/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

#include "TRMMonitorService.h"
#include "rdk_debug.h"

#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <cstring>

#include <uuid/uuid.h>
#include "trm/MessageProcessor.h"
#include "trm/Activity.h"
#include "trm/JsonEncoder.h"
#include "trm/JsonDecoder.h"

using namespace sdv;

#define LOG(...) RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", __VA_ARGS__) //printf(__VA_ARGS__); 
#define LOG_ERR(...) RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", __VA_ARGS__); //printf(__VA_ARGS__); 
#define LOG_DBG(...) RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.SDVAGENT", __VA_ARGS__);


class TRMMessageProcessor: public TRM::MessageProcessor
{
public :
    TRMMessageProcessor(TRMMonitorService *instance, SDVEventQueue *queue);
	void operator() (const TRM::NotifyTunerStatesUpdate &msg) ;
	void GetVersion(){}
private:
    TRMMonitorService *obj;
    SDVEventQueue *eventQueue;
};

TRMMonitorService::TUNER_USAGE_MAP TRMMonitorService::tunerUseMap;
pthread_mutex_t TRMMonitorService::mapLock;

TRMMonitorService::TRMMonitorService(SDVEventQueue* queue) {
	eventQueue = queue;
	isConnected = false;
	isRunning = false;

	usageChangeCallbackInstance = NULL;
	usageChangeCallback = NULL;
}


SDVService::SERVICE_RESULT TRMMonitorService::start() {
	LOG("[%s] start\n", CLASSNAME);

	isRunning = true;
	pthread_attr_init(&thread_attr);
	pthread_attr_setdetachstate(&thread_attr,  PTHREAD_CREATE_JOINABLE);
	pthread_create(&thread_id, &thread_attr, readServerThread, this);
    pthread_mutex_init(&mapLock, NULL);

	return SDVService::SUCCESS;
}

SDVService::SERVICE_RESULT TRMMonitorService::stop() {
	LOG("[%s] stop\n", CLASSNAME);

	isRunning = false;

	shutdown(socketFd, SHUT_RDWR);
	close(socketFd);

	pthread_join(thread_id, NULL);	// wait for socketReader thread to exit
	pthread_attr_destroy(&thread_attr);

    pthread_mutex_unlock(&mapLock);
    pthread_mutex_destroy(&mapLock);

	return SDVService::SUCCESS;
}

uint8_t TRMMonitorService::getTunerUsage(uint32_t sourceId){
    uint8_t result = FREE;
    pthread_mutex_lock(&mapLock);

    for(std::vector<TUNER_STATE>::iterator it = tunerUseMap.begin(); it != tunerUseMap.end(); ++it){
        if(it->sourceId == sourceId){
            result = it->state;
            break;
        }
    }
    pthread_mutex_unlock(&mapLock);
    return result;
}

void TRMMonitorService::notifyTunerStatesUpdate(SDVEventQueue::TASK_ID_t id, void *ptrObj, void *data ){
    LOG("[%s] %s: Received event from TRM JSON Decoder\n",CLASSNAME, __FUNCTION__);
    ((TRMMonitorService*)ptrObj)->updateTunerUsage((TRMMonitorService::TRM_TUNER_STATES*) data);
}

void TRMMonitorService::updateTunerUsage(TRMMonitorService::TRM_TUNER_STATES *tunerStateMap){
    TRMMonitorService::TUNER_USAGE_MAP changed;
    TRMMonitorService::TUNER_USAGE_MAP newMap;

    for(TRMMonitorService::TRM_TUNER_STATES::iterator it = tunerStateMap->begin(); it != tunerStateMap->end(); it++){ 
        TUNER_STATE tunerState = {0, FREE};
        TRMMonitorService::TRM_DETAILED_TUNER_STATE trmDTS = (TRMMonitorService::TRM_DETAILED_TUNER_STATE)*it;
 
        if(trmDTS.locator.compare(0,9,OCAP_LOCATOR_PREFIX) == 0){
            LOG_DBG("[%s] %s: Service locator prefix: ocap://\n",CLASSNAME, __FUNCTION__);
            std::string locator = trmDTS.locator.substr(9,std::string::npos); 
            tunerState.sourceId = (uint16_t) strtoul(locator.c_str(), NULL, 16);
            tunerState.state = 0;
        }
        else if(trmDTS.locator.compare(0,8,PPV_LOCATOR_PREFIX) == 0){
            LOG_DBG("[%s] %s: Service locator prefix: ppv://\n",CLASSNAME, __FUNCTION__);
            std::string locator = trmDTS.locator.substr(8,std::string::npos); 
            tunerState.sourceId = (uint16_t) strtoul(locator.c_str(), NULL, 16);
            tunerState.state = PPV;
        }
        else
        {
            LOG_ERR("Invalid locator: %s", trmDTS.locator.c_str());
            break;
        }

        uint8_t trmState = getTunerUseFromTRMTunerState(trmDTS.state);
        tunerState.state = tunerState.state | trmState; 

        LOG("[%s] %s: source: %d state %d\n",CLASSNAME, __FUNCTION__, tunerState.sourceId, tunerState.state);
        newMap.push_back(tunerState);
    }

    pthread_mutex_lock(&mapLock);

    // Find all brand new tuners uses and add to changed map
    for(TRMMonitorService::TUNER_USAGE_MAP::iterator newUse = newMap.begin(); newUse != newMap.end(); newUse++){
        bool isInUse = false;
        for(TRMMonitorService::TUNER_USAGE_MAP::iterator existingUse = tunerUseMap.begin(); existingUse != tunerUseMap.end(); existingUse++){
            if(existingUse->sourceId == newUse->sourceId){
                isInUse = true;
                break;
            }
        }
        if(!isInUse) {
            LOG("[%s] %s: new tuner use; sourceId=%d state=%d\n",CLASSNAME, __FUNCTION__, newUse->sourceId, newUse->state);
            changed.push_back(*newUse);
        }
    }

    // Find all existing tuners with usage change and add to changed map
    for(TRMMonitorService::TUNER_USAGE_MAP::iterator it = tunerUseMap.begin(); it != tunerUseMap.end(); it++){
        for(TRMMonitorService::TUNER_USAGE_MAP::iterator newUse = newMap.begin(); newUse != newMap.end(); newUse++){
            if(it->sourceId == newUse->sourceId){
                if(it->state != newUse->state){
                    LOG("[%s] %s: existing tuner use changed; sourceId=%d new state=%d\n",CLASSNAME, __FUNCTION__, newUse->sourceId, newUse->state);
                    changed.push_back(*newUse);
                }
                break;
            }
        }
    }

    tunerUseMap.clear();
    tunerUseMap = newMap;
    pthread_mutex_unlock(&mapLock);

    if (usageChangeCallback != NULL && usageChangeCallbackInstance != NULL) {
		for(TRMMonitorService::TUNER_USAGE_MAP::iterator it = changed.begin(); it != changed.end(); it++){
			LOG("[%s] %s: invoke usageChangeCallback for sourceId=%d usage=%d\n", CLASSNAME, __FUNCTION__,it->sourceId, it->state);
			usageChangeCallback(usageChangeCallbackInstance, it->sourceId, it->state);
		}
    }
    
    //Cleanup event data
    tunerStateMap->clear();
    delete tunerStateMap;
}

uint8_t TRMMonitorService::getTunerUseFromTRMTunerState(const std::string &trmTunerState){
    uint8_t ret;
    if(trmTunerState.compare("Live") == 0){
        ret = MAIN_TV;
    }
    else if(trmTunerState.compare("Record") == 0){
        ret = RECORDING;
    }
    else if(trmTunerState.compare("Hybrid") == 0){
        ret =  MAIN_TV | RECORDING;
    }
    else if(trmTunerState.compare("EAS") == 0){
        ret = MAIN_TV;
    }
    else {
        ret = FREE;
    }    

    return ret;
}

TRMMonitorService::~TRMMonitorService() {
	LOG("[%s] destructor\n", CLASSNAME);
}

void TRMMonitorService::setTunerUsageChangeCallback(void* callbackInstance, TUNER_USAGE_CHANGE_CALLBACK* callbackFunction) {
	usageChangeCallbackInstance = callbackInstance;
    usageChangeCallback = callbackFunction;
}

void TRMMonitorService::connectToServer() {
	struct sockaddr_in trm_address;
	int socket_fd;
	int socket_status = 0;


	if (!isConnected) {
		LOG("[%s] %s: connect to server\n" , CLASSNAME, __FUNCTION__);

		trm_address.sin_family = AF_INET;
		trm_address.sin_addr.s_addr = inet_addr(TRM_IP_ADDR);
		trm_address.sin_port = htons(TRM_PORT);
		socket_fd = socket(AF_INET, SOCK_STREAM, 0);

		while(1) {
			int retry_count = 10;
			socket_status = connect(socket_fd, (struct sockaddr *) &trm_address, sizeof(struct sockaddr_in));
			if (socket_status != 0  && retry_count > 0) {
				LOG_ERR("[%s] %s: connect failed - %d\n" , CLASSNAME, __FUNCTION__, socket_status);
				sleep(2);
				retry_count--;
			}
			else {
			   break;
			}
		}

		// If connect succeeded, configure socket calls as blocking
		if (socket_status == 0) {
			int current_flags = fcntl(socket_fd, F_GETFL, 0);
			current_flags &= (~O_NONBLOCK);
			fcntl(socket_fd, F_SETFL, current_flags);
			socketFd = socket_fd;
			isConnected = true;
		}
		else {
			LOG_ERR("[%s] %s: failed to connect to server; closing socket\n" , CLASSNAME, __FUNCTION__);
			close(socket_fd);
			socketFd = -1;
		}
	}
}

void TRMMonitorService::getTRMVersion(){
	LOG("[%s] %s: getting TRM Version\n" , CLASSNAME, __FUNCTION__);
 	uuid_t value;
	char guid[37];
	uuid_generate(value);
	uuid_unparse(value, guid);
	TRM::GetVersion msg(guid);
	std::vector<uint8_t> out;
	JsonEncode(msg, out);
	out.push_back( 0 );
	LOG("[%s] %s: getting TRM Version\n" , CLASSNAME, __FUNCTION__);
	postToServer((char *) &out[0], out.size());
}

void TRMMonitorService::processBuffer( const char* buf, int len)
{
	if (buf != NULL)
	{
		LOG("[%s] %s: message received from TRM:\n", CLASSNAME, __FUNCTION__);

		// For some reason rdklogger sometimes fails to log entire JSON buffer string; printf works
		if (rdk_dbg_enabled("LOG.RDK.SDVAGENT", RDK_LOG_INFO)) {
			printf("%s\n", buf);
		}

		std::vector<uint8_t> response;
		response.insert( response.begin(), buf, buf+len);
		TRMMessageProcessor trmProc = TRMMessageProcessor(this, eventQueue);
		TRM::JsonDecoder jdecoder(trmProc);
		jdecoder.decode(response);
	}
}

void* TRMMonitorService::readServerThread(void* arg) {
	TRMMonitorService* obj = (TRMMonitorService *)arg;

	int totalBytesRead = 0;
	int idx = 0;

	LOG("[%s] %s: starting thread\n" , CLASSNAME, __FUNCTION__);

	obj->connectToServer();
	if(!obj->isConnected) {
		LOG_ERR("[%s] %s: failed to connect to TRM server; exiting\n", CLASSNAME, __FUNCTION__);
		return NULL;
	}

	obj->getTRMVersion(); //needed?

	while (obj->isRunning) {
		if (obj->isConnected) {

			// First read TRM message header data
			uint8_t headerBuff[16];
			totalBytesRead = read(obj->socketFd, headerBuff, sizeof(headerBuff));
			LOG_DBG("[%s] %s: total header bytes read=%d\n", CLASSNAME, __FUNCTION__, totalBytesRead);

			if (totalBytesRead == sizeof(headerBuff)) {
				// Extract the payload length from the header
				int payload_length_offset = 12;
				uint32_t payloadLength =((((unsigned char)(headerBuff[payload_length_offset+0])) << 24) |
								 	 	(((unsigned char)(headerBuff[payload_length_offset+1])) << 16) |
								 	 	(((unsigned char)(headerBuff[payload_length_offset+2])) << 8 ) |
								 	 	(((unsigned char)(headerBuff[payload_length_offset+3])) << 0 ));
				LOG_DBG("[%s] %s: payload length=%d\n", CLASSNAME, __FUNCTION__, payloadLength);

				if (payloadLength > 0) {
					// Now read TRM message payload data
					uint8_t* payloadBuff = (uint8_t *) malloc(payloadLength+1); // +1 for string terminator
					totalBytesRead = read(obj->socketFd, payloadBuff, payloadLength);
					LOG_DBG("[%s] %s: total payload bytes read=%d\n", CLASSNAME, __FUNCTION__, totalBytesRead);

					if (totalBytesRead != 0) {
						// Process the payload message data
						payloadBuff[payloadLength] = '\0';
						obj->processBuffer((char *)payloadBuff, totalBytesRead);
					}
					else {
						LOG_ERR("[%s] %s: failed to to read payload; reconnecting...\n", CLASSNAME, __FUNCTION__);
						obj->isConnected = false;
					}
					free(payloadBuff);
					payloadBuff = NULL;
				}
				else {
					LOG_ERR("[%s] %s: payload length was zero; reconnecting...\n", CLASSNAME, __FUNCTION__);
					obj->isConnected = false;
				}
			}
			else {
				LOG_ERR("[%s] %s: invalid header size; reconnecting...\n", CLASSNAME, __FUNCTION__);
				obj->isConnected = false;
			}
		}
		else {
			LOG_ERR("[%s] %s: not connected to TRM server; sleep and retry...\n", CLASSNAME, __FUNCTION__);
			sleep(10);
			obj->connectToServer();
		}
	}
	LOG("[%s] %s: exit\n", CLASSNAME, __FUNCTION__);
}

bool TRMMonitorService::postToServer(const char *payload, int payloadLength) {
	unsigned char *buff = NULL;
	bool ret = false;

	connectToServer();
	LOG("[%s] %s: isConnected=%d\n", CLASSNAME, __FUNCTION__, isConnected);

	if (isConnected) {
		if (payloadLength != 0) {
			// Allocate buffer for header + payload
			static int message_id = 0x1000;
			const int header_length = 16;
			buff = (unsigned char *) malloc(payloadLength + header_length);

			int idx = 0;
			// Set header "Magic Word"
			buff[idx++] = 'T';
			buff[idx++] = 'R';
			buff[idx++] = 'M';
			buff[idx++] = 'S';
			// Set header type field to 0, as it is not used right now
			buff[idx++] = (0 & 0xFF000000) >> 24;
			buff[idx++] = (0 & 0x00FF0000) >> 16;
			buff[idx++] = (0 & 0x0000FF00) >> 8;
			buff[idx++] = (0 & 0x000000FF) >> 0;
			// Set header field for message id
			++message_id;
			buff[idx++] = (TRM_SDVAGENT_CONNECTION_ID & 0xFF000000) >> 24;
			buff[idx++] = (TRM_SDVAGENT_CONNECTION_ID & 0x00FF0000) >> 16;
			buff[idx++] = (TRM_SDVAGENT_CONNECTION_ID & 0x0000FF00) >> 8;
			buff[idx++] = (TRM_SDVAGENT_CONNECTION_ID & 0x000000FF) >> 0;
			// Set header field for payload length
			buff[idx++] = (payloadLength & 0xFF000000) >> 24;
			buff[idx++] = (payloadLength & 0x00FF0000) >> 16;
			buff[idx++] = (payloadLength & 0x0000FF00) >> 8;
			buff[idx++] = (payloadLength & 0x000000FF) >> 0;

			// Copy payload into buffer after header
			for (int i = 0; i < payloadLength; i++) {
				buff[idx + i] = payload[i];
			}

			LOG("[%s] %s: send message to TRM with payload length=%d\n", CLASSNAME, __FUNCTION__, payloadLength);

			// Write message to socket
			int write_count = write(socketFd, buff, payloadLength + header_length);
			free(buff);

			if (write_count == 0) {
				isConnected = false;
				LOG_ERR("[%s] %s: write failed\n", CLASSNAME, __FUNCTION__);
			} else {
				ret = true;
			}
		}
	}
	return ret;
}

TRMMessageProcessor::TRMMessageProcessor(TRMMonitorService *instance, SDVEventQueue *queue){
    obj = instance;
    eventQueue = queue;
}

void TRMMessageProcessor::operator() (const TRM::NotifyTunerStatesUpdate &msg)
{
    std::map<std::string, TRM::DetailedTunerState> trmData = msg.getTunerDetailedStates();
    LOG("[%s] %s: total tuner detailed states=%d\n", CLASSNAME, __FUNCTION__, trmData.size());
    TRMMonitorService::TRM_TUNER_STATES* data = new TRMMonitorService::TRM_TUNER_STATES();
    for(std::map<std::string, TRM::DetailedTunerState>::iterator it = trmData.begin(); it != trmData.end(); it++){
        TRMMonitorService::TRM_DETAILED_TUNER_STATE trmData = {it->second.getServiceLocator(), it->second.getState()};
        if(trmData.locator.compare("") != 0){
            data->push_back(trmData);
        }
    }
    eventQueue->push((void*)obj, obj->notifyTunerStatesUpdate, (void*)data, 0);
}




/*
// For testing without TRM
 *
#include <sstream>
#include <fstream>

void reportFakeTRM(TRMMonitorService* obj) {
    std::ifstream infile;

    RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","KT##: open /opt/faketrm.txt\n");
    infile.open("/opt/faketrm.txt", std::ifstream::in);
    if (infile.is_open()) {

        // Determine length of file
        infile.seekg (0, infile.end);
        int length = infile.tellg();
        infile.seekg (0, infile.beg);

        char* buff = (char*)malloc(length+1);
        RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","[TRMMonitorService] reading %d characters from faketrm.txt\n", length);

        // read data as a block:
        infile.read (buff, length);

        if (infile) {
            RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT", "[TRMMonitorService] read last char=%c , notify processBuffer\n", buff[length-1]);
            buff[length] = '\0';
            obj->processBuffer(buff, length+1);
        } else {
            RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT", "[TRMMonitorService] error: only could be read\n", infile.gcount());
        }

       infile.close();
       free(buff);
     }
}
*/
