/**
 * @file EventIndication.h
 */

#ifndef EVENTINDICATION_H_
#define EVENTINDICATION_H_

#include "SDVMessage.h"

namespace sdv {

/**
 * @class EventIndication
 * @brief CCMIS EvenIndication class definition
 *
 * @details Defines CCMIS EventIndication inbound and outbound message.
 */
class EventIndication : public SDVMessage {

public:

	/**
	 * @constructor
	 * @brief Constructor to create an inbound EventIndication SDV message from the specified raw data.
	 *
	 * @param [in] buffer raw data received from SDV Server
	 */
	EventIndication(uint8_t* buffer);

	/**
	 * @constructor
	 * @brief Constructor to create an outbound Tuner Use EventIndication SDV message for the specified parameters.
	 *
	 * @param [in] macAddress
	 * @param [in] tunerIndex
	 * @param [in] sourceId
	 * @param [in] tunerUse
	 */
	EventIndication(uint8_t* macAddress, uint8_t tunerIndex, uint32_t sourceId, uint8_t tunerUse);

	/**
	 * @fn serialize
	 * @brief Serialize this outbound message into the specified empty buffer.
	 * The serialized data will be ready for UDP transmission to the SDV Server.
	 *
	 * @param [out] emptyBuff - empty buffer to store serialized message
	 * @param [in] maxBuffLength - maximum length of empty buffer in bytes
	 * @return length of serialized message in bytes
	 */
	size_t serialize(uint8_t* emptyBuff, size_t buffLength);

	/**
	 * @fn getSourceId
	 * @brief Get the source ID in this message.
	 *
	 * @return source ID
	 */
	uint32_t getSourceId();

	/**
	 * @enum ATTRIBUTE_TAG
	 * @brief Defines values for attribute tag per CCMIS
	 */
	typedef enum attribute_tag {
		DISPLAY_BARKER = 0,
		TUNER_USE = 1
	} ATTRIBUTE_TAG;

	/**
	 * @fn getAttributeType
	 * @brief Get the type of attribute associated with this message
	 *
	 * @return value of message attribute tag
	 */
	ATTRIBUTE_TAG getAttributeType();

private:

	/**
	 * @struct ccmis_event_indciation
	 * @brief Data structure of CCMIS EventIndication that can be directly casted into packet buffer.
	 * Fields are defined in CCMIS 3.14
	 */
#pragma pack(push,1)
	typedef struct ccmis_event_indication {
		struct ccmis_header header;
		uint16_t reserved;
		uint16_t private_data_length;
		uint16_t reserved2;
		uint32_t source_id;
		uint8_t attribute_tag;
		uint8_t attribute_length;
		union {
			struct {
				uint16_t message_id;
			} display_barker; 	// applies only to inbound display barker message
			struct {
				uint8_t bit_flags;
				uint8_t reserved3;
			} tuner_use;		// applies only to outbound tuner use message
		} attribute_data;
	} CCMIS_EVENT_INDICATION;
#pragma pack(pop)

	static const uint16_t PAYLOAD_LENGTH = sizeof(struct ccmis_event_indication) - sizeof(struct ccmis_header); //!> excludes session ID
	static const uint16_t PRIVATE_DATA_LENGTH = 0x000a;
	static const uint8_t ATTRIBUTE_LENGTH = 2;

	uint32_t source_id;
	uint8_t tuner_use;
	ATTRIBUTE_TAG attribute_tag;
};

} /* namespace sdv */

#endif /* EVENTINDICATION_H_ */
