/**
 * @file ProgramSelectIndication.h
 *
 */

#ifndef PROGRAMSELECTINDICATION_H_
#define PROGRAMSELECTINDICATION_H_

#include "SDVMessage.h"

namespace sdv {

/**
 * @class ProgramSelectIndication
 * @brief CCMIS ProgramSelectIndication class definition
 *
 * @details Defines CCMIS ProgramSelectIndication inbound message.
 */
class ProgramSelectIndication : public SDVMessage {

public:

	/**
	 * @def REASON_xx
	 * @brief Reason codes for ProgramSelectIndication; values defined in CCMIS 3.14
	 */
	#define REASON_rspForceTune 	0x8000
	#define REASON_rspProgNotAvail 	0x8001

	/**
	 * @constructor
	 * @brief Constructor to create an ProgramSelectIndication SDV message from the specified raw data.
	 *
	 * @param [in] buffer raw data received from SDV Server
	 */
	ProgramSelectIndication(uint8_t* buffer);

	/**
	 * @fn getReason
	 * @brief Get the reason code in this message
	 *
	 * @return reason code, values defined in CCMIS 3.14
	 */
	uint16_t getReason();

	/**
	 * @fn getSourceId
	 * @brief Get the source ID in this message which indicates the service ID of the video stream now assigned to the tuner.
	 *
	 * @return source ID
	 */
	uint32_t getSourceId();

	/**
	 * @fn getModulation
	 * @brief Get the modulation format in this message.
	 *
	 * @return modulation format, values defined in CCMIS 3.14
	 */
	uint8_t getModulation();

	/**
	 * @fn getFrequency
	 * @brief Get the QAM tuning frequency in this message.
	 *
	 * @return tuning frequency in hertz
	 */
	uint32_t getFrequency();

	/**
	 * @fn getMpegProgramNumber
	 * @brief Get the MPEG program number in this message
	 *
	 * @return MPEG program number for transport stream
	 */
	uint16_t getMpegProgramNumber();

	/**
	 * @fn getCurrentSourceId
	 * @brief Get the source ID in this message which indicates the service ID the tuner is switching from.
	 *
	 * @return current source ID
	 */
	uint32_t getCurrentSourceId();


private:

	/**
	 * @struct ccmis_program_select_indication
	 * @brief Data structure of CCMIS ProgramSelectIndication that can be directly casted into packet buffer.
	 * Fields are defined in CCMIS 3.14
	 */
#pragma pack(push,1)
	typedef struct ccmis_program_select_indication {
		struct ccmis_header header;
		uint16_t reason;
		uint32_t source_id;
		uint16_t private_data_length;
		uint8_t reserved1;
		uint8_t modulation_format;
		uint32_t tuning_frequency;
		uint16_t mpeg_program_number;
		uint16_t reserved_2;
		uint32_t current_source_id;
	} CCMIS_PROGRAM_SELECT_INDICATION;
#pragma pack(pop)

	uint16_t reason;
	uint32_t source_id;
	uint8_t modulation_format;
	uint32_t tuning_frequency;
	uint16_t mpeg_program_number;
	uint32_t current_source_id;
};

} /* namespace sdv */

#endif /* PROGRAMSELECTINDICATION_H_ */
