/**
* @file SDVMessage.h
*
*/
#ifndef SDVMESSAGE_H_
#define SDVMESSAGE_H_


#include "stddef.h"
#include <cstdlib>
#include <stdint.h>



namespace sdv {

/**
 * @class SDVMessage
 * @brief SDVMessage class definition
 *
 * @details Base class for all SDV inbound and outbound messages specified in CCMIS 3.14 specification.
 * Defines and provides access to CCMIS message header data common to all messages.
 */
class SDVMessage {

public:
    /**
     * @enum MessageId
     *
     * @brief Enumeration of SDVMessaeg ids
     *
     * All Inbound and outbound SDVMessages will have one of these message IDs.
     */
	enum MessageId { ProgramSelectRequest = 0x0001,
		ProgramSelectConfirm = 0x0002,
		ProgramSelectIndication = 0x0003,
		ProgramSelectResponse = 0x0004,
		UserActivityReport = 0x8000,
		InitRequest = 0x8001,
		InitConfirm = 0x8002,
		QueryRequest = 0x8003,
		QueryConfirm = 0x8004,
		EventIndication = 0x8005,
		EventResponse = 0x8006
	};

    /**
     * Type of the transaction ids inside SDVMessages.
     */
	typedef uint32_t TRANACTION_ID_t;

    /**
     * @fn getMessageType
     *
     * @brief Determine what the type of SDVMessage is contained
     * within a buffer of serialized SDVMessage data.
     *
     * @param [in] buffer - buffer containing serialized SDVMessage data
     * @return enum indicating type of SDV message represented in this object
     */
	static MessageId getMessageType(unsigned char * buffer);

    /**
     * @fn getMessageId
     *
     * @brief Obtain the identifier of the message type represented by this object.
     *
     * @return the type of SDV message represented in this object
     */
	MessageId getMessageId() const {return _messageId;};

    /**
     * @fn getTransactionId
     *
     * @brief Obtain the transactionID of the message represented by this object.
     *
     * @return the transactionID represented in this object
     */
	TRANACTION_ID_t getTransactionId()const {return _transactionId;};

    /**
     * @fn getTransactionId
     *
     * @brief Obtain the ID of the tuner this message relates to
     *
     * @return the tuner ID or 0xFF if not relevant
     */
	int getTunerIndex() const {return _tunerIndex;};

	/**
	 * @fn serialize
	 *
	 * @brief Serialize the this outbound message into the specified empty buffer.
	 * The serialized data will be ready for UDP transmission to the SDV Server.
	 *
	 * @param [out] emptyBuff - empty buffer to store serialized message
	 * @param [in] maxBuffLength - maximum length of empty buffer in bytes
	 * @return length of serialized message in bytes
	 */
	virtual size_t serialize(uint8_t* emptyBuff, size_t maxBuffLength);


protected:

	/**
	 * @struct ccmis_header
	 *
	 * @brief Data structure of CCMIS header that can be directly casted into packet buffer.
	 * Fields are defined in CCMIS 3.14
	 *
	 * @details Note the 3 session fields at the end of this structure (10 bytes) are actually
	 * the very 1st field of message payload in CCMIS.  Because the session fields are common
	 * to all messages, we will handle them in the header structure.
	 */
#pragma pack(push,1)
	typedef struct ccmis_header {
		uint8_t protocol;
		uint8_t dsmcc_type;
		uint16_t message_id;
		uint32_t transaction_id;
		uint8_t reserved1;
		uint8_t adaptation_len;
		uint16_t message_len;
		uint8_t session_mac[6];
		uint8_t session_tuner_index;
		uint8_t session_reserved[3];
	} CCMIS_HEADER;
#pragma pack(pop)


    /**
     * @Constructor
     * @brief For unsolicited outbound Messages
     *
     * @param [in] macAddress - MAC address of the interface used to send or receive messages to SDVServer
     * @param [in] messageId - type of SDV message represented in this object.
     * @param [in] tunerIndex - ID of the tuner this message relates to; 0xFF if not relevant
     * @param [in] payloadLength - length of message payload
     */
    SDVMessage(uint8_t* macAddress, MessageId messageId, uint8_t tunerIndex, uint16_t payloadLength);


    /**
     * @Constructor
     * @brief For solicited outbound Messages
     *
     * @param [in] macAddress - MAC address of the interface used to send or receive messages to SDVServer
     * @param [in] messageId - type of SDV message represented in this object.
     * @param [in] tunerIndex - ID of the tuner this message relates to; 0xFF if not relevant
     * @param [in] payloadLength - length of message payload
     * @param [in] transactionId - ID of the transaction requesting this reply message
     */
    SDVMessage(uint8_t* macAddress, MessageId messageId, uint8_t tunerIndex, uint16_t payloadLength, uint32_t transactionId);

    /**
     * @Constructor
     * @brief For Inbound Messages
     *
     * @param [in] header - pointer to CCMIS_HEADER structure
     */
    SDVMessage(CCMIS_HEADER* header);


private:

    uint8_t _macAddress[6];           	/**< MAC address of the interface used to send message to SDVServer */
    int _tunerIndex;                    /**< ID of the tuner this message relates to of 0xFF if not relevant*/
    uint16_t _payloadLength;			/**< length in bytes of the CCMIS payload for this message */

    SDVMessage::MessageId _messageId;   /**< The type of SDV message represented in this object*/
    TRANACTION_ID_t _transactionId;     /**< Transaction identifier of the Inbound/Outbound Message*/

    static int lastTransactionId;   	/**< Counter for outbound transactionId generation*/

};

}
#endif /* SDVMESSAGE_H_ */
