/*
 * @file SDVEventQueue.cpp
 * @Author Jason Bedard jasbedar@cisco.com
 * @Date Jun 20 2014
 *
 * @brief Implementation of Unit test for the SDVEventQueue.
 */
#include "SDVEventQueue.h"
#include "gtest/gtest.h"

#include <unistd.h>
#include <map>

namespace sdv {

struct callback_record{
	void * _callback_data;
	SDVEventQueue::TASK_ID_t _callback_task_id;
	void* _callback_ptrInstance;
	callback_record(	void * callback_data, SDVEventQueue::TASK_ID_t callback_task_id, void* callback_ptrInstance){
		_callback_data = callback_data;
		_callback_task_id= callback_task_id;
		_callback_ptrInstance = callback_ptrInstance;
	}
};
std::map<int,callback_record*> callbackEvents;

void testCallback(SDVEventQueue::TASK_ID_t id, void* ptrInstance, void * data){
	callback_record * record = new callback_record(data, id, ptrInstance);
	callbackEvents.insert(std::pair<SDVEventQueue::TASK_ID_t,callback_record*>(id, record));
}

//TEST(SDVEventQueue, PostEventImmediate) {
//	SDVEventQueue queue = SDVEventQueue();
//	ASSERT_EQ(SDVService::SUCCESS,queue.start());
//
//	const char * excpeted_data[3];
//	excpeted_data[0]= "This is my test Data One";
//	excpeted_data[1]= "This is my test Data Two";
//	excpeted_data[2]= "This is my test Data Three";
//
//	SDVEventQueue::TASK_ID_t expectedeIDs[3];
//	for(int i=0;i!=sizeof(expectedeIDs);++i){
//		expectedeIDs[i] = queue.push((void*)this, &testCallback,(void*)&excpeted_data[i], 0);
//		ASSERT_EQ(expectedeIDs[i],i);
//	}
//	usleep(50000);
//	for(int i=0;i!=sizeof(expectedeIDs);++i){
//		std::map<int,callback_record*>::iterator it = callbackEvents.find(i);
//		ASSERT_EQ(sizeof(expectedeIDs), callbackEvents.size());
//		callback_record * rec = it->second;
//		ASSERT_EQ(expectedeIDs[i],rec->_callback_task_id);
//		ASSERT_EQ(&excpeted_data[i],rec->_callback_data);
//		ASSERT_EQ(this,rec->_callback_ptrInstance);
//		delete rec;
//	}
//	ASSERT_EQ(SDVService::SUCCESS,queue.stop());
//	callbackEvents.clear();
//}

TEST(SDVEventQueue, PostDelayed) {
	SDVEventQueue queue = SDVEventQueue();
	ASSERT_EQ(SDVService::SUCCESS,queue.start());

	const char * excpeted_data[0];
	excpeted_data[0]= "This is my test Data One";

	SDVEventQueue::TASK_ID_t expectedeIDs[1];
	expectedeIDs[0] = queue.push((void*)this, &testCallback,(void*)&excpeted_data[0], 250);
	ASSERT_EQ(expectedeIDs[0],0);

	usleep(150000);
	//Make sure event has not yet fired after 150 ms.
	std::map<int,callback_record*>::iterator it = callbackEvents.find(expectedeIDs[0]);
	ASSERT_EQ(0, callbackEvents.size());

	usleep(200000);
	//Make sure event has fired now.
	it = callbackEvents.find(expectedeIDs[0]);
	ASSERT_TRUE(it != callbackEvents.end());
	callback_record * rec = it->second;
	ASSERT_EQ(expectedeIDs[0],rec->_callback_task_id);
	ASSERT_EQ(&excpeted_data[0],rec->_callback_data);
	ASSERT_EQ(this,rec->_callback_ptrInstance);
	delete rec;
	ASSERT_EQ(SDVService::SUCCESS,queue.stop());

	callbackEvents.clear();
}
TEST(SDVEventQueue, MultipleDelayedEvents) {
	SDVEventQueue queue = SDVEventQueue();
	ASSERT_EQ(SDVService::SUCCESS,queue.start());

	const char * excpeted_data[3];
	excpeted_data[0]= "This is my test Data One";
	excpeted_data[1]= "This is my test Data Two";
	excpeted_data[2]= "This is my test Data Three";

	SDVEventQueue::TASK_ID_t expectedeIDs[3];
	expectedeIDs[0] = queue.push((void*)this, &testCallback,(void*)&excpeted_data[0], 250);
	expectedeIDs[1] = queue.push((void*)this, &testCallback,(void*)&excpeted_data[1], 50);
	expectedeIDs[2] = queue.push((void*)this, &testCallback,(void*)&excpeted_data[2], 100);

	usleep(25000);
	//Make sure event has not yet fired after 25 ms.
	std::map<int,callback_record*>::iterator it = callbackEvents.find(expectedeIDs[0]);
	ASSERT_EQ(0, callbackEvents.size());

	usleep(30000);
	//Make sure only event expectedeIDs[1] fired after > 55 ms
	ASSERT_EQ(callbackEvents.size(),1);
	it = callbackEvents.find(expectedeIDs[1]);
	ASSERT_TRUE(it != callbackEvents.end());
	callback_record * rec = it->second;
	ASSERT_EQ(expectedeIDs[1],rec->_callback_task_id);
	delete rec;

	usleep(50000);
	//Make sure only event expectedeIDs[2] fired after >105 ms
	ASSERT_EQ(callbackEvents.size(),2);
	it = callbackEvents.find(expectedeIDs[2]);
	ASSERT_TRUE(it != callbackEvents.end());
	rec = it->second;
	ASSERT_EQ(expectedeIDs[2],rec->_callback_task_id);
	delete rec;

	usleep(900000);
	//Make sure only event expectedeIDs[2] fired after >255 ms
	ASSERT_EQ(callbackEvents.size(),3);
	it = callbackEvents.find(expectedeIDs[0]);
	ASSERT_TRUE(it != callbackEvents.end());
	rec = it->second;
	ASSERT_EQ(expectedeIDs[0],rec->_callback_task_id);
	delete rec;

	ASSERT_EQ(SDVService::SUCCESS,queue.stop());
	callbackEvents.clear();
}
}
