/*
 * Broadcom oops capture and display (boops)
 *
 * Copyright (C) 2017 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <stdarg.h>
#include <stdint.h>
#include <unistd.h>
#include <uuid/uuid.h>
#include <argp.h>
#include <fcntl.h>
#include <boops_shared.h>

#define BOOPS_DUMP_START                   "==============================PREVIOUS_KERNEL_OOPS_DUMP_START==============================\n"
#define BOOPS_DUMP_SEPARATE                "===========================================================================================\n"
#define BOOPS_DUMP_END                     "===============================PREVIOUS_KERNEL_OOPS_DUMP_END===============================\n"
#define BOOPS_REPORTED_SIGNATURE           "BOOPS-REPORTED"
#define BOOPS_DEFAULT_OOPS_PATH            "/tmp/boops.dat"
#define BOOPS_MONITOR_SEC                  10
#define BOOPS_MONITOR_LOOP_PER_PRINT       180

#define BOOPS_READ_BUF_SIZE (2*1024)
#define BOOPS_PRINT_MAX_LINE_LENGTH (256)

/* Ensure we never send a line longer than the proc can handle. We can wrap, he won't */
#define BOOPS_MAX_LOG_LINE_SIZE (BOOPS_PROC_MAX_LOG_LINE_SIZE-2) /*-1 for \n, -1 for \0. */
#define BOOPS_MAX_SEGMENTS               128


enum boops_segment_type_t {
    BOOPS_HEADER_INVALID,
    BOOPS_HEADER_OOPS_NEW,
    BOOPS_HEADER_OOPS_REPORTED
};

enum boops_log_t {
    boops_log           = 1u << 0,
    boops_debug         = 1u << 1,
    boops_dump_headers  = 1u << 2,
    boops_dump_data     = 1u << 3,

    boops_message       = 1u << 29,
    boops_warning       = 1u << 30,
    boops_error         = 1u << 31,
};

struct boops_segment_t {
    struct boops_segment_header_t header;
    char* data;
    uint32_t data_size;
    off_t header_pos;
    off_t data_pos;
};

struct boops_oops_t {
    enum boops_segment_type_t type;
    struct boops_segment_t segment[BOOPS_MAX_SEGMENTS];
    uint32_t num_segments;
    off_t start_offset;

    struct boops_extended_header_t extended_info;
    char file_name[128];
    char uuid_str[37];
    char date_str[20];
};


struct boops_t {
    char line_buffer[BOOPS_PRINT_MAX_LINE_LENGTH+1];        /* Buffer for normal formatted prints */

    /* Buffer mechanism for printing log messages. When printing via the kernel we always print
       one full line so +1 for \n, +1 for \0.  */
    char log_buf[BOOPS_MAX_LOG_LINE_SIZE+2];
    char *log_buf_write;
    size_t log_buf_remaining;
    char log_prefix[16];
    bool log_prefix_needed;
    int kernel_fd;
    char current_boot_uuid[37]; /* This is the UUID of the current boot. NOT when the oops happned */

    const char* input_file_name;
    const char* output_file_name;
    off_t input_file_byte_offset;
    bool log_prefix_enabled;
    bool erase_oops;
    bool monitor;
    bool kernel_print;
    enum boops_log_t log_level;
};

////////////////////////////////////////////////////////////////
// Logging and Print
//////////////////

/* Logging functions to be used throughout the application */
#define boops_err(fmt, ...)  boops_print(boops, boops_error,   "ERROR: ",         NULL,         0,        fmt, ##__VA_ARGS__);
#define boops_wrn(fmt, ...)  boops_print(boops, boops_warning, "WARNING: ",       NULL,         0,        fmt, ##__VA_ARGS__);
#define boops_log(fmt, ...)  boops_print(boops, boops_log,     boops->log_prefix, NULL,         0,        fmt, ##__VA_ARGS__);
#define boops_msg(fmt, ...)  boops_print(boops, boops_message, "",                NULL,         0,        fmt, ##__VA_ARGS__);
#define boops_dbg(fmt, ...)  boops_print(boops, boops_debug,   "DBG: ",           __FUNCTION__, __LINE__, fmt, ##__VA_ARGS__);

static inline
bool boops_log_kernel_open(struct boops_t *boops) {
    boops->kernel_fd = open("/proc/" BOOPS_PROC_DIR "/" BOOPS_PROC_SEND_LOG, O_WRONLY | O_NOCTTY);
    if (boops->kernel_fd < 0) {
        return false;
    }
    return true;
}

static inline
void boops_log_kernel_close(struct boops_t *boops) {
    if (boops->kernel_fd >= 0)
        close(boops->kernel_fd);
}

static inline
void boops_put_log_kernel(struct boops_t *boops) {
    *(boops->log_buf_write)++='\n';
    *(boops->log_buf_write)++='\0';
    write(boops->kernel_fd, boops->log_buf, boops->log_buf_write - boops->log_buf);
    boops->log_buf_remaining = BOOPS_MAX_LOG_LINE_SIZE;
    boops->log_buf_write = boops->log_buf;
}

static inline
void boops_log_kernel(struct boops_t *boops, const char*buf) {
    if (!boops->log_buf_write) return;

    if (!buf)
        boops_put_log_kernel(boops);
    else {
        size_t buf_len = strlen(buf);
        while(buf_len) {
            size_t copy_size = (buf_len < boops->log_buf_remaining) ? buf_len : boops->log_buf_remaining;;
            strncpy(boops->log_buf_write, buf, copy_size);
            boops->log_buf_write+=copy_size;
            boops->log_buf_remaining-=copy_size;
            buf+=copy_size;
            buf_len-= copy_size;

            if (buf_len && !boops->log_buf_remaining) {
                boops_put_log_kernel(boops);
            }
        }
    }
}

static inline
void boops_put_log(struct boops_t *boops, const char*buf) {
    if (boops->kernel_print)
        boops_log_kernel(boops, buf);
    else
        fputs(buf ? buf : "\n", stdout);
}

static
void boops_log_buffer(struct boops_t *boops, char*buf) {
    char *line = NULL;

    if (boops && (boops->log_level & boops_log) != boops_log) return;

    while ((line = strsep(&buf, "\n")) != NULL) {
        if (boops->log_prefix_enabled && boops->log_prefix_needed && strlen(line)) {
            boops_put_log(boops, boops->log_prefix);
            boops->log_prefix_needed=false;
        }

        boops_put_log(boops, line);

        if (buf) {
            buf[-1] = '\n'; /* put the delim back in the buffer */
            if (boops->log_prefix_enabled && boops->log_prefix_needed) /* This was an empty line */
                boops_put_log(boops, boops->log_prefix);
            boops_put_log(boops, NULL); /* Send EOL */
            boops->log_prefix_needed=true;
        }
    }

    fflush(stdout);
}

static
void boops_print(struct boops_t *boops, const enum boops_log_t level, const char* prefix, const char* func, uint32_t line, const char*fmt, ...) {
    va_list ap;

    if (boops && (boops->log_level & level) != level) return;

    va_start(ap, fmt);
    vsnprintf(boops->line_buffer, sizeof(boops->line_buffer), fmt, ap);
    va_end(ap);

    if (level == boops_log)
        boops_log_buffer(boops, boops->line_buffer);
    else
        if (func)
            fprintf(stdout, "%s[%u] %s", __FUNCTION__, __LINE__, boops->line_buffer);
        else
            fputs(boops->line_buffer, stdout);
}


static
void boops_dump_buffer(struct boops_t *boops, const enum boops_log_t level, void*ctx, off_t start_offset, uint8_t*start, size_t size) {
    unsigned int file_offset = (unsigned int)(start_offset - boops->input_file_byte_offset);
    size_t start_align = file_offset & 0xF;
    uint8_t* pos = start - start_align;
    uint8_t* end = start + size;

    if (boops && (boops->log_level & level) != level) return;

    /* Some helper macros for this function */
    #define PRINT_HEX(x) ((((pos+x) < end) && ((pos+x) >= start)) ? ((uint8_t)pos[x]) : 0xFF)
    #define PRINT_CHAR(x) ((((pos+x) < end) && ((pos+x) >= start) && (pos[x] >=' ') && (pos[x] <=126)) ? pos[x] : '.' )
    #define ERASE_POS(ep) { \
        uint32_t es = ((ep) > 7) ? 1 : 0;\
        boops->line_buffer[(ep) * 3 + 10 + es]=' ';\
        boops->line_buffer[(ep) * 3 + 10 + es + 1]=' ';\
        boops->line_buffer[(ep) + 61]=' ';\
    }

    printf("Dump %u bytes [0x%08x-0x%08x]\n", size, file_offset, file_offset +size);
    while(pos < end) {
        snprintf(boops->line_buffer, sizeof(boops->line_buffer),
            "%08X  %02X %02X %02X %02X %02X %02X %02X %02X  %02X %02X %02X %02X %02X %02X %02X %02X  |%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c|\n",
            ((pos-start) + file_offset),
            PRINT_HEX(0),  PRINT_HEX(1),  PRINT_HEX(2),   PRINT_HEX(3),   PRINT_HEX(4),   PRINT_HEX(5),   PRINT_HEX(6),   PRINT_HEX(7),
            PRINT_HEX(8),  PRINT_HEX(9),  PRINT_HEX(10),  PRINT_HEX(11),  PRINT_HEX(12),  PRINT_HEX(13),  PRINT_HEX(14),  PRINT_HEX(15),
            PRINT_CHAR(0), PRINT_CHAR(1), PRINT_CHAR(2),  PRINT_CHAR(3),  PRINT_CHAR(4),  PRINT_CHAR(5),  PRINT_CHAR(6),  PRINT_CHAR(7),
            PRINT_CHAR(8), PRINT_CHAR(9), PRINT_CHAR(10), PRINT_CHAR(11), PRINT_CHAR(12), PRINT_CHAR(13), PRINT_CHAR(14), PRINT_CHAR(15));
        while(start_align) {
            ERASE_POS(start_align-1);
            start_align--;
        }
        pos+=16;
        while(pos > end) {
            ERASE_POS(0x10 - (pos - end));
            pos--;
        }
        printf("[%p] %s", ctx, boops->line_buffer);
    }
}

////////////////////////////////////////////////////////////////
// Command line arguments
//////////////////
const char *argp_program_version = "bOops 1.0";
const char *argp_program_bug_address = "Broadcom";
static char doc[] = "Show the contents of previous system oops messages";
static char args_doc[] = "";

static struct argp_option options[] = {
    /* File location. Group 0 */
    {"in_file",       'f', "FILE",   0,    "Name of the input file containing a raw oops dump", 0 },
    {"out_file",      'o', "OUTFILE",0,    "Name of the output file to store the oops", 0 },
    {"byte_offset",   'b', "START",  0,    "Byte offset in the file where the oops is store", 0 },

    /* Printing controls, group 1 */
    {"no_dump",       '0', 0,        0,    "Don't print oops dump data", 1 },
    {"debug",         '1', 0,        0,    "Enable debug prints", 1 },
    {"raw_header",    'h', 0,        0,    "Dump the raw header buffer data", 1 },
    {"raw_data",      'd', 0,        0,    "Dump the raw data buffer data", 1 },
    {"prefix",        'p', 0,        0,    "Prefix each line of the dump noting the type of oops", 1 },
    {"kernel",        'k', 0,        0,    "Force log messages through boops-k (kernel mode)", 1 },

    /* Control how the log messages are print. Group 2 */

    /* Commands. Group 3 */
    {"monitor",       'm', 0,        0,    "Remain and monitor for dumps", 2 },
    {"erase",         'e', 0,        0,    "Erase the oops stored in the device", 2 },
    { 0, 0, 0, 0, 0, 0}
};

static
error_t boops_parse_arguments(int key, char *arg, struct argp_state *state) {
    struct boops_t *boops = state->input;
    switch (key) {
    case 'f':
        boops->input_file_name = arg;
        break;
    case 'o':
        boops->output_file_name  = arg;
        break;
    case 'b':
        boops->input_file_byte_offset = atoll(arg);
        break;
    case 'p':
        boops->log_prefix_enabled = true;
        break;
    case 'e':
        boops->erase_oops = true;
        break;
    case 'm':
        boops->monitor = true;
        break;
    case 'k':
        boops->kernel_print = true;
        break;
    case '0':
        boops->log_level &= (~boops_log);
        break;
    case '1':
        boops->log_level |= boops_debug;
        break;
    case 'h':
        boops->log_level |= boops_dump_headers;
        break;
    case 'd':
        boops->log_level |= boops_dump_data;
        break;
    default:
        return ARGP_ERR_UNKNOWN;
    }
  return 0;
}
static struct argp argp = { options, boops_parse_arguments, args_doc, doc, 0, 0, 0 };


#define _fseeko(file, pos, type)        ({ int r = fseeko(file, pos, type); boops_dbg("[%p] Seek type '%s' of %lld bytes returned %u\n", file, #type, pos, r); r; });
#define _fopen(name, perms)             ({ FILE*f = fopen(name, perms); boops_dbg("[%p] Open '%s' with permissions '%s'\n", f, name, perms); f; })
#define _fclose(file)                   fclose(file); boops_dbg("[%p] Close handle %p\n", file, file);
#define _fread(buf, size, count, file)  ({ size_t v = fread(buf, size, count, file); boops_dbg("[%p] Read %u of %u bytes\n", file, v, (size)*(count)); v; })
#define _fwrite(buf, size, count, file)  ({ size_t v = fwrite(buf, size, count, file); boops_dbg("[%p] Wrote %u of %u bytes\n", file, v, (size)*(count)); v; })

////////////////////////////////////////////////////////////////
// Read/write oops capture
//////////////////

static
enum boops_segment_type_t boops_read_segment(struct boops_t *boops, FILE* device, struct boops_segment_t* out, struct boops_extended_header_t *ex_out, bool read_segments) {
    const size_t SEGMENT_HEADER_SIZE = sizeof(struct boops_segment_header_t);
    const size_t PRIMARY_HEADER_SIZE = sizeof(struct boops_primary_header_t);
    const size_t EXTENDED_HEADER_SIZE = sizeof(struct boops_extended_header_t);
    size_t bytes;
    off_t header_pos = ftello(device);
    enum boops_segment_type_t type = BOOPS_HEADER_INVALID;

    if (!out || (header_pos < 0)) {
        boops_err("Invalid parameters to boops_read_segment\n");
        return BOOPS_HEADER_INVALID;
    }

    bytes = _fread(out, 1, SEGMENT_HEADER_SIZE, device);
    boops_dump_buffer(boops, boops_dump_headers, device, header_pos, (void*)out, bytes);
    if (bytes != SEGMENT_HEADER_SIZE) {
        boops_err("Failed to read header (bytes:%d)\n", bytes);
        return BOOPS_HEADER_INVALID;
    }

    if (strncmp((const char*)out->header.signature, BOOPS_SIGNATURE, sizeof(BOOPS_SIGNATURE)) == 0)
        type=BOOPS_HEADER_OOPS_NEW;
    else if (strncmp((const char*)out->header.signature, BOOPS_REPORTED_SIGNATURE, sizeof(BOOPS_REPORTED_SIGNATURE)) == 0)
        type=BOOPS_HEADER_OOPS_REPORTED;
    else
        return BOOPS_HEADER_INVALID; /* There is no signature match */

    boops_dbg("Found segment type %u\n", type);

    if (out->header.bytes.total < (out->header.bytes.header + out->header.bytes.data)) {
        boops_err("Segment has invalid size [header:%u data:%u total:%u]\n",
                    out->header.bytes.header, out->header.bytes.data, out->header.bytes.total);
        return BOOPS_HEADER_INVALID;
    }

    if (out->header.bytes.header == PRIMARY_HEADER_SIZE) {
        /* THe extended header is after the primary header */
        if (ex_out) {
            off_t ex_pos = ftello(device);
            boops_dbg("Reading extended header %u bytes\n", EXTENDED_HEADER_SIZE);
            bytes = _fread(ex_out, 1, EXTENDED_HEADER_SIZE, device);
            boops_dump_buffer(boops, boops_dump_headers, device, ex_pos, (void*)ex_out, bytes);
            if (bytes != EXTENDED_HEADER_SIZE)
                boops_err("Failed to read extended header (bytes:%d)\n", bytes);
        }
    }
    else if (out->header.bytes.header != SEGMENT_HEADER_SIZE) {
        boops_err("Segment disagrees about header size [%u != %u && %u]\n", out->header.bytes.header, SEGMENT_HEADER_SIZE, PRIMARY_HEADER_SIZE);
        return BOOPS_HEADER_INVALID;
    }

    out->header_pos = header_pos;
    out->data_pos = header_pos + out->header.bytes.header;
    out->data_size = out->header.bytes.data;

    if (read_segments) { /* No need to read data if segments aren't requested */
        int err = _fseeko(device, out->data_pos, SEEK_SET);
        if (err) {
            boops_err("Failed to seek to data pos %u (%d)\n", out->data_pos, err);
            _fclose(device);
            return -1;
        }

        out->data = malloc(out->data_size);
        if (!boops) {
            boops_err("Failed to allocate %u bytes for segment data\n", out->header.bytes.data);
            return BOOPS_HEADER_INVALID;
        }

        bytes = _fread(out->data, 1, out->data_size, device);
        boops_dump_buffer(boops, boops_dump_data, device, out->data_pos, (void*)out->data, bytes);
        if (bytes != out->data_size) {
            boops_err("Failed to segment data (bytes_read:%d)\n", bytes);
            free(out->data);
            return BOOPS_HEADER_INVALID;
        }
    }

    return type;
}

static
int boops_mark_reported_oops(struct boops_t *boops, struct boops_oops_t* oops_data) {
    FILE* device = NULL;
    int err = 0;
    int i = 0;

    if (!oops_data || !oops_data->num_segments) {
        return 0;
    }

    if (oops_data->type == BOOPS_HEADER_OOPS_REPORTED) {
        boops_dbg("Oops already marked reported\n");
        return 0;
    }

    if (sizeof(BOOPS_REPORTED_SIGNATURE) > sizeof(BOOPS_SIGNATURE)) {
        boops_err("Reported signature too large to store.\n");
        return -1;
    }

    boops_dbg("Begin marking the oops reported\n");
    device = _fopen(oops_data->file_name, "r+b");
    if (!device) {
        boops_err("Failed to open file '%s' for writing\n", oops_data->file_name);
        return -1;
    }

    err = _fseeko(device, oops_data->start_offset, SEEK_SET);
    if (err) {
        boops_err("Failed to seek to start pos %u (%d)\n", oops_data->start_offset, err);
        _fclose(device);
        return -1;
    }

    for (i=0; (err == 0) && i < oops_data->num_segments; i++) {
        struct boops_segment_t *cur = &oops_data->segment[i];
        size_t bytes_written;

        memcpy((char*)cur->header.signature, BOOPS_REPORTED_SIGNATURE, sizeof(cur->header.signature));

        err = _fseeko(device, cur->header_pos, SEEK_SET);
        if (err) {
            boops_err("Failed to seek to block %u (%d)\n", cur->header_pos, err);
            break;
        }

        bytes_written = _fwrite(&cur->header, 1, sizeof(cur->header), device);
        if (bytes_written != sizeof(cur->header)) {
            boops_err("Failed to write header %u (bytes_written:%d)\n", sizeof(cur->header), bytes_written);
            err=-1;
            break;
        }
    }
    _fclose(device);

    boops_dbg("Done marking the oops reported\n");
    return err;
}

static
bool boops_print_oops(struct boops_t *boops, struct boops_oops_t* oops_data) {
    int i;

    if (!oops_data || !oops_data->num_segments) {
        boops_dbg("No valid oops file found\n");
        return false;
    }

    if (oops_data->type == BOOPS_HEADER_OOPS_REPORTED && boops->kernel_print) {
        boops_dbg("Oops already marked reported. We will not print via kernel message again\n");
        return false;
    }

    boops_dbg("Begin printing oops dump\n");

    snprintf(boops->log_prefix, sizeof(boops->log_prefix), "[%.5s] ", oops_data->uuid_str);

    boops_log(BOOPS_DUMP_START);
    boops_log("Info:\n");
    boops_log("   Oops Date:            %s UTC\n", oops_data->date_str);
    boops_log("   Build:                %s\n", oops_data->extended_info.build_version);
    boops_log("   Occurred During Boot: %s\n", oops_data->uuid_str);
    boops_log("   Reason:               %s\n", oops_data->extended_info.reason);
    boops_log(BOOPS_DUMP_SEPARATE);
    for (i=oops_data->num_segments-1; i >= 0; i--)
        boops_log_buffer(boops, oops_data->segment[i].data);
    boops_log("\n");
    boops_log(BOOPS_DUMP_END);

    boops->log_prefix[0]='\0';

    boops_dbg("Done printing oops dump\n");
    return true;
}

static
int boops_write_oops(struct boops_t *boops, struct boops_oops_t* oops_data, const char* file_name_in) {
    FILE* device = NULL;
    int err = 0;
    int i = 0;
    char file_name[256];
    char* f_out = file_name;
    const char* f_end = file_name + sizeof(file_name) - 1;

    if (!oops_data || !oops_data->num_segments)
        return 0;

    boops_dbg("Begin writing oops dump\n");

    memset(file_name, 0, sizeof(file_name));
    for(i =0; f_out < f_end && i < strlen(file_name_in); i++) {
        if (file_name_in[i] != '#') {
            *f_out++ = file_name_in[i];
            continue;
        }
        if (strncmp(&file_name_in[i], "#uuid", 5) == 0) {
            if (f_end-f_out > 14) {
                snprintf(f_out, 13, "uuid_%s", oops_data->uuid_str);
                f_out+=13;
            }
            i+=4;
        }
        else if (strncmp(&file_name_in[i], "#UUID", 5) == 0) {
            strncpy(f_out, oops_data->uuid_str, f_end-f_out);
            f_out+=strlen(oops_data->uuid_str);
            i+=4;
        }
        else if (strncasecmp(&file_name_in[i], "#date", 5) == 0) {
            snprintf(f_out, f_end-f_out, "%04d_%02d_%02d_%02d%02d",
                BOOPS_DATE_GET_YEAR(oops_data->extended_info.oops_date), BOOPS_DATE_GET_MONTH(oops_data->extended_info.oops_date), BOOPS_DATE_GET_DAY(oops_data->extended_info.oops_date),
                BOOPS_DATE_GET_HOUR(oops_data->extended_info.oops_date), BOOPS_DATE_GET_MIN(oops_data->extended_info.oops_date));
            f_out+=15;
            i+=4;
        }
    }

    if (f_out >= f_end) {
        boops_err("The file name '%s' is too long\n", file_name_in);
        return -1;
    }

    boops_dbg("Outputting to file '%s'\n", file_name);
    device = _fopen(file_name, "wb");
    if (!device) {
        boops_err("Failed to open file '%s' for writing\n", file_name);
        return -1;
    }

    fputs(BOOPS_DUMP_START, device);
    fputs("Info:\n", device);
    fprintf(device, "   Oops Date:            %s UTC\n", oops_data->date_str);
    fprintf(device, "   Build:                %s\n", oops_data->extended_info.build_version);
    fprintf(device, "   Occurred During Boot: %s\n", oops_data->uuid_str);
    fprintf(device, "   Reason:               %s\n", oops_data->extended_info.reason);
    fputs(BOOPS_DUMP_SEPARATE, device);
    for (i=oops_data->num_segments-1; i >= 0; i--) {
        struct boops_segment_t *cur = &oops_data->segment[i];

        boops_dbg("Writing %u bytes\n", cur->data_size);
        size_t bytes_written = _fwrite(cur->data, 1, cur->data_size, device);
        if (bytes_written != cur->data_size) {
            boops_err("Failed to write %u bytes (bytes_written:%d)\n", cur->data_size, bytes_written);
            break;
        }
    }
    fputs(BOOPS_DUMP_END, device);
    _fclose(device);

    boops_dbg("Done erasing oops dump\n");
    return err;
}

static
int boops_erase_oops(struct boops_t *boops, struct boops_oops_t* oops_data) {
    FILE* device = NULL;
    int err = 0;
    int i = 0;
    uint8_t* block_buffer = NULL;

    if (!oops_data || !oops_data->num_segments)
        return 0;

    boops_dbg("Begin erasing oops dump\n");

    device = _fopen(oops_data->file_name, "wb");
    if (!device) {
        boops_err("Failed to open file '%s' for writing\n", oops_data->file_name);
        return -1;
    }

    for (i=0; (err == 0) && i < oops_data->num_segments; i++) {
        struct boops_segment_t *cur = &oops_data->segment[i];

        block_buffer = malloc(cur->header.bytes.total);
        if (!block_buffer) {
            boops_err("Failed to allocate erase block buffer size %u\n", block_buffer);
            err=-1;
            break;
        }
        memset(block_buffer, 0, cur->header.bytes.total);

        err = _fseeko(device, cur->header_pos, SEEK_SET);
        if (err) {
            boops_err("Failed to seek to block %u (%d)\n", cur->header_pos, err);
            break;
        }

        boops_dbg("Erasing segment at offset %lld\n", ftello(device));
        size_t bytes_written = _fwrite(block_buffer, 1, cur->header.bytes.total, device);
        if (bytes_written != cur->header.bytes.total) {
            boops_err("Failed to erase %u bytes (bytes_written:%d)\n", cur->header.bytes.total, bytes_written);
            err=-1;
            break;
        }

        free(block_buffer);
        block_buffer=NULL;
    }

    if (block_buffer)
        free(block_buffer);

    _fclose(device);

    boops_dbg("Done erasing oops dump\n");
    return err;
}

static
void boops_destroy_oops(struct boops_t *boops, struct boops_oops_t* oops_data) {
    int i = 0;

    if (oops_data) {
        for (i=0; i < oops_data->num_segments; i++) {
            if (oops_data->segment[i].data)
                free(oops_data->segment[i].data);
        }
        free(oops_data);
    }
}

static
struct boops_oops_t* boops_read_oops(struct boops_t *boops, const char* file_name, off_t offset, bool read_segments) {
    struct boops_oops_t* oops_data;
    enum boops_segment_type_t type;
    FILE* device = NULL;
    int err = 0;

    oops_data = malloc(sizeof(*oops_data));
    if (!oops_data) {
        boops_err("Failed to allocate %u bytes for segment data\n", sizeof(*oops_data));
        return BOOPS_HEADER_INVALID;
    }
    memset(oops_data, 0, sizeof(*oops_data));
    strcpy(oops_data->file_name, file_name);
    oops_data->start_offset = offset;

    device = _fopen(oops_data->file_name, "r");
    if (!device) {
        boops_err("Failed to open file '%s' for reading\n", oops_data->file_name);
        free(oops_data);
        return NULL;
    }

    err = _fseeko(device, oops_data->start_offset, SEEK_SET);
    if (err) {
        boops_err("Failed to seek to block %u (%d)\n", oops_data->start_offset, err);
        free(oops_data);
        _fclose(device);
        return NULL;
    }

    oops_data->type = boops_read_segment(boops, device, &oops_data->segment[0], &oops_data->extended_info, read_segments);
    if (oops_data->type == BOOPS_HEADER_INVALID) {
        free(oops_data);
        _fclose(device);
        return NULL;
    }

    snprintf(oops_data->date_str, sizeof(oops_data->date_str), "%04d/%02d/%02d %02d:%02d:%02d",
            BOOPS_DATE_GET_YEAR(oops_data->extended_info.oops_date), BOOPS_DATE_GET_MONTH(oops_data->extended_info.oops_date), BOOPS_DATE_GET_DAY(oops_data->extended_info.oops_date),
            BOOPS_DATE_GET_HOUR(oops_data->extended_info.oops_date), BOOPS_DATE_GET_MIN(oops_data->extended_info.oops_date),   BOOPS_DATE_GET_SEC(oops_data->extended_info.oops_date));
    uuid_unparse_upper(oops_data->extended_info.uuid, oops_data->uuid_str);

    boops_dbg("Found oops dump type %u\n", oops_data->type);

    if (read_segments) {
        for (oops_data->num_segments=1; oops_data->num_segments < BOOPS_MAX_SEGMENTS; oops_data->num_segments++) {
            struct boops_segment_t *cur = &oops_data->segment[oops_data->num_segments];
            struct boops_segment_t *prev = &oops_data->segment[oops_data->num_segments - 1];

            err = _fseeko(device, prev->header_pos + prev->header.bytes.total, SEEK_SET);
            if (err) {
                boops_err("Failed to seek to block %u (%d)\n", oops_data->start_offset, err);
                free(oops_data);
                _fclose(device);
                return NULL;
            }

            type = boops_read_segment(boops, device, cur, NULL, read_segments);
            if (type == BOOPS_HEADER_INVALID) {
                boops_dbg("Found an invalid header. Most likely this is expected and the end of the dump\n");
                break;
            }

            if (type != oops_data->type) {
                boops_dbg("Next segment signature does not match this oops\n");
                break;
            }

            if (memcmp(cur->header.uuid, oops_data->extended_info.uuid, sizeof(cur->header.uuid)) != 0) {
                boops_dbg("Next segment UUID does not match this oops\n");
                break;
            }

            boops_dbg("Found next segment at offset %d\n", cur->header_pos);
        }
    }

    _fclose(device);
    boops_dbg("Successfully parsed oops dump\n");

    return oops_data;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////

static
int boops_monitor(struct boops_t *boops) {
    struct boops_oops_t *oops_mon = NULL;
    int print = BOOPS_MONITOR_LOOP_PER_PRINT;

    while(true) {
        if (print++ >= BOOPS_MONITOR_LOOP_PER_PRINT) {
            boops_log("Monitoring current boot: %s\n", boops->current_boot_uuid);
            print=0;
        }

        oops_mon = boops_read_oops(boops, boops->input_file_name, boops->input_file_byte_offset, false);
        if (oops_mon) {
            boops_destroy_oops(boops, oops_mon);
        }
        usleep(1000000*BOOPS_MONITOR_SEC);
    }

    return 0;
}


int main(const int argc, char *argv[]) {
    struct boops_t *boops = NULL;
    struct boops_oops_t *oops_data = NULL;
    int err = 0;
    FILE* uuid_proc;

    boops = malloc(sizeof(*boops));
    if (!boops) {
        fprintf(stderr, "Failed to allocate memory");
        exit(1);
    }

    memset(boops, 0, sizeof(*boops));
    boops->log_level = (boops_log | boops_message | boops_error | boops_warning );
    boops->log_buf_remaining = BOOPS_MAX_LOG_LINE_SIZE;
    boops->log_buf_write = boops->log_buf;
    boops->log_prefix_needed = true;
    boops->input_file_name = BOOPS_DEFAULT_OOPS_PATH;

    argp_parse(&argp, argc, argv, 0, 0, boops);

    if (boops->kernel_print && !boops_log_kernel_open(boops)) {
        boops_err("Failed to communicate with boops kernel module. Is it installed?\n");
        err=-1;
        goto out;
    }

    uuid_proc = _fopen("/proc/" BOOPS_PROC_DIR "/" BOOPS_PROC_UUID, "r");
    if (uuid_proc) {
        fgets(boops->current_boot_uuid, sizeof(boops->current_boot_uuid), uuid_proc);
        _fclose(uuid_proc);
        uuid_proc=NULL;
    }

    oops_data = boops_read_oops(boops, boops->input_file_name, boops->input_file_byte_offset, true);
    if (!boops_print_oops(boops, oops_data)) {
        boops_log("There was no OOPS in the previous run\n");
    }

    if (oops_data) {
        if (boops->output_file_name) {
            err = boops_write_oops(boops, oops_data, boops->output_file_name);
            if (err) {
                boops_err("Failed to write the oops (%d)\n", err);
                err=-3;
                goto out;
            }
        }

        if (boops->erase_oops) {
            err = boops_erase_oops(boops, oops_data);
            if (err) {
                boops_err("Failed to erase the oops (%d)\n", err);
                err=-4;
                goto out;
            }
        }
        else if (boops->kernel_print) {
            /* If we did a print via the kernel, this opps is considered "reported" */
            err = boops_mark_reported_oops(boops, oops_data);
            if (err) {
                boops_err("Failed to mark the oops reported (%d)\n", err);
                err=-5;
                goto out;
            }
        }
    }

    if (boops->monitor) {
        boops_monitor(boops);
    }

out:
    if (boops) {
        if (oops_data)
            boops_destroy_oops(boops, oops_data);

        boops_log_kernel_close(boops);
        free (boops);
        boops=NULL;
    }
    return err;
}
