#include "browser.h"

#include <QApplication>
#include <QGraphicsProxyWidget>
#include <QGraphicsScene>
#include <QLineEdit>
#include <QPushButton>
#include <QString>
#include <QMetaObject>

#include <QNetworkReply>
#ifndef QT_NO_OPENSSL
#include <QSslConfiguration>
#endif

#include <webplugin/webpluginfactoryprototype.h>

#include <cassert>

// FIXME: determine screen size
extern unsigned sDisplayWidth;
extern unsigned sDisplayHeight;

#ifdef ENABLE_AVE
extern "C"
{
    // Forward declare prototypes for the AVE libjsbindings-jspsdk.so library
    void loadAVEJavaScriptBindings(void* context);
    void unloadAVEJavaScriptBindings(void* context);
    void setTimingBaseTime(unsigned int baseTime);
    void TraceTimingSetBaseTime(unsigned int time);
    void setComcastSessionToken(const char* token);
}

void* GetClosedCaptionSurface()
{
    return NULL;
}

void ClearClosedCaptionSurface()
{
}

void HideClosedCaptionSurface()
{
}

void ShowClosedCaptionSurface()
{
}

void* CreateSurface()
{
     return NULL;
}

void DestroySurface(void* surface)
{
}

void GetSurfaceScale(double *pScaleX, double *pScaleY)
{
    *pScaleX = 1.0;
    *pScaleY = 1.0;
}

void SetSurfaceSize(void* surface, int width, int height)
{
}

void SetSurfacePos(void* surface, int x, int y)
{
}

extern "C"
{
unsigned  RISODoNotBackup(const unsigned char* path)
{
    return 0;
}
}

void* GetPlatformCallbackManager()
{
    return NULL;
}
#endif

HTTPNetworkAccessManager::HTTPNetworkAccessManager(QObject *parent, bool ignore_ssl_errors)
    : QNetworkAccessManager(parent)
    , m_ignoreSslErrors(ignore_ssl_errors)
{
#ifndef QT_NO_OPENSSL
    QObject::connect(this, SIGNAL(sslErrors(QNetworkReply*, QList<QSslError>)), SLOT(handleSslErrors(QNetworkReply*, QList<QSslError>)),Qt::DirectConnection);
#endif
}

#ifndef QT_NO_OPENSSL
void HTTPNetworkAccessManager::handleSslErrors(QNetworkReply* reply, QList<QSslError> sslErrors)
{
    if (m_ignoreSslErrors) {
        reply->ignoreSslErrors(sslErrors);
        printf("Ignoring SSL errors:\n");
    } else {
        printf("SSL errors:\n");
    }
    foreach (QSslError err, sslErrors) {
        printf("\t%d: '%s'\n", err.error(),qPrintable(err.errorString()));
    }
}
#endif

QNetworkReply * HTTPNetworkAccessManager::createRequest(Operation op, const QNetworkRequest &req, QIODevice *outgoingData)
{
//    fprintf(stderr, "Requesting url: %s \n", qPrintable(req.url().toString()));
//    fprintf(stderr, "---------------------------- HEADERS -------------------------------------------\n");
//    foreach(QByteArray head, req.rawHeaderList()) {
//         fprintf(stderr, "%s:%s\n", head.constData(), req.rawHeader(head).constData());
//    }
//    fprintf(stderr, "--------------------------------------------------------------------------------\n");
    QNetworkRequest request(req);
#ifndef QT_NO_OPENSSL
    QSslConfiguration config = req.sslConfiguration().isNull()
        ? QSslConfiguration::defaultConfiguration()
        : req.sslConfiguration();
    config.setPeerVerifyMode(QSslSocket::VerifyPeer);
    config.setProtocol(QSsl::AnyProtocol);
    request.setSslConfiguration(config);
#endif
    return QNetworkAccessManager::createRequest(op, request, outgoingData);
}

// Redirects JS console messages to stdout.
class JSLoggingWebPage: public QWebPage
{
public:
    explicit JSLoggingWebPage(QObject *parent = 0)
    :   QWebPage(parent)
    {
        // Initialize the page in the same way
        // as it's done in QGraphicsView::page().
        QPalette palette = QApplication::palette();
        palette.setBrush(QPalette::Base, QColor::fromRgbF(0, 0, 0, 0));
        setPalette(palette);
    }

    void setUserAgent(const QString& ua)
    {
        m_userAgent = ua;
    }

    QString userAgentForUrl(const QUrl&) const
    {
        if (m_userAgent.isEmpty())
            return "Mozilla/5.0 (Unknown; Linux i686) AppleWebKit/537.21 (KHTML, like Gecko) NativeXREReceiver"; // same as in receiver

        return m_userAgent;
    }

private:
    virtual void javaScriptConsoleMessage(const QString& message, int lineNumber, const QString& sourceID)
    {
        fprintf(stderr, "JSC [%s:%d]: %s\n",
            sourceID.toUtf8().constData(), lineNumber, message.toUtf8().constData());
    }

    QString m_userAgent;
};

static Browser *s_browser = NULL;

Browser::Browser(QGraphicsScene& scene, const QString& initial_url, bool is_video, bool enable_gui, bool enable_tiled_backing_store, const QString& user_agent, bool ignore_ssl_errors)
:   QGraphicsWebView(m_webFrame = new QGraphicsRectItem())
,   m_urlEditor(0), m_animDuration(500), m_animFPS(30), m_animLarge(true)
,   m_isVideo(is_video)
{
    s_browser = this;

    int page_w = sDisplayWidth;
    int page_h = sDisplayHeight;

    if (enable_gui)
    {
        const int WIDGET_PAD = 10;
        const int WIDGET_H = 30;
        const int LBL_W = 30;
        const int BTN_W = 100;
        const int PATHEDITOR_W = 400;

        // Create the "URL: " label.
        QGraphicsTextItem* lbl_item = scene.addText("URL:");
        lbl_item->setPos(WIDGET_PAD, WIDGET_PAD);

        QGraphicsProxyWidget* btn_proxy;
        QGraphicsProxyWidget* btn_last;
        QPushButton* btn;

        // Create the "Exit" button
        btn = new QPushButton("Exit");
        btn->setFocusPolicy (Qt::ClickFocus);
        QObject::connect(btn, SIGNAL(clicked()), this, SLOT(quitApp()));
        btn_proxy = scene.addWidget(btn);
        btn_proxy->setGeometry(QRectF(sDisplayWidth - BTN_W - WIDGET_PAD, WIDGET_PAD, BTN_W, WIDGET_H));
        btn_last = btn_proxy;

    #if 0
        // Create the "Scale" button (for testing animated scaling of video)
        btn = new QPushButton("Scale");
        QObject::connect(btn, SIGNAL(clicked()), this, SLOT(scaleVideo()));
        btn_proxy = scene.addWidget(btn);
        btn_proxy->setGeometry(QRectF(btn_last->pos().x() - BTN_W - WIDGET_PAD, WIDGET_PAD, BTN_W, WIDGET_H));
        btn_last = btn_proxy;
    #endif 
        // Create the "Go" button.
        btn = new QPushButton("Go");
        btn->setFocusPolicy (Qt::ClickFocus);
        QObject::connect(btn, SIGNAL(clicked()), this, SLOT(loadEnteredURL()));
        btn_proxy = scene.addWidget(btn);
        btn_proxy->setGeometry(QRectF(btn_last->pos().x() - BTN_W - WIDGET_PAD, WIDGET_PAD, BTN_W, WIDGET_H));
        btn_last = btn_proxy;

        // Create the "Make Screenshot" button.
        btn = new QPushButton("Screen it");
        btn->setFocusPolicy (Qt::ClickFocus);
        QObject::connect(btn, SIGNAL(clicked()), this, SLOT(makeScreenshot()));
        btn_proxy = scene.addWidget(btn);
        btn_proxy->setGeometry(QRectF(btn_last->pos().x() - BTN_W - WIDGET_PAD, WIDGET_PAD, BTN_W, WIDGET_H));
        btn_last = btn_proxy;

        // Create the URL input field.
        m_urlEditor = new QLineEdit;
        m_urlEditor->setText(initial_url);
        QGraphicsProxyWidget* line_edit_proxy = scene.addWidget(m_urlEditor);
        line_edit_proxy->setGeometry(QRectF(
                QPointF(lbl_item->pos().x() + LBL_W + WIDGET_PAD, WIDGET_PAD),
                QPointF(btn_last->pos().x() - WIDGET_PAD - PATHEDITOR_W, WIDGET_PAD + WIDGET_H)));
        //commented because of false triggering while loosing focus to "make screenshot" button
        //QObject::connect(m_urlEditor, SIGNAL(editingFinished()), this, SLOT(loadEnteredURL()));

        // Create the PATH for screenshot input field.
        m_pathEditor = new QLineEdit;
        m_pathEditor->setText("/home/screenshot.png");
        line_edit_proxy = scene.addWidget(m_pathEditor);
        line_edit_proxy->setGeometry(QRectF(
                QPointF(btn_last->pos().x() - WIDGET_PAD - PATHEDITOR_W, WIDGET_PAD),
                QPointF(btn_last->pos().x() - WIDGET_PAD, WIDGET_PAD + WIDGET_H)));


        // Create rectangle wrapping the web view.
        page_w = sDisplayWidth - 2*WIDGET_PAD;
        page_h = sDisplayHeight - 3*WIDGET_PAD - WIDGET_H;
        m_webFrame->setRect(WIDGET_PAD, 2*WIDGET_PAD + WIDGET_H, page_w, page_h);
    }
    else
    {
        // Create rectangle wrapping the web view.
        m_webFrame->setRect(0, 0, page_w, page_h);
    }

    m_webFrame->setBrush(QBrush(Qt::gray));
    scene.addItem(m_webFrame);

    // Set up the web view.
    m_largeRect = QRectF(m_webFrame->rect().left(), m_webFrame->rect().top(), page_w, page_h);
    m_smallRect = QRectF(m_webFrame->rect().left() + page_w - page_w/5, m_webFrame->rect().top(), page_w/5, page_h/5);

    JSLoggingWebPage* xpage = new JSLoggingWebPage(this);
    xpage->setUserAgent(user_agent);

    setPage(xpage);

    HTTPNetworkAccessManager *manager;
    manager = new HTTPNetworkAccessManager(page(), ignore_ssl_errors);
    page()->setNetworkAccessManager(manager);

    // page()->setProperty("_q_RepaintThrottlingPreset", "Heavy");

    setGeometry(m_largeRect);
    setMaximumSize(QSizeF(page_w, page_h));
    setMinimumSize(QSizeF(page_w, page_h));
    settings()->setAttribute(QWebSettings::PluginsEnabled,true);
    settings()->setAttribute(QWebSettings::JavascriptEnabled,true);
    settings()->setAttribute(QWebSettings::LinksIncludedInFocusChain,true);
    settings()->setAttribute(QWebSettings::LocalStorageEnabled, true);
    settings()->setAttribute(QWebSettings::AcceleratedCompositingEnabled, true);
    settings()->setAttribute(QWebSettings::TiledBackingStoreEnabled, enable_tiled_backing_store);
    settings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, true);
    printf("Tiled backing store %s\n", settings()->testAttribute(QWebSettings::TiledBackingStoreEnabled) ? "enabled" : "disabled");

    page()->mainFrame()->addToJavaScriptWindowObject("XREReceiver", this);
    QObject::connect(page()->mainFrame(),
                     SIGNAL(javaScriptWindowObjectCleared()),
                     SLOT(javaScriptWindowObjectCleared()));
    QObject::connect(this,SIGNAL(loadFinished(bool)),this,SLOT(onDocumentLoaded(bool)));
    page()->setProperty("_q_webInspectorServerPort", 9222);

    page()->mainFrame()->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);
    page()->mainFrame()->setScrollBarPolicy(Qt::Vertical,   Qt::ScrollBarAlwaysOff);

    //WebPlugins support
    page()->setPluginFactory(new WebPluginFactoryPrototype(this));


    loadURL(initial_url);
    setFocusPolicy (Qt::StrongFocus);
    setFocus();

    QTimer::singleShot(60 * 1000, this, SIGNAL(documentLoaded()));
    // traceWebKit();
}

Browser::~Browser()
{
    QObject::disconnect(page()->mainFrame(),
                        SIGNAL(javaScriptWindowObjectCleared()),
                        this,
                        SLOT(javaScriptWindowObjectCleared()));
#ifdef ENABLE_AVE
    void* context = page()->mainFrame()->jsGlobalContext();
    unloadAVEJavaScriptBindings(context);
#endif
    s_browser = NULL;
    setHtml("");
}

void Browser::javaScriptWindowObjectCleared()
{
    fprintf(stderr, "javaScriptWindowObjectCleared url=%s\n", qPrintable(page()->mainFrame()->url().toString()));
#ifdef ENABLE_AVE
    void* context = page()->mainFrame()->jsGlobalContext();
    loadAVEJavaScriptBindings(context);
#endif
}

void Browser::loadEnteredURL()
{
    assert(m_urlEditor);
    loadURL(m_urlEditor->text());
}

void Browser::scaleVideo()
{
    startAnimation();
}

void Browser::traceWebKit()
{
    QWebSettings::dumpWebCoreStatistics();
}

void Browser::sendCallMethod(QString methodName, QVariantList params)
{
    emit callMethod(methodName, params);
}

void Browser::onDocumentLoaded(bool success)
{
    QMetaObject::invokeMethod(this, "documentLoaded", Qt::QueuedConnection);
}

void Browser::quitApp()
{
    if (qGuiApp->thread() != this->thread()) {
        printf("***Error: Cannot handle 'quitApp' on non main thread\n");
        return;
    }
    setHtml("");
    fprintf(stderr, "Exit in 5 seconds\n");
    QTimer::singleShot(5 * 1000, qGuiApp, SLOT(quit()));
}

void Browser::onAnimate()
{
    m_animLatchTime = m_animTime.elapsed();

    qreal x, y, w, h;

    float percentComplete = (float)m_animLatchTime/m_animDuration;

    if(percentComplete < 1)
    {
        x = interpolate(m_animStartRect.x(), m_animEndRect.x(), percentComplete);
        y = interpolate(m_animStartRect.y(), m_animEndRect.y(), percentComplete);
        w = interpolate(m_animStartRect.width(), m_animEndRect.width(), percentComplete);
        h = interpolate(m_animStartRect.height(), m_animEndRect.height(), percentComplete);
    }
    else
    {
        stopAnimation();
        x = m_animEndRect.x();
        y = m_animEndRect.y();
        w = m_animEndRect.width();
        h = m_animEndRect.height();
    }

    setPos(x,y);
    setMaximumSize(QSizeF(w,h));
    setMinimumSize(QSizeF(w,h));
    resize(w,h);
}

void Browser::makeScreenshot()
{
    QWebFrame *mainFrame = page()->mainFrame();
    QPainter painter;
    QSize contentSize = mainFrame->contentsSize();

    page()->setViewportSize( mainFrame->contentsSize() );

    QImage image(contentSize, QImage::Format_ARGB32);
    painter.begin(&image);
    mainFrame->render(&painter);
    painter.end();

    QString path = m_screenshotPath;
    if(path.isEmpty())
        if(m_pathEditor)
            path = m_pathEditor->text();
    if(path.isEmpty())
        path = "/tmp/screenshot.png";

    image.save(path);
}

// static
QString Browser::constructVideoHtml(const QString& video_url)
{
    return QString().sprintf(
        "<body><table border=\"1\"><tr><td>"
        "<video src=\"%s\" autoplay />"
        "</td></tr></table></body>",
        video_url.toUtf8().constData());
}
void Browser::startAnimation()
{
    m_animLarge = !m_animLarge;

    if(m_animLarge)
    {
        m_animStartRect = m_smallRect;
        m_animEndRect = m_largeRect;
    }
    else
    {
        m_animStartRect = m_largeRect;
        m_animEndRect = m_smallRect;
    }

    QObject::connect(&m_animTimer,SIGNAL(timeout()),SLOT(onAnimate()));
    m_animTime.start();
    m_animLatchTime = 0;
    m_animTimer.start( (int)( 1000.0 / m_animFPS) );

}
void Browser::stopAnimation()
{
    m_animTimer.stop();
}

float Browser::interpolate(float start, float end, float percentComplete)
{
    if(percentComplete < 0)
        return start;
    if(percentComplete > 1)
        return end;
    return start + ((end-start)*percentComplete);
}

void Browser::loadURL(const QString& url)
{
    printf("Browser::loadURL: %s\n", url.toUtf8().constData());
    if (m_isVideo)
    {
        setHtml(constructVideoHtml(url));
        m_isVideo = false;
    }
    else
        load(QUrl(url));
}

void Browser::keyPressEvent(QKeyEvent *event)
{
    int key = event->key();
    switch(key){
        case Qt::Key_F10:
            makeScreenshot();
            break;
    }

    QGraphicsWebView::keyPressEvent(event);
}

void Browser::setScreenshotPath(const QString &value)
{
    m_screenshotPath = value;
}

void Browser::clearWebCoreCaches() {
    QWebSettings::clearMemoryCaches();
}

void Browser::dumpWebCoreStatistics() {
    QWebSettings::dumpWebCoreStatistics();
}

void Browser::emitClearWebCoreCaches() {
    QMetaObject::invokeMethod(s_browser, "clearWebCoreCaches", Qt::QueuedConnection);
}

void Browser::emitDumpWebCoreStatistics() {
    QMetaObject::invokeMethod(s_browser, "dumpWebCoreStatistics", Qt::QueuedConnection);
}

void Browser::emitQuitApp() {
    QMetaObject::invokeMethod(s_browser, "quitApp", Qt::QueuedConnection);
}
