/****************************************************************************
**
** Copyright (C) 2013 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of the plugins of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Digia.  For licensing terms and
** conditions see http://qt.digia.com/licensing.  For further information
** use the contact form at http://qt.digia.com/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Digia gives you certain additional
** rights.  These rights are described in the Digia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qeglfswindow.h"
#include "qeglfshooks.h"
#include <qpa/qwindowsysteminterface.h>

#include <QtPlatformSupport/private/qeglconvenience_p.h>

#include <QtDebug>

//COMCAST MODIFICATION BEGIN
struct SingleSurfaceData {
    SingleSurfaceData()
        : eglSurface(EGL_NO_SURFACE)
        , eglWindow(0) {
    }
    EGLSurface eglSurface;
    EGLNativeWindowType eglWindow;
    QSurfaceFormat format;
    QSet<QEglFSWindow*> refs;
};
Q_GLOBAL_STATIC(SingleSurfaceData, singleSurfaceData);
//COMCAST MODIFICATION END

QT_BEGIN_NAMESPACE

QEglFSWindow::QEglFSWindow(QWindow *w)
    : QPlatformWindow(w)
    , m_surface(0)
    , m_window(0)
{
    static int serialNo = 0;
    m_winid  = ++serialNo;
#ifdef QEGL_EXTRA_DEBUG
    qWarning("QEglWindow %p: %p 0x%x\n", this, w, uint(m_winid));
#endif
}

QEglFSWindow::~QEglFSWindow()
{
    destroy();
}

void QEglFSWindow::create()
{
    if (m_window)
        return;

    setWindowState(Qt::WindowFullScreen);

    if (window()->type() == Qt::Desktop) {
        QRect rect(QPoint(), QEglFSHooks::hooks()->screenSize());
        QPlatformWindow::setGeometry(rect);
        QWindowSystemInterface::handleGeometryChange(window(), rect);
        return;
    }

    EGLDisplay display = (static_cast<QEglFSScreen *>(window()->screen()->handle()))->display();
    QSurfaceFormat platformFormat = QEglFSHooks::hooks()->surfaceFormatFor(window()->requestedFormat());
    m_config = QEglFSIntegration::chooseConfig(display, platformFormat);
    m_format = q_glFormatFromConfig(display, m_config);
    resetSurface();
}

void QEglFSWindow::invalidateSurface()
{
    // Native surface has been deleted behind our backs
    m_window = 0;
    if (m_surface != 0) {
        EGLDisplay display = (static_cast<QEglFSScreen *>(window()->screen()->handle()))->display();
        eglDestroySurface(display, m_surface);
        m_surface = 0;
    }
}

void QEglFSWindow::resetSurface()
{
//COMCAST MODIFICATION BEGIN
    static const bool kSupportsMultipleWindows =
        QEglFSHooks::hooks()->hasCapability(QPlatformIntegration::Capability::MultipleWindows);
    if (!kSupportsMultipleWindows && singleSurfaceData()->eglSurface != EGL_NO_SURFACE) {
        m_surface = singleSurfaceData()->eglSurface;
        m_window = singleSurfaceData()->eglWindow;
        singleSurfaceData()->refs.insert(this);
#ifdef QEGL_EXTRA_DEBUG
        qWarning("Surface recreate request, re-using %x\n", m_surface);
#endif
        if (m_format != singleSurfaceData()->format) {
          qWarning() << "*** Error: format doesn't match: "
                     << m_format
                     << " vs. "
                     << singleSurfaceData()->format;
        }
        return;
    }
//COMCAST MODIFICATION END

    EGLDisplay display = static_cast<QEglFSScreen *>(screen())->display();

    m_window = QEglFSHooks::hooks()->createNativeWindow(QEglFSHooks::hooks()->screenSize(), m_format);
    m_surface = eglCreateWindowSurface(display, m_config, m_window, NULL);
    if (m_surface == EGL_NO_SURFACE) {
        EGLint error = eglGetError();
        eglTerminate(display);
        qFatal("EGL Error : Could not create the egl surface: error = 0x%x\n", error);
    }

//COMCAST MODIFICATION BEGIN
    if (!kSupportsMultipleWindows) {
        Q_ASSERT(singleSurfaceData()->refs.isEmpty());
        singleSurfaceData()->eglSurface = m_surface;
        singleSurfaceData()->eglWindow = m_window;
        singleSurfaceData()->format = m_format;
        singleSurfaceData()->refs.insert(this);
    }
//COMCAST MODIFICATION END
}

void QEglFSWindow::destroy()
{
//COMCAST MODIFICATION BEGIN
    singleSurfaceData()->refs.remove(this);
    if (!singleSurfaceData()->refs.isEmpty()) {
       return;
    }
    singleSurfaceData()->eglSurface = EGL_NO_SURFACE;
    singleSurfaceData()->eglWindow = 0;
//COMCAST MODIFICATION END

    if (m_surface) {
        EGLDisplay display = static_cast<QEglFSScreen *>(screen())->display();
        eglDestroySurface(display, m_surface);
        m_surface = 0;
    }

    if (m_window) {
        QEglFSHooks::hooks()->destroyNativeWindow(m_window);
        m_window = 0;
    }
}

void QEglFSWindow::setGeometry(const QRect &)
{
    // We only support full-screen windows
    QRect rect(screen()->availableGeometry());
    QPlatformWindow::setGeometry(rect);
    QWindowSystemInterface::handleGeometryChange(window(), rect);
    QWindowSystemInterface::handleExposeEvent(window(), QRegion(rect));
}

void QEglFSWindow::setWindowState(Qt::WindowState)
{
    setGeometry(QRect());
}

WId QEglFSWindow::winId() const
{
    return m_winid;
}

QSurfaceFormat QEglFSWindow::format() const
{
    return m_format;
}

//COMCAST MODIFICATION BEGIN
void QEglFSWindow::requestActivateWindow()
{
    // keep first created window focused
    if (m_winid == 1)
        QWindowSystemInterface::handleWindowActivated(window());
}

EGLSurface QEglFSWindow::surface() const
{
    return m_surface;
}

void QEglFSWindow::recreateSurface()
{
    singleSurfaceData()->eglSurface = EGL_NO_SURFACE;

    // recreate the surface
    EGLDisplay display = static_cast<QEglFSScreen *>(screen())->display();
    if (m_surface) {
        int ok = eglDestroySurface(display, m_surface);
        if ( !ok )  {
            qWarning("QEglFSWindow::recreateSurface: eglDestroySurface failed,"
                     " eglSurface: %p ok: %d", m_surface, ok );
        }
        m_surface = EGL_NO_SURFACE;
    }
    m_surface = eglCreateWindowSurface(display, m_config, m_window, NULL);
    if (m_surface == EGL_NO_SURFACE) {
        EGLint error = eglGetError();
        eglTerminate(display);
        qFatal("EGL Error : Could not create the egl surface:"
               " error = 0x%x\n", error);
    }

    // update global data and dependent windows
    singleSurfaceData()->eglSurface = m_surface;
    foreach(QEglFSWindow* win, singleSurfaceData()->refs) {
        win->m_surface = singleSurfaceData()->eglSurface;
    }
}
//COMCAST MODIFICATION END

QT_END_NAMESPACE
