/******************************************************************************
 *    (c)2010-2013 Broadcom Corporation
 *
 * This program is the proprietary software of Broadcom Corporation and/or its licensors,
 * and may only be used, duplicated, modified or distributed pursuant to the terms and
 * conditions of a separate, written license agreement executed between you and Broadcom
 * (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 * no license (express or implied), right to use, or waiver of any kind with respect to the
 * Software, and Broadcom expressly reserves all rights in and to the Software and all
 * intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 * HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 * NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 * Except as expressly set forth in the Authorized License,
 *
 * 1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 * secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 * and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 * 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 * AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 * WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 * THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 * OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 * LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 * OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 * USE OR PERFORMANCE OF THE SOFTWARE.
 *
 * 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 * LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 * EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 * USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 * ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 * LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 * ANY LIMITED REMEDY.
 *
 *****************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <fcntl.h>

#include "bcmnexus.h"
#include "directfb.h"
#include "idirectfb.h"

#include "display/idirectfbsurface.h"
#include "media/idirectfbvideoprovider.h"
#include "media/idirectfbdatabuffer.h"

#include "core/system.h"

#include "idirectfb_media_player_priv.h"


D_DEBUG_DOMAIN( bcmnexusVideoProvider,        "bcmNexus/VideoProvider",        "Broadcom NEXUS Video Provider module" );


static DFBResult
Probe( IDirectFBVideoProvider_ProbeContext *ctx );

static DFBResult
Construct( IDirectFBVideoProvider *thiz,
           IDirectFBDataBuffer    *buffer );


#include <direct/interface_implementation.h>

DIRECT_INTERFACE_IMPLEMENTATION( IDirectFBVideoProvider, mediaplayer )


/*
 * private data struct for video provider
 */

typedef struct {
    int                            count;

    pthread_mutex_t                lock;

    char                          *file_url;

    bool                           nxClient;
    NxClient_AllocResults          allocResults;
    unsigned                       connectId;
    bool                           setup_done;
    media_player_t                 player;
    int                            rate; /* trick speed */
    bool                           stopped;
    media_player_start_settings    start_settings;
    NEXUS_SurfaceClientHandle      surfaceClient;
    NEXUS_SurfaceClientHandle      videoSurfaceClient;

    media_player_status            media_status;

    DFBVideoProviderStatus         status;
    DFBVideoProviderPlaybackFlags  flags;

    IDirectFBEventBuffer          *events;
    NEXUS_Rect                     destinationRect;
    pthread_mutex_t                setDestinationLock;
    pthread_cond_t                 setDestinationSem;
    DirectThread                  *stream_info_thread_id;
    bool                           cancel_thread;
} IDirectFBVideoProvider_mediaplayer_data;


/*
 * Function prototypes for private functions
 */


static DFBResult
play_setup( IDirectFBVideoProvider_mediaplayer_data *data );

static void
send_event( IDirectFBVideoProvider_mediaplayer_data *data,
            DFBVideoProviderEventType         type );

static void complete(void *context);

static void*
stream_monitor_thread( DirectThread *self, void *arg );

/*
 * Static interface functions
 */

static void
IDirectFBVideoProvider_mediaplayer_Destruct( IDirectFBVideoProvider *thiz )
{
    IDirectFBVideoProvider_mediaplayer_data *data = thiz->priv;

    /*Stop the stream monitor thread which is dependent on player.*/
    data->cancel_thread = true;
    /*Wake up the stream_monitor_thread() to detect and initiate thread shutdown sequence */
    pthread_mutex_lock( &data->setDestinationLock );
    pthread_cond_signal( &data->setDestinationSem );
    pthread_mutex_unlock( &data->setDestinationLock );

    direct_thread_join( data->stream_info_thread_id);
    direct_thread_destroy( data->stream_info_thread_id);
    data->stream_info_thread_id = NULL;

    if ( data->player ) {
        if ( data->status == DVSTATE_PLAY ) {
            media_player_stop_priv( data->player );
        }
        media_player_destroy_priv( data->player );
        data->player = NULL;

        if ( data->videoSurfaceClient ) {
            NEXUS_SurfaceClient_ReleaseVideoWindow( data->videoSurfaceClient );
            data->videoSurfaceClient = NULL;
        }
        if ( data->surfaceClient ) {
            NEXUS_SurfaceClient_Release( data->surfaceClient );
            data->surfaceClient = NULL;
        }

        if ( data->nxClient && data->setup_done ) {
            NxClient_Disconnect(data->connectId);
            data->connectId = 0;
            NxClient_Free(&data->allocResults);
        }
    }

    if ( data->events )
        data->events->Release( data->events );

    D_FREE( data->file_url );

    /*Destroy  locks used in setDestination() public API.*/
    pthread_mutex_destroy( &data->setDestinationLock );
    pthread_cond_destroy( &data->setDestinationSem );

    pthread_mutex_destroy( &data->lock );

    DIRECT_DEALLOCATE_INTERFACE( thiz );
}

static DirectResult
IDirectFBVideoProvider_mediaplayer_AddRef( IDirectFBVideoProvider *thiz )
{
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    data->count++;

    return(DR_OK);
}

static DirectResult
IDirectFBVideoProvider_mediaplayer_Release( IDirectFBVideoProvider *thiz )
{
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( --data->count == 0 )
        IDirectFBVideoProvider_mediaplayer_Destruct( thiz );

    return(DR_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_GetCapabilities( IDirectFBVideoProvider       *thiz,
                                                    DFBVideoProviderCapabilities *caps )
{
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( !caps )
        return(DFB_INVARG);

    *caps = DVCAPS_BASIC       | DVCAPS_SCALE    | DVCAPS_SPEED      |
            DVCAPS_INTERACTIVE | DVCAPS_EVENT    | DVCAPS_SEEK;

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_GetSurfaceDescription( IDirectFBVideoProvider *thiz,
                                                          DFBSurfaceDescription  *desc )
{
    DFBResult result = DFB_OK;

    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( !desc )
        return(DFB_INVARG);

    pthread_mutex_lock( &data->lock );
    if ( data->status != DVSTATE_PLAY ) {
        result = play_setup( data );
    }

    if ( (result != DFB_OK) || !data->media_status.valid ) {
        pthread_mutex_unlock( &data->lock );
        return(DFB_FAILURE);
    }

    if ( data->media_status.has_video ) {
        desc->flags       = ( DSDESC_WIDTH | DSDESC_HEIGHT );
        desc->width       = data->media_status.width;
        desc->height      = data->media_status.height;

        if ( desc->width < 32 || desc->height < 32 ) {
            desc->width  = 320;
            desc->height = 240;
        }
    }
    pthread_mutex_unlock( &data->lock );
    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_PlayTo( IDirectFBVideoProvider *thiz,
                                           IDirectFBSurface       *destination,
                                           const DFBRectangle     *dest_rect,
                                           DVFrameCallback         callback,
                                           void                   *ctx )
{
    DFBResult result = DFB_OK;

    BSTD_UNUSED(callback);
    BSTD_UNUSED(ctx);

    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( !destination )
        return(DFB_INVARG);

    if ( !destination->priv )
        return(DFB_DESTROYED);

    pthread_mutex_lock( &data->lock );

    if ( data->status != DVSTATE_PLAY ) {
        int rc;
        NEXUS_Rect rect;
        IDirectFBSurface_data *dst_data;
        CoreSurface *dst_surface;

        dst_data = (IDirectFBSurface_data*) destination->priv;
        if ( !dst_data ) {
            D_ERROR("bcmNexus/VideoProvider: Invalid destination data !\n");
            pthread_mutex_unlock( &data->lock );
            return(DFB_FAILURE);
        }

        dst_surface = dst_data->surface;
        if ( !dst_surface ) {
            D_ERROR("bcmNexus/VideoProvider: Invalid destination surface !\n");
            pthread_mutex_unlock( &data->lock );
            return(DFB_FAILURE);
        }
        destination->Clear( destination, 0, 0, 0, 0 );

        if ( dest_rect ) {
            rect.x = dest_rect->x;
            rect.y = dest_rect->y;
            rect.width = dest_rect->w;
            rect.height = dest_rect->h;
        }
        else {

            rect.x = 0;
            rect.y = 0;
            rect.width = dst_surface->config.size.w;
            rect.height = dst_surface->config.size.h;
        }

        if ( !data->setup_done ) {
            result = play_setup( data );
            if ( result != DFB_OK ) {
                pthread_mutex_unlock( &data->lock );
                return(DFB_FAILURE);
            }
        }

        if ( !data->media_status.has_video && !data->media_status.has_audio ) {
            /* Probe has not been called or unsupported media */
            D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                     "unsupported media\n" );
            pthread_mutex_unlock( &data->lock );
            return(DFB_UNSUPPORTED);
        }

        if ( data->nxClient ) {
            NxClient_ConnectSettings connectSettings;
            NEXUS_SurfaceComposition comp;

            data->surfaceClient = NEXUS_SurfaceClient_Acquire(data->allocResults.surfaceClient[0].id);
            if ( data->surfaceClient ) {
                NxClient_GetSurfaceClientComposition(data->allocResults.surfaceClient[0].id, &comp);
                comp.position = rect;
                NxClient_SetSurfaceClientComposition(data->allocResults.surfaceClient[0].id, &comp);
                data->videoSurfaceClient = NEXUS_SurfaceClient_AcquireVideoWindow(data->surfaceClient, 0);
            }

            NxClient_GetDefaultConnectSettings(&connectSettings);
            connectSettings.simpleVideoDecoder[0].id = data->allocResults.simpleVideoDecoder[0].id;
            connectSettings.simpleAudioDecoder.id = data->allocResults.simpleAudioDecoder.id;
            connectSettings.simpleVideoDecoder[0].surfaceClientId = data->allocResults.surfaceClient[0].id;
            connectSettings.simpleVideoDecoder[0].windowId = 0;
            rc = NxClient_Connect(&connectSettings, &data->connectId);
            if ( rc ) {
                D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                         "NxClient_Connect failed\n" );
                pthread_mutex_unlock( &data->lock );
                return(DFB_FAILURE);
            }
        }

        rc = media_player_start_priv( data->player );
        if ( rc != NEXUS_SUCCESS ) {
            D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                     "start failed\n" );
            pthread_mutex_unlock( &data->lock );
            return(DFB_FAILURE);
        }

        data->status = DVSTATE_PLAY;

        send_event( data, DVPET_STARTED );
    }

    pthread_mutex_init( &data->setDestinationLock, NULL);
    pthread_cond_init( &data->setDestinationSem, NULL);

    if (!data->stream_info_thread_id) {
         data->stream_info_thread_id = direct_thread_create( DTT_DEFAULT, stream_monitor_thread,
                                                   (void*)thiz, "Stream Monitor" );
    }

    pthread_mutex_unlock( &data->lock );
    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_Stop( IDirectFBVideoProvider *thiz )
{
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    pthread_mutex_lock( &data->lock );

    if ( data->status == DVSTATE_STOP ) {
        pthread_mutex_unlock( &data->lock );
        return(DFB_OK);
    }

    if ( data->status == DVSTATE_PLAY ) {
        media_player_stop_priv( data->player );
    }

    data->status = DVSTATE_STOP;

    send_event( data, DVPET_STOPPED );

    pthread_mutex_unlock( &data->lock );

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_GetStatus( IDirectFBVideoProvider *thiz,
                                              DFBVideoProviderStatus *status )
{
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( !status )
        return(DFB_INVARG);

    *status = data->status;

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_SeekTo( IDirectFBVideoProvider *thiz,
                                           double                  seconds )
{
    int offset;
    int rc;
    unsigned length;
    NEXUS_PlaybackStatus playback_status;

    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( seconds < 0.0 )
        return(DFB_INVARG);

    pthread_mutex_lock( &data->lock );

    rc = media_player_get_playback_status_priv( data->player, &playback_status );
    if ( rc != NEXUS_SUCCESS ) {
        D_WARN( "DirectFB/VideoProvider_mediaplayer: "
                "status failed\n" );
        length = 0;
    }
    else {
        length = playback_status.last;
    }

    offset = (int) (seconds * 1000.0);
    if ( length > 0 && offset > (int)length ) {
        pthread_mutex_unlock( &data->lock );
        return(DFB_OK);
    }

    if ( data->status == DVSTATE_PLAY ) {
        if ( media_player_seek_priv( data->player, offset) != NEXUS_SUCCESS ) {
            D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                     "seek failed\n" );
            pthread_mutex_unlock( &data->lock );
            return(DFB_FAILURE);
        }
    }

    if ( data->rate != NEXUS_NORMAL_DECODE_RATE ) {
        if ( media_player_trick_priv(data->player, data->rate) != NEXUS_SUCCESS ) {
            D_WARN( "DirectFB/VideoProvider_mediaplayer: "
                    "set speed failed\n" );
            return(DFB_FAILURE);
        }
    }

    pthread_mutex_unlock( &data->lock );

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_GetPos( IDirectFBVideoProvider *thiz,
                                           double                 *seconds )
{
    int pos = 0;
    NEXUS_PlaybackStatus playback_status;

    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( !seconds )
        return(DFB_INVARG);

    if ( data->status == DVSTATE_PLAY ) {
        if ( media_player_get_playback_status_priv( data->player, &playback_status) != NEXUS_SUCCESS ) {
            D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                     "get status failed\n" );
            return(DFB_FAILURE);
        }
        pos = playback_status.position;
    }

    *seconds = (double)pos / 1000.0;

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_GetLength( IDirectFBVideoProvider *thiz,
                                              double                 *seconds )
{
    int rc;
    NEXUS_PlaybackStatus playback_status;
    unsigned length;

    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( !seconds )
        return(DFB_INVARG);

    rc = media_player_get_playback_status_priv( data->player, &playback_status );
    if ( rc != NEXUS_SUCCESS ) {
        D_WARN( "DirectFB/VideoProvider_mediaplayer: "
                "get status failed\n" );
        length = 0;
    }
    else {
        length = playback_status.last;
    }

    *seconds = (double)length / 1000.0;

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_SetPlaybackFlags( IDirectFBVideoProvider        *thiz,
                                                     DFBVideoProviderPlaybackFlags  flags )
{
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( flags & ~DVPLAY_LOOPING )
        return(DFB_UNSUPPORTED);

    data->flags = flags;

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_SetSpeed( IDirectFBVideoProvider *thiz,
                                             double                  multiplier )
{
    int speed = (int)(multiplier * (double)NEXUS_NORMAL_DECODE_RATE);
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( data->rate != speed ) {
        if ( media_player_trick_priv(data->player, speed) != NEXUS_SUCCESS ) {
            D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                     "set speed failed\n" );
            return(DFB_FAILURE);
        }

        data->rate = speed;
    }

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_GetSpeed( IDirectFBVideoProvider *thiz,
                                             double                 *ret_multiplier )
{
    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    if ( !ret_multiplier )
        return(DFB_INVARG);

    *ret_multiplier = (double)data->rate / (double)NEXUS_NORMAL_DECODE_RATE;

    return(DFB_OK);
}

static DFBResult
IDirectFBVideoProvider_mediaplayer_SetDestination( IDirectFBVideoProvider *thiz,
                                                   IDirectFBSurface       *dest,
                                                   const DFBRectangle     *dest_rect )
{
    BSTD_UNUSED(dest);

    if(!dest_rect)
    {
        return(DFB_INVARG);
    }

    D_DEBUG_AT( bcmnexusVideoProvider, "%s: %d,%d - %d,%d\n", __FUNCTION__, dest_rect->x, dest_rect->y, dest_rect->w, dest_rect->h );

    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    pthread_mutex_lock( &data->lock );

    data->destinationRect.x = dest_rect->x;
    data->destinationRect.y = dest_rect->y;
    data->destinationRect.width = dest_rect->w;
    data->destinationRect.height = dest_rect->h;

    pthread_mutex_lock( &data->setDestinationLock );
    pthread_cond_signal( &data->setDestinationSem );
    pthread_mutex_unlock( &data->setDestinationLock );

    pthread_mutex_unlock( &data->lock );

    return(DFB_OK);
}

static DFBResult IDirectFBVideoProvider_mediaplayer_GetVideowindowPosition( const NEXUS_CalculateVideoWindowPositionSettings *pPosition,
                                                                            NEXUS_VideoWindowSettings *pWindowSettings)
{
    unsigned clip;
    NEXUS_Rect viewport = pPosition->viewport; /* may need to modify for manual a/r correction */
    unsigned verticalClipping = pPosition->verticalClipping; /* may need to modify for manual a/r correction */
    unsigned horizontalClipping;/* may need to modify for manual a/r correction */
    unsigned displayAr, sourceAr; /* aspect ratio in 1/100th's units (e.g. 4:3 = 4/3 = 133 */
    unsigned aspectNumerator, aspectDenominator;

    if (!pPosition->displayWidth || !pPosition->displayHeight) {
        D_ERROR("bcmNexus/VideoProvider: Invalid Parameters displayWidth displayHeight !\n");
        return(DFB_INVARG);
    }

    horizontalClipping = pPosition->horizontalClipping;

    /* store AR as width/height in 1/100ths.
    This has the advantage of already being in units of a horizontal percentage for clipping-based zoom. */
    switch (pPosition->displayAspectRatio) {
    case NEXUS_DisplayAspectRatio_eAuto:
        displayAr = pPosition->displayWidth * 100 / pPosition->displayHeight;
        if (!displayAr) {
            D_ERROR("bcmNexus/VideoProvider: Invalid aspect ratio displayAr: %d !\n", displayAr);
            return(DFB_FAILURE);
        }
        break;
    case NEXUS_DisplayAspectRatio_e4x3:
        displayAr = 4 * 100 / 3;
        break;
    case NEXUS_DisplayAspectRatio_e16x9:
        displayAr = 16 * 100 / 9;
        break;
    default:
        D_ERROR("bcmNexus/VideoProvider: Unsupported displayAspectRatio : %d \n", pPosition->displayAspectRatio);
        return(DFB_UNSUPPORTED);
    }

    switch (pPosition->sourceAspectRatio) {
    case NEXUS_AspectRatio_e4x3:
        sourceAr = 400 / 3;
        aspectNumerator = 4;
        aspectDenominator = 3;
        break;
    case NEXUS_AspectRatio_e16x9:
        sourceAr = 1600 / 9;
        aspectNumerator = 16;
        aspectDenominator = 9;
        break;
    case NEXUS_AspectRatio_eSquarePixel:
        /* square pixel is the same as SAR 1:1 */
        if (pPosition->sourceHeight) {
            sourceAr = pPosition->sourceWidth * 100 / pPosition->sourceHeight;
            aspectNumerator = pPosition->sourceWidth;
            aspectDenominator = pPosition->sourceHeight;
        }
        else {
            D_ERROR("bcmNexus/VideoProvider: Invalid  sourceHeight : %d \n", pPosition->sourceHeight);
            return(DFB_FAILURE);
        }
        break;
    case NEXUS_AspectRatio_e221x1:
        sourceAr = 221;
        aspectNumerator = 221;
        aspectDenominator = 100;
        break;
    case NEXUS_AspectRatio_e15x9:
        sourceAr = 1500 / 9;
        aspectNumerator = 15;
        aspectDenominator = 9;
        break;
    case NEXUS_AspectRatio_eSar:
        if (pPosition->sampleAspectRatio.x && pPosition->sampleAspectRatio.y && pPosition->sourceWidth && pPosition->sourceHeight) {
            sourceAr = pPosition->sourceWidth * 100 * pPosition->sampleAspectRatio.x / pPosition->sampleAspectRatio.y / pPosition->sourceHeight;
            aspectNumerator = pPosition->sourceWidth * pPosition->sampleAspectRatio.x;
            aspectDenominator = pPosition->sourceHeight * pPosition->sampleAspectRatio.y;
        }
        else {
            D_ERROR("bcmNexus/VideoProvider: Invalid  source aspect ratio parameter(s)\n");
            return(DFB_FAILURE);
        }
        break;
    default:
        sourceAr = 0;
        aspectNumerator = 0;
        aspectDenominator = 1;
        break;
    }

    /**
    Manual aspect ratio correction involves either changes in the viewport or in clipping.
    **/
    switch (pPosition->manualAspectRatioCorrection) {
    case NEXUS_VideoWindowContentMode_eBox:
        if (!sourceAr) {
            D_ERROR("bcmNexus/VideoProvider: Invalid  sourceAr : %d at:%d\n", sourceAr, __LINE__);
            return(DFB_FAILURE);
        }

        /* if display and source ARs are equal, no letter/pillar boxing required */
        if(displayAr != sourceAr)
        {
            if (displayAr > sourceAr) {
                /* pillar box */
                unsigned w = viewport.height * aspectNumerator / aspectDenominator;
                viewport.x += (viewport.width - w) / 2;
                viewport.width = w;
            }
            else {
                /* letter box */
                unsigned h = viewport.width * aspectDenominator / aspectNumerator;
                viewport.y += (viewport.height - h) / 2;
                viewport.height = h;
            }
        }
        pWindowSettings->contentMode = NEXUS_VideoWindowContentMode_eBox;
        break;
    case NEXUS_VideoWindowContentMode_eZoom:
        if (!sourceAr) {
            D_ERROR("bcmNexus/VideoProvider: Invalid  sourceAr : %d at:%d\n", sourceAr, __LINE__);
            return(DFB_FAILURE);
        }

        /* if display and source ARs are equal, no adjustment required */
        if(displayAr != sourceAr)
        {
            if (displayAr > sourceAr) {
                /* vertical clipping - convert to height/width, then do the 1/100ths based math */
                verticalClipping += (100 * 100 / sourceAr) - (100 * 100 / displayAr);
            }
            else {
                /* horizontal clipping - units of sourceAr & displayAr are ready for direct math */
                horizontalClipping += sourceAr - displayAr;
            }
        }
        pWindowSettings->contentMode = NEXUS_VideoWindowContentMode_eFull;
        break;
    case NEXUS_VideoWindowContentMode_eFull:
        pWindowSettings->contentMode = NEXUS_VideoWindowContentMode_eFull;
        break;
    case NEXUS_VideoWindowContentMode_eFullNonLinear:
    case NEXUS_VideoWindowContentMode_ePanScan:
    case NEXUS_VideoWindowContentMode_ePanScanWithoutCorrection:
        /* only auto a/r correction supported for these modes */
        D_ERROR("bcmNexus/VideoProvider: unsupported ar correction format : %d at: %d\n", pPosition->manualAspectRatioCorrection, __LINE__);
        return(DFB_UNSUPPORTED);
    default:
        /* invalid value */
        D_ERROR("bcmNexus/VideoProvider: unsupported ar correction format : %d at %d \n", pPosition->manualAspectRatioCorrection,__LINE__);
        return(DFB_UNSUPPORTED);
    }

    /* use display coordinates as clipBase */
    pWindowSettings->clipBase.x = 0; /* unused */
    pWindowSettings->clipBase.y = 0; /* unused */
    pWindowSettings->clipBase.width = pPosition->displayWidth;
    pWindowSettings->clipBase.height = pPosition->displayHeight;

    /* start off with no clipping */
    pWindowSettings->clipRect = pWindowSettings->clipBase;

    /* apply user's clipping */
    clip = pWindowSettings->clipBase.width * horizontalClipping / 10000;
    pWindowSettings->clipRect.x += clip/2;
    pWindowSettings->clipRect.width -= clip;
    clip = pWindowSettings->clipBase.height * verticalClipping / 10000;
    pWindowSettings->clipRect.y += clip/2;
    pWindowSettings->clipRect.height -= clip;

    /* apply h/v position while maintaining the viewport and not changing scaling.
    let x/y go negative for now. */
    pWindowSettings->clipRect.x -= pPosition->horizontalPosition;
    pWindowSettings->clipRect.y -= pPosition->verticalPosition;

    /* convert viewport to window position, making adjustments to clipping as needed */
    pWindowSettings->position = viewport;


    if (viewport.x < 0) {
        int temp;
        temp = (-viewport.x) * pWindowSettings->clipBase.width / viewport.width;
        pWindowSettings->clipRect.x += temp;
        pWindowSettings->clipRect.width -= temp;
    }
    if (viewport.y < 0) {
        int temp;
        pWindowSettings->position.y = 0;

        temp = (-viewport.y) * pWindowSettings->clipBase.height / viewport.height;
        pWindowSettings->clipRect.y += temp;
        pWindowSettings->clipRect.height -= temp;
    }
    if (viewport.x + viewport.width > (int)pPosition->displayWidth) {
        pWindowSettings->clipRect.width -= (viewport.width - pWindowSettings->position.width) * pWindowSettings->clipBase.width / viewport.width;
    }
    if (viewport.y + viewport.height > (int)pPosition->displayHeight) {
        pWindowSettings->clipRect.height -= (viewport.height - pWindowSettings->position.height) * pWindowSettings->clipBase.height / viewport.height;
    }

    return(DFB_OK);

}

/*
*Thread to monitor stream format and destination rectangle.
*/

static void*
stream_monitor_thread( DirectThread *self, void *arg )
{
    IDirectFBVideoProvider *thiz = (IDirectFBVideoProvider *) arg;
    NEXUS_SimpleVideoDecoderHandle videoDecoderHandle;
    NEXUS_VideoDecoderStreamInformation streamInfo;
    NEXUS_VideoDecoderStreamInformation prevStreamInfo;
    NEXUS_SurfaceComposition comp;
    NEXUS_Rect prevDestinationRect;
    NEXUS_CalculateVideoWindowPositionSettings pos;
    NEXUS_Error rc;
    DFBResult ret;
    NEXUS_VideoWindowSettings windowSettings;
    bool streamChanged = false;
    bool updateRect = false;
    struct timespec   ts;
    struct timeval    tp;

    BSTD_UNUSED(self);

    DIRECT_INTERFACE_GET_DATA( IDirectFBVideoProvider_mediaplayer )

    videoDecoderHandle = media_player_get_video_decoder_priv(data->player);
    prevDestinationRect.x = prevDestinationRect.y = prevDestinationRect.width = prevDestinationRect.height = 0;
    memset(&prevStreamInfo, 0, sizeof(NEXUS_VideoDecoderStreamInformation));

    while (!data->cancel_thread ) {

        gettimeofday(&tp, NULL);

        /* Convert from timeval to timespec */
        ts.tv_sec  = tp.tv_sec;
        ts.tv_nsec = tp.tv_usec * 1000;
        ts.tv_nsec += 250*1000*1000;
        pthread_mutex_lock(&data->setDestinationLock);
        pthread_cond_timedwait( &data->setDestinationSem, &data->setDestinationLock, &ts );
        pthread_mutex_unlock(&data->setDestinationLock);
        if((!data->destinationRect.width) ||(!data->destinationRect.height)) {
            continue;
        }

        rc = media_player_get_streaminfo_priv(data->player, &streamInfo);
        if(rc) {
            continue;
        }

        if(!streamInfo.valid) {
            continue;
        }

        streamChanged = (prevStreamInfo.aspectRatio != streamInfo.aspectRatio)                      ||
                        (prevStreamInfo.sourceHorizontalSize != streamInfo.sourceHorizontalSize)    ||
                        (prevStreamInfo.sourceVerticalSize != streamInfo.sourceVerticalSize);

        updateRect = (prevDestinationRect.x != data->destinationRect.x)          ||
                     (prevDestinationRect.y != data->destinationRect.y)          ||
                     (prevDestinationRect.width != data->destinationRect.width)   ||
                     (prevDestinationRect.height != data->destinationRect.height);

        if(!(streamChanged || updateRect)) {
            continue;
          }

        NxClient_GetSurfaceClientComposition(data->allocResults.surfaceClient[0].id, &comp);

        memset(&windowSettings, 0, sizeof(NEXUS_VideoWindowSettings));
        memset(&pos, 0, sizeof(NEXUS_CalculateVideoWindowPositionSettings));
        windowSettings.clipRect = comp.clipRect;
        windowSettings.visible = true;
        pos.manualAspectRatioCorrection = NEXUS_VideoWindowContentMode_eBox;
        pos.viewport = data->destinationRect;
        pos.displayWidth = data->destinationRect.width;
        pos.displayHeight = data->destinationRect.height;
        pos.displayAspectRatio = NEXUS_DisplayAspectRatio_eAuto;
        pos.sourceAspectRatio = streamInfo.aspectRatio;
        pos.sampleAspectRatio.x = streamInfo.sampleAspectRatioX;
        pos.sampleAspectRatio.y = streamInfo.sampleAspectRatioY;
        pos.sourceWidth = streamInfo.sourceHorizontalSize;
        pos.sourceHeight = streamInfo.sourceVerticalSize;

        ret = IDirectFBVideoProvider_mediaplayer_GetVideowindowPosition(&pos, &windowSettings);
        if (ret) {
            D_ERROR("bcmNexus/VideoProvider: Error finding video window position %d!\n", ret);
        }

        comp.position = windowSettings.position;
        comp.clipRect = windowSettings.clipRect;
        comp.contentMode = windowSettings.contentMode;

        D_DEBUG_AT( bcmnexusVideoProvider, "%s: Setting video window composition:[%dx %dy %dw %dh]\n", __FUNCTION__, 
                    comp.position.x, comp.position.y, comp.position.width, comp.position.height );

        NxClient_SetSurfaceClientComposition(data->allocResults.surfaceClient[0].id, &comp);

        prevDestinationRect = data->destinationRect;
        prevStreamInfo = streamInfo;
    }
    return NULL;
}

/*
 * Public functions
 */

static DFBResult
Probe( IDirectFBVideoProvider_ProbeContext *ctx )
{
    /* Doing nothing in the probe for now*/
    BSTD_UNUSED(ctx);
    return (DFB_OK);
}

static DFBResult
Construct( IDirectFBVideoProvider *thiz,
           IDirectFBDataBuffer    *buffer  )
{
    IDirectFBDataBuffer_data *buffer_data;

    DIRECT_ALLOCATE_INTERFACE_DATA( thiz, IDirectFBVideoProvider_mediaplayer );

    data->count  = 1;
    data->player = NULL;
    data->rate   = NEXUS_NORMAL_DECODE_RATE;
    data->setup_done = false;
    data->status = DVSTATE_STOP;
    data->nxClient = false;
    data->connectId = 0;
    media_player_get_default_start_settings_priv( &data->start_settings );

    buffer_data = (IDirectFBDataBuffer_data*) buffer->priv;

    /* Supports path, file://, udp://, http://. - see parse_url in priv file */
    data->file_url = D_STRDUP( buffer_data->filename );

    direct_util_recursive_pthread_mutex_init( &data->lock );

    thiz->AddRef                = IDirectFBVideoProvider_mediaplayer_AddRef;
    thiz->Release               = IDirectFBVideoProvider_mediaplayer_Release;
    thiz->GetCapabilities       = IDirectFBVideoProvider_mediaplayer_GetCapabilities;
    thiz->GetSurfaceDescription = IDirectFBVideoProvider_mediaplayer_GetSurfaceDescription;
    thiz->PlayTo                = IDirectFBVideoProvider_mediaplayer_PlayTo;
    thiz->Stop                  = IDirectFBVideoProvider_mediaplayer_Stop;
    thiz->GetStatus             = IDirectFBVideoProvider_mediaplayer_GetStatus;
    thiz->SeekTo                = IDirectFBVideoProvider_mediaplayer_SeekTo;
    thiz->GetPos                = IDirectFBVideoProvider_mediaplayer_GetPos;
    thiz->GetLength             = IDirectFBVideoProvider_mediaplayer_GetLength;
    thiz->SetPlaybackFlags      = IDirectFBVideoProvider_mediaplayer_SetPlaybackFlags;
    thiz->SetSpeed              = IDirectFBVideoProvider_mediaplayer_SetSpeed;
    thiz->GetSpeed              = IDirectFBVideoProvider_mediaplayer_GetSpeed;
    thiz->SetDestination        = IDirectFBVideoProvider_mediaplayer_SetDestination;
    return(DFB_OK);
}


/*
 * Private functions
 */

static DFBResult
play_setup( IDirectFBVideoProvider_mediaplayer_data *data )
{
    DFBResult result = DFB_OK;
    media_player_create_settings create_settings;
    int rc;
    DFB_PlatformStatus platformStatus;

    if ( !data ) {
        D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                 "invalid parameter\n" );
        return(DFB_INVARG);
    }

    if ( data->setup_done ) {
        return(DFB_OK);
    }

    media_player_get_default_create_settings_priv(&create_settings);
    DFB_Platform_GetStatus( &platformStatus );
    if ( platformStatus.isXS ) {
        NxClient_AllocSettings allocSettings;
        NxClient_AllocResults allocResults;

        data->nxClient = true;

        NxClient_GetDefaultAllocSettings(&allocSettings);
        allocSettings.simpleVideoDecoder = create_settings.decodeVideo;
        allocSettings.simpleAudioDecoder = create_settings.decodeAudio;
        allocSettings.surfaceClient = 1;
        rc = NxClient_Alloc(&allocSettings, &allocResults);
        if ( rc ) {
            D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                     "NxClient_Alloc failed\n" );
            return(DFB_FAILURE);
        }

        data->allocResults = allocResults;
        create_settings.videoId = allocResults.simpleVideoDecoder[0].id;
        create_settings.audioId = allocResults.simpleAudioDecoder.id;
    }
    else {
        create_settings.videoId = NEXUS_ANY_ID;
        create_settings.audioId = NEXUS_ANY_ID;
    }

    data->player = media_player_create_priv( &create_settings );
    if ( !data->player ) {
        D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                 "create failed\n" );
        result = DFB_FAILURE;
        goto failure;
    }

    data->start_settings.eof = complete;
    data->start_settings.context = data;
    data->start_settings.loop = (data->flags & DVPLAY_LOOPING) ? true : false;
    data->start_settings.stream_url = data->file_url;

    rc = media_player_start_setup_priv( data->player, &data->start_settings );
    if ( rc ) {
        D_ERROR( "DirectFB/VideoProvider_mediaplayer: "
                 "start failed\n" );
        result = DFB_FAILURE;
        goto failure;
    }
    data->setup_done = true;

    rc = media_player_get_status_priv( data->player, &data->media_status );
    if ( rc || !data->media_status.valid ) {
        D_WARN( "DirectFB/VideoProvider_mediaplayer: "
                "get status failed\n" );
    }
    return(DFB_OK);

    failure:
    if ( data->nxClient ) {
        NxClient_Free(&data->allocResults);
    }
    return(result);
}

static void
send_event( IDirectFBVideoProvider_mediaplayer_data *data,
            DFBVideoProviderEventType         type )
{
    DFBEvent event;

    event.videoprovider.clazz = DFEC_VIDEOPROVIDER;
    event.videoprovider.type = type;
    event.videoprovider.data_type = DVPEDST_DATA;
    memset(&event.videoprovider.data, 0, sizeof(event.videoprovider.data));

    if ( data->events )
        data->events->PostEvent( data->events, &event );
}

static void complete(void *context)
{
    IDirectFBVideoProvider_mediaplayer_data *data = (IDirectFBVideoProvider_mediaplayer_data *)context;

    pthread_mutex_lock( &data->lock );
    if ( data->start_settings.loop ) {
        D_WARN( "DirectFB/VideoProvider_mediaplayer: "
                "end of stream - looping\n" );
        pthread_mutex_unlock( &data->lock );
        return;
    }

    if ( ( data->status == DVSTATE_PLAY ) && ( data->player ) ) {
        media_player_stop_priv( data->player );
    }
    data->status = DVSTATE_FINISHED;
    send_event( data, DVPET_FINISHED );
    pthread_mutex_unlock( &data->lock );
}

