#!/usr/bin/env ruby

# Copyright (C) 2014, 2015 Apple Inc. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
# THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
# THE POSSIBILITY OF SUCH DAMAGE.

require "pathname"
require "shellwords"

VERSION = "1.1"
DIRECTORY_NAME = "JetStream-#{VERSION}"

CDJS_FILES = [
    "constants.js",
    "util.js",
    "red_black_tree.js",
    "call_sign.js",
    "vector_2d.js",
    "vector_3d.js",
    "motion.js",
    "reduce_collision_set.js",
    "simulator.js",
    "collision.js",
    "collision_detector.js",
    "benchmark.js"
]

raise unless system("rm -rf " + DIRECTORY_NAME)
raise unless system("mkdir -p " + DIRECTORY_NAME)
raise unless system("mkdir -p #{DIRECTORY_NAME}/sunspider")
raise unless system("mkdir -p #{DIRECTORY_NAME}/sources")
raise unless system("cp sunspider/*.js #{DIRECTORY_NAME}/sunspider")
raise unless system("mkdir -p #{DIRECTORY_NAME}/cdjs")
CDJS_FILES.each {
    | filename |
    raise unless system("cp cdjs/#{filename} #{DIRECTORY_NAME}/cdjs")
}
raise unless system("cp -r JetStream.css JetStreamDriver.js LLVM-test-suite-LICENSE.txt simple Octane2 Octane2Setup.js SimpleSetup.js SunSpiderSetup.js Octane OctaneSetup.js CDjsSetup.js Reference.js TestingSetup.js JetStream-Logo.png JetStream-Logo@2x.png Swoosh.png Swoosh@2x.png " + DIRECTORY_NAME)

def detemplatize(basename)
    File.open(DIRECTORY_NAME + "/#{basename}.html", "w") {
        | outp |
        outp.write(IO::read("#{basename}-TEMPLATE.html").gsub(/@VERSION@/, VERSION.to_s))
    }
end
detemplatize("index")
detemplatize("in-depth")

def transferSource(benchmarkName, *files)
    files.each {
        | filename |
        pathname = Pathname.new(filename)
        
        originalBasename = pathname.basename
        extname = originalBasename.extname
        basename = originalBasename.basename(extname)
        
        if basename.to_s.start_with? benchmarkName
            resultPathname = Pathname.new(DIRECTORY_NAME) + "sources" + "#{basename}"
            resultName = basename.to_s
        else
            resultPathname = Pathname.new(DIRECTORY_NAME) + "sources" + "#{benchmarkName}-#{basename}"
            resultName = "#{benchmarkName} / #{basename}"
        end
        
        raise unless system("cp #{Shellwords.shellescape(pathname.to_s)} #{Shellwords.shellescape(resultPathname.to_s + extname)}")
    }
end

File.open(DIRECTORY_NAME + "/SimplePayload.js", "w") {
    | outp |
    outp.puts "// THIS IS AUTO-GENERATED BY create.rb"
    outp.puts "var SimplePayload = ["
    directory = Pathname.new("simple")
    Dir.foreach(directory) {
        | filename |
        next unless filename =~ /\.js$/
        baseName = $~.pre_match
        outp.puts "    {name: #{baseName.inspect}, content:#{IO::read(directory + filename).inspect}},"
        if filename.to_s =~ /\.(cp*)\.js/
            transferSource($~.pre_match, directory + ($~.pre_match + "." + $1), directory + filename)
        else
            transferSource(baseName, directory + filename)
        end
    }
    outp.puts "];"
}

File.open(DIRECTORY_NAME + "/SunSpiderPayload.js", "w") {
    | outp |
    outp.puts "// THIS IS AUTO-GENERATED BY create.rb"
    outp.puts "var SunSpiderPayload = ["
    directory = Pathname.new("sunspider")
    Dir.foreach(directory) {
        | filename |
        next unless filename =~ /\.js$/
        baseName = $~.pre_match
        outp.puts "    {name: #{baseName.inspect}, content:#{IO::read(directory + filename).inspect}},"
        transferSource(baseName, directory + filename)
    }
    outp.puts "];"
}

transferSource("code-multi-load", "Octane/code-load.js")
transferSource("richards", "Octane2/richards.js")
transferSource("deltablue", "Octane2/deltablue.js")
transferSource("crypto", "Octane2/crypto.js")
transferSource("raytrace", "Octane2/raytrace.js")
transferSource("earley-boyer", "Octane2/earley-boyer.js")
transferSource("regexp", "Octane2/regexp.js")
transferSource("splay", "Octane2/splay.js")
transferSource("navier-stokes", "Octane2/navier-stokes.js")
transferSource("pdfjs", "Octane2/pdfjs.js")
transferSource("mandreel", "Octane2/mandreel.js")
transferSource("gbemu", "Octane2/gbemu-part1.js", "Octane2/gbemu-part2.js")
transferSource("code-first-load", "Octane2/code-load.js")
transferSource("box2d", "Octane2/box2d.js")
transferSource("zlib", "Octane2/zlib.js", "Octane2/zlib-data.js")
transferSource("typescript", "Octane2/typescript.js", "Octane2/typescript-compiler.js", "Octane2/typescript-input.js")
transferSource("cdjs", *(CDJS_FILES.collect { | filename | "cdjs/#{filename}" }))

puts "You can now run JetStream by navigating to file://" + (Pathname.new(DIRECTORY_NAME) + "index.html").realpath.to_s

