/******************************************************************************
* Copyright (c) 2011 SeaChange International (SeaChange) and its Licensors. 
* All rights reserved.
*
* This software is the confidential and proprietary information of SeaChange
* ("Confidential Information"). You shall not disclose this source code or 
* such Confidential Information and shall use it only in accordance with the 
* terms of the license agreement you entered into.
*  
* SEACHANGE INERNATIONAL  MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE 
* SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT 
* LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
* PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SEACHANGE SHALL NOT BE LIABLE FOR 
* ANY DAMAGES SUFFERED BY LICENSEE NOR SHALL THEY BE RESPONSIBLE AS A RESULT 
* OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*******************************************************************************/




#ifdef USE_POD
//#if (defined MR05_P0 | defined MR05_P1)

#include "hal_api.h"
#include "pod_low_api.h"
#include "link.h"
#include "pod_api.h"
#include "pod_priv.h"
#include "hal_module.h"

#include "pod_module.h"
#include "cis.h"
#include "ci.h"
#include "rdk_debug.h"

#ifdef __cplusplus
extern "C" {
#endif
#undef FALSE
#undef TRUE
#define FALSE false
#define TRUE  true
/*----------------------------------- Global variables --------------------------------*/

/* CIS Information for module */
T_CIS_INFO g_pstCisInfo;

POD_MANUFACTURER_t bPodManufacturer = UNKNOWN_POD;

/****************************************************************************************
Name: 			f_ucCisLength
type:			function
Description: 	Calculate and return the length of the CIS.

In:             pucCIS : pointer on the CIS data

Out:			Nothing
Return value:	The length of the CIS
*****************************************************************************************/
UCHAR f_ucCisLength (UCHAR *pucCIS)
{
	UCHAR *pucTuple;

	/* The CA module card is supposed to be a 16-bits PC card, so,
	the CIS max length is 257 bytes. Actually I will suppose our CIS
	max size are not greater than 256 !.

	CIS termination : A 0xFF tuple code theorically indicates the end
	of the CIS. Nevertheless, 16-bits PC card can be terminated with
	a 0xFF tuple link field value (this termination case will not be
	taken in consideration here) */

	pucTuple = pucCIS;
	while ( *pucTuple != CISTPL_END )
		pucTuple += *(pucTuple+1) + 2;

	return ( pucTuple - pucCIS + 1 );
}


/****************************************************************************************
Name: 			f_bCisTupleVersion
type:			function
Description: 	Verification of the tuple version.

In:             pucTupleData : pointer on the tuple data.

Out:			Nothing
Return value:	FALSE : if the version id 5.0, TRUE otherwise.
*****************************************************************************************/
BOOLEAN f_bCisTupleVersion (UCHAR *pucTupleData)
{
	UCHAR *pucInfoString;
	BOOLEAN  bReturnCode;


	LOGINFO(HAL_MODULE_POD ,"f_bCisTupleVersion\r\n");

	/* Checking Major and minor version number */
	if ( (pucTupleData[0] != CIS_TPLLV1_MAJOR) || (pucTupleData[1] != CIS_TPLLV1_MINOR) )
	{
		RDK_LOG(RDK_LOG_ERROR, HAL_MODULE_POD,"%s() ERROR : f_bCisTupleVersion : pb in major/minor\r\n", __FUNCTION__);
		bReturnCode = FALSE;
	}
	else
	{
		/* Browsing information Strings */

		pucInfoString = &pucTupleData[2];

		while ( *pucInfoString != CISTPL_END )
		{
			while ( *pucInfoString != 0 )
			pucInfoString++;
			pucInfoString++;
		}

		bReturnCode = TRUE;
	}

	return bReturnCode;
}

/****************************************************************************************
Name: 			f_bCisTupleConfig
type:			function
Description: 	Tupple configuration

In:            pucTupleData    : pointer on the tuple data.
               ucTupleSize     : tuple size.
               pusConfRegAdr   : pointer on the address of the configuration register.
               pbModuleTypePod : pointer on the type of the module ( TRUE  : POD type )
                                                                   ( FALSE : DVB type ))

Out:			Nothing
Return value:	FALSE : error during the treatment.
*****************************************************************************************/
BOOLEAN f_bCisTupleConfig (UCHAR  *pucTupleData,  UCHAR ucTupleSize,
                        USHORT *pusConfRegAdr, BOOLEAN  *pbModuleTypePod)
{
	UCHAR ucTPCC_RMSize,  /* Number of bytes (mninus 1) in the Configuration Register presence mask field */
	ucTPCC_RASize,  /* Number of bytes (minus 1) in the Configuration Registers Base Address
					in Attribute Memory Space field.
					The address should not be greater than 0xFFE, so bTPCC_RASize <= 2 */
	ucTPCC_LAST;    /* Index of the last entry in the Card Configuration Table */
	BOOLEAN  bReturnCode = TRUE;
	UCHAR *pucSubTuple;
	UCHAR szInterfaceString[CIS_STRING_SIZE+1];
	UCHAR ucI;

	ucTPCC_RASize = (pucTupleData[0] & (BIT1 | BIT0)) + 1;
	ucTPCC_RMSize = ((pucTupleData[0] & (BIT5 | BIT4 | BIT3 | BIT2)) >> 2) + 1 ;
	ucTPCC_LAST = (pucTupleData[1] & (BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0));

    LOGINFO(HAL_MODULE_POD, "f_bCisTupleConfig :\r\n");

	/* Configuration Registers Base Address in Attribute Space. 1 to 4 bytes as indicated
	by the TPCC_RASZ field. POD module specs indicates this address should no be greater
	than 0x0FFE -> 2 bytes max */
	*pusConfRegAdr = (USHORT) pucTupleData[2];
	if ( ucTPCC_RASize > 1 )
		*pusConfRegAdr += (((USHORT) pucTupleData[3]) << 8);

	/* Configuration Registers Present Mask. 1 to 16 bytes as indicated by the TPCC_RMSZ field */

	/* Ignored ? */

	/*** CONFIGURATION SUBTUPLES ***/
	/* There should be only one subtuple : CISTPL_CIF */

	if ( (1 + 1 + ucTPCC_RASize + ucTPCC_RMSize) >= ucTupleSize )
	{
		RDK_LOG(RDK_LOG_ERROR, HAL_MODULE_POD,"%s() ERROR : f_ucCISTupleConfig : Cannot find Custom interface sub tuple\r\n", __FUNCTION__);
		bReturnCode = FALSE;
	}
	else
	{
		pucSubTuple = &pucTupleData[1 + 1 + ucTPCC_RASize + ucTPCC_RMSize];

		/* CISTPL_CIF : Custom interface subtuple */
		if ( pucSubTuple[0] == 0xC0 )
		{
			/* subtuple size : should not extend beyond parent tuple */
			if ( (pucSubTuple[1] + 1 + 1 + ucTPCC_RASize + ucTPCC_RMSize) > ucTupleSize )
			{
				RDK_LOG(RDK_LOG_ERROR, HAL_MODULE_POD,"%s() ERROR : f_ucCISTupleConfig : ERROR : SubTuple extends beyond config tuple\r\n",__FUNCTION__);
				bReturnCode = FALSE;
			}
			else
			{
				/* Interface ID number :
				Decoding the Interface Number is a little bit more complicated than this
				but it has been simplified since the id should be 0x0241 (DVB)
				or 0x0341 (POD) */

				if ( pucSubTuple[2] != 0x41 )
				{
					bReturnCode = FALSE;
				}
				else
				{
					switch ( pucSubTuple[3] )
					{
					case MODULE_TYPE_POD :  /* POD module id */
						LOGINFO(HAL_MODULE_POD, "This is a POD module\r\n");
						*pbModuleTypePod = TRUE;
						break;

					case MODULE_TYPE_DVB :  /* DVB module id */
						LOGINFO(HAL_MODULE_POD,"This is a DVB module\r\n");
						*pbModuleTypePod = FALSE;
						break;

					default :    /* unknown module id */
						bReturnCode = FALSE;
					}

					/* Interface description string (x.xx is the version)
					POD : "NRSS_CI_Vx.xx" not null ended.
					DVB : "DVB_CI_Vx.xx"  null ended.
					Practically this may be any thing ("OpenCable_POD_Vx.xx" for MOTOROLA)
					*/

					if ( ((pucSubTuple[1] - 2) <= CIS_STRING_SIZE) && bReturnCode == TRUE )
					{
						for ( ucI=0; ucI<pucSubTuple[1] - 2; ucI++ )
							szInterfaceString[ucI] = pucSubTuple[4+ucI];
						szInterfaceString[pucSubTuple[1] - 2] = 0;
					}
					else
					{
						if ( bReturnCode == FALSE )
						{
							RDK_LOG(RDK_LOG_ERROR, HAL_MODULE_POD, "%s() ERROR: Custom Interface desc. too long\r\n", __FUNCTION__);
						}
						bReturnCode = FALSE;
					}
				}
			}
		}
	}
	return bReturnCode;
}


/****************************************************************************************
Name: 			f_bCisTupleConfig
type:			function
Description: 	Check the CONFIG Table entry tuple, extract the Config Options values.

In:				UCHAR      *pTupleData		: pointer on the tuple data.
				UCHAR       bTupleSize		: Size of the tuple
                T_MODULE_CONFIG   *pstConfig: Contains the last config entry values.
Out:            UCHAR      *pucCOR: pointer on the COR to write
                BOOLEAN       *pbVpp: The Voltage must be changed ?

Out:			Nothing
Return value:	FALSE : error during the treatment.
*****************************************************************************************/

BOOLEAN cis_TupleCfgTableEntry (UCHAR      *pTupleData,
                             UCHAR       bTupleSize,
                             T_MODULE_CONFIG   *pstConfig,
                             UCHAR      *pucCOR,
                             BOOLEAN       *pbVpp)

{
	RDK_LOG(RDK_LOG_DEBUG,HAL_MODULE_POD, "%s bTupleSize %c \n",__FUNCTION__, bTupleSize);

	UCHAR ucFeatures;
	UCHAR ucNbAdrRange, ucAdrSize, ucLengthSize;
	T_CIS_POWER_DESC stVpp;

	/* This function will check if an entry matches DVB or POD configuration and retrieve the
	associated configuration entry number */

	/* A configuration entry can have a "default" flag which indicates that this entry
	provides default values for succeeding entries until another entry is encountered
	with its Default bit set.
	So, every "default" entry conditions should be stored to be used for following non-default
	entries */

	/*** TPCE_INDX : The Configuration Table Index Byte ***/

	/* TPCE_INDX : Configuration entry number */

	LOGINFO(HAL_MODULE_POD, "Calculating COR:  *pTupleData = 0x%02X\r\n",*pTupleData);

	*pucCOR = (*pTupleData) & (BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0);

	/* TPCE_INDX : Reset previous config if the "default" config flag is set */
	if ( *pTupleData & BIT7)
	{
		//MDEBUG (DPM_TEMP, "default entry");
		pstConfig->ucIntFace = 0;
		pstConfig->ucIo      = 0;
		pstConfig->ucPower   = 0;
		pstConfig->ucIRQ     = 0;
	}

	pTupleData++;

	/*** TPCE_IF: Interface Description Field ***/

	if ( *(pTupleData-1) & BIT6 )
	{
		pstConfig->ucIntFace = *pTupleData;
		pTupleData++;
	}

	/*** TPCE_FS: Feature Selection Byte ***/
	ucFeatures = *pTupleData;
	pTupleData++;

	/*** TPCE_PD : Power Description Structure ***/
	/* VPP requested value 5V : FALSE */
	*pbVpp = FALSE;

	if ( ucFeatures & (BIT1 | BIT0) )
	{
		/* Number of power descritpion fields :
		0 -> No power-description structures, use the default.
		1 -> VCC power-description-structure only.
		2 -> VCC and VPP[2::1] (VPP1 = VPP2) power-description-structures.
		3 -> VCC, VPP1 and VPP2 power-description-structures. */

		pstConfig->ucPower = ucFeatures & (BIT1 | BIT0);

		LOGINFO(HAL_MODULE_POD, "INFO: TPCE_FS feature select pwr = 0x%x\n", pstConfig->ucPower);

		/* Power description structure parameters :
		Each parameters (NomV, AvgI,...) contains a value (mantissa+exponent)
		and optional extensions.  Extensions are not taken in consideration,
		value is coded according to PC card standards volume 4 Metaformat specification
		Chapter "3.3.2.4 TPCE_PD: Power Description Structure" */

		if ( pstConfig->ucPower > 0 )
		{
			/* Vcc Power description */
			cis_GetPowerDesc (&pTupleData, NULL);

			if ( pstConfig->ucPower > 1 )
			{
				/* VPP1 Power description */
				cis_GetPowerDesc (&pTupleData, &stVpp);

				/* Test if VPP required Nominal Voltage is 5V */
				if ( (stVpp.NomV.bMantissa == 0x0A)&&(stVpp.NomV.bExponent == 0x05) )
					*pbVpp = TRUE;

				if ( pstConfig->ucPower > 2 )
				{
					/* VPP2 Power description */
					cis_GetPowerDesc (&pTupleData, NULL);
				}
			}
		}

#if (1)
        /* The above logic processes the full power description tuples without any exceptions.
        ** But here is the exception: If a host does not support individual VPP1
        ** and VPP2 (MR does not) and if power feature select field is 3 then
        ** only power 3.3 V should be set (This is as per SCTE28 section 6.1.2). */
        if ( (pstConfig->ucPower == 3) && (*pbVpp) )
        {
            *pbVpp = FALSE;
            LOGWARNING(HAL_MODULE_POD, "%s() WARN: VPP set back to 3.3v\n", __FUNCTION__);
        }
#endif

	}

	//*** TPCE_TD: Configuration Timing Information ***/
	if ( ucFeatures & BIT2 )
		pTupleData++;

	/*** TPCE_IO: I/O Space Addresses Required For This Configuration ***/
	if ( ucFeatures & BIT3 )
	{
		/* I/O space description byte */
		pstConfig->ucIo = *pTupleData;
		pTupleData++;

		/* Range flag : 0 no range, 1 : Presence of I/O addresses range */
		if ( pstConfig->ucIo & BIT7 )
		{
			/* I/O Range Descriptor byte */

			/* Number of I/O Address Ranges :
			This field specifies how many I/O Address Range Description
			fields will follow the I/O Range Descriptor byte.*/
			ucNbAdrRange = (*pTupleData & ( BIT3 | BIT2 | BIT1 | BIT0 )) + 1;

			/* Size of Address This field specifies how many bytes will be used
			to encode the Start of I/O Address block in each of the I/O Range
			Description fields that follow.
			0 -> 0 byte
			1 -> 1 bytes
			2 -> 2 bytes
			3 -> 4 bytes !!!
			*/

			ucAdrSize = (*pTupleData & ( BIT5 | BIT4 )) >> 4;
			if ( ucAdrSize == 0x03 )
				ucAdrSize = 0x04;

			/* Size of Length This field specifies how many bytes will be used
			to encode the Length of I/O Address block in each of the I/O Range
			Description fields that follow.
			0 -> 0 byte
			1 -> 1 bytes
			2 -> 2 bytes
			3 -> 4 bytes !!!
			*/
			ucLengthSize = (*pTupleData & ( BIT7 | BIT6 )) >> 6;
			if ( ucLengthSize == 0x03 )
				ucLengthSize = 0x04;

			/* Skipping I/OAddress ranges */
			pTupleData += ucNbAdrRange * (ucAdrSize+ucLengthSize) + 1;
		}
	}

	/*** TPCE_IR: Interrupt Request Description Structure ***/
	if ( ucFeatures & BIT4 )
	{
		pstConfig->ucIRQ = *pTupleData;
		if ( *pTupleData & BIT4 )
		pTupleData += 2;
		pTupleData++;
	}

	/*** TPCE_MS: Memory Space Description Structure ***/

	/* Don't care */

	/*** TPCE_MI: Miscellaneous Features Field ***/

	/* Don't care */
	/* The "ERRATA for EN50221" document (CIT057.doc) specifies that a valid DVB
	configuration has an IRQ descrition byte value equal to 0x20. This verification
	is not performed here since CA module cards tested do not specifies a valid IRQ.
	Other values are correctly tested */

	//checks that values are correct
	if ( (pstConfig->ucIntFace == 0x04) &&
					(pstConfig->ucIo      == 0x22) &&
					(pstConfig->ucPower   != 0x00) )
	{
		LOGINFO(HAL_MODULE_POD,"POD-DVB compliant\r\n");
		pstConfig->bValid = TRUE;
	}
	else
	{
		LOGINFO(HAL_MODULE_POD, "NOT POD-DVB compliant\r\n");
		pstConfig->bValid = FALSE;
	}

	/*** STCE_EV : Environment Descriptor Subtuple ***/

	/*** STCE_PD : Physical Device Name Subtuple ***/
	return TRUE;

}


/****************************************************************************************
Name: 			f_ucCisGetTuple
type:			function
Description: 	Extracts the tuple's size and data at the current CIS position

In:				pucTupleCurr  : pointer on the tuple packet
				ppucTupleData : pointer on the the address of the first tuple data.
                pucTupleSize  : The tuple size.

Out:			Nothing
Return value:	First value of tuple
*****************************************************************************************/
UCHAR f_ucCisGetTuple (UCHAR *pucTupleCurr,
                       UCHAR **ppucTupleData, UCHAR *pucTupleSize)
{
	*pucTupleSize  = pucTupleCurr[1];
	*ppucTupleData = &pucTupleCurr[2];
	return pucTupleCurr[0];
}

/****************************************************************************************
Name: 			f_bCisCheck
type:			function
Description: 	Checks CIS and extracts COR and COR base address

In:				pucCIS  : pointer on the CIS data.
                pstInfo : pointer on the CIS information structure.

Out:			Nothing
Return value:	FALSE : error on the module or verification.
*****************************************************************************************/
BOOLEAN f_bCisCheck (UCHAR *pucCIS, T_CIS_INFO *pstInfo)
{
	BOOLEAN            bModuleProblem;
	UCHAR           ucTupleCode, ucTupleSize;
	UCHAR           *pucCIS2;
	UCHAR           *pucTupleData;
	T_MODULE_CONFIG stConfig;

	stConfig.ucIntFace = 0;
	stConfig.ucIo      = 0;
	stConfig.ucPower   = 0;
	stConfig.ucIRQ     = 0;
	stConfig.bValid    = FALSE;

	LOGINFO(HAL_MODULE_POD,"f_bCisCheck\r\n");

	pucCIS2 = pucCIS;
	bModuleProblem = FALSE;

	while (  (	ucTupleCode = f_ucCisGetTuple (pucCIS2, &pucTupleData, &ucTupleSize)) != CISTPL_END
				&&  (! bModuleProblem) )
	{
		switch ( ucTupleCode )
		{
			case CISTPL_VERS_1 :
				LOGINFO(HAL_MODULE_POD, "CISTPL_VERS_1: Manufacturer name = %s\r\n", &pucCIS2[4] );
			//	RDK_LOG(RDK_LOG_DEBUG,HAL_MODULE_POD, "CISTPL_VERS_1: Manufacturer name = %s\r\n", &pucCIS2[4] );

				if ( f_bCisTupleVersion (pucTupleData) == FALSE )
					bModuleProblem = TRUE;
				break;

			case CISTPL_MANFID :
				LOGINFO(HAL_MODULE_POD,"CISTPL_MANFID = %02x %02x\r\n", pucCIS2[2], pucCIS2[3] );

				if ( (pucCIS2[2] == 0x04) && ( pucCIS2[3] == 0xD2 ) )
				{
				    LOGINFO(HAL_MODULE_POD, "DK (hpnx) Pod Detected!, ver = 0x%02x%02x\n",
				                                            pucCIS2[4], pucCIS2[5]);
				    bPodManufacturer = DK_HPNX_POD; /* Digital Keystone */
				}
				else if ( (pucCIS2[2] == 0) && ( pucCIS2[3] == 0x1F) )
				{
				    LOGINFO(HAL_MODULE_POD, "MO Pod Detected!, ver = 0x%02x%02x\n",
				                                            pucCIS2[4], pucCIS2[5]);
				    bPodManufacturer = MOT_POD;
				}
				else if ( (pucCIS2[2] == 0x37) && ( pucCIS2[3] == 0x02) )
				{
				    LOGINFO(HAL_MODULE_POD, "SA Pod Detected!, ver = 0x%02x%02x\n",
				                                            pucCIS2[4], pucCIS2[5]);
/*				    RDK_LOG(RDK_LOG_DEBUG,HAL_MODULE_POD, "SA Pod Detected!, ver = 0x%02x%02x\n",
				                                            pucCIS2[4], pucCIS2[5]);*/
				    bPodManufacturer = SA_POD;
				}
				else if ( (pucCIS2[2] == 0xff) && ( pucCIS2[3] == 0xff) )
				{
				    /* Currently NDS uses FFFF as manufacturer ID.
				    ** this is kind of bogus.   */
				    LOGINFO(HAL_MODULE_POD, "NDS Pod Detected!, ver = 0x%02x%02x\n",
				                                            pucCIS2[4], pucCIS2[5]);
				    bPodManufacturer = NDS_POD;
				}
				else
				{
				    /* Assume anything else is a flash card. */
				    LOGINFO(HAL_MODULE_POD,"Not a cable card! (flash?), ver = 0x%02x%02x\n",
				                                            pucCIS2[4], pucCIS2[5]);
				    bPodManufacturer = FLASH_CARD;
				    bModuleProblem = TRUE;
				}

				break;

			case CISTPL_CONFIG :
				//MDEBUG (DPM_TEMP, "CISTPL_CONFIG\r\n");
				if ( f_bCisTupleConfig (pucTupleData, ucTupleSize, &(pstInfo->usAddrCOR),
								  &(pstInfo->bModuleTypePOD)) == FALSE )
					bModuleProblem = TRUE;
				break;

			case CISTPL_CFTABLE_ENTRY :
				//MDEBUG (DPM_TEMP, "CISTPL_CFTABLE_ENTRY\r\n");
				if ( stConfig.bValid == FALSE )
					cis_TupleCfgTableEntry (pucTupleData, ucTupleSize,&stConfig, &(pstInfo->ucCOR),&(pstInfo->bVpp));
				break;

			case  CISTPL_DEVICE_OC :
				//MDEBUG (DPM_TEMP, "CISTPL_DEVICE_OC\r\n");
				break;

			case CISTPL_DEVICE_OA :
				//MDEBUG (DPM_TEMP, "CISTPL_DEVICE_OA\r\n");
				break;

			case CISTPL_STCE_PD :
				//MDEBUG (DPM_TEMP, "CISTPL_STCE_PD\r\n");
				break;

			case CISTPL_NO_LINK :
				//MDEBUG (DPM_TEMP, "CISTPL_NO_LINK\r\n");
				break;

			case CISTPL_LINKTARGET:
				//MDEBUG (DPM_TEMP, "CISTPL_LINKTARGET\r\n");
				break;

			default :
				//MDEBUG (DPM_TEMP, "Tuple Not Checked, not an error\r\n");
				//bModuleProblem = TRUE;
				break;
		} // end switch


		if ( (ucTupleCode == CISTPL_END) || (bModuleProblem == TRUE) )
			break;

		pucCIS2 += ucTupleSize + 2;

		ucTupleCode = f_ucCisGetTuple (pucCIS2, &pucTupleData, &ucTupleSize);
	} // end while

	/* Modify COR value depending on IRQ mode :
	* if module type is POD -then- BIT6 of COR value shall be set (indicating
	* LEVEL MODE interrupts) (SCTE DVS/221 REV7)
	* If a IRQ description field was found in a Config Table entry indicating
	* IRQ LEVEL mode, BIT6 is also set (This does not comes from DVB or POD
	* specification but is interpreted from PC card standards volume 2
	* (chapter 4.15.1) */

	if ( (stConfig.ucIRQ & BIT5) || (pstInfo->bModuleTypePOD == TRUE) )
	{
		LOGINFO(HAL_MODULE_POD, "Set BIT6 as formerly\r\n");
		pstInfo->ucCOR |= BIT6;
	}
	return ( !bModuleProblem );
}


/****************************************************************************************
Name: 			cis_GetPowerDesc
type:			function
Description: 	Fill a power description structure.

In:				UCHAR **ppTupleData: Pointer on the Address of the Dta
				T_CIS_POWER_DESC *pstPowerDesc: Struct to be filled

Out:			Nothing
Return value:	void
*****************************************************************************************/
void cis_GetPowerDesc (UCHAR **ppTupleData, T_CIS_POWER_DESC *pstPowerDesc)
{
	UCHAR bParamSelect;

	/* Reset Parameter description structure */
	if ( pstPowerDesc != NULL )
	{
		pstPowerDesc->NomV.bMantissa    = 0x00;
		pstPowerDesc->NomV.bExponent    = 0x00;
		pstPowerDesc->MinV.bMantissa    = 0x00;
		pstPowerDesc->MinV.bExponent    = 0x00;
		pstPowerDesc->MaxV.bMantissa    = 0x00;
		pstPowerDesc->MaxV.bExponent    = 0x00;
		pstPowerDesc->StaticI.bMantissa = 0x00;
		pstPowerDesc->StaticI.bExponent = 0x00;
		pstPowerDesc->AvgI.bMantissa    = 0x00;
		pstPowerDesc->AvgI.bExponent    = 0x00;
		pstPowerDesc->PeakI.bMantissa   = 0x00;
		pstPowerDesc->PeakI.bExponent   = 0x00;
		pstPowerDesc->PDwnI.bMantissa   = 0x00;
		pstPowerDesc->PDwnI.bExponent   = 0x00;
	}

	/* Parameter selection byte
	Each bit in the parameter selection byte indicates whether or not the corresponding
	parameter is described by parameter definitions which follow that byte. */
	bParamSelect = **ppTupleData;

	(*ppTupleData)++;

	/* Nominal operating supply voltage.
	In the absence of other information the nominal operating voltage has a tolerance of 5%. */
	if ( bParamSelect & BIT0 )
	{
		if ( pstPowerDesc != NULL )
			cis_GetPowerParam (ppTupleData, &(pstPowerDesc->NomV));
		else
			cis_GetPowerParam (ppTupleData, NULL);
	}

	/* Minimum operating supply voltage. */
	if ( bParamSelect & BIT1 )
	{
		if ( pstPowerDesc != NULL )
			cis_GetPowerParam (ppTupleData, &(pstPowerDesc->MinV));
		else
			cis_GetPowerParam (ppTupleData, NULL);
	}

	/* Maximum operating supply voltage. */
	if ( bParamSelect & BIT2 )
	{
		if ( pstPowerDesc != NULL )
			cis_GetPowerParam (ppTupleData, &(pstPowerDesc->MaxV));
		else
			cis_GetPowerParam (ppTupleData, NULL);
	}

	/* Continuous supply current required. */
	if ( bParamSelect & BIT3 )
	{
		if ( pstPowerDesc != NULL )
			cis_GetPowerParam (ppTupleData, &(pstPowerDesc->StaticI));
		else
			cis_GetPowerParam (ppTupleData, NULL);
	}

	/* Maximum current required averaged over 1 second. */
	if ( bParamSelect & BIT4 )
	{
		if ( pstPowerDesc != NULL )
			cis_GetPowerParam (ppTupleData, &(pstPowerDesc->AvgI));
		else
			cis_GetPowerParam (ppTupleData, NULL);
	}

	/* Peak I Maximum current required averaged over 10 milliseconds. */
	if ( bParamSelect & BIT5 )
	{
		if ( pstPowerDesc != NULL )
			cis_GetPowerParam (ppTupleData, &(pstPowerDesc->PeakI));
		else
			cis_GetPowerParam (ppTupleData, NULL);
	}

	/* PDwn I Power-down supply current required. */
	if ( bParamSelect & BIT6 )
	{
		if ( pstPowerDesc != NULL )
			cis_GetPowerParam (ppTupleData, &(pstPowerDesc->PDwnI));
		else
			cis_GetPowerParam (ppTupleData, NULL);
	}

	/* RFU Reserved for future standardization. */
	if ( bParamSelect & BIT6 )
		cis_GetPowerParam (ppTupleData, NULL);
}



/****************************************************************************************
Name: 			cis_GetPowerParam
type:			function
Description: 	Extract values from a power description parameter entry.

In:				UCHAR **ppTupleData: Pointer on the Address of the Data
				T_CIS_POWER_PARAM *pstPowerParam:
Out:			Nothing
Return value:	void
*****************************************************************************************/
void cis_GetPowerParam (UCHAR **ppTupleData, T_CIS_POWER_PARAM *pstPowerParam)
{

    if ( pstPowerParam != NULL )
	{
		pstPowerParam->bMantissa = (**ppTupleData & (BIT6+BIT5+BIT4+BIT3)) >> 3;
		pstPowerParam->bExponent = (**ppTupleData & ( BIT2+BIT1+BIT0 ));
	}

	/* Skip Power definition extensions */
	do
	{
		(*ppTupleData)++;
	}
	while ( **ppTupleData & BIT7 );
}



#ifdef __cplusplus
}
#endif

//#endif  // MR05_P0 | MR05_P1

#endif // USE_POD
