/*
 * drivers/amlogic/drm/meson_async_atomic.c
 *
 * Copyright (C) 2017 Amlogic, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 */

#include "meson_async_atomic.h"
#include "meson_plane.h"
#include <drm/drmP.h>
#include <drm/drm_atomic.h>
#include <drm/drm_mode.h>
#include <drm/drm_plane_helper.h>

static int atomic_set_prop(struct drm_atomic_state *state,
		struct drm_mode_object *obj, struct drm_property *prop,
		u64 prop_value)
{
	struct drm_mode_object *ref;
	int ret;

	if (!drm_property_change_valid_get(prop, prop_value, &ref))
		return -EINVAL;

	switch (obj->type) {
	case DRM_MODE_OBJECT_PLANE: {
		struct drm_plane *plane = obj_to_plane(obj);
		struct drm_plane_state *plane_state;

		plane_state = drm_atomic_get_plane_state(state, plane);
		if (IS_ERR(plane_state)) {
			ret = PTR_ERR(plane_state);
			break;
		}

		ret = drm_atomic_plane_set_property(plane,
				plane_state, prop, prop_value);
		break;
	}
	default:
		ret = -EINVAL;
		break;
	}

	drm_property_change_valid_put(prop, ref);
	return ret;
}

/*modified from drm_mode_atomic_ioctl() */
int meson_asyc_atomic_ioctl(struct drm_device *dev,
			  void *data, struct drm_file *file_priv)
{
	struct drm_mode_atomic *arg = data;
	u32 __user *objs_ptr = (u32 __user *)(unsigned long)(arg->objs_ptr);
	u32 __user *count_props_ptr = (u32 __user *)(unsigned long)(arg->count_props_ptr);
	u32 __user *props_ptr = (u32 __user *)(unsigned long)(arg->props_ptr);
	u64 __user *prop_values_ptr = (u64 __user *)(unsigned long)(arg->prop_values_ptr);
	unsigned int copied_objs, copied_props;
	struct drm_atomic_state *state;
	struct drm_modeset_acquire_ctx ctx;
	struct drm_plane *plane;
	struct drm_out_fence_state *fence_state;
	unsigned int plane_mask;
	int ret = 0;
	unsigned int i, j, num_fences;

	drm_modeset_acquire_init(&ctx, 0);

	state = drm_atomic_state_alloc(dev);
	if (!state)
		return -ENOMEM;

	state->allow_modeset = false;
	state->acquire_ctx = &ctx;

retry:
	plane_mask = 0;
	copied_objs = 0;
	copied_props = 0;
	fence_state = NULL;
	num_fences = 0;

	if (arg->count_objs > 1) {
		DRM_ERROR("only accept plane object update(%d).\n", arg->count_objs);
		ret = -EINVAL;
		goto out;
	}

	for (i = 0; i < arg->count_objs; i++) {
		u32 obj_id, count_props;
		struct drm_mode_object *obj;

		if (get_user(obj_id, objs_ptr + copied_objs)) {
			ret = -EFAULT;
			goto out;
		}

		obj = drm_mode_object_find(dev, obj_id, DRM_MODE_OBJECT_ANY);
		if (!obj) {
			ret = -ENOENT;
			goto out;
		}

		if (obj->type != DRM_MODE_OBJECT_PLANE) {
			DRM_ERROR("only accept plane object update, canot set (%d)\n", obj->type);

			ret = -EINVAL;
			goto out;
		}

		if (!obj->properties) {
			drm_mode_object_unreference(obj);
			ret = -ENOENT;
			goto out;
		}

		if (get_user(count_props, count_props_ptr + copied_objs)) {
			drm_mode_object_unreference(obj);
			ret = -EFAULT;
			goto out;
		}

		copied_objs++;

		for (j = 0; j < count_props; j++) {
			u32 prop_id;
			u64 prop_value;
			struct drm_property *prop;

			if (get_user(prop_id, props_ptr + copied_props)) {
				drm_mode_object_unreference(obj);
				ret = -EFAULT;
				goto out;
			}

			prop = drm_mode_obj_find_prop_id(obj, prop_id);
			if (!prop) {
				drm_mode_object_unreference(obj);
				ret = -ENOENT;
				goto out;
			}

			if (copy_from_user(&prop_value,
					   prop_values_ptr + copied_props,
					   sizeof(prop_value))) {
				drm_mode_object_unreference(obj);
				ret = -EFAULT;
				goto out;
			}

			ret = atomic_set_prop(state, obj, prop, prop_value);
			if (ret) {
				drm_mode_object_unreference(obj);
				goto out;
			}

			copied_props++;
		}

		if (count_props) {
			plane = obj_to_plane(obj);
			plane_mask |= (1 << drm_plane_index(plane));
			plane->old_fb = plane->fb;
		}
		drm_mode_object_unreference(obj);
	}

	ret = meson_async_atomic_commit(state);

out:
	drm_atomic_clean_old_fb(dev, plane_mask, ret);

	if (ret == -EDEADLK) {
		drm_atomic_state_clear(state);
		drm_modeset_backoff(&ctx);
		goto retry;
	}

	if (ret)
		drm_atomic_state_free(state);

	drm_modeset_drop_locks(&ctx);
	drm_modeset_acquire_fini(&ctx);
	return ret;
}

/*similar to drm_atomic_commit(), added async commit as kernel 5.4 did.
 *No lock in this function now, confirm it is called in lock range.
 */
int meson_async_atomic_commit(struct drm_atomic_state *state)
{
	struct drm_mode_config *config = &state->dev->mode_config;
	struct drm_plane *plane = NULL;
	struct drm_plane_state *plane_state = NULL;
	struct am_video_plane *videoplane = NULL;

	int i, plane_idx, n_planes = 0;
	int ret = 0;

	for (i = 0; i < config->num_total_plane; i++) {
		if (state->planes[i].ptr) {
			plane = state->planes[i].ptr;
			plane_idx = i;
			plane_state = state->planes[i].state;
			n_planes++;
			break;
		}
	}
	videoplane = to_am_video_plane(plane);

	DRM_DEBUG("async commit %p-%p-%d, %d, %s\n",
		state, plane_state, plane->index, n_planes,
		videoplane->plane_type ? "video" : "osd");

	/* FIXME: we support only single plane updates for now */
	if (n_planes != 1)
		return -EINVAL;

	/*async check*/
	if (videoplane->plane_type == VIDEO_PLANE)
		ret = meson_video_plane_async_check(plane, plane_state);
	else
		ret = meson_osd_plane_async_check(plane, plane_state);
	if (ret) {
		DRM_ERROR("ASYNC check failed (%d)\n", ret);
		return ret;
	}

	drm_atomic_helper_swap_state(state, true);

	/*get swapped plane state*/
	plane_state = state->planes[plane_idx].state;

	/*async commit*/
	if (videoplane->plane_type == VIDEO_PLANE)
		meson_video_plane_async_update(plane, plane_state);
	else
		meson_osd_plane_async_update(plane, plane_state);

	drm_atomic_helper_cleanup_planes(state->dev, state);
	drm_atomic_state_free(state);

	return ret;
}
