/*
 * drivers/amlogic/thermal/ddr_cooling.c
 *
 * Copyright (C) 2017 Amlogic, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 */

#include <linux/module.h>
#include <linux/thermal.h>
#include <linux/err.h>
#include <linux/slab.h>
#include <linux/amlogic/ddr_cooling.h>
#include <linux/amlogic/meson_cooldev.h>
#include <linux/io.h>
#include "../../thermal/thermal_core.h"

static DEFINE_IDR(ddr_idr);
static DEFINE_MUTEX(cooling_ddr_lock);
static DEFINE_MUTEX(cooling_list_lock);
static LIST_HEAD(ddr_dev_list);

static void __iomem *ddr_reg0;

/**
 * get_idr - function to get a unique id.
 * @idr: struct idr * handle used to create a id.
 * @id: int * value generated by this function.
 *
 * This function will populate @id with an unique
 * id, using the idr API.
 *
 * Return: 0 on success, an error code on failure.
 */
static int get_idr(struct idr *idr, int *id)
{
	int ret;

	mutex_lock(&cooling_ddr_lock);
	ret = idr_alloc(idr, NULL, 0, 0, GFP_KERNEL);
	mutex_unlock(&cooling_ddr_lock);
	if (unlikely(ret < 0))
		return ret;
	*id = ret;

	return 0;
}

/**
 * release_idr - function to free the unique id.
 * @idr: struct idr * handle used for creating the id.
 * @id: int value representing the unique id.
 */
static void release_idr(struct idr *idr, int id)
{
	mutex_lock(&cooling_ddr_lock);
	idr_remove(idr, id);
	mutex_unlock(&cooling_ddr_lock);
}

static int ddr_get_max_state(struct thermal_cooling_device *cdev,
			     unsigned long *state)
{
	struct ddr_cooling_device *ddr_device = cdev->devdata;

	*state = ddr_device->ddr_status;
	return 0;
}

static int ddr_get_cur_state(struct thermal_cooling_device *cdev,
			     unsigned long *state)
{
	*state = 0;
	return 0;
}

static int ddr_set_cur_state(struct thermal_cooling_device *cdev,
			     unsigned long state)
{
	state = 0;
	return state;
}

static int ddr_get_requested_power(struct thermal_cooling_device *cdev,
				   struct thermal_zone_device *tz,
				   u32 *power)
{
	*power = 0;
	return 0;
}

static int ddr_state2power(struct thermal_cooling_device *cdev,
			   struct thermal_zone_device *tz,
			   unsigned long state, u32 *power)
{
	*power = 0;
	return 0;
}

static int ddr_power2state(struct thermal_cooling_device *cdev,
			   struct thermal_zone_device *tz, u32 power,
			   unsigned long *state)
{
	cdev->ops->get_cur_state(cdev, state);
	return 0;
}

static int ddr_notify_state(struct thermal_cooling_device *cdev,
			    struct thermal_zone_device *tz,
			    enum thermal_trip_type type)
{
	struct ddr_cooling_device *ddr_device = cdev->devdata;
	int i;
	unsigned int val, val0, reg_val;

	reg_val = readl_relaxed(ddr_reg0);
	switch (type) {
	case THERMAL_TRIP_ACTIVE:
		val = ddr_device->ddr_data[0];

		for (i = 0; i < ddr_device->ddr_status - 1; i++) {
			if (tz->temperature >= ddr_device->ddr_temp[i])
				val = ddr_device->ddr_data[i + 1];
			else
				break;
		}
		pr_debug("chip temp: %d, set ddr reg bit val: %x\n", tz->temperature, val);

		val0 = reg_val & ddr_device->ddr_bits_keep;
		val0 = val0 >> ddr_device->ddr_bits[0];

		if (val0 == val)
			break;

		reg_val &= ddr_device->ddr_bits_keep;
		reg_val |= (val << ddr_device->ddr_bits[0]);
		pr_debug("last set ddr reg val: %x\n", reg_val);

		writel_relaxed(reg_val, ddr_reg0);
	default:
		break;
	}
	return 0;
}

static struct thermal_cooling_device_ops const ddr_cooling_ops = {
	.get_max_state = ddr_get_max_state,
	.get_cur_state = ddr_get_cur_state,
	.set_cur_state = ddr_set_cur_state,
	.state2power   = ddr_state2power,
	.power2state   = ddr_power2state,
	.notify_state  = ddr_notify_state,
	.get_requested_power = ddr_get_requested_power,
};

struct thermal_cooling_device *
ddr_cooling_register(struct device_node *np, struct cool_dev *cool)
{
	struct thermal_cooling_device *cool_dev;
	struct ddr_cooling_device *ddr_dev = NULL;
	struct thermal_instance *pos = NULL;
	char dev_name[THERMAL_NAME_LENGTH];
	int ret = 0, i;

	ddr_dev = kzalloc(sizeof(*ddr_dev), GFP_KERNEL);
	if (!ddr_dev)
		return ERR_PTR(-ENOMEM);

	ret = get_idr(&ddr_idr, &ddr_dev->id);
	if (ret) {
		kfree(ddr_dev);
		return ERR_PTR(-EINVAL);
	}

	ddr_reg0 = ioremap(cool->ddr_reg, 1);
	if (!ddr_reg0) {
		pr_err("thermal ddr cdev: ddr reg0 ioremap fail.\n");
		return ERR_PTR(-EINVAL);
	}

	snprintf(dev_name, sizeof(dev_name), "thermal-ddr-%d",
		 ddr_dev->id);

	ddr_dev->ddr_reg = cool->ddr_reg;
	ddr_dev->ddr_status = cool->ddr_status;
	for (i = 0; i < 2; i++)
		ddr_dev->ddr_bits[i] = cool->ddr_bits[i];
	ddr_dev->ddr_bits_keep = ~(0xffffffff << (ddr_dev->ddr_bits[1] + 1));
	ddr_dev->ddr_bits_keep = ~((ddr_dev->ddr_bits_keep >> ddr_dev->ddr_bits[0])
				   << ddr_dev->ddr_bits[0]);
	for (i = 0; i < cool->ddr_status; i++)
		ddr_dev->ddr_data[i] = cool->ddr_data[i];
	for (i = 0; i < cool->ddr_status - 1; i++)
		ddr_dev->ddr_temp[i] = cool->ddr_temp[i];

	cool_dev = thermal_of_cooling_device_register(np, dev_name, ddr_dev,
						      &ddr_cooling_ops);

	if (!cool_dev) {
		release_idr(&ddr_idr, ddr_dev->id);
		kfree(ddr_dev);
		return ERR_PTR(-EINVAL);
	}
	ddr_dev->cool_dev = cool_dev;

	list_for_each_entry(pos, &cool_dev->thermal_instances, cdev_node) {
		if (!strncmp(pos->cdev->type, dev_name, sizeof(dev_name))) {
			cool_dev->ops->notify_state(cool_dev, pos->tz, THERMAL_TRIP_ACTIVE);
			break;
		}
	}

	mutex_lock(&cooling_list_lock);
	list_add(&ddr_dev->node, &ddr_dev_list);
	mutex_unlock(&cooling_list_lock);

	return cool_dev;
}
EXPORT_SYMBOL_GPL(ddr_cooling_register);

/**
 * cpucore_cooling_unregister - function to remove cpucore cooling device.
 * @cdev: thermal cooling device pointer.
 *
 * This interface function unregisters the "thermal-cpucore-%x" cooling device.
 */
void ddr_cooling_unregister(struct thermal_cooling_device *cdev)
{
	struct ddr_cooling_device *ddr_dev;

	if (!cdev)
		return;

	iounmap(ddr_reg0);

	ddr_dev = cdev->devdata;

	mutex_lock(&cooling_list_lock);
	list_del(&ddr_dev->node);
	mutex_unlock(&cooling_list_lock);

	thermal_cooling_device_unregister(ddr_dev->cool_dev);
	release_idr(&ddr_idr, ddr_dev->id);
	kfree(ddr_dev);
}
EXPORT_SYMBOL_GPL(ddr_cooling_unregister);
