/****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ***************************************************************************/

#ifndef _FLOWMGR_H
#define _FLOWMGR_H

#include <linux/version.h>
#include <linux/workqueue.h>
#include <net/netfilter/nf_conntrack_offload.h>
#include "bcmport.h"
#include "bcmnethooks.h"

#define MAX_INF 32
#define MAX_MC2UC 4
#define NUM_MCTX 32

#define kflowmgr_wq flowmgr.fap_wq

enum flow_notrack_type {
	DISABLE_NOTRACK,
	ENABLE_NOTRACK,
	ENABLE_ALL_NOTRACK,
	DISABLE_ALL_NOTRACK,
	RESTORE_DEFAULT_NOTRACK,
	FLOW_NOTRACK_MAX_RESULT
};

enum flow_flush_type {
	FLOW_FLUSH_DISABLE,
	/* 1: Delete from FAP
	   Fast->Slow
	*/
	FLOW_FLUSH_FAP,
	/* 2: Delete from FAP and Unlearn flow
	   Fast->Slow->Fast for both US and DS flows with same conntrack
	*/
	FLOW_FLUSH_UNLEARN,
	/* 3: Remove conntrack
	   Fast->Slow->Fast for US with new conntrack
	*/
	FLOW_FLUSH_NF,
};

enum promote_mode {
	AUTO_PROMOTE = 0,
	MANUAL_PROMOTE_BY_CTMARK,
	MANUAL_PROMOTE_BY_CMD,
};

struct flow_astats {
	atomic64_t rx_packets;
	atomic64_t rx_bytes;
	atomic64_t tx_packets;
	atomic64_t tx_bytes;
} __aligned(4 * sizeof(atomic64_t));

struct link_stats {
	__u64 rx_packets;
	__u64 rx_bytes;
	__u64 tx_packets;
	__u64 tx_bytes;
} __aligned(4 * sizeof(u64));

/* Client DB Structure */
struct flowmgr_db_entry {
	struct hlist_node       hlist;

	struct rcu_head         rcu;
	__u8                    notrack;
	__u8                    steer;           /* Steering enable flag   */
	__u8                    accel;           /* Accel enable flag      */
	unsigned long           created;
	unsigned long           updated;
	int                     iif;             /* Input Interface Index  */
	atomic_t                use;             /* Active Use count       */
	atomic_t                rxflows;
	atomic_t                txflows;
	atomic_t                totrxflows;
	atomic_t                tottxflows;
	atomic_t                nsteers;
	__u8                    addr[6];         /* MAC address            */
	__u16                   vlan_id;
	struct flow_astats      stats;           /* Slow path Stats        */
	struct flow_astats      fstats;          /* Fast path Stats        */
	struct flow_astats      dscp_stats[DSCP_MAX_LIMIT];             /* DSCP Slow path Stats   */
	struct flow_astats      dscp_fstats[DSCP_MAX_LIMIT];            /* DSCP Fast path Stats   */
};

/* Multicast to Unicast Conversion Structure */
struct flowmgr_mc2uc {
	int			oif;             /* Output Interface Index */
	__u8			addr[6];         /* MAC address            */
};

/* Multicast Flow DB Structure */
struct flowmgr_mdb_entry {
	struct hlist_node	hlist;

	struct rcu_head		rcu;
	unsigned long		created;
	__be32			group[4];        /* For IPv4 use group[3]  */
	int			oif[NUM_MCTX];   /* Output Interface Index */
	__u8			num_mc2uc;
	struct flowmgr_mc2uc	mc2uc[MAX_MC2UC];
	int			iif;             /* Input Interface Index  */
	atomic_t		use;             /* Use count              */
	int			flow_id;         /* Flow ID                */
	int			flow_type;       /* Flow Type              */
	unsigned long		packets;         /* Packet Count           */
	unsigned long		bytes;         /* Byte Count           */
};


struct flowmgr_flow_usage {
	u32 add;
	u32 del;
	struct timeval ltv; /* Last flow duration */
	struct timeval mtv; /* Max flow duration */
};

struct flowmgr {
	int enable;
	struct net_device *wandev;
	struct ctl_table_header	*sysctl_header;
	struct flowmgr_flow_usage *flow_usage;
	int max_fap_flows;
	/* Enable/Disable 3-tuple bridge flow */
	int use_3tuple_br_flow;
	/* UDP packet threshold for promotion */
	int udp_pkt_threshold;
	/* TCP packet threshold for promotion */
	int tcp_pkt_threshold;
	/* ESP packet threshold for promotion */
	int esp_pkt_threshold;
	/* AH packet threshold for promotion */
	int ah_pkt_threshold;
	/* Enable/Disable drop connection for no socket */
	int nosock_drop;
	/* Accelerate timeout to check if connection is still alive */
	int fast_death_timeout;
	/* Accelerate TCP flow after 3way handshake */
	int use_tcp_3way_handshake;
	/* Disable acceleration of GRE or MAPT flows */
	/* 1: Disable US Encapsulation */
	/* 2: Disable DS Decapsulation */
	/* 3: Disable US and DS */
	int disable_gre_accel;
	int disable_mapt_accel;
	int enable_br_conntrack_call;
	int enable_expected_flow;
	int expected_flow_timeout;
	int check_status_timeout;
	/* Open TCP/UDP port tracking */
	int enable_tu_port_track;
	int tu_port_track_timeout;
	/* Flow promote mode */
	/* 0: Auto promote */
	/* 1: Manual promote by CT mark */
	/* 2: Manual promote by flowmgr cmd, such as ioctl, proc */
	int promote_mode;
	/* Flow promotion control for direction */
	/* Bit0: LAN2LAN */
	/* Bit1: WAN2LAN : Downstream */
	/* Bit2: LAN2WAN : Upstream */
	int promote_dir_ctl;
	int auto_flush_mac_mode;
	int auto_flush_inf_mode;
	atomic_t flush_worker_count;
	__u32 cnt_start[0];
	/* UDP packets processed by host after flow promotion */
	atomic64_t in_flight_udp_pkt_cnt[2];
	/* TCP packets processed by host after flow promotion */
	atomic64_t in_flight_tcp_pkt_cnt[2];
	/* ESP packets processed by host after flow promotion */
	atomic_t in_flight_esp_pkt_cnt[2];
	/* AH packets processed by host after flow promotion */
	atomic_t in_flight_ah_pkt_cnt[2];
	atomic64_t pkt_processed_cnt;
	atomic64_t pkt_skipoffload_cnt;
	atomic_t pkt_flood_cnt;
	atomic_t pkt_nosock_cnt;
	atomic_t flow_create_cnt;
	atomic_t flow_create_err_cnt;
	atomic_t flow_delete_cnt;
	atomic_t flow_delete_err_cnt;

	atomic_t flow_expected_create_cnt;
	atomic_t flow_expected_create_err_cnt;
	atomic_t flow_expected_delete_cnt;
	atomic_t flow_expected_delete_err_cnt;

	atomic_t flow_mcast_create_cnt;
	atomic_t flow_mcast_create_err_cnt;
	atomic_t flow_mcast_delete_cnt;
	atomic_t flow_ignore_cnt;
	atomic_t flow_tcp_cnt;
	atomic_t flow_udp_cnt;
	atomic_t flow_esp_cnt;
	atomic_t flow_ah_cnt;
	atomic_t flow_flush_mac_cnt;
	atomic_t flow_flush_inf_cnt;
	atomic_t flow_flush_wifi_cnt;
	atomic_t flow_flush_orig_cnt;
	atomic_t flow_flush_repl_cnt;
	atomic_t flow_unlearn_cnt;
	atomic_t flow_unlearn_orig_cnt;
	atomic_t flow_unlearn_repl_cnt;
	atomic_t pkt_ignore_nogroup_out_cnt;
	atomic_t pkt_ignore_nogroup_in_cnt;
	atomic_t pkt_ignore_null_in_cnt;
	atomic_t pkt_ignore_mc_cnt;
	atomic_t pkt_ignore_ipmc_cnt;
	atomic_t pkt_ignore_0port_cnt;
#ifdef CONFIG_BCM_FLOWMGR_FAP_WQ
	atomic_t fap_deferred_cnt;
#endif
	__u32 cnt_end[0];
	spinlock_t lock;
	spinlock_t lock_counter;
	u32 inf_db_info;
	struct proc_dir_entry *proc_dir;
	u32 sw_features;
	u32 hw_features;
	u32 features;
	int wantype;
	struct workqueue_struct *fap_wq;
};

struct flowmgr_tu_port_state {
	unsigned long ts;
};

void flowmgr_ignore_port_show(struct seq_file *s);
int flowmgr_ignore_port_add(u16 port);
int flowmgr_ignore_port_del(u16 port);
void flowmgr_ignore_port_clear(void);
void flowmgr_netdevice_show(struct seq_file *s);
char **flowmgr_netdevice_list(void);
int flowmgr_netdevice_add(char *name, char *type);
int flowmgr_netdevice_del(char *name);
void flowmgr_debug_show(struct seq_file *s);
void flowmgr_debug_counter_show(struct seq_file *s);
void flowmgr_debug_counter_clear(void);
void flowmgr_conntrack_clean_all(void);
void flowmgr_conntrack_clean_by_mapt_domain(int domain);
void flowmgr_set_wandev(struct net_device *dev);
struct net_device *flowmgr_get_wandev(void);
int flowmgr_set_wandev_by_name(char *name);
bool flowmgr_is_dev_tunnel(struct net_device *dev);
bool flowmgr_is_dev_gretap_tunnel(struct net_device *dev);
bool flowmgr_is_dev_ip6gretap_tunnel(struct net_device *dev);
int flowmgr_update_flow_stats(const struct nf_conn *ct,
				  struct nf_conn_offload *ct_offload,
				  int direction, int update);
int flowmgr_update_dev_slow_stats(struct net_device *dev, u32 packets,
				  u32 bytes, int rx, u8 dscp);
int flowmgr_dec_dev_slow_stats(struct net_device *dev, u32 packets,
			       u32 bytes, int rx, u8 dscp);

int flowmgr_db_init(void);
void flowmgr_db_fini(void);
struct flowmgr_db_entry *flowmgr_db_find(const unsigned char *addr);
void flowmgr_db_delete_by_port(int iif);
int flowmgr_db_delete(const unsigned char *addr);
int flowmgr_db_update(const struct net_device *in,
		      const unsigned char *addr, u16 vid);
void flowmgr_db_flush(void);
int flowmgr_db_dec_rxflow(struct flowmgr_db_entry *db);
int flowmgr_db_inc_rxflow(struct flowmgr_db_entry *db);
int flowmgr_db_dec_txflow(struct flowmgr_db_entry *db);
int flowmgr_db_inc_txflow(struct flowmgr_db_entry *db);

int flowmgr_db_clear_stats(struct flowmgr_db_entry *db);
int flowmgr_db_clear_stats_by_addr(unsigned char *addr);
int flowmgr_db_clear_stats_by_port(int iif);
void flowmgr_db_clear(void);

int flowmgr_dbflow_clear_steer_stats(struct flowmgr_db_entry *db);
int flowmgr_dbflow_clear_steer_stats_by_addr(unsigned char *addr);
int flowmgr_dbflow_clear_steer_stats_by_port(int iif);
void flowmgr_dbflow_clear_steer(void);

int flowmgr_dbflow_clear_stats(struct flowmgr_db_entry *db);
int flowmgr_dbflow_clear_stats_by_addr(unsigned char *addr);
int flowmgr_dbflow_clear_stats_by_port(int iif);
void flowmgr_dbflow_clear(void);

u32 flowmgr_dbflow_get_accel (struct flowmgr_db_entry *db);
int flowmgr_dbflow_set_accel(struct flowmgr_db_entry *db, u8 accel);
int flowmgr_dbflow_set_accel_by_addr(unsigned char *addr, u8 accel);
int flowmgr_dbflow_set_accel_by_port(int iif, u8 accel);
void flowmgr_dbflow_set_accel_all(u8 accel);

int flowmgr_dbflow_set_steer(struct flowmgr_db_entry *db, u8 steer);
int flowmgr_dbflow_set_steer_by_addr(unsigned char *addr, u8 steer);
int flowmgr_dbflow_set_steer_by_port(int iif, u8 steer);
void flowmgr_dbflow_set_steer_all(u8 steer);

int flowmgr_db_update_txstats_fast(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp);
int flowmgr_db_update_tx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp);
int flowmgr_db_dec_tx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp);
int flowmgr_db_update_rxstats_fast(struct flowmgr_db_entry *db,
				   unsigned int packets,
				   unsigned int bytes,
				   u8 dscp);
int flowmgr_db_update_rx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp);
int flowmgr_db_dec_rx_slow(const struct net_device *dev,
			      const unsigned char *addr,
			      unsigned int packets,
			      unsigned int bytes,
			      u8 dscp);
int flowmgr_db_get_stats64(struct flowmgr_db_entry *db,
			   struct link_stats *stats);
int flowmgr_db_get_stats64_slow(struct flowmgr_db_entry *db,
				struct link_stats *stats,
				int init);
int flowmgr_db_get_stats64_fast(struct flowmgr_db_entry *db,
				struct link_stats *stats);
int flowmgr_db_set_notrack(unsigned char *addr, enum flow_notrack_type notrack);
int flowmgr_db_set_dscp_notrack(u8 dscp, enum flow_notrack_type dscp_notrack);
int flowmgr_db_get_dscp_stats64_fast(struct flowmgr_db_entry *db,
				struct link_stats *stats, u64 *dscp_cgz);
int flowmgr_db_get_dscp_stats64_slow(struct flowmgr_db_entry *db,
				struct link_stats *stats,
				int init, u64 *dscp_cgz);
int flowmgr_db_get_dscp_stats64(struct flowmgr_db_entry *db,
			   struct link_stats *stats, u64 *dscp_cgz);
struct hlist_node  *flowmgr_db_get_first(int *hashid);
struct hlist_node *flowmgr_db_get_next(int *hashid,
				       struct hlist_node *head);
struct hlist_node *flowmgr_db_get_idx(int *hashid, loff_t pos);

int flowmgr_mdb_init(void);
void flowmgr_mdb_fini(void);
int flowmgr_mdb_insert(const struct net_device *in,
		       const struct net_device *out,
		       const __be32 *group,
		       const __u8 *mc2uc);
int flowmgr_mdb_delete(const struct net_device *in,
		       const struct net_device *out,
		       const __be32 *group,
		       const __u8 *mc2uc);
void flowmgr_mdb_delete_by_out_port(const struct net_device *out);
void flowmgr_mdb_flush(void);
struct hlist_node  *flowmgr_mdb_get_first(int *hashid);
struct hlist_node *flowmgr_mdb_get_next(int *hashid,
					struct hlist_node *head);
struct hlist_node *flowmgr_mdb_get_idx(int *hashid, loff_t pos);
int flowmgr_mdb_update_all_mcast_stats(void);
int flowmgr_mdb_update_dev_mcast_stats(struct net_device *dev);

int flowmgr_mcast_init(void);
void flowmgr_mcast_fini(void);
int flowmgr_mc2uc_enable(int enable);
int flowmgr_is_mc2uc(void);
int flowmgr_mcast_counter_start(void);

#ifdef CONFIG_BCM_FLOWMGR_IOCTL
int flowmgr_cdev_init(void);
void flowmgr_cdev_exit(void);
#endif

extern struct flowmgr flowmgr;
int is_flowmgr_db_ifenable(struct net_device *dev);
void flowmgr_db_ifenable(struct net_device *dev, int enable);

void flowmgr_enable_br_conntrack(int enable);
int flowmgr_flow_flush(struct net *net, const char *addr,
		       int ifindex, int type);
int flowmgr_flow_flush_wifi(struct net *net, const char *addr,
			    int flring, int mode);

int flowmgr_trace_init(void);
void flowmgr_trace_exit(void);
int flowmgr_trace_add_entry(char type, void *tuple, void *info);
int flowmgr_trace_show(struct seq_file *s);

int is_flowmgr_manual_enable(void);
void flowmgr_manual_enable(int enable);
void flowmgr_manual_trigger_clear(struct nf_conn *ct,
				  void *offload_info,
				  int direction);
int flowmgr_manual_trigger_check(struct nf_conn *ct, struct sk_buff *skb,
				 u8 start_promote);
int flowmgr_manual_promote(void *tuple, struct nf_conn *ct);
int flowmgr_manual_demote(void *tuple, struct nf_conn *ct);
int flowmgr_manual_init(void);
void flowmgr_manual_fini(void);

#define flowmgr_is_feature_enabled(f) (flowmgr.features & f)
void flowmgr_features_show(struct seq_file *s, char *name, int features);

int flowmgr_debug_init(void);
void flowmgr_debug_exit(void);
int flowmgr_ctdebug_check(struct nf_conn *ct,
			  void *ct_offload,
			  int dir);
int flowmgr_cttrace_check(void *ct_tuple, void *info);
int flowmgr_tu_port_test(u16 dport);
void flowmgr_tu_port_clear(u16 dport);
void flowmgr_tu_port_show(struct seq_file *s);
struct flowmgr_tu_port_state *flowmgr_tu_port_table(void);

void flowmgr_update_offload_retry_timeout(struct nf_conn *ct,
				struct nf_conn_offload *ct_offload);

#ifdef CONFIG_BCM_FLOWMGR_ARL
int flowmgr_arl_init(void);
void flowmgr_arl_exit(void);
#endif
#endif
