 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/if_vlan.h>
#include <linux/cpumask.h>
#include <linux/proc_fs.h>
#include <bcmnethooks.h>
#include <linux/brcmxcvr.h>
#include <net/ip.h>
#include <net/dst.h>
#include <net/xfrm.h>
#include <uapi/linux/if_bridge.h>
#if defined(CONFIG_BCM_PWR_RPC)
#include <brcm_pwr_rpc.h>
#endif
#include "dqnet.h"
#include "dqnet_priv.h"
#include "dqnet_dt.h"
#include "dqnet_ethtool.h"
#include "dqnet_dbg.h"
#include "dqnet_ethstats.h"
#include "dqnet_nethooks.h"
#include "dqnet_procfs.h"
#include "dqnet_wifi.h"
#include "dqnet_fap.h"
#include "dqnet_qmsg.h"
#include "dqnet_switch.h"
#include "dqnet_brcmtag.h"
#include "dqnet_svtag.h"
#include "dqm.h"
#include "fpm.h"
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
#include "dqskb.h"
#endif
#include "misc_services.h"

#define xstr(s) str(s)
#define str(s) #s
#define MFG_MAC_ADDR			{0x02, 0x10, 0x18}
#define DQNET_NAPI_WEIGHT_DEF		256
#define DQNET_NETDEV_BUDGET_DEF 	1000
#define DQNET_NETDEV_MAX_BACKLOG_DEF	3000
#define DQNET_WATCHDOG_TIMEOUT		(5*HZ)
#define ETH_CRC_LEN			(4)
#define SKB_CACHE_ALIGN_PAD		(256)

#define MIN_MTU_SIZE			64
#define MAX_MTU_SIZE			2000
#define IEEE_MAX_MTU_SIZE		9216

#if defined(CONFIG_BCM_PWR_RPC)
struct workqueue_struct *workqueue;
#endif

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
static
int swport_imp_call(struct ctl_table *ctl, int write,
		    void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	struct net_device *dev = ctl->extra1;
	int ret;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (!write) {
		goto done;
	}

	if (val < 0)
		ret = dqnet_fap_set_imp_lag_port(dev, 0,
						 DQNET_SET_IMP_AUTO);
	else
		ret = dqnet_fap_set_imp_lag_port(dev, val,
						 DQNET_SET_IMP_STATIC);
done:
	if (ret)
		*ppos = pos;
	return ret;
}
#endif


static int proc_phy_power_ctrl(struct ctl_table *ctl, int write,
		    void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	struct net_device *dev = ctl->extra1;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct phy_device *phydev = ndev->phydev;
	int ret = 0;

	if (!phydev && write)
		return 0;

	lctl = *ctl;
	if (!write)
		*valp = ndev->phy_power;
	else
		lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);
	if (write) {
		switch (val) {
		case HAL_LINK_STOP:
		case HAL_LINK_OPEN:
		case HAL_LINK_STOP_EXIT:
		case HAL_LINK_INIT_OPEN:
			ret |= haldev_config_link(&ndev->hal, val);
			ndev->phy_power = val;
			break;
		default:
			ret = -EINVAL;
		}
	}

	if (ret)
		*ppos = pos;

	return ret;
}

static int proc_phy_svtag(struct ctl_table *ctl, int write,
						 void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	struct net_device *dev = ctl->extra1;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int ret = 0;

	lctl = *ctl;
	if (!write)
		*valp = ntohl(ndev->svtag);
	else
		lctl.data = &val;

	ret = proc_douintvec(&lctl, write, buffer, lenp, ppos);
	if (write) {
		ndev->svtag = htonl(val);
	}

	if (ret)
		*ppos = pos;
	return ret;
}

static
int isolate_call(struct ctl_table *ctl, int write,
		 void __user *buffer, size_t *lenp, loff_t *ppos)
{
	int *valp = ctl->data;
	int val = *valp;
	loff_t pos = *ppos;
	struct ctl_table lctl;
	struct net_device *dev = ctl->extra1;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int ret, i, j;

	lctl = *ctl;
	lctl.data = &val;

	ret = proc_dointvec(&lctl, write, buffer, lenp, ppos);

	if (!write) {
		goto done;
	}

	if (dev->priv_flags & IFF_BRIDGE_PORT) {
		pr_err("DQNET: Remove ethernet %s from bridge before setting isolate value\n",
		       dev->name);
		return -1;
	}

	ret = dqnet_fap_set_isolate(dev, val);
	if (!ret && write) {
		*valp = val;
		i = ndev->if_sub_id;
		ndev->brcmtag_opc_uc = BRCM_OPCODE_1;
		for (j = 0; j < MAX_PHY_PORTS; j++) {
			if (i == j)
				continue;
			dqnet_swport_fwd[i][j] = val ? 1 : 0;
			dqnet_swport_fwd[j][i] = val ? 1 : 0;
		}
	}
done:
	if (ret)
		*ppos = pos;
	return ret;
}

static struct ctl_table ndev_sysctl_tbl[] = {
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	DQNET_DEV_SYSCTL_ENTRY(bmcast_enable,
			"enable_broadcast_multicast", 0644, proc_dointvec),
	DQNET_DEV_SYSCTL_ENTRY(brcmtag_opc_mc,
			"brcmtag_opcode_multicast", 0644, proc_dointvec),
	DQNET_DEV_SYSCTL_ENTRY(brcmtag_opc_uc,
			"brcmtag_opcode_unicast", 0644, proc_dointvec),
	DQNET_DEV_SYSCTL_ENTRY(swport_imp,
			"swport_imp", 0644, swport_imp_call),
#endif
#ifdef FPM_IN_SKB
	DQNET_DEV_SYSCTL_ENTRY(fpm_in_skb,
			"enable_fpm_in_skb", 0644, proc_dointvec),
#endif
	DQNET_DEV_SYSCTL_ENTRY(tx_fpm,
			"enable_tx_fpm", 0644, proc_dointvec),
	DQNET_DEV_SYSCTL_ENTRY(phy_power,
			"phy_power", 0644, proc_phy_power_ctrl),
	DQNET_DEV_SYSCTL_ENTRY(svtag,
			"svtag", 0644, proc_phy_svtag),
	DQNET_DEV_SYSCTL_ENTRY(isolate,
			"isolate", 0644, isolate_call),
	{}
};

struct net_bridge;

static struct net_device napi_dev;
struct dqnet_netdev ndevs;
static struct dqnet_list ndevs_open;
struct dqnet_list chans;
int if_id_max, if_sub_id_max;
struct dqnet_netdev ***demux_tbl;
spinlock_t demux_tbl_lock;
struct dqnet_err_stats err_stats;
u8 dqnet_swport_fwd[MAX_PHY_PORTS][MAX_PHY_PORTS];

static u8 mfg_mac_addr[ETH_MFG_ALEN] = MFG_MAC_ADDR;

static int dqnet_napi_weight = DQNET_NAPI_WEIGHT_DEF;
#ifndef MODULE
static int __init dqnet_napi_weight_setup(char *str)
{
	get_option(&str, &dqnet_napi_weight);
	return 1;
}
__setup("dqnet_napi_weight_max=", dqnet_napi_weight_setup);
#else
module_param_named(napi_weight, dqnet_napi_weight, int, 0644);
MODULE_PARM_DESC(napi_weight, "NAPI Weight Max");
#endif
#if defined(CONFIG_BCM_RPC_SERVICES)
extern int (*rpc_net_handler)(int dqm_tunnel,
			      struct misc_msg *msg);
#endif

struct net_device *dqskbdev;


/*
 * Passes an SKB to the netdev specified.
 *
 * Parameters
 *      ndev	private network device structure
 *      fb	packet buffer descriptor
 *
 * Returns
 *  success (0), don't free FPM (1), error (< 0)
 */
int dqnet_rx_skb(struct dqnet_netdev *ndev,
		 struct fpm_buff *fb)
{
	int status = 0;
	enum bcm_nethook_result result;
	struct sk_buff *skb = fb->skb;
	int pkt_len;

	netdev_dbg(ndev->dev, "-->\n");

#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
	if (skb->dev && (fb->type == BUF_TYPE_SKB)) {
		dqskbdev = skb->dev;
		dqskb_prepare_skb(dqskbdev, skb);
	}
#endif

	if (!netif_running(ndev->dev)) {
		netif_dbg(ndev, rx_err, ndev->dev,
			  "===============================================\n");
		netif_dbg(ndev, rx_err, ndev->dev,
			  "DROP: interface is down.\n");
		show_rx_dbg_pkt_desc(ndev, fb);
		show_rx_dbg_pkt(ndev, skb->data, skb->len);
		netif_dbg(ndev, rx_err, ndev->dev,
			  "===============================================\n");
		status = -EIO;
		goto done;
	}

	/* Remove BRCM tag before calling nethooks */
	if (ndev->brcm_tag) {
		status = dqnet_rm_brcm_tag(ndev, fb);
		if (status) {
			dev_kfree_skb_any(skb);
			ndev->stats.netdev.rx_errors++;
			ndev->stats.netdev.rx_dropped++;
			goto done;
		}
	}
	skb->priority = fb->priority;
	skb->dev = ndev->dev;
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	skb->dev_in = ndev->dev;
#endif
	skb_record_rx_queue(skb, fb->queue_id);

	netif_err(ndev, rx_status, ndev->dev,
		  "===============================================\n");
	netif_err(ndev, rx_status, ndev->dev, "RX packet\n");
	show_rx_pkt_desc(ndev, fb);
	show_rx_pkt(ndev, skb->data, skb->len);
	netif_err(ndev, rx_status, ndev->dev,
		  "===============================================\n");

	pkt_len = skb->len;
	skb->protocol = eth_type_trans(skb, ndev->dev);
	result = bcm_nethook_call_hooks(ndev->dev, BCM_NETHOOK_RX_SKB, skb);
	switch (result) {
	case BCM_NETHOOK_DROP:
		netif_dbg(ndev, drop, ndev->dev,
			  "===============================================\n");
		netif_dbg(ndev, drop, ndev->dev,
			  "DROP: RX SKB nethook\n");
		show_drop_pkt_desc(ndev, fb);
		show_drop_pkt(ndev, skb->data, skb->len);
		netif_dbg(ndev, drop, ndev->dev,
			  "===============================================\n");
		dev_kfree_skb_any(skb);
		ndev->stats.netdev.rx_dropped++;
		goto done;
	case BCM_NETHOOK_CONSUMED:
		ndev->stats.netdev.rx_packets++;
		ndev->stats.netdev.rx_bytes += pkt_len;
		ndev->stats.rx_nethook_consumed++;
		break;
	case BCM_NETHOOK_PASS:
	case BCM_NETHOOK_SKIP:
	default:
		ndev->stats.netdev.rx_packets++;
		ndev->stats.netdev.rx_bytes += pkt_len;
		if (skb->pkt_type == PACKET_BROADCAST ||
		    skb->pkt_type == PACKET_MULTICAST)
			ndev->stats.netdev.multicast++;
		netif_receive_skb(skb);
		break;
	}

done:
	netdev_dbg(ndev->dev, "<--\n");
	return status;
}

#ifdef FPM_IN_SKB
int dqnet_fpm_recycle(struct sk_buff *skb)
{
	u32 token = (u32) skb->recycle_arg;
	if (fpm_is_fpm_buf(skb->head) && fpm_is_valid_token(token)) {

		fpm_invalidate_token(token, 0, 0, 0);

		fpm_free_token(token);
		pr_debug("dqnet_fpm_recycle: %px %X %px\n", skb, skb->recycle_arg, skb->recycle);
		skb->recycle = 0;
		skb->recycle_arg = 0;
		skb->head = 0;
		skb->cloned = 0;
		atomic_inc(&err_stats.fpm_in_skb_free);
		return 0;
	}
	dump_stack();
	pr_err("%s: %x Error <--\n", skb->dev->name, (u32)skb->recycle_arg);
	return -1;
}

static inline bool is_fpm_in_skb_allow(struct fpm_buff *fb)
{
	struct ethhdr *eh = (struct ethhdr *) fb->data;
	__be16 eth_proto = ntohs(eh->h_proto);

	if (is_multicast_ether_addr(eh->h_dest))
		return false;

	if (eth_proto == ETH_P_IP) {
		struct iphdr *iph =  (struct iphdr *) (fb->data + ETH_HLEN);
		if (ip_is_fragment(iph))
			return false;
		if ((iph->protocol == IPPROTO_TCP) ||
		    (iph->protocol == IPPROTO_UDP))
			return true;
	} else if (eth_proto == ETH_P_IPV6) {
		struct ipv6hdr *iph =  (struct ipv6hdr *) (fb->data + ETH_HLEN);
		if (iph->nexthdr == NEXTHDR_FRAGMENT)
			return false;
		if ((iph->nexthdr == IPPROTO_TCP) ||
		    (iph->nexthdr == IPPROTO_UDP))
			return true;
	}

	return false;
}

struct sk_buff *dqnet_build_skb(struct fpm_buff *fb)
{
	struct skb_shared_info *shinfo;
	struct sk_buff *skb;
	unsigned int size = fb->len;
	int headroom = fb->data - fb->buf;
	int shinfo_in_headroom = 1;
	unsigned int shinfo_size = SKB_DATA_ALIGN(sizeof(struct skb_shared_info));
	int token_size = 0;
	int len = headroom + fb->len;

	/* Estimate the token size based on packet lenght and headroom */
	if (len < 512)
		token_size = 512;
	else if (len < 1024)
		token_size = 1024;
	else if (len < 2048)
		token_size = 2048;
	else if (len < 4096)
		token_size = 4096;

	/* Can shared info fit into headroom? */
	if (headroom < shinfo_size) {
		shinfo_in_headroom = 0;
		if ((len + shinfo_size) > token_size)
			return NULL;
	}

	fpm_set_token_size(&fb->token, token_size);

	skb = build_skb(fb->buf, size);
	if (!skb)
		return NULL;

	skb->truesize = SKB_TRUESIZE(size);
	skb->head = fb->buf;
	skb->data = fb->data;
	skb_reset_tail_pointer(skb);
	skb->end = skb->tail + size;
	skb->mac_header = (typeof(skb->mac_header))~0U;
	skb->transport_header = (typeof(skb->transport_header))~0U;
	skb->recycle = dqnet_fpm_recycle;
	skb->recycle_arg = fb->token;
	skb->cloned = 1;  /* This is needed */

	if (shinfo_in_headroom) {
		skb->recycle_shinfo = skb->head;
		skb->head += shinfo_size;
		atomic_inc(&err_stats.fpm_in_skb_shinfo_head);
	} else if (token_size) {
#ifdef NET_SKBUFF_DATA_USES_OFFSET
		skb->end = (sk_buff_data_t)(token_size - shinfo_size);
		skb->recycle_shinfo = skb->head + skb->end;
#else
		skb->end = (sk_buff_data_t)(fb->buf + token_size - shinfo_size);
		skb->recycle_shinfo = skb->end;
#endif
	} else {
		BUG();
	}
	skb->recycle_hw_data = 1;
	/* make sure we initialize shinfo sequentially */
	shinfo = skb_shinfo(skb);
	memset(shinfo, 0, offsetof(struct skb_shared_info, dataref));
	atomic_set(&shinfo->dataref, 1);

	fb->skb = skb;
	fb->token = 0;
	fb->type = BUF_TYPE_SKB;
	return skb;
}

int dqnet_rx_fpm_in_skb(struct dqnet_netdev *ndev,
			struct fpm_buff *fb)
{
	int status = 0;
	struct sk_buff *skb;

	if (!is_fpm_in_skb_allow(fb))
	    return -1;

	skb = dqnet_build_skb(fb);
	if (!skb)
	    return -1;

	skb_put(skb, fb->len);

	status = dqnet_rx_skb(ndev, fb);
	atomic_inc(&err_stats.fpm_in_skb);

	return status;
}
#endif

/*
 * Packages the packet into an SKB.
 *
 * Parameters
 *      ndev	private network device structure
 *      fb	packet buffer descriptor
 *
 * Returns
 *  success (0), don't free FPM (1), error (< 0)
 */
int dqnet_rx_fpm(struct dqnet_netdev *ndev,
		 struct fpm_buff *fb)
{
	int status = 0;
	enum bcm_nethook_result result;
	struct sk_buff *skb;
	union tok_src_dest tok_src;
	struct pcpu_sw_netstats *tstats = NULL;

	netdev_dbg(ndev->dev, "-->\n");

	if (!netif_running(ndev->dev)) {
		netif_dbg(ndev, rx_err, ndev->dev,
			  "===============================================\n");
		netif_dbg(ndev, rx_err, ndev->dev,
			  "DROP: interface is down.\n");
		show_rx_dbg_pkt_desc(ndev, fb);
		show_rx_dbg_pkt(ndev, fb->data, fb->len);
		netif_dbg(ndev, rx_err, ndev->dev,
			  "===============================================\n");
		status = -EIO;
		goto done;
	}

	tok_src.data = 0;
	tok_src.iface.id = fb->if_id;
	tok_src.iface.sub_id = fb->if_sub_id;
	fpm_track_token_src(fb->token, &tok_src);
	fb->dev_in = ndev->dev;
	/* Remove BRCM tag before calling nethooks */
	if (ndev->brcm_tag) {
		status = dqnet_rm_brcm_tag(ndev, fb);
		if (status) {
			ndev->stats.netdev.rx_errors++;
			ndev->stats.netdev.rx_dropped++;
			goto done;
		}
	}

	result = bcm_nethook_call_hooks(ndev->dev, BCM_NETHOOK_RX_FPM, fb);
	switch (result) {
	case BCM_NETHOOK_DROP:
		netif_dbg(ndev, drop, ndev->dev,
			  "===============================================\n");
		netif_dbg(ndev, drop, ndev->dev,
			  "DROP: RX FPM nethook\n");
		show_drop_pkt_desc(ndev, fb);
		show_drop_pkt(ndev, fb->data, fb->len);
		netif_dbg(ndev, drop, ndev->dev,
			  "===============================================\n");
		ndev->stats.netdev.rx_dropped++;
		goto done;
	case BCM_NETHOOK_CONSUMED:
		ndev->stats.rx_nethook_consumed++;
		ndev->stats.netdev.rx_packets++;
		ndev->stats.netdev.rx_bytes += fb->len;
		if (ndev->nethooks.offload_stats.slow) {
			tstats = this_cpu_ptr(ndev->nethooks.offload_stats.slow);
			if (tstats) {
				u64_stats_update_begin(&tstats->syncp);
				tstats->rx_packets++;
				tstats->rx_bytes += fb->len;
				u64_stats_update_end(&tstats->syncp);
			}
		}
		status = 1;
		goto done;
	case BCM_NETHOOK_PASS:
	case BCM_NETHOOK_SKIP:
	default:
		break;
	}

#ifdef FPM_IN_SKB
	if (ndev->fpm_in_skb) {
		if (dqnet_rx_fpm_in_skb(ndev, fb) == 0)
			goto done;
	}
#endif

	skb = dev_alloc_skb(fb->len /*+ SKB_CACHE_ALIGN_PAD + 2*/);
	if (!skb) {
		netif_err(ndev, rx_err, ndev->dev,
			  "===============================================\n");
		netif_err(ndev, rx_err, ndev->dev,
			  "DROP: unable to alloc skb.\n");
		show_rx_err_pkt_desc(ndev, fb);
		show_rx_err_pkt(ndev, fb->data, fb->len);
		netif_err(ndev, rx_err, ndev->dev,
			  "===============================================\n");
		ndev->stats.netdev.rx_errors++;
		ndev->stats.netdev.rx_dropped++;
		status = -ENOMEM;
		goto done;
	}
	fb->skb = skb;

	/*skb_reserve(skb, SKB_CACHE_ALIGN_PAD + 2);*/
	memcpy(skb->data, fb->data, fb->len);
	skb_put(skb, fb->len);

	status = dqnet_rx_skb(ndev, fb);

done:
	netdev_dbg(ndev->dev, "<--\n");
	return status;
}

int dqnet_rx_fpm_or_skb(struct dqnet_netdev *ndev,
			struct fpm_buff *fb)
{
	if (fb->type == BUF_TYPE_FPM)
		return dqnet_rx_fpm(ndev, fb);
	else if (fb->type == BUF_TYPE_SKB)
		return dqnet_rx_skb(ndev, fb);
	return -EIO;
}

/*
 * The FAP exception Q is muxed with packets from various devices that
 * we have to demux to the correct device. These packets are demuxed
 * via an interface ID. For some interfaces they must be further demuxed
 * using an interface sub-ID. For switch interfaces they must be further
 * demuxed using either an ARL lookup or a BRCM tag to determine the port #.
 * Since we don't know until we have a dqnet_netdev how to demux beyond
 * the interface ID we first retrieve a dqnet_netdev from the demux table
 * using the interface ID and sub-ID of if_sub_id_max that we ensured
 * had an entry when devices were probed.
 *
 * 	chan	DQM channel on which packet was received
 *      fb	FPM buffer descriptor
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_demux_if_id(struct dqnet_channel *chan,
			     struct fpm_buff *fb)
{
	int status = 0;
	unsigned long flags;
	struct dqnet_netdev *ndev;

	pr_debug("-->\n");

	spin_lock_irqsave(&demux_tbl_lock, flags);
	ndev = demux_tbl[fb->if_id][if_sub_id_max];
	spin_unlock_irqrestore(&demux_tbl_lock, flags);

	if (!ndev) {
		pr_debug("================================================\n");
		pr_debug("DROP: no device for interface ID %d.\n", fb->if_id);
		pr_debug("Chan %s RX packet\n", chan->name);
		show_dbg_pkt_desc(fb);
		if (fb->type == BUF_TYPE_FPM)
			show_dbg_pkt(fb->data, fb->len);
		else
			show_dbg_pkt(fb->skb->data, fb->skb->len);
		pr_debug("================================================\n");
		err_stats.if_id_lookup++;
		status = -EIO;
		goto done;
	}
	fb->brcm_tag_len = dqnet_get_brcm_tag_len(ndev);
	fb->sv_tag_len = dqnet_get_sv_tag_len(ndev);

	if (ndev->demux)
		status = ndev->demux(chan, fb);
	else
		status = chan->rx_pkt(ndev, fb);

done:
	pr_debug("<--\n");
	return status;
}

/*
 * The FAP exception Q is muxed with packets from various devices that
 * we have to demux to the correct device. Here we demux using
 * the interface ID and sub-ID from the packet descriptor to index into
 * the demux table to get the dqnet_netdev.
 *
 * 	chan	DQM channel on which packet was received
 *      fb	FPM buffer descriptor
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_demux_if_sub_id(struct dqnet_channel *chan,
				 struct fpm_buff *fb)
{
	int status = 0;
	unsigned long flags;
	struct dqnet_netdev *ndev;

	pr_debug("-->\n");

	spin_lock_irqsave(&demux_tbl_lock, flags);
	ndev = demux_tbl[fb->if_id][fb->if_sub_id];
	spin_unlock_irqrestore(&demux_tbl_lock, flags);

	if (!ndev) {
		pr_debug("================================================\n");
		pr_debug("DROP: no device for interface ID %d, sub ID %d.\n",
			 fb->if_id, fb->if_sub_id);
		pr_debug("Chan %s RX packet\n", chan->name);
		show_dbg_pkt_desc(fb);
		if (fb->type == BUF_TYPE_FPM)
			show_dbg_pkt(fb->data, fb->len);
		else
			show_dbg_pkt(fb->skb->data, fb->skb->len);
		pr_debug("================================================\n");
		err_stats.if_sub_id_lookup++;
		status = -EIO;
		goto done;
	}
	fb->brcm_tag_len = dqnet_get_brcm_tag_len(ndev);
	fb->sv_tag_len = dqnet_get_sv_tag_len(ndev);

	status = chan->rx_pkt(ndev, fb);

done:
	pr_debug("<--\n");
	return status;
}

/*
 * Read up to budget messages from DQM and send them for processing. The RX
 * DQM's upto max_queue for this channel are searched from highest to lowest
 * priority until up to budget packets are found. The packets are then sent for
 * further processing.
 *
 * Returns
 *	# of messages received
 */
static int dqnet_rx_q_msg(struct dqnet_channel *chan, int max_queue, int budget)
{
	int status = 0;
	int count = 0;
	int i, queue, msg_read;
	char str[128];
	struct fpm_buff fb;

	pr_debug("%s -->\n", chan->name);
	if (max_queue > chan->rx_q_count)
		max_queue = chan->rx_q_count;

	for (queue = 0; queue < max_queue; queue++) {
		msg_read = dqm_rx(chan->rx_q[queue],
				  budget - count,
				  chan->rx_q_info[queue].msg_size,
				  &chan->msgs[0]);
		if (msg_read < 0) {
			pr_err("Error receiving DQM %d\n on %s.\n",
				chan->rx_q_info[queue].q_num, chan->name);
			err_stats.dqm_rx++;
		} else {
			struct dqnet_q_info *qi = &chan->rx_q_info[queue];
			int msg_idx;
			int msg_size = qi->msg_size;
			for (i = 0, msg_idx = 0; i < msg_read; i++, msg_idx += msg_size) {
				if (chan->decode_q_msg(chan, &chan->msgs[msg_idx], &fb) < 0) {
					int ii;
					pr_err("Bad MSG: ");
					for (ii = 0; ii < msg_size; ii++)
						pr_err("%08x ", chan->msgs[msg_idx + ii]);
					pr_err("\n");
					snprintf(str, sizeof(str),"Bad Q MSG %d idx %d received on chan %s \n",
						 i, msg_idx, chan->name);
					err_stats.bad_q_msg++;
					goto err_drop_msg;
				}
				fb.queue_id = queue;
				if (fb.type == BUF_TYPE_FPM) {
#ifdef FPM_IN_SKB
					fpm_invalidate_token(fb.token, fb.offset, 0,
							     FPM_SYNC_HEAD);
#else
					fpm_invalidate_token(fb.token, fb.offset, 0,
							     FPM_SYNC_HEAD);
#endif
				} else {
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
					dqskb_invalidate_buffer(fb.skb->data, fb.skb->len);
#endif
				}

				if (chan->demux)
					status = chan->demux(chan, &fb);
				else
					status = chan->rx_pkt(chan->ndev, &fb);

				if (status > 0)
					continue;

				goto free_buffer;

err_drop_msg:
				pr_err("=================================================\n");
				pr_err("DROP: %s\n", str);
				show_err_pkt_desc(&fb);
				pr_err("=================================================\n");

free_buffer:
				if (fb.type == BUF_TYPE_FPM) {
					if (fpm_is_valid_token(fb.token)) {
						fpm_invalidate_token(fb.token,
								     fb.offset, 0,
								     FPM_SYNC_HEAD);
						fpm_free_token(fb.token);
					}
				}
			}
			count += msg_read;
		}
	}

	pr_debug("<--\n");
	return count;
}

/*
 * NAPI polling function for DQM RX messages
 */
static int dqnet_poll_napi(struct napi_struct *napi, int budget)
{
	struct dqnet_channel *chan =
		container_of(napi, struct dqnet_channel, napi);
	int work = 0, queue;
	unsigned long flags;
	int i;
	void (*napi_complete_hook)(void *);

	pr_debug("%s -->\n", chan->name);

	for (queue = 0; queue < chan->rx_q_count; queue++) {
		work += dqnet_rx_q_msg(chan, queue+1, budget-work);
		if (work >= budget)
			goto done;
	}
	if (work < budget) {
		napi_complete(napi);

		spin_lock_irqsave(&chan->rx_irq_lock, flags);
		for (i = 0; i < chan->rx_q_count; i++)
			dqm_enable_rx_cb(chan->rx_q[i]);
		spin_unlock_irqrestore(&chan->rx_irq_lock, flags);
	}
done:
	napi_complete_hook = rcu_dereference(chan->napi_complete.napi_complete_hook);
	if (napi_complete_hook && atomic_read(&chan->napi_complete.refcnt))
		napi_complete_hook(chan);

	pr_debug("<--\n");
	return work;
}

/*
 * Callback for all DQM RX not-empty events from the DQM driver. These
 * are used to know when there is a DQM message waiting to be read.
 *
 * Parameters
 *	q_h	DQM handle returned when the DQM was registered
 *	context	passed in when DQM was registered. In our case
 *      	it is a pointer the channel
 *      flags	indicates whether cb is for RX or TX
 */
static irqreturn_t dqnet_rx_isr(void *q_h, void *context, u32 flags)
{
	struct dqnet_channel *chan = context;
	unsigned long f;
	int i;

	pr_debug("IRQ --> q_h %p, chan %s\n", q_h, chan->name);

	spin_lock_irqsave(&chan->rx_irq_lock, f);
	for (i = 0; i < chan->rx_q_count; i++)
		dqm_disable_rx_cb(chan->rx_q[i]);
	spin_unlock_irqrestore(&chan->rx_irq_lock, f);

	napi_schedule(&chan->napi);

	pr_debug("IRQ <--\n");
	return IRQ_HANDLED;
}

/*
 * Callback handler for all DQM TX low-watermark events from the DQM
 * driver. These are used to know when there is room in the DQM for
 * additional messages (packets) to be sent.
 *
 * Parameters
 *	q_h	DQM handle returned when the DQM was registered
 *	context passed in when DQM was registered. In our case
 *		it is a pointer the tx_busy list of
 *      	dqnet_netdev's for this DQM.
 *      flags	indicates whether cb is for RX or TX
 */
static irqreturn_t dqnet_tx_isr(void *q_h, void *context, u32 flags)
{
	struct dqnet_list *tx_ndevs = context;
	struct list_head *pos, *pos_tmp;
	struct dqnet_netdev *ndev;
	unsigned long f;

	pr_debug("IRQ --> q_h %p\n", q_h);

	dqm_disable_tx_cb(q_h);
	spin_lock_irqsave(&tx_ndevs->lock, f);
	list_for_each_safe(pos, pos_tmp, &tx_ndevs->list)
	{
		ndev = list_entry(pos, struct dqnet_netdev, tx_busy_list);
		netif_wake_queue(ndev->dev);
		list_del(pos);
	}
	spin_unlock_irqrestore(&tx_ndevs->lock, f);

	pr_debug("IRQ <--\n");
	return IRQ_HANDLED;
}

/*
 * Transmit a buffer
 *
 * TODO: use TOS bits to determine which priority Q to use. For
 * now always use lowest priority Q.
 *
 * Parameters
 *	fb	buffer to send
 *	dev	network device which is sending
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_tx_buf(struct fpm_buff *fb, struct net_device *dev)
{
	int status = NETDEV_TX_OK;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	u32 msgdata[DQNET_MAX_MSGSZ];
	unsigned long flags;
	int tx_prio;
	enum dqnet_q_type q_type;
	struct list_head *pos;
	struct dqnet_netdev *ndev_tmp;
	bool ndev_found;
	union tok_src_dest tok_dest;
	bool free_buf = false;

	pr_debug("%s:-->\n",dev->name);

	fb->fap_tag_len = dqnet_fap_get_tag_len(dev);
	fb->brcm_tag_len = dqnet_get_brcm_tag_len(ndev);
	fb->sv_tag_len = dqnet_get_sv_tag_len(ndev);
	status = dqnet_add_sv_tag(ndev, fb);
	if (status)
		return status;
	status = dqnet_add_brcm_tag(ndev, fb);
	if (status)
		return status;
	status = dqnet_fap_add_tag(dev, fb);
	if (status)
		return status;

	fpm_set_token_size(&fb->token, fb->len);

	netif_err(ndev, tx_done, dev,
		  "===============================================\n");
	netif_err(ndev, tx_done, dev, "TX packet\n");
	show_tx_pkt_desc(ndev, fb);
	if (fb->type == BUF_TYPE_FPM)
		show_tx_pkt(ndev, fb->data, fb->len);
	else
		show_tx_pkt(ndev, fb->skb->data, fb->skb->len);
	netif_err(ndev, tx_done, dev,
		  "===============================================\n");

	/* Encoder might need to read the packet content. */
	/* So it needs to run before token invalidation */
	status = ndev->chan->encode_q_msg(ndev, fb, msgdata);

	/* Do not touch buffer contents after this point. */
	if (fb->type == BUF_TYPE_FPM) {
		tok_dest.data = 0;
		tok_dest.iface.id = fb->if_id;
		tok_dest.iface.sub_id = fb->if_sub_id;
		fpm_track_token_dest(fb->token, &tok_dest);
		fpm_track_token_tx(fb->token);

		fpm_flush_invalidate_token(fb->token, fb->offset,
					   ndev->tail_pad,
					   FPM_SYNC_HEAD | FPM_SYNC_TAIL);
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
	} else {
		phys_addr_t paddr;
		dma_addr_t dma;
		dma = dma_map_single(&ndev->dev->dev, fb->skb->data,
				     fb->skb->end - fb->skb->data,
				     DMA_TO_DEVICE);
		paddr = dma;
		memcpy(fb->skb->cb, &paddr, sizeof(phys_addr_t));
		memcpy(fb->skb->cb + sizeof(phys_addr_t), &paddr,
		       sizeof(phys_addr_t));
#endif
	}

	if (status) {
		if (status != -ECOMM) {
			netif_err(ndev, tx_err, dev,
				"===============================================\n");
			netif_err(ndev, tx_err, dev, "DROP: invalid channel type.\n");
			show_tx_err_pkt_desc(ndev, fb);
			if (fb->type == BUF_TYPE_FPM)
				show_tx_err_pkt(ndev, fb->data, fb->len);
			else
				show_tx_err_pkt(ndev, fb->skb->data, fb->skb->len);
			netif_err(ndev, tx_err, dev,
				  "===============================================\n");
		}
		ndev->stats.netdev.tx_errors++;
		ndev->stats.netdev.tx_dropped++;
		free_buf = true;
		status = NETDEV_TX_OK;
		goto done;
	}

	status = dqnet_fap_get_tx_prio(dev, &tx_prio);
	if (status) {
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		netif_err(ndev, tx_err, dev,
			  "DROP: erroneous channel priority.\n");
		show_tx_err_pkt_desc(ndev, fb);
		if (fb->type == BUF_TYPE_FPM)
			show_tx_err_pkt(ndev, fb->data, fb->len);
		else
			show_tx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		ndev->stats.netdev.tx_errors++;
		ndev->stats.netdev.tx_dropped++;
		free_buf = true;
		status = NETDEV_TX_OK;
		goto done;
	}
	status = dqnet_fap_get_tx_q_type(dev, &q_type);
	if (status) {
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		netif_err(ndev, tx_err, dev,
			  "DROP: erroneous channel US/DS.\n");
		show_tx_err_pkt_desc(ndev, fb);
		if (fb->type == BUF_TYPE_FPM)
			show_tx_err_pkt(ndev, fb->data, fb->len);
		else
			show_tx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		ndev->stats.netdev.tx_errors++;
		ndev->stats.netdev.tx_dropped++;
		free_buf = true;
		status = NETDEV_TX_OK;
		goto done;
	}

	status = dqm_tx(ndev->chan->tx_q[tx_prio][q_type], 1,
			ndev->chan->tx_q_info[(tx_prio * Q_US_DS_MAX) + q_type].msg_size,
			msgdata);
	switch (status) {
	case 0:
		ndev->stats.netdev.tx_bytes += fb->len - fb->brcm_tag_len -
				fb->fap_tag_len - fb->sv_tag_len + ETH_CRC_LEN;
		ndev->stats.netdev.tx_packets++;
		status = NETDEV_TX_OK;
		break;

	case -EAGAIN:
		netif_dbg(ndev, tx_queued, dev,
			  "Unable to send packet to DQM. Stopping ");
		netif_dbg(ndev, tx_queued, dev,
			  "netif Q until room available.\n");
		netif_stop_queue(dev);
		spin_lock_irqsave(&ndev->chan->tx_busy[tx_prio][q_type].lock,
				  flags);
		ndev_found = false;
		list_for_each(pos, &ndev->chan->tx_busy[tx_prio][q_type].list)
		{
			ndev_tmp = list_entry(pos, struct dqnet_netdev,
					      tx_busy_list);
			if (ndev_tmp == ndev) {
				ndev_found = true;
				break;
			}
		}
		if (!ndev_found)
			list_add(&ndev->tx_busy_list,
				 &ndev->chan->tx_busy[tx_prio][q_type].list);
		spin_unlock_irqrestore(&ndev->chan->tx_busy[tx_prio][q_type].lock,
				       flags);
		free_buf = true;
		dqm_enable_tx_cb(ndev->chan->tx_q[tx_prio][q_type]);
		ndev->stats.netdev.tx_fifo_errors++;
		ndev->stats.netdev.tx_errors++;
		err_stats.dqm_tx_busy++;
		status = NETDEV_TX_BUSY;
		goto done;

	default:
		netif_err(ndev, tx_err, dev,
			  "======================================\n");
		netif_err(ndev, tx_err, dev,
			  "DROP: error sending to DQM.\n");
		netif_err(ndev, tx_err, dev,
			  "TX (%d bytes)\n", fb->len);
		netif_err(ndev, tx_err, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		if (fb->type == BUF_TYPE_FPM)
			show_tx_err_pkt(ndev, fb->data, fb->len);
		else
			show_tx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, tx_err, dev,
			  "======================================\n");
		free_buf = true;
		ndev->stats.netdev.tx_errors++;
		err_stats.dqm_tx_failed++;
		status = NETDEV_TX_OK;
		goto done;
	}

done:
	if (free_buf) {
		if (fb->type == BUF_TYPE_FPM) {
			fpm_invalidate_token(fb->token, fb->offset,
					     ndev->tail_pad,
					     FPM_SYNC_HEAD | FPM_SYNC_TAIL);
			fpm_free_token(fb->token);
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
		} else {
			phys_addr_t paddr;
			dma_addr_t dma;
			memcpy(&paddr, fb->skb->cb, sizeof(phys_addr_t));
			dma = paddr;
			dma_unmap_single(&ndev->dev->dev, dma,
					 fb->skb->end - fb->skb->data,
					 DMA_FROM_DEVICE);
			if (status != NETDEV_TX_BUSY)
				dev_kfree_skb_any(fb->skb);
#endif
		}
	}
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Transmit an FPM buffer
 *
 * Parameters
 *	fb	buffer to send
 *	dev	network device which is sending
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_tx_fpm(struct fpm_buff *fb, struct net_device *dev)
{
	int status = NETDEV_TX_OK;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	enum bcm_nethook_result result;
	bool free_token = false;

	pr_debug("%s:-->\n",dev->name);

	/*
	 * Must set these here to handle someone calling this function directly
	 * from an SKB nethook where they don't have an FPMB (ex. SFAP).
	 */
	fb->if_id = ndev->if_id;
	fb->if_sub_id = ndev->if_sub_id;

	result = bcm_nethook_call_hooks(dev, BCM_NETHOOK_TX_FPM, fb);
	switch (result) {
	case BCM_NETHOOK_DROP:
		netif_dbg(ndev, drop, dev,
			  "===============================================\n");
		netif_dbg(ndev, drop, dev,
			  "DROP: TX FPM nethook\n");
		show_drop_pkt_desc(ndev, fb);
		show_drop_pkt(ndev, fb->data, fb->len);
		netif_dbg(ndev, drop, dev,
			  "===============================================\n");
		ndev->stats.netdev.tx_dropped++;
		free_token = true;
		goto done;
	case BCM_NETHOOK_CONSUMED:
		ndev->stats.tx_nethook_consumed++;
		goto done;
	case BCM_NETHOOK_PASS:
	case BCM_NETHOOK_SKIP:
	default:
		break;
	}

	status = dqnet_tx_buf(fb, dev);

done:
	if (free_token) {
		fpm_invalidate_token(fb->token, fb->offset,
				     ndev->tail_pad,
				     FPM_SYNC_HEAD | FPM_SYNC_TAIL);
		fpm_free_token(fb->token);
	}
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Function used by BCM nethooks to Transmit an FPM buffer
 *
 * Parameters
 *	fb	buffer to send
 *	dev	network device which is sending
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_nethook_tx_fpm(struct fpm_buff *fb, struct net_device *dev)
{
	int status = NETDEV_TX_OK;
	struct pcpu_sw_netstats *tstats = NULL;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();

	status = dqnet_tx_fpm(fb, dev);

	if (status == NETDEV_TX_OK && nethooks->offload_stats.slow) {
		tstats = this_cpu_ptr(nethooks->offload_stats.slow);
		if (tstats) {
			u64_stats_update_begin(&tstats->syncp);
			tstats->tx_packets++;
			tstats->tx_bytes += fb->len - fb->brcm_tag_len -
				fb->fap_tag_len + ETH_CRC_LEN;
			u64_stats_update_end(&tstats->syncp);
		}
	}

	return status;
}

/*
 * Copy an SKB to an FPM
 *
 * Copies an SKB's packet data into an FPM buffer. During this
 * process the packet is padded to the minimum Ethernet packet length
 * if necessary, and a BRCM tag and/or a DFAP header may be added
 * depending on the interface.
 *
 * Parameters
 *	ndev	network device private structure
 *      skb	SKB to copy
 *      fb	FPM to copy SKB into
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_copy_skb_to_fpm(struct dqnet_netdev *ndev,
				 struct fpm_buff *fb)
{
	int status = 0;
	int runt_pad_len;
	int pad_len;
	u8 *s, *f;

	pr_debug("%s:-->\n",ndev->name);

	/* Do not pad runt packets for wifi interface */
	if (ndev->link_type != DQNET_LINK_TYPE_WIFI)
		runt_pad_len = dqnet_get_runt_pad_len(fb->skb->len);
	fb->token = fpm_alloc_token(ndev->head_pad + fb->fap_tag_len +
								fb->sv_tag_len + fb->brcm_tag_len +
								fb->skb->len + runt_pad_len +
								ndev->tail_pad + 4);
	if (!fb->token) {
		netif_err(ndev, tx_err, ndev->dev,
			  "===============================================\n");
		netif_err(ndev, tx_err, ndev->dev,
			  "DROP: failed to alloc FPM buffer.\n");
		netif_err(ndev, tx_err, ndev->dev,
			  "TX packet (%d bytes)\n", fb->skb->len);
		netif_err(ndev, tx_err, ndev->dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, ndev->dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_tx_err_pkt(ndev, fb->skb->data, fb->skb->len);
		netif_err(ndev, tx_err, ndev->dev,
			  "===============================================\n");
		err_stats.fpm_alloc_failed++;
		status = -ENOMEM;
		goto done;
	}
	fb->buf = fpm_token_to_buffer(fb->token);
	fb->offset = ndev->head_pad;
	fb->data = fb->buf + fb->offset + fb->fap_tag_len + fb->brcm_tag_len;
	fb->data += fb->sv_tag_len;
	fb->offset += fb->fap_tag_len + fb->brcm_tag_len;
	fb->offset += fb->sv_tag_len;

	f = fb->data;
	s = fb->skb->data;
	memcpy(f, s, fb->skb->len);
	f += fb->skb->len;
	/* Do not pad runt packets for wifi interface */
	if (ndev->link_type != DQNET_LINK_TYPE_WIFI) {
		pad_len = dqnet_get_runt_pad_len(fb->skb->len);
		memset(f, 0, pad_len);
	}

	fb->len = fb->skb->len + pad_len;
	fpm_set_token_size(&fb->token, fb->len);

done:
	pr_debug("%s:<--\n",ndev->name);
	return status;
}

/*
 * Transmit an SKB
 *
 * TODO: use TOS bits to determine which priority Q to use. For
 * now always use lowest priority Q.
 *
 * Parameters
 *	skb	SKB to send
 *	dev	network device which is sending
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_tx_skb(struct sk_buff *skb, struct net_device *dev)
{
	int status = NETDEV_TX_OK;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	enum bcm_nethook_result result;
	struct fpm_buff fb;

	pr_debug("%s:-->\n",dev->name);

	local_bh_disable();

	if (ndev->chan->tx_disable) {
		ndev->stats.netdev.tx_dropped++;
		goto free_skb;
	}

	if (!netif_running(dev)) {
		netif_dbg(ndev, tx_err, dev,
			  "===============================================\n");
		netif_dbg(ndev, tx_err, dev,
			  "DROP: interface is down\n");
		netif_dbg(ndev, tx_err, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_dbg(ndev, tx_err, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_dbg(ndev, tx_err, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_tx_dbg_pkt(ndev, skb->data, skb->len);
		netif_dbg(ndev, tx_err, dev,
			  "===============================================\n");
		err_stats.drop_if_down++;
		goto free_skb;
	}

	if (!netif_carrier_ok(dev)) {
		netif_dbg(ndev, tx_err, dev,
			  "===============================================\n");
		netif_dbg(ndev, tx_err, dev,
			  "DROP: link is down\n");
		netif_dbg(ndev, tx_err, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_dbg(ndev, tx_err, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_dbg(ndev, tx_err, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_tx_dbg_pkt(ndev, skb->data, skb->len);
		netif_dbg(ndev, tx_err, dev,
			  "===============================================\n");
		ndev->stats.netdev.tx_carrier_errors++;
		ndev->stats.netdev.tx_errors++;
		err_stats.drop_carrier_off++;
		goto free_skb;
	}

	skb_reset_mac_header(skb);

	result = bcm_nethook_call_hooks(dev, BCM_NETHOOK_TX_SKB, skb);
	switch (result) {
	case BCM_NETHOOK_DROP:
		netif_dbg(ndev, drop, dev,
			  "=================================");
		netif_dbg(ndev, drop, dev,
			  "==============\n");
		netif_dbg(ndev, drop, dev,
			  "DROP: TX SKB nethook\n");
		netif_dbg(ndev, drop, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_dbg(ndev, drop, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_dbg(ndev, drop, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_drop_pkt(ndev, skb->data, skb->len);
		netif_dbg(ndev, drop, dev,
			  "=================================");
		netif_dbg(ndev, drop, dev,
			  "==============\n");
		ndev->stats.netdev.tx_dropped++;
		goto free_skb;
	case BCM_NETHOOK_CONSUMED:
		ndev->stats.tx_nethook_consumed++;
		goto enable_bh;
	case BCM_NETHOOK_PASS:
	case BCM_NETHOOK_SKIP:
	default:
		break;
	}

	fb.skb = skb;
	fb.if_id = ndev->if_id;
	fb.if_sub_id = ndev->if_sub_id;
	if (ndev->tx_qm_q) {
		/* Default pick 1st queue if supported by HW*/
		fb.queue_id = ndev->tx_qm_q[0];
	} else if (ndev->tx_if_id >= 0) {
		fb.queue_id = ndev->tx_if_id;
	}
	fb.fap_tag_len = dqnet_fap_get_tag_len(dev);
	fb.brcm_tag_len = dqnet_get_brcm_tag_len(ndev);
	fb.sv_tag_len = dqnet_get_sv_tag_len(ndev);
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	if (skb->dev_in)
		fb.dev_in = skb->dev_in;
	else
#endif
		fb.dev_in = __dev_get_by_index(&init_net, skb->skb_iif);
	fb.dev_out = dev;
	fb.priority = skb->priority;
	if (ndev->tx_fpm)
		fb.type = BUF_TYPE_FPM;
	else
		fb.type = ndev->chan->buf_type();
	if (fb.type == BUF_TYPE_FPM) {
		int ret;
		ret = dqnet_copy_skb_to_fpm(ndev, &fb);
		if (ret) {
			if (ret != -ENOMEM) {
				fpm_invalidate_token(fb.token, fb.offset,
						     ndev->tail_pad,
						     FPM_SYNC_HEAD |
						     FPM_SYNC_TAIL);
				fpm_free_token(fb.token);
			}
			ndev->stats.netdev.tx_errors++;
			ndev->stats.netdev.tx_dropped++;
			goto free_skb;
		}
	} else {
		/* Do not pad runt packets for wifi interface */
		if (ndev->link_type != DQNET_LINK_TYPE_WIFI) {
			if (skb->len < ETH_ZLEN)
				skb->len = ETH_ZLEN;
		}
	}

	if (ndev->tx_fpm)
		status = dqnet_tx_fpm(&fb, dev);
	else {
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
		if (fb.type == BUF_TYPE_SKB)
			dqskb_unprepare_skb(dqskbdev, skb);
#endif
		status = ndev->chan->tx_pkt(&fb, dev);
		if (fb.type == BUF_TYPE_SKB)
			goto enable_bh;
	}

free_skb:
	if (status != NETDEV_TX_BUSY)
		dev_kfree_skb_any(skb);

enable_bh:
	local_bh_enable();

	pr_debug("%s:<--\n",dev->name);
	return status;
}

static void dqnet_timeout(struct net_device *dev, unsigned int txqueue)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n",dev->name);

	netif_err(ndev, timer, dev, "Watchdog timer expired.\n");
	err_stats.wd_timeout++;

	pr_debug("%s:<--\n",dev->name);
}

static int dqnet_set_mtu(struct net_device *dev, int mtu)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = 0, min_mtu = MIN_MTU_SIZE, max_mtu = MAX_MTU_SIZE;

	if (ndev->hal.macdev)
		max_mtu = IEEE_MAX_MTU_SIZE;

	mtu = mtu + ETH_HLEN + VLAN_HLEN + ETH_FCS_LEN + ndev->mtu_pad;
	if (mtu < min_mtu || mtu > max_mtu)
	{
		pr_err("ERROR: %s MTU value %d is not within %d to %d!",
			  ndev->name, mtu, min_mtu, max_mtu);
		status = -EINVAL;
	}
	else
	{
		dev->mtu = mtu - ETH_HLEN - VLAN_HLEN - ETH_FCS_LEN - ndev->mtu_pad;
		if (ndev->hal.macdev)
			haldev_change_mtu(&ndev->hal, mtu);
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		else if (ndev->link_type == DQNET_LINK_TYPE_SWITCH)
			status = SWITCH->set_mtu(mtu);
#endif
		dqnet_fap_set_mtu(dev, mtu);
	}

	return status;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0))
/*
 * Handle device private ioctl requests to a device
 */
static int dqnet_siocdevprivate(struct net_device *dev, struct ifreq *rq,
				void *user, int cmd)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = -EOPNOTSUPP;

	pr_debug("%s:-->\n",dev->name);

	switch (cmd) {
	case SIOCDEVPRIVATE:
		if(get_user(ndev->mtu_pad,
			   ((unsigned int *)rq->ifr_ifru.ifru_data)))
			return -EFAULT;
		if(dqnet_set_mtu(dev, dev->mtu))
			return -EINVAL;
		status = 0;
		break;
	default:
		break;
	}

	pr_debug("%s:<--\n",dev->name);
	return status;
}
#endif

static int dqnet_mii_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct phy_device *phydev = dev->phydev;
	struct mii_ioctl_data *data = if_mii(rq);
	u32 reg_num;
	int ret;

	switch (cmd) {
	case SIOCGMIIPHY:
		data->phy_id = phydev->mdio.addr;
		break;
	case SIOCGMIIREG:
		reg_num = data->reg_num | (data->val_in << 16);
		ret = mdiobus_read(phydev->mdio.bus, data->phy_id, reg_num);
		if (ret < 0)
			return ret;
		data->val_out = ret;
		break;
	case SIOCSMIIREG:
		reg_num = data->reg_num | (data->val_out << 16);
		ret = mdiobus_write(phydev->mdio.bus, data->phy_id, reg_num, data->val_in);
		if (ret < 0)
			return ret;
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}

/*
 * Handle ioctl requests to a device
 */
static int dqnet_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	int status = -EOPNOTSUPP;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(5, 15, 0))
	struct dqnet_netdev *ndev = netdev_priv(dev);
#endif

	pr_debug("%s:-->\n",dev->name);

	switch (cmd) {
	case SIOCGMIIPHY:
	case SIOCGMIIREG:
	case SIOCSMIIREG:
		if (dev->phydev && dev->phydev->mdio.bus)
			return dqnet_mii_ioctl(dev, rq, cmd);
		break;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(5, 15, 0))
	case SIOCDEVPRIVATE:
		if(get_user(ndev->mtu_pad, ((unsigned int *)rq->ifr_ifru.ifru_data)))
			return -EFAULT;
		if(dqnet_set_mtu(dev, dev->mtu))
			return -EINVAL;
		status = 0;
		break;
#endif
	default:
		break;
	}

	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Query the device for network stats
 *
 * Ddepending on the device, these stats may come from an external
 * switch, the GFAP, or the endpoint device itself.
 *
 * Parameters
 *      dev	device to query
 *      stats	statistics struct to fill in
 *
 * Returns
 *	stats with values filled in
 */
static void
dqnet_get_stats(struct net_device *dev, struct rtnl_link_stats64 *stats)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n", dev->name);
	memcpy(stats, &ndev->stats.netdev, sizeof(ndev->stats.netdev));
	if (ndev->hal.macdev)
		haldev_link_stats(&ndev->hal, stats);
	else if (ndev->link_stats)
		ndev->link_stats(dev, stats);
	pr_debug("%s:<--\n", dev->name);
	return;
}

/*
 * Set a device's MAC address
 *
 * Parameters
 *	dev	device to modify
 *	p	new MAC address
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_set_mac_addr(struct net_device *dev, void *p)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = 0;
	struct sockaddr *addr = (struct sockaddr *)p;

	pr_debug("%s:-->\n",dev->name);

	if (netif_running(dev)) {
		netdev_err(dev, "Device busy.\n");
		status = -EBUSY;
		goto done;
	}

	memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);
	status = haldev_set_mac_address(&ndev->hal, dev->dev_addr);

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Handle phy link status callback
 *
 * Parameters
 *      dev	net device associated with phy
 */
static void dqnet_phy_link_cb(struct net_device *dev)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct phy_device *phydev = ndev->phydev;
	enum dqnet_link_state link_state;
	bool state_changed = false;

	pr_debug("%s:-->\n",dev->name);

	link_state = phydev->link ? DQNET_LINK_STATE_UP :
		DQNET_LINK_STATE_DOWN;

	if (ndev->link_state != link_state) {
		state_changed = true;
		ndev->link_state = link_state;
	}

	if (link_state == DQNET_LINK_STATE_UP) {
		/* check speed/duplex/pause changes */
		if (ndev->link_speed != phydev->speed) {
			state_changed = true;
			ndev->link_speed = phydev->speed;
		}

		if (ndev->link_duplex != phydev->duplex) {
			state_changed = true;
			ndev->link_duplex = phydev->duplex;
		}

		if (ndev->link_pause != phydev->pause) {
			state_changed = true;
			ndev->link_pause = phydev->pause;
		}

		/* done if nothing has changed */
		if (!state_changed)
			return;

		/*
		 * program UMAC and RGMII block based on established link
		 * speed, pause, and duplex.
		 */
		haldev_adjust_link(&ndev->hal);
	} else {
		/* done if nothing has changed */
		if (!state_changed)
			return;

		/* needed for fixed PHY to reflect correct link status */
		netif_carrier_off(dev);

		/* link down */
		haldev_remove_link(&ndev->hal);
	}

	phy_print_status(phydev);

	pr_debug("%s:<--\n",dev->name);
}

#if defined(CONFIG_BCM_RPC_SERVICES)
/*
 * Handle remote network interface message or request
 *
 * Parameters
 *      dqm_tunnel	DQM tunnel ID
 *      msg		RPC service callback message
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_rpc_message_handler(int dqm_tunnel, struct misc_msg *msg)
{
	int status = 0;
	char *name;
	bool ndev_found = false;
	struct dqnet_netdev *ndev = NULL;
	unsigned long flags;
	u8 *buf;
	u32 len, token;
	union tok_src_dest tok_op_data;

	if (unlikely(!fpm_is_valid_token(msg->data))) {
		pr_err("%s: Invalid token 0x%08x received.\n", __func__,
		       msg->data);
		msg->data = 0;
		status = -EINVAL;
		goto reply;
	}
	token = msg->data;
	fpm_track_token_rx(msg->data);
	tok_op_data.data = 0;
	tok_op_data.rpc_hdr = msg->dqm_header;
	fpm_track_token_src(msg->data, &tok_op_data);
	fpm_invalidate_token(msg->data, 0, 0, 0);
	name = (char *)fpm_token_to_buffer(msg->data);
	if (unlikely(!name)) {
		pr_err("%s: Unable to translate token 0x%08x to buffer.\n",
		       __func__, msg->data);
		msg->data = 0;
		status = -EINVAL;
		goto reply;
	}
	len = strlen(name);
	msg->data = 0;

	pr_debug("-->\n");

	spin_lock_irqsave(&ndevs_open.lock, flags);
	if (!list_empty(&ndevs_open.list)) {
		list_for_each_entry(ndev, &ndevs_open.list, open_list) {
			if (ndev->dt_name && !strncmp(ndev->dt_name, name, strlen(ndev->dt_name))) {
				ndev_found = true;
				break;
			}
		}
	}
	spin_unlock_irqrestore(&ndevs_open.lock, flags);
	if (!ndev_found) {
		pr_debug("dev: %s not found\n", name);
		err_stats.rpc_dev_unknown++;
		status = -EIO;
		goto free_buf;
	}

	pr_debug("dev: %s\n", ndev->name);

	switch (msg->type) {
	case IF_UP:
		pr_debug("IF_UP\n");
		if (netif_running(ndev->dev) &&
		    ndev->link_state == DQNET_LINK_STATE_DOWN) {
			netif_err(ndev, link, ndev->dev, "Link up.\n");
			ndev->link_state = DQNET_LINK_STATE_UP;
			netif_carrier_on(ndev->dev);
			send_netif_state(dqm_tunnel, ndev->dt_name, IF_UP);
		}
		break;

	case IF_DOWN:
		pr_debug("IF_down\n");
		if (netif_running(ndev->dev) &&
		    ndev->link_state == DQNET_LINK_STATE_UP) {
			netif_err(ndev, link, ndev->dev, "Link down.\n");
			ndev->link_state = DQNET_LINK_STATE_DOWN;
			netif_carrier_off(ndev->dev);
		}
		break;

	case IF_STATS:
		pr_debug("IF_STATS\n");
		msg->data = fpm_alloc_token(sizeof(ndev->stats));
		if (!msg->data) {
			netif_err(ndev, rx_err, ndev->dev,
				  "Unable to alloc token.\n");
			status = -ENOMEM;
			break;
		}
		fpm_set_token_size(&msg->data, sizeof(ndev->stats));
		buf = fpm_token_to_buffer(msg->data);
		memcpy(buf, &ndev->stats, sizeof(ndev->stats));
		tok_op_data.data = 0;
		tok_op_data.rpc_hdr = msg->dqm_header;
		fpm_track_token_dest(msg->data, &tok_op_data);
		fpm_track_token_tx(msg->data);
		fpm_flush_invalidate_token(msg->data, 0, 0, 0);
		break;

	default:
		pr_debug("unknown msg type\n");
		netif_err(ndev, link, ndev->dev,
		      "Unknown link status message.\n");
		err_stats.rpc_msg_bad++;
		status = -EINVAL;
		break;
	}

free_buf:
	fpm_invalidate_buffer((u8 *)name, len);
	fpm_free_token(token);

reply:
	pr_debug("sending response\n");
	if (rpc_msg_request((rpc_msg *)msg)) {
		msg->extra = status;
		rpc_send_reply(dqm_tunnel, (rpc_msg *)msg);
	}

	pr_debug("<--\n");
	return status;
}
EXPORT_SYMBOL(dqnet_rpc_message_handler);
#endif
int dqnet_priv_data_size(void)
{
	return DQNET_PRIV_DATA_SIZE();
}
EXPORT_SYMBOL(dqnet_priv_data_size);

int dqnet_switch_eee(struct net_device *dev, bool enable)
{
	int status = 0;
	struct dqnet_port_pwr port_pwr;

	if ((status = SWITCH->get_port_pwr(dev, &port_pwr)) == 0) {
		port_pwr.auto_pwr_down = true;
		port_pwr.eee_enabled = enable;
		port_pwr.eee_tx_lpi_enabled = true;
		status = SWITCH->set_port_pwr(dev, &port_pwr);
	}

	return status;
}

/*
 * Open a network device (i.e. ifconfig "device" up).
 *
 * The FAP exception Q is muxed with packets from various devices that
 * we have to demux to the correct device. These packets are demuxed
 * via an interface ID. For some interfaces they must be further demuxed
 * using an interface sub-ID. For switch interfaces they must be further
 * demuxed using either and ARL lookup or a BRCM tag to determine the port #.
 * Since we don't know until we have a dqnet_netdev how to demux beyond
 * the interface ID we first retrieve a dqnet_netdev from the demux table
 * using the interface ID and sub-ID of if_sub_id_max. This means
 * we always need a valid dqnet_netdev in the [IF ID][if_sub_id_max]
 * entry, so if this entry is NULL we populate it with our own.
 *
 * If the device has a Q of its own we don't need to demux, so we just
 * store the dqnet_netdev in the channel structure.
 *
 * Parameters
 *	dev	device to open
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_open(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct dqnet_channel *chan = ndev->chan;
	struct dqnet_netdev *ndev_tmp;
	bool ndev_found = false;
	unsigned long flags;
	int i;
	void *q_h;
	struct dqm_cb cb = {};
	struct list_head *pos, *pos_tmp;
	bool wifi = false;
	bool hostdrv = false;
	int prio;
	enum dqnet_q_type q_type;

	pr_debug("%s:-->\n",dev->name);

	if (!strcmp(ndev->name, DQNET_WIFI_NAME_STRING))
		wifi = true;
	else if (ndev->link_type == DQNET_LINK_TYPE_HOSTDRV)
		hostdrv = true;

	if (!wifi) {
		spin_lock_irqsave(&ndevs_open.lock, flags);
		if (!list_empty(&ndevs_open.list)) {
			list_for_each_entry(ndev_tmp,
				&ndevs_open.list, open_list) {
					if (ndev_tmp == ndev) {
						ndev_found = true;
						break;
					}
			}
		}
		spin_unlock_irqrestore(&ndevs_open.lock, flags);
		if (ndev_found) {
			netdev_err(dev,
				"Attempt to open device already open!\n");
			status = -EINVAL;
			goto done;
		}
	}

	if (chan->open_count == 0) {
		cb.fn = dqnet_tx_isr;
		for (i = 0; i < chan->tx_q_count; i++) {
			prio = chan->tx_q_info[i].priority;
			q_type = chan->tx_q_info[i].q_type;
			INIT_LIST_HEAD(&chan->tx_busy[prio][q_type].list);
			spin_lock_init(&chan->tx_busy[prio][q_type].lock);
			cb.context = &chan->tx_busy[prio][q_type];
			cb.name = chan->name;
			q_h = dqm_register(chan->tx_q_info[i].dev,
					   chan->tx_q_info[i].q_num,
					   &cb,
					   &chan->tx_q_info[i].msg_size,
					   DQM_F_TX);
			if (!q_h) {
				netdev_dbg(dev,
					"Failed dqm_register(%s, %s, %d)\n",
					chan->name, chan->tx_q_info[i].dev,
					chan->tx_q_info[i].q_num);
				status = -EIO;
				goto done;
			}
			chan->tx_q[prio][q_type] = q_h;
		}

		cb.context = chan;
		cb.fn = dqnet_rx_isr;
		cb.name = chan->name;
		for (i = 0; i < chan->rx_q_count; i++) {
			q_h = dqm_register(chan->rx_q_info[i].dev,
					   chan->rx_q_info[i].q_num,
					   &cb,
					   &chan->rx_q_info[i].msg_size,
					   DQM_F_RX);
			if (!q_h) {
				netdev_dbg(dev,
					"Failed dqm_register(%s, %s, %d)\n",
					chan->name, chan->rx_q_info[i].dev,
					chan->rx_q_info[i].q_num);
				status = -EIO;
				goto done;
			}
			chan->rx_q[i] = q_h;
		}
	}

	spin_lock_irqsave(&demux_tbl_lock, flags);
	switch (ndev->demux_type) {
	case DQNET_DEMUX_TYPE_IFID_SUBID:
		demux_tbl[ndev->if_id][ndev->if_sub_id] = ndev;
		if (demux_tbl[ndev->if_id][if_sub_id_max] == NULL)
			demux_tbl[ndev->if_id][if_sub_id_max] = ndev;
		if (chan->open_count == 0) {
			chan->demux = dqnet_demux_if_id;
			chan->ndev = ndev;
		}
		ndev->demux = dqnet_demux_if_sub_id;
		break;

	case DQNET_DEMUX_TYPE_IFID_ARL:
		demux_tbl[ndev->if_id][ndev->if_sub_id] = ndev;
		if (demux_tbl[ndev->if_id][if_sub_id_max] == NULL)
			demux_tbl[ndev->if_id][if_sub_id_max] = ndev;
		if (chan->open_count == 0) {
			chan->demux = dqnet_demux_if_id;
			chan->ndev = ndev;
		}
		ndev->demux = SWITCH->demux_arl;
		break;

	case DQNET_DEMUX_TYPE_NONE:
		chan->demux = NULL;
		chan->ndev = ndev;
		ndev->demux = NULL;
		break;

	default:
		netdev_err(dev, "Invalid DQNET_DEMUX_TYPE on %s!\n",
		    ndev->name);
		status = -EIO;
		break;
	}
	spin_unlock_irqrestore(&demux_tbl_lock, flags);
	if (status)
		goto err_release_dqms;

	if (chan->open_count == 0) {
		netif_napi_add(&napi_dev, &chan->napi, dqnet_poll_napi,
			chan->napi.weight);
		napi_enable(&chan->napi);
	}

	if (!wifi)
		netif_start_queue(dev);

	spin_lock_irqsave(&chan->rx_irq_lock, flags);
	for (i = 0; i < chan->rx_q_count; i++) {
		status = dqm_enable_rx_cb(chan->rx_q[i]);
		if (status)
			break;
	}
	spin_unlock_irqrestore(&chan->rx_irq_lock, flags);
	if (status) {
		netdev_err(dev, "Failed to enable RX ints on Q %d!\n",
		    chan->rx_q_info[i].q_num);
		goto err_disable_netif;
	}

	chan->open_count++;

	if (!wifi) {
		spin_lock_irqsave(&ndevs_open.lock, flags);
		list_add(&ndev->open_list, &ndevs_open.list);
		spin_unlock_irqrestore(&ndevs_open.lock, flags);

		switch (ndev->link_type) {
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		case DQNET_LINK_TYPE_SWITCH:
			status = dqnet_switch_eee(dev, 1);
			if (status)
				goto err_remove_ndev;
			status = SWITCH->enable_port(dev);
			if (status)
				goto err_remove_ndev;
			status = SWITCH->set_port_stp_state(dev,
				ndev->if_sub_id, BR_STATE_FORWARDING);
			if (status)
				goto err_remove_ndev;
			netdev_dbg(dev, "STP state set to forwarding.\n");
			break;
#endif
		case DQNET_LINK_TYPE_PHY:
			haldev_open(&ndev->hal);
			break;

		case DQNET_LINK_TYPE_RPC:
#if defined(CONFIG_BCM_RPC_SERVICES)
			netdev_dbg(dev, "Sending IF_UP RPC msg.\n");
			send_netif_state(ndev->rpc_tunnel, ndev->dt_name, IF_UP);
#endif
			break;
		case DQNET_LINK_TYPE_FIXED:
			ndev->link_state = DQNET_LINK_STATE_UP;
			netif_carrier_on(ndev->dev);
		   break;
		case DQNET_LINK_TYPE_HOSTDRV:
		   break;
		default:
			netdev_err(dev, "Unknown link type.\n");
			status = -EIO;
			goto err_remove_ndev;
		}
	}

	dev->min_mtu = MIN_MTU_SIZE;
	dev->max_mtu = MAX_MTU_SIZE;

	goto done;

err_remove_ndev:
	if (!wifi) {
		spin_lock_irqsave(&ndevs_open.lock, flags);
		list_for_each_safe(pos, pos_tmp, &ndevs_open.list)
		{
			ndev_tmp = list_entry(pos,
				struct dqnet_netdev, open_list);
			if (ndev_tmp == ndev)
				list_del(pos);
		}
		spin_unlock_irqrestore(&ndevs_open.lock, flags);
	}

err_disable_netif:
	if (!wifi)
		netif_stop_queue(dev);
	if (chan->open_count == 0) {
		napi_disable(&chan->napi);
		netif_napi_del(&chan->napi);
	}

	spin_lock_irqsave(&demux_tbl_lock, flags);
	if (ndev->demux_type != DQNET_DEMUX_TYPE_NONE) {
		demux_tbl[ndev->if_id][ndev->if_sub_id] = NULL;
		for (i = 0; i < if_sub_id_max; i++) {
			if (demux_tbl[ndev->if_id][i] != NULL)
				break;
		}
		if (i == if_sub_id_max)
			demux_tbl[ndev->if_id][if_sub_id_max] = NULL;
	}
	spin_unlock_irqrestore(&demux_tbl_lock, flags);
	if (chan->open_count == 0) {
		chan->demux = NULL;
		chan->ndev = NULL;
	}
	ndev->demux = NULL;

err_release_dqms:
	if (chan->open_count == 0) {
		for (i = 0; i < chan->tx_q_count; i++) {
			prio = chan->tx_q_info[i].priority;
			q_type = chan->tx_q_info[i].q_type;
			dqm_release(chan->tx_q[prio][q_type], DQM_F_TX);
		}
		for (i = 0; i < chan->rx_q_count; i++)
			dqm_release(chan->rx_q[i], DQM_F_RX);
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return status;
}

/*
 * Close a network device
 *
 * Parameters
 *	dev	device to be closed
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_close(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct dqnet_channel *chan = ndev->chan;
	struct dqnet_netdev *ndev_tmp;
	bool ndev_found = false;
	struct list_head *pos, *pos_tmp;
	unsigned long flags;
	int i;
	bool wifi = false;
	bool hostdrv = false;
	int prio;
	enum dqnet_q_type q_type;

	pr_debug("%s:-->\n",dev->name);

	if (!strcmp(ndev->name, DQNET_WIFI_NAME_STRING))
		wifi = true;
	else if (ndev->link_type == DQNET_LINK_TYPE_HOSTDRV)
		hostdrv = true;

	if (!wifi) {
		spin_lock_irqsave(&ndevs_open.lock, flags);
		if (!list_empty(&ndevs_open.list)) {
			list_for_each_entry(ndev_tmp, &ndevs_open.list,
					    open_list) {
					if (ndev_tmp == ndev) {
						ndev_found = true;
						break;
					}
			}
		}
		spin_unlock_irqrestore(&ndevs_open.lock, flags);
		if (!ndev_found) {
			netdev_err(dev, "Attempt to close device not open!\n");
			status = -EINVAL;
			goto done;
		}

		netif_stop_queue(ndev->dev);

		switch (ndev->link_type) {
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		case DQNET_LINK_TYPE_SWITCH:
			SWITCH->disable_port(dev);
			break;
#endif
		case DQNET_LINK_TYPE_PHY:
			haldev_stop(&ndev->hal);
			break;

		case DQNET_LINK_TYPE_RPC:
			netdev_dbg(dev, "Sending IF_DOWN RPC msg.\n");
#if defined(CONFIG_BCM_RPC_SERVICES)
			send_netif_state(ndev->rpc_tunnel, ndev->dt_name,
					 IF_DOWN);
#endif
			ndev->link_state = DQNET_LINK_STATE_DOWN;
			netif_carrier_off(ndev->dev);
			break;

		case DQNET_LINK_TYPE_FIXED:
			ndev->link_state = DQNET_LINK_STATE_DOWN;
			netif_carrier_off(ndev->dev);
			break;

		case DQNET_LINK_TYPE_HOSTDRV:
			break;

		default:
			break;
		}
	}

	chan->open_count--;
	if (chan->open_count == 0) {
		spin_lock_irqsave(&chan->rx_irq_lock, flags);
		for (i = 0; i < chan->rx_q_count; i++) {
			status = dqm_disable_rx_cb(chan->rx_q[i]);
			if (status) {
				netdev_err(dev,
					"Failed to disable RX ints on ");
				netdev_err(dev, "Q %d!\n", chan->rx_q_info[i].q_num);
			}
		}
		spin_unlock_irqrestore(&chan->rx_irq_lock, flags);

		napi_disable(&chan->napi);
		netif_napi_del(&chan->napi);

		for (i = 0; i < chan->tx_q_count; i++) {
			prio = chan->tx_q_info[i].priority;
			q_type = chan->tx_q_info[i].q_type;
			dqm_release(chan->tx_q[prio][q_type], DQM_F_TX);
		}
		for (i = 0; i < chan->rx_q_count; i++)
			dqm_release(chan->rx_q[i], DQM_F_RX);
	}

	if (wifi && chan->open_count != 0)
		napi_disable(&chan->napi);

	spin_lock_irqsave(&demux_tbl_lock, flags);
	if (ndev->demux_type != DQNET_DEMUX_TYPE_NONE) {
		demux_tbl[ndev->if_id][ndev->if_sub_id] = NULL;
		for (i = 0; i < if_sub_id_max; i++) {
			if (demux_tbl[ndev->if_id][i] != NULL)
				break;
		}
		if (i == if_sub_id_max)
			demux_tbl[ndev->if_id][if_sub_id_max] = NULL;
		else
			if(ndev == demux_tbl[ndev->if_id][if_sub_id_max])
				demux_tbl[ndev->if_id][if_sub_id_max] = demux_tbl[ndev->if_id][i];
	}

	spin_unlock_irqrestore(&demux_tbl_lock, flags);
	if (wifi && chan->open_count != 0) {
		napi_enable(&chan->napi);
		napi_schedule(&chan->napi);
	}

	if (chan->open_count == 0) {
		chan->demux = NULL;
		chan->ndev = NULL;
	}
	ndev->demux = NULL;

	if (!wifi) {
		spin_lock_irqsave(&ndevs_open.lock, flags);
		list_for_each_safe(pos, pos_tmp, &ndevs_open.list)
		{
			ndev_tmp = list_entry(pos, struct dqnet_netdev,
					      open_list);
			if (ndev_tmp == ndev)
				list_del(pos);
		}
		spin_unlock_irqrestore(&ndevs_open.lock, flags);
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return 0;
}

static int
dqnet_rxqueue_cpu_affinity(struct net_device *dev, int queue_id, cpumask_t mask)
{
	unsigned char path[100];
	unsigned char data[10];
	struct netdev_rx_queue *queue;
	struct rps_map *old_map, *map;
	static DEFINE_MUTEX(rps_map_mutex);
	int cpu, i;

	snprintf(path, sizeof(path), "/sys/class/net/%s/queues/rx-%d/rps_cpus",
		 dev->name, queue_id);
	snprintf(data, sizeof(data), "%*pb\n", cpumask_pr_args(&mask));

	pr_info("%s: Setting %s=%s", MODULE_NAME, path, data);

	queue = &dev->_rx[queue_id];
	if (queue->rps_map)
		return 0;

	map = kzalloc(max_t(unsigned int,
			RPS_MAP_SIZE(cpumask_weight(&mask)),
			L1_CACHE_BYTES), GFP_KERNEL);
	i = 0;
	for_each_cpu_and(cpu, &mask, cpu_online_mask)
		map->cpus[i++] = cpu;
	if (i) {
		map->len = i;
	} else {
		kfree(map);
		map = NULL;
	}
	mutex_lock(&rps_map_mutex);
	old_map = rcu_dereference_protected(queue->rps_map,
						mutex_is_locked(&rps_map_mutex));
	rcu_assign_pointer(queue->rps_map, map);

	if (map)
		static_branch_inc(&rps_needed);
	if (old_map)
		static_branch_dec(&rps_needed);

	mutex_unlock(&rps_map_mutex);

	if (old_map)
		kfree_rcu(old_map, rcu);

	return 0;
}

int dqnet_dev_weight(char *weight)
{
	pr_info("%s: Setting /proc/sys/net/core/dev_weight=%s\n",
		MODULE_NAME, weight);
	return procsys_file_write_string(
	   "/proc/sys/net/core/dev_weight", weight);
}

int dqnet_netdev_budget(char *budget)
{
	pr_info("%s: Setting /proc/sys/net/core/netdev_budget=%s\n",
		MODULE_NAME, budget);
	return procsys_file_write_string(
	   "/proc/sys/net/core/netdev_budget", budget);
}

int dqnet_netdev_max_backlog(char *backlog)
{
	pr_info("%s: Setting /proc/sys/net/core/netdev_max_backlog=%s\n",
		MODULE_NAME, backlog);
	return procsys_file_write_string(
	   "/proc/sys/net/core/netdev_max_backlog", backlog);
}

int dqnet_rxqueue_cpu_affinity_default(struct dqnet_netdev *ndev)
{
	int status = 0, i;
	cpumask_t affinity;

	for (i = 0; i < ndev->dev->num_rx_queues ; i++) {
		if (ndev->chan->rx_q_info[i].q_type == DQNET_QUEUE_DS) {
			*(affinity.bits) = 1;
			status = dqnet_rxqueue_cpu_affinity(ndev->dev,
							    i, affinity);
			pr_debug("%s: device %s queue %d affinity %*pb %d\n",
				MODULE_NAME, ndev->dev->name, i,
				cpumask_pr_args(&affinity), status);
		} else {
			affinity = *cpu_online_mask;
			status = dqnet_rxqueue_cpu_affinity(ndev->dev,
							    i, affinity);
			pr_debug("%s: device %s queue %d affinity %*pb %d\n",
				MODULE_NAME, ndev->dev->name, i,
				cpumask_pr_args(&affinity), status);
		}
	}
	return status;
}

#if defined(CONFIG_BCM_PWR_RPC)

static void dqnet_brcm_pwr_rpc_mbox_work_handler(struct work_struct *work)
{
	struct dqnet_netdev *ndev;

	ndev = container_of(work, struct dqnet_netdev, work);

	/* Powerdown Ethernet Phy */
	if (ndev->phy_power == HAL_LINK_STOP_EXIT)
		haldev_config_link(&ndev->hal, HAL_LINK_STOP_EXIT);
}

/*
 * NOTE: This function is designed to be called in ISR context. Make sure
 * any future additions are ISR safe.
 */
static int dqnet_brcm_pwr_rpc_mbox_event(
		struct notifier_block *this,
		unsigned long event, void *ptr)
{
	int status = NOTIFY_OK;
	struct brcm_pwr_rpc_mbox_info *states = ptr;
	struct dqnet_netdev *ndev;

	switch (event) {
	case PWR_MBOX_CHANGE_EVENT:
	{
		/*
		 * Currently, we only handle notifications for AC to battery
		 * transitions. Transitions to AC from battery will be handled
		 * by the powerman application.
		 */
		if (BATT_STATE(states->mbox[PWR_MBOX_BS]) == BS_BATTERY	 &&
				states->delay_battery_mode) {
			pr_debug("Power operated through battery, delay_battery_mode "
					 "enabled, so no action on Ethernet power control\n");
			goto done;
		}
		if (BATT_STATE(states->mbox[PWR_MBOX_BS]) == BS_BATTERY) {
			rcu_read_lock();
			list_for_each_entry_rcu(ndev, &ndevs.list, list) {
				if (!strstr(ndev->dev->name, "eth"))
					continue;

				if (ndev->link_type != DQNET_LINK_TYPE_PHY)
					continue;

				if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) ==
						BCM_NETDEVICE_GROUP_WAN)
					continue;

				ndev->phy_power = HAL_LINK_STOP_EXIT;
				/* Schedule Work on CPU 0 */
				if (workqueue)
					queue_work_on(0, workqueue, &ndev->work);
			}
			rcu_read_unlock();
		}
		break;
	}
	default:
		status = NOTIFY_DONE;
		goto done;
		break;
	}

done:
	return status;
}

static struct notifier_block brcm_pwr_rpc_mbox_notifier = {
	.notifier_call	= dqnet_brcm_pwr_rpc_mbox_event,
};
#endif
/*
 * Device probe
 *
 * This will be called by Linux for each device tree node matching our
 * driver name we passed when we called platform_driver_register().
 * Each node specifies a network device interface to create.
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_probe(struct platform_device *pdev)
{
	int status;
	struct dqnet_netdev *ndev = NULL;
	struct dqnet_channel *chan;
	struct net_device *dev = NULL;
	bool new_channel = true;
	unsigned long flags;
	struct dqnet_channel *chan_tmp;
	struct list_head *pos, *pos_tmp;
	u8 mac_unique[ETH_UNIQUE_ALEN];
	bool wifi = false;
	bool hostdrv = false;
	char path[sizeof(DQNET_SYSCTL_PATH) + IFNAMSIZ];
	int i;
	struct fpm_hw_info fpm_info;
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	struct dqnet_netdev *ndev_tmp = NULL;
	unsigned int switch_drop_hook = 0, switch_mcast_hook = 0;
#endif
	unsigned long *supported;

#ifdef CONFIG_SYSFS
	struct netdev_rx_queue	*_rx;
#endif
	pr_debug("-->\n");

	status = fpm_get_hw_info(&fpm_info);
	if (status)
		goto done;

	if (pdev->dev.platform_data) {
		dev = pdev->dev.platform_data;
		ndev = netdev_priv(dev);
		if (ndev) {
			if (!strncmp(ndev->name, DQNET_WIFI_NAME_STRING,
				     IFNAMSIZ))
				wifi = true;
			else if (ndev->link_type == DQNET_LINK_TYPE_HOSTDRV)
				hostdrv = true;
		}
	} else
		pr_debug("dnet_probe invalid platform data\n");

	if (ndev) {
		pr_debug("ndev=%px\n", ndev);
		if (ndev->name[0]) {
			pr_debug("ndev->name=\"%s\"\n", ndev->name);
		}
	}

	if (!wifi && !hostdrv) {
		dev = alloc_etherdev(sizeof(struct dqnet_netdev));
		if (!dev) {
			status = -ENOMEM;
			goto done;
		}
		ndev = netdev_priv(dev);
		memset(ndev, 0, sizeof(struct dqnet_netdev));
	}

	ndev->msg_enable = dqnet_default_msg;
	ndev->swport_imp = -1;
	ndev->phy_power = 1;

	pdev->dev.platform_data = ndev;
	ndev->pdev = pdev;
	ndev->dev = dev;

	ndev->head_pad = fpm_info.net_buf_head_pad;
	ndev->tail_pad = fpm_info.net_buf_tail_pad;

	bcm_nethook_dev_init(dev, dqnet_nethook_tx_fpm);

	chan = kmalloc(sizeof(struct dqnet_channel), GFP_KERNEL);
	if (!chan) {
		status = -ENOMEM;
		goto err_free_netdev;
	}
	memset(chan, 0, sizeof(struct dqnet_channel));
	ndev->chan = chan;

	if (wifi) {
		/* FAP dynamic channel info */
		dqnet_fap_get_q_info(dev, chan);

		spin_lock_irqsave(&chans.lock, flags);
		if (!list_empty(&chans.list)) {
			list_for_each_entry(chan_tmp, &chans.list, list)
			{
				if (!strncmp(ndev->chan->name, chan_tmp->name,
						 sizeof(ndev->chan->name))) {
					memcpy(chan, chan_tmp,
					       sizeof(struct dqnet_channel));
					break;
				}
			}
		}
		spin_unlock_irqrestore(&chans.lock, flags);

		ndev->bmcast_enable = true;
		ndev->brcm_tag = 0;

		if (dev->ethtool_ops) {
			/* Override msglevel APIs for ethtool_ops */
			ndev->ethtool_ops = kmemdup(dev->ethtool_ops,
						 sizeof(struct ethtool_ops),
						 GFP_KERNEL);
			ndev->ethtool_ops->set_msglevel = dqnet_set_msglevel;
			ndev->ethtool_ops->get_msglevel = dqnet_get_msglevel;
			ndev->ethtool_ops_orig = (struct ethtool_ops *)dev->ethtool_ops;
			dev->ethtool_ops = ndev->ethtool_ops;
		}


		/* If DHD Offload, register hook */
		if (ndev->dhd_rx_skb_nethook != NULL ||
			ndev->dhdol_find_sta != NULL) {
			status = bcm_nethook_register_hook(dev,
					BCM_NETHOOK_RX_SKB,
					RX_SKB_PRIO_DQNET_ETH_CLIENT_RETRANS_SRC,
					DQNET_ETH_CLIENT_RETRANSMIT_NAME,
					dqnet_eth_client_retrans_source_hook);
			if (status) {
				netdev_err(dev,
					"Failed to register nethook: \"");
				netdev_err(dev,
					DQNET_ETH_CLIENT_RETRANSMIT_NAME"\"\n");
				goto err_free_chan;
			}

			status = bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_RX_SKB,
					dqnet_eth_client_retrans_source_hook,
					true);
			if (status) {
				netdev_err(dev,
					"Failed to enable nethook: \"");
				netdev_err(dev,
					DQNET_ETH_CLIENT_RETRANSMIT_NAME"\"\n");
				goto err_unregister_hooks;
			}
		}
		if (ndev->dhd_tx_skb_nethook != NULL) {
			status = bcm_nethook_register_hook(dev,
					BCM_NETHOOK_TX_SKB,
					TX_SKB_PRIO_WMF,
					DQNET_WMF_NAME,
					dqnet_tx_skb_wlan_hook);
			if (status) {
				netdev_err(dev,
					"Failed to register nethook: \"");
				netdev_err(dev, DQNET_WMF_NAME"\"\n");
				goto err_free_chan;
			}

			status = bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_TX_SKB,
					dqnet_tx_skb_wlan_hook,
					true);
			if (status) {
				netdev_err(dev,
					"Failed to enable nethook: \"");
				netdev_err(dev,
					DQNET_WMF_NAME"\"\n");
				goto err_unregister_hooks;
			}
		}

	} else if (hostdrv) {
		/* FAP dynamic channel info */
		dqnet_fap_get_q_info(dev, chan);

		spin_lock_irqsave(&chans.lock, flags);
		if (!list_empty(&chans.list)) {
			list_for_each_entry(chan_tmp, &chans.list, list)
			{
				if (!strncmp(ndev->chan->name, chan_tmp->name,
						 sizeof(ndev->chan->name))) {
					memcpy(chan, chan_tmp,
					       sizeof(struct dqnet_channel));
					break;
				}
			}
		}
		spin_unlock_irqrestore(&chans.lock, flags);

		ndev->bmcast_enable = true;
		ndev->brcm_tag = 0;

		if (dev->ethtool_ops) {
			/* Override msglevel APIs for ethtool_ops */
			((struct ethtool_ops *) dev->ethtool_ops)->set_msglevel
				= dqnet_set_msglevel;
			((struct ethtool_ops *) dev->ethtool_ops)->get_msglevel
				= dqnet_get_msglevel;
		}
	} else {
		SET_NETDEV_DEV(dev, &pdev->dev);
		status = dqnet_parse_dt_node(pdev);
		if (status)
			goto err_free_chan;

		snprintf(dev->name, IFNAMSIZ, "%s", ndev->name);
		ndev->name = dev->name;

		if (is_zero_ether_addr(ndev->mac_addr)) {
			memcpy(ndev->mac_addr, mfg_mac_addr, ETH_MFG_ALEN);
			get_random_bytes(mac_unique, ETH_UNIQUE_ALEN);
			memcpy(&ndev->mac_addr[ETH_MFG_ALEN], mac_unique,
			       ETH_UNIQUE_ALEN);
		}
		memcpy(dev->dev_addr, ndev->mac_addr, sizeof(ndev->mac_addr));

		dev->ethtool_ops = &dqnet_ethernet_ethtool_ops;

		netif_carrier_off(dev);
		ndev->link_state = DQNET_LINK_STATE_DOWN;

		switch (ndev->link_type) {
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		case DQNET_LINK_TYPE_SWITCH:
			ndev->link_stats = SWITCH->link_stats;
			status = SWITCH->connect_port(dev);
			if (status)
				goto err_free_chan;
			ndev->bmcast_enable = false;
			ndev->brcmtag_opc_mc = true;
			status = bcm_nethook_register_hook(dev,
				BCM_NETHOOK_TX_SKB,
				TX_SKB_PRIO_DQNET_DROP_SWITCH_TO_SWITCH,
				DQNET_DROP_SWITCH_TO_SWITCH_NAME,
				dqnet_drop_switch_to_switch_hook);
			if (status) {
				netdev_err(dev,
					"Failed to register nethook: \"");
				netdev_err(dev,
					DQNET_DROP_SWITCH_TO_SWITCH_NAME"\"\n");
				goto err_close_link;
			}
			switch_drop_hook = 1;
			status = bcm_nethook_enable_hook(dev,
				BCM_NETHOOK_TX_SKB,
				dqnet_drop_switch_to_switch_hook,
				true);
			if (status) {
				netdev_err(dev,
					"Failed to enable nethook: \"");
				netdev_err(dev,
					DQNET_DROP_SWITCH_TO_SWITCH_NAME"\"\n");
				goto err_close_link;
			}
			if (!ndev->brcm_tag) {
				status = bcm_nethook_register_hook(dev,
					BCM_NETHOOK_TX_SKB,
					TX_SKB_PRIO_DQNET_TAG_SWITCH_MCASTS,
					DQNET_TAG_SWITCH_MCASTS_NAME,
					dqnet_tag_switch_mcasts_hook);
				if (status) {
					netdev_err(dev, "Failed to register nethook: \"");
					netdev_err(dev, DQNET_TAG_SWITCH_MCASTS_NAME"\"\n");
					goto err_close_link;
				}
				switch_mcast_hook = 1;
				status = bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_TX_SKB,
					dqnet_tag_switch_mcasts_hook,
					true);
				if (status) {
					netdev_err(dev, "Failed to enable nethook: \"");
					netdev_err(dev, DQNET_TAG_SWITCH_MCASTS_NAME"\"\n");
					goto err_close_link;
				}
				ndev->dqnet_vlan_enable =
					SWITCH->vlan_enabled();
			}
			break;
#endif
		case DQNET_LINK_TYPE_PHY:
			ndev->phydev = of_phy_connect(dev, ndev->phy_dn,
						      dqnet_phy_link_cb, 0,
						      ndev->phy_mode);
			if (!ndev->phydev) {
				netdev_err(dev, "Failed to connect to phy.\n");
				goto err_close_link;
			}

			phy_suspend(ndev->phydev);
			phy_support_asym_pause(ndev->phydev);
			if (!ndev->link_stats &&
			    ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT)
				ndev->link_stats = dqnet_fap_link_stats;

			ndev->bmcast_enable = true;
			break;

		case DQNET_LINK_TYPE_RPC:
			if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT ||
			    ndev->chan->type == DQNET_CHAN_TYPE_FAP_HOST)
				ndev->link_stats = dqnet_fap_link_stats;
			else {
				status = bcm_nethook_register_hook(dev,
					BCM_NETHOOK_TX_SKB,
					TX_SKB_PRIO_DQNET_DROP_PRIV,
					DQNET_DROP_PRIV_NAME,
					dqnet_tx_drop_priv_cm_hook);
				if (status) {
					netdev_err(dev,
						"Failed to register nethook tx: \"");
					netdev_err(dev,
						DQNET_DROP_PRIV_NAME"\"\n");
					goto err_close_link;
				}
				status = bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_TX_SKB,
					dqnet_tx_drop_priv_cm_hook,
					true);
				if (status) {
					netdev_err(dev,
						"Failed to enable nethook tx: \"");
					netdev_err(dev,
						DQNET_DROP_PRIV_NAME"\"\n");
					goto err_close_link;
				}
			}

			ndev->rpc_tunnel =
				rpc_get_fifo_tunnel_id(ndev->rpc_name);
			if (ndev->rpc_tunnel < 0) {
				netdev_err(dev,
					"Failed to get RPC tunnel %s.\n",
					ndev->rpc_name);
				status = ndev->rpc_tunnel;
				goto err_free_chan;
			}
			ndev->bmcast_enable = true;
			break;
		case DQNET_LINK_TYPE_FIXED:
			ndev->link_state = DQNET_LINK_STATE_UP;
			netif_carrier_on(ndev->dev);
			break;
		default:
			netdev_err(dev, "Unknown link type ndev->link_type=%d.\n",ndev->link_type);
			status = -EIO;
			goto err_free_chan;
		}
	}

	if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT ||
		ndev->chan->type == DQNET_CHAN_TYPE_FAP_HOST) {
		status = bcm_nethook_register_hook(dev,
			BCM_NETHOOK_TX_SKB,
			TX_SKB_PRIO_DQNET_DROP_PRIV,
			DQNET_DROP_PRIV_NAME,
			dqnet_tx_drop_priv_hook);
		if (status) {
			netdev_err(dev,
				"Failed to register nethook tx: \"");
			netdev_err(dev,
				DQNET_DROP_PRIV_NAME"\"\n");
			goto err_close_link;
		}
		status = bcm_nethook_enable_hook(dev,
			BCM_NETHOOK_TX_SKB,
			dqnet_tx_drop_priv_hook,
			true);
		if (status) {
			netdev_err(dev,
				"Failed to enable nethook tx: \"");
			netdev_err(dev,
				DQNET_DROP_PRIV_NAME"\"\n");
			goto err_close_link;
		}
		status = bcm_nethook_register_hook(dev,
			BCM_NETHOOK_RX_SKB,
			RX_SKB_PRIO_DQNET_DROP_PRIV,
			DQNET_DROP_PRIV_NAME,
			dqnet_rx_drop_priv_hook);
		if (status) {
			netdev_err(dev,
				"Failed to register nethook rx: \"");
			netdev_err(dev,
				DQNET_DROP_PRIV_NAME"\"\n");
			goto err_close_link;
		}
		status = bcm_nethook_enable_hook(dev,
			BCM_NETHOOK_RX_SKB,
			dqnet_rx_drop_priv_hook,
			true);
		if (status) {
			netdev_err(dev,
				"Failed to enable nethook rx: \"");
			netdev_err(dev,
				DQNET_DROP_PRIV_NAME"\"\n");
			goto err_close_link;
		}
	}

#ifdef FPM_IN_SKB
	ndev->fpm_in_skb = 0;
#endif

	status = dqnet_init_q_msg_funcs(chan);
	if (status) {
		netdev_err(dev, "Unknown Q msg format, %d, for %s.\n",
			   chan->qmsg_fmt, ndev->name);
		goto err_close_link;
	}

	status = dqnet_fap_init_port(dev);
	if (status) {
		netdev_err(dev, "Failed to init FAP port for %s.\n",
			   ndev->name);
		goto err_close_link;
	}

	spin_lock_irqsave(&chans.lock, flags);
	if (!list_empty(&chans.list)) {
		list_for_each_entry(chan, &chans.list, list)
		{
			if (!strncmp(ndev->chan->name, chan->name,
					 sizeof(ndev->chan->name))) {
				new_channel = false;
				break;
			}
		}
	}
	spin_unlock_irqrestore(&chans.lock, flags);

	if (!new_channel) {
		kfree(ndev->chan);
		ndev->chan = chan;
	} else {
		chan = ndev->chan;
		spin_lock_irqsave(&chans.lock, flags);
		list_add(&chan->list, &chans.list);
		spin_unlock_irqrestore(&chans.lock, flags);
		chan->msgs = kmalloc(sizeof(u32) * DQM_MAX_MSGSZ * dqnet_napi_weight,
				     GFP_KERNEL);
		chan->max_msg = dqnet_napi_weight;
		chan->napi.weight = dqnet_napi_weight;
		if (!chan->msgs) {
			pr_err("%s: Failed to alloc NAPI msgs buffer ", __func__);
			pr_err("for chan %s.\n", chan->name);
			goto err_close_link;
		}

	}
#if defined(CONFIG_BCM_PWR_RPC)
	/* init work for bottom half mbox irq*/
	if ((strstr(ndev->dev->name, "eth")) && (ndev->link_type == DQNET_LINK_TYPE_PHY))
		INIT_WORK(&ndev->work, dqnet_brcm_pwr_rpc_mbox_work_handler);
#endif

#ifdef CONFIG_SYSFS
	if (!wifi && !hostdrv) {
		_rx = dev->_rx;
		dev->_rx = kcalloc(chan->rx_q_count, sizeof(struct netdev_rx_queue),
				   GFP_KERNEL);
		dev->_rx[0] = _rx[0];
		for (i = 1; i < chan->rx_q_count; i++)
			dev->_rx[i].dev = dev;
		dev->num_rx_queues = chan->rx_q_count;
		rtnl_lock();
		netif_set_real_num_rx_queues(ndev->dev, chan->rx_q_count);
		rtnl_unlock();
		kfree(_rx);
		dqnet_rxqueue_cpu_affinity_default(ndev);
	}
#endif
	chan->ref_count++;
	spin_lock_irqsave(&ndevs_open.lock, flags);
	list_add_tail_rcu(&ndev->list, &ndevs.list);
	spin_unlock_irqrestore(&ndevs_open.lock, flags);
	synchronize_rcu();
	if(wifi) {
		ndev->wifi_ops = kmemdup(dev->netdev_ops,
					 sizeof(struct net_device_ops),
					 GFP_KERNEL);
		ndev->wifi_ops->ndo_start_xmit = dqnet_tx_skb;
		ndev->wifi_ops_orig = (struct net_device_ops *)dev->netdev_ops;
		dev->netdev_ops = ndev->wifi_ops;
		dev->group = BCM_NETDEVICE_GROUP_LAN;
	} else if (hostdrv) {
		ndev->hostdrv_ops = kmemdup(dev->netdev_ops,
					 sizeof(struct net_device_ops),
					 GFP_KERNEL);
		ndev->hostdrv_ops->ndo_start_xmit = dqnet_tx_skb;
		ndev->hostdrv_ops_orig = (struct net_device_ops *)dev->netdev_ops;
		dev->netdev_ops = ndev->hostdrv_ops;
		dev->group = BCM_NETDEVICE_GROUP_LAN;
	} else {
		netdev_dbg(dev, "Device allocated.\n");

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		if (ndev->link_type == DQNET_LINK_TYPE_SWITCH)
			dev->features |= NETIF_F_HW_VLAN_CTAG_FILTER;
#endif
		dev->netdev_ops = &dqnet_netdev_ops;
		dev->watchdog_timeo = DQNET_WATCHDOG_TIMEOUT;

		ndev->hal.phydev = dev->phydev;
		ndev->hal.netdev = dev;
		haldev_init(&ndev->hal);

		status = register_netdev(dev);
		if (status) {
			netdev_err(dev,
				"Failed to register network device.\n");
			goto err_remove_chan;
		}

		ndev->sysctl.tbl = kmemdup(&ndev_sysctl_tbl,
					   sizeof(ndev_sysctl_tbl),
					   GFP_KERNEL);
		if (!ndev->sysctl.tbl) {
			netdev_err(dev,
				"Failed to allocate sysctl table.\n");
			status = -ENOMEM;
			goto err_unregister;
		}
		for (i = 0; i < ARRAY_SIZE(ndev_sysctl_tbl) - 1; i++) {
			/* Set Sys Table data to current netdev address
			   from it's offset*/
			#if defined (CONFIG_ARM64)
				ndev->sysctl.tbl[i].data = ((void *) ndev +
					(u64) ndev->sysctl.tbl[i].data);
			#else
				ndev->sysctl.tbl[i].data = ((void *) ndev +
					(u32) ndev->sysctl.tbl[i].data);
			#endif
			ndev->sysctl.tbl[i].extra1 = dev;
		}
		snprintf(path, sizeof(path), DQNET_SYSCTL_PATH"%s",
			 ndev->name);
		ndev->sysctl.hdr = register_net_sysctl(&init_net, path,
						ndev->sysctl.tbl);
		if (!ndev->sysctl.hdr) {
			netdev_err(dev,
				"Failed to register sysctl table.\n");
			status = -EIO;
			goto err_free_sysctl;
		}

		if (!dev->phydev || (ndev->if_id == 7 && ndev->if_sub_id == 7))
			goto reg_done;

		supported = dev->phydev->supported;
		if (!(linkmode_test_bit(ETHTOOL_LINK_MODE_10000baseT_Full_BIT, supported) ||
				linkmode_test_bit(ETHTOOL_LINK_MODE_5000baseT_Full_BIT, supported) ||
				linkmode_test_bit(ETHTOOL_LINK_MODE_2500baseT_Full_BIT, supported)))
			goto reg_done;

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		ndev->swport_imp_reserve = true;
		dqnet_fap_set_imp_lag_port(dev, 1, DQNET_SET_IMP_RESERVE);
		rcu_read_lock();
		list_for_each_entry_rcu(ndev_tmp, &ndevs.list, list) {
			if (ndev_tmp->link_type != DQNET_LINK_TYPE_SWITCH)
				continue;

			dqnet_fap_set_imp_lag_port(ndev_tmp->dev, 0,
					DQNET_SET_IMP_AUTO);
		}
		rcu_read_unlock();
		synchronize_rcu();
#endif
reg_done:
		netdev_dbg(dev, "Device registered.\n");
	}

	dqnetstats_register_dev(ndev);

	pr_info("%s: net device %s\n", MODULE_NAME, ndev->name);

	goto done;

err_free_sysctl:
	kfree(ndev->sysctl.tbl);

err_unregister:
	unregister_netdev(dev);

err_remove_chan:
	rcu_read_lock();
	{
	struct dqnet_netdev *tmp_ndev = NULL;
	list_for_each_entry_rcu(tmp_ndev, &ndevs.list, list) {
		if (tmp_ndev == ndev) {
			spin_lock_irqsave(&ndevs_open.lock, flags);
			list_del_rcu(&tmp_ndev->list);
			spin_unlock_irqrestore(&ndevs_open.lock, flags);
			break;
		}
	}
	}
	rcu_read_unlock();
	synchronize_rcu();

	chan->ref_count--;
	if (chan->ref_count == 0) {
		spin_lock_irqsave(&chans.lock, flags);
		list_for_each_safe(pos, pos_tmp, &chans.list)
		{
			chan_tmp = list_entry(pos, struct dqnet_channel, list);
			if (chan_tmp == chan)
				break;
		}
		list_del(pos);
		spin_unlock_irqrestore(&chans.lock, flags);
		kfree(chan->msgs);
	}

err_close_link:
	if (ndev->phydev)
		phy_disconnect(ndev->phydev);

	hal_exit(&ndev->hal);

err_unregister_hooks:
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	if (switch_drop_hook)
		bcm_nethook_unregister_hook(dev, BCM_NETHOOK_TX_SKB,
				*dqnet_drop_switch_to_switch_hook);

	if (switch_mcast_hook)
		bcm_nethook_unregister_hook(dev, BCM_NETHOOK_TX_SKB,
				dqnet_tag_switch_mcasts_hook);
#endif
	if (wifi && (ndev->dhd_rx_skb_nethook != NULL ||
				 ndev->dhdol_find_sta != NULL))
		bcm_nethook_unregister_hook(dev, BCM_NETHOOK_RX_SKB,
			dqnet_eth_client_retrans_source_hook);

	if (wifi && (ndev->dhd_tx_skb_nethook != NULL))
		bcm_nethook_unregister_hook(dev, BCM_NETHOOK_TX_SKB,
			dqnet_tx_skb_wlan_hook);

	bcm_nethook_dev_fini(dev);
	pdev->dev.platform_data = NULL;

err_free_chan:
	if (chan->ref_count == 0)
		kfree(chan);

err_free_netdev:
	bcm_nethook_dev_fini(dev);
	pdev->dev.platform_data = NULL;

	if (!wifi)
		free_netdev(dev);

done:
	pr_debug("<--\n");
	return status;
}

/*
 * Device remove
 *
 * Returns
 *	0 for success, < 0 is error code
 */
int dqnet_remove(struct platform_device *pdev)
{
	int status = 0;
	struct dqnet_netdev *ndev = NULL;
	struct dqnet_channel *chan;
	unsigned long flags;
	struct dqnet_channel *chan_tmp;
	struct dqnet_netdev *ndev_tmp;
	struct list_head *pos, *pos_tmp;
	bool wifi = false;
	bool hostdrv = false;

	if (!pdev) {
		pr_err("%s pdev uninitialized!!!\n", __func__);
		status = -EINVAL;
		goto done;
	}
	ndev = pdev->dev.platform_data;
	if (!ndev) {
		pr_err("%s called with uninitialized platform_data!!!\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}

	if (!ndev->dev) {
		pr_err("%s called with uninitialized net_device!!!\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}

	netdev_dbg(ndev->dev, "-->\n");
#if defined(CONFIG_BCM_PWR_RPC)
	/* cancel scheduled work */
	if ((strstr(ndev->dev->name, "eth")) && (ndev->link_type == DQNET_LINK_TYPE_PHY))
		cancel_work_sync(&ndev->work);
#endif
	if (!strcmp(ndev->name, DQNET_WIFI_NAME_STRING))
		wifi = true;
	else if (ndev->link_type == DQNET_LINK_TYPE_HOSTDRV)
		hostdrv = true;

	dqnet_fap_exit_port(ndev->dev);

	dqnetstats_unregister_dev(ndev);
	if (!wifi && !hostdrv) {
		unregister_net_sysctl_table(ndev->sysctl.hdr);
		kfree(ndev->sysctl.tbl);
		haldev_uninit(&ndev->hal);
		unregister_netdev(ndev->dev);
	}

	chan = ndev->chan;
	chan->ref_count--;
	if (chan->ref_count == 0) {
		spin_lock_irqsave(&chans.lock, flags);
		list_for_each_safe(pos, pos_tmp, &chans.list)
		{
			chan_tmp = list_entry(pos, struct dqnet_channel, list);
			if (chan_tmp == chan)
				break;
		}
		list_del(pos);
		spin_unlock_irqrestore(&chans.lock, flags);
		kfree(chan->msgs);
		kfree(chan);
	}

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
	if (ndev->link_type == DQNET_LINK_TYPE_SWITCH) {
		bcm_nethook_unregister_hook(ndev->dev, BCM_NETHOOK_TX_SKB,
			*dqnet_drop_switch_to_switch_hook);
		if (!ndev->brcm_tag)
			bcm_nethook_unregister_hook(ndev->dev,
				BCM_NETHOOK_TX_SKB,
				dqnet_tag_switch_mcasts_hook);
	}
#endif
	if (wifi && (ndev->dhd_rx_skb_nethook != NULL ||
				 ndev->dhdol_find_sta != NULL)) {
		bcm_nethook_unregister_hook(ndev->dev, BCM_NETHOOK_RX_SKB,
			dqnet_eth_client_retrans_source_hook);
	}
	if (wifi && (ndev->dhd_tx_skb_nethook != NULL)) {
		bcm_nethook_unregister_hook(ndev->dev, BCM_NETHOOK_TX_SKB,
			dqnet_tx_skb_wlan_hook);
	}
	bcm_nethook_dev_fini(ndev->dev);
	pdev->dev.platform_data = NULL;

	if (wifi) {
		kfree(ndev->wifi_ops);
		ndev->dev->netdev_ops = ndev->wifi_ops_orig;
	} else if (hostdrv) {
		kfree(ndev->hostdrv_ops);
		ndev->dev->netdev_ops = ndev->hostdrv_ops_orig;
	} else {
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		if (ndev->link_type == DQNET_LINK_TYPE_SWITCH)
			SWITCH->disconnect_port(ndev->dev);
#endif
		if (ndev->link_type == DQNET_LINK_TYPE_PHY)
			phy_disconnect(ndev->phydev);

		kfree(ndev->wifi_ops);
		ndev->dev->netdev_ops = ndev->wifi_ops_orig;
		kfree(ndev->ethtool_ops);
		ndev->dev->ethtool_ops = ndev->ethtool_ops_orig;
		hal_exit(&ndev->hal);
		free_netdev(ndev->dev);
	}

	rcu_read_lock();
	list_for_each_entry_rcu(ndev_tmp, &ndevs.list, list) {
		if (ndev_tmp == ndev) {
			spin_lock_irqsave(&ndevs_open.lock, flags);
			list_del_rcu(&ndev_tmp->list);
			spin_unlock_irqrestore(&ndevs_open.lock, flags);
			break;
		}
	}
	rcu_read_unlock();
	synchronize_rcu();

done:
	pr_debug("<--\n");
	return status;
}

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
int dqnet_set_wan(char *ifname, int enable)
{
	struct dqnet_netdev *ndev;
	int j;
	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (strncmp(ndev->dev->name, ifname, sizeof(IFNAMSIZ)))
			continue;

		if (ndev->link_type != DQNET_LINK_TYPE_SWITCH)
			continue;

		if (enable) {
			if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) ==
			    BCM_NETDEVICE_GROUP_WAN) {
				goto done;
			}
		} else {
			if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) !=
			    BCM_NETDEVICE_GROUP_WAN) {
				goto done;
			}
		}
		for (j = 0; j < MAX_PHY_PORTS; j++) {
			if (j == ndev->if_sub_id)
				continue;
			dqnet_swport_fwd[ndev->if_sub_id][j] = enable;
			dqnet_swport_fwd[j][ndev->if_sub_id] = enable;
		}
	}
done:
	rcu_read_unlock();
	return dqnet_fap_set_wan(&ndevs, ifname, enable);
}
#else
int dqnet_set_wan(char *ifname, int enable)
{
	struct dqnet_netdev *ndev;

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (strncmp(ndev->dev->name, ifname, sizeof(IFNAMSIZ)))
			continue;

		if (enable) {
			if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) ==
				BCM_NETDEVICE_GROUP_WAN) {
				goto done;
			}
		} else {
			if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) !=
				BCM_NETDEVICE_GROUP_WAN) {
				goto done;
			}
		}
	}
done:
	rcu_read_unlock();
	return dqnet_fap_set_wan(&ndevs, ifname, enable);
}
#endif

int dqnet_balance_imp_lag_ports(int action, int poi)
{
	return dqnet_fap_balance_imp_lag_ports(&ndevs, action, poi);
}

int dqnet_get_wifi_idx(struct dqnet_netdev *ndev) {
	return dqnet_fap_get_wifi_idx(ndev);
}

void *dqnet_register_napi_complete(void *dev,
					 void (*napi_complete_hook)(void *))
{
	struct dqnet_netdev *ndev = netdev_priv((struct net_device *)dev);

	rcu_read_lock();
	rcu_assign_pointer(ndev->chan->napi_complete.napi_complete_hook,
			   napi_complete_hook);
	atomic_inc(&ndev->chan->napi_complete.refcnt);
	rcu_read_unlock();
	return ndev->chan;
}
EXPORT_SYMBOL(dqnet_register_napi_complete);

void dqnet_unregister_napi_complete(void *channel)
{
	struct dqnet_channel *chan = channel;
	int val;

	rcu_read_lock();
	val = atomic_dec_if_positive(&chan->napi_complete.refcnt);
	if (val == 0)
		rcu_assign_pointer(chan->napi_complete.napi_complete_hook, NULL);
	rcu_read_unlock();
}
EXPORT_SYMBOL(dqnet_unregister_napi_complete);

int bcm_nethook_dev(struct net_device *dev)
{
	struct dqnet_netdev *ndev;
	int found = 0;

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (ndev->dev == dev) {
			found = 1;
			break;
		}
	}
	rcu_read_unlock();
	synchronize_rcu();

	return found;
}
EXPORT_SYMBOL(bcm_nethook_dev);

int bcm_nethook_register_hook_devs(enum bcm_nethook_type type,
				   enum bcm_nethook_prio priority,
				   char *name, bcm_nethook_fn hook)
{
	struct dqnet_netdev *ndev;
	int devices = 0;

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (bcm_nethook_register_hook(ndev->dev, type, priority, name, hook) == 0) {
			pr_debug("dev %s registering hook %s\n", ndev->dev->name, name);
			devices++;
		}
		else {
			pr_err("failed dev %s registering hook %s\n", ndev->dev->name, name);
		}
	}
	rcu_read_unlock();
	synchronize_rcu();
	return devices;
}
EXPORT_SYMBOL(bcm_nethook_register_hook_devs);

int bcm_nethook_unregister_hook_devs(enum bcm_nethook_type type,
				     bcm_nethook_fn hook)
{
	struct dqnet_netdev *ndev;
	int devices = 0;

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (bcm_nethook_unregister_hook(ndev->dev, type, hook) == 0) {
			pr_debug("dev %s unregistering hook\n", ndev->dev->name);
			devices++;
		}
		else {
			pr_err("failed dev %s unregistering hook\n", ndev->dev->name);
		}
	}
	rcu_read_unlock();
	synchronize_rcu();
	return devices;
}
EXPORT_SYMBOL(bcm_nethook_unregister_hook_devs);

int bcm_nethook_enable_hook_devs(enum bcm_nethook_type type,
				 bcm_nethook_fn hook, bool enable)
{
	struct dqnet_netdev *ndev;
	int devices = 0;

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs.list, list) {
		if (bcm_nethook_enable_hook(ndev->dev, type, hook, enable) == 0) {
			pr_debug("dev %s %s hook\n",
				 ndev->dev->name, enable ? "enable" : "disable");
			devices++;
		}
		else {
			pr_err("failed dev %s %s hook\n",
			       ndev->dev->name, enable ? "enable" : "disable");
		}
		/* If all hooks are enabled (used by bspeed) then disable rx shaper */
		dqnet_fap_enable_rx_shaper(ndev->dev, enable ? false : true);
	}
	rcu_read_unlock();
	synchronize_rcu();
	return devices;
}
EXPORT_SYMBOL(bcm_nethook_enable_hook_devs);

struct net_device_ops dqnet_netdev_ops = {
	.ndo_open		= dqnet_open,
	.ndo_stop		= dqnet_close,
	.ndo_start_xmit		= dqnet_tx_skb,
	.ndo_set_mac_address	= dqnet_set_mac_addr,
	.ndo_validate_addr	= eth_validate_addr,
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0))
	.ndo_eth_ioctl		= dqnet_ioctl,
	.ndo_siocdevprivate	= dqnet_siocdevprivate,
#else
	.ndo_do_ioctl		= dqnet_ioctl,
#endif
	.ndo_tx_timeout		= dqnet_timeout,
	.ndo_get_stats64	= dqnet_get_stats,
	.ndo_change_mtu		= dqnet_set_mtu
};

static const struct of_device_id dqnet_of_match[] = {
	{.compatible = DQNET_OF_MATCH},
	{}
};
MODULE_DEVICE_TABLE(of, dqnet_of_match);
static struct platform_driver dqnet_driver = {
	.probe	= dqnet_probe,
	.remove	= dqnet_remove,
	.driver	= {
		.name		= MODULE_NAME,
		.owner		= THIS_MODULE,
		.of_match_table	= dqnet_of_match
	}
};

static int __init dqnet_init(void)
{
	int status = 0;
	int i;

	pr_debug("-->\n");

	pr_debug("%s driver v%s\n", MODULE_NAME, MODULE_VER);

	haldev_move(DQNET_OF_MATCH);
	dqnet_getsym_fap_ops();
	dqnet_getsym_qmsg_ops();

	dqnet_netdev_ops.ndo_vlan_rx_add_vid = SWITCH->vlan_rx_add_vid;
	dqnet_netdev_ops.ndo_vlan_rx_kill_vid = SWITCH->vlan_rx_kill_vid;

	init_dummy_netdev(&napi_dev);

	INIT_LIST_HEAD_RCU(&ndevs.list);
	INIT_LIST_HEAD(&ndevs_open.list);
	spin_lock_init(&ndevs_open.lock);
	INIT_LIST_HEAD(&chans.list);
	spin_lock_init(&chans.lock);

	if_id_max = dqnet_max_if_id();
	if_sub_id_max = dqnet_max_if_subid();
	if (SWITCH->get_max_ports() > if_sub_id_max)
		if_sub_id_max = SWITCH->get_max_ports();
	if (MAX_PHY_PORTS > if_sub_id_max)
		if_sub_id_max = MAX_PHY_PORTS;
	demux_tbl = kmalloc(sizeof(struct dqnet_netdev **) * if_id_max,
			    GFP_KERNEL);
	if (!demux_tbl) {
		pr_err("Unable to allocate demux_tbl!\n");
		status = -ENOMEM;
		goto done;
	}
	memset(demux_tbl, 0, sizeof(struct dqnet_netdev **) * if_id_max);
	for (i = 0; i < if_id_max; i++) {
		demux_tbl[i] = kmalloc(
			sizeof(struct dqnet_netdev *) * (if_sub_id_max + 1),
			GFP_KERNEL);
		if (!demux_tbl[i]) {
			pr_err("Unable to allocate demux_tbl[%d]!\n", i);
			status = -ENOMEM;
			goto err_free_demux_tbl;
		}
		memset(demux_tbl[i], 0,
		       sizeof(struct dqnet_netdev *) * (if_sub_id_max + 1));
	}
	spin_lock_init(&demux_tbl_lock);

	dqnet_proc_init();
	dqnet_ethstats_init();
	dqnet_dev_weight(xstr(DQNET_NAPI_WEIGHT_DEF));
	dqnet_netdev_budget(xstr(DQNET_NETDEV_BUDGET_DEF));
	dqnet_netdev_max_backlog(xstr(DQNET_NETDEV_MAX_BACKLOG_DEF));
#if defined(CONFIG_BCM_RPC_SERVICES)
	rpc_net_handler = dqnet_rpc_message_handler;
#endif
	dqnet_fap_init_system();
	/* Register notifier callback for power change requests */
#if defined(CONFIG_BCM_PWR_RPC)
	workqueue = alloc_workqueue("DQNETWQ", WQ_HIGHPRI, 0);
	if (!workqueue) {
		pr_err("%s: cannot alloc dqnet workqueue ", __func__);
		status = -ENOMEM;
		goto done;
	}
	brcm_pwr_rpc_mbox_register_notifier(&brcm_pwr_rpc_mbox_notifier);
#endif
	platform_driver_register(&dqnet_driver);
#ifdef CONFIG_NET_SWITCHDEV
	dqnet_swdev_init();
#endif
	goto done;

err_free_demux_tbl:
	for (i = 0; i < if_id_max; i++)
		if (demux_tbl[i])
			kfree(demux_tbl[i]);
	kfree(demux_tbl);

done:
	pr_debug("<--\n");
	return status;
}

static void dqnet_exit(void)
{
	int i;

	pr_debug("-->\n");

#if defined(CONFIG_BCM_PWR_RPC)
	brcm_pwr_rpc_mbox_unregister_notifier(&brcm_pwr_rpc_mbox_notifier);
	if (workqueue) {
		flush_workqueue(workqueue);
		destroy_workqueue(workqueue);
		workqueue = NULL;
	}
#endif
	dqnet_fap_exit_system();
	platform_driver_unregister(&dqnet_driver);
#if defined(CONFIG_BCM_RPC_SERVICES)
	rpc_net_handler = NULL;
#endif
	dqnet_proc_exit();
	dqnet_ethstats_exit();
	for (i = 0; i < if_id_max; i++)
		if (demux_tbl[i])
			kfree(demux_tbl[i]);
	kfree(demux_tbl);

#ifdef CONFIG_NET_SWITCHDEV
	dqnet_swdev_exit();
#endif
	dqnet_putsym_fap_ops();
	dqnet_putsym_qmsg_ops();

	pr_debug("<--\n");
	return;
}

module_init(dqnet_init);
module_exit(dqnet_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("dq_net");
