 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/if_vlan.h>
#include <linux/proc_fs.h>

#include <proc_cmd.h>
#include <bcmnethooks.h>
#include <linux/bcm_media_gw/msgfifo.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include <net/netfilter/nf_conntrack_core.h>
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
#include <net/netfilter/nf_conntrack_offload.h>
#endif

#include "dqnet.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "dqnet_fap.h"
#include "dqnet_nethooks.h"
#include "dqnet_brcmtag.h"
#include "dqnet_switch.h"
#include "fpm.h"

#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
/*
 * DQNet hooks (snoops)
 */
enum bcm_nethook_result dqnet_drop_switch_to_switch_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf;
	struct net_device *dev_in;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct dqnet_netdev *ndev_in;
	struct ethhdr *hdr;

	pr_debug("%s:-->\n",dev->name);
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
	dev_in = skb->dev_in;
#else
	dev_in = NULL;
#endif
	if (dev_in && dev_in->netdev_ops == &dqnet_netdev_ops) {
		enum ip_conntrack_info ctinfo;
		struct nf_conn *ct = nf_ct_get(skb, &ctinfo);
		struct nf_bridge_info *nf_bridge = NULL;
#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
		struct nf_conn_offload *ct_offload = NULL;
		struct offload_info *ct_offload_info = NULL;
#endif

		ndev_in = netdev_priv(dev_in);
		if (ndev->link_type != DQNET_LINK_TYPE_SWITCH ||
		    ndev_in->link_type != DQNET_LINK_TYPE_SWITCH)
			goto done;

		/* Is forwarding allowed from in -> out port */
		if (dqnet_swport_fwd[ndev_in->if_sub_id][ndev->if_sub_id])
			goto done;

		hdr = eth_hdr(skb);
		/* Drop Flooded/Multicast/Broadcast Packets */
		if (skb->pkt_flooded || is_multicast_ether_addr(hdr->h_dest))
			goto drop;

		/* Allow packets if no conntrack */
		if (!ct)
			goto done;

		/* Allow packet that has been nated */
		if ((ct->status & IPS_DST_NAT_DONE) ||
				   (ct->status & IPS_SRC_NAT_DONE))
			goto done;

#ifdef CONFIG_NF_CONNTRACK_OFFLOAD
		ct_offload = nf_conn_offload_find(ct);
		if (!ct_offload)
			goto done;

		ct_offload_info = &ct_offload->info[CTINFO2DIR(ctinfo)];
		nf_bridge = ct_offload_info->nf_bridge;
#endif

		if (!nf_bridge) {
			nf_bridge = nf_bridge_info_get(skb);
			if (!nf_bridge)
				goto done;
		}

#ifdef CONFIG_BCM_FLOWMGR
		if (nf_bridge->bridged)
			goto drop;
#endif

		/* Allow packets that are routed between switch ports */
		goto done;

drop:
		netif_err(ndev, drop, dev,
			  "=======================================\n");
		netif_err(ndev, drop, dev,
			  "DROP: Packet from switch port to switch port.\n");
		netif_err(ndev, drop, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_err(ndev, drop, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, drop, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_drop_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, drop, dev,
			  "=======================================\n");
		err_stats.drop_switch_to_switch++;
		result = BCM_NETHOOK_DROP;
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return result;
}

enum bcm_nethook_result dqnet_drop_redundant_switch_mcasts_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethhdr *hdr;

	pr_debug("%s:-->\n",dev->name);
	hdr = eth_hdr(skb);
	if (!hdr) {
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		netif_err(ndev, tx_err, dev,
			  "DROP: no ethernet header\n");
		netif_err(ndev, tx_err, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_err(ndev, tx_err, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_tx_err_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		err_stats.no_eth_hdr++;
		result = BCM_NETHOOK_DROP;
		goto done;
	}

	if (is_multicast_ether_addr(hdr->h_dest) &&
	    !ndev->bmcast_enable) {
		netif_err(ndev, drop, dev,
			  "===============================================\n");
		netif_err(ndev, drop, dev,
			  "DROP: Redundant (broad/multi)cast of\n");
		netif_err(ndev, drop, dev,
			  "packet to switch.\n");
		netif_err(ndev, drop, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_err(ndev, drop, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, drop, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_drop_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, drop, dev,
			  "===============================================\n");
		err_stats.drop_tx_rebroadcast++;
		result = BCM_NETHOOK_DROP;
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return result;
}

static bool is_eth_ipv6_multicast(const u8 *addr)
{
	/* From RFC-2464
	7.  Address Mapping -- Multicast

	An IPv6 packet with a multicast destination address DST, consisting
	of the sixteen octets DST[1] through DST[16], is transmitted to the
	Ethernet multicast address whose first two octets are the value 3333
	hexadecimal and whose last four octets are the last four octets of
	DST.

				+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
				|0 0 1 1 0 0 1 1|0 0 1 1 0 0 1 1|
				+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
				|   DST[13]     |   DST[14]     |
				+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
				|   DST[15]     |   DST[16]     |
				+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ */

	if ((addr[0] == 0x33) && (addr[1] == 0x33))
		return true;
	else
		return false;
}

#define VLAN_TAG_VID_BASE		(0x0ff0)
enum bcm_nethook_result dqnet_tag_switch_mcasts_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethhdr *hdr;
	struct vlan_ethhdr *veth;
	u16 vlan_tci;

	pr_debug("%s:-->\n",dev->name);
	hdr = eth_hdr(skb);
	if (!hdr) {
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		netif_err(ndev, tx_err, dev,
			  "DROP: no ethernet header\n");
		netif_err(ndev, tx_err, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_err(ndev, tx_err, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_tx_err_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		err_stats.no_eth_hdr++;
		result = BCM_NETHOOK_DROP;
		goto done;
	}

	if ((ndev->link_type == DQNET_LINK_TYPE_SWITCH) &&
	    is_multicast_ether_addr(hdr->h_dest)) {
		if ((ndev->dqnet_vlan_enable == 0)) {
			int start_port = 0;
			if (SWITCH->switch_ipv6_mc_flood_disabled()) {
				/* check for IPv6 multicast */
				if (is_eth_ipv6_multicast(hdr->h_dest)) {
					/* port is learned, start at destination
					 * port so anti-flooding logic doesn't
					 * accidentally drop it
					 */
					start_port = ndev->if_sub_id;
				}
			}
			if (SWITCH->switch_drop_brcst_mult(ndev->if_sub_id,
					start_port)) {
				/*show_err_pkt(skb->data, skb->len);*/
				result = BCM_NETHOOK_DROP;
				goto done;
			}
		} else {

			veth = (struct vlan_ethhdr *)skb->data;
			vlan_tci = VLAN_TAG_VID_BASE + ndev->if_sub_id;
			if ((veth->h_vlan_proto == htons(ETH_P_8021Q) &&
				 veth->h_vlan_TCI != 1))
				veth->h_vlan_TCI = htons(vlan_tci);
			else
				skb = vlan_insert_tag_set_proto(skb, htons(ETH_P_8021Q), vlan_tci);
			netif_dbg(ndev, tx_err, dev,
				  "===============================================\n");
			netif_dbg(ndev, tx_err, dev,
				  "TAGGING: %s packet\n",
				  is_broadcast_ether_addr(hdr->h_dest) ? "broadcast" :
				  "multicast");
			show_tx_dbg_pkt(ndev, skb->data, skb->len);
			netif_dbg(ndev, tx_err, dev,
				  "===============================================\n");
		}
	}
done:
	pr_debug("%s:<--\n",dev->name);
	return result;
}
#endif

#define ETHER_TYPE_IAPP_L2_UPDATE   0x6
enum bcm_nethook_result dqnet_eth_client_retrans_source_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf, *skb2;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethhdr *hdr;
	int ret, wifi_idx;

	pr_debug("%s:-->\n",dev->name);

	hdr = eth_hdr(skb);
	if (!hdr) {
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		netif_err(ndev, tx_err, dev,
			  "DROP: no ethernet header\n");
		netif_err(ndev, tx_err, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_err(ndev, tx_err, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_tx_err_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, tx_err, dev,
			  "===============================================\n");
		err_stats.no_eth_hdr++;
		result = BCM_NETHOOK_DROP;
		goto done;
	}

	wifi_idx = dqnet_get_wifi_idx(ndev);
	if (wifi_idx < 0)
		goto done;

	/* Is TX allowed ? */
	if (ndev->dhdol_find_sta) {
		if (!ndev->dhdol_find_sta(ndev->dhd_cntx, wifi_idx, (void *)hdr->h_dest))
			goto done;
	} else if (ndev->dhd_rx_skb_nethook) {
		if ((ret = ndev->dhd_rx_skb_nethook(ndev->dhd_cntx, wifi_idx, skb))
						!= BCM_NETHOOK_PASS)
			goto done;
	} else
		goto done;

	if (is_multicast_ether_addr(hdr->h_dest)) {
		if ((ntohs(hdr->h_proto) != ETHER_TYPE_IAPP_L2_UPDATE)) {
			skb2 = skb_copy(skb, GFP_ATOMIC);
			if (!skb2)
				goto done;

			skb_push(skb2, ETH_HLEN);
			if ((ret = dev->netdev_ops->ndo_start_xmit(skb2, ndev->dev)) < 0)
				pr_err("%s TX skb error %d\n", __func__, ret);
		}
	} else {
		pr_debug("Sending unicast back to wifi\n");
		skb_push(skb, ETH_HLEN);
		if ((ret = dev->netdev_ops->ndo_start_xmit(skb, ndev->dev)) < 0)
			pr_err("%s TX skb error %d\n", __func__, ret);

		result = BCM_NETHOOK_CONSUMED;
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return result;
}

enum bcm_nethook_result dqnet_tx_skb_wlan_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct ethhdr *hdr;
	int wifi_idx;
	/* TX queue to hold additional packets generated by WMF */
	struct sk_buff_head txq;

	pr_debug("%s:-->\n",dev->name);

	hdr = eth_hdr(skb);
	if (!hdr) {
		netif_err(ndev, tx_err, dev,
				  "===============================================\n");
		netif_err(ndev, tx_err, dev,
				  "DROP: no ethernet header\n");
		netif_err(ndev, tx_err, dev,
				  "TX (%d bytes)\n", skb->len);
		netif_err(ndev, tx_err, dev,
				  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, tx_err, dev,
				  "if_sub_id: %d\n", ndev->if_sub_id);
		show_tx_err_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, tx_err, dev,
				  "===============================================\n");
		err_stats.no_eth_hdr++;
		result = BCM_NETHOOK_DROP;
		goto done;
	}

	wifi_idx = dqnet_get_wifi_idx(ndev);
	if (wifi_idx < 0)
		goto done;

	if (!ndev->dhd_tx_skb_nethook || skb->peeked) {
		skb->peeked = 0;
		goto done;
	}

	skb_queue_head_init(&txq);
	result = ndev->dhd_tx_skb_nethook(ndev->dhd_cntx, wifi_idx, skb, &txq);
	if (result == BCM_NETHOOK_PASS) {
		while ((skb = skb_dequeue(&txq))) {
			skb->peeked = 1;
			dev->netdev_ops->ndo_start_xmit(skb, ndev->dev);
		}
		result = BCM_NETHOOK_CONSUMED;
	} else if (result == BCM_NETHOOK_DROP) {
		netif_err(ndev, drop, dev,
			  "===============================================\n");
		netif_err(ndev, drop, dev,
			  "DROP: WiFi Muticast coversion failure\n");
		netif_err(ndev, drop, dev,
			  "TX (%d bytes)\n", skb->len);
		netif_err(ndev, drop, dev,
			  "if_id: %d\n", ndev->if_id);
		netif_err(ndev, drop, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
		show_drop_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, drop, dev,
			  "===============================================\n");
		err_stats.wlan_conversion_failed++;
	}

done:
	pr_debug("%s:<--\n",dev->name);
	return result;
}

#define PRIV_IPV4_NET 0xAC1FFF00

enum bcm_nethook_result dqnet_rx_drop_priv_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf;
	struct ethhdr *eh;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int eth_proto;
	int protonum = -1;
	int encap = 0;
	int offset = ETH_HLEN;
	int len = skb->len + ETH_HLEN;

	pr_debug("%s:-->\n",dev->name);

	eh = eth_hdr(skb);
	eth_proto = eh->h_proto;
repeat:
	if (encap > 1)
		goto done;
	switch (eth_proto) {
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_IP):
		{
			const struct iphdr *iph;

			if ((offset+sizeof(struct iphdr)) > len)
				goto done;

			iph = (struct iphdr *)((void *) eh + offset);
			if ((iph->daddr & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET)) {
				err_stats.drop_priv_rx[encap][DQNET_PRIV_DROP_IPv4]++;
				goto drop;
			}

			protonum = iph->protocol;
			if (protonum == IPPROTO_GRE) {
				offset += (iph->ihl << 2) + 4;
				len -= offset;
				/* Parse inner gre packet */
				eh = (struct ethhdr *)((void *) eh + offset);
				eth_proto = eh->h_proto;
				offset = ETH_HLEN;
				encap++;
				goto repeat;
			}
		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_IPV6):
		{
			const struct ipv6hdr *ipv6h;

			if ((offset+sizeof(struct ipv6hdr)) > len)
				goto done;

			ipv6h = (struct ipv6hdr *)((void *) eh + offset);
			if (((ipv6h->daddr.s6_addr32[0] & htonl(0xFFC00000)) == htonl(0xFE800000)) &&
				((ipv6h->daddr.s6_addr32[3] & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET))) {
				err_stats.drop_priv_rx[encap][DQNET_PRIV_DROP_IPv6]++;
				goto drop;
			}

			protonum = ipv6h->nexthdr;
			offset += sizeof(*ipv6h);
			if (protonum == NEXTHDR_DEST) {
				struct ipv6_opt_hdr *opt;

				opt = (struct ipv6_opt_hdr *)((void *) eh + offset);
				protonum = opt->nexthdr;
				offset += (opt->hdrlen+1)*8;
			}
			if (protonum == IPPROTO_ICMPV6) {
				const struct nd_msg *nd;

				nd = (struct nd_msg *)((void *) eh + offset);

				if ((nd->icmph.icmp6_type == NDISC_NEIGHBOUR_SOLICITATION) ||
					(nd->icmph.icmp6_type == NDISC_NEIGHBOUR_ADVERTISEMENT)) {

					if (((nd->target.s6_addr32[0] & htonl(0xFFC00000)) == htonl(0xFE800000)) &&
						((nd->target.s6_addr32[3] & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET))) {
						err_stats.drop_priv_rx[encap][DQNET_PRIV_DROP_ICMPv6]++;
						goto drop;
					}
				}
			}

			if (protonum == IPPROTO_GRE) {
				offset += 4;
				len -= offset;
				/* Parse inner gre packet */
				eh = (struct ethhdr *)((void *) eh + offset);
				eth_proto = eh->h_proto;
				offset = ETH_HLEN;
				encap++;
				goto repeat;
			}
		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_ARP):
		{
			const struct arphdr *arph;
			unsigned char *arp_ptr;
			unsigned char *sha;
			__be32 sip, tip;

			if ((offset+sizeof(struct arphdr)) > len)
				goto done;

			arph = (struct arphdr *)((void *) eh + offset);

			if (arph->ar_op != htons(ARPOP_REPLY) &&
				arph->ar_op != htons(ARPOP_REQUEST))
				goto done;

			arp_ptr = (unsigned char *)(arph + 1);
			sha	= arp_ptr;
			arp_ptr += ETH_ALEN;
			memcpy(&sip, arp_ptr, 4);
			if ((sip & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET)) {
				err_stats.drop_priv_rx[encap][DQNET_PRIV_DROP_ARP]++;
				goto drop;
			}

			arp_ptr += 4;
			arp_ptr += ETH_ALEN;
			memcpy(&tip, arp_ptr, 4);
			if ((tip & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET)) {
				err_stats.drop_priv_rx[encap][DQNET_PRIV_DROP_ARP]++;
				goto drop;
			}
		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_8021Q):
		{
			struct vlan_hdr *vhdr;

			vhdr =  (struct vlan_hdr *) ((void *) eh + offset);
			eth_proto = vhdr->h_vlan_encapsulated_proto;
			offset += sizeof(struct vlan_hdr);
			goto repeat;
		}
		break;
	default:
		break;
	}

	goto done;

	/* Drop packets for priv addr */
drop:
	netif_err(ndev, drop, dev,
			  "=======================================\n");
	netif_err(ndev, drop, dev,
			  "DROP: Packet to privbr\n");
	netif_err(ndev, drop, dev,
			  "RX (%d bytes)\n", skb->len);
	netif_err(ndev, drop, dev,
			  "if_id: %d\n", ndev->if_id);
	netif_err(ndev, drop, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
	show_drop_pkt(ndev, skb->data, skb->len);
	netif_err(ndev, drop, dev,
			  "=======================================\n");
	result = BCM_NETHOOK_DROP;

done:
	pr_debug("%s:<--\n",dev->name);
	return result;
}

enum bcm_nethook_result dqnet_tx_drop_priv_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	enum bcm_nethook_result result = BCM_NETHOOK_PASS;
	struct sk_buff *skb = buf;
	struct ethhdr *eh;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int eth_proto;
	int protonum = -1;
	int encap = 0;
	int offset = ETH_HLEN;
	int len = skb->len + ETH_HLEN;

	pr_debug("%s:-->\n",dev->name);

	eh = eth_hdr(skb);
	eth_proto = eh->h_proto;
repeat:
	if (encap > 1)
		goto done;
	switch (eth_proto) {
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_IP):
		{
			const struct iphdr *iph;

			if ((offset+sizeof(struct iphdr)) > len)
				goto done;

			iph = (struct iphdr *)((void *) eh + offset);
			if ((iph->saddr & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET)) {
				err_stats.drop_priv_tx[encap][DQNET_PRIV_DROP_IPv4]++;
				goto drop;
			}

			protonum = iph->protocol;
			if (protonum == IPPROTO_GRE) {
				offset += (iph->ihl << 2) + 4;
				len -= offset;
				/* Parse inner gre packet */
				eh = (struct ethhdr *)((void *) eh + offset);
				eth_proto = eh->h_proto;
				offset = ETH_HLEN;
				encap++;
				goto repeat;
			}
		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_IPV6):
		{
			const struct ipv6hdr *ipv6h;

			if ((offset+sizeof(struct ipv6hdr)) > len)
				goto done;

			ipv6h = (struct ipv6hdr *)((void *) eh + offset);
			if (((ipv6h->saddr.s6_addr32[0] & htonl(0xFFC00000)) == htonl(0xFE800000)) &&
				((ipv6h->saddr.s6_addr32[3] & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET))) {
				err_stats.drop_priv_tx[encap][DQNET_PRIV_DROP_IPv6]++;
				goto drop;
			}

			protonum = ipv6h->nexthdr;
			offset += sizeof(*ipv6h);
			if (protonum == NEXTHDR_DEST) {
				struct ipv6_opt_hdr *opt;

				opt = (struct ipv6_opt_hdr *)((void *) eh + offset);
				protonum = opt->nexthdr;
				offset += (opt->hdrlen+1)*8;
			}
			if (protonum == IPPROTO_ICMPV6) {
				const struct nd_msg *nd;

				nd = (struct nd_msg *)((void *) eh + offset);

				if ((nd->icmph.icmp6_type == NDISC_NEIGHBOUR_SOLICITATION) ||
					(nd->icmph.icmp6_type == NDISC_NEIGHBOUR_ADVERTISEMENT)) {

					if (((nd->target.s6_addr32[0] & htonl(0xFFC00000)) == htonl(0xFE800000)) &&
						((nd->target.s6_addr32[3] & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET))) {
						err_stats.drop_priv_tx[encap][DQNET_PRIV_DROP_ICMPv6]++;
						goto drop;
					}
				}
			}

			if (protonum == IPPROTO_GRE) {
				offset += 4;
				len -= offset;
				/* Parse inner gre packet */
				eh = (struct ethhdr *)((void *) eh + offset);
				eth_proto = eh->h_proto;
				offset = ETH_HLEN;
				encap++;
				goto repeat;
			}

		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_ARP):
		{
			const struct arphdr *arph;
			unsigned char *arp_ptr;
			unsigned char *sha;
			__be32 sip, tip;

			if ((offset+sizeof(struct arphdr)) > len)
				goto done;

			arph = (struct arphdr *)((void *) eh + offset);

			if (arph->ar_op != htons(ARPOP_REPLY) &&
				arph->ar_op != htons(ARPOP_REQUEST))
				goto done;

			arp_ptr = (unsigned char *)(arph + 1);
			sha	= arp_ptr;
			arp_ptr += ETH_ALEN;
			memcpy(&sip, arp_ptr, 4);
			if ((sip & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET)) {
				err_stats.drop_priv_tx[encap][DQNET_PRIV_DROP_ARP]++;
				goto drop;
			}

			arp_ptr += 4;
			arp_ptr += ETH_ALEN;
			memcpy(&tip, arp_ptr, 4);
			if ((tip & htonl(0XFFFFFF00)) == htonl(PRIV_IPV4_NET)) {
				err_stats.drop_priv_tx[encap][DQNET_PRIV_DROP_ARP]++;
				goto drop;
			}
		}
		break;
	// coverity [bad_constant_function_call]
	case ntohs(ETH_P_8021Q):
		{
			struct vlan_hdr *vhdr;

			vhdr =  (struct vlan_hdr *) ((void *) eh + offset);
			eth_proto = vhdr->h_vlan_encapsulated_proto;
			offset += sizeof(struct vlan_hdr);
			goto repeat;
		}
		break;
	default:
		break;
	}

	goto done;
	/* Drop packets for priv addr */
drop:
	netif_err(ndev, drop, dev,
			  "=======================================\n");
	netif_err(ndev, drop, dev,
			  "DROP: Packet from privbr\n");
	netif_err(ndev, drop, dev,
			  "TX (%d bytes)\n", skb->len);
	netif_err(ndev, drop, dev,
			  "if_id: %d\n", ndev->if_id);
	netif_err(ndev, drop, dev,
			  "if_sub_id: %d\n", ndev->if_sub_id);
	show_drop_pkt(ndev, skb->data, skb->len);
	netif_err(ndev, drop, dev,
			  "=======================================\n");
	result = BCM_NETHOOK_DROP;

done:
	pr_debug("%s:<--\n",dev->name);
	return result;
}
