 /****************************************************************************
 *
 * Copyright (c) 2023-2023 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 *******************************************************************************
 *
 * Author: Steven Kern
 *
 ******************************************************************************/

#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/dma-mapping.h>
#include <linux/of_platform.h>

#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/bcm_media_gw/smc_otp/smc_otp.h>

// RPC events sent from eRouter and processed by SMC
typedef enum {
   RPC_ER_TO_SMC_OTP_FIELD_ID,                       /* 0 */
   RPC_ER_TO_SMC_OTP_FIELD_NAME,                     /* 1 */
   RPC_ER_TO_SMC_OTP_FIELD_GET,                      /* 2 */
   RPC_ER_TO_SMC_OTP_FIELD_SET,                      /* 3 */
   RPC_ER_TO_SMC_OTP_COMMIT,                         /* 4 */
   RPC_ER_TO_SMC_OTP_LOCK_REGION,                    /* 5 */
   RPC_ER_TO_SMC_OTP_MAX                             /* 6 */
} rpc_er_to_smc_otp_service;

/* Create a user space device for sending messages */
#define SMC_OTP_CLASS        "smc_otp"
#define SMC_OTP_DEVNAME      "smc_otp"
#define SMC_OTP_MAX_DEVS     1

#define SMC_OTP_OTP_ALIGN     8
#define SMC_OTP_MAX_OTP_SIZE  2048
#define SMC_OTP_RPC_TIMEOUT   5

static struct class     *smc_otp_class;
static int              smc_otp_major;
static dev_t            smc_otp_dev;
static struct cdev      smc_otp_cdev;
static struct device    *smc_otp_device;

static int rpc_smc_tunnel = -1;

static rpc_function smc_otp_table[] = {
   { NULL, 0 },
   { NULL, 0 },
   { NULL, 0 },
   { NULL, 0 }
};

static inline void smc_rpc_msg_set_addr(rpc_msg *msg, u64 v)
{
	msg->data[0] = (msg->data[0] & ~(0xff << 24)) |
		(u32)((v >> 8) & (0xff << 24));
	msg->data[1] = (u32)(v & 0xffffffff);
}

int smc_otp_open(struct inode *inode, struct file *file)
{
	return 0;
}

int smc_otp_release(struct inode *inode, struct file *file)
{
	return 0;
}

ssize_t smc_otp_read(struct file *file,
		                  char __user *buf,
		                  size_t count,
		                  loff_t *ppos)
{
	return 0;
}

ssize_t smc_otp_write(struct file *file,
			                const char __user *buf,
			                size_t count,
			                loff_t *ppos)
{
   return 0;
}

int rpc_send_smc_request_otp_field_id(char *field_name, unsigned int *field_id, unsigned int *field_size)
{
    rpc_msg rpc;
    int ret;

    pr_debug("rpc_send_smc_request_otp_field_id, rpc_smc_tunnel:%d", rpc_smc_tunnel);
    rpc_msg_init(&rpc, RPC_SERVICE_OTP, RPC_ER_TO_SMC_OTP_FIELD_ID, 0, 0, 0, 0);
    strncpy((char*)&rpc.data[1], field_name, 8);

    ret = rpc_send_request_timeout(rpc_smc_tunnel, (rpc_msg *)&rpc, SMC_OTP_RPC_TIMEOUT);
    if (!ret)
    {
        *field_id = rpc.data[0] & 0xff;
        *field_size = ( rpc.data[0] & 0xffff00 ) >> 8;
    }

    return ret;
}

int rpc_send_smc_request_otp_field_name(unsigned int field_id, char *field_name)
{
    rpc_msg rpc;
    int ret;

    pr_debug("rpc_send_smc_request_otp_field_name, rpc_smc_tunnel:%d", rpc_smc_tunnel);
    rpc_msg_init(&rpc, RPC_SERVICE_OTP, (uint32_t)(RPC_ER_TO_SMC_OTP_FIELD_NAME), 0, 0, 0, 0);
    rpc.data[0] = field_id;
    ret = rpc_send_request_timeout(rpc_smc_tunnel, (rpc_msg *)&rpc, SMC_OTP_RPC_TIMEOUT);
    if (!ret)
    {
        strncpy(field_name, (char*)&rpc.data[1], 8);
    }

    return ret;
}

int rpc_send_smc_request_otp_field_get(unsigned int field_id, dma_addr_t dma_addr, unsigned int buffer_size)
{
    rpc_msg rpc;
    int ret;

    pr_debug("rpc_send_smc_request_otp_field_get, rpc_smc_tunnel:%d", rpc_smc_tunnel);
    rpc_msg_init(&rpc, RPC_SERVICE_OTP, (uint32_t)(RPC_ER_TO_SMC_OTP_FIELD_GET), 0, 0, 0, 0);
    rpc.data[0] = field_id;
    smc_rpc_msg_set_addr(&rpc, (u64)dma_addr);
    dma_sync_single_for_device(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
    ret = rpc_send_request_timeout(rpc_smc_tunnel, (rpc_msg *)&rpc, SMC_OTP_RPC_TIMEOUT);
    dma_sync_single_for_cpu(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);

    return ret;
}

int rpc_send_smc_request_otp_field_set(unsigned int field_id, dma_addr_t dma_addr, unsigned int field_len, unsigned int buffer_size)
{
    rpc_msg rpc;
    int ret;

    pr_debug("rpc_send_smc_request_otp_field_set, rpc_smc_tunnel:%d", rpc_smc_tunnel);
    rpc_msg_init(&rpc, RPC_SERVICE_OTP, (uint32_t)(RPC_ER_TO_SMC_OTP_FIELD_SET), 0, 0, 0, 0);
    rpc.data[0] = ((field_len & 0xFFFF) << 8) | field_id;
    smc_rpc_msg_set_addr(&rpc, (u64)dma_addr);
    dma_sync_single_for_device(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
    ret = rpc_send_request_timeout(rpc_smc_tunnel, (rpc_msg *)&rpc, SMC_OTP_RPC_TIMEOUT);
    dma_sync_single_for_cpu(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);

    return ret;
}

int rpc_send_smc_request_otp_field_lock(char *field_name)
{
    rpc_msg rpc;
    int ret;

    pr_debug("rpc_send_smc_request_otp_field_lock, rpc_smc_tunnel:%d", rpc_smc_tunnel);
    rpc_msg_init(&rpc, RPC_SERVICE_OTP, (uint32_t)(RPC_ER_TO_SMC_OTP_LOCK_REGION), 0, 0, 0, 0);
    strncpy((char*)&rpc.data[1], field_name, 8);
    ret = rpc_send_request_timeout(rpc_smc_tunnel, (rpc_msg *)&rpc, SMC_OTP_RPC_TIMEOUT);

    return ret;
}

int rpc_send_smc_request_otp_field_commit(void)
{
    rpc_msg rpc;
    int ret;

    pr_debug("rpc_send_smc_request_otp_field_commit, rpc_smc_tunnel:%d", rpc_smc_tunnel);
    rpc_msg_init(&rpc, RPC_SERVICE_OTP, (uint32_t)(RPC_ER_TO_SMC_OTP_COMMIT), 0, 0, 0, 0);
    ret = rpc_send_request_timeout(rpc_smc_tunnel, (rpc_msg *)&rpc, SMC_OTP_RPC_TIMEOUT);

    return ret;
}

static char *smc_otp_alloc_buffer(struct smc_otp_req *smc_otp_request, unsigned int field_size, unsigned int *buf_size)
{
	char *buffer = NULL;
	unsigned int buffer_size;

	if( ( ( field_size + 7 ) / 8 ) > smc_otp_request->buff_length )
	{
		return NULL;
	}

	buffer_size = smc_otp_request->buff_length;
	if( buffer_size > SMC_OTP_MAX_OTP_SIZE )
	{
		return NULL;
	}
	buffer_size = ALIGN(buffer_size, SMC_OTP_OTP_ALIGN);
	buffer = devm_kzalloc(smc_otp_device, buffer_size, GFP_KERNEL);
	*buf_size = buffer_size;

	return buffer;
}

static long smc_otp_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	unsigned int field_id = 0;
	unsigned int field_size = 0;
	unsigned int buffer_size;
	char *buffer = NULL;
	struct smc_otp_req smc_otp_request;
   dma_addr_t dma_addr;

	ret = copy_from_user(&smc_otp_request, (void *)arg, sizeof(smc_otp_request));
	if (ret)
	{
		dev_err(smc_otp_device, "%s: can't copy smc_otp_req from userspace\n", __func__);
		return ret;
	}

	ret = rpc_send_smc_request_otp_field_id(smc_otp_request.otp_name, &field_id, &field_size);
	if( ret )
	{
		dev_err(smc_otp_device, "Failure finding OTP section %s", smc_otp_request.otp_name);
		return ret;
	}
	smc_otp_request.otp_field_id = field_id;
	smc_otp_request.otp_bits = field_size;

	switch (cmd)
	{
		case SMC_OTP_IOCTL_GET_OTP:
		{
			buffer = smc_otp_alloc_buffer( &smc_otp_request, field_size, &buffer_size );
			if( !buffer )
			{
				return -EINVAL;
			}
			dma_addr = dma_map_single(smc_otp_device, buffer, buffer_size, DMA_BIDIRECTIONAL);
			if( dma_mapping_error( smc_otp_device, dma_addr ) ) {
				dev_err( smc_otp_device, "DMA address not mapped\n" );
				devm_kfree(smc_otp_device, buffer);
				return -EINVAL;
			}

			ret = rpc_send_smc_request_otp_field_get(field_id, dma_addr, buffer_size);
			if( ret )
			{
				dev_err(smc_otp_device, "Failure retrieving OTP section %s", smc_otp_request.otp_name);
				dma_unmap_single(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
				devm_kfree(smc_otp_device, buffer);
				return ret;
			}

			ret = copy_to_user(smc_otp_request.buffer, buffer, buffer_size);

			dma_unmap_single(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
			devm_kfree(smc_otp_device, buffer);

			break;
		}

		case SMC_OTP_IOCTL_SET_OTP:
		{
			buffer = smc_otp_alloc_buffer( &smc_otp_request, field_size, &buffer_size );
			if( !buffer )
			{
				return -EINVAL;
			}
			dma_addr = dma_map_single(smc_otp_device, buffer, buffer_size, DMA_BIDIRECTIONAL);
			if( dma_mapping_error( smc_otp_device, dma_addr ) ) {
				dev_err( smc_otp_device, "DMA address not mapped\n" );
				devm_kfree(smc_otp_device, buffer);
				return -EINVAL;
			}

			ret = copy_from_user(buffer, smc_otp_request.buffer, buffer_size);
			if (ret)
			{
				dev_err(smc_otp_device, "Failure copying OTP data from user. Abort set.");
				dma_unmap_single(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
				devm_kfree(smc_otp_device, buffer);
				return ret;
			}

			ret = rpc_send_smc_request_otp_field_set(field_id, dma_addr, field_size, buffer_size);

			dma_unmap_single(smc_otp_device, dma_addr, buffer_size, DMA_BIDIRECTIONAL);
			devm_kfree(smc_otp_device, buffer);

			break;
		}

		case SMC_OTP_IOCTL_COMMIT_OTP:
		{
			ret = rpc_send_smc_request_otp_field_commit();
			break;
		}

		case SMC_OTP_IOCTL_LOCK_OTP:
		{
			ret = rpc_send_smc_request_otp_field_lock(smc_otp_request.otp_name);
			break;
		}

		default:
		{
			dev_err(smc_otp_device, "Unsupported IOCTL %d\n", cmd);
			ret = -EINVAL;
			break;
		}
	}

	if( ret )
	{
		dev_err(smc_otp_device, "Error during cmd %d\n", cmd);
		return ret;
	}

	// Copy results to user
	ret = copy_to_user((void *)arg, &smc_otp_request, sizeof(smc_otp_request));
	if (ret)
	{
		dev_err(smc_otp_device, "%s: can't copy smc_otp_req to userspace\n", __func__);
	}

	return ret;
}

static const struct file_operations smc_otp_fops = {
	.owner            = THIS_MODULE,
	.open             = smc_otp_open,
	.release          = smc_otp_release,
	.read             = smc_otp_read,
	.write            = smc_otp_write,
	.unlocked_ioctl   = smc_otp_ioctl,
};

int __init smc_otp_init(void)
{
	int status = 0;

	rpc_smc_tunnel = rpc_get_fifo_tunnel_id("rg-smc");
	if (rpc_smc_tunnel == -1)
		dev_err(smc_otp_device, "Unable to get rg-smc RPC tunnel");

	status = rpc_register_functions(RPC_SERVICE_OTP,
                                   smc_otp_table,
                                   sizeof(smc_otp_table)/
                                   sizeof(rpc_function));

	smc_otp_class = class_create(THIS_MODULE, SMC_OTP_CLASS);
	if (IS_ERR(smc_otp_class)) {
		dev_err(smc_otp_device, "class_create() failed for smc_otp_class\n");
		return PTR_ERR(smc_otp_class);
	}
	status = alloc_chrdev_region(&smc_otp_dev, 0, SMC_OTP_MAX_DEVS, SMC_OTP_CLASS);
	smc_otp_major = MAJOR(smc_otp_dev);
	if (status < 0) {
		dev_err(smc_otp_device, "%s: can't alloc chrdev region\n", __func__);
		class_destroy(smc_otp_class);
		return status;
	}
	cdev_init(&smc_otp_cdev, &smc_otp_fops);
	status = cdev_add(&smc_otp_cdev, smc_otp_dev, 1);
	if (status < 0) {
		dev_err(smc_otp_device, "can't register major %d\n", smc_otp_major);
		return status;
	}
	smc_otp_device = device_create(smc_otp_class, NULL, MKDEV(smc_otp_major, 0),
                                     NULL, SMC_OTP_DEVNAME);
	if (IS_ERR(smc_otp_device)) {
		dev_err(smc_otp_device, "%s: can't register class device\n", __func__);
		smc_otp_device = NULL;
		return PTR_ERR(smc_otp_device);
	}

	smc_otp_device->dma_mask = &smc_otp_device->coherent_dma_mask;
	dma_set_mask(smc_otp_device, DMA_BIT_MASK(64));

	return status;
}

void smc_otp_cleanup(void)
{
	if (smc_otp_class)
		class_destroy(smc_otp_class);

	if (smc_otp_major)
		unregister_chrdev_region(MKDEV(smc_otp_major, 0), SMC_OTP_MAX_DEVS);
}

module_init(smc_otp_init);
module_exit(smc_otp_cleanup);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Broadcom Corporation");
MODULE_DESCRIPTION("SMC OTP driver");
