/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2024 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom SMC Power Domain pseudo-Clock Driver
 *
 * This driver creates Linux enable/disable clock devices that can be used
 * by other drivers to switch SoC block power on/off. The Linux clock
 * devices are referred to herein as "pseudo clocks" because they are not
 * actually controlling a true clock within the SoC, but rather using the
 * power domain RPC messages to the SMC to switch the power on/off to SoC
 * blocks.
 *
 * Author: Tim Ross <tim.ross@broadcom.com>
 *****************************************************************************/


#include <linux/of.h>
#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/clk-provider.h>
#include <linux/clkdev.h>
#include <brcm_pwr_rpc.h>

#ifdef pr_fmt
#undef pr_fmt
#endif
#define pr_fmt(fmt) "brcm-pwr-rpc-pseudo-clk: " fmt

struct pwr_rpc_clk {
	struct clk_hw hw;
	char *name;
	int id;
	int state;
};

struct pwr_rpc_clk pwr_rpc_clks[PWR_DOMAINS_MAX];

#define to_pwr_rpc_clk(p) container_of(p, struct pwr_rpc_clk, hw)

static void pwr_rpc_clk_on_off(struct clk_hw *hw, bool on)
{
	struct pwr_rpc_clk *pwr = to_pwr_rpc_clk(hw);
	int state = on ? 0xf : 0;
	int reset = on ? 0 : 1;
	int rc;

	pr_debug("switching power domain %s %s, %s reset\n", pwr->name,
		state == 0 ? "off" : "on", reset ? "asserting" : "deasserting");
	rc = brcm_pwr_rpc_set_domain_state(pwr->name, state, reset);
	if (rc)
		pr_err("failed to set power domain %s's state to %d (%d)\n",
			pwr->name, state, rc);
}

static int pwr_rpc_clk_prepare(struct clk_hw *hw)
{
	pwr_rpc_clk_on_off(hw, true);
	return 0;
}

static void pwr_rpc_clk_unprepare(struct clk_hw *hw)
{
	pwr_rpc_clk_on_off(hw, false);
}

static int pwr_rpc_clk_is_prepared(struct clk_hw *hw)
{
	struct pwr_rpc_clk *pwr = to_pwr_rpc_clk(hw);

	/* 0 is unprepared, anything else is prepared */
	return (pwr->state != 0);
}

static const struct clk_ops pwr_rpc_clk_ops = {
	/*
	 * Need to use (un)prepare instead of (en/dis)able because we need
	 * to use synchronous RPC so we are sure power is on prior to
	 * returning and synchronous RPC will sleep.
	 */
	.prepare = pwr_rpc_clk_prepare,
	.unprepare = pwr_rpc_clk_unprepare,
	.is_prepared = pwr_rpc_clk_is_prepared,
};

int pwr_rpc_pseudo_clk_setup(struct device_node *node)
{
	char *name;
	int rc;
	int id;
	struct pwr_rpc_clk *pwr;
	struct clk_init_data init;
	int state, reset;

	rc = of_property_read_string(node, "pwr-domain-name", (const char **)&name);
	if (rc) {
		pr_err("failed to get pseudo-clock power domain name\n");
		goto done;
	}

	id = rc = brcm_pwr_rpc_register_domain(name);
	if (rc < 0) {
		pr_err("failed to register power domain %s (%d)\n", name, rc);
		goto done;
	}
	pwr = &pwr_rpc_clks[id];

	init.name = name;
	init.ops = &pwr_rpc_clk_ops;
	init.parent_names = NULL;
	init.num_parents = 0;
	init.flags = 0;

	pwr->name = name;
	pwr->id = id;
	rc = brcm_pwr_rpc_get_domain_state(name, &state, &reset);
	if (rc) {
		pr_err("failed to get power domain %s state (%d)\n", name, rc);
		goto err_unreg_pwr_dom;
	}
	pwr->state = state;
	pwr->hw.init = &init;

	rc = of_clk_hw_register(node, &pwr->hw);
	if (rc) {
		pr_err("failed to register hw clk for power domain %s (%d)\n",
		       name, rc);
		goto err_unreg_pwr_dom;
	}

	rc = of_clk_add_hw_provider(node, of_clk_hw_simple_get, pwr);
	if (rc) {
		pr_err("failed to add clk provider for power domain %s (%d)\n",
		       name, rc);
		goto err_unreg_clk_hw;
	}
	pr_info("created clock device for power domain %s\n", name);
	pr_debug("current state: power %s, reset %s\n",
		state ? "on" : "off", reset ? "asserted" : "deasserted");
	goto done;

err_unreg_clk_hw:
	clk_hw_unregister(&pwr->hw);
err_unreg_pwr_dom:
	brcm_pwr_rpc_unregister_domain(id);
done:
	return rc;
}
