/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2018 Broadcom. All rights reserved.
* The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
*
* This program is the proprietary software of Broadcom and/or
* its licensors, and may only be used, duplicated, modified or distributed
* pursuant to the terms and conditions of a separate, written license
* agreement executed between you and Broadcom (an "Authorized License").
* Except as set forth in an Authorized License, Broadcom grants no license
* (express or implied), right to use, or waiver of any kind with respect to
* the Software, and Broadcom expressly reserves all rights in and to the
* Software and all intellectual property rights therein. IF YOU HAVE NO
* AUTHORIZED LICENSE, THEN YOU HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY,
* AND SHOULD IMMEDIATELY NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE
* SOFTWARE.
*
* Except as expressly set forth in the Authorized License,
*
* 1. This program, including its structure, sequence and organization,
* constitutes the valuable trade secrets of Broadcom, and you shall use all
* reasonable efforts to protect the confidentiality thereof, and to use this
* information only in connection with your use of Broadcom integrated circuit
* products.
*
* 2. TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED
* "AS IS" AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS
* OR WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH
* RESPECT TO THE SOFTWARE. BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL
* IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR
* A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET
* ENJOYMENT, QUIET POSSESSION OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME
* THE ENTIRE RISK ARISING OUT OF USE OR PERFORMANCE OF THE SOFTWARE.
*
* 3. TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM
* OR ITS LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL,
* INDIRECT, OR EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY
* RELATING TO YOUR USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM
* HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN
* EXCESS OF THE AMOUNT ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1,
* WHICHEVER IS GREATER. THESE LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY
* FAILURE OF ESSENTIAL PURPOSE OF ANY LIMITED REMEDY.
*
****************************************************************************
*
* Filename:    ramlogrpc.h
*
****************************************************************************
*
* Description: ramlog RPC definitions
*
****************************************************************************/

/**
 * @file    ramlogrpc.h
 * @brief   ramlog RPC definitions
 */

#ifndef __RAMLOGRPC_H_
#define __RAMLOGRPC_H_

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/wait.h>
#include <linux/dma-mapping.h>

#define	RAMLOG_ALIGNED(x)		ALIGN((x), 8)

#define	RAMLOG_FIFO_COUNT		(32 * 1024)
#define	RAMLOG_MSG_BUFFER_SIZE		RAMLOG_ALIGNED(64 * 1024)

#define	RAMLOG_NULL_INDEX		(~0ULL)

#define	RAMLOGFLAG_SEVERITY_MASK	0x0007
#define	RAMLOGFLAG_SEVERITY_LENGTH	3
#define	RAMLOGFLAG_SEVERITY_SHIFT	0

#define	RAMLOGFLAG_SEVERITY_NONE	0x0000
#define	RAMLOGFLAG_SEVERITY_DEEP_DEBUG	0x0006
#define	RAMLOGFLAG_SEVERITY_DEBUG	0x0005
#define	RAMLOGFLAG_SEVERITY_MESSAGE	0x0004
#define	RAMLOGFLAG_SEVERITY_NOTICE	0x0003
#define	RAMLOGFLAG_SEVERITY_WARNING	0x0002
#define	RAMLOGFLAG_SEVERITY_ERROR	0x0001

#define	INVALID_RAMLOG_START_INDEX	0xffffffff
#define	RAMLOG_HEAD_START_INDEX		0xfffffffe
#define	RAMLOG_TAIL_START_INDEX		0xfffffffd

#define	CRYPTO_AES_IV_SIZE		16

/**
 * This is an encrypted log buffer header
 */
struct smc_ramlog_hdr {
	uint32_t magic;
	uint32_t length;
	uint8_t aes_iv[CRYPTO_AES_IV_SIZE];
	uint16_t aes_log_saltindex;
	uint8_t aes_log_keyindex;
	uint8_t hdr_ver;
	uint32_t reserved;
};

/**
 * This structure represents coordinates of SMC ramlog RPC tunnel
 */
struct ramlog_rpc_data {
	struct device *dev;
	int tunnel;
	char tunnel_name[32];
	struct semaphore sem;
};

/**
 * This routine registers all required resources to connect to SMC ramlog DQM
 *
 * @param[out]  ramlog              ramlog coordinates
 * @param[in]   dev                 a device pointer
 * @param[in]   tunnel_name         a tunnel name
 * @return                          zero on success or a negative error code
 */
int register_ramlog_rpc(struct ramlog_rpc_data *ramlog, struct device *dev,
	const char *tunnel_name);

/**
 * This routine registers all required resources to connect to SMC ramlog DQM
 * from a platform device
 *
 * @param[out]  ramlog              ramlog coordinates
 * @param[in]   pdev                a platform device pointer
 * @return                          zero on success or a negative error code
 */
int register_ramlog_rpc_from_platform_device(struct ramlog_rpc_data *ramlog,
	struct platform_device *pdev);

/**
 * This routine releases all required resources to connect to SMC ramlog DQM
 *
 * @param[out]  ramlog              ramlog coordinates
 */
void release_ramlog_rpc(struct ramlog_rpc_data *ramlog);

/**
 * This routine pulls encrypted records from ramlog into a buffer
 *
 * @param[out]  encrypted_dma_addr  store encrypted data there
 * @param[in]   maxcount            max. number of bytes that may be put
 *                                  into the buffer
 * @param[in]   ramlog              ramlog coordinates
 * @param[out]  count_ptr           if not null, a number of records actually
 *                                  pulled is stored there
 * @param[out]  update_ptr          if not null, an update stamp is stored
 *                                  here, but only when the last record was
 *                                  pulled
 * @param[in]   start_index         a position to start or one of the special
 *                                  values:
 *                                  INVALID_RAMLOG_START_INDEX  start at
 *                                                              the head
 *                                  RAMLOG_HEAD_START_INDEX     start at
 *                                                              the head
 *                                  RAMLOG_TAIL_START_INDEX     pull
 *                                                              maxcount most
 *                                                              recent records
 * @return                          an index of the next start position or
 *                                  INVALID_RAMLOG_START_INDEX
 */
uint32_t get_encrypted_ramlog_entries(dma_addr_t encrypted_dma_addr,
	uint32_t maxbufsize, struct ramlog_rpc_data *ramlog,
	uint32_t *count_ptr, uint32_t *update_ptr,
	uint32_t start_index);

/**
 * This routine makes ramlog empty
 *
 * @param[out]  ramlog              ramlog coordinates
 * @return                          zero or an error code
 */
int clear_ramlog(struct ramlog_rpc_data *ramlog);

/**
 * This routine adds another entry to the ramlog
 *
 * @param[out]  ramlog              ramlog coordinates
 * @param[in]   severity            a severity of the message
 * @param[in]   length              length of the message buffer
 *                                  (string length + 1)
 * @param[in]   message_dma_addr    contents of the actual message to report
 * @return                          zero on success or an error code
 */
int add_to_ramlog(struct ramlog_rpc_data *ramlog, unsigned int severity,
	unsigned int length, dma_addr_t message_dma_addr);

/**
 * This routine sets the current value of global reporting level.
 *
 * @param[out]  ramlog              ramlog coordinates
 * @param[in]   level               a new value of ramlog level
 * @param[in]   length              length of the component name buffer
 *                                  (string length + 1)
 * @param[in]   source_dma_addr     a name of the component for which level
 *                                  should be customized. if set to "*" or
 *                                  "!" all custom levels are removed and
 *                                  the global ramlog level is set.
 * @return                          zero on success or an error code
 */
int set_ramlog_level(struct ramlog_rpc_data *ramlog, unsigned int level,
	unsigned int length, dma_addr_t source_dma_addr);

/**
 * This routine sets up an UART flag for ramlog
 *
 * @param[out]  ramlog              ramlog coordinates
 * @param[in]   flag                a new UART flag
 * @return                          zero on success or an error code
 */
int set_ramlog_uart_flag(struct ramlog_rpc_data *ramlog, unsigned int flag);

/**
 * This routine returns the current number of entries present in the ramlog
 *
 * @param[in]   ramlog              ramlog coordinates
 * @return                          a number of entries currently sitting
 *                                  in the log
 */
unsigned int get_ramlog_count(struct ramlog_rpc_data *ramlog);

/**
 * This function returns a 32-bit number that represents the last update
 * of the ramlog
 *
 * @param[in]   ramlog              ramlog coordinates
 * @return                          the last update (counter)
 */
uint32_t get_ramlog_last_update(struct ramlog_rpc_data *ramlog);

#endif	/* __RAMLOGRPC_H_ */
