/*
 * Copyright (c) 2002-2012 Balabit
 * Copyright (c) 1998-2012 Balázs Scheidler
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 */

%code top {
#include "afsocket-parser.h"

}


%code {

#include "afsocket.h"
#include "cfg-parser.h"
#include "afunix-source.h"
#include "afunix-dest.h"
#include "afinet-source.h"
#include "afinet-dest.h"
#include "messages.h"
#include "syslog-names.h"
#include "plugin.h"
#include "cfg-grammar-internal.h"
#include "socket-options-inet.h"
#include "socket-options-unix.h"
#include "transport-mapper-inet.h"
#include "service-management.h"

#include "systemd-syslog-source.h"
#include "afsocket-systemd-override.h"

#include "tlscontext.h"


static SocketOptions *last_sock_options;
static TransportMapper *last_transport_mapper;

TLSContext *last_tls_context;


#if ! SYSLOG_NG_ENABLE_IPV6
#undef AF_INET6
#define AF_INET6 0; g_assert_not_reached()

#endif

static void
afsocket_grammar_set_source_driver(AFSocketSourceDriver *sd)
{
  last_driver = &sd->super.super;

  last_reader_options = &((AFSocketSourceDriver *) last_driver)->reader_options;
  last_sock_options = ((AFSocketSourceDriver *) last_driver)->socket_options;
  last_transport_mapper = ((AFSocketSourceDriver *) last_driver)->transport_mapper;
  last_proto_server_options = &last_reader_options->proto_options.super;
}

static void
afsocket_grammar_set_dest_driver(AFSocketDestDriver *dd)
{
  last_driver = &dd->super.super;

  last_writer_options = &((AFSocketDestDriver *) last_driver)->writer_options;
  last_sock_options = ((AFSocketDestDriver *) last_driver)->socket_options;
  last_transport_mapper = ((AFSocketDestDriver *) last_driver)->transport_mapper;
  last_proto_client_options = &last_writer_options->proto_options.super;
}

void
afunix_grammar_set_source_driver(AFUnixSourceDriver *sd)
{
  afsocket_grammar_set_source_driver(&sd->super);
  last_file_perm_options = &sd->file_perm_options;
}

static void
afinet_grammar_set_source_driver(AFInetSourceDriver *sd)
{
  afsocket_grammar_set_source_driver(&sd->super);
}

static void
afunix_grammar_set_dest_driver(AFUnixDestDriver *dd)
{
  afsocket_grammar_set_dest_driver(&dd->super);
}

static void
afinet_grammar_set_dest_driver(AFInetDestDriver *dd)
{
  afsocket_grammar_set_dest_driver(&dd->super);
}

void
systemd_syslog_grammar_set_source_driver(SystemDSyslogSourceDriver *sd)
{
  afsocket_grammar_set_source_driver(&sd->super);
}

}

%define api.prefix {afsocket_}

/* this parameter is needed in order to instruct bison to use a complete
 * argument list for yylex/yyerror */

%lex-param {CfgLexer *lexer}
%parse-param {CfgLexer *lexer}
%parse-param {LogDriver **instance}
%parse-param {gpointer arg}

%token KW_UNIX_STREAM 20000
%token KW_UNIX_DGRAM
%token KW_TCP
%token KW_UDP
%token KW_TCP6
%token KW_UDP6
%token KW_NETWORK
%token KW_TRANSPORT
%token KW_IP_PROTOCOL
%token KW_SYSTEMD_SYSLOG

%token KW_IP_TTL
%token KW_SO_BROADCAST
%token KW_IP_TOS
%token KW_IP_FREEBIND
%token KW_SO_SNDBUF
%token KW_SO_RCVBUF
%token KW_SO_KEEPALIVE
%token KW_SO_REUSEPORT
%token KW_TCP_KEEPALIVE_TIME
%token KW_TCP_KEEPALIVE_PROBES
%token KW_TCP_KEEPALIVE_INTVL
%token KW_SO_PASSCRED
%token KW_LISTEN_BACKLOG
%token KW_SPOOF_SOURCE
%token KW_SPOOF_SOURCE_MAX_MSGLEN

%token KW_KEEP_ALIVE
%token KW_MAX_CONNECTIONS
%token KW_CLOSE_ON_INPUT

%token KW_LOCALIP
%token KW_IP
%token KW_INTERFACE
%token KW_LOCALPORT
%token KW_DESTPORT
%token KW_FAILOVER_SERVERS
%token KW_FAILOVER
%token KW_SERVERS
%token KW_FAILBACK
%token KW_TCP_PROBE_INTERVAL
%token KW_SUCCESSFUL_PROBES_REQUIRED
%token KW_DYNAMIC_WINDOW_SIZE
%token KW_DYNAMIC_WINDOW_STATS_FREQ
%token KW_DYNAMIC_WINDOW_REALLOC_TICKS

/* SSL support */

%token KW_TLS
%token KW_PEER_VERIFY
%token KW_KEY_FILE
%token KW_CERT_FILE
%token KW_DHPARAM_FILE
%token KW_PKCS12_FILE
%token KW_CA_DIR
%token KW_CRL_DIR
%token KW_CA_FILE
%token KW_TRUSTED_KEYS
%token KW_TRUSTED_DN
%token KW_CIPHER_SUITE
%token KW_TLS12_AND_OLDER
%token KW_TLS13
%token KW_ECDH_CURVE_LIST
%token KW_SSL_OPTIONS
%token KW_SNI
%token KW_ALLOW_COMPRESS
%token KW_KEYLOG_FILE


%require "3.7.6"
%locations
%define api.pure
%define api.value.type {CFG_STYPE}
%define api.location.type {CFG_LTYPE}
%define parse.error verbose

%code {

# define YYLLOC_DEFAULT(Current, Rhs, N)                                \
  do {                                                                  \
    if (N)                                                              \
      {                                                                 \
        (Current).level = YYRHSLOC(Rhs, 1).level;                       \
        (Current).first_line   = YYRHSLOC (Rhs, 1).first_line;          \
        (Current).first_column = YYRHSLOC (Rhs, 1).first_column;        \
        (Current).last_line    = YYRHSLOC (Rhs, N).last_line;           \
        (Current).last_column  = YYRHSLOC (Rhs, N).last_column;         \
      }                                                                 \
    else                                                                \
      {                                                                 \
        (Current).level = YYRHSLOC(Rhs, 0).level;                       \
        (Current).first_line   = (Current).last_line   =                \
          YYRHSLOC (Rhs, 0).last_line;                                  \
        (Current).first_column = (Current).last_column =                \
          YYRHSLOC (Rhs, 0).last_column;                                \
      }                                                                 \
  } while (0)

#define CHECK_ERROR_WITHOUT_MESSAGE(val, token) do {                    \
    if (!(val))                                                         \
      {                                                                 \
        YYERROR;                                                        \
      }                                                                 \
  } while (0)

#define CHECK_ERROR(val, token, errorfmt, ...) do {                     \
    if (!(val))                                                         \
      {                                                                 \
        if (errorfmt)                                                   \
          {                                                             \
            gchar __buf[256];                                           \
            g_snprintf(__buf, sizeof(__buf), errorfmt, ## __VA_ARGS__); \
            yyerror(& (token), lexer, NULL, NULL, __buf);               \
          }                                                             \
        YYERROR;                                                        \
      }                                                                 \
  } while (0)

#define CHECK_ERROR_GERROR(val, token, error, errorfmt, ...) do {       \
    if (!(val))                                                         \
      {                                                                 \
        if (errorfmt)                                                   \
          {                                                             \
            gchar __buf[256];                                           \
            g_snprintf(__buf, sizeof(__buf), errorfmt ", error=%s", ## __VA_ARGS__, error->message); \
            yyerror(& (token), lexer, NULL, NULL, __buf);               \
          }                                                             \
        g_clear_error(&error);						\
        YYERROR;                                                        \
      }                                                                 \
  } while (0)

#define YYMAXDEPTH 20000


}

/* plugin types, must be equal to the numerical values of the plugin type in plugin.h */

%token LL_CONTEXT_ROOT                1
%token LL_CONTEXT_DESTINATION         2
%token LL_CONTEXT_SOURCE              3
%token LL_CONTEXT_PARSER              4
%token LL_CONTEXT_REWRITE             5
%token LL_CONTEXT_FILTER              6
%token LL_CONTEXT_LOG                 7
%token LL_CONTEXT_BLOCK_DEF           8
%token LL_CONTEXT_BLOCK_REF           9
%token LL_CONTEXT_BLOCK_CONTENT       10
%token LL_CONTEXT_BLOCK_ARG           11
%token LL_CONTEXT_PRAGMA              12
%token LL_CONTEXT_FORMAT              13
%token LL_CONTEXT_TEMPLATE_FUNC       14
%token LL_CONTEXT_INNER_DEST          15
%token LL_CONTEXT_INNER_SRC           16
%token LL_CONTEXT_CLIENT_PROTO        17
%token LL_CONTEXT_SERVER_PROTO        18
%token LL_CONTEXT_OPTIONS             19
%token LL_CONTEXT_CONFIG              20

/* this is a placeholder for unit tests, must be the latest & largest */
%token LL_CONTEXT_MAX                 21


/* statements */
%token KW_SOURCE                      10000
%token KW_FILTER                      10001
%token KW_PARSER                      10002
%token KW_DESTINATION                 10003
%token KW_LOG                         10004
%token KW_OPTIONS                     10005
%token KW_INCLUDE                     10006
%token KW_BLOCK                       10007
%token KW_JUNCTION                    10008
%token KW_CHANNEL                     10009
%token KW_IF                          10010
%token KW_ELSE                        10011
%token KW_ELIF                        10012

/* source & destination items */
%token KW_INTERNAL                    10020
%token KW_SYSLOG                      10060

/* option items */
%token KW_MARK_FREQ                   10071
%token KW_STATS_FREQ                  10072
%token KW_STATS_LEVEL                 10073
%token KW_STATS_LIFETIME              10074
%token KW_FLUSH_LINES                 10075
%token KW_SUPPRESS                    10076
%token KW_FLUSH_TIMEOUT               10077
%token KW_LOG_MSG_SIZE                10078
%token KW_FILE_TEMPLATE               10079
%token KW_PROTO_TEMPLATE              10080
%token KW_MARK_MODE                   10081
%token KW_ENCODING                    10082
%token KW_TYPE                        10083
%token KW_STATS_MAX_DYNAMIC           10084
%token KW_MIN_IW_SIZE_PER_READER      10085
%token KW_WORKERS                     10086
%token KW_BATCH_LINES                 10087
%token KW_BATCH_TIMEOUT               10088
%token KW_TRIM_LARGE_MESSAGES         10089

%token KW_CHAIN_HOSTNAMES             10090
%token KW_NORMALIZE_HOSTNAMES         10091
%token KW_KEEP_HOSTNAME               10092
%token KW_CHECK_HOSTNAME              10093
%token KW_BAD_HOSTNAME                10094

%token KW_KEEP_TIMESTAMP              10100

%token KW_USE_DNS                     10110
%token KW_USE_FQDN                    10111
%token KW_CUSTOM_DOMAIN               10112

%token KW_DNS_CACHE                   10120
%token KW_DNS_CACHE_SIZE              10121

%token KW_DNS_CACHE_EXPIRE            10130
%token KW_DNS_CACHE_EXPIRE_FAILED     10131
%token KW_DNS_CACHE_HOSTS             10132

%token KW_PERSIST_ONLY                10140
%token KW_USE_RCPTID                  10141
%token KW_USE_UNIQID                  10142

%token KW_TZ_CONVERT                  10150
%token KW_TS_FORMAT                   10151
%token KW_FRAC_DIGITS                 10152

%token KW_LOG_FIFO_SIZE               10160
%token KW_LOG_FETCH_LIMIT             10162
%token KW_LOG_IW_SIZE                 10163
%token KW_LOG_PREFIX                  10164
%token KW_PROGRAM_OVERRIDE            10165
%token KW_HOST_OVERRIDE               10166

%token KW_THROTTLE                    10170
%token KW_THREADED                    10171
%token KW_PASS_UNIX_CREDENTIALS       10231

%token KW_PERSIST_NAME                10302

%token KW_READ_OLD_RECORDS            10304
%token KW_USE_SYSLOGNG_PID            10305

/* log statement options */
%token KW_FLAGS                       10190

/* reader options */
%token KW_PAD_SIZE                    10200
%token KW_TIME_ZONE                   10201
%token KW_RECV_TIME_ZONE              10202
%token KW_SEND_TIME_ZONE              10203
%token KW_LOCAL_TIME_ZONE             10204
%token KW_FORMAT                      10205

/* destination writer options */
%token KW_TRUNCATE_SIZE               10206

/* timers */
%token KW_TIME_REOPEN                 10210
%token KW_TIME_REAP                   10211
%token KW_TIME_SLEEP                  10212

/* destination options */
%token KW_TMPL_ESCAPE                 10220

/* driver specific options */
%token KW_OPTIONAL                    10230

/* file related options */
%token KW_CREATE_DIRS                 10240

%token KW_OWNER                       10250
%token KW_GROUP                       10251
%token KW_PERM                        10252

%token KW_DIR_OWNER                   10260
%token KW_DIR_GROUP                   10261
%token KW_DIR_PERM                    10262

%token KW_TEMPLATE                    10270
%token KW_TEMPLATE_ESCAPE             10271
%token KW_TEMPLATE_FUNCTION           10272

%token KW_DEFAULT_FACILITY            10300
%token KW_DEFAULT_SEVERITY            10301

%token KW_PORT                        10323
/* misc options */

%token KW_USE_TIME_RECVD              10340

/* filter items*/
%token KW_FACILITY                    10350
%token KW_SEVERITY                    10351
%token KW_HOST                        10352
%token KW_MATCH                       10353
%token KW_MESSAGE                     10354
%token KW_NETMASK                     10355
%token KW_TAGS                        10356
%token KW_NETMASK6                    10357

/* parser items */

/* rewrite items */
%token KW_REWRITE                     10370
%token KW_CONDITION                   10371
%token KW_VALUE                       10372

/* yes/no switches */

%token KW_YES                         10380
%token KW_NO                          10381

%token KW_IFDEF                       10410
%token KW_ENDIF                       10411

%token LL_DOTDOT                      10420
%token LL_DOTDOTDOT                   10421
%token LL_PRAGMA                      10422
%token LL_EOL                         10423
%token LL_ERROR                       10424
%token LL_ARROW                       10425

%token <cptr> LL_IDENTIFIER           10430
%token <num>  LL_NUMBER               10431
%token <fnum> LL_FLOAT                10432
%token <cptr> LL_STRING               10433
%token <token> LL_TOKEN               10434
%token <cptr> LL_BLOCK                10435

%destructor { free($$); } <cptr>

/* value pairs */
%token KW_VALUE_PAIRS                 10500
%token KW_EXCLUDE                     10502
%token KW_PAIR                        10503
%token KW_KEY                         10504
%token KW_SCOPE                       10505
%token KW_SHIFT                       10506
%token KW_SHIFT_LEVELS                10507
%token KW_REKEY                       10508
%token KW_ADD_PREFIX                  10509
%token KW_REPLACE_PREFIX              10510
%token KW_CAST                        10511

%token KW_ON_ERROR                    10520

%token KW_RETRIES                     10521

%token KW_FETCH_NO_DATA_DELAY         10522

%type   <ptr> source_content
%type	<ptr> source_items
%type	<ptr> source_item
%type   <ptr> source_afinter
%type   <ptr> source_plugin
%type   <ptr> source_afinter_params

%type   <ptr> dest_content
%type	<ptr> dest_items
%type	<ptr> dest_item
%type   <ptr> dest_plugin

%type   <ptr> template_content
%type   <ptr> template_content_list

%type   <ptr> filter_content

%type   <ptr> parser_content

%type   <ptr> rewrite_content

%type	<ptr> log_items
%type	<ptr> log_item

%type   <ptr> log_last_junction
%type   <ptr> log_junction
%type   <ptr> log_content
%type   <ptr> log_conditional
%type   <ptr> log_if
%type   <ptr> log_forks
%type   <ptr> log_fork

%type	<num> log_flags
%type   <num> log_flags_items

%type   <ptr> value_pair_option

%type	<num> yesno
%type   <num> dnsmode
%type	<num> dest_writer_options_flags

%type	<cptr> string
%type	<cptr> string_or_number
%type	<cptr> normalized_flag
%type   <ptr> string_list
%type   <ptr> string_list_build
%type   <num> facility_string
%type   <num> severity_string

%type   <num> positive_integer
%type   <num> positive_integer64
%type   <num> nonnegative_integer
%type   <num> nonnegative_integer64
%type   <fnum> positive_float
%type   <fnum> nonnegative_float
%type	<cptr> path_no_check
%type	<cptr> path_secret
%type	<cptr> path_check
%type	<cptr> path


%type   <ptr> driver
%type	<ptr> source_afunix
%type	<ptr> source_afunix_dgram_params
%type	<ptr> source_afunix_stream_params
%type	<ptr> source_afinet
%type	<ptr> source_afinet_udp_params
%type	<ptr> source_afinet_tcp_params
%type	<ptr> source_afinet_udp6_params
%type	<ptr> source_afinet_tcp6_params
%type   <ptr> source_afsyslog
%type   <ptr> source_afsyslog_params
%type	<ptr> source_afnetwork
%type	<ptr> source_afnetwork_params
%type   <ptr> source_afsocket_stream_params
%type	<ptr> source_systemd_syslog
%type	<ptr> source_systemd_syslog_params

%type	<ptr> dest_afunix
%type	<ptr> dest_afunix_dgram_params
%type	<ptr> dest_afunix_stream_params
%type	<ptr> dest_afinet
%type	<ptr> dest_afinet_udp_params
%type	<ptr> dest_afinet_tcp_params
%type	<ptr> dest_afinet_udp6_params
%type	<ptr> dest_afinet_tcp6_params
%type   <ptr> dest_afsyslog
%type   <ptr> dest_afsyslog_params
%type   <ptr> dest_afnetwork
%type   <ptr> dest_afnetwork_params

%type   <num> inet_ip_protocol_option

%type <ptr> dest_afinet_option

%type <ptr> dest_failover_options
%type <ptr> dest_failover_modes_options

%%

start
        : driver
          {
            *instance = $1;
            if (yychar != AFSOCKET_EMPTY)
              cfg_lexer_unput_token(lexer, &yylval);
            YYACCEPT;
          }
        ;

driver
        : LL_CONTEXT_SOURCE source_afunix                     { $$ = $2; }
        | LL_CONTEXT_SOURCE source_afinet                     { $$ = $2; }
        | LL_CONTEXT_SOURCE source_afsyslog                   { $$ = $2; }
        | LL_CONTEXT_SOURCE source_afnetwork                  { $$ = $2; }
        | LL_CONTEXT_SOURCE source_systemd_syslog             { $$ = $2; }
        | LL_CONTEXT_DESTINATION dest_afunix                  { $$ = $2; }
        | LL_CONTEXT_DESTINATION dest_afinet                  { $$ = $2; }
        | LL_CONTEXT_DESTINATION dest_afsyslog                { $$ = $2; }
        | LL_CONTEXT_DESTINATION dest_afnetwork               { $$ = $2; }
        ;


source_afunix
        : KW_UNIX_DGRAM '(' source_afunix_dgram_params ')'	                                { $$ = $3; }
	| KW_UNIX_STREAM '(' source_afunix_stream_params ')' 	                                { $$ = $3; }
	;

source_afunix_dgram_params
	: string
	  {
            create_and_set_unix_dgram_or_systemd_syslog_source($1, configuration);
	  }
	  source_afunix_options			{ $$ = last_driver; free($1); }
	;

source_afunix_stream_params
	: string
	  {
            create_and_set_unix_stream_or_systemd_syslog_source($1, configuration);
	  }
	  source_afunix_options			{ $$ = last_driver; free($1); }
	;

/* options are common between dgram & stream */
source_afunix_options
	: source_afunix_option source_afunix_options
	|
	;

source_afunix_option
        : file_perm_option
	| source_afsocket_stream_params		{}
	| source_reader_option			{}
	| source_driver_option
	| unix_socket_option			{}
	| KW_OPTIONAL '(' yesno ')'		{ last_driver->optional = $3; }
	| KW_PASS_UNIX_CREDENTIALS '(' yesno ')'
	  {
	    AFUnixSourceDriver *self = (AFUnixSourceDriver*) last_driver;
	    afunix_sd_set_pass_unix_credentials(self, $3);
	  }
	| KW_CREATE_DIRS '(' yesno ')'
	  {
	    AFUnixSourceDriver *self = (AFUnixSourceDriver*) last_driver;
	    afunix_sd_set_create_dirs(self, $3);
	  }
	;

source_afinet
	: KW_UDP '(' source_afinet_udp_params ')'		{ $$ = $3; }
	| KW_TCP '(' source_afinet_tcp_params ')'		{ $$ = $3; }
	| KW_UDP6 '(' source_afinet_udp6_params ')'		{ $$ = $3; }
	| KW_TCP6 '(' source_afinet_tcp6_params ')'		{ $$ = $3; }
        ;

source_afinet_udp_params
        :
          {
            AFInetSourceDriver *d = afinet_sd_new_udp(configuration);

            afinet_grammar_set_source_driver(d);
	  }
	  source_afinet_udp_options		{ $$ = last_driver; }
	;

source_afinet_udp6_params
        :
          {
            AFInetSourceDriver *d = afinet_sd_new_udp6(configuration);

            afinet_grammar_set_source_driver(d);
          }
          source_afinet_udp_options		{ $$ = last_driver; }
        ;

source_afinet_udp_options
	: source_afinet_udp_option source_afinet_udp_options
	|
	;

source_afinet_udp_option
	: source_afinet_option
	;

source_afinet_option
	: KW_LOCALIP '(' string ')'		{ afinet_sd_set_localip(last_driver, $3); free($3); }
	| KW_IP '(' string ')'			{ afinet_sd_set_localip(last_driver, $3); free($3); }
	| KW_LOCALPORT '(' string_or_number ')'	{ afinet_sd_set_localport(last_driver, $3); free($3); }
	| KW_PORT '(' string_or_number ')'	{ afinet_sd_set_localport(last_driver, $3); free($3); }
	| source_reader_option
	| source_driver_option
	| inet_socket_option
	;

source_afinet_tcp_params
	:
	  {
	    AFInetSourceDriver *d = afinet_sd_new_tcp(configuration);

	    afinet_grammar_set_source_driver(d);
	  }
	  source_afinet_tcp_options	{ $$ = last_driver; }
	;

source_afinet_tcp6_params
	:
	  {
	    AFInetSourceDriver *d = afinet_sd_new_tcp6(configuration);

	    afinet_grammar_set_source_driver(d);
	  }
	  source_afinet_tcp_options	{ $$ = last_driver; }
	;

source_afinet_tcp_options
	: source_afinet_tcp_option source_afinet_tcp_options
	|
	;

source_afinet_tcp_option
        : source_afinet_option
	| KW_TLS
	  {
            gchar buf[256];

	    last_tls_context = tls_context_new(TM_SERVER, cfg_lexer_format_location(lexer, &@1, buf, sizeof(buf)));
	  }
	  '(' tls_options ')'
	  {
	    afinet_sd_set_tls_context(last_driver, last_tls_context);
          }
	| source_afsocket_stream_params		{}
	;

source_afsocket_stream_params
	: KW_KEEP_ALIVE '(' yesno ')'		{ afsocket_sd_set_keep_alive(last_driver, $3); }
	| KW_MAX_CONNECTIONS '(' positive_integer ')'	 { afsocket_sd_set_max_connections(last_driver, $3); }
	| KW_LISTEN_BACKLOG '(' positive_integer ')'	{ afsocket_sd_set_listen_backlog(last_driver, $3); }
	| KW_DYNAMIC_WINDOW_SIZE '(' nonnegative_integer ')' { afsocket_sd_set_dynamic_window_size(last_driver, $3); }
  | KW_DYNAMIC_WINDOW_STATS_FREQ '(' nonnegative_float ')' { afsocket_sd_set_dynamic_window_stats_freq(last_driver, $3); }
  | KW_DYNAMIC_WINDOW_REALLOC_TICKS '(' nonnegative_integer ')' { afsocket_sd_set_dynamic_window_realloc_ticks(last_driver, $3); }
	;

source_afsyslog
	: KW_SYSLOG '(' source_afsyslog_params ')'		{ $$ = $3; }
	;

source_afsyslog_params
	:
	  {
            /* we use transport(tcp) transport by default */
            AFInetSourceDriver *d = afinet_sd_new_syslog(configuration);

            afinet_grammar_set_source_driver(d);
	  }
	  source_afsyslog_options	{ $$ = last_driver; }
	;

source_afsyslog_options
	: source_afsyslog_option source_afsyslog_options
	|
	;

source_afsyslog_option
        : source_afinet_option
        | source_afsocket_transport
	| source_afsocket_stream_params		{}
	;

source_afnetwork
	: KW_NETWORK '(' source_afnetwork_params ')'    { $$ = $3; }
	;

source_afnetwork_params
	:
	  {
            /* we use transport(tcp) transport by default */
            AFInetSourceDriver *d = afinet_sd_new_network(configuration);

            afinet_grammar_set_source_driver(d);
	  }
	  source_afnetwork_options	{ $$ = last_driver; }
        ;

source_afnetwork_options
	: source_afnetwork_option source_afnetwork_options
	|
	;

source_afnetwork_option
        : source_afinet_option
        | source_afsocket_transport
	| source_afsocket_stream_params		{}
	;

source_afsocket_transport
	: afsocket_transport
        | KW_TRANSPORT '(' string
          {
            Plugin *p;
            gint context = LL_CONTEXT_SERVER_PROTO;

            p = cfg_find_plugin(configuration, context, $3);
            if (p)
              {
                /* for transports with grammar */
                if (p->parser)
                  {
                    LogProtoServerFactory *sf = cfg_parse_plugin(configuration, p, &@3, last_proto_server_options);
                    ((AFSocketSourceDriver *) last_driver)->proto_factory = sf;
                  }
              }
            transport_mapper_set_transport(last_transport_mapper, $3);
          }
        ')'	{ free($3); }
	| KW_TLS
	  {
            gchar buf[256];

	    last_tls_context = tls_context_new(TM_SERVER, cfg_lexer_format_location(lexer, &@1, buf, sizeof(buf)));
	  }
	  '(' tls_options ')'
	  {
	    afinet_sd_set_tls_context(last_driver, last_tls_context);
          }
        ;

source_systemd_syslog
	: KW_SYSTEMD_SYSLOG '(' source_systemd_syslog_params ')'  { $$ = $3; }
	;

source_systemd_syslog_params
	: {
#if ! SYSLOG_NG_ENABLE_SYSTEMD
            msg_error("systemd-syslog() source cannot be enabled and it is not"
                      " functioning. Please compile your syslog-ng with --enable-systemd"
                      " flag", cfg_lexer_format_location_tag(lexer, &@0));
            YYERROR;
#else
            SystemDSyslogSourceDriver *d = systemd_syslog_sd_new(configuration, FALSE);
            systemd_syslog_grammar_set_source_driver(d);
#endif
          }
          source_systemd_syslog_options { $$ = last_driver; }
        ;

source_systemd_syslog_options
        : source_systemd_syslog_option source_systemd_syslog_options
        |
        ;

source_systemd_syslog_option
        : source_reader_option
        | socket_option
	| source_driver_option
        ;

dest_afunix
	: KW_UNIX_DGRAM '(' dest_afunix_dgram_params ')'	{ $$ = $3; }
	| KW_UNIX_STREAM '(' dest_afunix_stream_params ')'	{ $$ = $3; }
	;

dest_afunix_dgram_params
	: string
	  {
	    AFUnixDestDriver *d = afunix_dd_new_dgram($1, configuration);

      afunix_grammar_set_dest_driver(d);
	  }
	  dest_afunix_options			{ $$ = last_driver; free($1); }
	;

dest_afunix_stream_params
	: string
	  {
	    AFUnixDestDriver *d = afunix_dd_new_stream($1, configuration);

      afunix_grammar_set_dest_driver(d);
	  }
	  dest_afunix_options			{ $$ = last_driver; free($1); }
	;

dest_afunix_options
	: dest_afunix_options dest_afunix_option
	|
	;

dest_afunix_option
	: dest_writer_option
	| dest_afsocket_option
	| socket_option
	| dest_driver_option
	;

dest_afinet
	: KW_UDP '(' dest_afinet_udp_params ')'			{ $$ = $3; }
	| KW_TCP '(' dest_afinet_tcp_params ')'			{ $$ = $3; }
	| KW_UDP6 '(' dest_afinet_udp6_params ')'		{ $$ = $3; }
	| KW_TCP6 '(' dest_afinet_tcp6_params ')'		{ $$ = $3; }
	;

dest_afinet_udp_params
	: string
	  {
	    AFInetDestDriver *d = afinet_dd_new_udp($1, configuration);

      afinet_grammar_set_dest_driver(d);
	  }
	  dest_afinet_udp_options		{ $$ = last_driver; free($1); }
	;

dest_afinet_udp6_params
	: string
	  {
	    AFInetDestDriver *d = afinet_dd_new_udp6($1, configuration);

	    afinet_grammar_set_dest_driver(d);
	  }
	  dest_afinet_udp_options		{ $$ = last_driver; free($1); }
	;


dest_afinet_udp_options
        : dest_afinet_udp_options dest_afinet_udp_option
	|
	;


dest_afinet_option
	: KW_LOCALIP '(' string ')'		{ afinet_dd_set_localip(last_driver, $3); free($3); }
	| KW_LOCALPORT '(' string_or_number ')'	{ afinet_dd_set_localport(last_driver, $3); free($3); }
	| KW_PORT '(' string_or_number ')'	{ afinet_dd_set_destport(last_driver, $3); free($3); }
	| KW_DESTPORT '(' string_or_number ')'	{ afinet_dd_set_destport(last_driver, $3); free($3); }
	| KW_FAILOVER_SERVERS { afinet_dd_enable_failover(last_driver); } '(' string_list ')'	{ afinet_dd_add_failovers(last_driver, $4); }
	| KW_FAILOVER { afinet_dd_enable_failover(last_driver); } '(' dest_failover_options ')'	{ $$ = $4; }
	| inet_socket_option
	| dest_writer_option
	| dest_afsocket_option
	| dest_driver_option
	;

dest_failover_options
	: KW_SERVERS '(' string_list ')' { afinet_dd_add_failovers(last_driver, $3); } dest_failover_modes_options
	;

dest_failover_modes_options
  : dest_failback_options
	|
	;

dest_failback_options
  : KW_FAILBACK { afinet_dd_enable_failback(last_driver); } '(' dest_failback_probe_options ')'
	;

dest_failback_probe_options
	: dest_failback_probe_options dest_failback_probe_option
	|
	;

dest_failback_probe_option
	: KW_TCP_PROBE_INTERVAL '(' positive_integer ')' { afinet_dd_set_failback_tcp_probe_interval(last_driver, $3); }
	| KW_SUCCESSFUL_PROBES_REQUIRED '(' positive_integer ')' { afinet_dd_set_failback_successful_probes_required(last_driver, $3); }
	;

dest_afinet_dgram_option
	: KW_SPOOF_SOURCE '(' yesno ')'		{ afinet_dd_set_spoof_source(last_driver, $3); }
	| KW_SPOOF_SOURCE_MAX_MSGLEN '(' positive_integer ')' { afinet_dd_set_spoof_source_max_msglen(last_driver, $3); }
	;

dest_afinet_udp_option
	: dest_afinet_option
	| dest_afinet_dgram_option
	;

dest_afinet_tcp_params
	: string
	  {
	    AFInetDestDriver *d = afinet_dd_new_tcp($1, configuration);

      afinet_grammar_set_dest_driver(d);
	  }
	  dest_afinet_tcp_options		{ $$ = last_driver; free($1); }
	;

dest_afinet_tcp6_params
	: string
	  {
	    AFInetDestDriver *d = afinet_dd_new_tcp6($1, configuration);

	    afinet_grammar_set_dest_driver(d);
	  }
	  dest_afinet_tcp_options		{ $$ = last_driver; free($1); }
	;

dest_afinet_tcp_options
	: dest_afinet_tcp_options dest_afinet_tcp_option
	|
	;

dest_afinet_tcp_option
	: dest_afinet_option
	| KW_TLS
	  {
            gchar buf[256];
	    last_tls_context = tls_context_new(TM_CLIENT, cfg_lexer_format_location(lexer, &@1, buf, sizeof(buf)));
	  }
	  '(' dest_tls_options ')'
	  {
	    afinet_dd_set_tls_context(last_driver, last_tls_context);
          }
	;

dest_afsocket_option
        : KW_KEEP_ALIVE '(' yesno ')'        { afsocket_dd_set_keep_alive(last_driver, $3); }
        | KW_CLOSE_ON_INPUT '(' yesno ')'
          {
            afsocket_dd_set_close_on_input(last_driver, $3);
            log_proto_client_options_set_drop_input(last_proto_client_options, !$3);
          }
        ;


dest_afsyslog
        : KW_SYSLOG '(' dest_afsyslog_params ')'   { $$ = $3; }

dest_afsyslog_params
        : string
          {
            AFInetDestDriver *d = afinet_dd_new_syslog($1, configuration);

            afinet_grammar_set_dest_driver(d);
	  }
	  dest_afsyslog_options			{ $$ = last_driver; free($1); }
        ;


dest_afsyslog_options
	: dest_afsyslog_options dest_afsyslog_option
	|
	;

dest_afsyslog_option
	: dest_afinet_option
	| dest_afsocket_transport
	;

dest_afnetwork
	: KW_NETWORK '(' dest_afnetwork_params ')'	{ $$ = $3; }
	;

dest_afnetwork_params
        : string
          {
            AFInetDestDriver *d = afinet_dd_new_network($1, configuration);

            afinet_grammar_set_dest_driver(d);
	  }
	  dest_afnetwork_options			{ $$ = last_driver; free($1); }
        ;

dest_afnetwork_options
	: dest_afnetwork_options dest_afnetwork_option
	|
	;

dest_afnetwork_option
	: dest_afinet_option
	| dest_afsocket_transport
	;

dest_afsocket_transport
	: afsocket_transport
        | KW_TRANSPORT '(' string
          {
            Plugin *p;
            gint context = LL_CONTEXT_CLIENT_PROTO;

            p = cfg_find_plugin(configuration, context, $3);
            if (p)
              {
                /* for transports with grammar */
                if (p->parser)
                  {
                    LogProtoClientFactory *cf = cfg_parse_plugin(configuration, p, &@3, last_proto_client_options);
                    ((AFSocketDestDriver *) last_driver)->proto_factory = cf;
                  }
              }
            transport_mapper_set_transport(last_transport_mapper, $3);
          }
        ')'	{ free($3); }
	| dest_afinet_dgram_option
	| KW_TLS
	  {
            gchar buf[256];

            last_tls_context = tls_context_new(TM_CLIENT, cfg_lexer_format_location(lexer, &@1, buf, sizeof(buf)));
          }
          '(' dest_tls_options ')'
          {
            afinet_dd_set_tls_context(last_driver, last_tls_context);
          }
        ;

afsocket_transport
        : KW_TRANSPORT '(' KW_TCP ')'                    { transport_mapper_set_transport(last_transport_mapper, "tcp"); }
        | KW_TRANSPORT '(' KW_UDP ')'                    { transport_mapper_set_transport(last_transport_mapper, "udp"); }
        | KW_TRANSPORT '(' KW_TLS ')'                    { transport_mapper_set_transport(last_transport_mapper, "tls"); }
        | KW_IP_PROTOCOL '(' inet_ip_protocol_option ')' { transport_mapper_set_address_family(last_transport_mapper, $3); }
        ;

dest_tls_options
        : dest_tls_option dest_tls_options
        |
        ;

dest_tls_option
        : tls_option
        | KW_SNI '(' yesno ')'
          {
            if ($3)
              tls_context_set_sni(last_tls_context, ((AFInetDestDriver *)last_driver)->primary);
          }
        ;

tls_options
	: tls_option tls_options
	|
	;

tls_option
        : KW_IFDEF {
}

	| KW_PEER_VERIFY '(' yesno ')'
	  {
	    gint verify_mode = $3 ? (TVM_REQUIRED | TVM_TRUSTED) : (TVM_OPTIONAL | TVM_UNTRUSTED);
	    tls_context_set_verify_mode(last_tls_context, verify_mode);
          }
	| KW_PEER_VERIFY '(' string ')'
	  {
	    CHECK_ERROR(tls_context_set_verify_mode_by_name(last_tls_context, $3), @3,
	                "unknown peer-verify() argument");
            free($3);
          }
	| KW_KEY_FILE '(' path_secret ')'
	  {
	    tls_context_set_key_file(last_tls_context, $3);
            free($3);
          }
	| KW_KEYLOG_FILE '(' string ')'
	  {
	    GError *error = NULL;
	    CHECK_ERROR_GERROR(tls_context_set_keylog_file(last_tls_context, $3, &error), @3, error, "Error setting keylog-file()");
	    free($3);
	  }
	| KW_CERT_FILE '(' path_check ')'
	  {
	    tls_context_set_cert_file(last_tls_context, $3);
            free($3);
          }
        | KW_DHPARAM_FILE '(' path_check ')'
          {
            tls_context_set_dhparam_file(last_tls_context, $3);
            free($3);
          }
        | KW_PKCS12_FILE '(' path_check ')'
          {
            tls_context_set_pkcs12_file(last_tls_context, $3);
            free($3);
          }
	| KW_CA_DIR '(' string ')'
	  {
	    tls_context_set_ca_dir(last_tls_context, $3);
            free($3);
          }
	| KW_CRL_DIR '(' string ')'
	  {
	    tls_context_set_crl_dir(last_tls_context, $3);
            free($3);
          }
        | KW_CA_FILE '(' path_check ')'
          {
            tls_context_set_ca_file(last_tls_context, $3);
            free($3);
          }
        | KW_TRUSTED_KEYS '(' string_list ')'
          {
            tls_session_set_trusted_fingerprints(last_tls_context, $3);
          }
        | KW_TRUSTED_DN '(' string_list ')'
          {
            tls_session_set_trusted_dn(last_tls_context, $3);
          }
        | KW_CIPHER_SUITE '(' tls_cipher_suites ')'
        | KW_CIPHER_SUITE '(' string ')'
          {
            /* compat for specifying TLS 1.2-or-older ciphers */
            tls_context_set_cipher_suite(last_tls_context, $3);
            free($3);
          }
        | KW_ECDH_CURVE_LIST '(' string ')'
          {
            tls_context_set_ecdh_curve_list(last_tls_context, $3);
            free($3);
          }
	| KW_SSL_OPTIONS '(' string_list ')'
	  {
            CHECK_ERROR(tls_context_set_ssl_options_by_name(last_tls_context, $3), @3,
                        "unknown ssl-options() argument");
	  }
        | KW_ALLOW_COMPRESS '(' yesno ')'
          {
             transport_mapper_inet_set_allow_compress(last_transport_mapper, $3);
          }
        | KW_ENDIF {
}
        ;

tls_cipher_suites
        : tls_cipher_suite tls_cipher_suites
        |
        ;

tls_cipher_suite
        : KW_TLS12_AND_OLDER '(' string ')'
          {
            tls_context_set_cipher_suite(last_tls_context, $3);
            free($3);
          }
        | KW_TLS13 '(' string ')'
          {
            GError *error = NULL;
            CHECK_ERROR_GERROR(tls_context_set_tls13_cipher_suite(last_tls_context, $3, &error), @3, error, "Error setting cipher-suite(tls13())");
            free($3);
          }
        ;

socket_option
	: KW_SO_SNDBUF '(' nonnegative_integer ')'
	{
		CHECK_ERROR($3 <= G_MAXINT, @3, "Invalid so_sndbuf, it has to be less than %d", G_MAXINT);
		last_sock_options->so_sndbuf = $3;
	}
	| KW_SO_RCVBUF '(' nonnegative_integer ')'
	{
		CHECK_ERROR($3 <= G_MAXINT, @3, "Invalid so_rcvbuf, it has to be less than %d", G_MAXINT);
		last_sock_options->so_rcvbuf = $3;
	}
	| KW_SO_BROADCAST '(' yesno ')'             { last_sock_options->so_broadcast = $3; }
	| KW_SO_KEEPALIVE '(' yesno ')'             { last_sock_options->so_keepalive = $3; }
	| KW_SO_REUSEPORT '(' yesno ')'             { last_sock_options->so_reuseport = $3; }
	;

unix_socket_option
        : socket_option
        | KW_SO_PASSCRED '(' yesno ')'              { socket_options_unix_set_so_passcred(last_sock_options, $3); }
        ;

inet_socket_option
	: socket_option
	| KW_IP_TTL '(' nonnegative_integer ')'               { ((SocketOptionsInet *) last_sock_options)->ip_ttl = $3; }
	| KW_IP_TOS '(' nonnegative_integer ')'               { ((SocketOptionsInet *) last_sock_options)->ip_tos = $3; }
	| KW_IP_FREEBIND '(' yesno ')'              { ((SocketOptionsInet *) last_sock_options)->ip_freebind = $3; }
	| KW_INTERFACE '(' string ')'		    { socket_options_inet_set_interface_name((SocketOptionsInet *) last_sock_options, $3); free($3); }
	| KW_TCP_KEEPALIVE_TIME '(' nonnegative_integer ')'
	  {
	    CHECK_ERROR(socket_options_inet_set_tcp_keepalive_time((SocketOptionsInet *) last_sock_options, $3), @1,
		            "The tcp-keepalive-time() option is not supported on this platform");
	  }
	| KW_TCP_KEEPALIVE_INTVL '(' nonnegative_integer ')'
	  {
	    CHECK_ERROR(socket_options_inet_set_tcp_keepalive_intvl((SocketOptionsInet *) last_sock_options, $3), @1,
		            "The tcp-keepalive-intvl() option is not supported on this platform");
	  }
	| KW_TCP_KEEPALIVE_PROBES '(' nonnegative_integer ')'
	  {
	    CHECK_ERROR(socket_options_inet_set_tcp_keepalive_probes((SocketOptionsInet *) last_sock_options, $3), @1,
		            "The tcp-keepalive-probes() option is not supported on this platform");
	  }
	;

inet_ip_protocol_option
        : LL_NUMBER
        {
          CHECK_ERROR($1 == 4 || $1 == 6, @1, "ip-protocol option can only be 4 or 6!");
          if ($1 == 4)
            {
              $$ = AF_INET;
            }
          else
            {
              $$ = AF_INET6;
            }
        }
        ;



source_content
        :
          { cfg_lexer_push_context(lexer, LL_CONTEXT_SOURCE, NULL, "source statement"); }
          source_items
          { cfg_lexer_pop_context(lexer); }
          {
            $$ = log_expr_node_new_junction($2, &@$);
          }
        ;

source_items
        : source_item semicolons source_items	{ $$ = log_expr_node_append_tail(log_expr_node_new_pipe($1, &@1), $3); }
        | log_fork semicolons source_items      { $$ = log_expr_node_append_tail($1,  $3); }
	|					{ $$ = NULL; }
	;

source_item
	: source_afinter			{ $$ = $1; }
        | source_plugin                         { $$ = $1; }
	;

source_plugin
        : LL_IDENTIFIER
          {
            Plugin *p;
            gint context = LL_CONTEXT_SOURCE;

            p = cfg_find_plugin(configuration, context, $1);
            CHECK_ERROR(p, @1, "%s plugin %s not found", cfg_lexer_lookup_context_name_by_type(context), $1);

            last_driver = (LogDriver *) cfg_parse_plugin(configuration, p, &@1, NULL);
            free($1);
            if (!last_driver)
              {
                YYERROR;
              }
            $$ = last_driver;
          }
        ;

source_afinter
	: KW_INTERNAL '(' source_afinter_params ')'			{ $$ = $3; }
	;

source_afinter_params
        : {
            last_driver = afinter_sd_new(configuration);
            last_source_options = &((AFInterSourceDriver *) last_driver)->source_options.super;
          }
          source_afinter_options { $$ = last_driver; }
        ;

source_afinter_options
        : source_afinter_option source_afinter_options
        |
        ;

source_afinter_option
        : KW_LOG_FIFO_SIZE '(' positive_integer ')'	{ ((AFInterSourceOptions *) last_source_options)->queue_capacity = $3; }
        | source_option
        ;


filter_content
        : {
            FilterExprNode *filter_expr = NULL;

	    CHECK_ERROR_WITHOUT_MESSAGE(cfg_parser_parse(&filter_expr_parser, lexer, (gpointer *) &filter_expr, NULL), @$);

            $$ = log_expr_node_new_pipe(log_filter_pipe_new(filter_expr, configuration), &@$);
	  }
	;

parser_content
        :
          {
            LogExprNode *last_parser_expr = NULL;

            CHECK_ERROR_WITHOUT_MESSAGE(cfg_parser_parse(&parser_expr_parser, lexer, (gpointer *) &last_parser_expr, NULL), @$);
            $$ = last_parser_expr;
          }
        ;

rewrite_content
        :
          {
            LogExprNode *last_rewrite_expr = NULL;

            CHECK_ERROR_WITHOUT_MESSAGE(cfg_parser_parse(&rewrite_expr_parser, lexer, (gpointer *) &last_rewrite_expr, NULL), @$);
            $$ = last_rewrite_expr;
          }
        ;

dest_content
         : { cfg_lexer_push_context(lexer, LL_CONTEXT_DESTINATION, NULL, "destination statement"); }
            dest_items
           { cfg_lexer_pop_context(lexer); }
           {
             $$ = log_expr_node_new_junction($2, &@$);
           }
         ;


dest_items
        /* all destination drivers are added as an independent branch in a junction*/
        : dest_item semicolons dest_items	{ $$ = log_expr_node_append_tail(log_expr_node_new_pipe($1, &@1), $3); }
        | log_fork semicolons dest_items        { $$ = log_expr_node_append_tail($1,  $3); }
	|					{ $$ = NULL; }
	;

dest_item
        : dest_plugin                           { $$ = $1; }
	;

dest_plugin
        : LL_IDENTIFIER
          {
            Plugin *p;
            gint context = LL_CONTEXT_DESTINATION;

            p = cfg_find_plugin(configuration, context, $1);
            CHECK_ERROR(p, @1, "%s plugin %s not found", cfg_lexer_lookup_context_name_by_type(context), $1);

            last_driver = (LogDriver *) cfg_parse_plugin(configuration, p, &@1, NULL);
            free($1);
            if (!last_driver)
              {
                YYERROR;
              }
            $$ = last_driver;
          }
        ;

log_items
	: log_item semicolons log_items		{ log_expr_node_append_tail($1, $3); $$ = $1; }
	|					{ $$ = NULL; }
	;

log_item
        : KW_SOURCE '(' string ')'		{ $$ = log_expr_node_new_source_reference($3, &@$); free($3); }
        | KW_SOURCE '{' source_content '}'      { $$ = log_expr_node_new_source(NULL, $3, &@$); }
        | KW_FILTER '(' string ')'		{ $$ = log_expr_node_new_filter_reference($3, &@$); free($3); }
        | KW_FILTER '{' filter_content '}'      { $$ = log_expr_node_new_filter(NULL, $3, &@$); }
        | KW_PARSER '(' string ')'              { $$ = log_expr_node_new_parser_reference($3, &@$); free($3); }
        | KW_PARSER '{' parser_content '}'      { $$ = log_expr_node_new_parser(NULL, $3, &@$); }
        | KW_REWRITE '(' string ')'             { $$ = log_expr_node_new_rewrite_reference($3, &@$); free($3); }
        | KW_REWRITE '{' rewrite_content '}'    { $$ = log_expr_node_new_rewrite(NULL, $3, &@$); }
        | KW_DESTINATION '(' string ')'		{ $$ = log_expr_node_new_destination_reference($3, &@$); free($3); }
        | KW_DESTINATION '{' dest_content '}'   { $$ = log_expr_node_new_destination(NULL, $3, &@$); }
        | log_conditional			{ $$ = $1; }
        | log_junction                          { $$ = $1; }
	;

log_junction
        : KW_JUNCTION '{' log_forks '}'         { $$ = log_expr_node_new_junction($3, &@$); }
        ;

log_last_junction

        /* this rule matches the last set of embedded log {}
         * statements at the end of the log {} block.
         * It is the final junction and was the only form of creating
         * a processing tree before syslog-ng 3.4.
         *
         * We emulate if the user was writing junction {} explicitly.
         */
        : log_forks                             { $$ = $1 ? log_expr_node_new_junction($1, &@1) :  NULL; }
        ;


log_forks
        : log_fork semicolons log_forks		{ log_expr_node_append_tail($1, $3); $$ = $1; }
        |                                       { $$ = NULL; }
        ;

log_fork
        : KW_LOG '{' log_content '}'            { $$ = $3; }
        | KW_CHANNEL '{' log_content '}'        { $$ = $3; }
        ;

log_conditional
        : log_if				{ $$ = $1; }
        | log_if KW_ELSE '{' log_content '}'
          {
            log_expr_node_conditional_set_false_branch_of_the_last_if($1, $4);
            $$ = $1;
          }
        ;

log_if
        : KW_IF '(' filter_content ')' '{' log_content '}'
          {
            $$ = log_expr_node_new_conditional_with_filter($3, $6, &@$);
          }
        | KW_IF '{' log_content '}'
          {
            $$ = log_expr_node_new_conditional_with_block($3, &@$);
          }
        | log_if KW_ELIF '(' filter_content ')' '{' log_content '}'
          {
            LogExprNode *false_branch;

            false_branch = log_expr_node_new_conditional_with_filter($4, $7, &@$);
            log_expr_node_conditional_set_false_branch_of_the_last_if($1, false_branch);
            $$ = $1;
          }
        | log_if KW_ELIF '{' log_content '}'
          {
            LogExprNode *false_branch;

            false_branch = log_expr_node_new_conditional_with_block($4, &@$);
            log_expr_node_conditional_set_false_branch_of_the_last_if($1, false_branch);
            $$ = $1;
          }
        ;

log_content
        : log_items log_last_junction log_flags                { $$ = log_expr_node_new_log(log_expr_node_append_tail($1, $2), $3, &@$); }
        ;

log_flags
	: KW_FLAGS '(' log_flags_items ')' semicolons	{ $$ = $3; }
	|					{ $$ = 0; }
	;

log_flags_items
	: normalized_flag log_flags_items	{ $$ = log_expr_node_lookup_flag($1) | $2; free($1); }
	|					{ $$ = 0; }
	;


template_content_inner
        : string
        {
          GError *error = NULL;

          CHECK_ERROR_GERROR(log_template_compile(last_template, $1, &error), @1, error, "Error compiling template");
          free($1);
        }
        | LL_IDENTIFIER '(' string ')'
        {
          GError *error = NULL;

          CHECK_ERROR_GERROR(log_template_compile(last_template, $3, &error), @3, error, "Error compiling template");
          free($3);

          CHECK_ERROR_GERROR(log_template_set_type_hint(last_template, $1, &error), @1, error, "Error setting the template type-hint \"%s\"", $1);
          free($1);
        }
        ;

template_content
        : { last_template = log_template_new(configuration, NULL); } template_content_inner	{ $$ = last_template; }
        ;

template_content_list
	: template_content template_content_list { $$ = g_list_prepend($2, $1); }
	| { $$ = NULL; }
	;



string
	: LL_IDENTIFIER
	| LL_STRING
	;

yesno
	: KW_YES				{ $$ = 1; }
	| KW_NO					{ $$ = 0; }
	| LL_NUMBER				{ $$ = $1; }
	;

dnsmode
	: yesno					{ $$ = $1; }
	| KW_PERSIST_ONLY                       { $$ = 2; }
	;

nonnegative_integer64
        : LL_NUMBER
          {
            CHECK_ERROR(($1 >= 0), @1, "It cannot be negative");
          }
        ;

nonnegative_integer
        : nonnegative_integer64
          {
            CHECK_ERROR(($1 <= G_MAXINT32), @1, "Must be smaller than 2^31");
          }
        ;

positive_integer64
        : LL_NUMBER
          {
            CHECK_ERROR(($1 > 0), @1, "Must be positive");
          }
        ;

positive_integer
        : positive_integer64
          {
            CHECK_ERROR(($1 <= G_MAXINT32), @1, "Must be smaller than 2^31");
          }
        ;

nonnegative_float
        : LL_FLOAT
          {
            CHECK_ERROR(($1 >= 0), @1, "It cannot be negative");
          }
        | nonnegative_integer
          {
            $$ = (double) $1;
          }
        ;

positive_float
        : LL_FLOAT
          {
            CHECK_ERROR(($1 > 0), @1, "Must be positive");
          }
        | positive_integer
          {
            $$ = (double) $1;
          }
        ;

string_or_number
        : string                                { $$ = $1; }
        | LL_NUMBER                             { $$ = strdup(lexer->token_text->str); }
        | LL_FLOAT                              { $$ = strdup(lexer->token_text->str); }
        ;

path
	: string
	  {
            struct stat buffer;
            int ret = stat($1, &buffer);
            CHECK_ERROR((ret == 0), @1, "File \"%s\" not found: %s", $1, strerror(errno));
            $$ = $1;
	  }
	;

path_check
    : path { cfg_path_track_file(configuration, $1, "path_check"); }
    ;

path_secret
    : path { cfg_path_track_file(configuration, $1, "path_secret"); }
    ;

path_no_check
    : string { cfg_path_track_file(configuration, $1, "path_no_check"); }
    ;

normalized_flag
        : string                                { $$ = normalize_flag($1); free($1); }
        ;

string_list
        : string_list_build                     { $$ = $1; }
        ;

string_list_build
        : string string_list_build		{ $$ = g_list_prepend($2, g_strdup($1)); free($1); }
        |					{ $$ = NULL; }
        ;

semicolons
        : ';'
        | ';' semicolons
        ;

severity_string
        : string
	  {
	    /* return the numeric value of the "level" */
	    int n = syslog_name_lookup_severity_by_name($1);
	    CHECK_ERROR((n != -1), @1, "Unknown priority level\"%s\"", $1);
	    free($1);
            $$ = n;
	  }
        ;

facility_string
        : string
          {
            /* return the numeric value of facility */
	    int n = syslog_name_lookup_facility_by_name($1);
	    CHECK_ERROR((n != -1), @1, "Unknown facility \"%s\"", $1);
	    free($1);
	    $$ = n;
	  }
        | KW_SYSLOG 				{ $$ = LOG_SYSLOG; }
        ;

parser_opt
        : KW_TEMPLATE '(' string ')'            {
                                                  LogTemplate *template;
                                                  GError *error = NULL;

                                                  template = cfg_tree_check_inline_template(&configuration->tree, $3, &error);
                                                  CHECK_ERROR_GERROR(template != NULL, @3, error, "Error compiling template");
                                                  log_parser_set_template(last_parser, template);
                                                  free($3);
                                                }
        ;

driver_option
        : KW_PERSIST_NAME '(' string ')' { log_pipe_set_persist_name(&last_driver->super, $3); free($3); }
        ;

inner_source
        : LL_IDENTIFIER
          {
            Plugin *p;
            gint context = LL_CONTEXT_INNER_SRC;
            gpointer value;

            p = cfg_find_plugin(configuration, context, $1);
            CHECK_ERROR(p, @1, "%s plugin %s not found", cfg_lexer_lookup_context_name_by_type(context), $1);

            value = cfg_parse_plugin(configuration, p, &@1, last_driver);

            free($1);
            if (!value)
              {
                YYERROR;
              }
            if (!log_driver_add_plugin(last_driver, (LogDriverPlugin *) value))
              {
                log_driver_plugin_free(value);
                CHECK_ERROR(TRUE, @1, "Error while registering the plugin %s in this destination", $1);
              }
          }
        ;

/* All source drivers should incorporate this rule, implies driver_option */
source_driver_option
        : inner_source
        | driver_option
        ;

inner_dest
        : LL_IDENTIFIER
          {
            Plugin *p;
            gint context = LL_CONTEXT_INNER_DEST;
            gpointer value;

            p = cfg_find_plugin(configuration, context, $1);
            CHECK_ERROR(p, @1, "%s plugin %s not found", cfg_lexer_lookup_context_name_by_type(context), $1);

            value = cfg_parse_plugin(configuration, p, &@1, last_driver);

            free($1);
            if (!value)
              {
                YYERROR;
              }
            if (!log_driver_add_plugin(last_driver, (LogDriverPlugin *) value))
              {
                log_driver_plugin_free(value);
                CHECK_ERROR(TRUE, @1, "Error while registering the plugin %s in this destination", $1);
              }
          }
        ;

/* implies driver_option */
dest_driver_option
        /* NOTE: plugins need to set "last_driver" in order to incorporate this rule in their grammar */

	: KW_LOG_FIFO_SIZE '(' positive_integer ')'	{ ((LogDestDriver *) last_driver)->log_fifo_size = $3; }
	| KW_THROTTLE '(' nonnegative_integer ')'         { ((LogDestDriver *) last_driver)->throttle = $3; }
        | inner_dest
        | driver_option
        ;

threaded_dest_driver_batch_option
        : KW_BATCH_LINES '(' nonnegative_integer ')' { log_threaded_dest_driver_set_batch_lines(last_driver, $3); }
        | KW_BATCH_TIMEOUT '(' positive_integer ')' { log_threaded_dest_driver_set_batch_timeout(last_driver, $3); }
        ;

threaded_dest_driver_workers_option
        : KW_WORKERS '(' positive_integer ')'  { log_threaded_dest_driver_set_num_workers(last_driver, $3); }
        ;

/* implies dest_driver_option */
threaded_dest_driver_general_option
	: KW_RETRIES '(' positive_integer ')'
        {
          log_threaded_dest_driver_set_max_retries_on_error(last_driver, $3);
        }
        | KW_TIME_REOPEN '(' positive_integer ')' { log_threaded_dest_driver_set_time_reopen(last_driver, $3); }
        | dest_driver_option
        ;

/* implies source_driver_option and source_option */
threaded_source_driver_option
	: KW_FORMAT '(' string ')' { log_threaded_source_driver_get_parse_options(last_driver)->format = g_strdup($3); free($3); }
        | KW_FLAGS '(' threaded_source_driver_option_flags ')'
        | { last_msg_format_options = log_threaded_source_driver_get_parse_options(last_driver); } msg_format_option
        | { last_source_options = log_threaded_source_driver_get_source_options(last_driver); } source_option
        | source_driver_option
        ;

threaded_fetcher_driver_option
        : KW_FETCH_NO_DATA_DELAY '(' nonnegative_float ')' { log_threaded_fetcher_driver_set_fetch_no_data_delay(last_driver, $3); }
        | KW_TIME_REOPEN '(' positive_integer ')' { log_threaded_fetcher_driver_set_time_reopen(last_driver, $3); }
        ;

threaded_source_driver_option_flags
	: string threaded_source_driver_option_flags
        {
          CHECK_ERROR(msg_format_options_process_flag(log_threaded_source_driver_get_parse_options(last_driver), $1), @1, "Unknown flag \"%s\"", $1);
          free($1);
        }
        |
        ;

/* LogSource related options */
source_option
        /* NOTE: plugins need to set "last_source_options" in order to incorporate this rule in their grammar */
	: KW_LOG_IW_SIZE '(' positive_integer ')'	{ last_source_options->init_window_size = $3; }
	| KW_CHAIN_HOSTNAMES '(' yesno ')'	{ last_source_options->chain_hostnames = $3; }
	| KW_KEEP_HOSTNAME '(' yesno ')'	{ last_source_options->keep_hostname = $3; }
	| KW_PROGRAM_OVERRIDE '(' string ')'	{ last_source_options->program_override = g_strdup($3); free($3); }
	| KW_HOST_OVERRIDE '(' string ')'	{ last_source_options->host_override = g_strdup($3); free($3); }
	| KW_LOG_PREFIX '(' string ')'	        { gchar *p = strrchr($3, ':'); if (p) *p = 0; last_source_options->program_override = g_strdup($3); free($3); }
	| KW_KEEP_TIMESTAMP '(' yesno ')'	{ last_source_options->keep_timestamp = $3; }
	| KW_READ_OLD_RECORDS '(' yesno ')'	{ last_source_options->read_old_records = $3; }
	| KW_USE_SYSLOGNG_PID '(' yesno ')'	{ last_source_options->use_syslogng_pid = $3; }
        | KW_TAGS '(' string_list ')'		{ log_source_options_set_tags(last_source_options, $3); }
        | { last_host_resolve_options = &last_source_options->host_resolve_options; } host_resolve_option
        ;

/* LogReader related options, implies source_option */
source_reader_option
        /* NOTE: plugins need to set "last_reader_options" in order to incorporate this rule in their grammar */

	: KW_CHECK_HOSTNAME '(' yesno ')'	{ last_reader_options->check_hostname = $3; }
	| KW_FLAGS '(' source_reader_option_flags ')'
	| KW_LOG_FETCH_LIMIT '(' positive_integer ')'	{ last_reader_options->fetch_limit = $3; }
        | KW_FORMAT '(' string ')'              { last_reader_options->parse_options.format = g_strdup($3); free($3); }
        | { last_source_options = &last_reader_options->super; } source_option
        | { last_proto_server_options = &last_reader_options->proto_options.super; } source_proto_option
        | { last_msg_format_options = &last_reader_options->parse_options; } msg_format_option
	;

source_reader_option_flags
        : string source_reader_option_flags     { CHECK_ERROR(log_reader_options_process_flag(last_reader_options, $1), @1, "Unknown flag \"%s\"", $1); free($1); }
        | KW_CHECK_HOSTNAME source_reader_option_flags     { log_reader_options_process_flag(last_reader_options, "check-hostname"); }
	|
	;

/* LogProtoSource related options */
source_proto_option
        : KW_ENCODING '(' string ')'
          {
            CHECK_ERROR(log_proto_server_options_set_encoding(last_proto_server_options, $3),
                        @3,
                        "unknown encoding \"%s\"", $3);
            free($3);
          }
        | KW_LOG_MSG_SIZE '(' positive_integer ')'      { last_proto_server_options->max_msg_size = $3; }
        | KW_TRIM_LARGE_MESSAGES '(' yesno ')'          { last_proto_server_options->trim_large_messages = $3; }
        ;

host_resolve_option
        : KW_USE_FQDN '(' yesno ')'             { last_host_resolve_options->use_fqdn = $3; }
        | KW_USE_DNS '(' dnsmode ')'            { last_host_resolve_options->use_dns = $3; }
	| KW_DNS_CACHE '(' yesno ')' 		{ last_host_resolve_options->use_dns_cache = $3; }
	| KW_NORMALIZE_HOSTNAMES '(' yesno ')'	{ last_host_resolve_options->normalize_hostnames = $3; }
	;

msg_format_option
	: KW_TIME_ZONE '(' string ')'		{ last_msg_format_options->recv_time_zone = g_strdup($3); free($3); }
	| KW_DEFAULT_SEVERITY '(' severity_string ')'
	  {
	    if (last_msg_format_options->default_pri == 0xFFFF)
	      last_msg_format_options->default_pri = LOG_USER;
	    last_msg_format_options->default_pri = (last_msg_format_options->default_pri & ~LOG_PRIMASK) | $3;
          }
	| KW_DEFAULT_FACILITY '(' facility_string ')'
	  {
	    if (last_msg_format_options->default_pri == 0xFFFF)
	      last_msg_format_options->default_pri = LOG_NOTICE;
	    last_msg_format_options->default_pri = (last_msg_format_options->default_pri & LOG_PRIMASK) | $3;
          }
        ;

dest_writer_options
	: dest_writer_option dest_writer_options
	|
	;

dest_writer_option
        /* NOTE: plugins need to set "last_writer_options" in order to incorporate this rule in their grammar */

	: KW_FLAGS '(' dest_writer_options_flags ')' { last_writer_options->options = $3; }
	| KW_FLUSH_LINES '(' nonnegative_integer ')'		{ last_writer_options->flush_lines = $3; }
	| KW_FLUSH_TIMEOUT '(' positive_integer ')'	{ }
        | KW_SUPPRESS '(' nonnegative_integer ')'            { last_writer_options->suppress = $3; }
	| KW_TEMPLATE '(' string ')'       	{
                                                  GError *error = NULL;

                                                  last_writer_options->template = cfg_tree_check_inline_template(&configuration->tree, $3, &error);
                                                  CHECK_ERROR_GERROR(last_writer_options->template != NULL, @3, error, "Error compiling template");
	                                          free($3);
	                                        }
	| KW_TEMPLATE_ESCAPE '(' yesno ')'	{ log_writer_options_set_template_escape(last_writer_options, $3); }
	| KW_PAD_SIZE '(' nonnegative_integer ')'         { last_writer_options->padding = $3; }
	| KW_TRUNCATE_SIZE '(' nonnegative_integer ')'         { last_writer_options->truncate_size = $3; }
	| KW_MARK_FREQ '(' nonnegative_integer ')'        { last_writer_options->mark_freq = $3; }
        | KW_MARK_MODE '(' KW_INTERNAL ')'      { log_writer_options_set_mark_mode(last_writer_options, "internal"); }
	| KW_MARK_MODE '(' string ')'
	  {
	    CHECK_ERROR(cfg_lookup_mark_mode($3) != -1, @3, "illegal mark mode \"%s\"", $3);
            log_writer_options_set_mark_mode(last_writer_options, $3);
            free($3);
          }
	| KW_TIME_REOPEN '(' positive_integer ')' { last_writer_options->time_reopen = $3; }
        | { last_template_options = &last_writer_options->template_options; } template_option
	;

dest_writer_options_flags
	: normalized_flag dest_writer_options_flags   { $$ = log_writer_options_lookup_flag($1) | $2; free($1); }
	|					      { $$ = 0; }
	;

file_perm_option
	: KW_OWNER '(' string_or_number ')'	{ file_perm_options_set_file_uid(last_file_perm_options, $3); free($3); }
	| KW_OWNER '(' ')'	                { file_perm_options_dont_change_file_uid(last_file_perm_options); }
	| KW_GROUP '(' string_or_number ')'	{ file_perm_options_set_file_gid(last_file_perm_options, $3); free($3); }
	| KW_GROUP '(' ')'	                { file_perm_options_dont_change_file_gid(last_file_perm_options); }
	| KW_PERM '(' LL_NUMBER ')'		{ file_perm_options_set_file_perm(last_file_perm_options, $3); }
	| KW_PERM '(' ')'		        { file_perm_options_dont_change_file_perm(last_file_perm_options); }
        | KW_DIR_OWNER '(' string_or_number ')'	{ file_perm_options_set_dir_uid(last_file_perm_options, $3); free($3); }
	| KW_DIR_OWNER '(' ')'	                { file_perm_options_dont_change_dir_uid(last_file_perm_options); }
	| KW_DIR_GROUP '(' string_or_number ')'	{ file_perm_options_set_dir_gid(last_file_perm_options, $3); free($3); }
	| KW_DIR_GROUP '(' ')'	                { file_perm_options_dont_change_dir_gid(last_file_perm_options); }
	| KW_DIR_PERM '(' LL_NUMBER ')'		{ file_perm_options_set_dir_perm(last_file_perm_options, $3); }
	| KW_DIR_PERM '(' ')'		        { file_perm_options_dont_change_dir_perm(last_file_perm_options); }
        ;

template_option
	: KW_TS_FORMAT '(' string ')'		{ last_template_options->ts_format = cfg_ts_format_value($3); free($3); }
	| KW_FRAC_DIGITS '(' nonnegative_integer ')'	{ last_template_options->frac_digits = $3; }
	| KW_TIME_ZONE '(' string ')'		{ last_template_options->time_zone[LTZ_SEND] = g_strdup($3); free($3); }
	| KW_SEND_TIME_ZONE '(' string ')'      { last_template_options->time_zone[LTZ_SEND] = g_strdup($3); free($3); }
	| KW_LOCAL_TIME_ZONE '(' string ')'     { last_template_options->time_zone[LTZ_LOCAL] = g_strdup($3); free($3); }
	| KW_ON_ERROR '(' string ')'
        {
          gint on_error;

          CHECK_ERROR(log_template_on_error_parse($3, &on_error), @3, "Invalid on-error() setting \"%s\"", $3);
          free($3);

          log_template_options_set_on_error(last_template_options, on_error);
        }
	;

matcher_option
        : KW_TYPE '(' string ')'		{ CHECK_ERROR(log_matcher_options_set_type(last_matcher_options, $3), @3, "unknown matcher type \"%s\"", $3); free($3); }
        | KW_FLAGS '(' matcher_flags ')'
        ;

matcher_flags
        : string matcher_flags			{ CHECK_ERROR(log_matcher_options_process_flag(last_matcher_options, $1), @1, "unknown matcher flag \"%s\"", $1); free($1); }
        |
        ;

value_pair_option
	: KW_VALUE_PAIRS
          {
            last_value_pairs = value_pairs_new(configuration);
          }
          '(' vp_options ')'
          { $$ = last_value_pairs; }
	;

vp_options
	: vp_option vp_options
	|
	;

vp_option
        : KW_PAIR '(' string ':' template_content ')'
          {
            value_pairs_add_pair(last_value_pairs, $3, $5);
            free($3);
          }
        | KW_PAIR '(' string template_content ')'
          {
            value_pairs_add_pair(last_value_pairs, $3, $4);
            free($3);
          }
        | KW_KEY '(' string KW_REKEY '('
          {
            last_vp_transset = value_pairs_transform_set_new($3);
            value_pairs_add_glob_pattern(last_value_pairs, $3, TRUE);
            free($3);
          }
          vp_rekey_options ')'                           { value_pairs_add_transforms(last_value_pairs, last_vp_transset); } ')'
	| KW_KEY '(' string_list ')'		         { value_pairs_add_glob_patterns(last_value_pairs, $3, TRUE); }
        | KW_REKEY '(' string
          {
            last_vp_transset = value_pairs_transform_set_new($3);
            free($3);
          }
          vp_rekey_options ')'                           { value_pairs_add_transforms(last_value_pairs, last_vp_transset); }
        | KW_EXCLUDE '(' string_list ')'                 { value_pairs_add_glob_patterns(last_value_pairs, $3, FALSE); }
	| KW_SCOPE '(' vp_scope_list ')'
	| KW_CAST '(' yesno ')'                          { value_pairs_set_cast_to_strings(last_value_pairs, $3); }
	;

vp_scope_list
	: string vp_scope_list                           { value_pairs_add_scope(last_value_pairs, $1); free($1); }
	|
	;

vp_rekey_options
	: vp_rekey_option vp_rekey_options
        |
	;

vp_rekey_option
	: KW_SHIFT '(' positive_integer ')' { value_pairs_transform_set_add_func(last_vp_transset, value_pairs_new_transform_shift($3)); }
	| KW_SHIFT_LEVELS '(' positive_integer ')' { value_pairs_transform_set_add_func(last_vp_transset, value_pairs_new_transform_shift_levels($3)); }
	| KW_ADD_PREFIX '(' string ')' { value_pairs_transform_set_add_func(last_vp_transset, value_pairs_new_transform_add_prefix($3)); free($3); }
	| KW_REPLACE_PREFIX '(' string string ')' { value_pairs_transform_set_add_func(last_vp_transset, value_pairs_new_transform_replace_prefix($3, $4)); free($3); free($4); }
	;

rewrite_expr_opt
        : KW_VALUE '(' string ')'
          {
            const gchar *p = $3;
            if (p[0] == '$')
              {
                msg_warning("Value references in rewrite rules should not use the '$' prefix, those are only needed in templates",
                            evt_tag_str("value", $3),
                            cfg_lexer_format_location_tag(lexer, &@3));
                p++;
              }
            last_rewrite->value_handle = log_msg_get_value_handle(p);
            CHECK_ERROR(!log_msg_is_handle_macro(last_rewrite->value_handle), @3, "%s is read-only, it cannot be changed in rewrite rules", p);
	    CHECK_ERROR(log_msg_is_value_name_valid(p), @3, "%s is not a valid name for a name-value pair, perhaps a misspelled .SDATA reference?", p);
            free($3);
          }
        | rewrite_condition_opt
        ;

rewrite_condition_opt
        : KW_CONDITION '('
          {
            FilterExprNode *filter_expr;

            CHECK_ERROR_WITHOUT_MESSAGE(cfg_parser_parse(&filter_expr_parser, lexer, (gpointer *) &filter_expr, NULL), @1);
            log_rewrite_set_condition(last_rewrite, filter_expr);
          } ')'
        ;



%%
