 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/times.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#if defined(CONFIG_BCM_KF_IGMP) && defined(CONFIG_BR_IGMP_SNOOP)
#include <br_igmp.h>
#endif
#include "flowmgr.h"

/* 1: Enable for wifi interfaces */
/* 2: Enable for all  interfaces */
int flowmgr_mc2uc = 1;

module_param_named(mc2uc_enable, flowmgr_mc2uc, int, 0644);
MODULE_PARM_DESC(mc2uc_enable, "Multicast-2-Unicast Enable/Disable");

struct delayed_work flowmgr_mcast_work;

int flowmgr_mcast_counter_start(void)
{
	unsigned long sec = msecs_to_jiffies(10 * 1000);
	if (sec)
	    schedule_delayed_work(&flowmgr_mcast_work, sec);
	return 0;
}

int flowmgr_mcast_counter_stop(void)
{
	cancel_delayed_work(&flowmgr_mcast_work);
	flush_delayed_work(&flowmgr_mcast_work);
	return 0;
}

static void flowmgr_mcast_work_handler(struct work_struct *work)
{
	flowmgr_mdb_update_all_mcast_stats();
	flowmgr_mcast_counter_start();
}

static int flowmgr_mcast_counter_init(void)
{
	INIT_DELAYED_WORK(&flowmgr_mcast_work, flowmgr_mcast_work_handler);
	return 0;
}

#ifdef CONFIG_BRIDGE_MCAST_OFFLOAD
#if defined(CONFIG_BCM_KF_IGMP) && defined(CONFIG_BR_IGMP_SNOOP)
int br_bcm_igmp_dbevent_add(struct net_bridge *br,
			    struct net_bridge_mc_fdb_entry *igmp_fdb,
			    struct net_bridge_mc_rep_entry *rep_entry)
{
	int ret;
	__be32 group[4];  /* For IPv4 use group[3] */
	struct net_device *from_dev;
	struct net_device *dst_dev;

	if (!igmp_fdb)
		return -1;

	from_dev = flowmgr_get_wandev();

	if (!from_dev)
		return -1;

	dst_dev = igmp_fdb->dst->dev;

	if (is_vlan_dev(dst_dev))
		dst_dev = vlan_dev_priv(dst_dev)->real_dev;

	pr_debug("MCAST: IGMP ADD %s %s %pI4 %pM\n",
		 from_dev->name,
		 dst_dev->name,
		 &igmp_fdb->grp.s_addr,
		 rep_entry->rep_mac.addr);

	memset(group, 0, sizeof(group));
	group[3] = igmp_fdb->grp.s_addr;

	if ((flowmgr_mc2uc == 2) ||
	    (flowmgr_mc2uc && strstr(dst_dev->name, "wl"))) {
		ret = flowmgr_mdb_insert(from_dev,
					 dst_dev,
					 group,
					 rep_entry->rep_mac.addr);
	} else {
		ret = flowmgr_mdb_insert(from_dev,
					 dst_dev,
					 group,
					 NULL);
	}
	return ret;
}

int br_bcm_igmp_dbevent_del(struct net_bridge *br,
			    struct net_bridge_mc_fdb_entry *igmp_fdb,
			    struct net_bridge_mc_rep_entry *rep_entry)
{
	int ret;
	__be32 group[4];  /* For IPv4 use group[3] */
	struct net_device *from_dev;
	struct net_device *dst_dev;

	if (!igmp_fdb)
		return -1;

	from_dev = flowmgr_get_wandev();

	if (!from_dev)
		return -1;

	dst_dev = igmp_fdb->dst->dev;

	if (is_vlan_dev(dst_dev))
		dst_dev = vlan_dev_priv(dst_dev)->real_dev;

	pr_debug("MCAST: IGMP DEL %s %s %pI4 %pM\n",
		 from_dev->name,
		 dst_dev->name,
		 &igmp_fdb->grp.s_addr,
		 rep_entry->rep_mac.addr);

	memset(group, 0, sizeof(group));
	group[3] = igmp_fdb->grp.s_addr;

	if ((flowmgr_mc2uc == 2) ||
	    (flowmgr_mc2uc && strstr(dst_dev->name, "wl"))) {
		ret = flowmgr_mdb_delete(from_dev,
					 dst_dev,
					 group,
					 rep_entry->rep_mac.addr);
	} else {
		ret = flowmgr_mdb_delete(from_dev,
					 dst_dev,
					 group,
					 NULL);
	}
	return ret;
}
#endif

int br_dbevent_cb(struct net_bridge *br, int dbtype, int event,
		  void *dbitem, void *dbitemext)
{
	if (!flowmgr.enable)
		return 0;

	if (!flowmgr_is_feature_enabled(FLOW_F_MCAST))
		return 0;

	pr_debug("MCAST: %s dbtype(%u) event(%u)\n",
		 br->dev->name, dbtype, event);

#if defined(CONFIG_BCM_KF_IGMP) && defined(CONFIG_BR_IGMP_SNOOP)
	if (dbtype == BR_MDB_BCM_IGMP) {
		if (event == BR_DBEVENT_ADD)
			br_bcm_igmp_dbevent_add(br, dbitem, dbitemext);
		else if (event == BR_DBEVENT_DEL)
			br_bcm_igmp_dbevent_del(br, dbitem, dbitemext);
	}
#endif
	return 0;
}

static struct br_dbevent_notifier notifier __read_mostly = {
	.fcn = br_dbevent_cb,
};
#endif

int flowmgr_mc2uc_enable(int enable)
{
	flowmgr_mc2uc = enable;
	return 0;
}
EXPORT_SYMBOL(flowmgr_mc2uc_enable);

int flowmgr_is_mc2uc(void)
{
	return flowmgr_mc2uc;
}
EXPORT_SYMBOL(flowmgr_is_mc2uc);

int flowmgr_mcast_init(void)
{
	flowmgr_mdb_init();
	flowmgr_mcast_counter_init();
#ifdef CONFIG_BRIDGE_MCAST_OFFLOAD
	br_register_dbevent_notifier(&notifier);
#endif
	return 0;
}

void flowmgr_mcast_fini(void)
{
	flowmgr_mcast_counter_stop();
	flowmgr_mdb_fini();
#ifdef CONFIG_BRIDGE_MCAST_OFFLOAD
	br_unregister_dbevent_notifier(&notifier);
#endif
}
