/****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2017 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Broadcom Power Domain RPC driver
 *
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 *		   Venky Selvaraj <venky.selvaraj@broadcom.com>
 *****************************************************************************/

#include <linux/bitops.h>
#include <linux/gpio/driver.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/of.h>
#include <linux/module.h>
#include <linux/irqdomain.h>
#include <linux/irqchip/chained_irq.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <proc_cmd.h>
#include <brcm_pwr_rpc.h>

#include "brcm_pwr_rpc_priv.h"

#ifdef pr_fmt
#undef pr_fmt
#endif
#define pr_fmt(fmt) MODULE_NAME ": " fmt

#define MODULE_NAME "brcm-pwr-rpc"
#define MODULE_VER	"1.0"

#define PWR_RPC_RG_SMC_TUNNEL_NAME "rg-smc"
#define BOOT_STATE_PROPERTY "boot-state"
#define INTR_NAME_PROPERTY "interrupt-names"
#define MAX_CB_NUM 256
#define ASYNC_CB_TIMEOUT 10 //10ms

#define PWR_RPC_TIMEOUT	 5

static int debug;
static pwr_domain domain_tbl[PWR_DOMAINS_MAX];
static struct brcm_pwr_rpc_dev_info *info;
static ATOMIC_NOTIFIER_HEAD(brcm_pwr_rpc_chain);
static int tunnel_id = -1;
static u32 async_count;
const char *boot_state_str = NULL;
static bool delay_batt_mode;
spinlock_t async_cb_lock;
struct async_cb_s
{
	pwr_rpc_cb_func cb_func;
	u32 tag;
	u32 dom_id;
	struct timer_list timer;
	bool cb_init;
};

struct async_cb_s async_cb[MAX_CB_NUM];

void clear_domain_tbl(void)
{
	int i;
	for (i = 0; i < PWR_DOMAINS_MAX; i++) {
		memset(domain_tbl[i].name, 0, PWR_DOMAIN_NAME_MAX_LEN);
		domain_tbl[i].id=-1;
		domain_tbl[i].state=-1;
		domain_tbl[i].reset=-1;
	}
}

static int pwr_get_domain_id(int tunnel, char *name, int *id)
{
	rpc_msg msg;
	int rc;

	rpc_msg_init(&msg, RPC_SERVICE_PWR, PWR_GET_DOMAIN_ID, 0,
		     0, 0, 0);
	strncpy((char *)&msg.data[1], name,
		PWR_DOMAIN_NAME_MAX_LEN-1);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_request_timeout(tunnel, &msg, PWR_RPC_TIMEOUT);
	if (rc) {
		pr_err(RED("%s : PWR_GET_DOMAIN_ID failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		goto done;
	}
	if (debug)
		rpc_dump_msg(&msg);
	rc = pwr_msg_get_retcode(&msg);
	if (rc) {
		pr_err(RED("brcm pwr rpc msg retcode %d\n"), (s8)rc);
		rpc_dump_msg(&msg);
		rc = -EIO;
		goto done;
	}
	*id = msg.data[0] & 0xFF;
	domain_tbl[*id].id = *id;
	strncpy(domain_tbl[*id].name, name,
		PWR_DOMAIN_NAME_MAX_LEN-1);

done:
	return rc;
}

static int pwr_get_domain_name(int tunnel, int id, char *name)
{
	rpc_msg msg;
	int rc;

	if (domain_tbl[id].id >= 0) {
		strncpy(name, domain_tbl[id].name,
			PWR_DOMAIN_NAME_MAX_LEN-1);
		return 0;
	}

	rpc_msg_init(&msg, RPC_SERVICE_PWR, PWR_GET_DOMAIN_NAME, 0,
		     id, 0, 0);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_request_timeout(tunnel, &msg, PWR_RPC_TIMEOUT);
	if (rc) {
		pr_err(RED("%s : PWR_GET_DOMAIN_NAME failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		goto done;
	}
	if (debug)
		rpc_dump_msg(&msg);
	rc = pwr_msg_get_retcode(&msg);
	if (rc) {
		pr_err(RED("brcm pwr rpc msg retcode %d\n"), (s8)rc);
		rpc_dump_msg(&msg);
		rc = -EIO;
		goto done;
	}
	strncpy(domain_tbl[id].name, (char *)&msg.data[1],
		PWR_DOMAIN_NAME_MAX_LEN-1);
	strncpy(name, (char *)&msg.data[1],
		PWR_DOMAIN_NAME_MAX_LEN-1);
	domain_tbl[id].id = id;

done:
	return rc;
}

static int pwr_get_domain_state(int tunnel, int id, int *state, int *reset)
{
	rpc_msg msg;
	int rc;

	rpc_msg_init(&msg, RPC_SERVICE_PWR, PWR_GET_DOMAIN_STATE, 0,
		     0, 0, 0);
	pwr_msg_set_domain_id(&msg, (u8)id);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_request_timeout(tunnel, &msg, PWR_RPC_TIMEOUT);
	if (rc) {
		pr_err(RED("%s : PWR_GET_DOMAIN_STATE failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		goto done;
	}
	if (debug)
		rpc_dump_msg(&msg);

	rc = pwr_msg_get_retcode(&msg);
	if (rc) {
		pr_err(RED("brcm pwr rpc msg retcode %d\n"), (s8)rc);
		rpc_dump_msg(&msg);
		rc = -EIO;
		goto done;
	}
	*state = pwr_msg_get_domain_state(&msg);
	*reset = pwr_msg_get_domain_reset(&msg);
	domain_tbl[id].state = *state;
	domain_tbl[id].reset = *reset;

done:
	return rc;
}

static int pwr_set_domain_state(int tunnel, int id, int state, int reset)
{
	rpc_msg msg;
	int rc;

	rpc_msg_init(&msg, RPC_SERVICE_PWR, PWR_SET_DOMAIN_STATE, 0,
		     0, 0, 0);
	pwr_msg_set_domain_reset(&msg, (u8)reset);
	pwr_msg_set_domain_state(&msg, (u8)state);
	pwr_msg_set_domain_id(&msg, (u8)id);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_request_timeout(tunnel, &msg, PWR_RPC_TIMEOUT);
	if (rc) {
		pr_err(RED("%s : PWR_SET_DOMAIN_STATE failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		goto done;
	}
	if (debug)
		rpc_dump_msg(&msg);
	rc = pwr_msg_get_retcode(&msg);
	if (rc) {
		pr_err(RED("brcm pwr rpc msg retcode %d\n"), (s8)rc);
		rpc_dump_msg(&msg);
		rc = -EIO;
		goto done;
	}
	domain_tbl[id].state = pwr_msg_get_domain_state(&msg);
	domain_tbl[id].reset = pwr_msg_get_domain_reset(&msg);

done:
	return rc;
}

static void brcm_pwr_rpc_aync_timer_cb(struct timer_list *tl)
{
	struct async_cb_s *async_cb = from_timer(async_cb, tl, timer);

	spin_lock(&async_cb_lock);
	if (async_cb->cb_init) {
		pr_info("ASYNC Calback Timeout for Tag(%u), so initialize callback function\n", async_cb->tag);
		async_cb->cb_func = NULL;
	}
	spin_unlock(&async_cb_lock);
}

static void brcm_pwr_rpc_add_async_timer_cb(struct async_cb_s *async_cb)
{
	timer_setup(&async_cb->timer, brcm_pwr_rpc_aync_timer_cb, 0);
	mod_timer(&async_cb->timer, jiffies + msecs_to_jiffies(ASYNC_CB_TIMEOUT));
}

static int pwr_get_domain_state_atomic(int tunnel, int id, pwr_rpc_cb_func cb_func)
{
	rpc_msg msg;
	int rc;
	u32 cnt;

	rpc_msg_init(&msg, RPC_SERVICE_PWR, PWR_ASYNC_GET_DOMAIN_STATE, 0,
		     0, 0, async_count);
	pwr_msg_set_domain_id(&msg, (u8)id);
	pr_debug("TAG-%x Domain state get Domain-ID-%d",
			 async_count, id);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_message(tunnel, &msg, false);
	if (rc) {
		pr_err(RED("%s : Req run state response failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		goto done;
	}

	cnt = (async_count%MAX_CB_NUM);
	if (async_cb[cnt].cb_init)
		pr_info("Previous Callback for ASYNC get message not processed\n");
	async_cb[cnt].dom_id = id;
	async_cb[cnt].tag = async_count;
	async_cb[cnt].cb_func = cb_func;
	async_cb[cnt].cb_init = true;
	if (cb_func)
		brcm_pwr_rpc_add_async_timer_cb(&async_cb[cnt]);

done:
	async_count++;
	return rc;
}

static int pwr_set_domain_state_atomic(int tunnel, int id, int state, int reset,
				       pwr_rpc_cb_func cb_func)
{
	rpc_msg msg;
	int rc;
	u32 cnt;

	rpc_msg_init(&msg, RPC_SERVICE_PWR, PWR_ASYNC_SET_DOMAIN_STATE, 0,
		     0, 0, async_count);
	pwr_msg_set_domain_reset(&msg, (u8)reset);
	pwr_msg_set_domain_state(&msg, (u8)state);
	pwr_msg_set_domain_id(&msg, (u8)id);
	pr_debug("TAG-%x Domain state set for Domain-ID-%d with State-%d and Reset-%d",
			 async_count, id, state, reset);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_message(tunnel, &msg, false);
	if (rc) {
		pr_err(RED("%s : Req run state response failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		goto done;
	}

	cnt = (async_count%MAX_CB_NUM);
	if (async_cb[cnt].cb_init)
		pr_info("Previous Callback for ASYNC set message not processed\n");
	async_cb[cnt].dom_id = id;
	async_cb[cnt].tag = async_count;
	async_cb[cnt].cb_func = cb_func;
	async_cb[cnt].cb_init = true;
	if (cb_func)
		brcm_pwr_rpc_add_async_timer_cb(&async_cb[cnt]);

done:
	async_count++;
	return rc;
}

static int brcm_pwr_rpc_get_fifo_tunnel_id(char *name)
{
	if (tunnel_id < 0) {
		tunnel_id = rpc_get_fifo_tunnel_id(name);
		if (tunnel_id < 0) {
			pr_err("%s : Error: invalid tunnel %s\n",
					MODULE_NAME, name);
		}
	}

	return tunnel_id;
}

int brcm_pwr_rpc_get_domain_state(char *name, int *state, int *reset)
{
	int tunnel = 0, id = 0;
	int rc = 0;

	if (!name) {
		pr_err("%s : Invalid argument name:%s\n",
			   MODULE_NAME, name);
		rc = -EIO;
		goto done;
	}

	tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(PWR_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, PWR_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto done;
	}

	if (pwr_get_domain_id(tunnel, name, &id))
		goto done;

	if (pwr_get_domain_state(tunnel, id, state, reset))
		goto done;
	pr_info("power domain %s presently set to state %d, reset %d\n",
		name, *state, *reset);

done:
	return rc;
}
EXPORT_SYMBOL(brcm_pwr_rpc_get_domain_state);

int brcm_pwr_rpc_set_domain_state(char *name, int state, int reset)
{
	int tunnel = 0, id = 0;
	int rc = 0;

	if (!name) {
		pr_err("%s : Invalid argument name:%s\n",
			   MODULE_NAME, name);
		rc = -EIO;
		goto done;
	}

	tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(PWR_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, PWR_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto done;
	}

	if (pwr_get_domain_id(tunnel, name, &id))
		goto done;

	if (pwr_set_domain_state(tunnel, id, state, reset))
		goto done;
	pr_info("set power domain %s to state %d, reset %d\n", name, state, reset);

done:
	return rc;
}
EXPORT_SYMBOL(brcm_pwr_rpc_set_domain_state);

int brcm_pwr_rpc_get_domain_state_atomic(int dom_id, pwr_rpc_cb_func cb_func)
{
	int rc = 0;

	if (dom_id <=0) {
		pr_err("%s : Invalid argument name:%d\n",
			   MODULE_NAME, dom_id);
		rc = -EIO;
		goto done;
	}

	if (pwr_get_domain_state_atomic(tunnel_id, dom_id, cb_func))
		goto done;

done:
	return rc;
}
EXPORT_SYMBOL(brcm_pwr_rpc_get_domain_state_atomic);

int brcm_pwr_rpc_set_domain_state_atomic(int dom_id, int state, int reset,
					 pwr_rpc_cb_func cb_func)
{
	int rc = 0;

	if (dom_id <=0) {
		pr_err("%s : Invalid argument dom_id:%d\n",
			   MODULE_NAME, dom_id);
		rc = -EIO;
		goto done;
	}

	pr_info("setting power domain ID %d to state %d, reset %d\n",
		dom_id, state, reset);
	if (pwr_set_domain_state_atomic(tunnel_id, dom_id, state, reset, cb_func))
		goto done;

done:
	return rc;
}
EXPORT_SYMBOL(brcm_pwr_rpc_set_domain_state_atomic);

int brcm_pwr_rpc_register_domain(char *name)
{
	int tunnel = 0, id = 0;
	int rc = 0;

	if (!name) {
		pr_err("%s : Invalid argument name:%s\n",
			   MODULE_NAME, name);
		rc = -EIO;
		goto done;
	}

	tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(PWR_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, PWR_RPC_RG_SMC_TUNNEL_NAME);
		rc = tunnel;
		goto done;
	}

	if (pwr_get_domain_id(tunnel, name, &id)){
		rc = -EIO;
		goto done;
	}
	domain_tbl[id].registered = true;

	return id;

done:
	return rc;
}
EXPORT_SYMBOL(brcm_pwr_rpc_register_domain);

void brcm_pwr_rpc_unregister_domain(int id)
{
	if (!domain_tbl[id].registered)
		pr_err("%s: ID %d is not a registered domain\n", __func__, id);

	domain_tbl[id].registered = false;
}
EXPORT_SYMBOL(brcm_pwr_rpc_unregister_domain);

bool brcm_pwr_rpc_is_registered_domain_turned_off(int id)
{
	int i = 0;

	if (id == PWR_DOMAINS_MAX) {
		for (i = 0; i < PWR_DOMAINS_MAX; i++) {
			if (domain_tbl[i].registered) {
				if (domain_tbl[i].state) {
					return false;
				}
			}
		}
	} else {
		if (domain_tbl[id].registered) {
			if (!domain_tbl[id].state)
				return true;
			else
				return false;
		}
	}

	return true;
}
EXPORT_SYMBOL(brcm_pwr_rpc_is_registered_domain_turned_off);

static int pwr_async_complete(int tunnel, rpc_msg *msg)
{
	int id = 0;
	u32 tag = msg->data[2], cnt = 0;
	pwr_rpc_cb_func cb_func;

	pr_debug("-->\n");

	if (debug)
		rpc_dump_msg(msg);

	pr_debug("TAG-%x Sync complete for Domain-ID-%d with State-%d and Reset-%d",
			 tag, pwr_msg_get_domain_id(msg),
			 pwr_msg_get_domain_state(msg),
			 pwr_msg_get_domain_reset(msg));

	id = pwr_msg_get_domain_id(msg);
	if (id >= PWR_DOMAINS_MAX) {
		pr_debug("Domain-ID-%d on Sync complete Invalid\n", id);
		return 0;
	}
	domain_tbl[id].id = id;
	domain_tbl[id].state = pwr_msg_get_domain_state(msg);
	domain_tbl[id].reset = pwr_msg_get_domain_reset(msg);

	cnt = (tag%MAX_CB_NUM);
	if (async_cb[cnt].tag == tag && async_cb[cnt].dom_id == id)
	{
		spin_lock(&async_cb_lock);
		async_cb[cnt].cb_init = false;
		del_timer(&async_cb[cnt].timer);
		cb_func = async_cb[cnt].cb_func;
		if (cb_func)
			cb_func(id, (void *)&domain_tbl[id].state,
				(void *)&domain_tbl[id].reset);
		async_cb[cnt].cb_func = NULL;
		spin_unlock(&async_cb_lock);
	}
	else
		pr_info("Async Call back not registered (or) time out\n");


	pr_debug("<--\n");
	return 0;
}

static rpc_function services_tbl[MAX_PWR_FUNC] = {
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ NULL,			0 },
	{ pwr_async_complete,	0 },
};

static void *seq_start(struct seq_file *seq, loff_t *pos)
{
	if (*pos < 1)
		return pos;

	return NULL;
}

static void *seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	++*pos;
	if (*pos < 1)
		return pos;

	return NULL;
}

static void seq_stop(struct seq_file *seq, void *v)
{
}

static int seq_show(struct seq_file *seq, void *v)
{
	int i;
	pr_seq(seq, "ID : Name     State Reset\n");
	for (i = 0; i < PWR_DOMAINS_MAX; i++) {
		if (domain_tbl[i].id < 0)
			continue;
		pr_seq(seq, "%-3d: %-8s %-2d    %-2d\n",
		       domain_tbl[i].id, domain_tbl[i].name,
		       domain_tbl[i].state, domain_tbl[i].reset);
	}

	return 0;
}

static const struct seq_operations command_seq_ops = {
	.start	= seq_start,
	.next	= seq_next,
	.stop	= seq_stop,
	.show	= seq_show,
};

static void cmd_debug_help(char *str)
{
	pr_alert("%s debug: Enable/disable debug prints\n", str);
	pr_alert("%s  debug <1|0>\n", str);
}

static int cmd_debug(int argc, char *argv[])
{
	if (argc  > 1) {
		if (kstrtoint(argv[1], 0, &debug)) {
			pr_err("Error %s: invalid debug %s\n",
				argv[0], argv[1]);
			goto done;
		}
		goto done;
	}
	pr_info("Debug: %d\n", debug);

done:
	return 0;
}

static void cmd_getid_help(char *str)
{
	pr_alert("%s getid: Get power domain id based on name\n", str);
	pr_alert("%s  getid <tunnel> <name>\n", str);
	pr_alert("%s   tunnel - rg-smc for SMC\n", str);
	pr_alert("%s   name - fpm, netport, gfap_pc, spu etc\n", str);
}

static int cmd_getid(int argc, char *argv[])
{
	int tunnel, id;
	if (argc  < 3) {
		cmd_getid_help("");
		return 0;
	}
	/* Validate tunnel name or id */
	if (kstrtoint(argv[1], 0, &tunnel)) {
		/* Not at number, check name */
		tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(argv[1]);
		if (tunnel < 0) {
			pr_err("Error %s: invalid tunnel %s\n",
				argv[0], argv[1]);
			goto done;
		}
	}
	if (strlen(argv[2]) > 8) {
		pr_err("Error %s: name %s length > 8\n",
			argv[0], argv[2]);
		goto done;
	}
	if (pwr_get_domain_id(tunnel, argv[2], &id))
		goto done;

	pr_info("Domain Id for name %-8s = %d\n", argv[2], id);

done:
	return 0;
}

static void cmd_getname_help(char *str)
{
	pr_alert("%s getname: Get power domain name based on id\n", str);
	pr_alert("%s  getname <tunnel> <id>\n", str);
	pr_alert("%s   tunnel - rg-smc for SMC\n", str);
	pr_alert("%s   id - 0 to 127\n", str);
}

static int cmd_getname(int argc, char *argv[])
{
	int tunnel, id;
	char name[PWR_DOMAIN_NAME_MAX_LEN] = {0};
	if (argc  < 3) {
		cmd_getname_help("");
		return 0;
	}
	/* Validate tunnel name or id */
	if (kstrtoint(argv[1], 0, &tunnel)) {
		/* Not at number, check name */
		tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(argv[1]);
		if (tunnel < 0) {
			pr_err("Error %s: invalid tunnel %s\n",
				 argv[0], argv[1]);
			goto done;
		}
	}
	if (kstrtoint(argv[2], 0, &id)) {
		pr_err("Error %s: invalid id %s\n",
			argv[0], argv[2]);
		goto done;
	}
	if (id > 0xFF) {
		pr_err("Error %s: id %s length > 0xFF\n",
			argv[0], argv[2]);
		goto done;
	}
	if (pwr_get_domain_name(tunnel, id, name))
		goto done;
	pr_info("Domain Name for id %d = %-8s\n", id, name);

done:
	return 0;
}

static void cmd_getstate_help(char *str)
{
	pr_alert("%s getstate: Get power domain state based on id or name\n", str);
	pr_alert("%s  getstate <tunnel> <id or name>\n", str);
	pr_alert("%s   tunnel - rg-smc for SMC\n", str);
	pr_alert("%s   name - fpm, netport, gfap_pc, spu etc\n", str);
	pr_alert("%s   id - 0 to 127\n", str);
}

static int cmd_getstate(int argc, char *argv[])
{
	int tunnel, id, state, reset;
	if (argc  < 3) {
		cmd_getstate_help("");
		return 0;
	}
	/* Validate tunnel name or id */
	if (kstrtoint(argv[1], 0, &tunnel)) {
		/* Not at number, check name */
		tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(argv[1]);
		if (tunnel < 0) {
			pr_err("Error %s: invalid tunnel %s\n",
				argv[0], argv[1]);
			goto done;
		}
	}
	if (kstrtoint(argv[2], 0, &id)) {
		/* Name to id */
		if (pwr_get_domain_id(tunnel, argv[2], &id)) {

			pr_err("Error %s: invalid id %s\n",
				argv[0], argv[2]);
			goto done;
		}
	}
	if (id > 0xFF) {
		pr_err("Error %s: id %s length > 0xFF\n",
			argv[0], argv[2]);
		goto done;
	}
	if (pwr_get_domain_state(tunnel, id, &state, &reset))
		goto done;
	pr_info("Domain State for id:%d State:0x%x Reset:%d\n", id, state,
		reset);

done:
	return 0;
}

static void cmd_setstate_help(char *str)
{
	pr_alert("%s setstate: Set power domain state based on id or name\n", str);
	pr_alert("%s  setstate <tunnel> <id or name> <state> <reset>\n", str);
	pr_alert("%s   tunnel - rg-smc for SMC\n", str);
	pr_alert("%s   name - fpm, netport, gfap_pc, spu etc\n", str);
	pr_alert("%s   id - 0 to 127\n", str);
	pr_alert("%s   state - 0 to 15\n", str);
	pr_alert("%s   reset - 0 or 1\n", str);
}

static int cmd_setstate(int argc, char *argv[])
{
	int tunnel, id, state, reset=0;
	if (argc  < 4) {
		cmd_setstate_help("");
		return 0;
	}
	/* Validate tunnel name or id */
	if (kstrtoint(argv[1], 0, &tunnel)) {
		/* Not at number, check name */
		tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(argv[1]);
		if (tunnel < 0) {
			pr_err("Error %s: invalid tunnel %s\n",
				argv[0], argv[1]);
			goto done;
		}
	}
	if (kstrtoint(argv[2], 0, &id)) {
		/* Name to id */
		if (pwr_get_domain_id(tunnel, argv[2], &id)) {

			pr_err("Error %s: invalid id %s\n",
				argv[0], argv[2]);
			goto done;
		}
	} else if (id > 0xFF) {
		pr_err("Error %s: id %s length > 0xFF\n",
			argv[0], argv[2]);
		goto done;
	}
	if (kstrtoint(argv[3], 0, &state)) {
		pr_err("Error %s: invalid state %s\n",
			argv[0], argv[3]);
		goto done;
	}
	if (argc == 5) {
		if (kstrtoint(argv[4], 0, &reset)) {
			pr_err("Error %s: invalid reset %s\n",
				argv[0], argv[4]);
			goto done;
		}
	}
	if (pwr_set_domain_state(tunnel, id, state, reset))
		goto done;

done:
	return 0;
}

static void cmd_clear_help(char *str)
{
	pr_alert("%s clear: Clear local cache of power domain table\n", str);
}

static int cmd_clear(int argc, char *argv[])
{
	clear_domain_tbl();

	return 0;
}

static void cmd_getall_help(char *str)
{
	pr_alert("%s getall: Get power domain state\n", str);
	pr_alert("%s  getall <tunnel>\n", str);
	pr_alert("%s   tunnel - rg-smc for SMC\n", str);
}

static int cmd_getall(int argc, char *argv[])
{
	int tunnel, i, state, reset=0;
	char name[PWR_DOMAIN_NAME_MAX_LEN] = {0};
	if (argc  !=  2) {
		cmd_getall_help("");
		return 0;
	}
	/* Validate tunnel name or id */
	if (kstrtoint(argv[1], 0, &tunnel)) {
		/* Not at number, check name */
		tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(argv[1]);
		if (tunnel < 0) {
			pr_err("Error %s: invalid tunnel %s\n",
				argv[0], argv[1]);
			goto done;
		}
	}
	for (i = 0; i < PWR_DOMAINS_MAX; i++) {
		if (pwr_get_domain_name(tunnel, i, name))
			continue;
		pr_info("Domain Name for id %d = %-8s\n", i, name);
		if (pwr_get_domain_state(tunnel, i, &state, &reset))
			continue;
		pr_info("Domain State for %d = 0x%x Reset state %d\n", i, state,
			reset);
	}

done:
	return 0;
}

static struct proc_cmd_ops command_entries[] = {
	PROC_CMD_INIT("clear",    cmd_clear),
	PROC_CMD_INIT("debug",    cmd_debug),
	PROC_CMD_INIT("getid",    cmd_getid),
	PROC_CMD_INIT("getname",  cmd_getname),
	PROC_CMD_INIT("getstate", cmd_getstate),
	PROC_CMD_INIT("getall",   cmd_getall),
	PROC_CMD_INIT("setstate", cmd_setstate),
};

struct proc_cmd_table brcm_pwr_rpc_command_table = {
	.module_name = MODULE_NAME,
	.size = ARRAY_SIZE(command_entries),
	.data_seq_read = (void *) &command_seq_ops,
	.ops = command_entries
};

/* MBOX notifier handlers */
int brcm_pwr_rpc_mbox_register_notifier(struct notifier_block *nb)
{
	return atomic_notifier_chain_register(&brcm_pwr_rpc_chain, nb);
}
EXPORT_SYMBOL(brcm_pwr_rpc_mbox_register_notifier);

int brcm_pwr_rpc_mbox_unregister_notifier(struct notifier_block *nb)
{
	return atomic_notifier_chain_unregister(&brcm_pwr_rpc_chain, nb);
}
EXPORT_SYMBOL(brcm_pwr_rpc_mbox_unregister_notifier);

static int pwr_rpc_mbox_notifier_call_chain(
				unsigned long val, void *v)
{
	return atomic_notifier_call_chain(&brcm_pwr_rpc_chain, val, v);
}

/**
* brcm_pwr_rpc_mbox_irq() - Mailbox IRQ handler
*
* This will handle any interrupt from registered mailbox's
* (Battery state change)
*/
static irqreturn_t brcm_pwr_rpc_mbox_irq(int irq, void *dev_id)
{
	u32 stat;
	struct brcm_pwr_rpc_mbox_info *states;
	unsigned long flags;
	int i;

	if ((dev_id != (void *)info) || (!info))
		return IRQ_NONE;

	states = &info->states;

	spin_lock_irqsave(&info->state_lock, flags);

	R32(stat, info->reg_status);

	/* Only get/check mbox's we're allowed to see */
	stat &= info->read_mask;
	if (!stat) {
		spin_unlock_irqrestore(&info->state_lock, flags);
		return IRQ_NONE; /* Not an IRQ we requested */
	}

	/* Clear the status. Must be done before reading */
	W32(info->reg_status, stat);

	for (i = 0; i < (PWR_MBOX_MAX); i++) {
		if (stat & MBOX_BIT(i)) {
			R32(states->mbox[i], MBOX(i));
			pr_debug("MBOX %d (%s): 0x%08x\n", i,
				   pwr_mbox_to_string(i), states->mbox[i]);
		}
	}
	states->mask = stat;

	pwr_rpc_mbox_notifier_call_chain(PWR_MBOX_CHANGE_EVENT, states);
	spin_unlock_irqrestore(&info->state_lock, flags);

	return IRQ_HANDLED;
}

/**
* brcm_pwr_rpc_of_map() - Get info from device tree.
*/
static int brcm_pwr_rpc_of_map(struct device_node *node)
{
	const __be32 *prop_reg;
	int len;

	pr_debug("-->\n");

	if (!node)
		pr_err("Invalid device node\n");

	/* Get read masks */
	prop_reg = (u32 *)of_get_property(node, "mbox-irq-read-mask", &len);
	if (prop_reg)
		info->read_mask = (u16)be32_to_cpu(*prop_reg);


	if (of_property_read_string(node, INTR_NAME_PROPERTY, &info->cpuc_irq))
		pr_info("%s : %s property not found\n",
				INTR_NAME_PROPERTY, MODULE_NAME);

	/* Check delay battery mode status from device tree */
	if (of_property_read_bool(node, "delay-battery-mode"))
		info->states.delay_battery_mode = true;
	else
		info->states.delay_battery_mode = false;
	delay_batt_mode = info->states.delay_battery_mode;

	if (of_property_read_string(node, BOOT_STATE_PROPERTY, &boot_state_str))
		pr_info("%s : %s property not found\n",
				BOOT_STATE_PROPERTY, MODULE_NAME);
	pr_debug("<--\n");
	return 0;
}

/**
* brcm_pwr_rpc_mbox_get_state() - Get current state of the
* system
*
* Will query Battery state (AC power, Battery power)
*/
int brcm_pwr_rpc_mbox_get_state(struct brcm_pwr_rpc_mbox_info *state)
{
	unsigned long flags;
	int i;

	pr_debug("-->\n");

	spin_lock_irqsave(&info->state_lock, flags);

	for (i = 0; i < PWR_MBOX_MAX; i++)
		if (info->read_mask & MBOX_BIT(i))
			R32(state->mbox[i], MBOX(i));

	spin_unlock_irqrestore(&info->state_lock, flags);

	pr_debug("<--\n");
	return 0;
}
EXPORT_SYMBOL(brcm_pwr_rpc_mbox_get_state);

void brcm_pwr_rpc_get_boot_state(const char **str)
{
	*str = boot_state_str;
}
EXPORT_SYMBOL_GPL(brcm_pwr_rpc_get_boot_state);

void brcm_pwr_rpc_get_delay_batt_mode(bool *state)
{
	*state = delay_batt_mode;
}
EXPORT_SYMBOL_GPL(brcm_pwr_rpc_get_delay_batt_mode);

/**
* brcm_pwr_rpc_probe()
*
* Get resources, get current power state,
* register IRQ (if so configured), create device node, check OF for settings.
*/
static int brcm_pwr_rpc_probe(struct platform_device *pdev)
{
	int ret = -1, tunnel = -1;
	u32 tmp;
	struct resource *res;
	struct device *dev = &pdev->dev;
	struct device_node *np;
	const char *str;

	pr_debug("-->\n");

	info = (struct brcm_pwr_rpc_dev_info *)devm_kzalloc(dev,
		sizeof(struct brcm_pwr_rpc_dev_info), GFP_KERNEL);
	if (!info) {
		pr_err("Failed to allocate memory!\n");
		ret = -ENOMEM;
		goto error_exit;
	}

	info->irq = -1;
	spin_lock_init(&info->state_lock);

	platform_set_drvdata(pdev, (void *)info);
	info->pdev = pdev;

	res = platform_get_resource(pdev, IORESOURCE_MEM, IO_MEM_MBOX);
	/* Exclusive access to MBOX registers */
	info->mbox = devm_ioremap_resource(dev, res);
	if ((!res) || !info->mbox) {
		pr_err("Error getting MBOX IO resource\n");
		ret = -EADDRNOTAVAIL;
		goto error_exit;
	}
	res = platform_get_resource(pdev, IORESOURCE_MEM, IO_MEM_COMM);
	if (!res) {
		pr_err("Error getting COMM IO resource\n");
		ret = -EADDRNOTAVAIL;
		goto error_exit;
	}
	/* Do NOT get exclusive access to CPU COMM registers */
	info->comm = devm_ioremap(dev, res->start, resource_size(res));
	if (!info->comm) {
		pr_err("Error getting COMM IO resource\n");
		ret = -EADDRNOTAVAIL;
		goto error_exit;
	}
	/* Collect any settings from device tree */
	if (brcm_pwr_rpc_of_map(dev->of_node) != 0) {
		pr_err("Error in OF configuration!\n");
		goto error_exit;
	}

	/* Request and register IRQ */
	info->irq = platform_get_irq(pdev, 0);
	if (info->irq < 0) {
		pr_err("Missing IRQ resource\n");
		ret = -ENXIO;
		goto error_exit;
	}

	/* Register for any interrupt specified in DT */
	info->read_mask = SET_CPU_COMM_MBOX_READ_MASK(info->read_mask);

	/* Which interrupt registers to use */
	if (!strcmp(info->cpuc_irq, "cpucomm_1_toB53_irq")) {
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_MASK;
	}
	else if (!strcmp(info->cpuc_irq, "cpucomm_0_toB53_irq")) {
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM0_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM0_MASK;
	}
	else if (!strcmp(info->cpuc_irq, "cpucomm_2_toB53_irq")) {
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM2_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM2_MASK;
	}
	else if (!strcmp(info->cpuc_irq, "cpucomm_3_toB53_irq")) {
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM3_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM3_MASK;
	}
	else {
		info->reg_status = CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_STATUS;
		info->reg_mask = CPU_COMM_REGS_CPUC_L2_IRQ_ARM1_MASK;
	}

	/* Get the current states of power/cpu's */
	/* Clear status that we care about, then check all to prevent
	 * unexpected interrupts from before driver comes on-line */
	W32(info->reg_status, info->read_mask);
	brcm_pwr_rpc_mbox_get_state(&info->states);

	ret = devm_request_irq(dev, info->irq, &brcm_pwr_rpc_mbox_irq, IRQF_SHARED,
			"brcm-mbox", (void *)info);
	if (ret < 0) {
		pr_err("error requesting IRQ #%d\n", info->irq);
		info->irq = -1; /* So that cleanup code works correctly */
		ret = -ENXIO;
		goto error_exit;
	}

	/* Enable MBOX interrupts.. TBD */
	R32(tmp, info->reg_mask);
	W32(info->reg_mask, info->read_mask | tmp);

	tunnel = brcm_pwr_rpc_get_fifo_tunnel_id(PWR_RPC_RG_SMC_TUNNEL_NAME);
	if (tunnel < 0) {
		pr_err("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, PWR_RPC_RG_SMC_TUNNEL_NAME);
		ret = tunnel;
		goto error_exit;
	}

	for_each_child_of_node(dev->of_node, np) {
		ret = of_property_read_string(np, "status", &str);
		if (ret == -EINVAL || !strncmp(str, "okay", 4) || !strncmp(str, "ok", 2)) {
			ret = pwr_rpc_pseudo_clk_setup(np);
			if (ret)
				goto error_exit;
		}
	}

	pr_debug("<--\n");
	return 0;

error_exit:

	if (info) {
		if (info->mbox)
			iounmap(info->mbox);
		if (info->comm)
			iounmap(info->comm);
		if (info->irq >= 0)
			devm_free_irq(dev, info->irq, (void *)info);

		devm_kfree(dev, (void *)info);
		info = NULL;
	}

	pr_debug("Error Exit <--\n");
	return ret;
}

/**
* brcm_pwr_rpc_remove()
*/
static int brcm_pwr_rpc_remove(struct platform_device *pdev)
{
	u32 tmp;
	struct brcm_pwr_rpc_dev_info *pinfo;
	int cnt;

	pr_debug("-->\n");

	pinfo = (struct brcm_pwr_rpc_dev_info *)platform_get_drvdata(pdev);
	if (!pinfo)
		goto error_exit;

	/* Disable MBOX interrupts */
	pr_debug("Enable MBOX interrupt mask\n");
	R32(tmp, info->reg_mask);
	tmp &= ~info->read_mask;
	W32(info->reg_mask, tmp);

	for (cnt = 0; cnt < MAX_CB_NUM; cnt++) {
		if(async_cb[cnt].cb_func){
			del_timer(&async_cb[cnt].timer);
			async_cb[cnt].cb_func = NULL;
			async_cb[cnt].cb_init = false;
		}
	}

	if (info->mbox)
		iounmap(info->mbox);
	if (info->comm)
		iounmap(info->comm);
	if (info->irq >= 0)
		devm_free_irq(&pdev->dev, info->irq, (void *)info);

	devm_kfree(&pdev->dev, (void *)info);
	pinfo = NULL;

error_exit:
	pr_debug("<--\n");
	return 0;
}

static const struct of_device_id brcm_pwr_rpc_of_match[] = {
	{ .compatible = PWR_RPC_OF_MATCH},
	{}
};
MODULE_DEVICE_TABLE(of, brcm_pwr_rpc_of_match);
static struct platform_driver brcm_pwr_rpc_driver = {
	.probe = brcm_pwr_rpc_probe,
	.remove = brcm_pwr_rpc_remove,
	.driver	= {
		.name		= MODULE_NAME,
		.owner		= THIS_MODULE,
		.of_match_table	= brcm_pwr_rpc_of_match
	}
};

static struct proc_dir_entry *pwr_rpc_proc_dir;

static int __init brcm_pwr_rpc_init(void)
{
	int rc = 0;

	rc = rpc_register_functions(RPC_SERVICE_PWR,
				    services_tbl,
				    MAX_PWR_FUNC);
	if (rc) {
		pr_err("%s: Failed to register PWR RPC function(s).\n",
			MODULE_NAME);
		return rc;
	}

	pwr_rpc_proc_dir = proc_mkdir("driver/itc-rpc/service/brcm-pwr-rpc", NULL);
	if (!pwr_rpc_proc_dir) {
		pr_err("%s: Warning: cannot create proc file\n",
		       MODULE_NAME);
		rpc_unregister_functions(RPC_SERVICE_PWR);
		return -ENOMEM;
	}
	proc_create_cmd("pwr", pwr_rpc_proc_dir, &brcm_pwr_rpc_command_table);
	pr_info("%s: Power Domain RPC module\n", __func__);
	spin_lock_init(&async_cb_lock);
	clear_domain_tbl();
	platform_driver_register(&brcm_pwr_rpc_driver);
	return rc;
}
subsys_initcall(brcm_pwr_rpc_init);

static void __exit brcm_pwr_rpc_exit(void)
{
	rpc_unregister_functions(RPC_SERVICE_PWR);
	platform_driver_unregister(&brcm_pwr_rpc_driver);
}
module_exit(brcm_pwr_rpc_exit);

MODULE_DESCRIPTION("BRCM Power Domain RPC Module");
MODULE_LICENSE("GPL v2");
