/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2015-2020 Broadcom.  All rights
* reserved.
* The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
*
* Filename: miimdiomux_core339x.c
*
****************************************************************************
* Description: RGMII and MDIO Mux Init driver for BCM339X
* Author: Mark Newcomer <mark.newcomer@broadcom.com>
****************************************************************************/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/of_platform.h>
#include <linux/io.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/brcmstb/brcmstb.h>
#include "miimdiomux339x.h"

#define MODULE_NAME	"brcm-miimdiomux339x"
#define MODULE_VER	"1.1"

#define MDIO_C_EN_MASK				0x00000008
#define MDIO_C_EN_SHIFT				3
#define MDIO_B_EN_MASK				0x00000004
#define MDIO_B_EN_SHIFT				2
#define MDIO_A_EN_MASK				0x00000002
#define MDIO_A_EN_SHIFT				1
#define MUX_UNIMAC_MDIO_EN_MASK		0x00000001
#define MUX_UNIMAC_MDIO_EN_SHIFT	0

struct mux_select select_map[] = {
	{
		"mdio_c_en",
		"mdio_reg",
		MDIO_C_EN_MASK,
		MDIO_C_EN_SHIFT,
		{
			{"disable", 0},
			{"enable",  1}
		},
		2
	},
	{
		"mdio_b_en",
		"mdio_reg",
		MDIO_B_EN_MASK,
		MDIO_B_EN_SHIFT,
		{
			{"disable", 0},
			{"enable",  1}
		},
		2
	},
	{
		"mdio_a_en",
		"mdio_reg",
		MDIO_A_EN_MASK,
		MDIO_A_EN_SHIFT,
		{
			{"disable", 0},
			{"enable",   1}
		},
		2
	},
	{
		"alt_rgmii",
		"mdio_reg",
		MUX_UNIMAC_MDIO_EN_MASK,
		MUX_UNIMAC_MDIO_EN_SHIFT,
		{
			{"a",	0},
			{"b",	1}
		},
		2
	},
};
#define SELECT_MAP_SIZE (sizeof(select_map)/sizeof(struct mux_select))

static int miimdiomux_select(struct platform_device *pdev,
		const char *mux_select_name, const char *mux_value_name)
{
	struct miimdiomux_priv *priv = platform_get_drvdata(pdev);
	struct mux_select *select;
	struct mux_value *value;
	int i, j;
	u32 val;
	void __iomem *reg;

	for (i = 0; i < SELECT_MAP_SIZE; i++) {
		if (!strcmp(mux_select_name, select_map[i].name))
			break;
	}

	if (i == SELECT_MAP_SIZE)
		return 0;

	if (!strcmp("mdio_reg", select_map[i].reg_name))
		reg = priv->mdio_enb_reg;
	else {
		printk("Invalid reg name value, %s, specified.\n",
			select_map[i].reg_name);
		return -EINVAL;
	}

	select = &select_map[i];
	for (j = 0; j < select->value_map_size; j++) {
		if (!strcmp(mux_value_name, select->value_map[j].name))
			break;
	}

	if (j == select->value_map_size) {
		printk("Invalid select value, %s, specified.\n",
			mux_value_name);
		return -EINVAL;
	}

	value = &select->value_map[j];
	val = __raw_readl(reg);
	val &= ~(select->mask);
	val |= ((value->val << select->shift) & select->mask);
	__raw_writel(val, reg);

	return 0;
}

static int miimdiomux_probe(struct platform_device *pdev)
{
	int status = 0;
	struct device *dev = &pdev->dev;
	struct resource *res;
	struct device_node *node = pdev->dev.of_node;
	struct property *prop;
	struct miimdiomux_priv *priv;

	pr_debug("-->\n");

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	res->flags &= ~IORESOURCE_CACHEABLE;
	priv->mdio_enb_reg = devm_ioremap_resource(dev, res);
	if (IS_ERR(priv->mdio_enb_reg)) {
		dev_err(dev, "Failed to map mdio enable register.");
		status = PTR_ERR(priv->mdio_enb_reg);
		goto done;
	}

	priv->notify = miimdiomux_select;
	platform_set_drvdata(pdev, priv);
	for_each_property_of_node(node, prop)
		if ((status = miimdiomux_select(pdev, prop->name, prop->value)) ==
				-EINVAL)
			break;

	dev_info(dev, "mdiomiimux339x: init complete!\n");
done:
	pr_debug("<--\n");
	return status;
}

static int miimdiomux_remove(struct platform_device *pdev)
{
	int status = 0;

	pr_debug("-->\n");

	pr_debug("<--\n");
	return status;
}

static const struct of_device_id miimdiomux_of_match[] = {
	{.compatible = "brcm,miimdiomux339x"},
	{.compatible = "brcm,bcm-fixed-syscon"},
	{}
};

MODULE_DEVICE_TABLE(of, miimdiomux_of_match);

static struct platform_driver miimdiomux_driver = {
	.probe  = miimdiomux_probe,
	.remove = miimdiomux_remove,
	.driver = {
		.name = MODULE_NAME,
		.owner = THIS_MODULE,
		.of_match_table = miimdiomux_of_match
	}
};

static int __init miimdiomux_init(void)
{
	pr_debug("%s driver v%s\n", MODULE_NAME, MODULE_VER);
	return platform_driver_register(&miimdiomux_driver);
}
subsys_initcall(miimdiomux_init);

static void __exit miimdiomux_exit(void)
{
	platform_driver_unregister(&miimdiomux_driver);
}
module_exit(miimdiomux_exit);

MODULE_AUTHOR("Mark Newcomer");
MODULE_DESCRIPTION("339x RGMII and MDIO Mux Init");
MODULE_LICENSE("GPL v2");
