 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/phy.h>

#include "ethsw.h"
#include "ethsw_priv.h"
#include "ethsw_core.h"
#include "ethsw_3390.h"

/*
 * Log Utilities
 */

/* define log module */
#define LOG_MODULE "ethsw_flow_control"

#define ACB_EN 0x1
#define ACB_ALGORITHM_1 0x0
#define ACB_ALGORITHM_2 0x1
#define XOFF_THRESHOLD 18
#define PRIORITY_TO_QID 0xfed6c9
#define ACB_REG_SIZE 4

static int ethsw_flowctl_port_addr_map[] = {
	BCHP_SWITCH_CORE_TC2COS_MAP_P0,
	BCHP_SWITCH_CORE_TC2COS_MAP_P1,
	BCHP_SWITCH_CORE_TC2COS_MAP_P2,
	BCHP_SWITCH_CORE_TC2COS_MAP_P3,
	BCHP_SWITCH_CORE_TC2COS_MAP_P4,
	BCHP_SWITCH_CORE_TC2COS_MAP_P5,
	0,
	BCHP_SWITCH_CORE_TC2COS_MAP_P7,
	BCHP_SWITCH_CORE_TC2COS_MAP_IMP
};

int ethsw_set_tc2cos_mapping(struct ethsw_device *swdev, u32 port, u32 tc, u32 cos)
{
	u32 mask, shift;

	if(swdev->port[port].type == INV_PORT) {
		LOG_WARNING("Attempted to set TC to COS mapping for unsupported port %d\n",
			port);
		return -EINVAL;
	}

	/* Sinle TC2COS_MAP register describes tc to cos mapping for a single
	   port. All registers are identical so we can use the same mask and
	   shift */
	mask = BCHP_SWITCH_CORE_TC2COS_MAP_P0_PRT000_TO_QID_MASK <<
		(BCHP_SWITCH_CORE_TC2COS_MAP_P0_PRT001_TO_QID_SHIFT * tc);
	shift = (BCHP_SWITCH_CORE_TC2COS_MAP_P0_PRT001_TO_QID_SHIFT * tc);

	LOG_DEBUG("port %d, tc 0x%x, cos 0x%x, mask 0x%x, shift %d\n",
		  port, tc, cos, mask, shift);

	ethsw_reg_set_field(swdev, ethsw_flowctl_port_addr_map[port],
			    mask, shift, cos);
	return 0;
}

int ethsw_get_tc2cos_mapping(struct ethsw_device *swdev, u32 port, u32 tc, u32 *cos)
{
	u32 mask, shift;

	if(swdev->port[port].type == INV_PORT) {
		LOG_WARNING("Attempted to set TC to COS mapping for unsupported port %d\n",
			    port);
		return -EINVAL;
	}

	/* Sinle TC2COS_MAP register describes tc to cos mapping for a single
	   port. All registers are identical so we can use the same mask and
	   shift */
	mask = BCHP_SWITCH_CORE_TC2COS_MAP_P0_PRT000_TO_QID_MASK <<
		(BCHP_SWITCH_CORE_TC2COS_MAP_P0_PRT001_TO_QID_SHIFT * tc);
	shift = (BCHP_SWITCH_CORE_TC2COS_MAP_P0_PRT001_TO_QID_SHIFT * tc);

	*cos = ethsw_reg_get_field(swdev, ethsw_flowctl_port_addr_map[port],
				   mask, shift);
	return 0;
}

int ethsw_set_acb_threshold(struct ethsw_device *swdev, u32 port, u32 qid, u32 threshold)
{
	u32 reg;

	if ((swdev->port[port].type == INV_PORT) ||
		((swdev == ethsw_dev[INTERNAL_SW]) && (port == 6))) {
			LOG_WARNING("Attempted to set TC to COS mapping for unsupported port %d\n",
			    port);
		return -EINVAL;
	}

	/* Admission Control Block (ACB) monitors queue depths for all switch queues
	   and provides congestion status information to IMP port connected blocks.
	   IMP port connected blocks monitor congestion status and do not schedule
	   packets for congested queues.

	   Queue identification Signaled as
	   {PORT_ID[2:0], QID[2:0]}. PORT_ID[2:0] is encoded as:
	   3'b000 - 3'b101 - P0 to P5
	   3'b110 - P7
	   3'b111 - P8
	   P6 is not used for SF2. */
	if (swdev == ethsw_dev[INTERNAL_SW])
		port = (port < 6) ? port : (port - 1);

	reg = BCHP_SWITCH_ACB_ACB_QUEUE_0_CONFIGURATION +
		(((port * ETHSW_ACB_QUEUES_PER_PORT) + qid) * ACB_REG_SIZE);
	/* All ACB_QUEUE registers are identical so we can use the same mask and
	   shift */
	ethsw_reg_set_field(swdev, reg,
			    BCHP_SWITCH_ACB_ACB_QUEUE_0_CONFIGURATION_xoff_threshold_MASK,
			    BCHP_SWITCH_ACB_ACB_QUEUE_0_CONFIGURATION_xoff_threshold_SHIFT,
			    threshold);
	return 0;
}

int ethsw_get_acb_threshold(struct ethsw_device *swdev, u32 port, u32 qid, u32 *threshold)
{
	u32 reg;

	if(swdev->port[port].type == INV_PORT) {
		LOG_WARNING("Attempted to set TC to COS mapping for unsupported port %d\n",
			    port);
		return -EINVAL;
	}

	reg = BCHP_SWITCH_ACB_ACB_QUEUE_0_CONFIGURATION +
		(((port * ETHSW_ACB_QUEUES_PER_PORT) + qid) * ACB_REG_SIZE);
	/* All ACB_QUEUE registers are identical so we can use the same mask and
	   shift */
	*threshold = ethsw_reg_get_field(swdev, reg,
		BCHP_SWITCH_ACB_ACB_QUEUE_0_CONFIGURATION_xoff_threshold_MASK,
		BCHP_SWITCH_ACB_ACB_QUEUE_0_CONFIGURATION_xoff_threshold_SHIFT);
	return 0;

}

int ethsw_acb_enable(struct ethsw_device *swdev)
{
	int ret = 0;
	int p, q, tc;
	int cos_map[] = { 1, 1, 3 ,3, 5 ,5, 7, 7 };
	int cos_map_size = sizeof(cos_map) / sizeof(int);

	FUNC_ENTER();

	LOG_DEBUG("Programming TC -> COS mappings\n");
	/* The lines below set mapping in the following way:
	   TC 7,6 -> COS 7
	   TC 5,4 -> COS 5
	   TC 3,2 -> COS 3
	   TC 1,0 -> COS 1
	   COS and QID are synonymous. It is important to note that the mappings
	   are programmed with ingress port number (from switch perspective)
	   but affect egress ports queues */
	for(p = 0; p < ETHSW_PORT_MAX; p ++) {
		if(swdev->port[p].type == IMP_PORT) {
			for(tc = 0; tc < cos_map_size; tc ++) {
				ret = ethsw_set_tc2cos_mapping(swdev, p, tc, cos_map[tc]);
				if(ret) goto ERROR;
			}
		}
	}

	LOG_DEBUG("Programming XOFF_THRESHOLD for all IMP->LAN queues\n");
	/* Set xoff_threshold for COS (QID) 1,3,5,7 for all LAN port: 0-4, 7 */
	for(p = 0; p < ETHSW_PORT_MAX; p ++) {
		if(swdev->port[p].type == PHY_PORT || swdev->port[p].type == MII_PORT ) {
			for(q = 1; q < ETHSW_ACB_QUEUES_PER_PORT; q += 2) {
				ret = ethsw_set_acb_threshold(swdev, p, q, XOFF_THRESHOLD);
				if(ret) goto ERROR;
			}
		}
	}

	LOG_INFO("Enabling ACB\n");
	/* Enable ACB */
	ethsw_reg_set_field(swdev, BCHP_SWITCH_ACB_ACB_CONTROL,
			    BCHP_SWITCH_ACB_ACB_CONTROL_acb_en_MASK,
			    BCHP_SWITCH_ACB_ACB_CONTROL_acb_en_SHIFT,
			    ACB_EN);

        /* Use ACB algorithm 1 */
	ethsw_reg_set_field(swdev, BCHP_SWITCH_ACB_ACB_CONTROL,
			    BCHP_SWITCH_ACB_ACB_CONTROL_acb_algorithm_MASK,
			    BCHP_SWITCH_ACB_ACB_CONTROL_acb_algorithm_SHIFT,
			    ACB_ALGORITHM_1);

	FUNC_LEAVE();
ERROR:
	if(ret) {
		LOG_ERR("failed to enable ACB\n");
	}
	return ret;
}
