/*
 * Test Driver for Broadcom RPC LED Service
 *
 * Copyright (C) 2020 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/bitops.h>
#include <linux/gpio/driver.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/of.h>
#include <linux/module.h>
#include <linux/irqdomain.h>
#include <linux/irqchip/chained_irq.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/types.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/delay.h>
#include <linux/notifier.h>
#include <proc_cmd.h>
#include <brcm_ba_rpc.h>

#define MODULE_NAME "rpc_led"
#define LED_EVENT_MAX 256
#define LED_EVENT_NAME_MAX_LEN 13
#define BA_NAME_MAX_LEN 9
#define LED_RPC_TIMEOUT 5

#define ANSI_RED	"\e[31;1m"
#define ANSI_YELLOW	"\e[33;1m"
#define ANSI_RESET	"\e[0m"
#define RED(str)	ANSI_RED str ANSI_RESET
#define YLW(str)	ANSI_YELLOW str ANSI_RESET

static uint32_t bfrg_alarm_id;
static int rg_smc_tunnel;

struct led_event {
	uint32_t id;
	char name[LED_EVENT_NAME_MAX_LEN];
};

enum led_rpc_func {
	GET_EVENT_ID,
	NOTIFY_EVENT,
	LED_MAX_FUNC
};

enum ba_rpc_func {
	GET_CPU_ID,
	GET_CPU_NAME,
	GET_RUN_STATE_ID,
	GET_RUN_STATE_NAME,
	GET_RUN_STATE,
	NOTIFY_RUN_STATE,
	REQUEST_RUN_STATE,
	REQUEST_RUN_STATE_RESPONSE,
	SET_RUN_STATE,
	BA_MAX_FUNC
};

struct ba_msg {
	uint32_t	hdr;
	union {
		uint32_t	rsvd0;
		struct {
			uint8_t	cpu_id;
			uint8_t	rs_id;
			union {
				uint8_t	be_rude:1;
				uint8_t	rsvd1:7;

				uint8_t	response:4;
				uint8_t rsvd2:4;
			};
			uint8_t	rc:8;
		};
	};
	union {
		uint32_t	rsvd3[2];
		char		name[8];
	};
};

static int debug;
static struct led_event event_history[LED_EVENT_MAX];

void clear_event_history(void)
{
	int i;

	for (i = 0; i < LED_EVENT_MAX; i++) {
		memset(event_history[i].name, 0, LED_EVENT_NAME_MAX_LEN);
		event_history[i].id = -1;
	}
}

static int led_get_event_id(int tunnel, char *name, uint32_t *id)
{
	rpc_msg msg;
	int rc = 0;

	rpc_msg_init(&msg, RPC_SERVICE_LED, GET_EVENT_ID, 0,
		     0, 0, 0);
	strncpy((char *)&msg.data[0], name,
		LED_EVENT_NAME_MAX_LEN-1);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_request_timeout(tunnel, &msg, LED_RPC_TIMEOUT);
	if (rc) {
		pr_err(RED("%s : rpc_send_request_timeout failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
		goto done;
	} else {
		if (debug)
			rpc_dump_msg(&msg);
		if ((msg.data[0] >> 24) == 0xff) {
			pr_err(RED("%s : invalid LED event name(%s) %d\n"),
			   MODULE_NAME, name, rc);
			rpc_dump_msg(&msg);
			rc = -EINVAL;
			goto done;
		}
		*id = msg.data[0] & 0xFFFF;
		event_history[*id%LED_EVENT_MAX].id = *id;
		strncpy(event_history[*id%LED_EVENT_MAX].name, name,
			LED_EVENT_NAME_MAX_LEN-1);
	}
done:
	return rc;
}

static int led_notify_event(int tunnel, uint32_t id)
{
	rpc_msg msg;
	int rc = 0;

	rpc_msg_init(&msg, RPC_SERVICE_LED, NOTIFY_EVENT, 0,
		     (id & 0xFFFF), 0, 0);

	if (debug)
		rpc_dump_msg(&msg);

	rc = rpc_send_message(tunnel, &msg, false);
	if (unlikely(rc)) {
		pr_err(RED("%s : rpc_send_message failure (%d)\n"),
			   MODULE_NAME, rc);
		rpc_dump_msg(&msg);
	}
	return rc;
}

int bcm_rpc_led_notify(char *alarm)
{
	int tunnel = 0, rc = 0;
	uint32_t id = 0;
	char *tname = "rg-smc";

	if (!alarm) {
		pr_err(RED("%s : Invalid argument alarm:%s\n"),
			   MODULE_NAME, alarm);
		rc = -EINVAL;
		goto done;
	}

	/* Not at number, check name */
	tunnel = rpc_get_fifo_tunnel_id(tname);
	if (tunnel < 0) {
		pr_info("%s : Error: invalid tunnel %s\n",
			MODULE_NAME, tname);
		rc = tunnel;
		goto done;
	}
	rc = led_get_event_id(tunnel, alarm, &id);
	if (rc) {
		pr_info("%s : LED get event id failure for alarm(%s)\n",
			MODULE_NAME, alarm);
		goto done;
	}
	rc = led_notify_event(tunnel, id);
	if (rc) {
		pr_info("%s : LED event notify failure for alarm(%s)\n",
			MODULE_NAME, alarm);
		goto done;
	}
done:
	return rc;
}
EXPORT_SYMBOL(bcm_rpc_led_notify);

static void *seq_start(struct seq_file *seq, loff_t *pos)
{
	if (*pos < 1)
		return pos;
	return NULL;
}

static void *seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	++*pos;
	if (*pos < 1)
		return pos;
	return NULL;
}

static void seq_stop(struct seq_file *seq, void *v)
{
}

static int seq_show(struct seq_file *seq, void *v)
{
	int i = 0;

	pr_seq(seq, "ID : Name\n");
	for (i = 0; i < LED_EVENT_MAX; i++) {
		if (event_history[i].id < 0)
			continue;
		pr_seq(seq, "%-3d: %-12s\n",
		       event_history[i].id, event_history[i].name);
	}

	return 0;
}

static const struct seq_operations command_seq_ops = {
	.start	= seq_start,
	.next	= seq_next,
	.stop	= seq_stop,
	.show	= seq_show,
};

static void cmd_debug_help(char *str)
{
	pr_alert("%s debug: Enable/Disable debug prints\n", str);
	pr_alert("%s   debug <1|0>\n", str);
}

static int cmd_debug(int argc, char *argv[])
{
	if (argc  > 1) {
		if (kstrtoint(argv[1], 0, &debug)) {
			pr_info("Error %s: invalid debug %s\n",
				argv[0], argv[1]);
			goto done;
		}
		goto done;
	}
	pr_info("Debug: %d\n", debug);
done:
	return 0;
}

static void cmd_getid_help(char *str)
{
	pr_alert("%s getid: Get led id based on name\n", str);
	pr_alert("%s  getid <tunnel> <name>\n", str);
}

static int cmd_getid(int argc, char *argv[])
{
	int tunnel = 0, rc = 0;
	uint32_t id = 0;

	if (argc  < 3) {
		cmd_getid_help("");
		return 0;
	}
	/* Validate tunnel name or id */
	if (kstrtoint(argv[1], 0, &tunnel)) {
		/* Not at number, check name */
		tunnel = rpc_get_fifo_tunnel_id(argv[1]);
		if (tunnel < 0) {
			pr_info("Error %s: invalid tunnel %s\n",
				argv[0], argv[1]);
			goto done;
		}
	}
	if (strlen(argv[2]) > 12) {
		pr_info("Error %s: name %s length > 12\n",
			argv[0], argv[2]);
		goto done;
	}
	rc = led_get_event_id(tunnel, argv[2], &id);
	if (rc) {
		pr_info("LED Get Event Id failure for name %-12s\n", argv[2]);
		goto done;
	}

	pr_info("LED Event Id for name %-12s = %d\n", argv[2], id);
done:
	return 0;
}

static void cmd_getname_help(char *str)
{
	pr_alert("%s getname: Get led name based on id\n", str);
	pr_alert("%s  getname <id>\n", str);
}

static int cmd_getname(int argc, char *argv[])
{
	int i = 0;
	uint32_t id = 0;

	if (argc  < 2) {
		cmd_getname_help("");
		return 0;
	}
	if (kstrtoint(argv[1], 0, &id)) {
		pr_info("Error %s: invalid id %s\n",
			argv[0], argv[1]);
		goto done;
	}
	if (id > 0xFFFF) {
		pr_info("Error %s: id %s length > 0xFFFF\n",
			argv[0], argv[1]);
		goto done;
	}

	for (i = 0; i < LED_EVENT_MAX; i++) {
		if (event_history[i].id == id)
			pr_info("LED event Name for id %d = %-12s\n",
					id, event_history[i].name);
	}
done:
	return 0;
}

static void cmd_notifyevent_help(char *str)
{
	pr_alert("%s notifyevent: Notify led event based based on name or id\n", str);
	pr_alert("%s  notifyevent <tunnel> <id>\n", str);
}

static int cmd_notifyevent(int argc, char *argv[])
{
	int tunnel = 0, rc = 0;
	uint32_t id = 0;

	if (argc  < 3) {
		cmd_notifyevent_help("");
		return 0;
	}
	/* Validate tunnel name or id */
	if (kstrtoint(argv[1], 0, &tunnel)) {
		/* Not at number, check name */
		tunnel = rpc_get_fifo_tunnel_id(argv[1]);
		if (tunnel < 0) {
			pr_info("Error %s: invalid tunnel %s\n",
				argv[0], argv[1]);
			goto done;
		}
	}
	if (kstrtoint(argv[2], 0, &id)) {
		/* Name to id */
		rc = led_get_event_id(tunnel, argv[2], &id);
		if (rc) {
			pr_info("Error %s: invalid name %s\n",
				argv[0], argv[2]);
			goto done;
		}
	} else if (id > 0xFFFF) {
		pr_info("Error %s: id %s length > 0xFFFF\n",
			argv[0], argv[2]);
		goto done;
	}
	rc = led_notify_event(tunnel, id);
	if (rc) {
		pr_info("%s : LED event notify failure for alarm(%s)\n",
			MODULE_NAME, argv[2]);
		goto done;
	}
done:
	return 0;
}

static void cmd_clear_help(char *str)
{
	pr_alert("%s clear: Clear local history\n", str);
}

static int cmd_clear(int argc, char *argv[])
{
	clear_event_history();
	return 0;
}

static struct proc_cmd_ops rpc_led_cmd_entries[] = {
	PROC_CMD_INIT("clear",   cmd_clear),
	PROC_CMD_INIT("debug",   cmd_debug),
	PROC_CMD_INIT("getid",   cmd_getid),
	PROC_CMD_INIT("getname", cmd_getname),
	PROC_CMD_INIT("notifyevent",  cmd_notifyevent),
};

struct proc_cmd_table rpc_led_cmd_table = {
	.module_name = MODULE_NAME,
	.size = ARRAY_SIZE(rpc_led_cmd_entries),
	.data_seq_read = (void *) &command_seq_ops,
	.ops = rpc_led_cmd_entries
};

static int panic_callback(struct notifier_block *self,
			  unsigned long event, void *ctx)
{
	time64_t now;
	struct tm tm_val;
	char rs_name[BA_NAME_MAX_LEN] = {0};
	int rc = 0;

	now = ktime_get_real_seconds();
	time64_to_tm(now, 0, &tm_val);

	pr_emerg("---[ LED Notify: Start %d/%d/%ld %02d:%02d:%02d ]---\n",
		 tm_val.tm_mon + 1, tm_val.tm_mday, 1900 + tm_val.tm_year,
		 tm_val.tm_hour, tm_val.tm_min, tm_val.tm_sec);
	rc = bcm_ba_get_cpu_run_state(rs_name, BA_RPC_RG_CPU_NAME);
	if (rc) {
		pr_emerg("---[ LED Notify: Failure ]---\n");
		return NOTIFY_OK;
	}
	if (!(!strncmp(rs_name, BA_RPC_RS_READY, sizeof(BA_RPC_RS_READY))) ||
         (!strncmp(rs_name, BA_RPC_RS_BATTERY, sizeof(BA_RPC_RS_BATTERY)))) {
		led_notify_event(rg_smc_tunnel, bfrg_alarm_id);
		pr_emerg("---[ LED Notify: BOOTFAILRG ]---\n");
	}
	pr_emerg("---[ LED Notify: End ]---\n");

	return NOTIFY_OK;
}

static struct notifier_block nb_panic = {
	.notifier_call  = panic_callback,
};

static struct proc_dir_entry *rpc_led_proc_dir;

static int __init bcm_rpc_led_init(void)
{
	int rc;

	atomic_notifier_chain_register(&panic_notifier_list,
				       &nb_panic);
	rpc_led_proc_dir = proc_mkdir("driver/itc-rpc/service/rpc-led", NULL);
	if (rpc_led_proc_dir == NULL) {
		pr_err("%s: Warning: cannot create proc file\n",
		       MODULE_NAME);
		rpc_unregister_functions(RPC_SERVICE_LED);
		return -ENOMEM;
	}
	proc_create_cmd("led", rpc_led_proc_dir, &rpc_led_cmd_table);
	pr_info("%s: LED RPC Service module\n", __func__);
	clear_event_history();

	/* Get Info for panic handler during bootup*/
	rg_smc_tunnel = rpc_get_fifo_tunnel_id("rg-smc");
	if (rg_smc_tunnel < 0) {
		pr_err("%s : Error: invalid tunnel rg-smc\n",
			MODULE_NAME);
		return -EINVAL;
	}
	rc = led_get_event_id(rg_smc_tunnel, "BOOTFAILRG", &bfrg_alarm_id);
	if (rc) {
		pr_err("%s : LED get event id failure for alarm(BOOTFAILRG)\n",
			MODULE_NAME);
		return -EINVAL;
	}

	return 0;
}
subsys_initcall(bcm_rpc_led_init);

static void __exit bcm_rpc_led_exit(void)
{
	atomic_notifier_chain_unregister(&panic_notifier_list,
					 &nb_panic);
}
module_exit(bcm_rpc_led_exit);

MODULE_DESCRIPTION("BCM RPC LED service Module");
MODULE_LICENSE("GPL v2");
