/*
 * Broadcom oops capture and display (boops)
 *
 * Copyright (C) 2017 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <linux/module.h>
#include <linux/kmsg_dump.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/random.h>
#include <linux/slab.h>
#include <linux/cpu.h>
#include <linux/ktime.h>
#include <linux/version.h>
#include <linux/kmsg_dump.h>

#include "boops_k.h"
#include "boops_k_proc.h"
#include "boops_k_common.h"
#include "boops_k_storage.h"

#define BOOPS_MIN_BLOCK_SIZE    512
#define BOOPS_MIN_BLOCK_COUNT   8
#define BOOPS_INVALID           0xFFFFFFFF
#define BOOPS_SEGMENT_BYTES     4096


/////////////////////////////////////////////////
// Global Parameters
////////////////////////////////////
static char storage_type[BOOPS_STORAGE_TYPE_LENGTH] = { 0 };
module_param_string(storage_type, storage_type, sizeof(storage_type), 0400);
MODULE_PARM_DESC(storage_type,
        "Indicate the type of storage to be used [mmc, ubi]");

static unsigned long segment_bytes = BOOPS_SEGMENT_BYTES;
module_param(segment_bytes, ulong, 0400);
MODULE_PARM_DESC(segment_bytes,
        "The amount of data between headers. Also how much data we attempt to read from kmsg buffer at once");

static char build_version[64] = { 0 };
module_param_string(build_version, build_version, sizeof(build_version), 0400);
MODULE_PARM_DESC(build_version,
        "The versionof software running on the device");

// UBI specific params
static unsigned long ubi_device = BOOPS_INVALID;
module_param(ubi_device, ulong, 0400);
MODULE_PARM_DESC(ubi_device,
        "The UBI device number for OOPS storage");

static unsigned long ubi_volume = BOOPS_INVALID;
module_param(ubi_volume, ulong, 0400);
MODULE_PARM_DESC(ubi_volume,
        "The UBI volume to use for OOPS storage");


// MMC specific params
static unsigned long mmc_block_first = BOOPS_INVALID;
module_param(mmc_block_first, ulong, 0400);
MODULE_PARM_DESC(mmc_block_first,
        "First writable block to dump the panic");

static unsigned long mmc_block_count = BOOPS_INVALID;
module_param(mmc_block_count, ulong, 0400);
MODULE_PARM_DESC(mmc_block_count,
        "Number of writable blocks");

struct boops_platform_data {
    char*    storage_type;
    uint32_t ubi_volume;
    uint32_t ubi_device;
    uint32_t mmc_block_first;
    uint32_t mmc_block_count;
    uint32_t segment_bytes;
};

static struct platform_device *dummy = NULL;
static struct boops_platform_data *dummy_data = NULL;

/* Needed global for device prints. Should be added to a print struct and pass to storage/proc devices */
char g_boops_current_boot_uuid[BOOPS_UUID_SIZE];
char g_storage_type[BOOPS_STORAGE_TYPE_LENGTH];


static const char *boops_get_reason_str(struct boops_t *boops, enum kmsg_dump_reason reason)
{
    if (boops->simulation_mode)
        return "Simulated Oops";

    switch (reason) {
    case KMSG_DUMP_PANIC:
        return "Kernel Panic";
    case KMSG_DUMP_OOPS:
        return "Kernel Oops";
    case KMSG_DUMP_EMERG:
        return "Emergency";
#if LINUX_VERSION_CODE < KERNEL_VERSION(5,15,0)
    case KMSG_DUMP_RESTART:
        return "Restart";
    case KMSG_DUMP_HALT:
        return "Halt";
    case KMSG_DUMP_POWEROFF:
        return "Poweroff";
#else
    case KMSG_DUMP_SHUTDOWN:
        return "Shutdown";
#endif
    default:
        return "Unknown";
    }
}

void boops_capture_oops(struct boops_t *boops, boops_dump_get_buffer get_buffer, void* context, enum kmsg_dump_reason reason) {
    struct boops_storage_t *storage = boops->storage;
    struct tm tm_val;
    bool data_remaining = true;
    int err = 0;
    uint32_t used_blocks = 0;
    uint8_t* segment_buffer = boops->block_buffer;
    struct boops_primary_header_t *primary_header = (struct boops_primary_header_t *)(segment_buffer);
    struct boops_extended_header_t *extended_header = &primary_header->extended_header;
    size_t length;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,9,0)
    struct timespec64 now;
    ktime_get_real_ts64(&now);
    time64_to_tm(now.tv_sec, 0, &tm_val);
#else
    struct timeval now;
    do_gettimeofday(&now);
    time_to_tm(now.tv_sec, 0, &tm_val);
#endif

    boops_info("Dumping OOPS for this boot cycle %pUB\n", g_boops_current_boot_uuid);

    /* Setup the primary header */
    memcpy(extended_header->uuid, g_boops_current_boot_uuid, sizeof(extended_header->uuid));
    strncpy(extended_header->build_version, build_version, sizeof(extended_header->build_version));
    strncpy(extended_header->reason, boops_get_reason_str(boops, reason), sizeof(extended_header->reason));
    extended_header->oops_date = BOOPS_DATE_SET_YEAR(tm_val.tm_year+1900) |
                                 BOOPS_DATE_SET_MONTH(tm_val.tm_mon+1) |
                                 BOOPS_DATE_SET_DAY(tm_val.tm_mday) |
                                 BOOPS_DATE_SET_HOUR(tm_val.tm_hour) |
                                 BOOPS_DATE_SET_MIN(tm_val.tm_min) |
                                 BOOPS_DATE_SET_SEC(tm_val.tm_sec);

    /* Loop until we've read all the data */
    while(data_remaining && used_blocks < storage->block.count) {
        struct boops_segment_header_t *segment_header=(struct boops_segment_header_t *)(segment_buffer);
        memcpy(segment_header->signature, BOOPS_SIGNATURE, sizeof(segment_header->signature));
        memcpy(segment_header->uuid, extended_header->uuid, sizeof(segment_header->uuid));
        segment_header->bytes.header = (used_blocks == 0)? sizeof(struct boops_primary_header_t) : sizeof(struct boops_segment_header_t);
        segment_header->bytes.total = boops->segment.bytes;

        /* Fill the segment payload */
        data_remaining = get_buffer(context, false,
                                    segment_buffer + segment_header->bytes.header,
                                    segment_header->bytes.total - segment_header->bytes.header,
                                    &length);

        if (!length) {
            if (!used_blocks) {
                boops_err("Nothing to dump from the kmsg buffer for boot %pUB\n", g_boops_current_boot_uuid);
                length = snprintf(segment_buffer + segment_header->bytes.header, segment_header->bytes.total - segment_header->bytes.header, "\n*** No available logs in the kmsg buffer ***\n");
            }
            data_remaining=false;
        }

        if (length > segment_header->bytes.total - segment_header->bytes.header) {
            boops_err("Overflow dumping the kmsg buffer for boot %pUB. Returned %lu bytes but limiting to %u\n", g_boops_current_boot_uuid, length, segment_header->bytes.total - segment_header->bytes.header);
            length=segment_header->bytes.total - segment_header->bytes.header;
            data_remaining=false;
        }
        segment_header->bytes.data = (uint32_t)(length);

        boops_info("Filled %u of %u bytes in segment\n", segment_header->bytes.data, segment_header->bytes.total - segment_header->bytes.header);
        used_blocks += boops->segment.blocks;
        segment_buffer += segment_header->bytes.total;
    }

    boops_info("Writing %u blocks\n", used_blocks);
    err=storage->write_blocks(storage, boops->block_buffer, used_blocks);
    if (err) {
        boops_err("Failed writing segment to the storage device\n");
        return;
    }

    boops_info("Completed OOPS dump for boot %pUB\n", g_boops_current_boot_uuid);

    /* Reset the block_buffer in case this was a test run */
    memset(boops->block_buffer, 0, boops->block_buffer_size);
}

static void boops_capture_oops_cb(struct kmsg_dumper *dumper, enum kmsg_dump_reason reason) {
    struct boops_t *boops = container_of(dumper, struct boops_t, dump);
    volatile int i;
    void * context = dumper;

    /* Block all cpus and let the last one in do the actual dump */
    if (boops->dump_requested++) {
        boops_info("CPU %u will block and wait for oops dump completion\n", smp_processor_id());
        while(!boops->dump_completed) {};
        boops_info("CPU %u unblocking...\n", smp_processor_id());
        return;
    }

    boops_info("CPU %u will perform the oops dump after a small delay...\n", smp_processor_id());
    for(i=0; i<boops->oops_dump_delay; i++) {} /* Give some time for others CPUS to dump their stacks if they are going to */

    boops_info("CPU %u delay complete\n", smp_processor_id());
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0)
    {
        struct kmsg_dump_iter iter;
        kmsg_dump_rewind(&iter);
        context = &iter;
    }
#endif

    boops_capture_oops(boops, kmsg_dump_get_buffer, context, reason);
    boops->dump_completed=true;
}

static int boops_probe(struct platform_device *pdev) {
    struct boops_platform_data *pdata = pdev->dev.platform_data;
    struct device *dev = &pdev->dev;
    struct boops_t *boops = NULL;
    struct boops_storage_t *storage = NULL;
    int err = 0;

    if (!dummy_data) {
        boops_err("Param info was not provided\n");
        return -EINVAL;
    }

    generate_random_uuid(g_boops_current_boot_uuid);
    boops = devm_kzalloc(dev, sizeof(*boops), GFP_KERNEL);
    if (!boops) {
        return -ENOMEM;
    }
    memset(boops, 0, sizeof(*boops));
    platform_set_drvdata(pdev, boops);

#ifdef CONFIG_MTD_UBI
    if (strcasecmp("ubi", dummy_data->storage_type) == 0) {
        boops->storage=boops_storage_ubi_create(pdata->ubi_device, pdata->ubi_volume);
        boops->oops_dump_delay=0x0FFFFFFF; /*  Mips platforms are slower than ARM so make this lower. This is very much a hack, hopefully temporary... */
    }
#endif
#ifdef CONFIG_MMC
    if (strcasecmp("mmc", dummy_data->storage_type) == 0) {
        boops->storage=boops_storage_mmc_create(pdata->mmc_block_first, pdata->mmc_block_count);
        boops->oops_dump_delay=0x7FFFFFFF; /*  Arm platforms are a bit fase so this is higher. This is very much a hack, hopefully temporary... */
    }
#endif
    if (!boops->storage) {
        boops_err("Failed to setup storage device\n");
        return -EINVAL;
    }
    storage=boops->storage;
    snprintf(g_storage_type, sizeof(g_storage_type), " %s", dummy_data->storage_type);

    if ((storage->block.count == BOOPS_INVALID) ||
        (storage->block.bytes == BOOPS_INVALID) ||
        (storage->block.bytes < BOOPS_MIN_BLOCK_SIZE) ||
        (storage->block.count < BOOPS_MIN_BLOCK_COUNT)) {
        boops_err("Invalid storage geometry Block[bytes=%u, count=%lu]\n", storage->block.bytes, storage->block.count);

        if (boops->storage->destroy)
            boops->storage->destroy(boops->storage);
        return -EINVAL;
    }

    if (storage->block.bytes < sizeof(struct boops_segment_header_t)) {
        boops_err("The segment header must be able to fit in a single block (block:%u header:%lu)\n", storage->block.bytes, sizeof(struct boops_segment_header_t));
        return -EINVAL;
    }

    if ((dummy_data->segment_bytes % storage->block.bytes) != 0) {
        uint32_t ns = dummy_data->segment_bytes - (dummy_data->segment_bytes % storage->block.bytes);
        boops_info("The segment size (%u) is not aligned to the block size (%u). Rounding down to %u\n", dummy_data->segment_bytes, storage->block.bytes, ns);
        dummy_data->segment_bytes = ns;
    }

    if (dummy_data->segment_bytes < storage->block.bytes || dummy_data->segment_bytes > (storage->block.bytes*storage->block.count)) {
        boops_err("The segment size (%u) must be larger than the block size (%u) and smaller than total bytes (%lu)\n", dummy_data->segment_bytes, storage->block.bytes, storage->block.bytes*storage->block.count);
        return -EINVAL;
    }

    boops->segment.bytes = dummy_data->segment_bytes;
    boops->segment.blocks = (boops->segment.bytes + (storage->block.bytes-1))  / storage->block.bytes; /* Calculate the blocks for the segment, rounding up */
    boops->segment.count = boops->storage->block.count / boops->segment.blocks;
    boops->block_buffer_size = boops->segment.count * boops->segment.bytes;

    if (boops->segment.bytes < sizeof(struct boops_primary_header_t)) {
        boops_err("The primary header must be able to fit in a single segment (segment:%u header:%lu)\n", boops->segment.bytes, sizeof(struct boops_primary_header_t));
        return -EINVAL;
    }

    boops_info("Created %s storage with %lu bytes block[bytes=%u, count=%lu] segment[blocks=%u, bytes=%u, count=%lu]\n",
            dummy_data->storage_type, boops->block_buffer_size, storage->block.bytes, storage->block.count,
            boops->segment.blocks, boops->segment.bytes, boops->segment.count);

    boops->block_buffer = devm_kzalloc(dev, boops->block_buffer_size, GFP_KERNEL);
    if (boops->block_buffer == NULL) {
        boops_info("Failed allocating segment buffer (size:%lu)\n", boops->block_buffer_size);
        return -EINVAL;
    }
    memset(boops->block_buffer, 0, boops->block_buffer_size);

    /* Register dump function with kmsger */
    boops->dump.dump = boops_capture_oops_cb;
    err = kmsg_dump_register(&boops->dump);
    if (err) {
        boops_info("Failed to register kmsg dumper\n");
        return -EINVAL;
    }

    boops->boops_proc = boops_proc_init(boops);
    boops_info("boops activated for current boot UUID: %pUB\n", g_boops_current_boot_uuid);
    return 0;
}


static int boops_remove(struct platform_device *pdev) {
    struct boops_t *boops = platform_get_drvdata(pdev);

    if (boops) {
        struct boops_storage_t *storage = boops->storage;

        if (boops->boops_proc)
            boops_proc_remove(boops->boops_proc);

        if (storage && storage->destroy) {
            boops_info("Destroying storage interface\n");
            storage->destroy(storage);
            storage=NULL;
        }

        if (kmsg_dump_unregister(&boops->dump) < 0)
            boops_err("Failed to unregister kmsg dumper\n");

        boops_info("boops disabled\n");
    }
    return 0;
}

static struct platform_driver boops_driver = {
    .remove = boops_remove,
    .probe  = boops_probe,
    .driver = {
    .name   = "boops",
    .owner  = THIS_MODULE,
    },
};

static int __init boops_init(void) {
    bool supported=false;
    if (platform_driver_probe(&boops_driver, boops_probe) == -ENODEV) {
        /*
        * If we didn't find a platform device, we use module parameters
        * building platform data on the fly.
        */
        if (dummy || dummy_data) {
            boops_err("boops is already running\n");
            return -EINVAL;
        }
#ifdef CONFIG_MTD_UBI
        if (strcasecmp("ubi", storage_type) == 0) {
            if (ubi_volume == BOOPS_INVALID || ubi_device == BOOPS_INVALID) {
                boops_err("For UBI storage a device and volume must be provided\n");
                return -EINVAL;
            }
            supported = true;
        }
#endif
#ifdef CONFIG_MMC
        if (strcasecmp("mmc", storage_type) == 0) {
            if (mmc_block_first == BOOPS_INVALID || mmc_block_count == BOOPS_INVALID) {
                boops_err("For MMC storage block_first and block_count must be provided\n");
                return -EINVAL;
            }
            supported = true;
        }
#endif
        if (!supported) {
            boops_err("Unsupported storage type '%s'\n", storage_type);
            return -EINVAL;
        }

        dummy_data = kzalloc(sizeof(*dummy_data), GFP_KERNEL);
        if (!dummy_data)
            return -ENOMEM;

        memset(dummy_data, 0, sizeof(*dummy_data));
        dummy_data->storage_type = storage_type;
        dummy_data->ubi_volume = ubi_volume;
        dummy_data->ubi_device = ubi_device;
        dummy_data->mmc_block_first = mmc_block_first;
        dummy_data->mmc_block_count = mmc_block_count;
        dummy_data->segment_bytes = segment_bytes;

        dummy = platform_create_bundle(&boops_driver, boops_probe,
                                        NULL, 0, dummy_data, sizeof(*dummy_data));
        if (IS_ERR(dummy)) {
            boops_err("Failed to create platform bundle\n");
            kfree(dummy_data);
            dummy_data = NULL;
            return PTR_ERR(dummy);
        }
    }
    return 0;
}

static void __exit boops_exit(void)
{
    if (dummy) {
        platform_device_unregister(dummy);
        platform_driver_unregister(&boops_driver);
        if (dummy_data) {
            kfree(dummy_data);
            dummy_data = NULL;
        }
        dummy = NULL;
    }
}

module_init(boops_init);
module_exit(boops_exit);

MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("BROADCOM");
MODULE_DESCRIPTION("Driver for Broadcom Kernel OOPS Capture");
