// SPDX-License-Identifier: GPL-2.0-only
/*
 * Copyright © 2014 NVIDIA Corporation
 * Copyright © 2015 Broadcom Corporation
 */

#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/slab.h>
#include <linux/soc/brcmstb/brcmstb.h>
#include <linux/sys_soc.h>

static u32 family_id;
static u32 product_id;
#ifdef CONFIG_BCM_KF_CM
#define MFG_REG_INFO "mfg-reg-info"
static u32 manufact_id;
static u32 manufact_bits;
static u32 manufact_mask;
static u32 manufact_shift;
static u32 cm_variant_id;
static u32 fab_id;
static u32 fab_bits;
static u32 fab_mask;
static u32 fab_shift;

enum {
	DT_MANUFACT_BITS_POS,
	DT_MANUFACT_MASK_POS,
	DT_MANUFACT_SHIFT_POS,
	DT_MANUFACT_COUNT,
};

enum {
	DT_FAB_ID_BITS_POS,
	DT_FAB_ID_MASK_POS,
	DT_FAB_ID_SHIFT_POS,
	DT_FAB_ID_COUNT,
};
#endif /* CONFIG_BCM_KF_CM */

u32 brcmstb_get_family_id(void)
{
	return family_id;
}
EXPORT_SYMBOL(brcmstb_get_family_id);

u32 brcmstb_get_product_id(void)
{
	return product_id;
}
EXPORT_SYMBOL(brcmstb_get_product_id);

#ifdef CONFIG_BCM_KF_CM
u32 brcmstb_get_manufact_id(void)
{
	return manufact_id;
}
EXPORT_SYMBOL(brcmstb_get_manufact_id);

u32 brcmstb_get_manufact_bits(void)
{
	return manufact_bits;
}
EXPORT_SYMBOL(brcmstb_get_manufact_bits);

u32 brcmstb_get_manufact_mask(void)
{
	return manufact_mask;
}
EXPORT_SYMBOL(brcmstb_get_manufact_mask);

u32 brcmstb_get_manufact_shift(void)
{
	return manufact_shift;
}
EXPORT_SYMBOL(brcmstb_get_manufact_shift);

u32 brcmcm_get_variant_id(void)
{
	return cm_variant_id;
}
EXPORT_SYMBOL(brcmcm_get_variant_id);

u32 brcmstb_get_fab_id(void)
{
	return fab_id;
}
EXPORT_SYMBOL(brcmstb_get_fab_id);

u32 brcmstb_get_fab_bits(void)
{
	return fab_bits;
}
EXPORT_SYMBOL(brcmstb_get_fab_bits);

u32 brcmstb_get_fab_mask(void)
{
	return fab_mask;
}
EXPORT_SYMBOL(brcmstb_get_fab_mask);

u32 brcmstb_get_fab_shift(void)
{
	return fab_shift;
}
EXPORT_SYMBOL(brcmstb_get_fab_shift);

static const struct of_device_id jtag_otp_match[] = {
	{ .compatible = "brcm,brcmstb-jtag-otp", },
	{ }
};

#endif /* CONFIG_BCM_KF_CM */

static const struct of_device_id sun_top_ctrl_match[] = {
	{ .compatible = "brcm,bcm7125-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7346-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7358-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7360-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7362-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7420-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7425-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7429-sun-top-ctrl", },
	{ .compatible = "brcm,bcm7435-sun-top-ctrl", },
	{ .compatible = "brcm,brcmstb-sun-top-ctrl", },
	{ }
};

static int __init brcmstb_soc_device_early_init(void)
{
	struct device_node *sun_top_ctrl;
	void __iomem *sun_top_ctrl_base;
	int ret = 0;

	/* We could be on a multi-platform kernel, don't make this fatal but
	 * bail out early
	 */
	sun_top_ctrl = of_find_matching_node(NULL, sun_top_ctrl_match);
	if (!sun_top_ctrl)
		return ret;

	sun_top_ctrl_base = of_iomap(sun_top_ctrl, 0);
	if (!sun_top_ctrl_base) {
		ret = -ENODEV;
		goto out;
	}

	family_id = readl(sun_top_ctrl_base);
	product_id = readl(sun_top_ctrl_base + 0x4);
	iounmap(sun_top_ctrl_base);
out:
	of_node_put(sun_top_ctrl);
	return ret;
}
early_initcall(brcmstb_soc_device_early_init);

static int __init brcmstb_soc_device_init(void)
{
	struct soc_device_attribute *soc_dev_attr;
#ifdef CONFIG_BCM_KF_CM
	struct device_node *fam_prod = NULL;
	void __iomem *fam_prod_base = NULL;
	struct device_node *jtag_otp = NULL;
	void __iomem *jtag_otp_base = NULL;
	unsigned int val[DT_MANUFACT_COUNT] = {0x20, 0xf000, 0xc};
	unsigned int fab_val[DT_FAB_ID_COUNT] = {0x28, 0x78000, 0xf};
#else
	struct device_node *sun_top_ctrl;
#endif /* CONFIG_BCM_KF_CM */
	struct soc_device *soc_dev;
	int ret = 0;

#ifdef CONFIG_BCM_KF_CM
	fam_prod = of_find_matching_node(NULL, sun_top_ctrl_match);
	if (fam_prod) {
		fam_prod_base = of_iomap(fam_prod, 0);
		if (!fam_prod_base) {
			ret = -ENODEV;
			goto out;
		}

		family_id = readl(fam_prod_base);
		product_id = readl(fam_prod_base + 0x4);
	} else {
		fam_prod = of_find_node_by_name(NULL, "bolt");
		if (!fam_prod) {
			ret = -ENODEV;
			goto out;
		}
		ret = of_property_read_u32(fam_prod, "family-id",
					   &family_id);
		if (ret)
			goto out;
		ret = of_property_read_u32(fam_prod, "product-id",
					   &product_id);
		if (ret)
			goto out;
	}
	pr_debug("family ID: 0x%08x\n", family_id);
	pr_debug("product ID: 0x%08x\n", product_id);

	jtag_otp = of_find_matching_node(NULL, jtag_otp_match);
	if (jtag_otp)
		jtag_otp_base = of_iomap(jtag_otp, 0);

	if (jtag_otp && jtag_otp_base) {
		if (of_property_read_u32_array(jtag_otp,
			MFG_REG_INFO, val, DT_MANUFACT_COUNT)) {
			pr_err("Missing %s property. Setting default value\n",
				   MFG_REG_INFO);
		}
		manufact_bits = readl(jtag_otp_base + val[DT_MANUFACT_BITS_POS]);
		if (of_machine_is_compatible("brcm,bcm3390") &&
				((family_id >> 16) == 0x3390))
			cm_variant_id = (readl(jtag_otp_base + 0x24) &
					GENMASK(25, 22)) >> 22;
	}
	manufact_mask = val[DT_MANUFACT_MASK_POS];
	manufact_shift = val[DT_MANUFACT_SHIFT_POS];
	manufact_id = (manufact_bits & manufact_mask) >> manufact_shift;

	/* Read the CHIP type for SMIC */
	if (jtag_otp && jtag_otp_base) {
		fab_bits = readl(jtag_otp_base + fab_val[DT_FAB_ID_BITS_POS]);
	}
	fab_mask = fab_val[DT_FAB_ID_MASK_POS];
	fab_shift = fab_val[DT_FAB_ID_SHIFT_POS];
	fab_id = (fab_bits & fab_mask) >> fab_shift;
#else
	/* We could be on a multi-platform kernel, don't make this fatal but
	 * bail out early
	 */
	sun_top_ctrl = of_find_matching_node(NULL, sun_top_ctrl_match);
	if (!sun_top_ctrl)
		return ret;
#endif /* CONFIG_BCM_KF_CM */

	soc_dev_attr = kzalloc(sizeof(*soc_dev_attr), GFP_KERNEL);
	if (!soc_dev_attr) {
		ret = -ENOMEM;
		goto out;
	}

	soc_dev_attr->family = kasprintf(GFP_KERNEL, "%x",
					 family_id >> 28 ?
					 family_id >> 16 : family_id >> 8);
	soc_dev_attr->soc_id = kasprintf(GFP_KERNEL, "%x",
					 product_id >> 28 ?
					 product_id >> 16 : product_id >> 8);
	soc_dev_attr->revision = kasprintf(GFP_KERNEL, "%c%d",
					 ((product_id & 0xf0) >> 4) + 'A',
					   product_id & 0xf);

	soc_dev = soc_device_register(soc_dev_attr);
	if (IS_ERR(soc_dev)) {
		kfree(soc_dev_attr->family);
		kfree(soc_dev_attr->soc_id);
		kfree(soc_dev_attr->revision);
		kfree(soc_dev_attr);
		ret = -ENOMEM;
	}
out:
#ifdef CONFIG_BCM_KF_CM
	if (fam_prod)
		of_node_put(fam_prod);
	if (fam_prod_base)
		iounmap(fam_prod_base);
#else
	of_node_put(sun_top_ctrl);
#endif /* CONFIG_BCM_KF_CM */
	return ret;
}
arch_initcall(brcmstb_soc_device_init);
