 /****************************************************************************
 *
 * Copyright (c) 2021-2025 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 ****************************************************************************/

#ifdef CONFIG_BCM_KF_CM

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/atomic.h>
#include <linux/time.h>
#include <linux/module.h>
#include <asm/smp.h>
#include <asm/cpu.h>
#include <linux/slab.h>
#include <asm/exception.h>
#include <linux/watchdog.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <asm/irq_regs.h>
#include <linux/kdebug.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/vmstat.h>
#include <linux/kmsg_dump.h>
#include <asm/ptrace.h>
#include <linux/printk.h>
#include <asm/delay.h>
#include <asm/stacktrace.h>
#include <asm/traps.h>
#include <linux/reboot.h>
#include <linux/kallsyms.h>
#include <linux/oom.h>
#include <linux/nmi.h>
#include <linux/seq_file.h>
#include <brcm_mbox.h>
#include <linux/brcmstb/sram_dump.h>
#include "sram_dump_priv.h"
#include <linux/time64.h>
#include <linux/panic_notifier.h>
#include <linux/blkdev.h>
#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>

/* This is only for driver debugging purpose. */
//#define SRAM_DEBUG

/*
 * For now, 764 bytes is not enough for adding secondary backtracing.
 * So disabling now, if we increase SRAM size, we can enable it later.
 */
//#define ENABLE_SECONDARY_CPU_BACKTRACE

#define AON_SRAM_PRIMARY_DUMP_BUF_SIZE		1024
#define AON_SRAM_SECONDARY_DUMP_BUF_SIZE	512
#define AON_SRAM_KDUMP_LINE_BUF_SIZE		1024
#define AON_SRAM_MAGIC_MARK			0xadefcdab

enum sram_crash_type_t {
	SRAM_CRASH_NONE = 0,
	SRAM_CRASH_NMI_WATCHDOG,
	SRAM_CRASH_DIE,
	SRAM_CRASH_PANIC,
	SRAM_CRASH_OOM,
	SRAM_CRASH_SOFTLOCKUP,
	SRAM_CRASH_HUNGTASK,
	SRAM_CRASH_RCU_STALL,
	SRAM_CRASH_EMERGENCY_RESTART,
	SRAM_CRASH_MAX
};

static char *sram_crash_type_str[SRAM_CRASH_MAX] = {
	"No crash",
	"NMI Watchdog",
	"Die",
	"Panic",
	"Out of memory",
	"Soft lockup",
	"Hung task",
	"RCU stall",
	"Emergency restart",
};

#define RPC_RESET_REASON_MAX 24
static char *rpc_reset_str[RPC_RESET_REASON_MAX] = {
	"SW Master Reset",
	"SMC Watchdog Timer",
	"B53 Watchdog Timer 0",
	"B53 Watchdog Timer 1",
	"B53 Watchdog Timer 2",
	"B53 Watchdog Timer 3",
	"Viper Watchdog Timer",
	"TPMI Watchdog Timer",
	"AVS Overtemp",
	"VDDC Under Voltage",
	"VDDC Over Voltage",
	"B53 Under Voltage",
	"B53 Over Voltage",
	"VDDM Under Voltage",
	"VDDM Over Voltage",
	"AVDD Under Voltage",
	"AVDD Over Voltage",
	"External Pin",
	"POR",
	"GFAP Watchdog Timer",
	"SMC SW Request",
	"GFAP SW Request",
	"B53 SW Request",
	"Viper SW Request",
};

struct sram_nmi_watchdog_info_t {
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	unsigned char pcomm_str[8];
	unsigned char pbt_str[320];
	unsigned char scomm_str[8];
	unsigned char sbt_str[128];
	unsigned char kdump_str[184+(AON_SRAM_KDUMP_LINE_BUF_SIZE-184)];
#else
	unsigned char pcomm_str[8];
	unsigned char pbt_str[512];
	unsigned char kdump_str[128+(AON_SRAM_KDUMP_LINE_BUF_SIZE-128)];
#endif
};

struct sram_die_info_t {
	struct pt_regs regs;
	long die_err;
	unsigned char die_str[16];
	unsigned char pcomm_str[8];
	unsigned char pbt_str[384];
	unsigned char kdump_str[256+(AON_SRAM_KDUMP_LINE_BUF_SIZE-256)];
};

struct sram_panic_info_t {
	unsigned char pcomm_str[8];
	unsigned char pbt_str[328];
	unsigned char kdump_str[384+(AON_SRAM_KDUMP_LINE_BUF_SIZE-384)];
};

struct sram_oom_info_t {
	unsigned int  max_rss_size;
	unsigned char max_rss_comm[8];
	unsigned char pcomm_str[8];
	unsigned char pbt_str[128];
	unsigned char kdump_str[572+(AON_SRAM_KDUMP_LINE_BUF_SIZE-572)];
};

struct sram_lockup_info_t {
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	unsigned char pcomm_str[8];
	unsigned char pbt_str[320];
	unsigned char scomm_str[8];
	unsigned char sbt_str[160];
	unsigned char kdump_str[224+(AON_SRAM_KDUMP_LINE_BUF_SIZE-224)];
#else
	unsigned char pcomm_str[8];
	unsigned char pbt_str[512];
	unsigned char kdump_str[200+(AON_SRAM_KDUMP_LINE_BUF_SIZE-200)];
#endif
};

struct sram_dump_t {
	unsigned int  magic;
	enum sram_crash_type_t crash_type;
	unsigned int  crash_cpu;
	unsigned int  crash_time;
	unsigned int  cm_mbox_event;
	unsigned int  restart_time;
	unsigned char restart_comm_str[8];
	unsigned char restart_parent_comm_str[8];
	unsigned int  ext_reserved;
	union {
		struct sram_nmi_watchdog_info_t wd;
		struct sram_die_info_t di;
		struct sram_panic_info_t pn;
		struct sram_oom_info_t om;
		struct sram_lockup_info_t lo;
	} us;
};

static struct sram_dump_t *sram_base_addr = NULL;
static struct sram_dump_t *sram_base_addr_prev_backup;
static volatile enum sram_crash_type_t sram_crash_type;
static volatile unsigned long sram_crash_dump_started;
static volatile unsigned long sram_crash_primary_cpu;
static volatile unsigned long sram_primary_backtrace_dump_started;
static char *sram_primary_dump_buf;
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
static volatile unsigned int   sram_secondary_backtrace_dump_started;
static char *sram_secondary_dump_buf;
#endif
static char *sram_kdump_line_buf;
static char *previous_reset_history_str = "Unknown";

#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
static int get_secondary_cpu_priv(int primary_cpu)
{
	return ((primary_cpu + 1) % NR_CPUS);
}
#endif

extern struct task_struct *find_lock_task_mm(struct task_struct *p);

static const char *sram_vf_name = "/dev/flash-debugo";
struct file *sram_vf_file = NULL;
struct block_device *sram_vf_bdev;
static int *sram_vf_buf;
#define SRAM_VF_BUF_SIZE 4096

static int sram_vfo(void)
{
	if (sram_vf_file)
		return 0;

	sram_vf_file = filp_open(sram_vf_name, O_RDWR, 0);
	if (IS_ERR(sram_vf_file)) {
		pr_err("sram_vfo: %s open errno %ld\n",
		       sram_vf_name, PTR_ERR(sram_vf_file));
		return -ENOENT;
	}
	sram_vf_bdev = I_BDEV(sram_vf_file->f_mapping->host);
	return 0;
}

static int sram_vfr(unsigned long long offset)
{
	int ret = 0;
	int data32 = 0;
	ret = kernel_read(sram_vf_file, &data32, 4, &offset);
	pr_debug("sram_vfr: %lld %x ret=%d\n", offset, data32, ret);
	return data32;
}

static int sram_vfw(int data32, unsigned long long offset)
{
	int ret = 0;
	ret = kernel_write(sram_vf_file, &data32, 4, &offset);
	pr_debug("sram_vfw: %lld %x ret=%d\n", offset, data32, ret);
	return ret;
}

inline int sram_vfc(void)
{
	if (sram_vf_file) {
		filp_close(sram_vf_file, NULL);
		sram_vf_file = NULL;
		sram_vf_bdev = 0;
	}
	return 0;
}

extern int vfbio_crash_write(struct block_device *bdev, u32 start_blk, void *buf,
		      u32 n_blks);

static int vfbio_fwrite_block(void)
{
	if (sram_vfo()) {
		pr_err("vfw_block: failed to open vfbio crash log file\n");
		return -EINVAL;
	}
#ifdef SRAM_DEBUG
	pr_err("sram_dump64: vfbio_fwrite_block\n");
#endif
	vfbio_crash_write(sram_vf_bdev, 0, sram_vf_buf, 1);
	return 0;
}

static int sram_open(void)
{
	BUILD_BUG_ON(sizeof(struct sram_dump_t) > SRAM_VF_BUF_SIZE);

	sram_vf_buf = kzalloc(SRAM_VF_BUF_SIZE, GFP_KERNEL);
	if (!sram_vf_buf) {
		pr_err("sram_open kzalloc failed!\n");
		return -ENOMEM;
	}
	sram_base_addr = (struct sram_dump_t *)sram_vf_buf;
	return 0;
}

static int sram_readl(unsigned int *src)
{
	pr_debug("sram_readl: %px %x\n", src, *src);
	return *src;
}

static int sram_writel(int data32, unsigned int *dst)
{
	int ret = 0;
	*dst = data32;
	pr_debug("sram_writel: %px %x\n", dst, data32);
	return ret;
}

static int sram_memread32(unsigned int *dst,
	unsigned int *src, int count) {
	int i;
	unsigned int data;

	if (!dst)
		return -1;

	for (i=0; i<count; i++) {
		data = sram_readl(src+i);
		dst[i] = data;
	}

	return 0;
}

static int sram_memwrite32(unsigned int *dst,
	unsigned int *src, int count) {
	int i;
	unsigned int  data;

	if (!src)
		return -1;

	for (i=0; i<count; i++) {
		data = src[i];
		sram_writel(data, dst+i);
	}

	return 0;
}

static int sram_memset32(unsigned int *dst, unsigned int data32,
	int count)
{
	int i;

	if (!dst)
		return -1;

	for (i=0; i<count; i++) {
		sram_writel(data32, dst+i);
	}

	return 0;
}

/*
 * AON SRAM doesn't have enough space to store the full text
 * So we remap 8bit char data to 6bit data. Some chars will be ignored.
 * Here is the mapping formula.
 *       0 ->     0 : NULL
 *      10 ->     1 : LF
 *   32~36 ->   2~6 : Special chars " !"#$%"
 *   40~64 ->  7~31 : Special chars and numbers "()*+,-./0123456789:;<=>?@"
 *   65~90 -> 37~62 : Upper case alphabet is remapped to lower case.
 *   91~95 -> 32~36 : Special chars "[\]^_"
 *  97~122 -> 37~62 : Lower case alphabet "abcdefghijklmnopqrstuvwxyz"
 *  others ->    63 : all not mapped chars will be fallen to this and skipped.
 */

/* convert_char_to_sram_char_priv: convert sram char to mem char */
static unsigned char convert_char_to_sram_char_priv(char inchar)
{
	unsigned char outchar;

	/* remap */
	switch (inchar) {
		case 0:
			outchar = 0;
			break;
		case 10:
			outchar = 1;
			break;
		case 32 ... 36:
			outchar = inchar-30;
			break;
		case 40 ... 64:
			outchar = inchar-33;
			break;
		case 65 ... 90:
			outchar = inchar-28;
			break;
		case 91 ... 95:
			outchar = inchar-59;
			break;
		case 97 ... 122:
			outchar = inchar-60;
			break;
		default:
			outchar = 63;
			break;
	}
	return outchar;
}

/* convert_sram_char_to_char_priv: convert mem char to sram char */
static char convert_sram_char_to_char_priv(unsigned inchar)
{
	unsigned char outchar;

	/* remap */
	switch (inchar) {
		case 0:
			outchar = 0;
			break;
		case 1:
			outchar = 10;
			break;
		case 2 ... 6:
			outchar = inchar+30;
			break;
		case 7 ... 31:
			outchar = inchar+33;
			break;
		case 32 ... 36:
			outchar = inchar+59;
			break;
		case 37 ... 62:
			outchar = inchar+60;
			break;
		default:
			/* This should not happen, just convert to '?' */
			outchar = '*';
			break;
	}
	return outchar;
}

/*
 * static strcpy_to_sram_priv: copy string to sram
 *   sram_dst should be aligned to 4
 *   actual the number of copied chars are depends on both of src/dst size
 *   return value is the number of copied mem chars, not aschar
 */
static size_t strcpy_to_sram_priv(unsigned char *sram_dst, char *mem_src,
	size_t dst_size, size_t src_size)
{
	unsigned int i, j;
	unsigned int  data;
	size_t src_long_size;
	size_t dst_long_size;
	size_t data_long_size;
	size_t copied_chars=0;

	/* Check input parameters */
	if (!sram_dst||!mem_src||!dst_size||!src_size) {
		pr_err("strcpy_to_sram_priv: error - invalid parameters, "
			"sram_dst:%px, mem_src:%px, "
			"dst_size:%ld, src_size:%ld\n",
		        sram_dst, mem_src,
			dst_size, src_size);
		return copied_chars;
	}
	if ((unsigned long)sram_dst%4) {
		pr_err("strcpy_to_sram_priv: error - unaligned sram_dst, "
			"sram_dst:%px, mem_src:%px, "
			"dst_size:%ld, src_size:%ld\n",
			sram_dst, mem_src,
			dst_size, src_size);
		return copied_chars;
	}

	/* Copy string to aon sram buffer */
	/* num_data is calculated based on minimum of dst_size*5 and src*4 */
	/* This is due to 4:5 mapping in char(8bit) and aschar(6bit) */
	dst_long_size = dst_size/4;
	src_long_size = (src_size+4)/5;
	data_long_size = min(src_long_size, dst_long_size);
	for (i=0; i<data_long_size; i++) {
		data = 0;
		for (j=0; j<5; j++) {
			if (i*5+j >= src_size) {
				copied_chars = src_size;
				break;
			}
			data += convert_char_to_sram_char_priv(mem_src[i*5+j])
				<< (6*j);
		}
		sram_writel(data, (unsigned int *)(sram_dst+i*4));
	}

	if (!copied_chars)
		copied_chars = data_long_size*5;

	return copied_chars;
}

/*
 * static strcpy_from_sram_priv: copy string from sram
 *   sram_src should be aligned to 4
 *   actual the number of copied chars are depends on both of src/dst size
 *   return value is the number of copied mem chars
 */
static size_t strcpy_from_sram_priv(char *mem_dst, unsigned char *sram_src,
	size_t dst_size, size_t src_size)
{
	unsigned int i, j;
	unsigned int  data;
	size_t src_long_size;
	size_t dst_long_size;
	size_t data_long_size;
	size_t copied_chars=0;

	/* Check input parameters */
	if (!mem_dst||!sram_src||!dst_size||!src_size) {
		pr_err("strcpy_from_sram_priv: error - invalid parameters, "
			"mem_dst:%px, sram_src:%px, "
			"dst_size:%ld, src_size:%ld\n",
			mem_dst, sram_src,
			dst_size, src_size);
		return copied_chars;
	}
	if ((unsigned long)sram_src%4) {
		pr_err("strcpy_from_sram_priv: error - unaligned sram_src, "
			"mem_dst:%px, sram_src:%px, "
			"dst_size:%ld, src_size:%ld\n",
			mem_dst, sram_src,
			dst_size, src_size);
		return copied_chars;
	}

	/* Copy string to mem buffer */
	/* num_data is calculated based on minimum of dst_size*5 and src*4 */
	/* This is due to 4:5 mapping in char(8bit) and aschar(6bit) */
	dst_long_size = (dst_size+4)/5;
	src_long_size = src_size/4;
	data_long_size = min(src_long_size, dst_long_size);
	for (i=0; i<data_long_size; i++) {
		data = sram_readl((unsigned int *)(sram_src+i*4));
		for (j=0; j<5; j++) {
			if (i*5+j >= dst_size) {
				copied_chars = dst_size;
				break;
			}
			mem_dst[i*5+j] =
				convert_sram_char_to_char_priv(data >> (6*j) &
					0x3f);
		}
	}

	if (!copied_chars)
		copied_chars = data_long_size*5;

	return copied_chars;
}

/* Disable/Enable printk should not be called across printk_nmi_enter/exit */
/* Disable/Enable printk should be called with preemption disabled */

/* Disable printk */
static int disable_printk_priv(void)
{
	int orig_suppress_printk;

	orig_suppress_printk = suppress_printk;
	suppress_printk = 1;

	return orig_suppress_printk;
}

/* Enable printk */
static void enable_printk_priv(int stored_printk)
{
	suppress_printk = stored_printk;
}

/* brcm_sram_store_crash_info_priv: store crash info to sram */
static void brcm_sram_store_crash_info_priv(void)
{
	struct sram_dump_t *base = sram_base_addr;
	struct timespec64 curtime;

	/* Store crash info */
	sram_writel(sram_crash_type, &base->crash_type);
	sram_writel(sram_crash_primary_cpu, &base->crash_cpu);
	ktime_get_real_ts64(&curtime);
        sram_writel((long)curtime.tv_sec, &base->crash_time);
}

/* brcm_sram_store_regs_priv: store regs to sram */
static void brcm_sram_store_regs_priv(struct pt_regs *regs)
{
	struct sram_dump_t *base = sram_base_addr;
	struct pt_regs *sram_regs;

	if (!regs)
		return;

	/* Store regs */
	if (sram_crash_type == SRAM_CRASH_DIE)
		sram_regs = &base->us.di.regs;
	else
		return;

	sram_memwrite32((unsigned int  *)sram_regs, (unsigned int  *)regs,
		sizeof(struct pt_regs)/4);
}

/* brcm_sram_store_kdump_priv: store kmsg to sram */
/* sram_primary_dump_buf is commonly used in backtrace and kmsg dump */
/* So we need to ensure this buffer is not used at the same time */
static void brcm_sram_store_kdump_priv(void)
{
	struct sram_dump_t *base = sram_base_addr;
	char *strbuf = sram_primary_dump_buf;
	char *linebuf = sram_kdump_line_buf;
	enum sram_crash_type_t type = sram_crash_type;
	unsigned char *sram_kdump_str;
	size_t sram_kdump_str_size;
	size_t strmax, len;
	struct kmsg_dump_iter iter;

	if (!strbuf || !linebuf)
		return;

	switch (type) {
	case SRAM_CRASH_NMI_WATCHDOG:
		sram_kdump_str = base->us.wd.kdump_str;
		sram_kdump_str_size = sizeof(base->us.wd.kdump_str);
		break;
	case SRAM_CRASH_DIE:
		sram_kdump_str = base->us.di.kdump_str;
		sram_kdump_str_size = sizeof(base->us.di.kdump_str);
		break;
	case SRAM_CRASH_PANIC:
	case SRAM_CRASH_EMERGENCY_RESTART:
		sram_kdump_str = base->us.pn.kdump_str;
		sram_kdump_str_size = sizeof(base->us.pn.kdump_str);
		break;
	case SRAM_CRASH_OOM:
		sram_kdump_str = base->us.om.kdump_str;
		sram_kdump_str_size = sizeof(base->us.om.kdump_str);
		break;
	case SRAM_CRASH_SOFTLOCKUP:
	case SRAM_CRASH_HUNGTASK:
	case SRAM_CRASH_RCU_STALL:
		sram_kdump_str = base->us.lo.kdump_str;
		sram_kdump_str_size = sizeof(base->us.lo.kdump_str);
		break;
	default:
		return;
	}

	/* Dump kmsg */
	strmax = sram_kdump_str_size*5/4;

	kmsg_dump_rewind(&iter);
	if (type == SRAM_CRASH_OOM)
		kmsg_dump_get_buffer(&iter, false, strbuf, 120, &len);
	kmsg_dump_get_buffer(&iter, false, strbuf, strmax, &len);
	strbuf[len] = 0;

	/* Save Kdump to sram */
	strbuf[strmax-1] = 0;
	strcpy_to_sram_priv(sram_kdump_str, strbuf, sram_kdump_str_size,
		strmax);
}

/* brcm_sram_store_oom_info_priv: store oom info to sram */
static void brcm_sram_store_oom_info_priv(void)
{
	struct sram_dump_t *base = sram_base_addr;
	struct task_struct *p;
	struct task_struct *task;
	struct task_struct *max_rss_task=NULL;
	unsigned int  max_rss_size = 0;
	unsigned int  cur_task_rss_size;

	/* Find the max RSS task */
	rcu_read_lock();
	for_each_process(p) {
		task = find_lock_task_mm(p);
		if (!task)
			continue;

		cur_task_rss_size = get_mm_rss(task->mm);
		if (cur_task_rss_size > max_rss_size) {
			max_rss_size = cur_task_rss_size;
			max_rss_task = task;
		}
		task_unlock(task);
	}

	/* Store max RSS process info */
	if (max_rss_task) {
		strcpy_to_sram_priv(base->us.om.max_rss_comm,
		max_rss_task->comm, sizeof(base->us.om.max_rss_comm),
		sizeof(max_rss_task->comm));
		sram_writel(max_rss_size, &base->us.om.max_rss_size);
	}

	rcu_read_unlock();
}

/*
 * bsbt: brcm_sram_dump_ipi_cpu_backtrace
 */
void bsbt(struct pt_regs *regs)
{
	//register unsigned int  current_sp asm ("sp");
	struct sram_dump_t *base = sram_base_addr;
	enum sram_crash_type_t type = sram_crash_type;
	int cpu = smp_processor_id();
	int pcpu = sram_crash_primary_cpu;
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	int scpu = get_secondary_cpu_priv(pcpu);
#endif
	char *strbuf;
	volatile unsigned long *dump_started;
	unsigned char *sram_comm_str, *sram_bt_str;
	size_t sram_comm_str_size, sram_bt_str_size;
	size_t strmax, len, offset;
	struct stackframe frame;
	int  prev_printk;

	/* check if there was a crash */
	if (type == SRAM_CRASH_NONE || type >= SRAM_CRASH_MAX)
		return;

	/* Set strbuf, dump_started addr */
	if (cpu == pcpu) {
		strbuf = sram_primary_dump_buf;
		dump_started = &sram_primary_backtrace_dump_started;
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	} else if (cpu == scpu) {
		strbuf = sram_secondary_dump_buf;
		dump_started = &sram_secondary_backtrace_dump_started;
#endif
	} else {
		return;
	}

	if (!strbuf)
		return;

	/* Skip backtrace if already started */
	if (test_and_set_bit(0, dump_started))
		return;

	switch (sram_crash_type) {
	case SRAM_CRASH_NMI_WATCHDOG:
		if (cpu == pcpu) {
			sram_comm_str = base->us.wd.pcomm_str;
			sram_comm_str_size = sizeof(base->us.wd.pcomm_str);
			sram_bt_str = base->us.wd.pbt_str;
			sram_bt_str_size = sizeof(base->us.wd.pbt_str);
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
		} else {
			sram_comm_str = base->us.wd.scomm_str;
			sram_comm_str_size = sizeof(base->us.wd.scomm_str);
			sram_bt_str = base->us.wd.sbt_str;
			sram_bt_str_size = sizeof(base->us.wd.sbt_str);
#endif
		}
		break;
	case SRAM_CRASH_DIE:
		if (cpu == pcpu) {
			sram_comm_str = base->us.di.pcomm_str;
			sram_comm_str_size = sizeof(base->us.di.pcomm_str);
			sram_bt_str = base->us.di.pbt_str;
			sram_bt_str_size = sizeof(base->us.di.pbt_str);
		}
		break;
	case SRAM_CRASH_PANIC:
	case SRAM_CRASH_EMERGENCY_RESTART:
		if (cpu == pcpu) {
			sram_comm_str = base->us.pn.pcomm_str;
			sram_comm_str_size = sizeof(base->us.pn.pcomm_str);
			sram_bt_str = base->us.pn.pbt_str;
			sram_bt_str_size = sizeof(base->us.pn.pbt_str);
		}
		break;
	case SRAM_CRASH_OOM:
		if (cpu == pcpu) {
			sram_comm_str = base->us.om.pcomm_str;
			sram_comm_str_size = sizeof(base->us.om.pcomm_str);
			sram_bt_str = base->us.om.pbt_str;
			sram_bt_str_size = sizeof(base->us.om.pbt_str);
		}
		break;
	case SRAM_CRASH_SOFTLOCKUP:
	case SRAM_CRASH_HUNGTASK:
	case SRAM_CRASH_RCU_STALL:
		if (cpu == pcpu) {
			sram_comm_str = base->us.lo.pcomm_str;
			sram_comm_str_size = sizeof(base->us.lo.pcomm_str);
			sram_bt_str = base->us.lo.pbt_str;
			sram_bt_str_size = sizeof(base->us.lo.pbt_str);
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
		} else {
			sram_comm_str = base->us.lo.scomm_str;
			sram_comm_str_size = sizeof(base->us.lo.scomm_str);
			sram_bt_str = base->us.lo.sbt_str;
			sram_bt_str_size = sizeof(base->us.lo.sbt_str);
#endif
		}
		break;
	default:
		return;
	}

	/* Disable printk */
	prev_printk = disable_printk_priv();

	/* Store current process info */
	strcpy_to_sram_priv(sram_comm_str, current->comm, sram_comm_str_size,
		sizeof(current->comm));

	if (regs) {
		start_backtrace(&frame, regs->regs[29], regs->pc);
	} else {
		start_backtrace(&frame,
				(unsigned long)__builtin_frame_address(0),
				(unsigned long)bsbt);
	}
	strmax = sram_bt_str_size*5/4;
	memset(strbuf, 0, strmax);
	offset = 0;
	while (1) {
		int urc;
		unsigned long where = frame.pc;

		urc = unwind_frame(NULL, &frame);
		if (urc < 0)
			break;
		/* copy symbol text to strbuf */
		len = sprint_symbol(strbuf+offset, where);
		offset += len;
		/* Add trailing '\n' */
		strbuf[offset++] = '\n';
		if (offset >= strmax)
			break;
	}
	strcpy_to_sram_priv(sram_bt_str, strbuf, sram_bt_str_size, strmax);

	/* re-enable printk */
	enable_printk_priv(prev_printk);
}

/* bswd: brcm_sram_dump_nmi_watchdog_callback */
void bswd(struct pt_regs *regs)
{
//	struct sram_dump_t *base = sram_base_addr;
	int cpu = smp_processor_id();
	int prev_printk;

#ifdef SRAM_DEBUG
	pr_err("bswd - start\n");
#endif

	/* Skip dumping if already started */
	if (test_and_set_bit(0, &sram_crash_dump_started)) {
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
		/* Wait until crash cpu/type is updated, without lock */
		udelay(1);
		/* Store secondary cpu backtrace */
		if (cpu == get_secondary_cpu_priv(sram_crash_primary_cpu))
			bsbt(regs);
#endif
		return;
	}

	/* Set crash type */
	sram_crash_primary_cpu = cpu;
	sram_crash_type = SRAM_CRASH_NMI_WATCHDOG;
	smp_mb();

	/* Disable printk */
	prev_printk = disable_printk_priv();

	/* Store primary cpu backtrace */
	bsbt(regs);

	/* Store regs */
	brcm_sram_store_regs_priv(regs);

	/* Store kdump */
	brcm_sram_store_kdump_priv();

	/* Store crash info */
	brcm_sram_store_crash_info_priv();

	/* re-enable printk */
	enable_printk_priv(prev_printk);

	vfbio_fwrite_block();
#ifdef SRAM_DEBUG
	pr_err("bswd - end\n");
#endif
}

/* bsc: brcm_sram_dump_common_callback: common internal only callback */
static void bsc(enum sram_crash_type_t type,
	struct pt_regs *regs)
{
//	struct sram_dump_t *base = sram_base_addr;
	int cpu;
	int prev_printk;

	/* Skip dumping if already started */
	if (test_and_set_bit(0, &sram_crash_dump_started)) {
		return;
	}

	/* Disable preemption and get CPU number */
	cpu = get_cpu();

	/* Set crash type */
	sram_crash_primary_cpu = cpu;
	sram_crash_type = type;
	smp_mb();

	/* Disable printk */
	prev_printk = disable_printk_priv();

	/* Store crash info */
	brcm_sram_store_crash_info_priv();

	/* Store regs */
	if (regs)
		brcm_sram_store_regs_priv(regs);

	/* store kdump */
	brcm_sram_store_kdump_priv();

	/* Store oom info */
	if (type == SRAM_CRASH_OOM)
		brcm_sram_store_oom_info_priv();

	/* Store backtrace */
	bsbt(regs);
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	trigger_allbutself_cpu_backtrace();
#endif

	/* re-enable printk */
	enable_printk_priv(prev_printk);

	vfbio_fwrite_block();

	/* Enable preemption */
	put_cpu();
}

/* bsd: brcm_sram_dump_die_callback */
static int bsd(struct notifier_block *nb,
	unsigned long reason, void *args)
{
	struct sram_dump_t *base = sram_base_addr;
	struct die_args *die_args = args;
	int len, strmax;

	/* save to sram */
#ifdef SRAM_DEBUG
	pr_err("bsd - start\n");
#endif
	base->us.di.die_err = die_args->err;
	len = strlen(die_args->str);
	strmax = sizeof(base->us.di.die_str)*5/4;
	if (len > strmax)
		len = strmax;
	strcpy_to_sram_priv(base->us.di.die_str, (char *) die_args->str,
			    sizeof(base->us.di.die_str), len);
	bsc(SRAM_CRASH_DIE, die_args->regs);
#ifdef SRAM_DEBUG
	pr_err("bsd - end\n");
#endif
	return NOTIFY_DONE;
}

/* bsp: int brcm_sram_dump_panic_callback */
static int bsp(struct notifier_block *nb,
	unsigned long reason, void *args)
{
	/* save to sram */
#ifdef SRAM_DEBUG
	pr_err("bsp - start\n");
#endif
	bsc(SRAM_CRASH_PANIC, NULL);
#ifdef SRAM_DEBUG
	pr_err("bsp - end\n");
#endif
	return NOTIFY_DONE;
}

/* bsr: brcm_sram_dump_reboot_callback_priv */
static int bsr(struct notifier_block *nb,
	unsigned long reason, void *args)
{
	struct sram_dump_t *base = sram_base_addr;
	struct timespec64 curtime;

#ifdef SRAM_DEBUG
	pr_err("bsr - start\n");
#endif
	/* Store current process info */
	strcpy_to_sram_priv(base->restart_comm_str, current->comm,
		sizeof(base->restart_comm_str), sizeof(current->comm));

	/* Save parent process info */
	if (current->parent)
		strcpy_to_sram_priv(base->restart_parent_comm_str,
			current->parent->comm,
			sizeof(base->restart_parent_comm_str),
			sizeof(current->parent->comm));

	/* Save reboot time */
	ktime_get_real_ts64(&curtime);
	sram_writel((long)curtime.tv_sec, &base->restart_time);

#ifdef SRAM_DEBUG
	pr_err("bsr - end\n");
#endif
	vfbio_fwrite_block();
	return NOTIFY_DONE;
}

/* bso: brcm_sram_dump_oom_callback */
static int bso(struct notifier_block *nb,
	unsigned long reason, void *args)
{
	/* save to sram */
#ifdef SRAM_DEBUG
	pr_err("bso - start\n");
#endif
	show_free_areas(0, NULL);
	bsc(SRAM_CRASH_OOM, NULL);
#ifdef SRAM_DEBUG
	pr_err("bso - end\n");
#endif
	return NOTIFY_DONE;
}

/* bssl: brcm_sram_dump_softlockup_callback */
void bssl(void)
{
#ifdef SRAM_DEBUG
	pr_err("bssl - start\n");
#endif
	/* save to sram */
	bsc(SRAM_CRASH_SOFTLOCKUP, NULL);
#ifdef SRAM_DEBUG
	pr_err("bssl - end\n");
#endif
}

/* bsht: brcm_sram_dump_hungtask_callback */
void bsht(void)
{
#ifdef SRAM_DEBUG
	pr_err("bsht - start\n");
#endif
	/* save to sram */
	bsc(SRAM_CRASH_HUNGTASK, NULL);
#ifdef SRAM_DEBUG
	pr_err("bsht - end\n");
#endif
}

/* bsrs: brcm_sram_dump_rcu_stall_callback */
void bsrs(void)
{
#ifdef SRAM_DEBUG
	pr_err("bsrs - start\n");
#endif
	/* save to sram */
	bsc(SRAM_CRASH_RCU_STALL, NULL);
#ifdef SRAM_DEBUG
	pr_err("bsrs - end\n");
#endif
}

/* bser: brcm_sram_dump_emergency_restart_callback */
void bser(void)
{
#ifdef SRAM_DEBUG
	pr_err("bser - start\n");
#endif
	/* save to sram */
	bsc(SRAM_CRASH_EMERGENCY_RESTART, NULL);
#ifdef SRAM_DEBUG
	pr_err("bser - end\n");
#endif
}

#if defined(CONFIG_BCM_MBOX)
/* bscm: brcm_sram_dump_cm_mbox_event_callback */
void bscm(unsigned long event)
{
	struct sram_dump_t *base = sram_base_addr;

	static unsigned int  history = 0;
	static unsigned int  old = 0;

#ifdef SRAM_DEBUG
	pr_err("bscm - start\n");
#endif
	if (STATE(old) == STATE(event))
		return;
	else
		old = STATE(event);

	history = (history<<8) + old;

	/* save to sram */
	sram_writel(history, &base->cm_mbox_event);

#ifdef SRAM_DEBUG
	pr_err("bscm - end\n");
#endif
}
#endif

static void brcm_sram_dump_crash_info_show_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	char *strbuf = sram_primary_dump_buf;
	enum sram_crash_type_t type;
	struct tm tmval;

	if (!s || !strbuf)
		return;

	type = base->crash_type;
	if (type == SRAM_CRASH_NONE) {
		/* No crash */
		seq_printf(s, "* Crash type: No crash\n");
		return;
	} else if (type >= SRAM_CRASH_MAX) {
		seq_printf(s, "* Crash type: invalid crash type [%d]\n",
			(int)type);
		return;
	}

	seq_printf(s, "* Crash type: %s\n", sram_crash_type_str[type]);
	time64_to_tm ((time64_t)base->crash_time, 0, &tmval);
	seq_printf(s, "* Crash date: %d-%d-%d %02d:%02d:%02d\n",
		(int)tmval.tm_year+1900, tmval.tm_mon+1, tmval.tm_mday,
		tmval.tm_hour, tmval.tm_min, tmval.tm_sec);
	seq_printf(s, "* Crash cpu: %u\n", base->crash_cpu);
}

static void brcm_sram_dump_crash_regs_info_show_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	enum sram_crash_type_t type;
	struct pt_regs *regs;
	char *strbuf = sram_primary_dump_buf;
	size_t strmax, len;
	int i;

	if (!s || !strbuf)
		return;

	type = base->crash_type;
	if (type == SRAM_CRASH_DIE)
		regs = &base->us.di.regs;
	else
		return;

	strmax = sizeof(base->us.di.die_str)*5/4;
	len = strcpy_from_sram_priv(strbuf, base->us.di.die_str,
		strmax, sizeof(base->us.di.die_str));
	strbuf[len] = 0;
	len = strlen(strbuf);
	seq_printf(s, "* Error msg: %s code: %016lx\n", (len>0)?strbuf:"NULL",
		   base->us.di.die_err);
	seq_printf(s, "* Regs:\n");
	seq_printf(s, "  pc : %016llx lr : %016llx sp : %016llx\n",
		   regs->pc, (u64)ptrauth_strip_insn_pac(regs->regs[30]), regs->sp);
	seq_printf(s, "  pmr_save: %08llx\n", regs->pmr_save);
	for (i = 29; i >= 2; i-=3) {
		seq_printf(s, "  x%-2d: %016llx x%-2d: %016llx x%-2d: %016llx\n",
			   i, regs->regs[i],
			   i-1, regs->regs[i-1],
			   i-2, regs->regs[i-2]);
	}
}

static void brcm_sram_dump_crash_backtrace_info_show_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	char *strbuf = sram_primary_dump_buf;
	int pcpu;
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	int scpu;
#endif
	size_t strmax, len;

	if (!s || !strbuf)
		return;

	pcpu = base->crash_cpu;
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	scpu = get_secondary_cpu_priv(pcpu);
#endif

	switch (base->crash_type) {
	case SRAM_CRASH_NMI_WATCHDOG:
		strmax = sizeof(base->us.wd.pcomm_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.wd.pcomm_str,
			strmax, sizeof(base->us.wd.pcomm_str));
		strbuf[len] = 0;
		len = strlen(strbuf);
		seq_printf(s, "* Process: [%s]\n", (len>0)?strbuf:"NULL");
		strmax = sizeof(base->us.wd.pbt_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.wd.pbt_str,
			strmax, sizeof(base->us.wd.pbt_str));
		strbuf[len] = 0;
		seq_printf(s, "* Backtrace:\n%s\n", strbuf);
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
		seq_printf(s, "* Secondary cpu: %d\n", scpu);
		strmax = sizeof(base->us.wd.scomm_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.wd.scomm_str,
			strmax, sizeof(base->us.wd.scomm_str));
		strbuf[len] = 0;
		len = strlen(strbuf);
		seq_printf(s, "* Process: [%s]\n", (len>0)?strbuf:"NULL");
		strmax = sizeof(base->us.wd.sbt_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.wd.sbt_str,
			strmax, sizeof(base->us.wd.sbt_str));
		strbuf[len] = 0;
		seq_printf(s, "* Backtrace:\n%s\n", strbuf);
#endif
		break;
	case SRAM_CRASH_DIE:
		strmax = sizeof(base->us.di.pcomm_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.di.pcomm_str,
			strmax, sizeof(base->us.di.pcomm_str));
		strbuf[len] = 0;
		len = strlen(strbuf);
		seq_printf(s, "* Process: [%s]\n", (len>0)?strbuf:"NULL");
		strmax = sizeof(base->us.di.pbt_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.di.pbt_str,
			strmax, sizeof(base->us.di.pbt_str));
		strbuf[len] = 0;
		seq_printf(s, "* Backtrace:\n%s\n", strbuf);
		break;
	case SRAM_CRASH_PANIC:
	case SRAM_CRASH_EMERGENCY_RESTART:
		strmax = sizeof(base->us.pn.pcomm_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.pn.pcomm_str,
			strmax, sizeof(base->us.pn.pcomm_str));
		strbuf[len] = 0;
		len = strlen(strbuf);
		seq_printf(s, "* Process: [%s]\n", (len>0)?strbuf:"NULL");
		strmax = sizeof(base->us.pn.pbt_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.pn.pbt_str,
			strmax, sizeof(base->us.pn.pbt_str));
		strbuf[len] = 0;
		seq_printf(s, "* Backtrace:\n%s\n", strbuf);
		break;
	case SRAM_CRASH_OOM:
		strmax = sizeof(base->us.om.pcomm_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.om.pcomm_str,
			strmax, sizeof(base->us.om.pcomm_str));
		strbuf[len] = 0;
		len = strlen(strbuf);
		seq_printf(s, "* Process: [%s]\n", (len>0)?strbuf:"NULL");
		strmax = sizeof(base->us.om.pbt_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.om.pbt_str,
			strmax, sizeof(base->us.om.pbt_str));
		strbuf[len] = 0;
		seq_printf(s, "* Backtrace:\n%s\n", strbuf);
		break;
	case SRAM_CRASH_SOFTLOCKUP:
	case SRAM_CRASH_HUNGTASK:
	case SRAM_CRASH_RCU_STALL:
		strmax = sizeof(base->us.lo.pcomm_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.lo.pcomm_str,
			strmax, sizeof(base->us.lo.pcomm_str));
		strbuf[len] = 0;
		len = strlen(strbuf);
		seq_printf(s, "* Process: [%s]\n", (len>0)?strbuf:"NULL");
		strmax = sizeof(base->us.lo.pbt_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.lo.pbt_str,
			strmax, sizeof(base->us.lo.pbt_str));
		strbuf[len] = 0;
		seq_printf(s, "* Backtrace:\n%s\n", strbuf);
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
		seq_printf(s, "* Secondary cpu: %d\n", scpu);
		strmax = sizeof(base->us.lo.scomm_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.lo.scomm_str,
			strmax, sizeof(base->us.lo.scomm_str));
		strbuf[len] = 0;
		len = strlen(strbuf);
		seq_printf(s, "* Process: [%s]\n", (len>0)?strbuf:"NULL");
		strmax = sizeof(base->us.lo.sbt_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.lo.sbt_str,
			strmax, sizeof(base->us.lo.sbt_str));
		strbuf[len] = 0;
		seq_printf(s, "* Backtrace:\n%s\n", strbuf);
#endif
		break;
	default:
		return;
	}
}

static void brcm_sram_dump_crash_kdump_info_show_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	char *strbuf = sram_primary_dump_buf;
	size_t strmax, len;

	if (!s || !strbuf)
		return;

	switch (base->crash_type) {
	case SRAM_CRASH_NMI_WATCHDOG:
		strmax = sizeof(base->us.wd.kdump_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.wd.kdump_str,
			strmax, sizeof(base->us.wd.kdump_str));
		strbuf[len] = 0;
		seq_printf(s, "* kdump:\n%s\n", strbuf);
		break;
	case SRAM_CRASH_DIE:
		strmax = sizeof(base->us.di.kdump_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.di.kdump_str,
			strmax, sizeof(base->us.di.kdump_str));
		strbuf[len] = 0;
		seq_printf(s, "* kdump:\n%s\n", strbuf);
		break;
	case SRAM_CRASH_PANIC:
	case SRAM_CRASH_EMERGENCY_RESTART:
		strmax = sizeof(base->us.pn.kdump_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.pn.kdump_str,
			strmax, sizeof(base->us.pn.kdump_str));
		strbuf[len] = 0;
		seq_printf(s, "* kdump:\n%s\n", strbuf);
		break;
	case SRAM_CRASH_OOM:
		strmax = sizeof(base->us.om.kdump_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.om.kdump_str,
			strmax, sizeof(base->us.om.kdump_str));
		strbuf[len] = 0;
		seq_printf(s, "* kdump:\n%s\n", strbuf);
		break;
	case SRAM_CRASH_SOFTLOCKUP:
	case SRAM_CRASH_HUNGTASK:
	case SRAM_CRASH_RCU_STALL:
		strmax = sizeof(base->us.lo.kdump_str)*5/4;
		len = strcpy_from_sram_priv(strbuf, base->us.lo.kdump_str,
			strmax, sizeof(base->us.lo.kdump_str));
		strbuf[len] = 0;
		seq_printf(s, "* kdump:\n%s\n", strbuf);
		break;
	default:
		return;
	}
}

static void brcm_sram_dump_oom_info_show_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	char *strbuf = sram_primary_dump_buf;
	size_t strmax, len;

	if (!s || !strbuf)
		return;

	if (base->crash_type != SRAM_CRASH_OOM)
		return;

	strmax = sizeof(base->us.om.max_rss_comm)*5/4;
	len = strcpy_from_sram_priv(strbuf, base->us.om.max_rss_comm,
		strmax, sizeof(base->us.om.max_rss_comm));
	strbuf[len] = 0;
	len = strlen(strbuf);
	seq_printf(s, "* Max rss process: [%s]\n", (len > 0) ? strbuf : "NULL");
	seq_printf(s, "* Max rss size: %u kB\n", base->us.om.max_rss_size * 4);
}

#if defined(CONFIG_BCM_MBOX)
static void brcm_sram_dump_cm_mbox_event_info_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	unsigned int  history;
	enum cm_state state;

	int i;

	if (!s)
		return;

	history = base->cm_mbox_event;
	seq_printf(s, "* Last 4 cm mbox event:\n  ");
	for (i=3; i>0; i--) {
		state = STATE(history >> i*8);
		seq_printf(s, "[%s] -> ", cmstate_to_string(state));
	}
	state = STATE(history);
	seq_printf(s, "[%s]\n", cmstate_to_string(state));
}
#endif

static void brcm_sram_dump_reboot_event_info_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	char comm[24], parent_comm[24];
	struct tm tmval;
	size_t strmax, comm_len, parent_comm_len;

	if (!s)
		return;

	if (base->restart_time > 10) {
		strmax = sizeof(base->restart_comm_str)*5/4;
		comm_len = strcpy_from_sram_priv(comm, base->restart_comm_str,
			strmax, sizeof(base->restart_comm_str));
		comm[comm_len] = 0;
		comm_len = strlen(comm);
		strmax = sizeof(base->restart_parent_comm_str)*5/4;
		parent_comm_len = strcpy_from_sram_priv(parent_comm,
			base->restart_parent_comm_str,
			strmax, sizeof(base->restart_parent_comm_str));
		parent_comm[parent_comm_len] = 0;
		parent_comm_len = strlen(parent_comm);
		seq_printf(s, "* Who called reboot: [%s] -> [%s]\n",
			(parent_comm_len > 0) ? parent_comm : "NULL",
			(comm_len > 0) ? comm : "NULL");

		time64_to_tm ((time64_t)base->restart_time, 0, &tmval);
		seq_printf(s, "* Reboot request date: "
			"%d-%d-%d %02d:%02d:%02d\n",
			(int)tmval.tm_year+1900, tmval.tm_mon+1, tmval.tm_mday,
			tmval.tm_hour, tmval.tm_min, tmval.tm_sec);
	}
}

static int brcm_sram_dump_show_common_priv(struct seq_file *s,
	struct sram_dump_t *base)
{
	if (!s)
		return -EINVAL;

#ifdef SRAM_DEBUG1
	{
		int i;
		unsigned int  *addr = (unsigned int  *)base;
		seq_printf(s, "======== SRAM DEBUG start ========\n");
		for (i=0; i<sizeof(struct sram_dump_t)/4; i++) {
			seq_printf(s, "[0x%08llx]: 0x%08x\n",
				(unsigned long long)(addr+i),
				(unsigned int )(addr[i]));
		}
		seq_printf(s, "========= SRAM DEBUG end =========\n");
	}
#endif
	seq_printf(s, "======== Broadcom SRAM debug dump start ========\n");
	seq_printf(s, "* HW reset cause: %s\n",
		previous_reset_history_str ?
		previous_reset_history_str : "unknown");
	if (base->magic == AON_SRAM_MAGIC_MARK) {
		if (base->crash_type > SRAM_CRASH_NONE &&
			base->crash_type < SRAM_CRASH_MAX) {
			seq_printf(s, "* Kernel crash detected: Yes\n");
		} else {
			seq_printf(s, "* Kernel crash detected: No\n");
		}
		brcm_sram_dump_reboot_event_info_priv(s, base);
		brcm_sram_dump_crash_info_show_priv(s, base);
		brcm_sram_dump_crash_regs_info_show_priv(s, base);
		brcm_sram_dump_crash_backtrace_info_show_priv(s, base);
		brcm_sram_dump_crash_kdump_info_show_priv(s, base);
		brcm_sram_dump_oom_info_show_priv(s, base);
#if defined(CONFIG_BCM_MBOX)
		brcm_sram_dump_cm_mbox_event_info_priv(s, base);
#endif
		seq_printf(s, "* Extra status: 0x%08x\n", base->ext_reserved);
	}
	seq_printf(s, "======== Broadcom SRAM debug dump end ==========\n");

	return 0;
}

int brcm_sram_dump_show_prev_priv(struct seq_file *s)
{
	int i;
	int *base = (unsigned int *)sram_base_addr_prev_backup;

	if (sram_base_addr_prev_backup->magic != AON_SRAM_MAGIC_MARK) {
		/* Backup previous life's SRAM data */
		if (sram_vfo()) {
			pr_err("brcm_sram_dump_show_now_priv: failed to open vfbio crash log file\n");
			return -EINVAL;
		}
		for (i=0; i<sizeof(struct sram_dump_t)/4; i++)
			base[i] = sram_vfr(i*4);
		for (i=0; i<sizeof(struct sram_dump_t)/4; i++)
			sram_vfw(0, i*4);
		sram_vfc();
	}
	return brcm_sram_dump_show_common_priv(s, sram_base_addr_prev_backup);
}

int brcm_sram_dump_show_now_priv(struct seq_file *s)
{
	struct sram_dump_t *base;
	int ret;

	base = (struct sram_dump_t *)kzalloc(sizeof(struct sram_dump_t),
		GFP_KERNEL);
	if (!base) {
		pr_err("brcm_sram_dump_show_now_priv: failed to kmalloc\n");
		return -ENOMEM;
	}
	sram_memread32((unsigned int  *)base, (unsigned int  *)sram_base_addr,
		sizeof(struct sram_dump_t)/4);

	ret = brcm_sram_dump_show_common_priv(s, base);

	kfree(base);

	return ret;
}

void brcm_sram_dump_clear_priv(void)
{
	/* Clear AON SRAM */
	sram_memset32((unsigned int  *)sram_base_addr, 0x0,
		sizeof(struct sram_dump_t)/4);
	sram_writel(AON_SRAM_MAGIC_MARK, &sram_base_addr->magic);

	/* Clear internal static variables */
	sram_crash_type = SRAM_CRASH_NONE;
	sram_crash_primary_cpu = 0;
	clear_bit(0, &sram_crash_dump_started);
	clear_bit(0, &sram_primary_backtrace_dump_started);
#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	clear_bit(0, &sram_secondary_backtrace_dump_started);
#endif
	mb();
}

static struct notifier_block brcm_sram_dump_die_notifier = {
	.notifier_call = bsd,
	.priority = 0xffff,
};

static struct notifier_block brcm_sram_dump_panic_notifier = {
	.notifier_call = bsp,
	.priority = 0xffff,
};

static struct notifier_block brcm_sram_dump_reboot_notifier = {
	.notifier_call = bsr,
	.priority = 0xffff,
};

static struct notifier_block brcm_sram_dump_oom_notifier = {
	.notifier_call = bso,
	.priority = 0xffff,
};

#if defined(CONFIG_BCM_MBOX)
static int nbrcm_ba_cm_mbox_event(
		struct notifier_block *this,
		unsigned int  event, void *ptr)
{
	struct brcm_mbox_info *states = ptr;

	switch (event) {
	case MBOX_CHANGE_EVENT:
		bscm(states->mbox[MBOX_CM]);
		break;
	default:
		break;
	}
	return 0;
}

static struct notifier_block  brcm_sram_dump_cm_mbox_notifier = {
	.notifier_call  = nbrcm_ba_cm_mbox_event,
	.priority = 0xffff,
};
#endif

#define RPC_FUNCTION_GET_RESET_REASON 33
#define	RPC_DQM_TIMEOUT			10000	/* milliseconds */
unsigned int rpc_sys_get_reset_reson(char *rpcname)
{
	rpc_msg msg;
	int status;
	int tunnel;

	tunnel = rpc_get_fifo_tunnel_id(rpcname);
	if (tunnel < 0) {
		pr_err("%s: Unable to obtain RPC tunnel ID.\n",
			__func__);
		return -1;
	}

	memset(&msg, 0, sizeof(msg));
	rpc_msg_init(&msg, RPC_SERVICE_SYS,
		RPC_FUNCTION_GET_RESET_REASON, 0, 0, 0, 0);
	status = rpc_send_request_timeout(tunnel, &msg,
		RPC_DQM_TIMEOUT / 1000);
	if (status < 0)
		return 0;
	return (unsigned int) (msg.data[0] & 0xff);
}

static int __init brcm_sram_dump_init(void)
{
	struct device_node *node;
	char *rpcrgsmc_name;
	int ret;

	ret = sram_open();
	if (ret)
		return ret;

	/* Backup previous life's SRAM data */
	sram_base_addr_prev_backup = (struct sram_dump_t *)kzalloc(
		sizeof(struct sram_dump_t), GFP_KERNEL);
	if (!sram_base_addr_prev_backup) {
		pr_err("sram_base_addr_prev_backup devm_kmalloc failed!\n");
		return -ENOMEM;
	}

	/* Allocate primary/secondary/line dump buffer memory */
	sram_primary_dump_buf = (char *)kmalloc(
		AON_SRAM_PRIMARY_DUMP_BUF_SIZE, GFP_KERNEL);
	if (!sram_primary_dump_buf) {
		pr_err("sram_primary_dump_buf devm_kmalloc failed!\n");
		return -ENOMEM;
	}

#ifdef ENABLE_SECONDARY_CPU_BACKTRACE
	sram_secondary_dump_buf = (char *)kmalloc(
		AON_SRAM_SECONDARY_DUMP_BUF_SIZE, GFP_KERNEL);
	if (!sram_secondary_dump_buf) {
		pr_err("sram_secondary_dump_buf devm_kmalloc failed!\n");
		return -ENOMEM;
	}
#endif
	sram_kdump_line_buf = (char *)kmalloc(
		AON_SRAM_KDUMP_LINE_BUF_SIZE, GFP_KERNEL);
	if (!sram_kdump_line_buf) {
		pr_err("sram_kdump_line_buf devm_kmalloc failed!\n");
		return -ENOMEM;
	}

	/* Reset SRAM to the default clear state, since we have the backup */
	brcm_sram_dump_clear_priv();

	/* Register proc interface */
	brcm_sram_dump_proc_init();

	/* Register crash notifiers */
#if defined(CONFIG_BCM_MBOX)
	brcm_mbox_register_notifier(&brcm_sram_dump_cm_mbox_notifier);
#endif
	register_reboot_notifier(&brcm_sram_dump_reboot_notifier);
	register_die_notifier(&brcm_sram_dump_die_notifier);
	register_oom_notifier(&brcm_sram_dump_oom_notifier);
	atomic_notifier_chain_register(&panic_notifier_list,
		&brcm_sram_dump_panic_notifier);

	/* Save the previous reset history */
	node = of_find_node_by_name(NULL, "rpc");
	if (!node) {
		pr_err("rpc device-tree node was not found\n");
		goto done;
	}
	node = of_find_node_by_name(node, "rpcrgsmc");
	if (!node) {
		pr_err("rpcrgsmc device-tree node was not found\n");
		goto done;
	}
	if (of_property_read_string(node, "dev-name",
				    (const char **)&rpcrgsmc_name)) {
		pr_err("reset-list property was not found\n");
		goto done;
	}
	ret = rpc_sys_get_reset_reson(rpcrgsmc_name);
	if (ret >= 0)
		previous_reset_history_str = rpc_reset_str[ret];
done:
	/* Done */
	pr_info("brcm_sram_dump_init\n");
	return 0;
}

static void __exit brcm_sram_dump_remove(void)
{
	/* Clear internal static variables */
	brcm_sram_dump_clear_priv();
	sram_base_addr = NULL;

	/* Unregister proc interface */
	brcm_sram_dump_proc_exit();

	/* Unregister crash notifiers */
#if defined(CONFIG_BCM_MBOX)
	brcm_mbox_unregister_notifier(&brcm_sram_dump_cm_mbox_notifier);
#endif
	unregister_reboot_notifier(&brcm_sram_dump_reboot_notifier);
	unregister_die_notifier(&brcm_sram_dump_die_notifier);
	unregister_oom_notifier(&brcm_sram_dump_oom_notifier);
	atomic_notifier_chain_unregister(&panic_notifier_list,
		&brcm_sram_dump_panic_notifier);

	/* Done */
	pr_info("brcm_sram_dump_remove\n");
}

late_initcall_sync(brcm_sram_dump_init);
module_exit(brcm_sram_dump_remove);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jayesh Patel <jayesh.patel@broadcom.com>");
MODULE_AUTHOR("Changseob Kim <changseob.kim@broadcom.com>");

#endif /* CONFIG_BCM_KF_CM */
