/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Copyright (C) 2012 ARM Ltd.
 */
#ifndef __ASM_IRQFLAGS_H
#define __ASM_IRQFLAGS_H

#include <asm/alternative.h>
#include <asm/barrier.h>
#include <asm/ptrace.h>
#include <asm/sysreg.h>
#ifdef CONFIG_ARM64_GIC_CPUIF_MEMREGS
#include <linux/cache.h>

extern void __cacheline_aligned __iomem *gicc_pmr;

#define write_memreg_s(_v, _r) do {										\
	asm volatile("str %w0, [%1]" : : "rZ" (_v), "r" (_r) : "memory");	\
} while (0)

#define read_memreg_s(_r) ({											\
	u32 _v;																\
	asm volatile("ldr %w0, [%1]" : "=&r" (_v) : "r" (_r) : "memory");	\
	_v;																	\
})
#endif

/*
 * Aarch64 has flags for masking: Debug, Asynchronous (serror), Interrupts and
 * FIQ exceptions, in the 'daif' register. We mask and unmask them in 'daif'
 * order:
 * Masking debug exceptions causes all other exceptions to be masked too/
 * Masking SError masks IRQ/FIQ, but not debug exceptions. IRQ and FIQ are
 * always masked and unmasked together, and have no side effects for other
 * flags. Keeping to this order makes it easier for entry.S to know which
 * exceptions should be unmasked.
 */

/*
 * CPU interrupt mask handling.
 */
#ifdef CONFIG_ARM64_GIC_CPUIF_MEMREGS
static __always_inline void __daif_local_irq_enable(void)
{
	barrier();
	asm volatile("msr daifclr, #3");
	barrier();
}

static __always_inline void __sys_pmr_local_irq_enable(void)
{
	if (system_has_prio_mask_debugging()) {
		u32 pmr = read_sysreg_s(SYS_ICC_PMR_EL1);
		WARN_ON_ONCE(pmr != GIC_PRIO_IRQON && pmr != GIC_PRIO_IRQOFF);
	}

	barrier();
	write_sysreg_s(GIC_PRIO_IRQON, SYS_ICC_PMR_EL1);
	pmr_sync();
	barrier();
}

static __always_inline void __mem_pmr_local_irq_enable(void)
{
	if (system_has_prio_mask_debugging()) {
		u32 pmr = read_memreg_s(gicc_pmr);
		WARN_ON_ONCE(pmr != GIC_PRIO_IRQON && pmr != GIC_PRIO_IRQOFF);
	}

	write_memreg_s(GIC_PRIO_IRQON, gicc_pmr);
}

static inline void arch_local_irq_enable(void)
{
	if (!system_uses_irq_prio_masking()) {
		__daif_local_irq_enable();
	} else if (system_supports_gic_cpuif_sysregs()) {
		__sys_pmr_local_irq_enable();
	} else {
		__mem_pmr_local_irq_enable();
	}
}
#else
static inline void arch_local_irq_enable(void)
{
	if (system_has_prio_mask_debugging()) {
		u32 pmr = read_sysreg_s(SYS_ICC_PMR_EL1);

		WARN_ON_ONCE(pmr != GIC_PRIO_IRQON && pmr != GIC_PRIO_IRQOFF);
	}

	asm volatile(ALTERNATIVE(
		"msr	daifclr, #3		// arch_local_irq_enable",
		__msr_s(SYS_ICC_PMR_EL1, "%0"),
		ARM64_HAS_IRQ_PRIO_MASKING)
		:
		: "r" ((unsigned long) GIC_PRIO_IRQON)
		: "memory");

	pmr_sync();
}
#endif

#ifdef CONFIG_ARM64_GIC_CPUIF_MEMREGS
static __always_inline void __daif_local_irq_disable(void)
{
	barrier();
	asm volatile("msr daifset, #3");
	barrier();
}

static __always_inline void __sys_pmr_local_irq_disable(void)
{
	if (system_has_prio_mask_debugging()) {
		u32 pmr = read_sysreg_s(SYS_ICC_PMR_EL1);
		WARN_ON_ONCE(pmr != GIC_PRIO_IRQON && pmr != GIC_PRIO_IRQOFF);
	}

	barrier();
	write_sysreg_s(GIC_PRIO_IRQOFF, SYS_ICC_PMR_EL1);
	barrier();
}

static __always_inline void __mem_pmr_local_irq_disable(void)
{
	if (system_has_prio_mask_debugging()) {
		u32 pmr = read_memreg_s(gicc_pmr);
		WARN_ON_ONCE(pmr != GIC_PRIO_IRQON && pmr != GIC_PRIO_IRQOFF);
	}

	write_memreg_s(GIC_PRIO_IRQOFF, gicc_pmr);
}

static inline void arch_local_irq_disable(void)
{
	if (!system_uses_irq_prio_masking()) {
		__daif_local_irq_disable();
	} else if (system_supports_gic_cpuif_sysregs()) {
		__sys_pmr_local_irq_disable();
	} else {
		__mem_pmr_local_irq_disable();
	}
}
#else
static inline void arch_local_irq_disable(void)
{
	if (system_has_prio_mask_debugging()) {
		u32 pmr = read_sysreg_s(SYS_ICC_PMR_EL1);

		WARN_ON_ONCE(pmr != GIC_PRIO_IRQON && pmr != GIC_PRIO_IRQOFF);
	}

	asm volatile(ALTERNATIVE(
		"msr	daifset, #3		// arch_local_irq_disable",
		__msr_s(SYS_ICC_PMR_EL1, "%0"),
		ARM64_HAS_IRQ_PRIO_MASKING)
		:
		: "r" ((unsigned long) GIC_PRIO_IRQOFF)
		: "memory");
}
#endif

/*
 * Save the current interrupt enable state.
 */
#ifdef CONFIG_ARM64_GIC_CPUIF_MEMREGS
static inline unsigned long arch_local_save_flags(void)
{
	unsigned long flags = 0;

	if (!system_uses_irq_prio_masking()) {
		flags = read_sysreg(daif);
	} else if (system_supports_gic_cpuif_sysregs()) {
		flags = read_sysreg_s(SYS_ICC_PMR_EL1);
	} else {
		flags = read_memreg_s(gicc_pmr);
	}

	return flags;
}
#else
static inline unsigned long arch_local_save_flags(void)
{
	unsigned long flags;

	asm volatile(ALTERNATIVE(
		"mrs	%0, daif",
		__mrs_s("%0", SYS_ICC_PMR_EL1),
		ARM64_HAS_IRQ_PRIO_MASKING)
		: "=&r" (flags)
		:
		: "memory");

	return flags;
}
#endif

static inline int arch_irqs_disabled_flags(unsigned long flags)
{
	int res;

	asm volatile(ALTERNATIVE(
		"and	%w0, %w1, #" __stringify(PSR_I_BIT),
		"eor	%w0, %w1, #" __stringify(GIC_PRIO_IRQON),
		ARM64_HAS_IRQ_PRIO_MASKING)
		: "=&r" (res)
		: "r" ((int) flags)
		: "memory");

	return res;
}

static inline int arch_irqs_disabled(void)
{
	return arch_irqs_disabled_flags(arch_local_save_flags());
}

static inline unsigned long arch_local_irq_save(void)
{
	unsigned long flags;

	flags = arch_local_save_flags();

	/*
	 * There are too many states with IRQs disabled, just keep the current
	 * state if interrupts are already disabled/masked.
	 */
	if (!arch_irqs_disabled_flags(flags))
		arch_local_irq_disable();

	return flags;
}

/*
 * restore saved IRQ state
 */
#ifdef CONFIG_ARM64_GIC_CPUIF_MEMREGS
static __always_inline void __daif_local_irq_restore(unsigned long flags)
{
	barrier();
	write_sysreg(flags, daif);
	barrier();
}

static __always_inline void __sys_pmr_local_irq_restore(unsigned long flags)
{
	barrier();
	write_sysreg_s(flags, SYS_ICC_PMR_EL1);
	pmr_sync();
	barrier();
}

static __always_inline void __mem_pmr_local_irq_restore(unsigned long flags)
{
	write_memreg_s(flags, gicc_pmr);
}

static inline void arch_local_irq_restore(unsigned long flags)
{
	if (!system_uses_irq_prio_masking()) {
		__daif_local_irq_restore(flags);
	} else if (system_supports_gic_cpuif_sysregs()) {
		__sys_pmr_local_irq_restore(flags);
	} else {
		__mem_pmr_local_irq_restore(flags);
	}
}
#else
static inline void arch_local_irq_restore(unsigned long flags)
{
	asm volatile(ALTERNATIVE(
		"msr	daif, %0",
		__msr_s(SYS_ICC_PMR_EL1, "%0"),
		ARM64_HAS_IRQ_PRIO_MASKING)
		:
		: "r" (flags)
		: "memory");

	pmr_sync();
}
#endif

#endif /* __ASM_IRQFLAGS_H */
