 /****************************************************************************
 *
 * Copyright (c) 2015 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/kernel.h>
#include <linux/spi/spi.h>
#include <linux/delay.h>
#include <asm/byteorder.h>

#include "ethsw.h"
#include "ethsw_priv.h"

/*
   This file implements functionality required by SPI interface to the
   switch. It is required on 3384 since the switch is not an internal module on
   this chip.
*/


static DEFINE_MUTEX(spi_mutex);

/* define log module */
#define LOG_MODULE "ethsw_spi"

typedef struct {
	u8 opcode;
	u8 addr;
} ethsw_cmd;

#define NORMAL_CMD_BYTE_READ  0x60
#define NORMAL_CMD_BYTE_WRITE 0x61
#define SPI_REGS_PAGE  0x00
#define SPI_PAGE_REG   0xFF
#define SPI_STATUS_REG 0xFE
#define SPI_DATA_REG   0xF0
#define SPI_STATUS_REG_SPIF 7
#define SPI_STATUS_REG_RACK 5

#define MAX_SPI_RW_SIZE 8

#define CHECK_BIT(var,pos) ((var) & (1<<(pos)))


int ethsw_spi_read_one(u8 reg_addr, void *reg_cont, u8 reg_size)
{
	int ret = 0;
	ethsw_cmd cmd;

	/* Issue read command*/
	cmd.opcode = NORMAL_CMD_BYTE_READ;
	cmd.addr = reg_addr;
	ret = spi_write_then_read(ethsw_dev[EXTERNAL_SPI_SW]->sdev, &cmd, sizeof(cmd), reg_cont, reg_size);
	if(ret)
		LOG_ERR("SPI reg 0x%x failed", reg_addr);

	return ret;
}

int ethsw_spi_write_one(u8 reg_addr, void *reg_cont, u8 reg_size)
{
	int ret = 0;
	u8 buf[sizeof(ethsw_cmd) + MAX_SPI_RW_SIZE]; // Maximum reg_size 8 bytes
	u32 buf_size = sizeof(ethsw_cmd) + reg_size;
	ethsw_cmd *cmd = (ethsw_cmd *)buf;
	void *data = (u8 *)buf + sizeof(ethsw_cmd);

	if(reg_size > MAX_SPI_RW_SIZE)
		return -EINVAL;

	/* Issue read command*/
	cmd->opcode = NORMAL_CMD_BYTE_WRITE;
	cmd->addr = reg_addr;
	memcpy(data, reg_cont, reg_size);
	spi_write(ethsw_dev[EXTERNAL_SPI_SW]->sdev, buf, buf_size);
	if(ret)
		LOG_ERR("SPI reg 0x%x failed", reg_addr);

	return ret;
}


#define SPI_MAX_RETRY 10

int ethsw_spi_test_status_reg_spif(void)
{
	int ret = 0;
	u8  reg_cont;
	int i = SPI_MAX_RETRY;

	// check SPIF bit 7==0
	while(1) {
		ret = ethsw_spi_read_one(SPI_STATUS_REG, &reg_cont, sizeof(reg_cont));
		if(ret) goto EXIT;

		if(CHECK_BIT(reg_cont, SPI_STATUS_REG_SPIF) == 0) break;

		LOG_ERR("SPIF retry of 0x%x reg_cont 0x%x\n",
			    SPI_STATUS_REG, reg_cont);

		i --;
		if (i < 1) {
			LOG_ERR("SPIF exceeded retry of %u SPI status reg content 0x%x\n",
				SPI_MAX_RETRY, reg_cont);
			ret = -ETIME;
			goto EXIT;
		}
		mdelay(1);
	}
EXIT:
	return ret;
}


int ethsw_spi_test_status_reg_rack(void)
{
	int ret = 0;
	u8  reg_cont;
	int i = SPI_MAX_RETRY;

	while (1) {
		ret = ethsw_spi_read_one(SPI_STATUS_REG, &reg_cont, sizeof(reg_cont));
		if(ret) goto EXIT;

		if(CHECK_BIT(reg_cont, SPI_STATUS_REG_RACK)) break;

		LOG_DEBUG("RACK retry of 0x%x reg_cont 0x%x\n",
			    SPI_STATUS_REG, reg_cont);

		i --;
		if (i < 1) {
			LOG_ERR("RACK exceeded retry of %u SPI status reg content 0x%x\n",
				 SPI_MAX_RETRY, reg_cont);
			ret = -ETIME;
			goto EXIT;
		}
		mdelay(1);
	}
EXIT:
	return ret;
}

void ethsw_spi_abort_transaction(u8 page)
{
	LOG_WARNING("Aboring transation for page 0x%x", page);
	ethsw_spi_write_one(SPI_PAGE_REG, &page, sizeof(page));
	mdelay(100);
}


// Temporary code to check invalid page access for 3384
int ethsw_invalid_page_map[256];

void ethsw_init_invalid_page_map(void)
{
	int i;

	for (i = 0x05 + 1; i < 0x08; i ++)
		ethsw_invalid_page_map[i] = 1;

	/* Page 8 is undocumented in customer available data sheet */
	for (i = 0x08 + 1; i < 0x11; i ++)
		ethsw_invalid_page_map[i] = 1;

	for (i = 0x14 + 1; i < 0x20; i ++)
		ethsw_invalid_page_map[i] = 1;

	ethsw_invalid_page_map[0x29] = 1;
	ethsw_invalid_page_map[0x33] = 1;
	ethsw_invalid_page_map[0x35] = 1;

	for (i = 0x36 + 1; i < 0x40; i ++)
		ethsw_invalid_page_map[i] = 1;

	for (i = 0x43 + 1; i < 0x70; i ++)
		ethsw_invalid_page_map[i] = 1;

	for (i = 0x71 + 1; i < 0x85; i ++)
		ethsw_invalid_page_map[i] = 1;

	for (i = 0x85 + 1; i < 0x88; i ++)
		ethsw_invalid_page_map[i] = 1;

	ethsw_invalid_page_map[0x89] = 1;

	for (i = 0x92 + 1; i < 0xF0; i ++)
		ethsw_invalid_page_map[i] = 1;

	for (i = 0xF0 + 1; i < 0xFE; i ++)
		ethsw_invalid_page_map[i] = 1;
}



int ethsw_spi_read(u8 page, u8 reg_addr, void *reg_val, u8 reg_size)
{
	int ret = 0;
	u8 reg_cont;
	int step = 0;

	//if(ethsw_invalid_page_map[page]) {
	//	LOG_ERR("Attempted to read invalid page 0x%x\n", page);
	//	ret = -EINVAL;
	 //   goto EXIT;
	//}

	switch(reg_size) {
	case 8:
	case 4:
	case 2:
	case 1:
		break;
	default:
		LOG_ERR("Failed to read %u bytes using SPI. Maximum size of SPI read is %u\n",
			reg_size, MAX_SPI_RW_SIZE);
		ret = -EINVAL;
		goto EXIT;
	}
	ret = mutex_lock_killable(&spi_mutex);
	if(ret)
		goto EXIT;

	ret =  ethsw_spi_test_status_reg_spif();
	if(ret) {
		ethsw_spi_abort_transaction(page);
		step = 1;
		goto EXIT;
	}

	ret = ethsw_spi_write_one(SPI_PAGE_REG, &page, sizeof(page));
	if(ret)  {
		step = 2;
		goto EXIT;
	}

	ret = ethsw_spi_read_one(reg_addr, &reg_cont, sizeof(reg_cont));
	if(ret)  {
		step = 3;
		goto EXIT;
	}

	ret = ethsw_spi_test_status_reg_rack();
	if(ret)  {
		ethsw_spi_abort_transaction(page);
		step = 4;
		goto EXIT;
	}

	ret = ethsw_spi_read_one(SPI_DATA_REG, reg_val, reg_size);
	if(ret)  {
		step = 5;
		goto EXIT;
	}

	mutex_unlock(&spi_mutex);

	switch(reg_size) {
	case 8:
		le64_to_cpus(reg_val);
		LOG_DEBUG("page %x reg_addr %x, reg_val %llx, reg_size %u\n",
			  page, reg_addr, *(u64 *)reg_val, reg_size);
		break;
	case 4:
		le32_to_cpus(reg_val);
		LOG_DEBUG("page %x reg_addr %x, reg_val %x, reg_size %u\n",
			  page, reg_addr, *(u32 *)reg_val, reg_size);
		break;
	case 2:
		le16_to_cpus(reg_val);
		LOG_DEBUG("page %x reg_addr %x, reg_val %x, reg_size %u\n",
			  page, reg_addr, *(u16 *)reg_val, reg_size);
		break;
	case 1:
		LOG_DEBUG("page %x reg_addr %x, reg_val %x, reg_size %u\n",
			  page, reg_addr, *(u8 *)reg_val, reg_size);
		break;
	}

EXIT:
	if(ret) {
		LOG_ERR("%s for page 0x%x, reg 0x%x failed at step %d; ret 0x%x\n",
			__FUNCTION__, page, reg_addr, step, (u32)ret);
		if(step != 0)
			mutex_unlock(&spi_mutex);
	}
	return ret;
}

u64 cpu_to_le48p(u64 *in)
{
	u64 out;

	out = cpu_to_le64p(in);
	out = out << 16;
	return out;
}


int ethsw_spi_write(u8 page, u8 reg_addr, void *reg_val, u8 reg_size)
{
	int ret = 0;
	u64 data64;
	u32 data32;
	u16 data16;
	void *datap;
	int step = 0;

	if(ethsw_invalid_page_map[page]) {
		LOG_ERR("Attempted to write invalid page 0x%x\n", page);
		ret = -EINVAL;
		goto EXIT;
	}

	switch(reg_size) {
	case 8:
		data64 = cpu_to_le64p((u64 *)reg_val);
		datap = &data64;
		break;
	case 6:
		data64 = cpu_to_le48p((u64 *)reg_val);
		// For cbr8 we need to shift by 16 to get
		// rid of zeros.
		data64 >>= 16;
		datap = &data64;
		break;
	case 4:
		data32 = cpu_to_le32p((u32 *)reg_val);
		datap = &data32;
		break;
	case 2:
		data16 = cpu_to_le16p((u16 *)reg_val);
		datap = &data16;
		break;
	case 1:
		datap = reg_val;
		break;
	default:
		LOG_ERR("Failed to write %u bytes using SPI. Maximum size of SPI write is %u\n",
			reg_size, MAX_SPI_RW_SIZE);
		ret = -EINVAL;
		step = 0;
		goto EXIT;
	}

	ret = mutex_lock_killable(&spi_mutex);
	if(ret)
		goto EXIT;


	ret =  ethsw_spi_test_status_reg_spif();
	if(ret) {
		ethsw_spi_abort_transaction(page);
		step = 1;
		goto EXIT;
	}

	ret = ethsw_spi_write_one(SPI_PAGE_REG, &page, sizeof(page));
        if(ret) {
		step = 2;
		goto EXIT;
	}

	ret = ethsw_spi_write_one(reg_addr, datap, reg_size);
	if(ret) {
		step = 3;
		goto EXIT;
	}

	mutex_unlock(&spi_mutex);

	switch(reg_size) {
	case 8:
		LOG_DEBUG("%s: page %x reg_addr %x, reg_val %llx, reg_size %u\n",
			  __FUNCTION__, page, reg_addr, *(u64 *)reg_val, reg_size);
		break;
	case 6:
		LOG_DEBUG("%s: page %x reg_addr %x, reg_val %llx, reg_size %u\n",
			  __FUNCTION__, page, reg_addr, *(u64 *)reg_val >> 16, reg_size);
		break;
	case 4:
		LOG_DEBUG("%s: page %x reg_addr %x, reg_val %x, reg_size %u\n",
			  __FUNCTION__, page, reg_addr, *(u32 *)reg_val, reg_size);
		break;
	case 2:
		LOG_DEBUG("%s: page %x reg_addr %x, reg_val %x, reg_size %u\n",
			  __FUNCTION__, page, reg_addr, *(u16 *)reg_val, reg_size);
		break;
	case 1:
		LOG_DEBUG("%s: page %x reg_addr %x, reg_val %x, reg_size %u\n",
			  __FUNCTION__, page, reg_addr, *(u8 *)reg_val, reg_size);
		break;
	}

EXIT:
	if(ret) {
		LOG_ERR("%s for page 0x%x, reg 0x%x failed at step %d; ret 0x%x\n",
			__FUNCTION__, page, reg_addr, step, (u32)ret);
		if(step != 0)
			mutex_unlock(&spi_mutex);
	}
	return ret;
}


int ethsw_creg_write_spi(struct ethsw_device *swdev,
		     int page_num, int reg_addr, u8 *data, int size)
{
	if (swdev->ethsw_pwr_status)
		return ethsw_spi_write((u8)page_num, (u8)reg_addr, data, (u8)size);
	else {
		LOG_DEBUG("%s: Ethernet switch has been powered down\n",__FUNCTION__);
		return -EINVAL;
	}
}

int ethsw_creg_read_spi(struct ethsw_device *swdev,
		    int page_num, int reg_addr, u8 *data, int size)
{
	if (swdev->ethsw_pwr_status)
		return ethsw_spi_read(page_num, reg_addr, data, size);
	else {
		LOG_DEBUG("%s: Ethernet switch has been powered down\n",__FUNCTION__);
		return -EINVAL;
	}
}
