#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <ion/ion.h>
#include "IONmem.h"

#define ALOGD(...)
#define ALOGE(...)

static int cmem_fd = -2;
static int ref_count = 0;


static int validate_init()
{
    switch (cmem_fd) {
      case -3:
        ALOGE("CMEM_exit() already called, check stderr output for earlier "
            "CMEM failure messages (possibly version mismatch).\n");

        return 0;

      case -2:
        ALOGE("CMEM_init() not called, you must initialize CMEM before "
            "making API calls.\n");

        return 0;

      case -1:
        ALOGE("CMEM file descriptor -1 (failed 'open()'), ensure CMEMK "
            "kernel module cmemk.ko has been installed with 'insmod'");

        return 0;

      default:
        return 1;
    }
}

int CMEM_init(void)
{
    int flags;

    ALOGD("init: entered - ref_count %d, cmem_fd %d\n", ref_count, cmem_fd);

    if (cmem_fd >= 0) {
        ref_count++;
        ALOGD("init: /dev/ion already opened, incremented ref_count %d\n",
            ref_count);
        return 0;
    }

    cmem_fd = ion_open();

    if (cmem_fd < 0) {
        ALOGE("init: Failed to open /dev/ion: '%s'\n", strerror(errno));
        return -1;
    }

    ref_count++;

    ALOGD("init: successfully opened /dev/ion...\n");

    ALOGD("init: exiting, returning success\n");

    return 0;
}


unsigned long CMEM_alloc(size_t size, IONMEM_AllocParams *params)
{
    int ret = 0;
    unsigned long PhyAdrr = 0;
    struct meson_phys_data phy_data;
    struct ion_custom_data custom_data;

    if (!validate_init()) {
        return 0;
    }

    ret = ion_alloc(cmem_fd, size, 0, ION_HEAP_CARVEOUT_MASK, 0, &params->mIonHnd);
    if (ret < 0) {
        ALOGE("ion alloc from custom\n");
        ret = ion_alloc(cmem_fd, size, 0, 1<<5, 0, &params->mIonHnd); //ION_HEAP_TYPE_CUSTOM
        if (ret < 0) {
            ion_close(cmem_fd);
            ALOGE("ion_alloc failed, errno=%d", errno);
            cmem_fd = -1;
            return -ENOMEM;
        }
    }
    ret = ion_share(cmem_fd, params->mIonHnd, &params->mImageFd);
    if (ret < 0) {
        ALOGE("ion_share failed, errno=%d", errno);
        ion_free(cmem_fd, params->mIonHnd);
        ion_close(cmem_fd);
        cmem_fd = -1;
        return -EINVAL;
    }

    #if 0
    phy_data.handle = params->mImageFd;
    phy_data.phys_addr = 0;
    phy_data.size = 0;

    custom_data.cmd = ION_IOC_MESON_PHYS_ADDR;
    custom_data.arg = (unsigned long)&phy_data;

    ret = ioctl(cmem_fd, ION_IOC_CUSTOM, (unsigned long)&custom_data);
    if (ret < 0) {
        ALOGD("ion custom ioctl %x failed with code %d: %s\n",
            ION_IOC_MESON_PHYS_ADDR, ret, strerror(errno));
    } else {
        PhyAdrr = phy_data.phys_addr;
    }
    ALOGD("allocate ion buffer for capture, ret=%d, bytes=%d, PysAdrr=%ld .\n",
        ret, size, PhyAdrr);
    return PhyAdrr;
    #endif
    return ret;
}

int CMEM_getPhyPtr(IONMEM_AllocParams *params, unsigned long *PhyAdrr)
{
    int ret = 0;
    struct meson_phys_data phy_data;
    struct ion_custom_data custom_data;

    if (!validate_init()) {
        return -EBADF;
    }

    phy_data.handle = params->mImageFd;
    phy_data.phys_addr = 0;
    phy_data.size = 0;
    custom_data.cmd = ION_IOC_MESON_PHYS_ADDR;
    custom_data.arg = (unsigned long)&phy_data;
    ret = ioctl(cmem_fd, ION_IOC_CUSTOM, (unsigned long)&custom_data);
    if (ret < 0) {
        *PhyAdrr = 0;
    } else {
        *PhyAdrr = phy_data.phys_addr;
    }
    return ret;
}

#if 0
void* CMEM_getUsrPtr(unsigned long PhyAdr, int size)
{
    void *userp = NULL;
    /* Map the physical address to user space */
    userp = mmap(0,                       // Preferred start address
                 size,                    // Length to be mapped
                 PROT_WRITE | PROT_READ,  // Read and write access
                 MAP_SHARED,              // Shared memory
                 cmem_fd,                 // File descriptor
                 PhyAdr);               // The byte offset from fd

    if (userp == MAP_FAILED) {
        ALOGE("registerAlloc: Failed to mmap buffer at physical address %#lx\n",
            PhyAdr);
        return NULL;
    }
    ALOGD("mmap succeeded, returning virt buffer %p\n", userp);

    return userp;
}
#endif

int CMEM_free(IONMEM_AllocParams *params)
{
    if (!validate_init()) {
        return -1;
    }
    ALOGD("CMEM_free,mIonHnd=%x free\n", params->mIonHnd);
    ion_free(cmem_fd, params->mIonHnd);
    close(params->mImageFd);

    return 0;
}


int CMEM_exit(void)
{
    int result = 0;

    ALOGD("exit: entered - ref_count %d, cmem_fd %d\n", ref_count, cmem_fd);

    if (!validate_init()) {
        return -1;
    }

    ALOGD("exit: decrementing ref_count\n");

    ref_count--;
    if (ref_count == 0) {
        result = ion_close(cmem_fd);

        ALOGD("exit: ref_count == 0, closed /dev/ion (%s)\n",
            result == -1 ? strerror(errno) : "succeeded");

        /* setting -3 allows to distinguish CMEM exit from CMEM failed */
        cmem_fd = -3;
    }

    ALOGD("exit: exiting, returning %d\n", result);

    return result;
}

