/*
 * Copyright (c) 2014 Amlogic, Inc. All rights reserved.
 *
 * This source code is subject to the terms and conditions defined in the
 * file 'LICENSE' which is part of this source code package.
 *
 * Description:
 */


#ifndef HLS_LIST_HEADERS_H
#define HLS_LIST_HEADERS_H

#ifdef __cplusplus
extern "C" {
#endif
#define LIST_POISON1  ((void *) 0x00100100)
#define LIST_POISON2  ((void *) 0x00200200)
#ifndef offsetof
#define offsetof(TYPE, MEMBER) ((size_t) (&((TYPE *)0)->MEMBER))
#endif
#define prefetch(x) (x)

#ifndef typeof
#define typeof(T)   __typeof__(T)
#endif
    /**
     * container_of - cast a member of a structure out to the containing structure
     * @ptr:    the pointer to the member.
     * @type:   the type of the container struct this is embedded in.
     * @member: the name of the member within the struct.
     *
     */
#define container_of(ptr, type, member) ({          \
        const typeof( ((type *)0)->member ) *__mptr = (ptr);    \
        (type *)( (char *)__mptr - offsetof(type,member) );})
    /*
     * Simple doubly linked list implementation.
     *
     * Some of the internal functions ("__xxx") are useful when
     * manipulating whole lists rather than single entries, as
     * sometimes we already know the next/prev entries and we can
     * generate better code by using them directly rather than
     * using the generic single-entry routines.
     */
    struct list_head {
        struct list_head *next;
        struct list_head *prev;
    };

#define LIST_HEAD_INIT(name) { &(name), &(name) }

#define LIST_HEAD(name) \
    struct list_head name = LIST_HEAD_INIT(name)
    static inline void INIT_LIST_HEAD(struct list_head *list)
    {
        list->next = list;
        list->prev = list;
    }
    /*
     * Insert a new entry between two known consecutive entries.
     *
     * This is only for internal list manipulation where we know
     * the prev/next entries already!
     */
    static inline void __list_add(struct list_head *new2add,
                                  struct list_head *prev,
                                  struct list_head *next)
    {
        next->prev = new2add;
        new2add->next = next;
        new2add->prev = prev;
        prev->next = new2add;
    }

    /**
     * list_add - add a new entry
     * @new2add: new entry to be added
     * @head: list head to add it after
     *
     * Insert a new entry after the specified head.
     * This is good for implementing stacks.
     */
    static inline void list_add(struct list_head *new2add, struct list_head *head)
    {
        __list_add(new2add, head, head->next);
    }
    /**
     * list_add_tail - add a new entry
     * @new2add: new entry to be added
     * @head: list head to add it before
     *
     * Insert a new entry before the specified head.
     * This is useful for implementing queues.
     */
    static inline void list_add_tail(struct list_head *new2add, struct list_head *head)
    {
        __list_add(new2add, head->prev, head);
    }

    /*
     * Delete a list entry by making the prev/next entries
     * point to each other.
     *
     * This is only for internal list manipulation where we know
     * the prev/next entries already!
     */
    static inline void __list_del(struct list_head * prev, struct list_head * next)
    {
        next->prev = prev;
        prev->next = next;
    }

    /**
     * list_del - deletes entry from list.
     * @entry: the element to delete from the list.
     * Note: list_empty() on entry does not return true after this, the entry is
     * in an undefined state.
     */
    static inline void list_del(struct list_head *entry)
    {
        __list_del(entry->prev, entry->next);
        entry->next = (struct list_head *)LIST_POISON1;
        entry->prev = (struct list_head *)LIST_POISON2;
    }


    /**
     * list_replace - replace old entry by new one
     * @old : the element to be replaced
     * @new2add : the new element to insert
     *
     * If @old was empty, it will be overwritten.
     */
    static inline void list_replace(struct list_head *old,
                                    struct list_head *new2add)
    {
        new2add->next = old->next;
        new2add->next->prev = new2add;
        new2add->prev = old->prev;
        new2add->prev->next = new2add;
    }

    static inline void list_replace_init(struct list_head *old,
                                         struct list_head *new2add)
    {
        list_replace(old, new2add);
        INIT_LIST_HEAD(old);
    }

    /**
     * list_del_init - deletes entry from list and reinitialize it.
     * @entry: the element to delete from the list.
     */
    static inline void list_del_init(struct list_head *entry)
    {
        __list_del(entry->prev, entry->next);
        INIT_LIST_HEAD(entry);
    }

    /**
     * list_move - delete from one list and add as another's head
     * @list: the entry to move
     * @head: the head that will precede our entry
     */
    static inline void list_move(struct list_head *list, struct list_head *head)
    {
        __list_del(list->prev, list->next);
        list_add(list, head);
    }

    /**
     * list_move_tail - delete from one list and add as another's tail
     * @list: the entry to move
     * @head: the head that will follow our entry
     */
    static inline void list_move_tail(struct list_head *list,
                                      struct list_head *head)
    {
        __list_del(list->prev, list->next);
        list_add_tail(list, head);
    }

    /**
     * list_is_last - tests whether @list is the last entry in list @head
     * @list: the entry to test
     * @head: the head of the list
     */
    static inline int list_is_last(const struct list_head *list,
                                   const struct list_head *head)
    {
        return list->next == head;
    }

    /**
     * list_empty - tests whether a list is empty
     * @head: the list to test.
     */
    static inline int list_empty(const struct list_head *head)
    {
        return head->next == head;
    }

    /**
     * list_empty_careful - tests whether a list is empty and not being modified
     * @head: the list to test
     *
     * Description:
     * tests whether a list is empty _and_ checks that no other CPU might be
     * in the process of modifying either member (next or prev)
     *
     * NOTE: using list_empty_careful() without synchronization
     * can only be safe if the only activity that can happen
     * to the list entry is list_del_init(). Eg. it cannot be used
     * if another CPU could re-list_add() it.
     */
    static inline int list_empty_careful(const struct list_head *head)
    {
        struct list_head *next = head->next;
        return (next == head) && (next == head->prev);
    }

    static inline void __list_splice(struct list_head *list,
                                     struct list_head *head)
    {
        struct list_head *first = list->next;
        struct list_head *last = list->prev;
        struct list_head *at = head->next;

        first->prev = head;
        head->next = first;

        last->next = at;
        at->prev = last;
    }

    /**
     * list_splice - join two lists
     * @list: the new list to add.
     * @head: the place to add it in the first list.
     */
    static inline void list_splice(struct list_head *list, struct list_head *head)
    {
        if (!list_empty(list)) {
            __list_splice(list, head);
        }
    }

    /**
     * list_splice_init - join two lists and reinitialise the emptied list.
     * @list: the new list to add.
     * @head: the place to add it in the first list.
     *
     * The list at @list is reinitialised
     */
    static inline void list_splice_init(struct list_head *list,
                                        struct list_head *head)
    {
        if (!list_empty(list)) {
            __list_splice(list, head);
            INIT_LIST_HEAD(list);
        }
    }

    /**
     * list_entry - get the struct for this entry
     * @ptr:    the &struct list_head pointer.
     * @type:   the type of the struct this is embedded in.
     * @member: the name of the list_struct within the struct.
     */
#define list_entry(ptr, type, member) \
    container_of(ptr, type, member)
    /**
     * list_first_entry - get the first element from a list
     * @ptr:    the list head to take the element from.
     * @type:   the type of the struct this is embedded in.
     * @member: the name of the list_struct within the struct.
     *
     * Note, that list is expected to be not empty.
     */
#define list_first_entry(ptr, type, member) \
    list_entry((ptr)->next, type, member)

    /**
     * list_last_entry - get the last element from a list
     * @ptr:    the list head to take the element from.
     * @type:   the type of the struct this is embedded in.
     * @member: the name of the list_struct within the struct.
     *
     * Note, that list is expected to be not empty.
     */
#define list_last_entry(ptr, type, member) \
    list_entry((ptr)->prev, type, member)
    /**
     * list_for_each    -   iterate over a list
     * @pos:    the &struct list_head to use as a loop cursor.
     * @head:   the head for your list.
     */
#define list_for_each(pos, head) \
    for (pos = (head)->next; pos != (head); \
            pos = pos->next)

    /**
     * __list_for_each  -   iterate over a list
     * @pos:    the &struct list_head to use as a loop cursor.
     * @head:   the head for your list.
     *
     * This variant differs from list_for_each() in that it's the
     * simplest possible list iteration code, no prefetching is done.
     * Use this for code that knows the list to be very short (empty
     * or 1 entry) most of the time.
     */
#define __list_for_each(pos, head) \
    for (pos = (head)->next; pos != (head); pos = pos->next)

    /**
     * list_for_each_prev   -   iterate over a list backwards
     * @pos:    the &struct list_head to use as a loop cursor.
     * @head:   the head for your list.
     */
#define list_for_each_prev(pos, head) \
    for (pos = (head)->prev; pos != (head); \
            pos = pos->prev)

    /**
     * list_for_each_safe - iterate over a list safe against removal of list entry
     * @pos:    the &struct list_head to use as a loop cursor.
     * @n:      another &struct list_head to use as temporary storage
     * @head:   the head for your list.
     */
#define list_for_each_safe(pos, n, head) \
    for (pos = (head)->next, n = pos->next; pos != (head); \
        pos = n, n = pos->next)

    /**
     * list_for_each_entry  -   iterate over list of given type
     * @pos:    the type * to use as a loop cursor.
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     */
#define list_for_each_entry(pos, head, member)              \
    for (pos = list_entry((head)->next, typeof(*pos), member);  \
         &pos->member != (head);    \
         pos = list_entry(pos->member.next, typeof(*pos), member))

    /**
     * list_for_each_entry_reverse - iterate backwards over list of given type.
     * @pos:    the type * to use as a loop cursor.
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     */
#define list_for_each_entry_reverse(pos, head, member)          \
    for (pos = list_entry((head)->prev, typeof(*pos), member);  \
         &pos->member != (head);    \
         pos = list_entry(pos->member.prev, typeof(*pos), member))

    /**
     * list_prepare_entry - prepare a pos entry for use in list_for_each_entry_continue()
     * @pos:    the type * to use as a start point
     * @head:   the head of the list
     * @member: the name of the list_struct within the struct.
     *
     * Prepares a pos entry for use as a start point in list_for_each_entry_continue().
     */
#define list_prepare_entry(pos, head, member) \
    ((pos) ? : list_entry(head, typeof(*pos), member))

    /**
     * list_for_each_entry_continue - continue iteration over list of given type
     * @pos:    the type * to use as a loop cursor.
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     *
     * Continue to iterate over list of given type, continuing after
     * the current position.
     */
#define list_for_each_entry_continue(pos, head, member)         \
    for (pos = list_entry(pos->member.next, typeof(*pos), member);  \
         &pos->member != (head);    \
         pos = list_entry(pos->member.next, typeof(*pos), member))

    /**
     * list_for_each_entry_from - iterate over list of given type from the current point
     * @pos:    the type * to use as a loop cursor.
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     *
     * Iterate over list of given type, continuing from current position.
     */
#define list_for_each_entry_from(pos, head, member)             \
    for (; &pos->member != (head);  \
         pos = list_entry(pos->member.next, typeof(*pos), member))

    /**
     * list_for_each_entry_safe - iterate over list of given type safe against removal of list entry
     * @pos:    the type * to use as a loop cursor.
     * @n:      another type * to use as temporary storage
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     */
#define list_for_each_entry_safe(pos, n, head, member)          \
    for (pos = list_entry((head)->next, typeof(*pos), member),  \
        n = list_entry(pos->member.next, typeof(*pos), member); \
         &pos->member != (head);                    \
         pos = n, n = list_entry(n->member.next, typeof(*n), member))

    /**
     * list_for_each_entry_safe_continue
     * @pos:    the type * to use as a loop cursor.
     * @n:      another type * to use as temporary storage
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     *
     * Iterate over list of given type, continuing after current point,
     * safe against removal of list entry.
     */
#define list_for_each_entry_safe_continue(pos, n, head, member)         \
    for (pos = list_entry(pos->member.next, typeof(*pos), member),      \
        n = list_entry(pos->member.next, typeof(*pos), member);     \
         &pos->member != (head);                        \
         pos = n, n = list_entry(n->member.next, typeof(*n), member))

    /**
     * list_for_each_entry_safe_from
     * @pos:    the type * to use as a loop cursor.
     * @n:      another type * to use as temporary storage
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     *
     * Iterate over list of given type from current point, safe against
     * removal of list entry.
     */
#define list_for_each_entry_safe_from(pos, n, head, member)             \
    for (n = list_entry(pos->member.next, typeof(*pos), member);        \
         &pos->member != (head);                        \
         pos = n, n = list_entry(n->member.next, typeof(*n), member))

    /**
     * list_for_each_entry_safe_reverse
     * @pos:    the type * to use as a loop cursor.
     * @n:      another type * to use as temporary storage
     * @head:   the head for your list.
     * @member: the name of the list_struct within the struct.
     *
     * Iterate backwards over list of given type, safe against removal
     * of list entry.
     */
#define list_for_each_entry_safe_reverse(pos, n, head, member)      \
    for (pos = list_entry((head)->prev, typeof(*pos), member),  \
        n = list_entry(pos->member.prev, typeof(*pos), member); \
         &pos->member != (head);                    \
         pos = n, n = list_entry(n->member.prev, typeof(*n), member))

#ifdef __cplusplus
}
#endif

#endif
