/* SPDX-License-Identifier: (GPL-2.0+ OR MIT)
 *
 * Copyright (c) 2019 Amlogic, Inc. All rights reserved.
 */
#include <linux/module.h>
#include <linux/err.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/io.h>
#include <linux/cdev.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/device.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/uaccess.h>
#include <linux/of.h>
#include <linux/memblock.h>
#include <linux/random.h>
#include <linux/compat.h>
#include "ring_buffer.h"

#define BL30MSG_LIMIT_ADDR	    (0x0F000000)

static struct proc_dir_entry *bl30msg_proc;
static unsigned long bl30msg_mem_base;
static unsigned long bl30msg_mem_size;
static int flush_msg = 1;

static int bl30msg_proc_show(struct seq_file *file, void *v)
{
	unsigned int count;

    if (bl30msg_mem_base == 0 || bl30msg_mem_base > BL30MSG_LIMIT_ADDR ||
        bl30msg_mem_size == 0)
        return 0;
    
    if (flush_msg)
	    count = read_remove_ring_buffer(bl30msg_mem_base, bl30msg_mem_size);
    else
	    count = read_keep_ring_buffer(bl30msg_mem_base, bl30msg_mem_size);

    if (count > 0) {
        void * mem_base_virt = phys_to_virt(bl30msg_mem_base);
        seq_write(file, (void *)mem_base_virt, count);
    }

	return 0;
}

static int bl30msg_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, bl30msg_proc_show, NULL);
}

static const struct file_operations bl30msg_proc_fops = {
	.open = bl30msg_proc_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release
};

static ssize_t flush_msg_show(struct class *cla,
				struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", flush_msg);
}

static ssize_t flush_msg_store(struct class *cla,
				 struct class_attribute *attr,
				 const char *buf, size_t count)
{
	unsigned int len;

	if (buf[count - 1] == '\n')
		len = count - 1;
	else
		len = count;

	if (!strncmp(buf, "0", len)) {
		flush_msg = 0;
	}
    else {
        flush_msg = 1;
    }

	return count;
}

static CLASS_ATTR_RW(flush_msg);

static struct attribute *bl30msg_attrs[] = {
	&class_attr_flush_msg.attr,
	NULL,
};

ATTRIBUTE_GROUPS(bl30msg);

static struct class bl30msg_class = {
	.name		= "bl30msg",
	.owner		= THIS_MODULE,
	.class_groups = bl30msg_groups,
};

static int __init early_bl30msg_para(char *buf)
{
	int ret;

	if (!buf)
		return -EINVAL;

	ret = sscanf(buf, "%lx,%lx",
		     &bl30msg_mem_base, &bl30msg_mem_size);
	if (ret != 2) {
		pr_err("invalid boot args \"defendkey\"\n");
		return -EINVAL;
	}

	if (bl30msg_mem_base > BL30MSG_LIMIT_ADDR) {
		pr_err("bl30msg reserved memory base overflow\n");
		return -EINVAL;
	}

	pr_info("%s, base:%lx, size:%lx\n",
		__func__, bl30msg_mem_base, bl30msg_mem_size);

	ret = memblock_reserve(bl30msg_mem_base,
			       PAGE_ALIGN(bl30msg_mem_size));
	if (ret < 0) {
		pr_info("reserve memblock %lx - %lx failed\n",
			bl30msg_mem_base,
			bl30msg_mem_base + PAGE_ALIGN(bl30msg_mem_size));
		return -EINVAL;
	}

    pr_info("reserve memblock %lx - %lx\n",
        bl30msg_mem_base,
        bl30msg_mem_base + PAGE_ALIGN(bl30msg_mem_size));

	return 0;
}

early_param("bl30msg", early_bl30msg_para);


static int bl30msg_probe(struct platform_device *pdev)
{
	bl30msg_proc = proc_create("bl30msg", 0644, NULL, &bl30msg_proc_fops);
    class_register(&bl30msg_class);
    
	return 0;
}

static int bl30msg_remove(struct platform_device *pdev)
{
    if (bl30msg_proc) {
	    proc_remove(bl30msg_proc);
    }
    sysfs_remove_group(&pdev->dev.kobj,(struct attribute_group *)&bl30msg_groups);
	return 0;
}

static const struct of_device_id bl30msg_of_match[] = {
	{	.compatible = "amlogic, meson_bl30msg",
	},
	{},
};

static struct platform_driver bl30msg_driver = {
	.probe = bl30msg_probe,
	.remove = bl30msg_remove,
	.driver = {
		.owner = THIS_MODULE,
		.name = "meson_bl30msg",
		.of_match_table = bl30msg_of_match,
	},
};

static int __init bl30msg_init(void)
{
  	pr_warn("[%s]module init begin\n", __func__);
	return platform_driver_register(&bl30msg_driver);
}
core_initcall(bl30msg_init);

static void __exit bl30msg_exit(void)
{
	platform_driver_unregister(&bl30msg_driver);
}
module_exit(bl30msg_exit);

MODULE_AUTHOR("xingxing.wang@amlogic.com");
MODULE_DESCRIPTION("meson bl30msg driver");
MODULE_LICENSE("GPL");
