/**
 * Copyright (C) ARM Limited 2011-2016. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/cpufreq.h>
#include <trace/events/power.h>

#if defined(__arm__)

#include <asm/mach-types.h>

#define implements_wfi() (!machine_is_omap3_beagle())

#else

#define implements_wfi() false

#endif

/* cpu_frequency and cpu_idle trace points were introduced in Linux
 * kernel v2.6.38 the now deprecated power_frequency trace point was
 * available prior to 2.6.38, but only for x86
 */
#if GATOR_CPU_FREQ_SUPPORT

static DEFINE_PER_CPU(ulong, idle_prev_state);
static ulong power_cpu_enabled[GATOR_CLUSTER_COUNT];
static bool power_cpu_enabled_any;
static ulong power_cpu_key[GATOR_CLUSTER_COUNT];

static int gator_trace_power_create_files(struct super_block *sb, struct dentry *root)
{
    struct dentry *dir;
    int cpu;
    bool found_nonzero_freq = false;

    /* Even if CONFIG_CPU_FREQ is defined, it still may not be
     * used. Check for non-zero values from cpufreq_quick_get
     */
    for_each_online_cpu(cpu) {
        if (cpufreq_quick_get(cpu) > 0) {
            found_nonzero_freq = true;
            break;
        }
    }

    if (found_nonzero_freq) {
        char buf[40];
        int i;

        /* cpu_frequency */
        for (i = 0; i < gator_cluster_count; i++) {
            snprintf(buf, sizeof(buf), "%s_freq", gator_clusters[i]->pmnc_name);
            dir = gatorfs_mkdir(sb, root, buf);
            if (!dir)
                return -1;
            gatorfs_create_ulong(sb, dir, "enabled", &power_cpu_enabled[i]);
            gatorfs_create_ro_ulong(sb, dir, "key", &power_cpu_key[i]);
        }
    }

    return 0;
}

/* 'cpu' may not equal smp_processor_id(), i.e. may not be running on the core that is having the freq/idle state change */
GATOR_DEFINE_PROBE(cpu_frequency, TP_PROTO(unsigned int frequency, unsigned int cpu))
{
    cpu = lcpu_to_pcpu(cpu);
    marshal_event_single64(cpu, power_cpu_key[gator_clusterids[cpu]], frequency * 1000L);
}

GATOR_DEFINE_PROBE(cpu_idle, TP_PROTO(unsigned int state, unsigned int cpu))
{
    cpu = lcpu_to_pcpu(cpu);

    if (state == per_cpu(idle_prev_state, cpu))
        return;

    if (implements_wfi())
        marshal_idle(cpu, state);

    per_cpu(idle_prev_state, cpu) = state;
}

static void gator_trace_power_online(void)
{
    int pcpu = get_physical_cpu();
    int lcpu = get_logical_cpu();

    if (power_cpu_enabled[gator_clusterids[pcpu]])
        marshal_event_single64(pcpu, power_cpu_key[gator_clusterids[pcpu]], cpufreq_quick_get(lcpu) * 1000L);
}

static void gator_trace_power_offline(void)
{
    /* Set frequency to zero on an offline */
    int cpu = get_physical_cpu();

    if (power_cpu_enabled[gator_clusterids[cpu]])
        marshal_event_single(cpu, power_cpu_key[gator_clusterids[cpu]], 0);
}

static int gator_trace_power_start(void)
{
    int cpu;
    int i;

    power_cpu_enabled_any = false;
    for (i = 0; i < gator_cluster_count; i++) {
        if (power_cpu_enabled[i]) {
            power_cpu_enabled_any = true;
            break;
        }
    }

    /* register tracepoints */
    if (power_cpu_enabled_any)
        if (GATOR_REGISTER_TRACE(cpu_frequency))
            goto fail_cpu_frequency_exit;

    /* Always register for cpu_idle for detecting WFI */
    if (GATOR_REGISTER_TRACE(cpu_idle))
        goto fail_cpu_idle_exit;
    pr_debug("gator: registered power event tracepoints\n");

    for_each_present_cpu(cpu) {
        per_cpu(idle_prev_state, cpu) = 0;
    }

    return 0;

    /* unregister tracepoints on error */
fail_cpu_idle_exit:
    if (power_cpu_enabled_any)
        GATOR_UNREGISTER_TRACE(cpu_frequency);
fail_cpu_frequency_exit:
    pr_err("gator: power event tracepoints failed to activate, please verify that tracepoints are enabled in the linux kernel\n");

    return -1;
}

static void gator_trace_power_stop(void)
{
    int i;

    if (power_cpu_enabled_any)
        GATOR_UNREGISTER_TRACE(cpu_frequency);
    GATOR_UNREGISTER_TRACE(cpu_idle);
    pr_debug("gator: unregistered power event tracepoints\n");

    for (i = 0; i < gator_cluster_count; i++)
        power_cpu_enabled[i] = 0;
}

static void gator_trace_power_init(void)
{
    int i;

    for (i = 0; i < gator_cluster_count; i++) {
        power_cpu_enabled[i] = 0;
        power_cpu_key[i] = gator_events_get_key();
    }
}
#else
static int gator_trace_power_create_files(struct super_block *sb, struct dentry *root)
{
    return 0;
}

static void gator_trace_power_online(void)
{
}

static void gator_trace_power_offline(void)
{
}

static int gator_trace_power_start(void)
{
    return 0;
}

static void gator_trace_power_stop(void)
{
}

static void gator_trace_power_init(void)
{
}
#endif
