// SPDX-License-Identifier: (GPL-2.0+ OR MIT)
/*
 * Copyright (c) 2021 Amlogic, Inc. All rights reserved.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/dma-mapping.h>
#include <linux/device.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/of_address.h>
#include <linux/of_platform.h>
#include <linux/uaccess.h>
#include <linux/compat.h>
#include <linux/io.h>
#include <linux/mutex.h>
#include <linux/arm-smccc.h>
#include <asm/cacheflush.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/amlogic/iomap.h>
#include <linux/amlogic/secmon.h>
#include <linux/ion.h>
#include <linux/meson_ion.h>

#include "secure_mcp.h"

#define S_MCP_DEVICE_NAME "secure_mcp"
#define SMC_RETURN_OK 0

#define AES128_TYPE 0
#define AES256_TYPE 1
#define AES128_LENGTH 16
#define AES256_LENGTH 32

static struct platform_device *local_pdev;
static struct device *mcp_device;
static struct cdev mcp_dev;
static dev_t devno;
static void __iomem *sharemem_input_base;

struct class *secure_mcp_dev_class;

static DEFINE_MUTEX(s_mcp_mutex);

/*
 * Func : llama_load_key
 * Desc : Load the key (and optionally) the IV for an encrypt/decrypt
 * Parm : key        : array of key
 *        iv         : Optional IV - or NULL. If NULL, fill with 0's
 *        key_type   : AES128 (0) OR AES256 (1)
 *        reference  : Pointer for reference of key (return)
 * Retn : 0          : success, others fail
 */
static int32_t llama_load_key(const u8 key[32], const u8 iv[16], u8 key_type, u32 *reference)
{
	long ret;
	struct arm_smccc_res res;
	u8 data[48] = {0};
	int key_length = AES256_LENGTH;

	if (!sharemem_input_base)
		return -1;

	if (key_type != AES128_TYPE && key_type != AES256_TYPE)
		return -1;

	if (key_type == AES128_TYPE)
		key_length = AES128_LENGTH;

	memcpy(data, key, key_length);
	memcpy(&data[32], iv, 16);

	meson_sm_mutex_lock();

	/* Copy the key and the IV into place */
	memcpy((void *)sharemem_input_base, data, sizeof(data));
	asm __volatile__("" : : : "memory");

	arm_smccc_smc(LLAMA_LOAD_KEY, key_type, 0, 0, 0, 0, 0, 0, &res);
	ret = res.a0;
	*reference = res.a1;
	meson_sm_mutex_unlock();

	/* 0 is okay */
	return ret;
}

/*
 * Func : llama_release_key
 * Desc : open function of mcp dev
 * Parm : ref  : context of key
 * Retn : 0    : success, others fail
 */
static int32_t llama_release_key(u32 ref)
{
	long ret;
	struct arm_smccc_res res;

	asm __volatile__("" : : : "memory");

	arm_smccc_smc(LLAMA_RELEASE_KEY, ref, 0, 0, 0, 0, 0, 0, &res);
	ret = res.a0;

	/* 0 is okay */
	return ret;
}

/*
 * Func : llama_setup_transfer
 * Desc : open function of mcp dev
 * Parm : ref        : context of key
 *        key_type   : type of key - AES128 (0) or AES256 (1)
 *        chain_type : type of chain - ECB, (0) CBC (1) or CTR (2)
 *        iv         : initialisation vector
 * Retn : 0 : success, others fail
 */
static int32_t llama_setup_transfer(u32 ref, u8 key_type, u8 chain_type, const u8 *iv)
{
	long ret;
	struct arm_smccc_res res;

	if (!sharemem_input_base)
		return -1;

	meson_sm_mutex_lock();
	memcpy((void *)sharemem_input_base, (const void *)iv, 16);

	asm __volatile__("" : : : "memory");

	arm_smccc_smc(LLAMA_SETUP_TRANSFER, ref, key_type, chain_type, 0, 0, 0, 0, &res);
	ret = res.a0;

	meson_sm_mutex_unlock();

	/* 0 is okay */
	return ret;
}

/*
 * Func : llama_encrypt_transfer
 * Desc : Encrypt some data (Needs to have setup transfer)
 * Parm : ref        : context of key
 *        src        : source buffer
 *        src_length : source buffer length
 *        dst        : destination buffer (May be the same as src)
 * Retn : 0          : success, others fail
 */
static int32_t llama_encrypt_transfer(u32 ref, uintptr_t src, u32 src_length, uintptr_t dst)
{
	long ret;
	struct arm_smccc_res res;

	asm __volatile__("" : : : "memory");

	arm_smccc_smc(LLAMA_ENCRYPT_TRANSFER, ref, src, src_length, dst, 0, 0, 0, &res);
	ret = res.a0;

	return ret;
}

/*
 * Func : llama_decrypt_transfer
 * Desc : Decrypt some data (Needs to have setup transfer)
 * Parm : ref        : context of key
 *        src        : source buffer
 *        src_length : source buffer length
 *        dst        : destination buffer (May be the same as src)
 * Retn : 0          : success, others fail
 */
static int32_t llama_decrypt_transfer(u32 ref, uintptr_t src, u32 src_length, uintptr_t dst)
{
	long ret;
	struct arm_smccc_res res;

	asm __volatile__("" : : : "memory");
	arm_smccc_smc(LLAMA_DERYPT_TRANSFER, ref, src, src_length, dst, 0, 0, 0, &res);
	ret = res.a0;

	return ret;
}

/*
 * Func : s_mcp_dev_open
 * Desc : open function of mcp dev
 * Parm : inode : inode of dev
 *        file  : context of file
 * Retn : 0 : success, others fail
 */
static int s_mcp_dev_open(struct inode *inode, struct file *file)
{
	return 0;
}

/*
 * Func : s_mcp_dev_release
 * Desc : release function of mcp dev
 * Parm : inode : inode of dev
 *        file  : context of file
 * Retn : 0 : success, others fail
 */
static int s_mcp_dev_release(struct inode *inode, struct file *file)
{
	return 0;
}

/*
 * Func : compat_s_mcp_dev_ioctl
 * Desc : ioctl function of mcp dev
 * Parm : inode : inode of dev
 * file : context o    le
 * cmd : control command
 * arg : arguments
 * Retn : 0 : success, others fail
 */
//static long compat_s_mcp_dev_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
//{
//printk("%s:%d\n", __func__,__LINE__);
//return 0;
//}
/*
 * Func : s_mcp_dev_ioctl
 * Desc : ioctl function of mcp dev
 * Parm : inode : inode of dev
 * file : context of file
 * cmd : control command
 * arg : arguments
 * Retn : 0 : success, others fail
 */
static long s_mcp_dev_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct secure_mcp_key_data mcp_key_data = {0};
	int ret = 0;
	struct key_process_info_t *mcp_key_process_info = NULL;
	unsigned int *reference_val = NULL;

	switch (cmd) {
	case SECURE_MCP_LOAD_KEY:
		mutex_lock(&s_mcp_mutex);
		mcp_key_process_info = kzalloc(sizeof(*mcp_key_process_info), GFP_KERNEL);
		reference_val = kzalloc(sizeof(unsigned int), GFP_KERNEL);
		memset(mcp_key_process_info, 0, sizeof(*mcp_key_process_info));

		if (copy_from_user(mcp_key_process_info,
			(void *)arg, sizeof(*mcp_key_process_info))) {
			secure_mcp_warning("do ioctl failed-copy secure_mcp_key_data from user failed\n");
			ret = 1;
			mutex_unlock(&s_mcp_mutex);
			goto result;
		}

		ret = llama_load_key((u8 *)mcp_key_process_info->key,
							(u8 *)mcp_key_process_info->iv,
							mcp_key_process_info->key_type,
							reference_val);
		if (ret != 0) {
			secure_mcp_warning("load key failed\n");
			mutex_unlock(&s_mcp_mutex);
			goto result;
		}
		mcp_key_process_info->dst_addr = *reference_val;
		if (copy_to_user((void *)arg, mcp_key_process_info,
				sizeof(*mcp_key_process_info))) {
			secure_mcp_warning("do ioctl failed-copy secure_mcp_key_data to user failed\n");
			ret = 1;
			mutex_unlock(&s_mcp_mutex);
			goto result;
		}

		mutex_unlock(&s_mcp_mutex);

		break;

	case SECURE_MCP_TRANSFER:
		mutex_lock(&s_mcp_mutex);
		mcp_key_process_info = kzalloc(sizeof(*mcp_key_process_info), GFP_KERNEL);
		reference_val = kzalloc(sizeof(unsigned int), GFP_KERNEL);
		memset(mcp_key_process_info, 0, sizeof(*mcp_key_process_info));

		if (copy_from_user(mcp_key_process_info,
			(void *)arg, sizeof(*mcp_key_process_info))) {
			secure_mcp_warning("do ioctl failed-copy secure_mcp_key_data from user failed\n");
			ret = 1;
			mutex_unlock(&s_mcp_mutex);
			goto result;
		}

		ret = llama_setup_transfer(mcp_key_process_info->key_addr,
				mcp_key_process_info->key_type,
				mcp_key_process_info->bcm,
				(u8 *)mcp_key_process_info->iv);
		if (ret == 0) {
			if (mcp_key_process_info->en_de == 0) {
				ret = llama_decrypt_transfer(mcp_key_process_info->key_addr,
							mcp_key_process_info->src_addr,
							mcp_key_process_info->length,
							mcp_key_process_info->dst_addr);
			} else {
				ret = llama_encrypt_transfer(mcp_key_process_info->key_addr,
							mcp_key_process_info->src_addr,
							mcp_key_process_info->length,
							mcp_key_process_info->dst_addr);
			}
		} else {
			secure_mcp_warning("setup transfer failed\n");
		}
		if (ret != 0) {
			mutex_unlock(&s_mcp_mutex);
			goto result;
		}

		mutex_unlock(&s_mcp_mutex);
		break;

	case SECURE_MCP_RELEASE_KEY:
		mutex_lock(&s_mcp_mutex);

		if (copy_from_user(&mcp_key_data,
				(void *)arg,
				sizeof(struct secure_mcp_key_data))) {
			secure_mcp_warning("do ioctl failed-copy secure_mcp_key_data from user failed\n");
			ret = 1;
			mutex_unlock(&s_mcp_mutex);
			goto result;
		}
		ret = llama_release_key(mcp_key_data.key_no);
		if (ret != 0) {
			ret = 1;
			secure_mcp_warning("do ioctl command failed - release_key failed\n");
			mutex_unlock(&s_mcp_mutex);
			goto result;
		}

		mutex_unlock(&s_mcp_mutex);
		break;

	default:
		secure_mcp_warning("do ioctl command failed - unknown ioctl command(%d)\n", cmd);
		ret = 1;
		goto result;
	}

result:
	kfree(mcp_key_process_info);
	kfree(reference_val);

	if (ret == 0)
		return ret;
	else
		return -EFAULT;
}

static const struct file_operations mcp_ops = {
	.owner = THIS_MODULE,
	//.ioctl = s_mcp_dev_ioctl,
	.unlocked_ioctl = s_mcp_dev_ioctl,
	.compat_ioctl = s_mcp_dev_ioctl,
	.open = s_mcp_dev_open,
	.release = s_mcp_dev_release,
};

static int s_mcp_probe(struct platform_device *pdev)
{
	local_pdev = pdev;

	secure_mcp_info("driver initial begin.\n");

	cdev_init(&mcp_dev, &mcp_ops);

	if (alloc_chrdev_region(&devno, 0, 1, S_MCP_DEVICE_NAME) != 0) {
		cdev_del(&mcp_dev);
		secure_mcp_warning("%s %s %d", __FILE__, __func__, __LINE__);
		return -EFAULT;
	}

	if (cdev_add(&mcp_dev, devno, 1) < 0)
		return -EFAULT;

	secure_mcp_dev_class = class_create(THIS_MODULE, S_MCP_DEVICE_NAME);

	mcp_device = device_create(secure_mcp_dev_class, NULL, devno, NULL, "secure_mcp_core");

	sharemem_input_base = get_meson_sm_input_base();

	secure_mcp_info("driver initial done.\n");

	return 0;
}

static int s_mcp_remove(struct platform_device *dev)
{
	secure_mcp_info("driver remove begin.\n");

	device_destroy(secure_mcp_dev_class, mcp_device->devt);

	cdev_del(&mcp_dev);

	unregister_chrdev_region(devno, 1);

	secure_mcp_info("driver remove done.\n");

	return 0;
}

static int s_mcp_suspend(struct platform_device *dev, pm_message_t state)
{
	return 0;
}

static int s_mcp_resume(struct platform_device *dev)
{
	return 0;
}

static const struct of_device_id secure_mcp_ids[] = {
	{ .compatible = "amlogic,secure-mcp" },
	{ /* Sentinel */ },
};

static struct platform_driver secure_mcp_driver = {
	.probe = s_mcp_probe,
	.remove = s_mcp_remove,
	.suspend = s_mcp_suspend,
	.resume = s_mcp_resume,
	.driver = {
		.name = "SECURE_MCP",
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(secure_mcp_ids),
	},
};

module_platform_driver(secure_mcp_driver);
