
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/mm.h>
#include <linux/kdev_t.h>
#include <asm/page.h>
#include <linux/cdev.h>

#include <linux/device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/acpi.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>

#include <linux/interrupt.h>
#include <linux/irq.h>

#include <linux/wait.h>       // wait_event_interruptible
#include <linux/poll.h>
#ifdef CONFIG_AMLOGIC_LEGACY_EARLY_SUSPEND
#include <linux/amlogic/pm.h>
#endif

#define HEADPHONE_DETECT_DEVICE "hp_detect"

static int g_hpd_major;
static u32  g_minor;

module_param(g_hpd_major, int, 0);

struct hpd_data {
	dev_t             hpdDev;
	u32               hpd_gpio;
	u32               hpd_irq_num;
};

static struct hpd_data g_hpd_data;

volatile unsigned long g_hpd;

DECLARE_WAIT_QUEUE_HEAD(hpd_queue);

irqreturn_t hpd_irq_handler(int irq, void *data)
{
	set_bit(1, &g_hpd);
	
	wake_up_interruptible(&hpd_queue);
	return IRQ_HANDLED;
}

unsigned int hpd_poll(struct file *filp, poll_table *wait)
{
	unsigned int mask = 0;

	poll_wait(filp, &hpd_queue, wait);

	if (test_and_clear_bit(1, &g_hpd))
		mask |= POLLPRI;

	return mask;
}

static ssize_t hpd_read(struct file *filp,
		char *buffer, size_t len, loff_t *offs)
{
	unsigned int ret = 0;
	int hp_det;

	hp_det = gpio_get_value(g_hpd_data.hpd_gpio);
	pr_info("Headphone connection : %d  \r\n", hp_det);

	if (buffer) {
		ret = copy_to_user(buffer, &hp_det, len);
		if (ret != 0)
			pr_err("Failed to copy data to user. ret = %d\n", ret);
	} else {
		pr_err("error buffer is null \r\n");
		ret = -1;
	}

	return ret;
}

static int hpd_open(struct inode *inode, struct file *filp)
{
	g_minor = MINOR(inode->i_rdev);

	if (g_minor >= 1) {
		pr_err("Failed -> invalid device instance %d", g_minor);
		return(-ENODEV);
	}

	return(0);
}

static int hpd_release(struct inode *inode, struct file *filp)
{
	return 0;
}

#ifdef CONFIG_AMLOGIC_LEGACY_EARLY_SUSPEND
static void hdp_early_suspend(struct early_suspend *h)
{
}

static void hdp_late_resume(struct early_suspend *h)
{
	hpd_irq_handler(1, NULL);
}

static struct early_suspend hdp_suspend_handler = {
	.suspend = hdp_early_suspend,
	.resume  = hdp_late_resume,
};
#endif

static const struct file_operations hpd_ops = {
	.owner   = THIS_MODULE,
	.open    = hpd_open,
	.read	 = hpd_read,
	.release = hpd_release,
	.poll    = hpd_poll,
};

static struct class *hpd_class;

static int hpd_init(void)
{
	struct device_node *np;
	int status = -1;
	int gpio_pin = -1;
	struct device *dev;

	g_hpd_major = register_chrdev(g_hpd_major, HEADPHONE_DETECT_DEVICE, &hpd_ops);

	if (g_hpd_major < 0)
		return -1;

	hpd_class = class_create(THIS_MODULE, HEADPHONE_DETECT_DEVICE);
	if (IS_ERR(hpd_class)) {
		unregister_chrdev(g_hpd_major, HEADPHONE_DETECT_DEVICE);
		return PTR_ERR(hpd_class);
	}

	g_hpd_data.hpdDev = MKDEV(g_hpd_major, 0);
	dev = device_create(hpd_class, NULL, g_hpd_data.hpdDev,
			NULL, HEADPHONE_DETECT_DEVICE);
	if (!dev) {
		pr_err("Device creation failed\n");
		return(-ENODEV);
	}

	np = of_find_compatible_node(NULL, NULL, "amlogic,headphone");

	gpio_pin = of_get_named_gpio(np, "hp-det-gpio", 0);

	status = gpio_request(gpio_pin, "hp-det-gpio");

	if (status != 0) {
		pr_err("GPIO_CC request failed\n");
		return(-1);
	}

	g_hpd_data.hpd_gpio = gpio_pin;
	gpio_direction_input(g_hpd_data.hpd_gpio);
	g_hpd_data.hpd_irq_num = gpio_to_irq(g_hpd_data.hpd_gpio);

	status = request_irq(g_hpd_data.hpd_irq_num, (irq_handler_t)hpd_irq_handler,
			IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, HEADPHONE_DETECT_DEVICE, NULL);
	if (status < 0) {
		pr_err("request irq failed :%d\n", status);
		return(-1);
	}

#ifdef CONFIG_AMLOGIC_LEGACY_EARLY_SUSPEND
	register_early_suspend(&hdp_suspend_handler);
#endif

	return 0;
}

static void hpd_cleanup(void)
{
	free_irq(g_hpd_data.hpd_irq_num, NULL);
	gpio_free(g_hpd_data.hpd_gpio);
	device_destroy(hpd_class, g_hpd_data.hpdDev);
	class_destroy(hpd_class);
	unregister_chrdev(g_hpd_major, HEADPHONE_DETECT_DEVICE);
}

module_init(hpd_init);
module_exit(hpd_cleanup);

MODULE_AUTHOR("Sky");
MODULE_LICENSE("GPL");
