/*
 * Copyright 2008, Freescale Semiconductor, Inc
 * Andy Fleming
 *
 * Based vaguely on the Linux code
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <config.h>
#include <common.h>
#include <command.h>
#include <errno.h>
#include <mmc.h>
#include <part.h>
#include <malloc.h>
#include <linux/list.h>
#include <div64.h>
#include "mmc_private.h"
#include <asm/cpu_id.h>
#include <asm/arch/cpu_sdio.h>
#include <asm/arch/sd_emmc.h>

#include <emmc_partitions.h>
#ifdef CONFIG_STORE_COMPATIBLE
#include <partition_table.h>
#endif
#include <amlogic/secure_storage.h>

#define CONFIG_DISABLE_USER_WP
#define CONFIG_DISABLE_PERM_USER_WP 1
#define USER_WP_VALUE  ((1<<4)|(1<<6))
//#define CONFIG_MMC_TRACE 1

#define stamp_after(a, b) ((int)(b) - (int)(a) < 0)

#define KEY_BACKUP

static struct list_head mmc_devices;
static int cur_dev_num = -1;

int amlmmc_is_inited(void) {
	return device_boot_flag;
}

struct aml_pattern aml_pattern_table[] = {
	AML_PATTERN_ELEMENT(MMC_PATTERN_NAME, CALI_PATTERN),
	AML_PATTERN_ELEMENT(MMC_MAGIC_NAME, MAGIC_PATTERN),
	AML_PATTERN_ELEMENT(MMC_RANDOM_NAME, RANDOM_PATTERN),
};

#ifdef KEY_BACKUP
static struct aml_key_info key_infos[2] = { {0, 0, 0}, {0, 0, 0} };
#endif

bool emmckey_is_access_range_legal (struct mmc *mmc, ulong start, lbaint_t blkcnt) {
#ifdef CONFIG_STORE_COMPATIBLE
	ulong key_start_blk, key_end_blk;
	u64 key_glb_offset;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;
#ifdef KEY_BACKUP
	int cpy = KEY_COPIES;
#endif
#endif
	if (aml_is_emmc_tsd(mmc)) {
#ifdef CONFIG_STORE_COMPATIBLE
		vpart = aml_get_virtual_partition_by_name(MMC_KEY_NAME);
		part = aml_get_partition_by_name(MMC_RESERVED_NAME);
		key_glb_offset = part->offset + vpart->offset;
		key_start_blk = (key_glb_offset / MMC_BLOCK_SIZE);
#ifdef KEY_BACKUP
		key_end_blk = ((key_glb_offset + vpart->size * cpy) / MMC_BLOCK_SIZE - 1) + 2;
#else
		key_end_blk = ((key_glb_offset + vpart->size) / MMC_BLOCK_SIZE - 1);
#endif
		if (!(info_disprotect & DISPROTECT_KEY)) {
			if ((key_start_blk <= (start + blkcnt -1))
				&& (key_end_blk >= start)
				&& (blkcnt != start)) {
				printf("%s, keys %ld, keye %ld, start %ld, blkcnt %ld\n", __func__,
					key_start_blk, key_end_blk, start, blkcnt);
				printf("Emmckey: Access range is illegal!\n");
				return 0;
			}
		}
#endif
	}

	return 1;
}

__weak int board_mmc_getwp(struct mmc *mmc)
{
	return -1;
}

int mmc_getwp(struct mmc *mmc)
{
	int wp;

	wp = board_mmc_getwp(mmc);

	if (wp < 0) {
		if (mmc->cfg->ops->getwp)
			wp = mmc->cfg->ops->getwp(mmc);
		else
			wp = 0;
	}

	return wp;
}

__weak int board_mmc_getcd(struct mmc *mmc)
{
	return -1;
}

int mmc_send_cmd(struct mmc *mmc, struct mmc_cmd *cmd, struct mmc_data *data)
{
	int ret;

#ifdef CONFIG_MMC_TRACE
	int i;
	u8 *ptr;

	printf("CMD_SEND:%d\n", cmd->cmdidx);
	printf("\t\tARG\t\t\t 0x%08X\n", cmd->cmdarg);
	ret = mmc->cfg->ops->send_cmd(mmc, cmd, data);
	switch (cmd->resp_type) {
		case MMC_RSP_NONE:
			printf("\t\tMMC_RSP_NONE\n");
			break;
		case MMC_RSP_R1:
			printf("\t\tMMC_RSP_R1,5,6,7 \t 0x%08X \n",
				cmd->response[0]);
			break;
		case MMC_RSP_R1b:
			printf("\t\tMMC_RSP_R1b\t\t 0x%08X \n",
				cmd->response[0]);
			break;
		case MMC_RSP_R2:
			printf("\t\tMMC_RSP_R2\t\t 0x%08X \n",
				cmd->response[0]);
			printf("\t\t          \t\t 0x%08X \n",
				cmd->response[1]);
			printf("\t\t          \t\t 0x%08X \n",
				cmd->response[2]);
			printf("\t\t          \t\t 0x%08X \n",
				cmd->response[3]);
			printf("\n");
			printf("\t\t\t\t\tDUMPING DATA\n");
			for (i = 0; i < 4; i++) {
				int j;
				printf("\t\t\t\t\t%03d - ", i*4);
				ptr = (u8 *)&cmd->response[i];
				ptr += 3;
				for (j = 0; j < 4; j++)

					printf("%02X ", *ptr--);
				printf("\n");
			}
			break;
		case MMC_RSP_R3:
			printf("\t\tMMC_RSP_R3,4\t\t 0x%08X \n",
				cmd->response[0]);
			break;
		default:
			printf("\t\tERROR MMC rsp not supported\n");
			break;
	}
#else
		ret = mmc->cfg->ops->send_cmd(mmc, cmd, data);
#endif
	return ret;
}

int mmc_send_status(struct mmc *mmc, int timeout)
{
	struct mmc_cmd cmd;
	int err, retries = 5;
#ifdef CONFIG_MMC_TRACE
	int status;
#endif

	cmd.cmdidx = MMC_CMD_SEND_STATUS;
	cmd.resp_type = MMC_RSP_R1;
	if (!mmc_host_is_spi(mmc))
		cmd.cmdarg = mmc->rca << 16;

	do {
		err = mmc_send_cmd(mmc, &cmd, NULL);
		if (!err) {
			if ((cmd.response[0] & MMC_STATUS_RDY_FOR_DATA) &&
			    (cmd.response[0] & MMC_STATUS_CURR_STATE) !=
			     MMC_STATE_PRG)
				break;
			else if (cmd.response[0] & MMC_STATUS_MASK) {
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
				printf("Status Error: 0x%08X\n",
					cmd.response[0]);
#endif
				return COMM_ERR;
			}
		} else if (--retries < 0)
			return err;

		udelay(1000);

	} while (timeout--);

#ifdef CONFIG_MMC_TRACE
	status = (cmd.response[0] & MMC_STATUS_CURR_STATE) >> 9;
	printf("CURR STATE:%d\n", status);
#endif
	if (timeout <= 0) {
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
		printf("Timeout waiting card ready\n");
#endif
		return TIMEOUT;
	}
	if (cmd.response[0] & MMC_STATUS_SWITCH_ERROR)
		return SWITCH_ERR;

	return 0;
}

int mmc_set_blocklen(struct mmc *mmc, int len)
{
	struct mmc_cmd cmd;

	if (mmc->ddr_mode)
		return 0;

	cmd.cmdidx = MMC_CMD_SET_BLOCKLEN;
	cmd.resp_type = MMC_RSP_R1;
	cmd.cmdarg = len;

	return mmc_send_cmd(mmc, &cmd, NULL);
}

struct mmc *find_mmc_device(int dev_num)
{
	struct mmc *m;
	struct list_head *entry;

	list_for_each(entry, &mmc_devices) {
		m = list_entry(entry, struct mmc, link);
		if (m->block_dev.dev == dev_num)
			return m;
	}

#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
	printf("MMC Device %d not found\n", dev_num);
#endif

	return NULL;
}

static int mmc_read_blocks(struct mmc *mmc, void *dst, lbaint_t start,
			   lbaint_t blkcnt)
{
	struct mmc_cmd cmd;
	struct mmc_data data;
	int ret = 0, err = 0, err_flag = 0, retries = 0;

__RETRY:
#ifdef MMC_CMD23
	if (blkcnt > 1) {
		cmd.cmdidx = MMC_CMD_SET_BLOCK_COUNT;
		cmd.cmdarg = blkcnt & 0xFFFF;
		cmd.resp_type = MMC_RSP_R1;
		err = mmc_send_cmd(mmc, &cmd, NULL);
		if (err)
			printf("mmc set blkcnt failed\n");
	}
#endif

	if (blkcnt > 1)
		cmd.cmdidx = MMC_CMD_READ_MULTIPLE_BLOCK;
	else
		cmd.cmdidx = MMC_CMD_READ_SINGLE_BLOCK;

	if (mmc->high_capacity)
		cmd.cmdarg = start;
	else
		cmd.cmdarg = start * mmc->read_bl_len;

	cmd.resp_type = MMC_RSP_R1;

	data.dest = dst;
	data.blocks = blkcnt;
	data.blocksize = mmc->read_bl_len;
	data.flags = MMC_DATA_READ;

	ret = mmc_send_cmd(mmc, &cmd, &data);
#ifndef MMC_CMD23
	if (blkcnt > 1) {
		cmd.cmdidx = MMC_CMD_STOP_TRANSMISSION;
		cmd.cmdarg = 0;
		cmd.resp_type = MMC_RSP_R1b;
		err = mmc_send_cmd(mmc, &cmd, NULL);
		if (err) {
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
			printf("mmc fail to send stop cmd\n");
#endif
		}
	}
#endif
	if ((ret || err) && mmc->refix == 1)
		return 0;

	if (ret || err) {
		if (err_flag == 0) {
			err_flag = 1;
			retries = 5;
		}
		if (retries) {
			printf("retry read, count: %d\n", retries);
			retries--;
			goto __RETRY;
		}
		printf("retry read error !!!\n");
		return 0;
	}
	return blkcnt;
}

ulong mmc_bread(int dev_num, lbaint_t start, lbaint_t blkcnt, void *dst)
{
	lbaint_t cur, blocks_todo = blkcnt;

	if (blkcnt == 0)
		return 0;

	struct mmc *mmc = find_mmc_device(dev_num);
	if (!mmc)
		return 0;

	if ((start + blkcnt) > mmc->block_dev.lba) {
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
		printf("MMC: block number 0x" LBAF " exceeds max(0x" LBAF ")\n",
			start + blkcnt, mmc->block_dev.lba);
#endif
		return 0;
	}
	if (!emmckey_is_access_range_legal(mmc, start, blkcnt))
		return 0;
#ifndef MMC_HS400_MODE
	if (mmc_set_blocklen(mmc, mmc->read_bl_len))
		return 0;
#endif

	do {
		cur = (blocks_todo > mmc->cfg->b_max) ?
			mmc->cfg->b_max : blocks_todo;
		if(mmc_read_blocks(mmc, dst, start, cur) != cur)
			return 0;
		blocks_todo -= cur;
		start += cur;
		dst += cur * mmc->read_bl_len;
	} while (blocks_todo > 0);

	return blkcnt;
}

static int mmc_go_idle(struct mmc *mmc)
{
	struct mmc_cmd cmd;
	int err;

	udelay(1000);

	cmd.cmdidx = MMC_CMD_GO_IDLE_STATE;
	cmd.cmdarg = 0;
	cmd.resp_type = MMC_RSP_NONE;

	err = mmc_send_cmd(mmc, &cmd, NULL);

	if (err)
		return err;

	udelay(2000);

	return 0;
}

static int sd_send_op_cond(struct mmc *mmc)
{
	int timeout = 1000;
	int err;
	struct mmc_cmd cmd;

	do {
		cmd.cmdidx = MMC_CMD_APP_CMD;
		cmd.resp_type = MMC_RSP_R1;
		cmd.cmdarg = 0;

		err = mmc_send_cmd(mmc, &cmd, NULL);

		if (err)
			return err;

		cmd.cmdidx = SD_CMD_APP_SEND_OP_COND;
		cmd.resp_type = MMC_RSP_R3;

		/*
		 * Most cards do not answer if some reserved bits
		 * in the ocr are set. However, Some controller
		 * can set bit 7 (reserved for low voltages), but
		 * how to manage low voltages SD card is not yet
		 * specified.
		 */
		cmd.cmdarg = mmc_host_is_spi(mmc) ? 0 :
			(mmc->cfg->voltages & 0xff8000);

		if (mmc->version == SD_VERSION_2)
			cmd.cmdarg |= OCR_HCS;

		err = mmc_send_cmd(mmc, &cmd, NULL);

		if (err)
			return err;

		udelay(1000);
	} while ((!(cmd.response[0] & OCR_BUSY)) && timeout--);

	if (timeout <= 0)
		return UNUSABLE_ERR;

	if (mmc->version != SD_VERSION_2)
		mmc->version = SD_VERSION_1_0;

	if (mmc_host_is_spi(mmc)) { /* read OCR for spi */
		cmd.cmdidx = MMC_CMD_SPI_READ_OCR;
		cmd.resp_type = MMC_RSP_R3;
		cmd.cmdarg = 0;

		err = mmc_send_cmd(mmc, &cmd, NULL);

		if (err)
			return err;
	}

	mmc->ocr = cmd.response[0];

	mmc->high_capacity = ((mmc->ocr & OCR_HCS) == OCR_HCS);
	mmc->rca = 0;

	return 0;
}

/* We pass in the cmd since otherwise the init seems to fail */
static int mmc_send_op_cond_iter(struct mmc *mmc, struct mmc_cmd *cmd,
		int use_arg)
{
	int err;

	cmd->cmdidx = MMC_CMD_SEND_OP_COND;
	cmd->resp_type = MMC_RSP_R3;
	cmd->cmdarg = 0;
	if (use_arg && !mmc_host_is_spi(mmc)) {
		cmd->cmdarg =
			(mmc->cfg->voltages &
			(mmc->op_cond_response & OCR_VOLTAGE_MASK)) |
			(mmc->op_cond_response & OCR_ACCESS_MODE);

		if (mmc->cfg->host_caps & MMC_MODE_HC)
			cmd->cmdarg |= OCR_HCS;
	}
	err = mmc_send_cmd(mmc, cmd, NULL);
	if (err)
		return err;
	mmc->op_cond_response = cmd->response[0];
	return 0;
}

static int mmc_send_op_cond(struct mmc *mmc)
{
//	struct mmc_cmd cmd;
//	int err, i;

	/* Some cards seem to need this */
	mmc_go_idle(mmc);

 	/* Asking to the card its capabilities */
	mmc->op_cond_pending = 1;

	return IN_PROGRESS;
/*
 *	for (i = 0; i < 2; i++) {
 *		err = mmc_send_op_cond_iter(mmc, &cmd, i != 0);
 *		if (err)
 *			return err;
 *
 *		if (mmc->op_cond_response & OCR_BUSY)
 *			return 0;
 *	}
 *	return IN_PROGRESS;
 */
}

static int mmc_complete_op_cond(struct mmc *mmc)
{
	struct mmc_cmd cmd;
	int timeout = 1000;
	uint start;
	int err;

	mmc->op_cond_pending = 0;
	start = get_timer(0);
	do {
		err = mmc_send_op_cond_iter(mmc, &cmd, 1);
		if (err)
			return err;
		if (get_timer(start) > timeout)
			return UNUSABLE_ERR;
		udelay(100);
	} while (!(mmc->op_cond_response & OCR_BUSY));

	if (mmc_host_is_spi(mmc)) { /* read OCR for spi */
		cmd.cmdidx = MMC_CMD_SPI_READ_OCR;
		cmd.resp_type = MMC_RSP_R3;
		cmd.cmdarg = 0;

		err = mmc_send_cmd(mmc, &cmd, NULL);

		if (err)
			return err;
	}

	mmc->version = MMC_VERSION_UNKNOWN;
	mmc->ocr = cmd.response[0];

	mmc->high_capacity = ((mmc->ocr & OCR_HCS) == OCR_HCS);
	mmc->rca = 1;

	return 0;
}


static int mmc_send_ext_csd(struct mmc *mmc, u8 *ext_csd)
{
	struct mmc_cmd cmd;
	struct mmc_data data;
	int err;

	/* Get the Card Status Register */
	cmd.cmdidx = MMC_CMD_SEND_EXT_CSD;
	cmd.resp_type = MMC_RSP_R1;
	cmd.cmdarg = 0;

	data.dest = (char *)ext_csd;
	data.blocks = 1;
	data.blocksize = MMC_MAX_BLOCK_LEN;
	data.flags = MMC_DATA_READ;

	err = mmc_send_cmd(mmc, &cmd, &data);

	return err;
}

#ifdef CONFIG_DISABLE_USER_WP
int mmc_switch_by_bit(struct mmc *mmc, u8 set, u8 index, u8 value)
{
	struct mmc_cmd cmd;
	int timeout = 1000;
	int ret;

	cmd.cmdidx = MMC_CMD_SWITCH;
	cmd.resp_type = MMC_RSP_R1b;
	cmd.cmdarg = (MMC_SWITCH_MODE_SET_BITS << 24) |
						(index << 16) |
						(value << 8);

	ret = mmc_send_cmd(mmc, &cmd, NULL);

	/* Waiting for the ready status */
	if (!ret)
		ret = mmc_send_status(mmc, timeout);

	return ret;
}
#endif

int mmc_switch(struct mmc *mmc, u8 set, u8 index, u8 value)
{
	struct mmc_cmd cmd;
	int timeout = 1000;
	int ret;

	cmd.cmdidx = MMC_CMD_SWITCH;
	cmd.resp_type = MMC_RSP_R1b;
	cmd.cmdarg = (MMC_SWITCH_MODE_WRITE_BYTE << 24) |
				 (index << 16) |
				 (value << 8);

	ret = mmc_send_cmd(mmc, &cmd, NULL);

	/* Waiting for the ready status */
	if (!ret)
		ret = mmc_send_status(mmc, timeout);

	return ret;

}

int mmc_hwpart_config(struct mmc *mmc,
		      const struct mmc_hwpart_conf *conf,
		      enum mmc_hwpart_conf_mode mode)
{
	u8 part_attrs = 0;
	u32 enh_size_mult;
	u32 enh_start_addr;
	u32 gp_size_mult[4];
	u32 max_enh_size_mult;
	u32 tot_enh_size_mult = 0;
	u8 wr_rel_set;
	int i, pidx, err;
	ALLOC_CACHE_ALIGN_BUFFER(u8, ext_csd, MMC_MAX_BLOCK_LEN);

	if (mode < MMC_HWPART_CONF_CHECK || mode > MMC_HWPART_CONF_COMPLETE)
		return -EINVAL;

	if (IS_SD(mmc) || (mmc->version < MMC_VERSION_4_41)) {
		printf("eMMC >= 4.4 required for enhanced user data area\n");
		return -EMEDIUMTYPE;
	}

	if (!(mmc->part_support & PART_SUPPORT)) {
		printf("Card does not support partitioning\n");
		return -EMEDIUMTYPE;
	}

	if (!mmc->hc_wp_grp_size) {
		printf("Card does not define HC WP group size\n");
		return -EMEDIUMTYPE;
	}

	/* check partition alignment and total enhanced size */
	if (conf->user.enh_size) {
		if (conf->user.enh_size % mmc->hc_wp_grp_size ||
		    conf->user.enh_start % mmc->hc_wp_grp_size) {
			printf("User data enhanced area not HC WP group "
			       "size aligned\n");
			return -EINVAL;
		}
		part_attrs |= EXT_CSD_ENH_USR;
		enh_size_mult = conf->user.enh_size / mmc->hc_wp_grp_size;
		if (mmc->high_capacity) {
			enh_start_addr = conf->user.enh_start;
		} else {
			enh_start_addr = (conf->user.enh_start << 9);
		}
	} else {
		enh_size_mult = 0;
		enh_start_addr = 0;
	}
	tot_enh_size_mult += enh_size_mult;

	for (pidx = 0; pidx < 4; pidx++) {
		if (conf->gp_part[pidx].size % mmc->hc_wp_grp_size) {
			printf("GP%i partition not HC WP group size "
			       "aligned\n", pidx+1);
			return -EINVAL;
		}
		gp_size_mult[pidx] = conf->gp_part[pidx].size / mmc->hc_wp_grp_size;
		if (conf->gp_part[pidx].size && conf->gp_part[pidx].enhanced) {
			part_attrs |= EXT_CSD_ENH_GP(pidx);
			tot_enh_size_mult += gp_size_mult[pidx];
		}
	}

	if (part_attrs && ! (mmc->part_support & ENHNCD_SUPPORT)) {
		printf("Card does not support enhanced attribute\n");
		return -EMEDIUMTYPE;
	}

	err = mmc_send_ext_csd(mmc, ext_csd);
	if (err)
		return err;

	max_enh_size_mult =
		(ext_csd[EXT_CSD_MAX_ENH_SIZE_MULT+2] << 16) +
		(ext_csd[EXT_CSD_MAX_ENH_SIZE_MULT+1] << 8) +
		ext_csd[EXT_CSD_MAX_ENH_SIZE_MULT];
	if (tot_enh_size_mult > max_enh_size_mult) {
		printf("Total enhanced size exceeds maximum (%u > %u)\n",
		       tot_enh_size_mult, max_enh_size_mult);
		return -EMEDIUMTYPE;
	}

	/* The default value of EXT_CSD_WR_REL_SET is device
	 * dependent, the values can only be changed if the
	 * EXT_CSD_HS_CTRL_REL bit is set. The values can be
	 * changed only once and before partitioning is completed. */
	wr_rel_set = ext_csd[EXT_CSD_WR_REL_SET];
	if (conf->user.wr_rel_change) {
		if (conf->user.wr_rel_set)
			wr_rel_set |= EXT_CSD_WR_DATA_REL_USR;
		else
			wr_rel_set &= ~EXT_CSD_WR_DATA_REL_USR;
	}
	for (pidx = 0; pidx < 4; pidx++) {
		if (conf->gp_part[pidx].wr_rel_change) {
			if (conf->gp_part[pidx].wr_rel_set)
				wr_rel_set |= EXT_CSD_WR_DATA_REL_GP(pidx);
			else
				wr_rel_set &= ~EXT_CSD_WR_DATA_REL_GP(pidx);
		}
	}

	if (wr_rel_set != ext_csd[EXT_CSD_WR_REL_SET] &&
	    !(ext_csd[EXT_CSD_WR_REL_PARAM] & EXT_CSD_HS_CTRL_REL)) {
		puts("Card does not support host controlled partition write "
		     "reliability settings\n");
		return -EMEDIUMTYPE;
	}

	if (ext_csd[EXT_CSD_PARTITION_SETTING] &
	    EXT_CSD_PARTITION_SETTING_COMPLETED) {
		printf("Card already partitioned\n");
		puts("\tUser Enhanced Start: ");
		u64 temp,  j;
		for (j = 0, temp = 0; j < 4; j++) {
			temp |= ext_csd[EXT_CSD_ENH_START_ADDR + 3 - j];
			temp = temp << 8;
		}
		print_size(temp << 9, "\n");
		puts("\tUser Enhanced Size: ");
		for (j = 0, temp = 0; j < 3; j++) {
			temp |= ext_csd[EXT_CSD_ENH_SIZE_MULT + 2 - j];
			temp = temp << 8;
		}
		temp *= mmc->hc_wp_grp_size;
		print_size(temp << 9, "\n");
		return -EPERM;
	}

	if (mode == MMC_HWPART_CONF_CHECK)
		return 0;

	/* Partitioning requires high-capacity size definitions */
	if (!(ext_csd[EXT_CSD_ERASE_GROUP_DEF] & 0x01)) {
		err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
				 EXT_CSD_ERASE_GROUP_DEF, 1);

		if (err)
			return err;

		ext_csd[EXT_CSD_ERASE_GROUP_DEF] = 1;

		mmc->erase_grp_size =
			ext_csd[EXT_CSD_HC_ERASE_GRP_SIZE] * 1024;

	}

	/* all OK, write the configuration */
	for (i = 0; i < 4; i++) {
		err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
				 EXT_CSD_ENH_START_ADDR+i,
				 (enh_start_addr >> (i*8)) & 0xFF);
		if (err)
			return err;
	}
	for (i = 0; i < 3; i++) {
		err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
				 EXT_CSD_ENH_SIZE_MULT+i,
				 (enh_size_mult >> (i*8)) & 0xFF);
		if (err)
			return err;
	}
	for (pidx = 0; pidx < 4; pidx++) {
		for (i = 0; i < 3; i++) {
			err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
					 EXT_CSD_GP_SIZE_MULT+pidx*3+i,
					 (gp_size_mult[pidx] >> (i*8)) & 0xFF);
			if (err)
				return err;
		}
	}
	err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
			 EXT_CSD_PARTITIONS_ATTRIBUTE, part_attrs);
	if (err)
		return err;

	if (mode == MMC_HWPART_CONF_SET)
		return 0;

	/* The WR_REL_SET is a write-once register but shall be
	 * written before setting PART_SETTING_COMPLETED. As it is
	 * write-once we can only write it when completing the
	 * partitioning. */
	if (wr_rel_set != ext_csd[EXT_CSD_WR_REL_SET]) {
		err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
				 EXT_CSD_WR_REL_SET, wr_rel_set);
		if (err)
			return err;
	}

	/* Setting PART_SETTING_COMPLETED confirms the partition
	 * configuration but it only becomes effective after power
	 * cycle, so we do not adjust the partition related settings
	 * in the mmc struct. */

	err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
			 EXT_CSD_PARTITION_SETTING,
			 EXT_CSD_PARTITION_SETTING_COMPLETED);
	if (err)
		return err;

	return 0;
}

u8 ext_csd_w[] = {191, 187, 185, 183, 179, 178, 177, 175,
					173, 171, 169, 167, 165, 164, 163, 162,
					161, 156, 155, 143, 140, 136, 134, 133,
					132, 131, 62, 59, 56, 52, 37, 34,
					33, 32, 31, 30, 29, 22, 17, 16, 15};

int mmc_set_ext_csd(struct mmc *mmc, u8 index, u8 value)
{
	int ret = SWITCH_ERR, i;

	for (i = 0; i < sizeof(ext_csd_w); i++) {
		if (ext_csd_w[i] == index)
			break;
	}
	if (i != sizeof(ext_csd_w))
		ret = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, index, value);

	return ret;
}

int mmc_get_ext_csd(struct mmc *mmc, u8 *ext_csd)
{
	return mmc_send_ext_csd(mmc, ext_csd);
}

static int mmc_change_freq(struct mmc *mmc)
{
	ALLOC_CACHE_ALIGN_BUFFER(u8, ext_csd, MMC_MAX_BLOCK_LEN);
	char cardtype;
	int err;

	mmc->card_caps = MMC_MODE_4BIT | MMC_MODE_8BIT;

	if (mmc_host_is_spi(mmc))
		return 0;

	/* Only version 4 supports high-speed */
	if (mmc->version < MMC_VERSION_4)
		return 0;

	err = mmc_send_ext_csd(mmc, ext_csd);

	if (err)
		return err;

	cardtype = ext_csd[EXT_CSD_CARD_TYPE] & 0xf;

	err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, EXT_CSD_HS_TIMING, 1);

	if (err)
		return err == SWITCH_ERR ? 0 : err;

	/* Now check to see that it worked */
	err = mmc_send_ext_csd(mmc, ext_csd);

	if (err)
		return err;

	/* No high-speed support */
	if (!ext_csd[EXT_CSD_HS_TIMING])
		return 0;

	/* High Speed is set, there are two types: 52MHz and 26MHz */
	if (cardtype & EXT_CSD_CARD_TYPE_52) {
		if (cardtype & EXT_CSD_CARD_TYPE_DDR_1_8V)
			mmc->card_caps |= MMC_MODE_DDR_52MHz;
		mmc->card_caps |= MMC_MODE_HS_52MHz | MMC_MODE_HS;
	} else {
		mmc->card_caps |= MMC_MODE_HS;
	}

	return 0;
}

static int mmc_set_capacity(struct mmc *mmc, int part_num)
{
	switch (part_num) {
	case 0:
		mmc->capacity = mmc->capacity_user;
		break;
	case 1:
	case 2:
		mmc->capacity = mmc->capacity_boot;
		break;
	case 3:
		mmc->capacity = mmc->capacity_rpmb;
		break;
	case 4:
	case 5:
	case 6:
	case 7:
		mmc->capacity = mmc->capacity_gp[part_num - 4];
		break;
	default:
		return -1;
	}

	mmc->block_dev.lba = lldiv(mmc->capacity, mmc->read_bl_len);

	return 0;
}

int mmc_select_hwpart(int dev_num, int hwpart)
{
	struct mmc *mmc = find_mmc_device(dev_num);
	int ret;

	if (!mmc)
		return -ENODEV;

	if (mmc->part_num == hwpart)
		return 0;

	if (mmc->part_config == MMCPART_NOAVAILABLE) {
		printf("Card doesn't support part_switch\n");
		return -EMEDIUMTYPE;
	}

	ret = mmc_switch_part(dev_num, hwpart);
	if (ret)
		return ret;

	mmc->part_num = hwpart;

	return 0;
}

int mmc_switch_part(int dev_num, unsigned int part_num)
{
	struct mmc *mmc = find_mmc_device(dev_num);
	int ret;

	if (!mmc)
		return -1;

	ret = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, EXT_CSD_PART_CONF,
			 (mmc->part_config & ~PART_ACCESS_MASK)
			 | (part_num & PART_ACCESS_MASK));

	/*
	 * Set the capacity if the switch succeeded or was intended
	 * to return to representing the raw device.
	 */
	if ((ret == 0) || ((ret == -ENODEV) && (part_num == 0)))
		ret = mmc_set_capacity(mmc, part_num);

	return ret;
}

int mmc_getcd(struct mmc *mmc)
{
	int cd;

	cd = board_mmc_getcd(mmc);

	if (cd < 0) {
		if (mmc->cfg->ops->getcd)
			cd = mmc->cfg->ops->getcd(mmc);
		else
			cd = 1;
	}

	return cd;
}

static int sd_switch(struct mmc *mmc, int mode, int group, u8 value, u8 *resp)
{
	struct mmc_cmd cmd;
	struct mmc_data data;

	/* Switch the frequency */
	cmd.cmdidx = SD_CMD_SWITCH_FUNC;
	cmd.resp_type = MMC_RSP_R1;
	cmd.cmdarg = (mode << 31) | 0xffffff;
	cmd.cmdarg &= ~(0xf << (group * 4));
	cmd.cmdarg |= value << (group * 4);

	data.dest = (char *)resp;
	data.blocksize = 64;
	data.blocks = 1;
	data.flags = MMC_DATA_READ;

	return mmc_send_cmd(mmc, &cmd, &data);
}


static int sd_change_freq(struct mmc *mmc)
{
	int err;
	struct mmc_cmd cmd;
	ALLOC_CACHE_ALIGN_BUFFER(uint, scr, 2);
	ALLOC_CACHE_ALIGN_BUFFER(uint, switch_status, 16);
	struct mmc_data data;
	int timeout;

	mmc->card_caps = 0;

	if (mmc_host_is_spi(mmc))
		return 0;

	/* Read the SCR to find out if this card supports higher speeds */
	cmd.cmdidx = MMC_CMD_APP_CMD;
	cmd.resp_type = MMC_RSP_R1;
	cmd.cmdarg = mmc->rca << 16;

	err = mmc_send_cmd(mmc, &cmd, NULL);

	if (err)
		return err;

	cmd.cmdidx = SD_CMD_APP_SEND_SCR;
	cmd.resp_type = MMC_RSP_R1;
	cmd.cmdarg = 0;

	timeout = 3;

retry_scr:
	data.dest = (char *)scr;
	data.blocksize = 8;
	data.blocks = 1;
	data.flags = MMC_DATA_READ;

	err = mmc_send_cmd(mmc, &cmd, &data);

	if (err) {
		if (timeout--)
			goto retry_scr;

		return err;
	}

	mmc->scr[0] = __be32_to_cpu(scr[0]);
	mmc->scr[1] = __be32_to_cpu(scr[1]);

	switch ((mmc->scr[0] >> 24) & 0xf) {
		case 0:
			mmc->version = SD_VERSION_1_0;
			break;
		case 1:
			mmc->version = SD_VERSION_1_10;
			break;
		case 2:
			mmc->version = SD_VERSION_2;
			if ((mmc->scr[0] >> 15) & 0x1)
				mmc->version = SD_VERSION_3;
			break;
		default:
			mmc->version = SD_VERSION_1_0;
			break;
	}

	if (mmc->scr[0] & SD_DATA_4BIT)
		mmc->card_caps |= MMC_MODE_4BIT;

	/* Version 1.0 doesn't support switching */
	if (mmc->version == SD_VERSION_1_0)
		return 0;

	timeout = 4;
	while (timeout--) {
		err = sd_switch(mmc, SD_SWITCH_CHECK, 0, 1,
				(u8 *)switch_status);

		if (err)
			return err;

		/* The high-speed function is busy.  Try again */
		if (!(__be32_to_cpu(switch_status[7]) & SD_HIGHSPEED_BUSY))
			break;
	}

	/* If high-speed isn't supported, we return */
	if (!(__be32_to_cpu(switch_status[3]) & SD_HIGHSPEED_SUPPORTED))
		return 0;

	/*
	 * If the host doesn't support SD_HIGHSPEED, do not switch card to
	 * HIGHSPEED mode even if the card support SD_HIGHSPEED.
	 * This can avoid furthur problem when the card runs in different
	 * mode between the host.
	 */
	if (!((mmc->cfg->host_caps & MMC_MODE_HS_52MHz) &&
		(mmc->cfg->host_caps & MMC_MODE_HS)))
		return 0;

	err = sd_switch(mmc, SD_SWITCH_SWITCH, 0, 1, (u8 *)switch_status);

	if (err)
		return err;

	if ((__be32_to_cpu(switch_status[4]) & 0x0f000000) == 0x01000000)
		mmc->card_caps |= MMC_MODE_HS;

	return 0;
}

/* frequency bases */
/* divided by 10 to be nice to platforms without floating point */
static const int fbase[] = {
	10000,
	100000,
	1000000,
	10000000,
};

/* Multiplier values for TRAN_SPEED.  Multiplied by 10 to be nice
 * to platforms without floating point.
 */
static const int multipliers[] = {
	0,	/* reserved */
	10,
	12,
	13,
	15,
	20,
	25,
	30,
	35,
	40,
	45,
	50,
	55,
	60,
	70,
	80,
};

static void mmc_set_ios(struct mmc *mmc)
{
	if (mmc->cfg->ops->set_ios)
		mmc->cfg->ops->set_ios(mmc);
}

int aml_emmc_refix(struct mmc *mmc)
{
	int ret = 0;
	mmc->refix = 1;
	if (0)
		mmc->cfg->ops->calibration(mmc);
	if (mmc->cfg->ops->calc_fixed_adj != NULL)
		mmc->cfg->ops->calc_fixed_adj(mmc);
	else if (mmc->cfg->ops->refix != NULL) {
		ret = mmc->cfg->ops->refix(mmc);
		if (!ret)
			printf("[%s] mmc refix success\n", __func__);
		else
			printf("[%s] mmc refix error\n", __func__);
	}
	mmc->refix = 0;
	return ret;
}

void mmc_set_clock(struct mmc *mmc, uint clock)
{
	if (clock > mmc->cfg->f_max)
		clock = mmc->cfg->f_max;

	if (clock < mmc->cfg->f_min)
		clock = mmc->cfg->f_min;

	mmc->clock = clock;

	mmc_set_ios(mmc);
}

void mmc_set_bus_width(struct mmc *mmc, uint width)
{
	mmc->bus_width = width;

	mmc_set_ios(mmc);
}

//#ifdef CONFIG_DISABLE_USER_WP
int mmc_disable_usr_wp(struct mmc *mmc, u8 *ext_csd)
{
	int err= 0;
	u8 cur_ext_csd_171 = 0;
	u8 user_wp_disable = 0;
	cur_ext_csd_171 = (u8)ext_csd[171];
	printf("original ext_csd[171] USE_WP field value is %02x\n",cur_ext_csd_171);
	user_wp_disable = (cur_ext_csd_171^USER_WP_VALUE)&USER_WP_VALUE;
	printf("user_wp_disable is %d\n",user_wp_disable);
	if (user_wp_disable) {
		err = mmc_switch_by_bit(mmc,MMC_SWITCH_MODE_SET_BITS,EXT_CSD_USER_WP,user_wp_disable);
		if (err)
		    printf("mmc_switch mistake when disable user_wp, err value is %d\n",err);

		err = mmc_get_ext_csd(mmc,ext_csd);
		if (err)
			printf("mmc get ext csd mistake in %d\n",err);
		else
			printf("ext_csd[171] USER_WP filed value after switch is %02x\n",ext_csd[171]);
	}
	else {
		printf("Disable bits in ext_csd[171] USE_WP field has been set to \"1\"\n");
	}
	return err;
}

static int mmc_startup(struct mmc *mmc)
{
	int err, i;
	uint mult, freq;
	u64 cmult, csize, capacity;
	struct mmc_cmd cmd;
	ALLOC_CACHE_ALIGN_BUFFER(u8, ext_csd, MMC_MAX_BLOCK_LEN);
	ALLOC_CACHE_ALIGN_BUFFER(u8, test_csd, MMC_MAX_BLOCK_LEN);
	int timeout = 1000;

#ifdef CONFIG_MMC_SPI_CRC_ON
	if (mmc_host_is_spi(mmc)) { /* enable CRC check for spi */
		cmd.cmdidx = MMC_CMD_SPI_CRC_ON_OFF;
		cmd.resp_type = MMC_RSP_R1;
		cmd.cmdarg = 1;
		err = mmc_send_cmd(mmc, &cmd, NULL);

		if (err)
			return err;
	}
#endif

	/* Put the Card in Identify Mode */
	cmd.cmdidx = mmc_host_is_spi(mmc) ? MMC_CMD_SEND_CID :
		MMC_CMD_ALL_SEND_CID; /* cmd not supported in spi */
	cmd.resp_type = MMC_RSP_R2;
	cmd.cmdarg = 0;

	err = mmc_send_cmd(mmc, &cmd, NULL);

	if (err)
		return err;

	memcpy(mmc->cid, cmd.response, 16);

	/*
	 * For MMC cards, set the Relative Address.
	 * For SD cards, get the Relatvie Address.
	 * This also puts the cards into Standby State
	 */
	if (!mmc_host_is_spi(mmc)) { /* cmd not supported in spi */
		cmd.cmdidx = SD_CMD_SEND_RELATIVE_ADDR;
		cmd.cmdarg = mmc->rca << 16;
		cmd.resp_type = MMC_RSP_R6;

		err = mmc_send_cmd(mmc, &cmd, NULL);

		if (err)
			return err;

		if (IS_SD(mmc))
			mmc->rca = (cmd.response[0] >> 16) & 0xffff;
	}

	/* Get the Card-Specific Data */
	cmd.cmdidx = MMC_CMD_SEND_CSD;
	cmd.resp_type = MMC_RSP_R2;
	cmd.cmdarg = mmc->rca << 16;

	err = mmc_send_cmd(mmc, &cmd, NULL);

	/* Waiting for the ready status */
	mmc_send_status(mmc, timeout);

	if (err)
		return err;

	mmc->csd[0] = cmd.response[0];
	mmc->csd[1] = cmd.response[1];
	mmc->csd[2] = cmd.response[2];
	mmc->csd[3] = cmd.response[3];

	if (mmc->version == MMC_VERSION_UNKNOWN) {
		int version = (cmd.response[0] >> 26) & 0xf;

		switch (version) {
			case 0:
				mmc->version = MMC_VERSION_1_2;
				break;
			case 1:
				mmc->version = MMC_VERSION_1_4;
				break;
			case 2:
				mmc->version = MMC_VERSION_2_2;
				break;
			case 3:
				mmc->version = MMC_VERSION_3;
				break;
			case 4:
				mmc->version = MMC_VERSION_4;
				break;
			default:
				mmc->version = MMC_VERSION_1_2;
				break;
		}
	}

	/* divide frequency by 10, since the mults are 10x bigger */
	freq = fbase[(cmd.response[0] & 0x7)];
	mult = multipliers[((cmd.response[0] >> 3) & 0xf)];

	mmc->tran_speed = freq * mult;

	mmc->dsr_imp = ((cmd.response[1] >> 12) & 0x1);
	mmc->read_bl_len = 1 << ((cmd.response[1] >> 16) & 0xf);

	if (IS_SD(mmc))
		mmc->write_bl_len = mmc->read_bl_len;
	else
		mmc->write_bl_len = 1 << ((cmd.response[3] >> 22) & 0xf);

	if (mmc->high_capacity) {
		csize = (mmc->csd[1] & 0x3f) << 16
			| (mmc->csd[2] & 0xffff0000) >> 16;
		cmult = 8;
	} else {
		csize = (mmc->csd[1] & 0x3ff) << 2
			| (mmc->csd[2] & 0xc0000000) >> 30;
		cmult = (mmc->csd[2] & 0x00038000) >> 15;
	}

	mmc->capacity_user = (csize + 1) << (cmult + 2);
	mmc->capacity_user *= mmc->read_bl_len;
	mmc->capacity_boot = 0;
	mmc->capacity_rpmb = 0;
	for (i = 0; i < 4; i++)
		mmc->capacity_gp[i] = 0;

	if (mmc->read_bl_len > MMC_MAX_BLOCK_LEN)
		mmc->read_bl_len = MMC_MAX_BLOCK_LEN;

	if (mmc->write_bl_len > MMC_MAX_BLOCK_LEN)
		mmc->write_bl_len = MMC_MAX_BLOCK_LEN;

	if ((mmc->dsr_imp) && (0xffffffff != mmc->dsr)) {
		cmd.cmdidx = MMC_CMD_SET_DSR;
		cmd.cmdarg = (mmc->dsr & 0xffff) << 16;
		cmd.resp_type = MMC_RSP_NONE;
		if (mmc_send_cmd(mmc, &cmd, NULL))
			printf("MMC: SET_DSR failed\n");
	}

	/* Select the card, and put it into Transfer Mode */
	if (!mmc_host_is_spi(mmc)) { /* cmd not supported in spi */
		cmd.cmdidx = MMC_CMD_SELECT_CARD;
		cmd.resp_type = MMC_RSP_R1;
		cmd.cmdarg = mmc->rca << 16;
		err = mmc_send_cmd(mmc, &cmd, NULL);

		if (err)
			return err;
	}

	/*
	 * For SD, its erase group is always one sector
	 */
	mmc->erase_grp_size = 1;
	mmc->part_config = MMCPART_NOAVAILABLE;
	if (!IS_SD(mmc) && (mmc->version >= MMC_VERSION_4)) {
		/* check  ext_csd version and capacity */
		err = mmc_send_ext_csd(mmc, ext_csd);
		if (!err && (ext_csd[EXT_CSD_REV] >= 2)) {
			/*
			 * According to the JEDEC Standard, the value of
			 * ext_csd's capacity is valid if the value is more
			 * than 2GB
			 */
			capacity = (u64)(ext_csd[EXT_CSD_SEC_CNT] << 0
					| ext_csd[EXT_CSD_SEC_CNT + 1] << 8
					| ext_csd[EXT_CSD_SEC_CNT + 2] << 16
					| ext_csd[EXT_CSD_SEC_CNT + 3] << 24);
			capacity *= MMC_MAX_BLOCK_LEN;
			if ((capacity >> 20) > 2 * 1024)
				mmc->capacity_user = capacity;
		}

		switch (ext_csd[EXT_CSD_REV]) {
		case 1:
			mmc->version = MMC_VERSION_4_1;
			break;
		case 2:
			mmc->version = MMC_VERSION_4_2;
			break;
		case 3:
			mmc->version = MMC_VERSION_4_3;
			break;
		case 5:
			mmc->version = MMC_VERSION_4_41;
			break;
		case 6:
			mmc->version = MMC_VERSION_4_5;
			break;
		case 7:
			mmc->version = MMC_VERSION_5_0;
			break;
		case 8:
			mmc->version = MMC_VERSION_5_1;
			break;
		}

		/* dev life time estimate type A/B */
		mmc->dev_lifetime_est_typ_a
			= ext_csd[EXT_CSD_DEV_LIFETIME_EST_TYP_A];
		mmc->dev_lifetime_est_typ_b
			= ext_csd[EXT_CSD_DEV_LIFETIME_EST_TYP_B];

		/*
		 * Host needs to enable ERASE_GRP_DEF bit if device is
		 * partitioned. This bit will be lost every time after a reset
		 * or power off. This will affect erase size.
		 */
		if ((ext_csd[EXT_CSD_PARTITIONING_SUPPORT] & PART_SUPPORT) &&
		    (ext_csd[EXT_CSD_PARTITIONS_ATTRIBUTE] & PART_ENH_ATTRIB)) {
			err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
				EXT_CSD_ERASE_GROUP_DEF, 1);

			if (err)
				return err;
			else
				ext_csd[EXT_CSD_ERASE_GROUP_DEF] = 1;

			/* Read out group size from ext_csd */
			mmc->erase_grp_size =
				ext_csd[EXT_CSD_HC_ERASE_GRP_SIZE] * 1024;
			/*
			 * if high capacity and partition setting completed
			 * SEC_COUNT is valid even if it is smaller than 2 GiB
			 * JEDEC Standard JESD84-B45, 6.2.4
			 */
			if (mmc->high_capacity &&
			    (ext_csd[EXT_CSD_PARTITION_SETTING] &
			     EXT_CSD_PARTITION_SETTING_COMPLETED)) {
				capacity = (u64)((ext_csd[EXT_CSD_SEC_CNT]) |
					(ext_csd[EXT_CSD_SEC_CNT + 1] << 8) |
					(ext_csd[EXT_CSD_SEC_CNT + 2] << 16) |
					(ext_csd[EXT_CSD_SEC_CNT + 3] << 24));
				capacity *= MMC_MAX_BLOCK_LEN;
				mmc->capacity_user = capacity;
			}
		} else {
			/* Calculate the group size from the csd value. */
			int erase_gsz, erase_gmul;
			erase_gsz = (mmc->csd[2] & 0x00007c00) >> 10;
			erase_gmul = (mmc->csd[2] & 0x000003e0) >> 5;
			mmc->erase_grp_size = (erase_gsz + 1)
				* (erase_gmul + 1);
		}

		/* store the partition info of emmc */
		if ((ext_csd[EXT_CSD_PARTITIONING_SUPPORT] & PART_SUPPORT) ||
		    ext_csd[EXT_CSD_BOOT_MULT])
			mmc->part_config = ext_csd[EXT_CSD_PART_CONF];

		mmc->capacity_boot = ext_csd[EXT_CSD_BOOT_MULT] << 17;

		mmc->capacity_rpmb = ext_csd[EXT_CSD_RPMB_MULT] << 17;

		for (i = 0; i < 4; i++) {
			int idx = EXT_CSD_GP_SIZE_MULT + i * 3;
			mmc->capacity_gp[i] = (ext_csd[idx + 2] << 16) +
				(ext_csd[idx + 1] << 8) + ext_csd[idx];
			mmc->capacity_gp[i] *=
				ext_csd[EXT_CSD_HC_ERASE_GRP_SIZE];
			mmc->capacity_gp[i] *= ext_csd[EXT_CSD_HC_WP_GRP_SIZE];
		}
		mmc->hc_wp_grp_size = 1024
			* ext_csd[EXT_CSD_HC_ERASE_GRP_SIZE]
			* ext_csd[EXT_CSD_HC_WP_GRP_SIZE];

		mmc->part_support = ext_csd[EXT_CSD_PARTITIONING_SUPPORT];
	}

	err = mmc_set_capacity(mmc, mmc->part_num);
	if (err)
		return err;

	if (IS_SD(mmc))
		err = sd_change_freq(mmc);
	else
		err = mmc_change_freq(mmc);

	if (err)
		return err;

#ifdef CONFIG_DISABLE_USER_WP
//	mmc_disable_usr_wp(mmc,ext_csd);
#endif

	/* Restrict card's capabilities by what the host can do */
	mmc->card_caps &= mmc->cfg->host_caps;

	if (IS_SD(mmc)) {
		if (mmc->card_caps & MMC_MODE_4BIT) {
			cmd.cmdidx = MMC_CMD_APP_CMD;
			cmd.resp_type = MMC_RSP_R1;
			cmd.cmdarg = mmc->rca << 16;

			err = mmc_send_cmd(mmc, &cmd, NULL);
			if (err)
				return err;

			cmd.cmdidx = SD_CMD_APP_SET_BUS_WIDTH;
			cmd.resp_type = MMC_RSP_R1;
			cmd.cmdarg = 2;
			err = mmc_send_cmd(mmc, &cmd, NULL);
			if (err)
				return err;

			mmc_set_bus_width(mmc, 4);
		}

		if (mmc->card_caps & MMC_MODE_HS)
			mmc->tran_speed = 50000000;
		else
			mmc->tran_speed = 25000000;
	} else {
		int idx;

		/* An array of possible bus widths in order of preference */
		static unsigned ext_csd_bits[] = {
			EXT_CSD_DDR_BUS_WIDTH_8,
			EXT_CSD_DDR_BUS_WIDTH_4,
			EXT_CSD_BUS_WIDTH_8,
			EXT_CSD_BUS_WIDTH_4,
			EXT_CSD_BUS_WIDTH_1,
		};

		/* An array to map CSD bus widths to host cap bits */
		static unsigned ext_to_hostcaps[] = {
			[EXT_CSD_DDR_BUS_WIDTH_4] =
				MMC_MODE_DDR_52MHz | MMC_MODE_4BIT,
			[EXT_CSD_DDR_BUS_WIDTH_8] =
				MMC_MODE_DDR_52MHz | MMC_MODE_8BIT,
			[EXT_CSD_BUS_WIDTH_4] = MMC_MODE_4BIT,
			[EXT_CSD_BUS_WIDTH_8] = MMC_MODE_8BIT,
		};

		/* An array to map chosen bus width to an integer */
		static unsigned widths[] = {
			8, 4, 8, 4, 1,
		};

		for (idx=0; idx < ARRAY_SIZE(ext_csd_bits); idx++) {
			unsigned int extw = ext_csd_bits[idx];
			unsigned int caps = ext_to_hostcaps[extw];

			/*
			 * Check to make sure the card and controller support
			 * these capabilities
			 */
			if ((mmc->card_caps & caps) != caps)
				continue;

			err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL,
					EXT_CSD_BUS_WIDTH, extw);

			if (err)
				continue;

			mmc->ddr_mode = (caps & MMC_MODE_DDR_52MHz) ? 1 : 0;
			mmc_set_bus_width(mmc, widths[idx]);
			err = mmc_send_ext_csd(mmc, test_csd);

			if (err)
				continue;

			/* Only compare read only fields */
			if (ext_csd[EXT_CSD_PARTITIONING_SUPPORT]
				== test_csd[EXT_CSD_PARTITIONING_SUPPORT] &&
			    ext_csd[EXT_CSD_HC_WP_GRP_SIZE]
				== test_csd[EXT_CSD_HC_WP_GRP_SIZE] &&
			    ext_csd[EXT_CSD_REV]
				== test_csd[EXT_CSD_REV] &&
			    ext_csd[EXT_CSD_HC_ERASE_GRP_SIZE]
				== test_csd[EXT_CSD_HC_ERASE_GRP_SIZE] &&
			    memcmp(&ext_csd[EXT_CSD_SEC_CNT],
				   &test_csd[EXT_CSD_SEC_CNT], 4) == 0)
				break;
			else
				err = SWITCH_ERR;
		}

		if (err)
			return err;

		if (mmc->card_caps & MMC_MODE_HS) {
			if (mmc->card_caps & MMC_MODE_HS_52MHz)
				mmc->tran_speed = 52000000;
			else
				mmc->tran_speed = 26000000;
		}
	}

	mmc_set_clock(mmc, mmc->tran_speed);

	if (mmc->card_caps & MMC_MODE_HS_52MHz) {
		err = aml_emmc_refix(mmc);
		if (err)
			return err;
	}

	/* Fix the block length for DDR mode */
	if (mmc->ddr_mode) {
		mmc->read_bl_len = MMC_MAX_BLOCK_LEN;
		mmc->write_bl_len = MMC_MAX_BLOCK_LEN;
	}

	/* fill in device description */
	mmc->block_dev.lun = 0;
	mmc->block_dev.type = 0;
	mmc->block_dev.blksz = mmc->read_bl_len;
	mmc->block_dev.log2blksz = LOG2(mmc->block_dev.blksz);
	mmc->block_dev.lba = lldiv(mmc->capacity, mmc->read_bl_len);
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
	sprintf(mmc->block_dev.vendor, "Man %06x Snr %04x%04x",
		mmc->cid[0] >> 24, (mmc->cid[2] & 0xffff),
		(mmc->cid[3] >> 16) & 0xffff);
	sprintf(mmc->block_dev.product, "%c%c%c%c%c%c", mmc->cid[0] & 0xff,
		(mmc->cid[1] >> 24), (mmc->cid[1] >> 16) & 0xff,
		(mmc->cid[1] >> 8) & 0xff, mmc->cid[1] & 0xff,
		(mmc->cid[2] >> 24) & 0xff);
	sprintf(mmc->block_dev.revision, "%d.%d", (mmc->cid[2] >> 20) & 0xf,
		(mmc->cid[2] >> 16) & 0xf);
#else
	mmc->block_dev.vendor[0] = 0;
	mmc->block_dev.product[0] = 0;
	mmc->block_dev.revision[0] = 0;
#endif
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBDISK_SUPPORT)
	init_part(&mmc->block_dev);
#endif

	return 0;
}

static int mmc_send_if_cond(struct mmc *mmc)
{
	struct mmc_cmd cmd;
	int err;

	cmd.cmdidx = SD_CMD_SEND_IF_COND;
	/* We set the bit if the host supports voltages between 2.7 and 3.6 V */
	cmd.cmdarg = ((mmc->cfg->voltages & 0xff8000) != 0) << 8 | 0xaa;
	cmd.resp_type = MMC_RSP_R7;

	err = mmc_send_cmd(mmc, &cmd, NULL);

	if (err)
		return err;

	if ((cmd.response[0] & 0xff) != 0xaa)
		return UNUSABLE_ERR;
	else
		mmc->version = SD_VERSION_2;

	return 0;
}

/* not used any more */
int __deprecated mmc_register(struct mmc *mmc)
{
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
	printf("%s is deprecated! use mmc_create() instead.\n", __func__);
#endif
	return -1;
}

struct mmc *mmc_create(const struct mmc_config *cfg, void *priv)
{
	struct mmc *mmc;

	/* quick validation */
	if (cfg == NULL || cfg->ops == NULL || cfg->ops->send_cmd == NULL ||
			cfg->f_min == 0 || cfg->f_max == 0 || cfg->b_max == 0)
		return NULL;

	mmc = calloc(1, sizeof(*mmc));
	if (mmc == NULL)
		return NULL;

	mmc->cfg = cfg;
	mmc->priv = priv;

	/* the following chunk was mmc_register() */

	/* Setup dsr related values */
	mmc->dsr_imp = 0;
	mmc->dsr = 0xffffffff;
	/* Setup the universal parts of the block interface just once */
	mmc->block_dev.if_type = IF_TYPE_MMC;
	mmc->block_dev.dev = cur_dev_num++;
	mmc->block_dev.removable = 1;
	mmc->block_dev.block_read = mmc_bread;
	mmc->block_dev.block_write = mmc_bwrite;
	mmc->block_dev.block_erase = mmc_berase;

	/* setup initial part type */
	mmc->block_dev.part_type = mmc->cfg->part_type;

	INIT_LIST_HEAD(&mmc->link);

	list_add_tail(&mmc->link, &mmc_devices);

	return mmc;
}

void mmc_destroy(struct mmc *mmc)
{
	/* only freeing memory for now */
	free(mmc);
}

#ifdef CONFIG_PARTITIONS
block_dev_desc_t *mmc_get_dev(int dev)
{
	struct mmc *mmc = find_mmc_device(dev);
	if (!mmc || mmc_init(mmc))
		return NULL;

	return &mmc->block_dev;
}
#endif

/* board-specific MMC power initializations. */
__weak void board_mmc_power_init(void)
{
}

int mmc_start_init(struct mmc *mmc)
{
	int err;

	/* we pretend there's no card when init is NULL */
	if (mmc_getcd(mmc) == 0 || mmc->cfg->ops->init == NULL) {
		mmc->has_init = 0;
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
		printf("MMC: no card present\n");
#endif
		return NO_CARD_ERR;
	}

	if (mmc->has_init)
		return 0;

	board_mmc_power_init();

	/* made sure it's not NULL earlier */
	err = mmc->cfg->ops->init(mmc);

	if (err)
		return err;

	mmc->ddr_mode = 0;
	mmc_set_bus_width(mmc, 1);
	mmc_set_clock(mmc, 1);

	/* Reset the Card */
	err = mmc_go_idle(mmc);

	if (err)
		return err;

	/* The internal partition reset to user partition(0) at every CMD0*/
	mmc->part_num = 0;

	/* Test for SD version 2 */
	err = mmc_send_if_cond(mmc);

	/* Now try to get the SD card's operating condition */
	err = sd_send_op_cond(mmc);

	/* If the command timed out, we check for an MMC card */
	if (err == TIMEOUT) {
		err = mmc_send_op_cond(mmc);

		if (err && err != IN_PROGRESS) {
#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)
			printf("Card did not respond to voltage select!\n");
#endif
			return UNUSABLE_ERR;
		}
	}

	if (err == IN_PROGRESS)
		mmc->init_in_progress = 1;

	return err;
}

void mmc_write_cali_mattern(void *addr, struct aml_pattern *table)
{
	int i = 0;
	unsigned int s = 10;
	u32 *mattern = (u32 *)addr;
	struct virtual_partition *vpart =
		aml_get_virtual_partition_by_name(table->name);
	for (i = 0;i < (vpart->size)/4 - 1;i++) {
		if (!strcmp(table->name, "random"))
			mattern[i] = rand_r(&s);
		else
			mattern[i] = table->pattern;
	}
	mattern[i] = crc32(0, (u8 *)addr, (vpart->size - 4));
	return;
}

int mmc_pattern_check(struct mmc *mmc, struct aml_pattern *table)
{
	void *addr = NULL;
	int dev = EMMC_DTB_DEV;
	u64 cnt = 0, n = 0, blk = 0;
	u32 *buf = NULL;
	u32 crc32_s = 0;
	struct partitions *part = NULL;
	struct virtual_partition *vpart = NULL;
	vpart = aml_get_virtual_partition_by_name(MMC_KEY_NAME);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);

	addr = (void *)malloc(vpart->size);
	if (!addr) {
		printf("pattern malloc failed\n");
		return 1;
	}
	vpart = aml_get_virtual_partition_by_name(table->name);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	blk = (part->offset + vpart->offset) / mmc->read_bl_len;
	cnt = vpart->size / mmc->read_bl_len;
	n = mmc_bread(dev, blk, cnt, addr);
	if (n != cnt) {
		printf("read pattern failed\n");
		free(addr);
		return 1;
	} else {
		buf = (u32 *)addr;
		crc32_s = crc32(0, (u8 *)addr, (vpart->size - 4));
		if (crc32_s != buf[vpart->size/4 - 1]) {
			printf("check %s failed,need to write\n",
						table->name);
			mmc_write_cali_mattern(addr, table);
			n = mmc_bwrite(dev, blk, cnt, addr);
			printf("several 0x%x pattern blocks write %s\n",
				table->pattern, (n == cnt) ? "OK" : "ERROR");
		}
		printf("crc32_s:0x%x == storage crc_pattern:0x%x!!!\n",
				crc32_s, buf[vpart->size/4 - 1]);
	}
	free(addr);
	return (n == cnt) ? 0 : 1;
}

static int mmc_complete_init(struct mmc *mmc)
{
	int err = 0;

	if (mmc->op_cond_pending)
		err = mmc_complete_op_cond(mmc);

	if (!err)
		err = mmc_startup(mmc);
	if (err)
		mmc->has_init = 0;
	else
		mmc->has_init = 1;
	mmc->init_in_progress = 0;
	return err;
}

#ifdef MMC_WRITE_CHIP_ID
/* 1. read board chip_id
 * 2. read chip_id writed on emmc
 * 3. compare two chip_id
 * 4. compare vddee
 *	if same return
 *	else write chip_id or vddee on emmc
 */
int aml_write_chip_id(struct mmc *mmc)
{
	struct aml_card_sd_info *aml_priv = mmc->priv;
	struct tuning_para *parameter = aml_priv->para;
	struct tuning_para *para;
	uint8_t chip_id[16];
	void *buf;
	int same_para = 1;
	int i, blk, off;
	int n;
	int para_size;
	int size;
	unsigned int vddee = CONFIG_VDDEE_INIT_VOLTAGE;

	para_size = sizeof(struct tuning_para);
	blk = (para_size - 1) / 512 + 1;
	size = blk * 512;

	buf = (void *)malloc(size);
	if (!buf) {
		printf("buffer malloc failed\n");
		return 1;
	}
	memset(buf, 0, size);

	if (get_chip_id(&chip_id[0], sizeof(chip_id)) != 0) {
		printf("get chip id error\n");
		free(buf);
		return 1;
	}

	off = MMC_TUNING_OFFSET;
	n = mmc->block_dev.block_read(1, off, blk, buf);
	if (n == blk) {
		para = (struct tuning_para *)buf;
		for (i = 0; i < sizeof(chip_id); i++) {
			if (para->chip_id[i] != chip_id[i]) {
				same_para = 0;
				break;
			}
		}

		if (vddee != para->vddee)
			same_para = 0;

		if (same_para == 0) {
			printf("chip_id is 0x:");
			memset(para, 0, para_size);
			para->vddee = vddee;
			for (i = 0; i < sizeof(chip_id); i++) {
				para->chip_id[i] = chip_id[i];
				printf("%02x ", para->chip_id[i]);
			}
			printf("\n");
			mmc->block_dev.block_write(1, off, blk, buf);
		}
	}
	memcpy(parameter, buf, para_size);
	free(buf);
	return 0;
}
#endif

#ifdef MMC_HS400_MODE
void reset_all_reg(struct mmc *mmc)
{
	struct aml_card_sd_info *aml_priv = mmc->priv;
	struct sd_emmc_global_regs *sd_emmc_regs = aml_priv->sd_emmc_reg;
	void *buf;
	unsigned long  addr;
	int size;
	u64 writeval;
	unsigned long byte;

	if (aml_priv->sd_emmc_port != 2)
		return;
	sd_emmc_regs->gclock = 0x1000204;
	sd_emmc_regs->gcfg = 0x4791;
	sd_emmc_regs->gadjust = 0;
	sd_emmc_regs->gdelay = 0;
	sd_emmc_regs->gdelay1 = 0;
	sd_emmc_regs->gintf3 = 0;

	aml_priv->cfg.f_max = 40000000;

	size = 4;
	byte = size;
	addr = CLKSRC_BASE + 0x25c;
	writeval = 0x080;
	buf = map_sysmem(addr, byte);
	*((u32 *)buf) = (u32)writeval;
	unmap_sysmem(buf);
}

static long long para_checksum_calc(struct tuning_para *para)
{
	int i = 0;
	int size = sizeof(struct tuning_para) - 6 * sizeof(unsigned int);
	unsigned int *buffer;
	long long checksum = 0;

	size = size >> 2;
	buffer = (unsigned int *)para;
	while (i < size)
		checksum += buffer[i++];

	return checksum;
}

static int mmc_read_single_block(struct mmc *mmc, void *dst, lbaint_t start)
{
	struct mmc_cmd cmd;
	struct mmc_data data;
	int ret = 0;

	cmd.cmdidx = MMC_CMD_READ_SINGLE_BLOCK;

	if (mmc->high_capacity)
		cmd.cmdarg = start;
	else
		cmd.cmdarg = start * mmc->read_bl_len;

	cmd.resp_type = MMC_RSP_R1;

	data.dest = dst;
	data.blocks = 1;
	data.blocksize = mmc->read_bl_len;
	data.flags = MMC_DATA_READ;

	ret = mmc_send_cmd(mmc, &cmd, &data);
	return ret;
}

/*
 * check if tuning parameter is exist
 * check if temperature is in the 0~69
 * check if the parameter has been tuning
 *		under the current temperature
 * check if the data had been broken by  checksum
 *
 * if all four condition above is yes, the tuning parameter
 *		could be use directly
 * otherwise returning and save parameter
 */
int aml_para_is_exist(struct mmc *mmc)
{
	int temperature;
	int temp_index;
	long long checksum;
	struct aml_card_sd_info *aml_priv = mmc->priv;
	struct tuning_para *para = aml_priv->para;
	int ret;

	para->update = 1;
	ret = r1p1_temp_read(1, &temperature);
	if (ret == -1) {
		para->update = 0;
		printf("get temperature failed\n");
		return 0;
	}

	checksum = para_checksum_calc(para);
	if (checksum != para->checksum) {
		printf("warning: checksum is not match\n");
		return 0;
	}

	if (para->magic != TUNED_MAGIC) {
		printf("[%s] magic is not match\n", __func__);
		return 0;
	}

	if (para->version != TUNED_VERSION) {
		printf("[%s] VERSION is not match\n", __func__);
		return 0;
	}

	printf("current temperature is %d\n", temperature);
	temp_index = temperature  / 10;
	para->temperature = temperature;
	/* temperature range is 0 ~ 69 */
	if (temp_index < 0 || temp_index > 6) {
		printf("temperature is out of normal range\n");
		return 0;
	}

	if (para->hs4[temp_index].flag != TUNED_FLAG) {
		printf("current temperature %d degree not tuning yet\n",
			temperature / 10);
		return 0;
	}

	printf("the hs400 parameter is useful\n");
	para->update = 0;
	return 1;
}

/*
 * read tuning para from reserved partition
 * and copy it to pdata->para
 */
int aml_read_tuning_para(struct mmc *mmc)
{
	struct aml_card_sd_info *aml_priv = mmc->priv;
	struct tuning_para *para = aml_priv->para;
	int offset, blk;
	int ret;
	int para_size;
	void *addr;

	para_size = sizeof(struct tuning_para);

	blk = (para_size - 1) / 512 + 1;
	offset = MMC_TUNING_OFFSET;
	addr = malloc(blk * 512);
	if (!addr)
		return -1;

	memset(addr, 0, blk * 512);

	if (blk == 1)
		ret = mmc_read_single_block(mmc, addr, offset);
	else
		ret = mmc_bread(1, offset, blk, addr);
	if (ret) {
		printf("read tuning parameter failed\n");
		free(addr);
		return ret;
	}
	memcpy(para, addr, para_size);
	free(addr);
	return ret;
}
#endif

int mmc_init(struct mmc *mmc)
{
	struct aml_card_sd_info *aml_priv = mmc->priv;
	int err = IN_PROGRESS, i;
	unsigned start;

	if (mmc->has_init)
		return 0;
#ifdef MMC_HS400_MODE
	reset_all_reg(mmc);
#endif

	start = get_timer(0);

	if (!mmc->init_in_progress)
		err = mmc_start_init(mmc);

	if (!err || err == IN_PROGRESS)
		err = mmc_complete_init(mmc);
	debug("%s: %d, time %lu\n", __func__, err, get_timer(start));
	if (err)
		return err;
	printf("[%s] mmc init success\n", __func__);
	if (mmc->block_dev.dev == CONFIG_SYS_MMC_ENV_DEV)  {
		device_boot_flag = EMMC_BOOT_FLAG;
		secure_storage_set_info(STORAGE_DEV_EMMC);

	}
#ifdef CONFIG_STORE_COMPATIBLE
	info_disprotect |= DISPROTECT_KEY;
	if (aml_is_emmc_tsd(mmc)) { // eMMC OR TSD
		if (!is_partition_checked) {
			if (mmc_device_init(mmc) == 0) {
				is_partition_checked = true;
				printf("eMMC/TSD partition table have been checked OK!\n");
				for (i = 0; i < ARRAY_SIZE(aml_pattern_table); i++)
					mmc_pattern_check(mmc, &aml_pattern_table[i]);
			}
		}
	}
	info_disprotect &= ~DISPROTECT_KEY;

#endif

	if (aml_priv->sd_emmc_port != 2)
		return err;

#ifdef MMC_WRITE_CHIP_ID
	err = aml_write_chip_id(mmc);
	if (err)
		printf("write chip id and vddee failed\n");
#endif

#ifdef MMC_HS400_MODE
	err = aml_read_tuning_para(mmc);
	if (err)
		printf("read tuning parameter failed\n");

#ifdef MMC_SET_TUNING_PARA
	if (aml_para_is_exist(mmc) == 0)
		return err;
#endif

	err = mmc_set_hs200_mode(mmc);
	if (err) {
		printf("set hs200 mode failed\n");
		return err;
	}
	err = mmc_set_hs400_mode(mmc);
	if (err) {
		printf("set hs400 mode failed\n");
		return err;
	}
#endif

	return err;
}

int mmc_ffu_op(int dev, u64 ffu_ver, void *addr, u64 cnt)
{
	int err, i, supported_modes, fw_cfg, ffu_status;
	u64 fw_ver = 0, n;
	u8 ext_csd_ffu[512] = {0};
	lbaint_t ffu_addr=0;
	struct mmc *mmc = find_mmc_device(dev);
	if (!mmc)
		return -ENODEV;

	printf("ffu update start\n");
	/* check Manufacturer MID */
	if ((mmc->cid[0] >> 24) == SAMSUNG_MID) {
		ffu_addr = SAMSUNG_FFU_ADDR;
	} else if ((mmc->cid[0] >> 24) == KINGSTON_MID) {
		ffu_addr = KINGSTON_FFU_ADDR;
	} else if ((mmc->cid[0] >> 24) == BIWIN_MID) {
		ffu_addr = BIWIN_FFU_ADDR;
	} else if ((mmc->cid[0] >> 24) == TOSHIBA_MID) {
		ffu_addr = TOSHIBA_FFU_ADDR;
	} else {
		printf("FFU update for this manufacturer not support yet\n");
		return -1;
	}

	/*
	 * check FFU Supportability
	 * check FFU Prohibited or not
	 * check current firmware version
	 */
	memset(ext_csd_ffu, 0, 512);
	err = mmc_get_ext_csd(mmc, ext_csd_ffu);
	if (err)
		return err;

	supported_modes = ext_csd_ffu[EXT_CSD_SUPPORTED_MODES] & 0x1;
	fw_cfg = ext_csd_ffu[EXT_CSD_FW_CFG] & 0x1;
	for (i = 0; i < 8; i++) {
		fw_ver |= ext_csd_ffu[EXT_CSD_FW_VERSION + 7 - i];
		if (i < 7)
			fw_ver <<= 8;
	}
	if ((mmc->cid[0] >> 24) == BIWIN_MID)
		fw_ver = ((fw_ver >> 16) & 0xffffffff);
	printf("old fw_ver = %llx\n", fw_ver);
	if (!supported_modes || fw_cfg || (fw_ver >= ffu_ver))
		return -1;

	/* Set FFU Mode */
	err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, EXT_CSD_MODE_CFG, 1);
	if (err) {
		printf("Failed: set FFU mode\n");
		return err;
	}

	/* Write patch file at one write command */
	n = mmc_ffu_write(dev, ffu_addr, cnt, addr);
	if (n != cnt) {
		printf("target is %llx block, but only %llx block has been write\n", cnt, n);
		return -1;
	}

	memset(ext_csd_ffu, 0, 512);
	err = mmc_get_ext_csd(mmc, ext_csd_ffu);
	if (err)
		return err;

	for (i = 0; i < 8; i++) {
		fw_ver |= ext_csd_ffu[EXT_CSD_FW_VERSION + 7 - i];
		if (i < 7)
			fw_ver <<= 8;
	}
	if ((mmc->cid[0] >> 24) == BIWIN_MID)
		fw_ver = ((fw_ver >> 16) & 0xffffffff);
	printf("new fw_ver = %llx\n", fw_ver);
	if ((mmc->cid[0] >> 24) == SAMSUNG_MID) {
		/* Set Normal Mode */
		err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, EXT_CSD_MODE_CFG, 0);
		if (err)
			return err;
	}

	/* reset devices */
	err = mmc_go_idle(mmc);
	if (err)
		return err;

	/* Initialization */
	mmc->has_init = 0;
	err = mmc_init(mmc);
	if (err)
		return err;

	/* Read ffu_status, check ffu_version */
	memset(ext_csd_ffu, 0, 512);
	err = mmc_get_ext_csd(mmc, ext_csd_ffu);
	if (err)
		return err;
	ffu_status = ext_csd_ffu[EXT_CSD_FFU_STATUS] & 0xff;
	fw_ver = 0;
	for (i = 0; i < 8; i++) {
		fw_ver |= ext_csd_ffu[EXT_CSD_FW_VERSION + 7 - i];
		if (i < 7)
			fw_ver <<= 8;
	}
	if ((mmc->cid[0] >> 24) == BIWIN_MID)
		fw_ver = ((fw_ver >> 16) & 0xffffffff);
	printf("new fw_ver = %llx\n", fw_ver);
	if (ffu_status || (fw_ver != ffu_ver))
		return ffu_status;

	printf("FFU update ok!\n");
	return 0;
}

int get_boot_size(char *name, uint64_t* size)
{
	int ret = 0;
	struct mmc *mmc = find_mmc_device(1);

	/* do nothing */
	if (!name)
		goto _out;

	if (!mmc) {
		printf("%s() %d: not valid emmc\n", __func__, __LINE__);
		ret = -1;
		goto _out;
	}
	/* make sure mmc is initialized! */
	ret = mmc_init(mmc);
	if (ret) {
		printf("%s() %d: emmc init %d\n", __func__, __LINE__, ret);
		ret = -2;
		goto _out;
	}

	if (!strcmp(name, "boot0") || !strcmp(name, "boot1"))
		*size = mmc->capacity_boot;

_out:
	return ret;
}

int mmc_set_dsr(struct mmc *mmc, u16 val)
{
	mmc->dsr = val;
	return 0;
}

/* CPU-specific MMC initializations */
__weak int cpu_mmc_init(bd_t *bis)
{
	return -1;
}

/* board-specific MMC initializations. */
__weak int board_mmc_init(bd_t *bis)
{
	return -1;
}

#if !defined(CONFIG_SPL_BUILD) || defined(CONFIG_SPL_LIBCOMMON_SUPPORT)

void print_mmc_devices(char separator)
{
	struct mmc *m;
	struct list_head *entry;

	list_for_each(entry, &mmc_devices) {
		m = list_entry(entry, struct mmc, link);

		printf("%s: %d", m->cfg->name, m->block_dev.dev);

		if (entry->next != &mmc_devices) {
			printf("%c", separator);
			if (separator != '\n')
				puts (" ");
		}
	}

	printf("\n");
}

#else
void print_mmc_devices(char separator) { }
#endif

int get_mmc_num(void)
{
	return cur_dev_num;
}

void mmc_set_preinit(struct mmc *mmc, int preinit)
{
	mmc->preinit = preinit;
}

static void do_preinit(void)
{
	struct mmc *m;
	struct list_head *entry;

	list_for_each(entry, &mmc_devices) {
		m = list_entry(entry, struct mmc, link);

		if (m->preinit)
			mmc_start_init(m);
	}
}


int mmc_initialize(bd_t *bis)
{
	INIT_LIST_HEAD (&mmc_devices);
	cur_dev_num = 0;

	if (board_mmc_init(bis) < 0)
		cpu_mmc_init(bis);

#ifndef CONFIG_SPL_BUILD
	print_mmc_devices(',');
#endif

	do_preinit();
	return 0;
}

#ifdef CONFIG_SUPPORT_EMMC_BOOT
/*
 * This function changes the size of boot partition and the size of rpmb
 * partition present on EMMC devices.
 *
 * Input Parameters:
 * struct *mmc: pointer for the mmc device strcuture
 * bootsize: size of boot partition
 * rpmbsize: size of rpmb partition
 *
 * Returns 0 on success.
 */

int mmc_boot_partition_size_change(struct mmc *mmc, unsigned long bootsize,
				unsigned long rpmbsize)
{
	int err;
	struct mmc_cmd cmd;

	/* Only use this command for raw EMMC moviNAND. Enter backdoor mode */
	cmd.cmdidx = MMC_CMD_RES_MAN;
	cmd.resp_type = MMC_RSP_R1b;
	cmd.cmdarg = MMC_CMD62_ARG1;

	err = mmc_send_cmd(mmc, &cmd, NULL);
	if (err) {
		debug("mmc_boot_partition_size_change: Error1 = %d\n", err);
		return err;
	}

	/* Boot partition changing mode */
	cmd.cmdidx = MMC_CMD_RES_MAN;
	cmd.resp_type = MMC_RSP_R1b;
	cmd.cmdarg = MMC_CMD62_ARG2;

	err = mmc_send_cmd(mmc, &cmd, NULL);
	if (err) {
		debug("mmc_boot_partition_size_change: Error2 = %d\n", err);
		return err;
	}
	/* boot partition size is multiple of 128KB */
	bootsize = (bootsize * 1024) / 128;

	/* Arg: boot partition size */
	cmd.cmdidx = MMC_CMD_RES_MAN;
	cmd.resp_type = MMC_RSP_R1b;
	cmd.cmdarg = bootsize;

	err = mmc_send_cmd(mmc, &cmd, NULL);
	if (err) {
		debug("mmc_boot_partition_size_change: Error3 = %d\n", err);
		return err;
	}
	/* RPMB partition size is multiple of 128KB */
	rpmbsize = (rpmbsize * 1024) / 128;
	/* Arg: RPMB partition size */
	cmd.cmdidx = MMC_CMD_RES_MAN;
	cmd.resp_type = MMC_RSP_R1b;
	cmd.cmdarg = rpmbsize;

	err = mmc_send_cmd(mmc, &cmd, NULL);
	if (err) {
		debug("mmc_boot_partition_size_change: Error4 = %d\n", err);
		return err;
	}
	return 0;
}

/*
 * Modify EXT_CSD[177] which is BOOT_BUS_WIDTH
 * based on the passed in values for BOOT_BUS_WIDTH, RESET_BOOT_BUS_WIDTH
 * and BOOT_MODE.
 *
 * Returns 0 on success.
 */
int mmc_set_boot_bus_width(struct mmc *mmc, u8 width, u8 reset, u8 mode)
{
	int err;

	err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, EXT_CSD_BOOT_BUS_WIDTH,
			 EXT_CSD_BOOT_BUS_WIDTH_MODE(mode) |
			 EXT_CSD_BOOT_BUS_WIDTH_RESET(reset) |
			 EXT_CSD_BOOT_BUS_WIDTH_WIDTH(width));

	if (err)
		return err;
	return 0;
}

/*
 * Modify EXT_CSD[179] which is PARTITION_CONFIG (formerly BOOT_CONFIG)
 * based on the passed in values for BOOT_ACK, BOOT_PARTITION_ENABLE and
 * PARTITION_ACCESS.
 *
 * Returns 0 on success.
 */
int mmc_set_part_conf(struct mmc *mmc, u8 ack, u8 part_num, u8 access)
{
	int err;

	err = mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, EXT_CSD_PART_CONF,
			 EXT_CSD_BOOT_ACK(ack) |
			 EXT_CSD_BOOT_PART_NUM(part_num) |
			 EXT_CSD_PARTITION_ACCESS(access));

	if (err)
		return err;
	return 0;
}

/*
 * Modify EXT_CSD[162] which is RST_n_FUNCTION based on the given value
 * for enable.  Note that this is a write-once field for non-zero values.
 *
 * Returns 0 on success.
 */
int mmc_set_rst_n_function(struct mmc *mmc, u8 enable)
{
	return mmc_switch(mmc, EXT_CSD_CMD_SET_NORMAL, EXT_CSD_RST_N_FUNCTION,
			  enable);
}
#endif

extern ulong mmc_bwrite(int dev_num, lbaint_t start,
				lbaint_t blkcnt, const void *src);
extern unsigned long mmc_berase(int dev_num,
				lbaint_t start, lbaint_t blkcnt);
static int mmc_reserved_read(const char *name,
			     unsigned char *buf, unsigned int size)
{
	ulong start, start_blk, blkcnt, ret;
	int dev = EMMC_DTB_DEV;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;

	vpart = aml_get_virtual_partition_by_name(name);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	start = part->offset + vpart->offset;
	start_blk = (start / MMC_BLOCK_SIZE);
	blkcnt = (size / MMC_BLOCK_SIZE);
	ret = mmc_bread(dev, start_blk, blkcnt, buf);
	if (ret != blkcnt) {
		printf("[%s] %d, mmc_bread %s error\n",
			__func__, __LINE__, name);
		return 1;
	}
	return 0;
}

static int mmc_reserved_write(const char *name,
			      unsigned char *buf, unsigned int size)
{
	ulong start, start_blk, blkcnt, ret;
	int i = 2, dev = EMMC_DTB_DEV;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;

	vpart = aml_get_virtual_partition_by_name(name);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	start = part->offset + vpart->offset;
	start_blk = (start / MMC_BLOCK_SIZE);
	blkcnt = (size / MMC_BLOCK_SIZE);
	do {
		ret = mmc_bwrite(dev, start_blk, blkcnt, buf);
		if (ret != blkcnt) {
			printf("[%s] %d, mmc_bwrite %s error\n",
				__func__, __LINE__, name);
			return 1;
		}
		start_blk += vpart->size / MMC_BLOCK_SIZE;
	} while (--i);
	return 0;
}

static int mmc_reserved_erase(const char *name, int number)
{
	ulong start, start_blk, blkcnt, ret;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;
	int dev = EMMC_DTB_DEV;

	vpart = aml_get_virtual_partition_by_name(name);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	start = part->offset + vpart->offset;
	start_blk = (start / MMC_BLOCK_SIZE);
#ifdef KEY_BACKUP
	blkcnt = (vpart->size / MMC_BLOCK_SIZE) * number + 2;
#else
	blkcnt = (vpart->size / MMC_BLOCK_SIZE) * number;
#endif
	ret = mmc_berase(dev, start_blk, blkcnt);
	if (ret) {
		printf("[%s] %d mmc_berase %s error\n",
		       __func__, __LINE__, name);
		return 1;
	}
	return 0;
}


#ifdef KEY_BACKUP
/* unifykey backup distribution */

/*--------------------------------------------------------
 * offset |  0x12020  |  0x12220  |  0x12420  |  0x12421  |
 *--------------------------------------------------------
 *  size  | 200 block | 200 block |  1 block  |  1 block  |
 *--------------------------------------------------------
 *content |    key1   |    key2   | checksum1 | checksum2 |
 *--------------------------------------------------------
 */
static u64 _calc_key_checksum(void *addr, int size)
{
	int i = 0;
	u32 *buffer;
	u64 checksum = 0;

	if ((u64)addr % 4 != 0)
		BUG();

	buffer = (u32 *)addr;
	size = size >> 2;
	while (i < size)
		checksum += buffer[i++];

	return checksum;
}

static int _key_read(struct mmc *mmc, u64 blk, u64 cnt, void * addr)
{
	int dev = EMMC_DTB_DEV;
	u64 n;
	n = mmc->block_dev.block_read(dev, blk, cnt, addr);
	if (n != cnt) {
		printf("%s: dev # %d, block # %#llx, count # %#llx ERROR!\n",
				__func__, dev, blk, cnt);
	}

	return (n != cnt);
}

static int _verify_key_checksum(struct mmc *mmc, void *addr, int cpy)
{
	u64 checksum;
	int ret = 0;
	u64 blk, key_glb_offset;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;
	char checksum_info[512] = {0};

	vpart = aml_get_virtual_partition_by_name(MMC_KEY_NAME);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	key_glb_offset = part->offset + vpart->offset;

	blk = (key_glb_offset + 2 * (vpart->size)) / MMC_BLOCK_SIZE + cpy;
	ret = _key_read(mmc, blk, 1, (void *)checksum_info);
	if (ret)
		return -1;

	memcpy(&key_infos[cpy], checksum_info, sizeof(struct aml_key_info));

	checksum = _calc_key_checksum(addr, vpart->size);
	printf("calc %llx, store %llx\n", checksum, key_infos[cpy].checksum);

	return !(checksum == key_infos[cpy].checksum);
}

static int update_key_info(struct mmc *mmc, unsigned char *addr)
{
	int ret = 0;
	u64 blk, cnt, key_glb_offset;
	int cpy = 1;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;
	int valid_flag = 0;

	vpart = aml_get_virtual_partition_by_name(MMC_KEY_NAME);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	key_glb_offset = part->offset + vpart->offset;

	while (cpy >= 0) {
		blk = (key_glb_offset + cpy * (vpart->size)) / MMC_BLOCK_SIZE;
		cnt = vpart->size / mmc->read_bl_len;
		ret = _key_read(mmc, blk, cnt, addr);
		if (ret) {
			printf("%s: block # %#llx, cnt # %#llx ERROR!\n",
				__func__, blk, cnt);
			return -1;
		}

		ret = _verify_key_checksum(mmc, addr, cpy);
		if (!ret && key_infos[cpy].magic != 0)
			valid_flag += cpy + 1;
		else
			printf("cpy %d is not valid\n", cpy);
		cpy--;
	}

	if (key_infos[0].stamp > key_infos[1].stamp)
		mmc->key_stamp = key_infos[0].stamp;
	else
		mmc->key_stamp = key_infos[1].stamp;

	return valid_flag;
}

static int _key_write(struct mmc *mmc, u64 blk, u64 cnt, void *addr)
{
	int dev = EMMC_KEY_DEV;
	u32 n;
	n = mmc->block_dev.block_write(dev, blk, cnt, addr);
	if (n != cnt) {
		printf("%s: dev # %d, block # %#llx, count # %#llx ERROR!\n",
				__func__, dev, blk, cnt);
	}

	return (n != cnt);
}

static int write_invalid_key(struct mmc *mmc, void *addr, int valid_flag)
{
	u64 blk, cnt, key_glb_offset;
	int ret = 0;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;
	char checksum_info[512] = {0};

	if (valid_flag > 2 || valid_flag < 1)
		return 1;

	vpart = aml_get_virtual_partition_by_name(MMC_KEY_NAME);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	key_glb_offset = part->offset + vpart->offset;

	blk = (key_glb_offset + (valid_flag - 1) * (vpart->size)) / MMC_BLOCK_SIZE;
	cnt = vpart->size / mmc->read_bl_len;

	if (_key_read(mmc, blk, cnt, addr)) {
	printf("%s: block # %#llx,cnt # %#llx ERROR!\n",
			__func__, blk, cnt);
		ret = -2;
	}
	/* fixme, update the invalid one - key1 */
	blk = (key_glb_offset + (valid_flag % 2) * vpart->size) / MMC_BLOCK_SIZE;
	if (_key_write(mmc, blk, cnt, addr)) {
		printf("%s: block # %#llx,cnt # %#llx ERROR!\n",
			__func__, blk, cnt);
		ret = -4;
	}

	memcpy(checksum_info, &key_infos[valid_flag - 1], sizeof(struct aml_key_info));
	blk = (key_glb_offset + 2 * (vpart->size)) / MMC_BLOCK_SIZE + valid_flag % 2;
	if (_key_write(mmc, blk, 1, checksum_info)) {
		printf("%s: block # %#llx,cnt # %#llx ERROR!\n",
			__func__, blk, cnt);
		ret = -4;
	}

	return ret;
}

static int update_invalid_key(struct mmc *mmc, void *addr, int valid_flag)
{
	int ret = 0, dev = EMMC_KEY_DEV;
	u64 blk, cnt, key_glb_offset;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;
	char checksum_info[512] = {0};

	vpart = aml_get_virtual_partition_by_name(MMC_KEY_NAME);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	key_glb_offset = part->offset + vpart->offset;
	cnt = vpart->size / mmc->read_bl_len;

	if (valid_flag == 2) {
		printf("update key1");
		ret = write_invalid_key(mmc, addr, valid_flag);
		if (ret)
			ret = -2;
	} else {
		printf("update key2");
		blk = (key_glb_offset + vpart->size) / MMC_BLOCK_SIZE;
		if (_key_write(mmc, blk, cnt, addr)) {
			printf("%s: dev # %d, block # %#llx,cnt # %#llx ERROR!\n",
				__func__, dev, blk, cnt);
			ret = -2;
		}
		memcpy(checksum_info, &key_infos[valid_flag - 1],
				sizeof(struct aml_key_info));
		blk = (key_glb_offset + 2 * (vpart->size)) / MMC_BLOCK_SIZE + valid_flag % 2;
		if (_key_write(mmc, blk, 1, checksum_info)) {
			printf("%s: block # %#llx,cnt # %#llx ERROR!\n",
				__func__, blk, cnt);
			ret = -4;
		}
	}
	return ret;
}

int update_old_key(struct mmc *mmc, void *addr)
{
	int ret = 0;
	int valid_flag;

	if (stamp_after(key_infos[1].stamp, key_infos[0].stamp)) {
		memcpy(&key_infos[1], &key_infos[0], sizeof(struct aml_key_info));
		valid_flag = 2;
	} else if (stamp_after(key_infos[0].stamp, key_infos[1].stamp)) {
		memcpy(&key_infos[0], &key_infos[1], sizeof(struct aml_key_info));
		valid_flag = 1;
	} else {
		printf("do nothing\n");
		return ret;
	}

	ret = write_invalid_key(mmc, addr, valid_flag);
	/*update key*/
	if (ret)
		ret = -3;
	mmc->key_stamp = key_infos[0].stamp;
	return ret;
}

static struct mmc *_rsv_init(void)
{
	struct mmc *mmc = find_mmc_device(EMMC_KEY_DEV);
	if (!mmc) {
		printf("not find mmc\n");
		return NULL;
	}

	if (mmc_init(mmc)) {
		printf("mmc init failed\n");
		return NULL;
	}

	return mmc;
}

static int mmc_key_write_backup(const char *name,
			      unsigned char *addr, unsigned int size)
{
	int ret = 0;
	u64 blk, cnt, key_glb_offset;
	int cpy;
	struct mmc * mmc;
	struct partitions * part = NULL;
	struct virtual_partition *vpart = NULL;
	char checksum_info[512] = {0};

	vpart = aml_get_virtual_partition_by_name(MMC_KEY_NAME);
	part = aml_get_partition_by_name(MMC_RESERVED_NAME);
	key_glb_offset = part->offset + vpart->offset;

	mmc = _rsv_init();
	if (mmc == NULL)
		return -10;

	key_infos[0].stamp =  mmc->key_stamp + 1;
	key_infos[0].magic = 9;
	key_infos[0].checksum = _calc_key_checksum(addr, vpart->size);
	printf("new stamp %d, checksum 0x%llx, magic %d\n",
		key_infos[0].stamp, key_infos[0].checksum, key_infos[0].magic);

	memcpy(checksum_info, &key_infos[0], sizeof(struct aml_key_info));

	for (cpy = 0; cpy < KEY_COPIES; cpy++) {
		blk = (key_glb_offset + cpy * (vpart->size)) / MMC_BLOCK_SIZE;
		cnt = vpart->size / mmc->read_bl_len;
		ret |= _key_write(mmc, blk, cnt, addr);

		blk = (key_glb_offset + 2 * (vpart->size)) / MMC_BLOCK_SIZE + cpy;
		ret |= _key_write(mmc, blk, 1, checksum_info);
	}

	if (ret) {
		printf("%s() %d: emmc init %d\n", __func__, __LINE__, ret);
		ret = -2;
	}

	return ret;
}

static int mmc_key_read_backup(const char *name,
			      unsigned char *addr, unsigned int size)
{
	int valid = 0;
	struct mmc *mmc;

	mmc = _rsv_init();
	if (mmc == NULL)
		return -10;

	/* check valid key flag , addr save the first key content */
	valid = update_key_info(mmc, addr);
	switch (valid) {
		/* none is valid, using the 1st one for compatibility*/
		case 0:
			goto _out;
		break;
		/* only first is valid, using the first update the second */
		case 1:
			update_invalid_key(mmc, addr, 1);
		break;
		/* only second is valid, using the second */
		case 2:
			update_invalid_key(mmc, addr, 2);
		break;
		case 3:
		/*update the old key */
			update_old_key(mmc, addr);
		break;
		default:
			printf("impossible valid values.\n");
			BUG();
		break;
	}
_out:
	return 0;
}
#endif

int mmc_key_write(unsigned char *buf,
		  unsigned int size, uint32_t *actual_length)
{
	int ret;

	info_disprotect |= DISPROTECT_KEY;
#ifdef KEY_BACKUP
	ret = mmc_key_write_backup(MMC_KEY_NAME, buf, size);
#else
	ret = mmc_reserved_write(MMC_KEY_NAME, buf, size);
#endif
	info_disprotect &= ~DISPROTECT_KEY;

	return ret;
}

int mmc_key_read(unsigned char *buf,
		 unsigned int size, uint32_t *actual_length)
{
	int ret;

	info_disprotect |= DISPROTECT_KEY;
#ifdef KEY_BACKUP
	ret = mmc_key_read_backup(MMC_KEY_NAME, buf, size);
#else
	ret = mmc_reserved_read(MMC_KEY_NAME, buf, size);
#endif
	info_disprotect &= ~DISPROTECT_KEY;

	/*key size is 256KB*/
	*actual_length =  0x40000;

	return ret;
}

int mmc_key_erase(void)
{
	int ret;

	info_disprotect |= DISPROTECT_KEY;
	/* when write the 2nd key? */
	ret = mmc_reserved_erase(MMC_KEY_NAME, 2);
	info_disprotect &= ~DISPROTECT_KEY;

	return ret;
}

int mmc_ddr_parameter_read(unsigned char *buf,
			   unsigned int size)
{
	return mmc_reserved_read(MMC_DDR_PARAMETER_NAME, buf, size);
}

int mmc_ddr_parameter_write(unsigned char *buf,
			    unsigned int size)
{
	return mmc_reserved_write(MMC_DDR_PARAMETER_NAME, buf, size);
}

int mmc_ddr_parameter_erase(void)
{
	return mmc_reserved_erase(MMC_DDR_PARAMETER_NAME, 1);
}
