/* SPDX-License-Identifier: GPL-2.0 WITH Linux-syscall-note */
#ifndef _LINUX_PRCTL_H
#define _LINUX_PRCTL_H

#include <linux/types.h>

/* Values to pass as first argument to prctl() */

#define PR_SET_PDEATHSIG  1  /* Second arg is a signal */
#define PR_GET_PDEATHSIG  2  /* Second arg is a ptr to return the signal */

/* Get/set current->mm->dumpable */
#define PR_GET_DUMPABLE   3
#define PR_SET_DUMPABLE   4

/* Get/set unaligned access control bits (if meaningful) */
#define PR_GET_UNALIGN	  5
#define PR_SET_UNALIGN	  6
# define PR_UNALIGN_NOPRINT	1	/* silently fix up unaligned user accesses */
# define PR_UNALIGN_SIGBUS	2	/* generate SIGBUS on unaligned user access */

/* Get/set whether or not to drop capabilities on setuid() away from
 * uid 0 (as per security/commoncap.c) */
#define PR_GET_KEEPCAPS   7
#define PR_SET_KEEPCAPS   8

/* Get/set floating-point emulation control bits (if meaningful) */
#define PR_GET_FPEMU  9
#define PR_SET_FPEMU 10
# define PR_FPEMU_NOPRINT	1	/* silently emulate fp operations accesses */
# define PR_FPEMU_SIGFPE	2	/* don't emulate fp operations, send SIGFPE instead */

/* Get/set floating-point exception mode (if meaningful) */
#define PR_GET_FPEXC	11
#define PR_SET_FPEXC	12
# define PR_FP_EXC_SW_ENABLE	0x80	/* Use FPEXC for FP exception enables */
# define PR_FP_EXC_DIV		0x010000	/* floating point divide by zero */
# define PR_FP_EXC_OVF		0x020000	/* floating point overflow */
# define PR_FP_EXC_UND		0x040000	/* floating point underflow */
# define PR_FP_EXC_RES		0x080000	/* floating point inexact result */
# define PR_FP_EXC_INV		0x100000	/* floating point invalid operation */
# define PR_FP_EXC_DISABLED	0	/* FP exceptions disabled */
# define PR_FP_EXC_NONRECOV	1	/* async non-recoverable exc. mode */
# define PR_FP_EXC_ASYNC	2	/* async recoverable exception mode */
# define PR_FP_EXC_PRECISE	3	/* precise exception mode */

/* Get/set whether we use statistical process timing or accurate timestamp
 * based process timing */
#define PR_GET_TIMING   13
#define PR_SET_TIMING   14
# define PR_TIMING_STATISTICAL  0       /* Normal, traditional,
                                                   statistical process timing */
# define PR_TIMING_TIMESTAMP    1       /* Accurate timestamp based
                                                   process timing */

#define PR_SET_NAME    15		/* Set process name */
#define PR_GET_NAME    16		/* Get process name */

/* Get/set process endian */
#define PR_GET_ENDIAN	19
#define PR_SET_ENDIAN	20
# define PR_ENDIAN_BIG		0
# define PR_ENDIAN_LITTLE	1	/* True little endian mode */
# define PR_ENDIAN_PPC_LITTLE	2	/* "PowerPC" pseudo little endian */

/* Get/set process seccomp mode */
#define PR_GET_SECCOMP	21
#define PR_SET_SECCOMP	22

/* Get/set the capability bounding set (as per security/commoncap.c) */
#define PR_CAPBSET_READ 23
#define PR_CAPBSET_DROP 24

/* Get/set the process' ability to use the timestamp counter instruction */
#define PR_GET_TSC 25
#define PR_SET_TSC 26
# define PR_TSC_ENABLE		1	/* allow the use of the timestamp counter */
# define PR_TSC_SIGSEGV		2	/* throw a SIGSEGV instead of reading the TSC */

/* Get/set securebits (as per security/commoncap.c) */
#define PR_GET_SECUREBITS 27
#define PR_SET_SECUREBITS 28

/*
 * Get/set the timerslack as used by poll/select/nanosleep
 * A value of 0 means "use default"
 */
#define PR_SET_TIMERSLACK 29
#define PR_GET_TIMERSLACK 30

#define PR_TASK_PERF_EVENTS_DISABLE		31
#define PR_TASK_PERF_EVENTS_ENABLE		32

/*
 * Set early/late kill mode for hwpoison memory corruption.
 * This influences when the process gets killed on a memory corruption.
 */
#define PR_MCE_KILL	33
# define PR_MCE_KILL_CLEAR   0
# define PR_MCE_KILL_SET     1

# define PR_MCE_KILL_LATE    0
# define PR_MCE_KILL_EARLY   1
# define PR_MCE_KILL_DEFAULT 2

#define PR_MCE_KILL_GET 34

/*
 * Tune up process memory map specifics.
 */
#define PR_SET_MM		35
# define PR_SET_MM_START_CODE		1
# define PR_SET_MM_END_CODE		2
# define PR_SET_MM_START_DATA		3
# define PR_SET_MM_END_DATA		4
# define PR_SET_MM_START_STACK		5
# define PR_SET_MM_START_BRK		6
# define PR_SET_MM_BRK			7
# define PR_SET_MM_ARG_START		8
# define PR_SET_MM_ARG_END		9
# define PR_SET_MM_ENV_START		10
# define PR_SET_MM_ENV_END		11
# define PR_SET_MM_AUXV			12
# define PR_SET_MM_EXE_FILE		13
# define PR_SET_MM_MAP			14
# define PR_SET_MM_MAP_SIZE		15

/*
 * This structure provides new memory descriptor
 * map which mostly modifies /proc/pid/stat[m]
 * output for a task. This mostly done in a
 * sake of checkpoint/restore functionality.
 */
struct prctl_mm_map {
	__u64	start_code;		/* code section bounds */
	__u64	end_code;
	__u64	start_data;		/* data section bounds */
	__u64	end_data;
	__u64	start_brk;		/* heap for brk() syscall */
	__u64	brk;
	__u64	start_stack;		/* stack starts at */
	__u64	arg_start;		/* command line arguments bounds */
	__u64	arg_end;
	__u64	env_start;		/* environment variables bounds */
	__u64	env_end;
	__u64	*auxv;			/* auxiliary vector */
	__u32	auxv_size;		/* vector size */
	__u32	exe_fd;			/* /proc/$pid/exe link file */
};

/*
 * Set specific pid that is allowed to ptrace the current task.
 * A value of 0 mean "no process".
 */
#define PR_SET_PTRACER 0x59616d61
# define PR_SET_PTRACER_ANY ((unsigned long)-1)

#define PR_SET_CHILD_SUBREAPER	36
#define PR_GET_CHILD_SUBREAPER	37

/*
 * If no_new_privs is set, then operations that grant new privileges (i.e.
 * execve) will either fail or not grant them.  This affects suid/sgid,
 * file capabilities, and LSMs.
 *
 * Operations that merely manipulate or drop existing privileges (setresuid,
 * capset, etc.) will still work.  Drop those privileges if you want them gone.
 *
 * Changing LSM security domain is considered a new privilege.  So, for example,
 * asking selinux for a specific new context (e.g. with runcon) will result
 * in execve returning -EPERM.
 *
 * See Documentation/userspace-api/no_new_privs.rst for more details.
 */
#define PR_SET_NO_NEW_PRIVS	38
#define PR_GET_NO_NEW_PRIVS	39

#define PR_GET_TID_ADDRESS	40

#define PR_SET_THP_DISABLE	41
#define PR_GET_THP_DISABLE	42

/*
 * No longer implemented, but left here to ensure the numbers stay reserved:
 */
#define PR_MPX_ENABLE_MANAGEMENT  43
#define PR_MPX_DISABLE_MANAGEMENT 44

#define PR_SET_FP_MODE		45
#define PR_GET_FP_MODE		46
# define PR_FP_MODE_FR		(1 << 0)	/* 64b FP registers */
# define PR_FP_MODE_FRE		(1 << 1)	/* 32b compatibility */

/* Control the ambient capability set */
#define PR_CAP_AMBIENT			47
# define PR_CAP_AMBIENT_IS_SET		1
# define PR_CAP_AMBIENT_RAISE		2
# define PR_CAP_AMBIENT_LOWER		3
# define PR_CAP_AMBIENT_CLEAR_ALL	4

/* arm64 Scalable Vector Extension controls */
/* Flag values must be kept in sync with ptrace NT_ARM_SVE interface */
#define PR_SVE_SET_VL			50	/* set task vector length */
# define PR_SVE_SET_VL_ONEXEC		(1 << 18) /* defer effect until exec */
#define PR_SVE_GET_VL			51	/* get task vector length */
/* Bits common to PR_SVE_SET_VL and PR_SVE_GET_VL */
# define PR_SVE_VL_LEN_MASK		0xffff
# define PR_SVE_VL_INHERIT		(1 << 17) /* inherit across exec */

/* Per task speculation control */
#define PR_GET_SPECULATION_CTRL		52
#define PR_SET_SPECULATION_CTRL		53
/* Speculation control variants */
# define PR_SPEC_STORE_BYPASS		0
# define PR_SPEC_INDIRECT_BRANCH	1
/* Return and control values for PR_SET/GET_SPECULATION_CTRL */
# define PR_SPEC_NOT_AFFECTED		0
# define PR_SPEC_PRCTL			(1UL << 0)
# define PR_SPEC_ENABLE			(1UL << 1)
# define PR_SPEC_DISABLE		(1UL << 2)
# define PR_SPEC_FORCE_DISABLE		(1UL << 3)
# define PR_SPEC_DISABLE_NOEXEC		(1UL << 4)

/* Reset arm64 pointer authentication keys */
#define PR_PAC_RESET_KEYS		54
# define PR_PAC_APIAKEY			(1UL << 0)
# define PR_PAC_APIBKEY			(1UL << 1)
# define PR_PAC_APDAKEY			(1UL << 2)
# define PR_PAC_APDBKEY			(1UL << 3)
# define PR_PAC_APGAKEY			(1UL << 4)

/* Tagged user address controls for arm64 */
#define PR_SET_TAGGED_ADDR_CTRL		55
#define PR_GET_TAGGED_ADDR_CTRL		56
# define PR_TAGGED_ADDR_ENABLE		(1UL << 0)

#define PR_SET_VMA		0x53564d41
# define PR_SET_VMA_ANON_NAME		0

#define PR_SET_FTRACE_MARKER	99

/* sets max size of string buffer */
#define FTRACE_BUF_SIZE	1024

/* define structure(s) used for logging video and audio metrics */
typedef struct {
	__u64 lastSyncTime;
	__u64 maxSyncTime;
	__u64 minSyncTime;
	__u64 sumSyncTime;
	__u32 numVsyncs1ms;
	__u32 numVsyncs2ms;
	__u32 numVsyncs3ms;
	__u32 numVsyncs4ms;
	__u32 numVsyncs5ms;
	__u32 numVsyncs6ms;
	__u32 numVsyncs7ms;
	__u32 numVsyncs8ms;
	__u32 numVsyncs9ms;
	__u32 numVsyncs10ms;
	__u32 vsyncCount;
	__u32 framesDisplayed;
	__u32 framesFRC;
	__u32 framesQDropped;
	__u32 framesPTSLateDropped;
	__u32 framesSyncFreeDropped;
	__u32 framesRepeats;
	__u32 underruns;
	__u32 Qunderflow;
	__u32 gapUnderflow;
	__u32 maxConsecQDrops;
	__u32 maxConsecPTSLateDrops;
	__u32 maxSyncFreeDropped;
	__u32 maxConsecRepeats;
	__u32 maxUnderruns;
	__u32 maxQunderflow;
	__u32 maxGapUnderflow;
	__u32 consecQDrops;
	__u32 consecPTSLateDrops;
	__u32 consecSyncFreeDropped;
	__u32 consecRepeats;
	__u32 consecUnderruns;
	__u32 consecQunderflow;
	__u32 consecGapUnderflow;
	__u32 syncRate;
	__u32 syncMode;
	__u32 retuneCount;
	__u32 decodeErrorCount;
	__u32 numMissed1Vsyncs;
	__u32 numMissed2Vsyncs;
	__u32 numMissed3Vsyncs;
	__u32 numMissed4Vsyncs;
	__u32 numMissed5Vsyncs;
	__u32 numMissed6PlusVsyncs;
	__u32 maxMissedVsyncs;
	__u32 sumMissedVsyncs;
	__u32 missedVsyncsCount;
	__u32 cause_wstServer;
	__u32 cause_calloc;
	__u32 cause_wstSyncWaitFence;
	__u32 cause_eglSwapBuffers;
	__u64 maxVSyncIrqDelay;
	/* pad out structure to be an integer number of cache lines (64bytes/cache line) */
	__u32 reserved[3];
} VIDEO_METRICS;
/* max two AV channels supported */
#define MAX_VIDEO_CHANNELS	2

#define PR_GET_VIDEO_METRICS	100
#define PR_SET_VIDEO_METRICS	101

typedef enum {
	AUDIO_METRIC_TYPE_STREAM_INPUT = 0,
	AUDIO_METRIC_TYPE_SYSTEM_INPUT,
	AUDIO_METRIC_TYPE_AD_INPUT,
	AUDIO_METRIC_TYPE_MAIN_INPUT,
	AUDIO_METRIC_TYPE_MAX_INPUTS,
	AUDIO_METRIC_TYPE_OUTPUT
} AUDIO_METRIC_TYPE;

/* define structure(s) used for logging audio metrics */
typedef struct {
	__u32 underrunTime;
} AUDIO_OUTPUT_METRICS;

typedef struct {
	__u32 underruns;
	__u32 bufFullTime;
	__u32 bufLevel;
	__u32 sampleRate;
	__u32 numChannels;
	__u32 discardPkt;
	__u32 reconfigCount;
	__u32 frameRepeat;
	__u32 frameSkip;
	__u32 frameInSync;
	__u32 resync;
	__u32 reserve[5];
} AUDIO_INPUT_CHAN_METRICS;

typedef struct {
	__u32 count;
	__u32 underruns;
	__u32 bufFullTime;
	__u32 minBufFullTime;
	__u32 maxBufFullTime;
	__u32 bufLevelBytes;
	__u32 maxBufLevelBytes;
	__u32 discardPkt;
	__u32 reconfigCount;
	__u32 sampleRate;
	__u32 numChannels;
	__u32 resyncCount;
	__u32 frameRepeatCount;
	__u32 consecRepeats;
	__u32 maxConsecRepeats;
	__u32 frameSkipCount;
	__u32 frameInSync;
	__u32 consecSkips;
	__u32 maxConsecSkips;
	/* pad out to 1 cache line (64 bytes) */
	__u32 reserved[13];
} AUDIO_INPUT_METRICS;

typedef struct {
	__u32 underrunTime;
	__u32 underrunCount;
	__u32 minUnderrunTime;
	__u32 maxUnderrunTime;
	AUDIO_INPUT_METRICS input[AUDIO_METRIC_TYPE_MAX_INPUTS];
	/* pad out structure to be an integer number of cache lines (64bytes/cache line) */
} AUDIO_METRICS;
/* max two AV channels supported */
#define MAX_AUDIO_CHANNELS	2

#define PR_GET_AUDIO_METRICS	102
#define PR_SET_AUDIO_METRICS	103

#define NUM_CPUS 4

/* define structure(s) used for logging idle metrics */
typedef struct {
	__u64 metricStartTime;
	__u64 metricEndTime;
	__u64 idle_time[NUM_CPUS];
} IDLE_METRICS;

typedef struct {
	__u64 sum_idle_time;
	__u32 max_idle_time;
	__u32 idle_count;
} PER_CPU_IDLE_METRIC;

/* define structure(s) used for logging idle metrics */
typedef struct {
	__u64 metricStartTime;
	__u64 metricEndTime;
	PER_CPU_IDLE_METRIC idle[NUM_CPUS];
	__u64 sum_all_cpus_running_time;
	__u64 sum_run_time_lt_1ms;
	__u64 sum_run_time_gt_1ms;
	__u64 sum_run_time_gt_5ms;
	__u64 sum_run_time_gt_10ms;
	__u64 sum_run_time_gt_20ms;
	__u64 sum_run_time_gt_30ms;
	__u64 sum_run_time_gt_40ms;
	__u64 sum_run_time_gt_50ms;
	__u64 sum_run_time_gt_75ms;
	__u64 sum_run_time_gt_100ms;
	__u32 max_run_time_lt_1ms;
	__u32 max_run_time_gt_1ms;
	__u32 max_run_time_gt_5ms;
	__u32 max_run_time_gt_10ms;
	__u32 max_run_time_gt_20ms;
	__u32 max_run_time_gt_30ms;
	__u32 max_run_time_gt_40ms;
	__u32 max_run_time_gt_50ms;
	__u32 max_run_time_gt_75ms;
	__u32 max_run_time_gt_100ms;
	__u32 max_all_cpus_running_time;
	__u32 count;
	__u32 run_time_gt_1ms;
	__u32 run_time_gt_5ms;
	__u32 run_time_gt_10ms;
	__u32 run_time_gt_20ms;
	__u32 run_time_gt_30ms;
	__u32 run_time_gt_40ms;
	__u32 run_time_gt_50ms;
	__u32 run_time_gt_75ms;
	__u32 run_time_gt_100ms;
} IDLE_METRICS_V2;

#define PR_GET_IDLE_METRICS		104
#define PR_GET_IDLE_METRICS_VERSION	105

typedef enum {
	IDLE_METRICS_VERSION_V1 = 0,
	IDLE_METRICS_VERSION_V2,
	IDLE_METRICS_VERSION_NUM_VERSIONS
} IDLE_METRICS_VERSION;

#endif /* _LINUX_PRCTL_H */
