#include <uapi/linux/dvb/frontend.h>

#include "si2151_func.h"
#include "si2151.h"


#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0)
extern struct si2151_device_s *si2151_devp;
extern void si2151_set_std(void);
extern int dtv_out_amp_val;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 4, 0)
static void si2151_release_ex(struct dvb_frontend *fe);
#else
static int si2151_release_ex(struct dvb_frontend *fe);
#endif
static int si2151_init_ex(struct dvb_frontend *fe);
static int si2151_sleep_ex(struct dvb_frontend *fe);
static int si2151_suspend_ex(struct dvb_frontend *fe);
static int si2151_resume_ex(struct dvb_frontend *fe);
static int si2151_set_params_ex(struct dvb_frontend *fe);
static int si2151_set_analog_params_ex(struct dvb_frontend *fe,
		struct analog_parameters *p);
static int si2151_calc_regs_ex(struct dvb_frontend *fe, u8 *buf, int buf_len);
static int si2151_set_config_ex(struct dvb_frontend *fe, void *priv_cfg);
static int si2151_get_frequency_ex(struct dvb_frontend *fe, u32 *frequency);
static int si2151_get_bandwidth_ex(struct dvb_frontend *fe, u32 *frequency);
static int si2151_get_if_frequency_ex(struct dvb_frontend *fe, u32 *frequency);
static int si2151_get_status_ex(struct dvb_frontend *fe, u32 *status);
static int si2151_get_rf_strength_ex(struct dvb_frontend *fe, u16 *strength);
#ifdef CONFIG_AMLOGIC_DVB_COMPAT
static int si2151_get_strength_ex(struct dvb_frontend *fe, s16 *strength);
#endif
static int si2151_get_afc_ex(struct dvb_frontend *fe, s32 *afc);
static int si2151_set_frequency_ex(struct dvb_frontend *fe, u32 frequency);
static int si2151_set_bandwidth_ex(struct dvb_frontend *fe, u32 bandwidth);


static int si2151_init_configure(struct si2151_device_s *si2151,
		unsigned char mode)
{
	int ret = 0;

	if (si2151->inited && mode == si2151->parm.mode) {
		pr_info("[si2151] %s: had already done(mode %d).\n",
				__func__, si2151->parm.mode);

		return 0;
	}

	ret = si2151_init(&si2151->tuner_client,
			&si2151->si_cmd_reply,
			&si2151->si_common_reply);
	if (ret) {
		pr_err("[si2151] %s: init si2151 error: ret %d.\n",
				__func__, ret);

		return ret;
	}

	ret = si2151_configure(&si2151->tuner_client,
			&si2151->si_prop,
			&si2151->si_cmd_reply,
			&si2151->si_common_reply);
	if (ret) {
		pr_err("[si2151] %s: configure si2151 error: ret %d.\n",
				__func__, ret);

		return ret;
	}

	si2151->parm.mode = mode;
	si2151->inited = true;
	si2151->suspended = false;

	return 0;
}

static int si2151_powerdown(struct si2151_device_s *si2151)
{
	int ret = 0;

	if (!si2151->suspended) {
		si2151->parm.mode = 0xf; /* setting invalid value */
		si2151->parm.frequency = 0;
		si2151->parm.std = 0;
		si2151->parm.audmode = 0;
		si2151->fre_offset = 0;
		ret = si2151_power_down(&si2151->tuner_client,
				&si2151->si_cmd_reply);
		if (ret)
			pr_err("[si2151] %s: si2151_power_down error: ret %d.\n",
					__func__, ret);

		si2151->suspended = true;
		si2151->inited = false;
	}

	return ret;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 4, 0)
static void si2151_release_ex(struct dvb_frontend *fe)
#else
static int si2151_release_ex(struct dvb_frontend *fe)
#endif
{
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	si2151_powerdown(si2151);

	pr_info("[si2151] %s: OK.\n", __func__);

	mutex_unlock(&si2151->mutex);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 4, 0)
#else
	return 0;
#endif
}

static int si2151_init_ex(struct dvb_frontend *fe)
{
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;
	unsigned char mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV;

	mutex_lock(&si2151->mutex);

	if (fe->ops.info.type == FE_ANALOG)
		mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV;
	else
		mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_DTV;

	si2151_init_configure(si2151, mode);

	pr_info("[si2151] %s: type %d OK.\n", __func__, fe->ops.info.type);

	mutex_unlock(&si2151->mutex);

	return 0;
}

static int si2151_sleep_ex(struct dvb_frontend *fe)
{
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	si2151_powerdown(si2151);

	pr_info("[si2151] %s: OK.\n", __func__);

	mutex_unlock(&si2151->mutex);

	return 0;
}

static int si2151_suspend_ex(struct dvb_frontend *fe)
{
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	si2151_powerdown(si2151);

	pr_info("[si2151] %s: OK.\n", __func__);

	mutex_unlock(&si2151->mutex);

	return 0;
}

static int si2151_resume_ex(struct dvb_frontend *fe)
{
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;
	unsigned char mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV;

	mutex_lock(&si2151->mutex);

	if (fe->ops.info.type == FE_ANALOG)
		mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV;
	else
		mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_DTV;

	si2151_init_configure(si2151, mode);

	pr_info("[si2151] %s: OK.\n", __func__);

	mutex_unlock(&si2151->mutex);

	return 0;
}

/* This is the recomended way to set the tuner */
static int si2151_set_params_ex(struct dvb_frontend *fe)
{
	struct dtv_frontend_properties *c = &fe->dtv_property_cache;
	int strength = 100;
	int data = 0, ret = 0;
	bool dtv_agc_auto_freeze = false;
	int bw = SI2151_DTV_MODE_PROP_BW_BW_8MHZ;
	int modulation = SI2151_DTV_MODE_PROP_MODULATION_DEFAULT;
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	si2151->parm.mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_DTV;

	if (FE_ATSC == fe->ops.info.type) {
		si2151->si_prop.dtv_lif_freq.offset = 5000;

		/* cheng.tong:20170410 modify */
		si2151->si_prop.dtv_lif_out.amp = dtv_out_amp_val;
		bw = SI2151_DTV_MODE_PROP_BW_BW_6MHZ;

		if (c->modulation == VSB_8 || c->modulation == VSB_16) {
			modulation = SI2151_DTV_MODE_PROP_MODULATION_ATSC;
		} else {
			modulation = SI2151_DTV_MODE_PROP_MODULATION_QAM_US;
		}
	} else if (FE_QAM == fe->ops.info.type) {
		si2151->si_prop.dtv_lif_freq.offset = 5000;

		if (c->modulation == QAM_256) {
			si2151->si_prop.dtv_lif_out.amp = 23;
		} else {
			si2151->si_prop.dtv_lif_out.amp = 23;
		}

		/* bruce.huang:20170113 modify */
		si2151->si_prop.dtv_lif_out.amp = dtv_out_amp_val;

		bw = SI2151_DTV_MODE_PROP_BW_BW_8MHZ;
		modulation = SI2151_DTV_MODE_PROP_MODULATION_DVBC;
	} else if (FE_DTMB == fe->ops.info.type) {
		/* cheng.tong:20170410 modify */
		si2151->si_prop.dtv_lif_out.amp = dtv_out_amp_val;

		bw = SI2151_DTV_MODE_PROP_BW_BW_8MHZ;
		modulation = SI2151_DTV_MODE_PROP_MODULATION_DTMB;
	} else if (FE_OFDM == fe->ops.info.type) {
		si2151->si_prop.dtv_lif_freq.offset = 5000;
		si2151->si_prop.dtv_lif_out.amp = dtv_out_amp_val;
		si2151->si_prop.dtv_mode.invert_spectrum = SI2151_DTV_MODE_PROP_INVERT_SPECTRUM_INVERTED;

		modulation = SI2151_DTV_MODE_PROP_MODULATION_DVBT;
		switch (c->bandwidth_hz) {
		case 1700000:
		case 1712000:
			bw = SI2151_DTV_MODE_PROP_BW_BW_1P7MHZ;
			break;
		case 5000000:
		case 6000000:
			bw = SI2151_DTV_MODE_PROP_BW_BW_6MHZ;
			break;
		case 7000000:
			bw = SI2151_DTV_MODE_PROP_BW_BW_7MHZ;
			break;
		case 8000000:
			bw = SI2151_DTV_MODE_PROP_BW_BW_8MHZ;
			break;
		default:
			bw = SI2151_DTV_MODE_PROP_BW_BW_8MHZ;
			break;
		}

		if (c->delivery_system == SYS_DVBT2) {
			/* for with FEF mode */
			if (si2151->si_prop.dtv_agc_auto_freeze.thld != 9 ||
				si2151->si_prop.dtv_agc_auto_freeze.timeout != 63) {
				si2151->si_prop.dtv_agc_auto_freeze.thld = 9;
				si2151->si_prop.dtv_agc_auto_freeze.timeout = 63;
				dtv_agc_auto_freeze = true;
			}
		} else {
			if (si2151->si_prop.dtv_agc_auto_freeze.thld != SI2151_DTV_AGC_AUTO_FREEZE_PROP_THLD_DEFAULT ||
				si2151->si_prop.dtv_agc_auto_freeze.timeout != SI2151_DTV_AGC_AUTO_FREEZE_PROP_TIMEOUT_DEFAULT) {
				si2151->si_prop.dtv_agc_auto_freeze.thld = SI2151_DTV_AGC_AUTO_FREEZE_PROP_THLD_DEFAULT;
				si2151->si_prop.dtv_agc_auto_freeze.timeout = SI2151_DTV_AGC_AUTO_FREEZE_PROP_TIMEOUT_DEFAULT;
				dtv_agc_auto_freeze = true;
			}
		}

		if (dtv_agc_auto_freeze) {
			ret = si2151_sendproperty(&si2151->tuner_client, SI2151_DTV_AGC_AUTO_FREEZE_PROP,
					&si2151->si_prop, &si2151->si_cmd_reply);
			if (ret)
				pr_err("[si2151] %s: set dtv agc auto freeze: ret %d.\n",
						__func__, ret);
		}
	} else if (FE_ISDBT == fe->ops.info.type) {
		si2151->si_prop.dtv_lif_freq.offset = 5000;
		si2151->si_prop.dtv_lif_out.amp = dtv_out_amp_val;
		si2151->si_prop.dtv_mode.invert_spectrum = SI2151_DTV_MODE_PROP_INVERT_SPECTRUM_INVERTED;

		switch (c->bandwidth_hz) {
		case 5000000:
		case 6000000:
			bw = SI2151_DTV_MODE_PROP_BW_BW_6MHZ;
			break;
		case 7000000:
			bw = SI2151_DTV_MODE_PROP_BW_BW_7MHZ;
			break;
		case 8000000:
			bw = SI2151_DTV_MODE_PROP_BW_BW_8MHZ;
			break;
		default:
			bw = SI2151_DTV_MODE_PROP_BW_BW_6MHZ;
			break;
		}

		modulation = SI2151_DTV_MODE_PROP_MODULATION_ISDBT;
	}

	ret = si2151_set_property(&si2151->tuner_client, 0, SI2151_DTV_LIF_FREQ_PROP,
			si2151->si_prop.dtv_lif_freq.offset, &si2151->si_cmd_reply);
	if (ret)
		pr_err("[si2151] %s: set dtv lif out if error: ret %d.\n",
				__func__, ret);

	ret = si2151_set_property(&si2151->tuner_client, 0, SI2151_DTV_LIF_OUT_PROP,
			(si2151->si_prop.dtv_lif_out.offset) + ((si2151->si_prop.dtv_lif_out.amp) << 8),
					&si2151->si_cmd_reply);
	if (ret)
		pr_err("[si2151] %s: set dtv lif out amp error: ret %d.\n",
				__func__, ret);

	si2151->si_prop.dtv_mode.bw = bw;
	si2151->si_prop.dtv_mode.modulation = modulation;

	data = (si2151->si_prop.dtv_mode.bw & SI2151_DTV_MODE_PROP_BW_MASK)
			<< SI2151_DTV_MODE_PROP_BW_LSB
			| (si2151->si_prop.dtv_mode.modulation
					& SI2151_DTV_MODE_PROP_MODULATION_MASK)
					<< SI2151_DTV_MODE_PROP_MODULATION_LSB
			| (si2151->si_prop.dtv_mode.invert_spectrum
					& SI2151_DTV_MODE_PROP_INVERT_SPECTRUM_MASK)
					<< SI2151_DTV_MODE_PROP_INVERT_SPECTRUM_LSB;
	if ((ret = si2151_set_property(&si2151->tuner_client, 0, SI2151_DTV_MODE_PROP,
			data, &si2151->si_cmd_reply)))
		pr_err("[si2151] %s: set SI2151_DTV_MODE_PROP error: %d.\n",
				__func__, ret);

	/* Bruce Add for RL */
	/* Set the Tuner return loss optimize and TF1 boundary if in DTMB mode,
	 * otherwise reset it to default values.
	 */
	si2151->si_prop.tuner_return_loss_optimize.config = 91;
	si2151->si_prop.tuner_return_loss_optimize_2.thld = 15; /* (default 31) */
	si2151->si_prop.tuner_return_loss_optimize_2.window = 5; /* (default 0) */
	si2151->si_prop.tuner_return_loss_optimize_2.engagement_delay = 3; /* (default 15) */
	si2151->si_prop.tuner_tf1_boundary_offset.tf1_boundary_offset = 22;
	/* set the remaining optimize values to their defaults */
	si2151->si_prop.tuner_return_loss_optimize.thld = 0;
	si2151->si_prop.tuner_return_loss_optimize.engagement_delay = 7;
	si2151->si_prop.tuner_return_loss_optimize.disengagement_delay = 10;

	data = (si2151->si_prop.tuner_return_loss_optimize.thld
				& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_THLD_MASK)
				<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_THLD_LSB
				| (si2151->si_prop.tuner_return_loss_optimize.config
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_CONFIG_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_CONFIG_LSB
				| (si2151->si_prop.tuner_return_loss_optimize.engagement_delay
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_ENGAGEMENT_DELAY_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_ENGAGEMENT_DELAY_LSB
				| (si2151->si_prop.tuner_return_loss_optimize.disengagement_delay
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_DISENGAGEMENT_DELAY_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_DISENGAGEMENT_DELAY_LSB;
	if ((ret = si2151_set_property(&si2151->tuner_client, 0,
			Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP, data,
			&si2151->si_cmd_reply)))
		pr_err("[si2151] %s: set Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP error: %d.\n",
				__func__, ret);

	data = (si2151->si_prop.tuner_return_loss_optimize_2.thld
				& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_THLD_MASK)
				<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_THLD_LSB
				| (si2151->si_prop.tuner_return_loss_optimize_2.window
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_WINDOW_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_WINDOW_LSB
				| (si2151->si_prop.tuner_return_loss_optimize_2.engagement_delay
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_ENGAGEMENT_DELAY_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_ENGAGEMENT_DELAY_LSB;
	if ((ret = si2151_set_property(&si2151->tuner_client, 0,
			Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP, data,
			&si2151->si_cmd_reply)))
		pr_err("[si2151] %s: set Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP error: %d.\n",
				__func__, ret);

	data = (si2151->si_prop.tuner_tf1_boundary_offset.tf1_boundary_offset
			& Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP_TF1_BOUNDARY_OFFSET_MASK)
			<< Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP_TF1_BOUNDARY_OFFSET_LSB;
	if ((ret = si2151_set_property(&si2151->tuner_client, 0,
			Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP, data,
			&si2151->si_cmd_reply)))
		pr_err("[si2151] %s: set Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP error: %d.\n",
				__func__, ret);

	si2151->fre_offset = 0;
	si2151->parm.frequency = c->frequency;
	if ((ret = si2151_tune(&si2151->tuner_client, si2151->parm.mode,
			si2151->parm.frequency, &si2151->si_cmd_reply,
			&si2151->si_common_reply))) {
		pr_err("[si2151] %s: tune frequency error: %d.\n",
				__func__, ret);

		mutex_unlock(&si2151->mutex);

		return -1;
	}

	if ((ret = si2151_tuner_status(&si2151->tuner_client,
			SI2151_ATV_STATUS_CMD_INTACK_OK, &si2151->si_cmd_reply)) != 0)
		pr_err("[si2151] %s: si2151_tuner_status error: %d.\n",
				__func__, ret);
	else
		strength = si2151->si_cmd_reply.tuner_status.rssi - 256;

	pr_dbg("[si2151] %s: fe type %d, frequency %u, bw %d, modul %d, lif_freq %d, offset %d, amp %d, strength %d, dtv_agc_auto_freeze %d.\n",
			__func__, fe->ops.info.type,
			si2151->parm.frequency,
			si2151->si_prop.dtv_mode.bw,
			si2151->si_prop.dtv_mode.modulation,
			si2151->si_prop.dtv_lif_freq.offset,
			si2151->si_prop.dtv_lif_out.offset,
			si2151->si_prop.dtv_lif_out.amp,
			strength,
			dtv_agc_auto_freeze);

	mutex_unlock(&si2151->mutex);

	return 0;
}

static int si2151_set_analog_params_ex(struct dvb_frontend *fe, struct analog_parameters *p)
{
	int data = 0, ret = 0;
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	si2151->parm.mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV;

	/* if(c->analog.std != si2151->parm.std) */ {
		si2151->parm.std = p->std;
		si2151->parm.audmode = p->audmode;
		si2151_set_std();
		pr_dbg("[si2151] %s: set std color %s, audio type %s.\n",
				__func__,
				v4l2_std_to_str((0xff000000 & si2151->parm.std)),
				v4l2_std_to_str((0xffffff & si2151->parm.std)));
	}

	/* Bruce Add for return loss */
	/* In ATV application, following optimize values for DTMB application
	 * should be set back to their defaults.
	 */
	si2151->si_prop.tuner_return_loss_optimize.config =
			Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_CONFIG_DISABLE;
	si2151->si_prop.tuner_return_loss_optimize.thld = 0;
	si2151->si_prop.tuner_return_loss_optimize.engagement_delay = 7;
	si2151->si_prop.tuner_return_loss_optimize.disengagement_delay =
			10;
	si2151->si_prop.tuner_return_loss_optimize_2.thld = 31; /* (default 31) */
	si2151->si_prop.tuner_return_loss_optimize_2.window = 0; /* (default 0) */
	si2151->si_prop.tuner_return_loss_optimize_2.engagement_delay = 15; /* (default 15) */
	si2151->si_prop.tuner_tf1_boundary_offset.tf1_boundary_offset = 0;

	data = (si2151->si_prop.tuner_return_loss_optimize.thld
				& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_THLD_MASK)
				<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_THLD_LSB
				| (si2151->si_prop.tuner_return_loss_optimize.config
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_CONFIG_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_CONFIG_LSB
				| (si2151->si_prop.tuner_return_loss_optimize.engagement_delay
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_ENGAGEMENT_DELAY_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_ENGAGEMENT_DELAY_LSB
				| (si2151->si_prop.tuner_return_loss_optimize.disengagement_delay
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_DISENGAGEMENT_DELAY_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP_DISENGAGEMENT_DELAY_LSB;
	if ((ret = si2151_set_property(&si2151->tuner_client, 0,
			Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP, data,
			&si2151->si_cmd_reply)))
		pr_err("[si2151] %s: set Si2151_TUNER_RETURN_LOSS_OPTIMIZE_PROP error: %d.\n",
				__func__, ret);

	data = (si2151->si_prop.tuner_return_loss_optimize_2.thld
				& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_THLD_MASK)
				<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_THLD_LSB
				| (si2151->si_prop.tuner_return_loss_optimize_2.window
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_WINDOW_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_WINDOW_LSB
				| (si2151->si_prop.tuner_return_loss_optimize_2.engagement_delay
						& Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_ENGAGEMENT_DELAY_MASK)
						<< Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP_ENGAGEMENT_DELAY_LSB;
	if ((ret = si2151_set_property(&si2151->tuner_client, 0,
			Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP, data,
			&si2151->si_cmd_reply)))
		pr_err("[si2151] %s: set Si2151_TUNER_RETURN_LOSS_OPTIMIZE_2_PROP error: %d.\n",
				__func__, ret);

	data = (si2151->si_prop.tuner_tf1_boundary_offset.tf1_boundary_offset
				& Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP_TF1_BOUNDARY_OFFSET_MASK)
				<< Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP_TF1_BOUNDARY_OFFSET_LSB;
	if ((ret = si2151_set_property(&si2151->tuner_client, 0,
			Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP, data,
			&si2151->si_cmd_reply)))
		pr_err("[si2151] %s: set Si2151_TUNER_TF1_BOUNDARY_OFFSET_PROP error: %d.\n",
				__func__, ret);

	si2151->parm.frequency = p->frequency;
	if ((ret = si2151_tune(&si2151->tuner_client, si2151->parm.mode,
			si2151->parm.frequency + si2151->fre_offset, &si2151->si_cmd_reply,
			&si2151->si_common_reply))) {
		pr_err("[si2151] %s: tune frequency error: %d.\n", __func__, ret);

		mutex_unlock(&si2151->mutex);

		return -1;
	}

	pr_dbg("[si2151] %s: set frequency %u, frequency offset is %d.\n",
			__func__, si2151->parm.frequency, si2151->fre_offset);

	mutex_unlock(&si2151->mutex);

	return 0;
}

static int si2151_set_config_ex(struct dvb_frontend *fe, void *priv_cfg)
{
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;
	unsigned char mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV;

	mutex_lock(&si2151->mutex);

	if (fe->ops.info.type == FE_ANALOG)
		mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV;
	else
		mode = SI2151_TUNER_TUNE_FREQ_CMD_MODE_DTV;

	si2151_init_configure(si2151, mode);

	pr_info("[si2151] %s: fe type %d OK.\n", __func__, fe->ops.info.type);

	mutex_unlock(&si2151->mutex);

	return 0;
}

static int si2151_get_frequency_ex(struct dvb_frontend *fe, u32 *frequency)
{
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	*frequency = si2151->parm.frequency;

	return 0;
}

static int si2151_get_bandwidth_ex(struct dvb_frontend *fe, u32 *frequency)
{
	return 0;
}

static int si2151_get_if_frequency_ex(struct dvb_frontend *fe, u32 *frequency)
{
	/* frequency[0]: if_inv.
	 * frequency[1]: if_freq.
	 */

	int err_code = 0;
	int video_sys = 0;
	unsigned char inv_bit = 0;
	unsigned char mode_bit = 0;
	unsigned char mode_mask = 0;
	unsigned int prop_mode = 0;
	unsigned int prop_lif = 0;
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	if (si2151->parm.mode == SI2151_TUNER_TUNE_FREQ_CMD_MODE_ATV) {
		inv_bit = 0x09;
		mode_bit = 0x00;
		mode_mask = 0x07;
		prop_mode = SI2151_ATV_VIDEO_MODE_PROP;
		prop_lif = Si2151_ATV_LIF_FREQ_PROP;
	} else {
		inv_bit = 0x08;
		mode_bit = 0x04;
		mode_mask = 0x0f;
		prop_mode = SI2151_DTV_MODE_PROP;
		prop_lif = SI2151_DTV_LIF_FREQ_PROP;
	}

	/* lif_inv */
	err_code = si2151_get_property(&si2151->tuner_client, 0,
			prop_mode, &si2151->si_cmd_reply);
	if (err_code) {
		pr_err("[si2151] %s: si2151_get_property prop_mode error: %d.\n",
				__func__, err_code);

		mutex_unlock(&si2151->mutex);

		return err_code;
	}

	si2151->if_inv = (si2151->si_cmd_reply.get_property.data
			>> inv_bit) & 0x01;
	video_sys = (si2151->si_cmd_reply.get_property.data >> mode_bit) & mode_mask;
	/* lif_freq */
	err_code = si2151_get_property(&si2151->tuner_client, 0, prop_lif,
			&si2151->si_cmd_reply);
	if (err_code) {
		pr_err("[si2151] %s: si2151_get_property prop_lif error: %d.\n",
				__func__, err_code);

		mutex_unlock(&si2151->mutex);

		return err_code;
	}

	si2151->if_freq = si2151->si_cmd_reply.get_property.data * 1000;

	if (si2151->if_inv == 0 || video_sys == SI2151_ATV_VIDEO_MODE_PROP_VIDEO_SYS_LP) {
		if (video_sys == SI2151_ATV_VIDEO_MODE_PROP_VIDEO_SYS_LP)
			frequency[0] = 1;
		else
			frequency[0] = 0;

		frequency[1] = si2151->if_freq - si2151->fre_offset;
	} else {
		frequency[0] = 1;
		frequency[1] = si2151->if_freq + si2151->fre_offset;
	}

	pr_dbg("[si2151] %s: if_inv: %d, if_freq: %d.\n",
			__func__, frequency[0], frequency[1]);

	mutex_unlock(&si2151->mutex);

	return 0;
}

static int si2151_get_status_ex(struct dvb_frontend *fe, u32 *status)
{
#if 0
	int ret = 0;
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	if (!status) {
		pr_err("[si2151] %s: null pointer error.\n", __func__);

		return 0;
	}

	mutex_lock(&si2151->mutex);

	if ((ret = si2151_atv_status(&si2151->tuner_client,
			SI2151_ATV_STATUS_CMD_INTACK_OK,
			&si2151->si_cmd_reply))) {
		pr_info("[si2151] %s: si2151_atv_status error: %d.\n",
				__func__, ret);

		mutex_unlock(&si2151->mutex);

		return -1;
	} else {
		if (si2151->si_cmd_reply.atv_status.chl)
			*status = FE_HAS_LOCK;
		else
			*status = FE_TIMEDOUT;
	}

	mutex_unlock(&si2151->mutex);
#else
	*status = FE_TIMEDOUT;
#endif

	return 0;
}

static int si2151_get_rf_strength_ex(struct dvb_frontend *fe, u16 *strength)
{
#if 0
	int ret = 0;
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	if ((ret = si2151_tuner_status(&si2151->tuner_client,
			SI2151_ATV_STATUS_CMD_INTACK_OK,
			&si2151->si_cmd_reply))) {
		pr_err("[si2151] %s: si2151_tuner_status error: %d.\n",
				__func__, ret);

		mutex_unlock(&si2151->mutex);

		return -1;
	} else
		*strength = si2151->si_cmd_reply.tuner_status.rssi - 256;

	pr_info("[si2151] %s: strength = %d dbm\n", __func__, *strength);

	mutex_unlock(&si2151->mutex);
#else
	pr_info("[si2151] %s: please use get_strength.\n", __func__);
#endif
	return 0;
}

#ifdef CONFIG_AMLOGIC_DVB_COMPAT
static int si2151_get_strength_ex(struct dvb_frontend *fe, s16 *strength)
{
	int ret = 0;
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;
	char str;

	mutex_lock(&si2151->mutex);

	if ((ret = si2151_tuner_status(&si2151->tuner_client,
			SI2151_ATV_STATUS_CMD_INTACK_OK,
			&si2151->si_cmd_reply))) {
		pr_err("[si2151] %s: si2151_tuner_status error: %d.\n",
				__func__, ret);

		mutex_unlock(&si2151->mutex);

		return -1;
	} else {
		str = si2151->si_cmd_reply.tuner_status.rssi;
		if (str & 0x80)
			*strength = 0xFF80 | (str & 0x7F);
		else
			*strength = str & 0x7F;
	}
	pr_dbg("[si2151] %s: strength = %d dbm\n", __func__, *strength);

	mutex_unlock(&si2151->mutex);

	return 0;
}
#endif

static int si2151_get_afc_ex(struct dvb_frontend *fe, s32 *afc)
{
	return 0;
}

/*
 * This is support for demods like the mt352 - fills out the supplied
 * buffer with what to write.
 *
 * Don't use on newer drivers.
 */
static int si2151_calc_regs_ex(struct dvb_frontend *fe, u8 *buf, int buf_len)
{
	return 0;
}

/*
 * These are provided separately from set_params in order to
 * facilitate silicon tuners which require sophisticated tuning loops,
 * controlling each parameter separately.
 *
 * Don't use on newer drivers.
 */
static int si2151_set_frequency_ex(struct dvb_frontend *fe, u32 frequency)
{
	int ret = 0;
	struct si2151_device_s *si2151 = (struct si2151_device_s *)fe->tuner_priv;

	mutex_lock(&si2151->mutex);

	pr_dbg("[si2151] %s: now mode is %d(0-DTV,1-ATV)\n",
			__func__, si2151->parm.mode);

	si2151->parm.frequency = frequency;

	ret = si2151_tune(&si2151->tuner_client, si2151->parm.mode,
			si2151->parm.frequency + si2151->fre_offset,
			&si2151->si_cmd_reply,
			&si2151->si_common_reply);
	if (ret)
		pr_err("[si2151] %s: si2151_tune error: %d.\n", __func__, ret);

	mutex_unlock(&si2151->mutex);

	return ret;
}

static int si2151_set_bandwidth_ex(struct dvb_frontend *fe, u32 bandwidth)
{
	return 0;
}

static struct dvb_tuner_ops si2151_tuner_ops = {
	.info = {
		.name           = "si2151",
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 4, 0)
		.frequency_min_hz  = SI2151_TUNER_TUNE_FREQ_CMD_FREQ_MIN,
		.frequency_max_hz  = SI2151_TUNER_TUNE_FREQ_CMD_FREQ_MAX,
		.frequency_step_hz = 0,
#else
		.frequency_min  = SI2151_TUNER_TUNE_FREQ_CMD_FREQ_MIN,
		.frequency_max  = SI2151_TUNER_TUNE_FREQ_CMD_FREQ_MAX,
		.frequency_step = 0,
#endif
		.bandwidth_min  = 6000000,
		.bandwidth_max  = 8000000,
		.bandwidth_step = 0,
	},
	.release = si2151_release_ex,
	.init = si2151_init_ex,
	.sleep = si2151_sleep_ex,
	.suspend = si2151_suspend_ex,
	.resume = si2151_resume_ex,
	.set_params = si2151_set_params_ex,
	.set_analog_params = si2151_set_analog_params_ex,
	.calc_regs = si2151_calc_regs_ex,
	.set_config = si2151_set_config_ex,
	.get_frequency = si2151_get_frequency_ex,
	.get_bandwidth = si2151_get_bandwidth_ex,
	.get_if_frequency = si2151_get_if_frequency_ex,
	.get_status = si2151_get_status_ex,
	.get_rf_strength = si2151_get_rf_strength_ex,
#ifdef CONFIG_AMLOGIC_DVB_COMPAT
	.get_strength = si2151_get_strength_ex,
#endif
	.get_afc = si2151_get_afc_ex,
	.set_frequency = si2151_set_frequency_ex,
	.set_bandwidth = si2151_set_bandwidth_ex,
};

struct dvb_frontend *si2151_attach(struct dvb_frontend *fe,
		const struct tuner_config *cfg)
{
	struct dtv_frontend_properties *c = &fe->dtv_property_cache;

	if (!fe || !cfg || !cfg->i2c_adap || !si2151_devp) {
		pr_info("[si2151] %s NULL pointer of fe(%p) or cfg(%p) or i2c_adap(%p) or si2151_devp(%p).\n",
				__func__, fe, cfg, cfg ? cfg->i2c_adap : NULL, si2151_devp);
		return NULL;
	}

	fe->tuner_priv = si2151_devp;

	si2151_devp->tuner_client.adapter = cfg->i2c_adap;
	si2151_devp->tuner_client.addr = cfg->i2c_addr;

	if (!sprintf(si2151_devp->tuner_client.name, "si2151_tuner_i2c")) {
		pr_err("[si2151] %s: sprintf name error.\n", __func__);
	}

	if (cfg->xtal_mode == 0)
		SI2151_SHARE_XTAL = SI2151_POWER_UP_CMD_CLOCK_MODE_XTAL;
	else if (cfg->xtal_mode == 3)
		SI2151_SHARE_XTAL = SI2151_POWER_UP_CMD_CLOCK_MODE_EXTCLK;
	else
		pr_info("[si2151] %s: nonsupport xtal_mode %d, use default [%d].\n",
				__func__, cfg->xtal, SI2151_SHARE_XTAL);

	if (cfg->xtal_cap == 1)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_4P7PF;
	else if (cfg->xtal_cap == 2)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_5P2PF;
	else if (cfg->xtal_cap == 3)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_5P7PF;
	else if (cfg->xtal_cap == 4)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_6P2PF;
	else if (cfg->xtal_cap == 5)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_6P7PF;
	else if (cfg->xtal_cap == 6)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_7P2PF;
	else if (cfg->xtal_cap == 7)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_7P7PF;
	else if (cfg->xtal_cap == 8)
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_8P2PF;
	else
		SI2151_XTAL_CAP = SI2151_CRYSTAL_TRIM_PROP_XO_CAP_DEFAULT;

	memcpy(&fe->ops.tuner_ops, &si2151_tuner_ops,
			sizeof(struct dvb_tuner_ops));

	pr_info("[si2151] %s: done.\n", __func__);

	si2151_init_ex(fe);
	c->modulation = VSB_8;
	c->frequency = 803000000;
	//msleep(500);
	si2151_set_params_ex(fe);

	return fe;
}
EXPORT_SYMBOL(si2151_attach);
#endif /* LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0) */
