/* SPDX-License-Identifier: GPL-2.0 WITH Linux-syscall-note */
/*
 *
 * (C) COPYRIGHT 2015-2022 ARM Limited. All rights reserved.
 *
 * This program is free software and is provided to you under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation, and any use by you of this program is subject to the terms
 * of such GNU license.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can access it online at
 * http://www.gnu.org/licenses/gpl-2.0.html.
 *
 */

#if !defined(_KBASE_TIMELINE_H)
#define _KBASE_TIMELINE_H

#include <mali_kbase.h>

/*****************************************************************************/

struct kbase_timeline;

/**
 * kbase_timeline_init - initialize timeline infrastructure in kernel
 * @timeline:       Newly created instance of kbase_timeline will be stored in
 *                  this pointer.
 * @timeline_flags: Timeline status will be written to this variable when a
 *                  client is attached/detached. The variable must be valid
 *                  while timeline instance is valid.
 * Return: zero on success, negative number on error
 */
int kbase_timeline_init(struct kbase_timeline **timeline,
	atomic_t *timeline_flags);

/**
 * kbase_timeline_term - terminate timeline infrastructure in kernel
 *
 * @timeline:     Timeline instance to be terminated. It must be previously created
 *                with kbase_timeline_init().
 */
void kbase_timeline_term(struct kbase_timeline *timeline);

/**
 * kbase_timeline_io_acquire - acquire timeline stream file descriptor
 * @kbdev:     Kbase device
 * @flags:     Timeline stream flags
 *
 * This descriptor is meant to be used by userspace timeline to gain access to
 * kernel timeline stream. This stream is later broadcasted by user space to the
 * timeline client.
 * Only one entity can own the descriptor at any given time. Descriptor shall be
 * closed if unused. If descriptor cannot be obtained (i.e. when it is already
 * being used) return will be a negative value.
 *
 * Return: file descriptor on success, negative number on error
 */
int kbase_timeline_io_acquire(struct kbase_device *kbdev, u32 flags);

/**
 * kbase_timeline_streams_flush - flush timeline streams.
 * @timeline:     Timeline instance
 *
 * Function will flush pending data in all timeline streams.
 *
 * Return: Zero on success, errno on failure.
 */
int kbase_timeline_streams_flush(struct kbase_timeline *timeline);

/**
 * kbase_timeline_streams_body_reset - reset timeline body streams.
 * @timeline:     Timeline instance
 *
 * Function will discard pending data in all timeline body streams.
 */
void kbase_timeline_streams_body_reset(struct kbase_timeline *timeline);

/**
 * kbase_timeline_post_kbase_context_create - Inform timeline that a new KBase
 *                                            Context has been created.
 * @kctx:    KBase Context
 */
void kbase_timeline_post_kbase_context_create(struct kbase_context *kctx);

/**
 * kbase_timeline_pre_kbase_context_destroy - Inform timeline that a KBase
 *                                            Context is about to be destroyed.
 * @kctx:    KBase Context
 */
void kbase_timeline_pre_kbase_context_destroy(struct kbase_context *kctx);

/**
 * kbase_timeline_post_kbase_context_destroy - Inform timeline that a KBase
 *                                             Context has been destroyed.
 * @kctx:    KBase Context
 *
 * Should be called immediately before the memory is freed, and the context ID
 * and kbdev pointer should still be valid.
 */
void kbase_timeline_post_kbase_context_destroy(struct kbase_context *kctx);

#if MALI_UNIT_TEST

/**
 * kbase_timeline_stats - read timeline stream statistics
 * @timeline:        Timeline instance
 * @bytes_collected: Will hold number of bytes read by the user
 * @bytes_generated: Will hold number of bytes generated by trace points
 */
void kbase_timeline_stats(struct kbase_timeline *timeline, u32 *bytes_collected, u32 *bytes_generated);
#endif /* MALI_UNIT_TEST */

/**
 * kbase_timeline_io_debugfs_init - Add a debugfs entry for reading timeline stream data
 *
 * @kbdev: An instance of the GPU platform device, allocated from the probe
 *         method of the driver.
 */
void kbase_timeline_io_debugfs_init(struct kbase_device *kbdev);

#endif /* _KBASE_TIMELINE_H */
