/*
** FAAD2 - Freeware Advanced Audio (AAC) Decoder including SBR decoding
** Copyright (C) 2003-2005 M. Bakker, Nero AG, http://www.nero.com
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** Any non-GPL usage of this software or parts of this software is strictly
** forbidden.
**
** The "appropriate copyright message" mentioned in section 2c of the GPLv2
** must read: "Code from FAAD2 is copyright (c) Nero AG, www.nero.com"
**
** Commercial non-GPL licensing of this software is possible.
** For more info contact Nero AG through Mpeg4AAClicense@nero.com.
**
** $Id: sbr_e_nf.c,v 1.22 2008/03/23 23:03:29 menno Exp $
**/
#include <stdlib.h>
#include "common.h"
#include "structs.h"

#ifdef SBR_DEC


#include "sbr_syntax.h"
#include "sbr_e_nf.h"

void extract_envelope_data(sbr_info *sbr, uint8_t ch)
{
    uint8_t l, k;

    for (l = 0; l < sbr->L_E[ch]; l++) {
        if (sbr->bs_df_env[ch][l] == 0) {
            for (k = 1; k < sbr->n[sbr->f[ch][l]]; k++) {
                sbr->E[ch][k][l] = sbr->E[ch][k - 1][l] + sbr->E[ch][k][l];
                if (sbr->E[ch][k][l] < 0) {
                    sbr->E[ch][k][l] = 0;
                }
            }

        } else { /* bs_df_env == 1 */

            uint8_t g = (l == 0) ? sbr->f_prev[ch] : sbr->f[ch][l - 1];
            int16_t E_prev;

            if (sbr->f[ch][l] == g) {
                for (k = 0; k < sbr->n[sbr->f[ch][l]]; k++) {
                    if (l == 0) {
                        E_prev = sbr->E_prev[ch][k];
                    } else {
                        E_prev = sbr->E[ch][k][l - 1];
                    }

                    sbr->E[ch][k][l] = E_prev + sbr->E[ch][k][l];
                }

            } else if ((g == 1) && (sbr->f[ch][l] == 0)) {
                uint8_t i;

                for (k = 0; k < sbr->n[sbr->f[ch][l]]; k++) {
                    for (i = 0; i < sbr->N_high; i++) {
                        if (sbr->f_table_res[HI_RES][i] == sbr->f_table_res[LO_RES][k]) {
                            if (l == 0) {
                                E_prev = sbr->E_prev[ch][i];
                            } else {
                                E_prev = sbr->E[ch][i][l - 1];
                            }

                            sbr->E[ch][k][l] = E_prev + sbr->E[ch][k][l];
                        }
                    }
                }

            } else if ((g == 0) && (sbr->f[ch][l] == 1)) {
                uint8_t i;

                for (k = 0; k < sbr->n[sbr->f[ch][l]]; k++) {
                    for (i = 0; i < sbr->N_low; i++) {
                        if ((sbr->f_table_res[LO_RES][i] <= sbr->f_table_res[HI_RES][k]) &&
                            (sbr->f_table_res[HI_RES][k] < sbr->f_table_res[LO_RES][i + 1])) {
                            if (l == 0) {
                                E_prev = sbr->E_prev[ch][i];
                            } else {
                                E_prev = sbr->E[ch][i][l - 1];
                            }

                            sbr->E[ch][k][l] = E_prev + sbr->E[ch][k][l];
                        }
                    }
                }
            }
        }
    }
}

void extract_noise_floor_data(sbr_info *sbr, uint8_t ch)
{
    uint8_t l, k;

    for (l = 0; l < sbr->L_Q[ch]; l++) {
        if (sbr->bs_df_noise[ch][l] == 0) {
            for (k = 1; k < sbr->N_Q; k++) {
                sbr->Q[ch][k][l] = sbr->Q[ch][k][l] + sbr->Q[ch][k - 1][l];
            }
        } else {
            if (l == 0) {
                for (k = 0; k < sbr->N_Q; k++) {
                    sbr->Q[ch][k][l] = sbr->Q_prev[ch][k] + sbr->Q[ch][k][0];
                }
            } else {
                for (k = 0; k < sbr->N_Q; k++) {
                    sbr->Q[ch][k][l] = sbr->Q[ch][k][l - 1] + sbr->Q[ch][k][l];
                }
            }
        }
    }
}

#ifndef FIXED_POINT

/* table for Q_div values when no coupling */
static const real_t Q_div_tab[31] = {
    FRAC_CONST(0.0153846), FRAC_CONST(0.030303),
    FRAC_CONST(0.0588235), FRAC_CONST(0.111111),
    FRAC_CONST(0.2),       FRAC_CONST(0.333333),
    FRAC_CONST(0.5),       FRAC_CONST(0.666667),
    FRAC_CONST(0.8),       FRAC_CONST(0.888889),
    FRAC_CONST(0.941176),  FRAC_CONST(0.969697),
    FRAC_CONST(0.984615),  FRAC_CONST(0.992248),
    FRAC_CONST(0.996109),  FRAC_CONST(0.998051),
    FRAC_CONST(0.999024),  FRAC_CONST(0.999512),
    FRAC_CONST(0.999756),  FRAC_CONST(0.999878),
    FRAC_CONST(0.999939),  FRAC_CONST(0.999969),
    FRAC_CONST(0.999985),  FRAC_CONST(0.999992),
    FRAC_CONST(0.999996),  FRAC_CONST(0.999998),
    FRAC_CONST(0.999999),  FRAC_CONST(1),
    FRAC_CONST(1),         FRAC_CONST(1),
    FRAC_CONST(1)
};

static const real_t Q_div_tab_left[31][13] = {
    { FRAC_CONST(0.969704), FRAC_CONST(0.888985), FRAC_CONST(0.667532), FRAC_CONST(0.336788), FRAC_CONST(0.117241), FRAC_CONST(0.037594), FRAC_CONST(0.0153846), FRAC_CONST(0.00967118), FRAC_CONST(0.00823245), FRAC_CONST(0.00787211), FRAC_CONST(0.00778198), FRAC_CONST(0.00775945), FRAC_CONST(0.00775382) },
    { FRAC_CONST(0.984619), FRAC_CONST(0.94123), FRAC_CONST(0.800623), FRAC_CONST(0.503876), FRAC_CONST(0.209877), FRAC_CONST(0.0724638), FRAC_CONST(0.030303), FRAC_CONST(0.0191571), FRAC_CONST(0.0163305), FRAC_CONST(0.0156212), FRAC_CONST(0.0154438), FRAC_CONST(0.0153994), FRAC_CONST(0.0153883) },
    { FRAC_CONST(0.99225), FRAC_CONST(0.969726), FRAC_CONST(0.889273), FRAC_CONST(0.670103), FRAC_CONST(0.346939), FRAC_CONST(0.135135), FRAC_CONST(0.0588235), FRAC_CONST(0.037594), FRAC_CONST(0.0321361), FRAC_CONST(0.0307619), FRAC_CONST(0.0304178), FRAC_CONST(0.0303317), FRAC_CONST(0.0303102) },
    { FRAC_CONST(0.99611), FRAC_CONST(0.98463), FRAC_CONST(0.941392), FRAC_CONST(0.802469), FRAC_CONST(0.515152), FRAC_CONST(0.238095), FRAC_CONST(0.111111), FRAC_CONST(0.0724638), FRAC_CONST(0.0622711), FRAC_CONST(0.0596878), FRAC_CONST(0.0590397), FRAC_CONST(0.0588776), FRAC_CONST(0.058837) },
    { FRAC_CONST(0.998051), FRAC_CONST(0.992256), FRAC_CONST(0.969811), FRAC_CONST(0.890411), FRAC_CONST(0.68), FRAC_CONST(0.384615), FRAC_CONST(0.2), FRAC_CONST(0.135135), FRAC_CONST(0.117241), FRAC_CONST(0.112652), FRAC_CONST(0.111497), FRAC_CONST(0.111208), FRAC_CONST(0.111135) },
    { FRAC_CONST(0.999025), FRAC_CONST(0.996113), FRAC_CONST(0.984674), FRAC_CONST(0.942029), FRAC_CONST(0.809524), FRAC_CONST(0.555556), FRAC_CONST(0.333333), FRAC_CONST(0.238095), FRAC_CONST(0.209877), FRAC_CONST(0.202492), FRAC_CONST(0.200625), FRAC_CONST(0.200156), FRAC_CONST(0.200039) },
    { FRAC_CONST(0.999512), FRAC_CONST(0.998053), FRAC_CONST(0.992278), FRAC_CONST(0.970149), FRAC_CONST(0.894737), FRAC_CONST(0.714286), FRAC_CONST(0.5), FRAC_CONST(0.384615), FRAC_CONST(0.346939), FRAC_CONST(0.336788), FRAC_CONST(0.3342), FRAC_CONST(0.33355), FRAC_CONST(0.333388) },
    { FRAC_CONST(0.999756), FRAC_CONST(0.999025), FRAC_CONST(0.996124), FRAC_CONST(0.984848), FRAC_CONST(0.944444), FRAC_CONST(0.833333), FRAC_CONST(0.666667), FRAC_CONST(0.555556), FRAC_CONST(0.515152), FRAC_CONST(0.503876), FRAC_CONST(0.500975), FRAC_CONST(0.500244), FRAC_CONST(0.500061) },
    { FRAC_CONST(0.999878), FRAC_CONST(0.999512), FRAC_CONST(0.998058), FRAC_CONST(0.992366), FRAC_CONST(0.971429), FRAC_CONST(0.909091), FRAC_CONST(0.8), FRAC_CONST(0.714286), FRAC_CONST(0.68), FRAC_CONST(0.670103), FRAC_CONST(0.667532), FRAC_CONST(0.666884), FRAC_CONST(0.666721) },
    { FRAC_CONST(0.999939), FRAC_CONST(0.999756), FRAC_CONST(0.999028), FRAC_CONST(0.996169), FRAC_CONST(0.985507), FRAC_CONST(0.952381), FRAC_CONST(0.888889), FRAC_CONST(0.833333), FRAC_CONST(0.809524), FRAC_CONST(0.802469), FRAC_CONST(0.800623), FRAC_CONST(0.800156), FRAC_CONST(0.800039) },
    { FRAC_CONST(0.999969), FRAC_CONST(0.999878), FRAC_CONST(0.999514), FRAC_CONST(0.998081), FRAC_CONST(0.992701), FRAC_CONST(0.97561), FRAC_CONST(0.941176), FRAC_CONST(0.909091), FRAC_CONST(0.894737), FRAC_CONST(0.890411), FRAC_CONST(0.889273), FRAC_CONST(0.888985), FRAC_CONST(0.888913) },
    { FRAC_CONST(0.999985), FRAC_CONST(0.999939), FRAC_CONST(0.999757), FRAC_CONST(0.999039), FRAC_CONST(0.996337), FRAC_CONST(0.987654), FRAC_CONST(0.969697), FRAC_CONST(0.952381), FRAC_CONST(0.944444), FRAC_CONST(0.942029), FRAC_CONST(0.941392), FRAC_CONST(0.94123), FRAC_CONST(0.94119) },
    { FRAC_CONST(0.999992), FRAC_CONST(0.99997), FRAC_CONST(0.999878), FRAC_CONST(0.999519), FRAC_CONST(0.998165), FRAC_CONST(0.993789), FRAC_CONST(0.984615), FRAC_CONST(0.97561), FRAC_CONST(0.971429), FRAC_CONST(0.970149), FRAC_CONST(0.969811), FRAC_CONST(0.969726), FRAC_CONST(0.969704) },
    { FRAC_CONST(0.999996), FRAC_CONST(0.999985), FRAC_CONST(0.999939), FRAC_CONST(0.99976), FRAC_CONST(0.999082), FRAC_CONST(0.996885), FRAC_CONST(0.992248), FRAC_CONST(0.987654), FRAC_CONST(0.985507), FRAC_CONST(0.984848), FRAC_CONST(0.984674), FRAC_CONST(0.98463), FRAC_CONST(0.984619) },
    { FRAC_CONST(0.999998), FRAC_CONST(0.999992), FRAC_CONST(0.99997), FRAC_CONST(0.99988), FRAC_CONST(0.999541), FRAC_CONST(0.99844), FRAC_CONST(0.996109), FRAC_CONST(0.993789), FRAC_CONST(0.992701), FRAC_CONST(0.992366), FRAC_CONST(0.992278), FRAC_CONST(0.992256), FRAC_CONST(0.99225) },
    { FRAC_CONST(0.999999), FRAC_CONST(0.999996), FRAC_CONST(0.999985), FRAC_CONST(0.99994), FRAC_CONST(0.99977), FRAC_CONST(0.999219), FRAC_CONST(0.998051), FRAC_CONST(0.996885), FRAC_CONST(0.996337), FRAC_CONST(0.996169), FRAC_CONST(0.996124), FRAC_CONST(0.996113), FRAC_CONST(0.99611) },
    { FRAC_CONST(1), FRAC_CONST(0.999998), FRAC_CONST(0.999992), FRAC_CONST(0.99997), FRAC_CONST(0.999885), FRAC_CONST(0.99961), FRAC_CONST(0.999024), FRAC_CONST(0.99844), FRAC_CONST(0.998165), FRAC_CONST(0.998081), FRAC_CONST(0.998058), FRAC_CONST(0.998053), FRAC_CONST(0.998051) },
    { FRAC_CONST(1), FRAC_CONST(0.999999), FRAC_CONST(0.999996), FRAC_CONST(0.999985), FRAC_CONST(0.999943), FRAC_CONST(0.999805), FRAC_CONST(0.999512), FRAC_CONST(0.999219), FRAC_CONST(0.999082), FRAC_CONST(0.999039), FRAC_CONST(0.999028), FRAC_CONST(0.999025), FRAC_CONST(0.999025) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999998), FRAC_CONST(0.999992), FRAC_CONST(0.999971), FRAC_CONST(0.999902), FRAC_CONST(0.999756), FRAC_CONST(0.99961), FRAC_CONST(0.999541), FRAC_CONST(0.999519), FRAC_CONST(0.999514), FRAC_CONST(0.999512), FRAC_CONST(0.999512) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999999), FRAC_CONST(0.999996), FRAC_CONST(0.999986), FRAC_CONST(0.999951), FRAC_CONST(0.999878), FRAC_CONST(0.999805), FRAC_CONST(0.99977), FRAC_CONST(0.99976), FRAC_CONST(0.999757), FRAC_CONST(0.999756), FRAC_CONST(0.999756) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999998), FRAC_CONST(0.999993), FRAC_CONST(0.999976), FRAC_CONST(0.999939), FRAC_CONST(0.999902), FRAC_CONST(0.999885), FRAC_CONST(0.99988), FRAC_CONST(0.999878), FRAC_CONST(0.999878), FRAC_CONST(0.999878) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999999), FRAC_CONST(0.999996), FRAC_CONST(0.999988), FRAC_CONST(0.999969), FRAC_CONST(0.999951), FRAC_CONST(0.999943), FRAC_CONST(0.99994), FRAC_CONST(0.999939), FRAC_CONST(0.999939), FRAC_CONST(0.999939) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999998), FRAC_CONST(0.999994), FRAC_CONST(0.999985), FRAC_CONST(0.999976), FRAC_CONST(0.999971), FRAC_CONST(0.99997), FRAC_CONST(0.99997), FRAC_CONST(0.99997), FRAC_CONST(0.999969) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999999), FRAC_CONST(0.999997), FRAC_CONST(0.999992), FRAC_CONST(0.999988), FRAC_CONST(0.999986), FRAC_CONST(0.999985), FRAC_CONST(0.999985), FRAC_CONST(0.999985), FRAC_CONST(0.999985) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999998), FRAC_CONST(0.999996), FRAC_CONST(0.999994), FRAC_CONST(0.999993), FRAC_CONST(0.999992), FRAC_CONST(0.999992), FRAC_CONST(0.999992), FRAC_CONST(0.999992) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999999), FRAC_CONST(0.999998), FRAC_CONST(0.999997), FRAC_CONST(0.999996), FRAC_CONST(0.999996), FRAC_CONST(0.999996), FRAC_CONST(0.999996), FRAC_CONST(0.999996) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999999), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) }
};

static const real_t Q_div_tab_right[31][13] = {
    { FRAC_CONST(0.00775382), FRAC_CONST(0.00775945), FRAC_CONST(0.00778198), FRAC_CONST(0.00787211), FRAC_CONST(0.00823245), FRAC_CONST(0.00967118), FRAC_CONST(0.0153846), FRAC_CONST(0.037594), FRAC_CONST(0.117241), FRAC_CONST(0.336788), FRAC_CONST(0.667532), FRAC_CONST(0.888985), FRAC_CONST(0.969704) },
    { FRAC_CONST(0.0153883), FRAC_CONST(0.0153994), FRAC_CONST(0.0154438), FRAC_CONST(0.0156212), FRAC_CONST(0.0163305), FRAC_CONST(0.0191571), FRAC_CONST(0.030303), FRAC_CONST(0.0724638), FRAC_CONST(0.209877), FRAC_CONST(0.503876), FRAC_CONST(0.800623), FRAC_CONST(0.94123), FRAC_CONST(0.984619) },
    { FRAC_CONST(0.0303102), FRAC_CONST(0.0303317), FRAC_CONST(0.0304178), FRAC_CONST(0.0307619), FRAC_CONST(0.0321361), FRAC_CONST(0.037594), FRAC_CONST(0.0588235), FRAC_CONST(0.135135), FRAC_CONST(0.346939), FRAC_CONST(0.670103), FRAC_CONST(0.889273), FRAC_CONST(0.969726), FRAC_CONST(0.99225) },
    { FRAC_CONST(0.058837), FRAC_CONST(0.0588776), FRAC_CONST(0.0590397), FRAC_CONST(0.0596878), FRAC_CONST(0.0622711), FRAC_CONST(0.0724638), FRAC_CONST(0.111111), FRAC_CONST(0.238095), FRAC_CONST(0.515152), FRAC_CONST(0.802469), FRAC_CONST(0.941392), FRAC_CONST(0.98463), FRAC_CONST(0.99611) },
    { FRAC_CONST(0.111135), FRAC_CONST(0.111208), FRAC_CONST(0.111497), FRAC_CONST(0.112652), FRAC_CONST(0.117241), FRAC_CONST(0.135135), FRAC_CONST(0.2), FRAC_CONST(0.384615), FRAC_CONST(0.68), FRAC_CONST(0.890411), FRAC_CONST(0.969811), FRAC_CONST(0.992256), FRAC_CONST(0.998051) },
    { FRAC_CONST(0.200039), FRAC_CONST(0.200156), FRAC_CONST(0.200625), FRAC_CONST(0.202492), FRAC_CONST(0.209877), FRAC_CONST(0.238095), FRAC_CONST(0.333333), FRAC_CONST(0.555556), FRAC_CONST(0.809524), FRAC_CONST(0.942029), FRAC_CONST(0.984674), FRAC_CONST(0.996113), FRAC_CONST(0.999025) },
    { FRAC_CONST(0.333388), FRAC_CONST(0.33355), FRAC_CONST(0.3342), FRAC_CONST(0.336788), FRAC_CONST(0.346939), FRAC_CONST(0.384615), FRAC_CONST(0.5), FRAC_CONST(0.714286), FRAC_CONST(0.894737), FRAC_CONST(0.970149), FRAC_CONST(0.992278), FRAC_CONST(0.998053), FRAC_CONST(0.999512) },
    { FRAC_CONST(0.500061), FRAC_CONST(0.500244), FRAC_CONST(0.500975), FRAC_CONST(0.503876), FRAC_CONST(0.515152), FRAC_CONST(0.555556), FRAC_CONST(0.666667), FRAC_CONST(0.833333), FRAC_CONST(0.944444), FRAC_CONST(0.984848), FRAC_CONST(0.996124), FRAC_CONST(0.999025), FRAC_CONST(0.999756) },
    { FRAC_CONST(0.666721), FRAC_CONST(0.666884), FRAC_CONST(0.667532), FRAC_CONST(0.670103), FRAC_CONST(0.68), FRAC_CONST(0.714286), FRAC_CONST(0.8), FRAC_CONST(0.909091), FRAC_CONST(0.971429), FRAC_CONST(0.992366), FRAC_CONST(0.998058), FRAC_CONST(0.999512), FRAC_CONST(0.999878) },
    { FRAC_CONST(0.800039), FRAC_CONST(0.800156), FRAC_CONST(0.800623), FRAC_CONST(0.802469), FRAC_CONST(0.809524), FRAC_CONST(0.833333), FRAC_CONST(0.888889), FRAC_CONST(0.952381), FRAC_CONST(0.985507), FRAC_CONST(0.996169), FRAC_CONST(0.999028), FRAC_CONST(0.999756), FRAC_CONST(0.999939) },
    { FRAC_CONST(0.888913), FRAC_CONST(0.888985), FRAC_CONST(0.889273), FRAC_CONST(0.890411), FRAC_CONST(0.894737), FRAC_CONST(0.909091), FRAC_CONST(0.941176), FRAC_CONST(0.97561), FRAC_CONST(0.992701), FRAC_CONST(0.998081), FRAC_CONST(0.999514), FRAC_CONST(0.999878), FRAC_CONST(0.999969) },
    { FRAC_CONST(0.94119), FRAC_CONST(0.94123), FRAC_CONST(0.941392), FRAC_CONST(0.942029), FRAC_CONST(0.944444), FRAC_CONST(0.952381), FRAC_CONST(0.969697), FRAC_CONST(0.987654), FRAC_CONST(0.996337), FRAC_CONST(0.999039), FRAC_CONST(0.999757), FRAC_CONST(0.999939), FRAC_CONST(0.999985) },
    { FRAC_CONST(0.969704), FRAC_CONST(0.969726), FRAC_CONST(0.969811), FRAC_CONST(0.970149), FRAC_CONST(0.971429), FRAC_CONST(0.97561), FRAC_CONST(0.984615), FRAC_CONST(0.993789), FRAC_CONST(0.998165), FRAC_CONST(0.999519), FRAC_CONST(0.999878), FRAC_CONST(0.99997), FRAC_CONST(0.999992) },
    { FRAC_CONST(0.984619), FRAC_CONST(0.98463), FRAC_CONST(0.984674), FRAC_CONST(0.984848), FRAC_CONST(0.985507), FRAC_CONST(0.987654), FRAC_CONST(0.992248), FRAC_CONST(0.996885), FRAC_CONST(0.999082), FRAC_CONST(0.99976), FRAC_CONST(0.999939), FRAC_CONST(0.999985), FRAC_CONST(0.999996) },
    { FRAC_CONST(0.99225), FRAC_CONST(0.992256), FRAC_CONST(0.992278), FRAC_CONST(0.992366), FRAC_CONST(0.992701), FRAC_CONST(0.993789), FRAC_CONST(0.996109), FRAC_CONST(0.99844), FRAC_CONST(0.999541), FRAC_CONST(0.99988), FRAC_CONST(0.99997), FRAC_CONST(0.999992), FRAC_CONST(0.999998) },
    { FRAC_CONST(0.99611), FRAC_CONST(0.996113), FRAC_CONST(0.996124), FRAC_CONST(0.996169), FRAC_CONST(0.996337), FRAC_CONST(0.996885), FRAC_CONST(0.998051), FRAC_CONST(0.999219), FRAC_CONST(0.99977), FRAC_CONST(0.99994), FRAC_CONST(0.999985), FRAC_CONST(0.999996), FRAC_CONST(0.999999) },
    { FRAC_CONST(0.998051), FRAC_CONST(0.998053), FRAC_CONST(0.998058), FRAC_CONST(0.998081), FRAC_CONST(0.998165), FRAC_CONST(0.99844), FRAC_CONST(0.999024), FRAC_CONST(0.99961), FRAC_CONST(0.999885), FRAC_CONST(0.99997), FRAC_CONST(0.999992), FRAC_CONST(0.999998), FRAC_CONST(1) },
    { FRAC_CONST(0.999025), FRAC_CONST(0.999025), FRAC_CONST(0.999028), FRAC_CONST(0.999039), FRAC_CONST(0.999082), FRAC_CONST(0.999219), FRAC_CONST(0.999512), FRAC_CONST(0.999805), FRAC_CONST(0.999943), FRAC_CONST(0.999985), FRAC_CONST(0.999996), FRAC_CONST(0.999999), FRAC_CONST(1) },
    { FRAC_CONST(0.999512), FRAC_CONST(0.999512), FRAC_CONST(0.999514), FRAC_CONST(0.999519), FRAC_CONST(0.999541), FRAC_CONST(0.99961), FRAC_CONST(0.999756), FRAC_CONST(0.999902), FRAC_CONST(0.999971), FRAC_CONST(0.999992), FRAC_CONST(0.999998), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999756), FRAC_CONST(0.999756), FRAC_CONST(0.999757), FRAC_CONST(0.99976), FRAC_CONST(0.99977), FRAC_CONST(0.999805), FRAC_CONST(0.999878), FRAC_CONST(0.999951), FRAC_CONST(0.999986), FRAC_CONST(0.999996), FRAC_CONST(0.999999), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999878), FRAC_CONST(0.999878), FRAC_CONST(0.999878), FRAC_CONST(0.99988), FRAC_CONST(0.999885), FRAC_CONST(0.999902), FRAC_CONST(0.999939), FRAC_CONST(0.999976), FRAC_CONST(0.999993), FRAC_CONST(0.999998), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999939), FRAC_CONST(0.999939), FRAC_CONST(0.999939), FRAC_CONST(0.99994), FRAC_CONST(0.999943), FRAC_CONST(0.999951), FRAC_CONST(0.999969), FRAC_CONST(0.999988), FRAC_CONST(0.999996), FRAC_CONST(0.999999), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999969), FRAC_CONST(0.99997), FRAC_CONST(0.99997), FRAC_CONST(0.99997), FRAC_CONST(0.999971), FRAC_CONST(0.999976), FRAC_CONST(0.999985), FRAC_CONST(0.999994), FRAC_CONST(0.999998), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999985), FRAC_CONST(0.999985), FRAC_CONST(0.999985), FRAC_CONST(0.999985), FRAC_CONST(0.999986), FRAC_CONST(0.999988), FRAC_CONST(0.999992), FRAC_CONST(0.999997), FRAC_CONST(0.999999), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999992), FRAC_CONST(0.999992), FRAC_CONST(0.999992), FRAC_CONST(0.999992), FRAC_CONST(0.999993), FRAC_CONST(0.999994), FRAC_CONST(0.999996), FRAC_CONST(0.999998), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999996), FRAC_CONST(0.999996), FRAC_CONST(0.999996), FRAC_CONST(0.999996), FRAC_CONST(0.999996), FRAC_CONST(0.999997), FRAC_CONST(0.999998), FRAC_CONST(0.999999), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999998), FRAC_CONST(0.999999), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(0.999999), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) },
    { FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1), FRAC_CONST(1) }
};

/* calculates 1/(1+Q) */
/* [0..1] */
static real_t calc_Q_div(sbr_info *sbr, uint8_t ch, uint8_t m, uint8_t l)
{
    if (sbr->bs_coupling) {
        /* left channel */
        if ((sbr->Q[0][m][l] < 0 || sbr->Q[0][m][l] > 30) ||
            (sbr->Q[1][m][l] < 0 || sbr->Q[1][m][l] > 24 /* 2*panOffset(1) */)) {
            return 0;
        } else {
            /* the pan parameter is always even */
            if (ch == 0) {
                return Q_div_tab_left[sbr->Q[0][m][l]][sbr->Q[1][m][l] >> 1];
            } else {
                return Q_div_tab_right[sbr->Q[0][m][l]][sbr->Q[1][m][l] >> 1];
            }
        }
    } else {
        /* no coupling */
        if (sbr->Q[ch][m][l] < 0 || sbr->Q[ch][m][l] > 30) {
            return 0;
        } else {
            return Q_div_tab[sbr->Q[ch][m][l]];
        }
    }
}

/* table for Q_div2 values when no coupling */
static const real_t Q_div2_tab[31] = {
    FRAC_CONST(0.984615),     FRAC_CONST(0.969697),
    FRAC_CONST(0.941176),     FRAC_CONST(0.888889),
    FRAC_CONST(0.8),          FRAC_CONST(0.666667),
    FRAC_CONST(0.5),          FRAC_CONST(0.333333),
    FRAC_CONST(0.2),          FRAC_CONST(0.111111),
    FRAC_CONST(0.0588235),    FRAC_CONST(0.030303),
    FRAC_CONST(0.0153846),    FRAC_CONST(0.00775194),
    FRAC_CONST(0.00389105),   FRAC_CONST(0.00194932),
    FRAC_CONST(0.00097561),   FRAC_CONST(0.000488043),
    FRAC_CONST(0.000244081),  FRAC_CONST(0.000122055),
    FRAC_CONST(6.10314E-005), FRAC_CONST(3.05166E-005),
    FRAC_CONST(1.52586E-005), FRAC_CONST(7.62934E-006),
    FRAC_CONST(3.81468E-006), FRAC_CONST(1.90734E-006),
    FRAC_CONST(9.53673E-007), FRAC_CONST(4.76837E-007),
    FRAC_CONST(2.38419E-007), FRAC_CONST(1.19209E-007),
    FRAC_CONST(5.96046E-008)
};

static const real_t Q_div2_tab_left[31][13] = {
    { FRAC_CONST(0.0302959), FRAC_CONST(0.111015), FRAC_CONST(0.332468), FRAC_CONST(0.663212), FRAC_CONST(0.882759), FRAC_CONST(0.962406), FRAC_CONST(0.984615), FRAC_CONST(0.990329), FRAC_CONST(0.991768), FRAC_CONST(0.992128), FRAC_CONST(0.992218), FRAC_CONST(0.992241), FRAC_CONST(0.992246) },
    { FRAC_CONST(0.0153809), FRAC_CONST(0.0587695), FRAC_CONST(0.199377), FRAC_CONST(0.496124), FRAC_CONST(0.790123), FRAC_CONST(0.927536), FRAC_CONST(0.969697), FRAC_CONST(0.980843), FRAC_CONST(0.98367), FRAC_CONST(0.984379), FRAC_CONST(0.984556), FRAC_CONST(0.984601), FRAC_CONST(0.984612) },
    { FRAC_CONST(0.00775006), FRAC_CONST(0.0302744), FRAC_CONST(0.110727), FRAC_CONST(0.329897), FRAC_CONST(0.653061), FRAC_CONST(0.864865), FRAC_CONST(0.941176), FRAC_CONST(0.962406), FRAC_CONST(0.967864), FRAC_CONST(0.969238), FRAC_CONST(0.969582), FRAC_CONST(0.969668), FRAC_CONST(0.96969) },
    { FRAC_CONST(0.0038901), FRAC_CONST(0.0153698), FRAC_CONST(0.0586081), FRAC_CONST(0.197531), FRAC_CONST(0.484848), FRAC_CONST(0.761905), FRAC_CONST(0.888889), FRAC_CONST(0.927536), FRAC_CONST(0.937729), FRAC_CONST(0.940312), FRAC_CONST(0.94096), FRAC_CONST(0.941122), FRAC_CONST(0.941163) },
    { FRAC_CONST(0.00194884), FRAC_CONST(0.00774443), FRAC_CONST(0.0301887), FRAC_CONST(0.109589), FRAC_CONST(0.32), FRAC_CONST(0.615385), FRAC_CONST(0.8), FRAC_CONST(0.864865), FRAC_CONST(0.882759), FRAC_CONST(0.887348), FRAC_CONST(0.888503), FRAC_CONST(0.888792), FRAC_CONST(0.888865) },
    { FRAC_CONST(0.000975372), FRAC_CONST(0.00388727), FRAC_CONST(0.0153257), FRAC_CONST(0.057971), FRAC_CONST(0.190476), FRAC_CONST(0.444444), FRAC_CONST(0.666667), FRAC_CONST(0.761905), FRAC_CONST(0.790123), FRAC_CONST(0.797508), FRAC_CONST(0.799375), FRAC_CONST(0.799844), FRAC_CONST(0.799961) },
    { FRAC_CONST(0.000487924), FRAC_CONST(0.00194742), FRAC_CONST(0.00772201), FRAC_CONST(0.0298507), FRAC_CONST(0.105263), FRAC_CONST(0.285714), FRAC_CONST(0.5), FRAC_CONST(0.615385), FRAC_CONST(0.653061), FRAC_CONST(0.663212), FRAC_CONST(0.6658), FRAC_CONST(0.66645), FRAC_CONST(0.666612) },
    { FRAC_CONST(0.000244021), FRAC_CONST(0.000974659), FRAC_CONST(0.00387597), FRAC_CONST(0.0151515), FRAC_CONST(0.0555556), FRAC_CONST(0.166667), FRAC_CONST(0.333333), FRAC_CONST(0.444444), FRAC_CONST(0.484848), FRAC_CONST(0.496124), FRAC_CONST(0.499025), FRAC_CONST(0.499756), FRAC_CONST(0.499939) },
    { FRAC_CONST(0.000122026), FRAC_CONST(0.000487567), FRAC_CONST(0.00194175), FRAC_CONST(0.00763359), FRAC_CONST(0.0285714), FRAC_CONST(0.0909091), FRAC_CONST(0.2), FRAC_CONST(0.285714), FRAC_CONST(0.32), FRAC_CONST(0.329897), FRAC_CONST(0.332468), FRAC_CONST(0.333116), FRAC_CONST(0.333279) },
    { FRAC_CONST(6.10165E-005), FRAC_CONST(0.000243843), FRAC_CONST(0.000971817), FRAC_CONST(0.00383142), FRAC_CONST(0.0144928), FRAC_CONST(0.047619), FRAC_CONST(0.111111), FRAC_CONST(0.166667), FRAC_CONST(0.190476), FRAC_CONST(0.197531), FRAC_CONST(0.199377), FRAC_CONST(0.199844), FRAC_CONST(0.199961) },
    { FRAC_CONST(3.05092E-005), FRAC_CONST(0.000121936), FRAC_CONST(0.000486145), FRAC_CONST(0.00191939), FRAC_CONST(0.00729927), FRAC_CONST(0.0243902), FRAC_CONST(0.0588235), FRAC_CONST(0.0909091), FRAC_CONST(0.105263), FRAC_CONST(0.109589), FRAC_CONST(0.110727), FRAC_CONST(0.111015), FRAC_CONST(0.111087) },
    { FRAC_CONST(1.52548E-005), FRAC_CONST(6.09719E-005), FRAC_CONST(0.000243132), FRAC_CONST(0.000960615), FRAC_CONST(0.003663), FRAC_CONST(0.0123457), FRAC_CONST(0.030303), FRAC_CONST(0.047619), FRAC_CONST(0.0555556), FRAC_CONST(0.057971), FRAC_CONST(0.0586081), FRAC_CONST(0.0587695), FRAC_CONST(0.05881) },
    { FRAC_CONST(7.62747E-006), FRAC_CONST(3.04869E-005), FRAC_CONST(0.000121581), FRAC_CONST(0.000480538), FRAC_CONST(0.00183486), FRAC_CONST(0.00621118), FRAC_CONST(0.0153846), FRAC_CONST(0.0243902), FRAC_CONST(0.0285714), FRAC_CONST(0.0298507), FRAC_CONST(0.0301887), FRAC_CONST(0.0302744), FRAC_CONST(0.0302959) },
    { FRAC_CONST(3.81375E-006), FRAC_CONST(1.52437E-005), FRAC_CONST(6.0794E-005), FRAC_CONST(0.000240327), FRAC_CONST(0.000918274), FRAC_CONST(0.00311526), FRAC_CONST(0.00775194), FRAC_CONST(0.0123457), FRAC_CONST(0.0144928), FRAC_CONST(0.0151515), FRAC_CONST(0.0153257), FRAC_CONST(0.0153698), FRAC_CONST(0.0153809) },
    { FRAC_CONST(1.90688E-006), FRAC_CONST(7.62189E-006), FRAC_CONST(3.03979E-005), FRAC_CONST(0.000120178), FRAC_CONST(0.000459348), FRAC_CONST(0.00156006), FRAC_CONST(0.00389105), FRAC_CONST(0.00621118), FRAC_CONST(0.00729927), FRAC_CONST(0.00763359), FRAC_CONST(0.00772201), FRAC_CONST(0.00774443), FRAC_CONST(0.00775006) },
    { FRAC_CONST(9.53441E-007), FRAC_CONST(3.81096E-006), FRAC_CONST(1.51992E-005), FRAC_CONST(6.00925E-005), FRAC_CONST(0.000229727), FRAC_CONST(0.00078064), FRAC_CONST(0.00194932), FRAC_CONST(0.00311526), FRAC_CONST(0.003663), FRAC_CONST(0.00383142), FRAC_CONST(0.00387597), FRAC_CONST(0.00388727), FRAC_CONST(0.0038901) },
    { FRAC_CONST(4.76721E-007), FRAC_CONST(1.90548E-006), FRAC_CONST(7.59965E-006), FRAC_CONST(3.00472E-005), FRAC_CONST(0.000114877), FRAC_CONST(0.000390472), FRAC_CONST(0.00097561), FRAC_CONST(0.00156006), FRAC_CONST(0.00183486), FRAC_CONST(0.00191939), FRAC_CONST(0.00194175), FRAC_CONST(0.00194742), FRAC_CONST(0.00194884) },
    { FRAC_CONST(2.3836E-007), FRAC_CONST(9.52743E-007), FRAC_CONST(3.79984E-006), FRAC_CONST(1.50238E-005), FRAC_CONST(5.74416E-005), FRAC_CONST(0.000195274), FRAC_CONST(0.000488043), FRAC_CONST(0.00078064), FRAC_CONST(0.000918274), FRAC_CONST(0.000960615), FRAC_CONST(0.000971817), FRAC_CONST(0.000974659), FRAC_CONST(0.000975372) },
    { FRAC_CONST(1.1918E-007), FRAC_CONST(4.76372E-007), FRAC_CONST(1.89992E-006), FRAC_CONST(7.51196E-006), FRAC_CONST(2.87216E-005), FRAC_CONST(9.76467E-005), FRAC_CONST(0.000244081), FRAC_CONST(0.000390472), FRAC_CONST(0.000459348), FRAC_CONST(0.000480538), FRAC_CONST(0.000486145), FRAC_CONST(0.000487567), FRAC_CONST(0.000487924) },
    { FRAC_CONST(5.95901E-008), FRAC_CONST(2.38186E-007), FRAC_CONST(9.49963E-007), FRAC_CONST(3.756E-006), FRAC_CONST(1.4361E-005), FRAC_CONST(4.88257E-005), FRAC_CONST(0.000122055), FRAC_CONST(0.000195274), FRAC_CONST(0.000229727), FRAC_CONST(0.000240327), FRAC_CONST(0.000243132), FRAC_CONST(0.000243843), FRAC_CONST(0.000244021) },
    { FRAC_CONST(2.9795E-008), FRAC_CONST(1.19093E-007), FRAC_CONST(4.74982E-007), FRAC_CONST(1.878E-006), FRAC_CONST(7.18056E-006), FRAC_CONST(2.44135E-005), FRAC_CONST(6.10314E-005), FRAC_CONST(9.76467E-005), FRAC_CONST(0.000114877), FRAC_CONST(0.000120178), FRAC_CONST(0.000121581), FRAC_CONST(0.000121936), FRAC_CONST(0.000122026) },
    { FRAC_CONST(1.48975E-008), FRAC_CONST(5.95465E-008), FRAC_CONST(2.37491E-007), FRAC_CONST(9.39002E-007), FRAC_CONST(3.59029E-006), FRAC_CONST(1.22069E-005), FRAC_CONST(3.05166E-005), FRAC_CONST(4.88257E-005), FRAC_CONST(5.74416E-005), FRAC_CONST(6.00925E-005), FRAC_CONST(6.0794E-005), FRAC_CONST(6.09719E-005), FRAC_CONST(6.10165E-005) },
    { FRAC_CONST(7.44876E-009), FRAC_CONST(2.97732E-008), FRAC_CONST(1.18745E-007), FRAC_CONST(4.69501E-007), FRAC_CONST(1.79515E-006), FRAC_CONST(6.10348E-006), FRAC_CONST(1.52586E-005), FRAC_CONST(2.44135E-005), FRAC_CONST(2.87216E-005), FRAC_CONST(3.00472E-005), FRAC_CONST(3.03979E-005), FRAC_CONST(3.04869E-005), FRAC_CONST(3.05092E-005) },
    { FRAC_CONST(3.72438E-009), FRAC_CONST(1.48866E-008), FRAC_CONST(5.93727E-008), FRAC_CONST(2.34751E-007), FRAC_CONST(8.97575E-007), FRAC_CONST(3.05175E-006), FRAC_CONST(7.62934E-006), FRAC_CONST(1.22069E-005), FRAC_CONST(1.4361E-005), FRAC_CONST(1.50238E-005), FRAC_CONST(1.51992E-005), FRAC_CONST(1.52437E-005), FRAC_CONST(1.52548E-005) },
    { FRAC_CONST(1.86219E-009), FRAC_CONST(7.44331E-009), FRAC_CONST(2.96864E-008), FRAC_CONST(1.17375E-007), FRAC_CONST(4.48788E-007), FRAC_CONST(1.52588E-006), FRAC_CONST(3.81468E-006), FRAC_CONST(6.10348E-006), FRAC_CONST(7.18056E-006), FRAC_CONST(7.51196E-006), FRAC_CONST(7.59965E-006), FRAC_CONST(7.62189E-006), FRAC_CONST(7.62747E-006) },
    { FRAC_CONST(9.31095E-010), FRAC_CONST(3.72166E-009), FRAC_CONST(1.48432E-008), FRAC_CONST(5.86876E-008), FRAC_CONST(2.24394E-007), FRAC_CONST(7.62939E-007), FRAC_CONST(1.90734E-006), FRAC_CONST(3.05175E-006), FRAC_CONST(3.59029E-006), FRAC_CONST(3.756E-006), FRAC_CONST(3.79984E-006), FRAC_CONST(3.81096E-006), FRAC_CONST(3.81375E-006) },
    { FRAC_CONST(4.65548E-010), FRAC_CONST(1.86083E-009), FRAC_CONST(7.42159E-009), FRAC_CONST(2.93438E-008), FRAC_CONST(1.12197E-007), FRAC_CONST(3.8147E-007), FRAC_CONST(9.53673E-007), FRAC_CONST(1.52588E-006), FRAC_CONST(1.79515E-006), FRAC_CONST(1.878E-006), FRAC_CONST(1.89992E-006), FRAC_CONST(1.90548E-006), FRAC_CONST(1.90688E-006) },
    { FRAC_CONST(2.32774E-010), FRAC_CONST(9.30414E-010), FRAC_CONST(3.71079E-009), FRAC_CONST(1.46719E-008), FRAC_CONST(5.60985E-008), FRAC_CONST(1.90735E-007), FRAC_CONST(4.76837E-007), FRAC_CONST(7.62939E-007), FRAC_CONST(8.97575E-007), FRAC_CONST(9.39002E-007), FRAC_CONST(9.49963E-007), FRAC_CONST(9.52743E-007), FRAC_CONST(9.53441E-007) },
    { FRAC_CONST(1.16387E-010), FRAC_CONST(4.65207E-010), FRAC_CONST(1.8554E-009), FRAC_CONST(7.33596E-009), FRAC_CONST(2.80492E-008), FRAC_CONST(9.53674E-008), FRAC_CONST(2.38419E-007), FRAC_CONST(3.8147E-007), FRAC_CONST(4.48788E-007), FRAC_CONST(4.69501E-007), FRAC_CONST(4.74982E-007), FRAC_CONST(4.76372E-007), FRAC_CONST(4.76721E-007) },
    { FRAC_CONST(5.81935E-011), FRAC_CONST(2.32603E-010), FRAC_CONST(9.27699E-010), FRAC_CONST(3.66798E-009), FRAC_CONST(1.40246E-008), FRAC_CONST(4.76837E-008), FRAC_CONST(1.19209E-007), FRAC_CONST(1.90735E-007), FRAC_CONST(2.24394E-007), FRAC_CONST(2.34751E-007), FRAC_CONST(2.37491E-007), FRAC_CONST(2.38186E-007), FRAC_CONST(2.3836E-007) },
    { FRAC_CONST(2.90967E-011), FRAC_CONST(1.16302E-010), FRAC_CONST(4.63849E-010), FRAC_CONST(1.83399E-009), FRAC_CONST(7.01231E-009), FRAC_CONST(2.38419E-008), FRAC_CONST(5.96046E-008), FRAC_CONST(9.53674E-008), FRAC_CONST(1.12197E-007), FRAC_CONST(1.17375E-007), FRAC_CONST(1.18745E-007), FRAC_CONST(1.19093E-007), FRAC_CONST(1.1918E-007) }
};

static const real_t Q_div2_tab_right[31][13] = {
    { FRAC_CONST(0.992246), FRAC_CONST(0.992241), FRAC_CONST(0.992218), FRAC_CONST(0.992128), FRAC_CONST(0.991768), FRAC_CONST(0.990329), FRAC_CONST(0.984615), FRAC_CONST(0.962406), FRAC_CONST(0.882759), FRAC_CONST(0.663212), FRAC_CONST(0.332468), FRAC_CONST(0.111015), FRAC_CONST(0.0302959) },
    { FRAC_CONST(0.984612), FRAC_CONST(0.984601), FRAC_CONST(0.984556), FRAC_CONST(0.984379), FRAC_CONST(0.98367), FRAC_CONST(0.980843), FRAC_CONST(0.969697), FRAC_CONST(0.927536), FRAC_CONST(0.790123), FRAC_CONST(0.496124), FRAC_CONST(0.199377), FRAC_CONST(0.0587695), FRAC_CONST(0.0153809) },
    { FRAC_CONST(0.96969), FRAC_CONST(0.969668), FRAC_CONST(0.969582), FRAC_CONST(0.969238), FRAC_CONST(0.967864), FRAC_CONST(0.962406), FRAC_CONST(0.941176), FRAC_CONST(0.864865), FRAC_CONST(0.653061), FRAC_CONST(0.329897), FRAC_CONST(0.110727), FRAC_CONST(0.0302744), FRAC_CONST(0.00775006) },
    { FRAC_CONST(0.941163), FRAC_CONST(0.941122), FRAC_CONST(0.94096), FRAC_CONST(0.940312), FRAC_CONST(0.937729), FRAC_CONST(0.927536), FRAC_CONST(0.888889), FRAC_CONST(0.761905), FRAC_CONST(0.484848), FRAC_CONST(0.197531), FRAC_CONST(0.0586081), FRAC_CONST(0.0153698), FRAC_CONST(0.0038901) },
    { FRAC_CONST(0.888865), FRAC_CONST(0.888792), FRAC_CONST(0.888503), FRAC_CONST(0.887348), FRAC_CONST(0.882759), FRAC_CONST(0.864865), FRAC_CONST(0.8), FRAC_CONST(0.615385), FRAC_CONST(0.32), FRAC_CONST(0.109589), FRAC_CONST(0.0301887), FRAC_CONST(0.00774443), FRAC_CONST(0.00194884) },
    { FRAC_CONST(0.799961), FRAC_CONST(0.799844), FRAC_CONST(0.799375), FRAC_CONST(0.797508), FRAC_CONST(0.790123), FRAC_CONST(0.761905), FRAC_CONST(0.666667), FRAC_CONST(0.444444), FRAC_CONST(0.190476), FRAC_CONST(0.057971), FRAC_CONST(0.0153257), FRAC_CONST(0.00388727), FRAC_CONST(0.000975372) },
    { FRAC_CONST(0.666612), FRAC_CONST(0.66645), FRAC_CONST(0.6658), FRAC_CONST(0.663212), FRAC_CONST(0.653061), FRAC_CONST(0.615385), FRAC_CONST(0.5), FRAC_CONST(0.285714), FRAC_CONST(0.105263), FRAC_CONST(0.0298507), FRAC_CONST(0.00772201), FRAC_CONST(0.00194742), FRAC_CONST(0.000487924) },
    { FRAC_CONST(0.499939), FRAC_CONST(0.499756), FRAC_CONST(0.499025), FRAC_CONST(0.496124), FRAC_CONST(0.484848), FRAC_CONST(0.444444), FRAC_CONST(0.333333), FRAC_CONST(0.166667), FRAC_CONST(0.0555556), FRAC_CONST(0.0151515), FRAC_CONST(0.00387597), FRAC_CONST(0.000974659), FRAC_CONST(0.000244021) },
    { FRAC_CONST(0.333279), FRAC_CONST(0.333116), FRAC_CONST(0.332468), FRAC_CONST(0.329897), FRAC_CONST(0.32), FRAC_CONST(0.285714), FRAC_CONST(0.2), FRAC_CONST(0.0909091), FRAC_CONST(0.0285714), FRAC_CONST(0.00763359), FRAC_CONST(0.00194175), FRAC_CONST(0.000487567), FRAC_CONST(0.000122026) },
    { FRAC_CONST(0.199961), FRAC_CONST(0.199844), FRAC_CONST(0.199377), FRAC_CONST(0.197531), FRAC_CONST(0.190476), FRAC_CONST(0.166667), FRAC_CONST(0.111111), FRAC_CONST(0.047619), FRAC_CONST(0.0144928), FRAC_CONST(0.00383142), FRAC_CONST(0.000971817), FRAC_CONST(0.000243843), FRAC_CONST(6.10165E-005) },
    { FRAC_CONST(0.111087), FRAC_CONST(0.111015), FRAC_CONST(0.110727), FRAC_CONST(0.109589), FRAC_CONST(0.105263), FRAC_CONST(0.0909091), FRAC_CONST(0.0588235), FRAC_CONST(0.0243902), FRAC_CONST(0.00729927), FRAC_CONST(0.00191939), FRAC_CONST(0.000486145), FRAC_CONST(0.000121936), FRAC_CONST(3.05092E-005) },
    { FRAC_CONST(0.05881), FRAC_CONST(0.0587695), FRAC_CONST(0.0586081), FRAC_CONST(0.057971), FRAC_CONST(0.0555556), FRAC_CONST(0.047619), FRAC_CONST(0.030303), FRAC_CONST(0.0123457), FRAC_CONST(0.003663), FRAC_CONST(0.000960615), FRAC_CONST(0.000243132), FRAC_CONST(6.09719E-005), FRAC_CONST(1.52548E-005) },
    { FRAC_CONST(0.0302959), FRAC_CONST(0.0302744), FRAC_CONST(0.0301887), FRAC_CONST(0.0298507), FRAC_CONST(0.0285714), FRAC_CONST(0.0243902), FRAC_CONST(0.0153846), FRAC_CONST(0.00621118), FRAC_CONST(0.00183486), FRAC_CONST(0.000480538), FRAC_CONST(0.000121581), FRAC_CONST(3.04869E-005), FRAC_CONST(7.62747E-006) },
    { FRAC_CONST(0.0153809), FRAC_CONST(0.0153698), FRAC_CONST(0.0153257), FRAC_CONST(0.0151515), FRAC_CONST(0.0144928), FRAC_CONST(0.0123457), FRAC_CONST(0.00775194), FRAC_CONST(0.00311526), FRAC_CONST(0.000918274), FRAC_CONST(0.000240327), FRAC_CONST(6.0794E-005), FRAC_CONST(1.52437E-005), FRAC_CONST(3.81375E-006) },
    { FRAC_CONST(0.00775006), FRAC_CONST(0.00774443), FRAC_CONST(0.00772201), FRAC_CONST(0.00763359), FRAC_CONST(0.00729927), FRAC_CONST(0.00621118), FRAC_CONST(0.00389105), FRAC_CONST(0.00156006), FRAC_CONST(0.000459348), FRAC_CONST(0.000120178), FRAC_CONST(3.03979E-005), FRAC_CONST(7.62189E-006), FRAC_CONST(1.90688E-006) },
    { FRAC_CONST(0.0038901), FRAC_CONST(0.00388727), FRAC_CONST(0.00387597), FRAC_CONST(0.00383142), FRAC_CONST(0.003663), FRAC_CONST(0.00311526), FRAC_CONST(0.00194932), FRAC_CONST(0.00078064), FRAC_CONST(0.000229727), FRAC_CONST(6.00925E-005), FRAC_CONST(1.51992E-005), FRAC_CONST(3.81096E-006), FRAC_CONST(9.53441E-007) },
    { FRAC_CONST(0.00194884), FRAC_CONST(0.00194742), FRAC_CONST(0.00194175), FRAC_CONST(0.00191939), FRAC_CONST(0.00183486), FRAC_CONST(0.00156006), FRAC_CONST(0.00097561), FRAC_CONST(0.000390472), FRAC_CONST(0.000114877), FRAC_CONST(3.00472E-005), FRAC_CONST(7.59965E-006), FRAC_CONST(1.90548E-006), FRAC_CONST(4.76721E-007) },
    { FRAC_CONST(0.000975372), FRAC_CONST(0.000974659), FRAC_CONST(0.000971817), FRAC_CONST(0.000960615), FRAC_CONST(0.000918274), FRAC_CONST(0.00078064), FRAC_CONST(0.000488043), FRAC_CONST(0.000195274), FRAC_CONST(5.74416E-005), FRAC_CONST(1.50238E-005), FRAC_CONST(3.79984E-006), FRAC_CONST(9.52743E-007), FRAC_CONST(2.3836E-007) },
    { FRAC_CONST(0.000487924), FRAC_CONST(0.000487567), FRAC_CONST(0.000486145), FRAC_CONST(0.000480538), FRAC_CONST(0.000459348), FRAC_CONST(0.000390472), FRAC_CONST(0.000244081), FRAC_CONST(9.76467E-005), FRAC_CONST(2.87216E-005), FRAC_CONST(7.51196E-006), FRAC_CONST(1.89992E-006), FRAC_CONST(4.76372E-007), FRAC_CONST(1.1918E-007) },
    { FRAC_CONST(0.000244021), FRAC_CONST(0.000243843), FRAC_CONST(0.000243132), FRAC_CONST(0.000240327), FRAC_CONST(0.000229727), FRAC_CONST(0.000195274), FRAC_CONST(0.000122055), FRAC_CONST(4.88257E-005), FRAC_CONST(1.4361E-005), FRAC_CONST(3.756E-006), FRAC_CONST(9.49963E-007), FRAC_CONST(2.38186E-007), FRAC_CONST(5.95901E-008) },
    { FRAC_CONST(0.000122026), FRAC_CONST(0.000121936), FRAC_CONST(0.000121581), FRAC_CONST(0.000120178), FRAC_CONST(0.000114877), FRAC_CONST(9.76467E-005), FRAC_CONST(6.10314E-005), FRAC_CONST(2.44135E-005), FRAC_CONST(7.18056E-006), FRAC_CONST(1.878E-006), FRAC_CONST(4.74982E-007), FRAC_CONST(1.19093E-007), FRAC_CONST(2.9795E-008) },
    { FRAC_CONST(6.10165E-005), FRAC_CONST(6.09719E-005), FRAC_CONST(6.0794E-005), FRAC_CONST(6.00925E-005), FRAC_CONST(5.74416E-005), FRAC_CONST(4.88257E-005), FRAC_CONST(3.05166E-005), FRAC_CONST(1.22069E-005), FRAC_CONST(3.59029E-006), FRAC_CONST(9.39002E-007), FRAC_CONST(2.37491E-007), FRAC_CONST(5.95465E-008), FRAC_CONST(1.48975E-008) },
    { FRAC_CONST(3.05092E-005), FRAC_CONST(3.04869E-005), FRAC_CONST(3.03979E-005), FRAC_CONST(3.00472E-005), FRAC_CONST(2.87216E-005), FRAC_CONST(2.44135E-005), FRAC_CONST(1.52586E-005), FRAC_CONST(6.10348E-006), FRAC_CONST(1.79515E-006), FRAC_CONST(4.69501E-007), FRAC_CONST(1.18745E-007), FRAC_CONST(2.97732E-008), FRAC_CONST(7.44876E-009) },
    { FRAC_CONST(1.52548E-005), FRAC_CONST(1.52437E-005), FRAC_CONST(1.51992E-005), FRAC_CONST(1.50238E-005), FRAC_CONST(1.4361E-005), FRAC_CONST(1.22069E-005), FRAC_CONST(7.62934E-006), FRAC_CONST(3.05175E-006), FRAC_CONST(8.97575E-007), FRAC_CONST(2.34751E-007), FRAC_CONST(5.93727E-008), FRAC_CONST(1.48866E-008), FRAC_CONST(3.72438E-009) },
    { FRAC_CONST(7.62747E-006), FRAC_CONST(7.62189E-006), FRAC_CONST(7.59965E-006), FRAC_CONST(7.51196E-006), FRAC_CONST(7.18056E-006), FRAC_CONST(6.10348E-006), FRAC_CONST(3.81468E-006), FRAC_CONST(1.52588E-006), FRAC_CONST(4.48788E-007), FRAC_CONST(1.17375E-007), FRAC_CONST(2.96864E-008), FRAC_CONST(7.44331E-009), FRAC_CONST(1.86219E-009) },
    { FRAC_CONST(3.81375E-006), FRAC_CONST(3.81096E-006), FRAC_CONST(3.79984E-006), FRAC_CONST(3.756E-006), FRAC_CONST(3.59029E-006), FRAC_CONST(3.05175E-006), FRAC_CONST(1.90734E-006), FRAC_CONST(7.62939E-007), FRAC_CONST(2.24394E-007), FRAC_CONST(5.86876E-008), FRAC_CONST(1.48432E-008), FRAC_CONST(3.72166E-009), FRAC_CONST(9.31095E-010) },
    { FRAC_CONST(1.90688E-006), FRAC_CONST(1.90548E-006), FRAC_CONST(1.89992E-006), FRAC_CONST(1.878E-006), FRAC_CONST(1.79515E-006), FRAC_CONST(1.52588E-006), FRAC_CONST(9.53673E-007), FRAC_CONST(3.8147E-007), FRAC_CONST(1.12197E-007), FRAC_CONST(2.93438E-008), FRAC_CONST(7.42159E-009), FRAC_CONST(1.86083E-009), FRAC_CONST(4.65548E-010) },
    { FRAC_CONST(9.53441E-007), FRAC_CONST(9.52743E-007), FRAC_CONST(9.49963E-007), FRAC_CONST(9.39002E-007), FRAC_CONST(8.97575E-007), FRAC_CONST(7.62939E-007), FRAC_CONST(4.76837E-007), FRAC_CONST(1.90735E-007), FRAC_CONST(5.60985E-008), FRAC_CONST(1.46719E-008), FRAC_CONST(3.71079E-009), FRAC_CONST(9.30414E-010), FRAC_CONST(2.32774E-010) },
    { FRAC_CONST(4.76721E-007), FRAC_CONST(4.76372E-007), FRAC_CONST(4.74982E-007), FRAC_CONST(4.69501E-007), FRAC_CONST(4.48788E-007), FRAC_CONST(3.8147E-007), FRAC_CONST(2.38419E-007), FRAC_CONST(9.53674E-008), FRAC_CONST(2.80492E-008), FRAC_CONST(7.33596E-009), FRAC_CONST(1.8554E-009), FRAC_CONST(4.65207E-010), FRAC_CONST(1.16387E-010) },
    { FRAC_CONST(2.3836E-007), FRAC_CONST(2.38186E-007), FRAC_CONST(2.37491E-007), FRAC_CONST(2.34751E-007), FRAC_CONST(2.24394E-007), FRAC_CONST(1.90735E-007), FRAC_CONST(1.19209E-007), FRAC_CONST(4.76837E-008), FRAC_CONST(1.40246E-008), FRAC_CONST(3.66798E-009), FRAC_CONST(9.27699E-010), FRAC_CONST(2.32603E-010), FRAC_CONST(5.81935E-011) },
    { FRAC_CONST(1.1918E-007), FRAC_CONST(1.19093E-007), FRAC_CONST(1.18745E-007), FRAC_CONST(1.17375E-007), FRAC_CONST(1.12197E-007), FRAC_CONST(9.53674E-008), FRAC_CONST(5.96046E-008), FRAC_CONST(2.38419E-008), FRAC_CONST(7.01231E-009), FRAC_CONST(1.83399E-009), FRAC_CONST(4.63849E-010), FRAC_CONST(1.16302E-010), FRAC_CONST(2.90967E-011) }
};

/* calculates Q/(1+Q) */
/* [0..1] */
static real_t calc_Q_div2(sbr_info *sbr, uint8_t ch, uint8_t m, uint8_t l)
{
    if (sbr->bs_coupling) {
        if ((sbr->Q[0][m][l] < 0 || sbr->Q[0][m][l] > 30) ||
            (sbr->Q[1][m][l] < 0 || sbr->Q[1][m][l] > 24 /* 2*panOffset(1) */)) {
            return 0;
        } else {
            /* the pan parameter is always even */
            if (ch == 0) {
                return Q_div2_tab_left[sbr->Q[0][m][l]][sbr->Q[1][m][l] >> 1];
            } else {
                return Q_div2_tab_right[sbr->Q[0][m][l]][sbr->Q[1][m][l] >> 1];
            }
        }
    } else {
        /* no coupling */
        if (sbr->Q[ch][m][l] < 0 || sbr->Q[ch][m][l] > 30) {
            return 0;
        } else {
            return Q_div2_tab[sbr->Q[ch][m][l]];
        }
    }
}

static const real_t E_deq_tab[64] = {
    64.0f, 128.0f, 256.0f, 512.0f, 1024.0f, 2048.0f, 4096.0f, 8192.0f,
    16384.0f, 32768.0f, 65536.0f, 131072.0f, 262144.0f, 524288.0f, 1.04858E+006f, 2.09715E+006f,
    4.1943E+006f, 8.38861E+006f, 1.67772E+007f, 3.35544E+007f, 6.71089E+007f, 1.34218E+008f, 2.68435E+008f, 5.36871E+008f,
    1.07374E+009f, 2.14748E+009f, 4.29497E+009f, 8.58993E+009f, 1.71799E+010f, 3.43597E+010f, 6.87195E+010f, 1.37439E+011f,
    2.74878E+011f, 5.49756E+011f, 1.09951E+012f, 2.19902E+012f, 4.39805E+012f, 8.79609E+012f, 1.75922E+013f, 3.51844E+013f,
    7.03687E+013f, 1.40737E+014f, 2.81475E+014f, 5.6295E+014f, 1.1259E+015f, 2.2518E+015f, 4.5036E+015f, 9.0072E+015f,
    1.80144E+016f, 3.60288E+016f, 7.20576E+016f, 1.44115E+017f, 2.8823E+017f, 5.76461E+017f, 1.15292E+018f, 2.30584E+018f,
    4.61169E+018f, 9.22337E+018f, 1.84467E+019f, 3.68935E+019f, 7.3787E+019f, 1.47574E+020f, 2.95148E+020f, 5.90296E+020f
};

void envelope_noise_dequantisation(sbr_info *sbr, uint8_t ch)
{
    if (sbr->bs_coupling == 0) {
        int16_t exp;
        uint8_t l, k;
        uint8_t amp = (sbr->amp_res[ch]) ? 0 : 1;

        for (l = 0; l < sbr->L_E[ch]; l++) {
            for (k = 0; k < sbr->n[sbr->f[ch][l]]; k++) {
                /* +6 for the *64 and -10 for the /32 in the synthesis QMF (fixed)
                 * since this is a energy value: (x/32)^2 = (x^2)/1024
                 */
                /* exp = (sbr->E[ch][k][l] >> amp) + 6; */
                exp = (sbr->E[ch][k][l] >> amp);

                if ((exp < 0) || (exp >= 64)) {
                    sbr->E_orig[ch][k][l] = 0;
                } else {
                    sbr->E_orig[ch][k][l] = E_deq_tab[exp];

                    /* save half the table size at the cost of 1 multiply */
                    if (amp && (sbr->E[ch][k][l] & 1)) {
                        sbr->E_orig[ch][k][l] = MUL_C(sbr->E_orig[ch][k][l], COEF_CONST(1.414213562));
                    }
                }
            }
        }

        for (l = 0; l < sbr->L_Q[ch]; l++) {
            for (k = 0; k < sbr->N_Q; k++) {
                sbr->Q_div[ch][k][l] = calc_Q_div(sbr, ch, k, l);
                sbr->Q_div2[ch][k][l] = calc_Q_div2(sbr, ch, k, l);
            }
        }
    }
}

static const real_t E_pan_tab[25] = {
    FRAC_CONST(0.000244081), FRAC_CONST(0.000488043),
    FRAC_CONST(0.00097561),  FRAC_CONST(0.00194932),
    FRAC_CONST(0.00389105),  FRAC_CONST(0.00775194),
    FRAC_CONST(0.0153846),   FRAC_CONST(0.030303),
    FRAC_CONST(0.0588235),   FRAC_CONST(0.111111),
    FRAC_CONST(0.2),         FRAC_CONST(0.333333),
    FRAC_CONST(0.5),         FRAC_CONST(0.666667),
    FRAC_CONST(0.8),         FRAC_CONST(0.888889),
    FRAC_CONST(0.941176),    FRAC_CONST(0.969697),
    FRAC_CONST(0.984615),    FRAC_CONST(0.992248),
    FRAC_CONST(0.996109),    FRAC_CONST(0.998051),
    FRAC_CONST(0.999024),    FRAC_CONST(0.999512),
    FRAC_CONST(0.999756)
};

void unmap_envelope_noise(sbr_info *sbr)
{
    real_t tmp;
    int16_t exp0, exp1;
    uint8_t l, k;
    uint8_t amp0 = (sbr->amp_res[0]) ? 0 : 1;
    uint8_t amp1 = (sbr->amp_res[1]) ? 0 : 1;

    for (l = 0; l < sbr->L_E[0]; l++) {
        for (k = 0; k < sbr->n[sbr->f[0][l]]; k++) {
            /* +6: * 64 ; +1: * 2 ; */
            exp0 = (sbr->E[0][k][l] >> amp0) + 1;

            /* UN_MAP removed: (x / 4096) same as (x >> 12) */
            /* E[1] is always even so no need for compensating the divide by 2 with
             * an extra multiplication
             */
            /* exp1 = (sbr->E[1][k][l] >> amp1) - 12; */
            exp1 = (sbr->E[1][k][l] >> amp1);

            if ((exp0 < 0) || (exp0 >= 64) ||
                (exp1 < 0) || (exp1 > 24)) {
                sbr->E_orig[1][k][l] = 0;
                sbr->E_orig[0][k][l] = 0;
            } else {
                tmp = E_deq_tab[exp0];
                if (amp0 && (sbr->E[0][k][l] & 1)) {
                    tmp = MUL_C(tmp, COEF_CONST(1.414213562));
                }

                /* panning */
                sbr->E_orig[0][k][l] = MUL_F(tmp, E_pan_tab[exp1]);
                sbr->E_orig[1][k][l] = MUL_F(tmp, E_pan_tab[24 - exp1]);
            }
        }
    }

    for (l = 0; l < sbr->L_Q[0]; l++) {
        for (k = 0; k < sbr->N_Q; k++) {
            sbr->Q_div[0][k][l] = calc_Q_div(sbr, 0, k, l);
            sbr->Q_div[1][k][l] = calc_Q_div(sbr, 1, k, l);
            sbr->Q_div2[0][k][l] = calc_Q_div2(sbr, 0, k, l);
            sbr->Q_div2[1][k][l] = calc_Q_div2(sbr, 1, k, l);
        }
    }
}

#endif

#endif
