/*
 * drivers/amlogic/drm/meson_fbdev.c
 *
 * Copyright (C) 2017 Amlogic, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 */
#include <drm/drm.h>
#include <drm/drmP.h>
#include <drm/drm_crtc_helper.h>
#include <drm/drm_atomic.h>
#include <drm/drm_atomic_helper.h>

#include "meson_drv.h"
#include "meson_gem.h"
#include "meson_fb.h"
#include "meson_fbdev.h"
#include "meson_plane.h"
#include "meson_async_atomic.h"

#define PREFERRED_BPP		32
#define PREFERRED_DEPTH	32
#define MESON_DRM_MAX_CONNECTOR	2

static int am_meson_fbdev_alloc_fb_gem(struct fb_info *info)
{
	struct am_meson_fb *meson_fb;
	struct drm_fb_helper *helper = info->par;
	struct meson_drm_fbdev *fbdev = container_of(helper, struct meson_drm_fbdev, base);
	struct drm_framebuffer *fb = helper->fb;
	struct drm_device *dev = helper->dev;
	size_t size = info->fix.smem_len;
	struct am_meson_gem_object *meson_gem;

	if (!fbdev->fb_gem) {
		meson_gem = am_meson_gem_object_create(dev, 0, size);
		if (IS_ERR(meson_gem)) {
			DRM_ERROR("alloc memory %d fail\n", (u32)size);
			return -ENOMEM;
		}
		fbdev->fb_gem = &meson_gem->base;
		fb = helper->fb;
		meson_fb = container_of(fb, struct am_meson_fb, base);
		if (!meson_fb) {
			DRM_INFO("meson_fb is NULL!\n");
			return -EINVAL;
		}
		meson_fb->bufp[0] = meson_gem;
		DRM_DEBUG("alloc memory %d done\n", (u32)size);
	} else {
		DRM_DEBUG("no need repeate alloc memory %d\n", (u32)size);
	}
	return 0;
}

static void am_meson_fbdev_free_fb_gem(struct fb_info *info)
{
	struct drm_fb_helper *helper = info->par;
	struct meson_drm_fbdev *fbdev = container_of(helper, struct meson_drm_fbdev, base);

	if (fbdev->fb_gem) {
		am_meson_gem_object_free(fbdev->fb_gem);
		fbdev->fb_gem = NULL;
		DRM_DEBUG("free memory done\n");
	} else {
		DRM_DEBUG("memory already free before\n");
	}
}

static void am_meson_fbdev_update_fb(struct fb_info *info,
				     struct fb_var_screeninfo *var)
{
	struct drm_fb_helper *helper = info->par;
	struct drm_framebuffer *fb = helper->fb;
	int bytes_per_pixel, depth;

	fb->width = var->xres_virtual;
	fb->height = var->yres_virtual;
	bytes_per_pixel = DIV_ROUND_UP(var->bits_per_pixel, 8);
	fb->pitches[0] = ALIGN(var->xres_virtual * bytes_per_pixel, 64);
	if (var->bits_per_pixel == 32)
		depth = (var->transp.length > 0) ? 32 : 24;
	else
		depth = var->bits_per_pixel;
	fb->pixel_format = drm_mode_legacy_fb_format(var->bits_per_pixel,
						     depth);
}

static int am_meson_fbdev_open(struct fb_info *info, int arg)
{
	int ret = 0;
	struct drm_fb_helper *helper = info->par;
	struct meson_drm_fbdev *fbdev = container_of(helper, struct meson_drm_fbdev, base);
	struct am_osd_plane *osdplane = container_of(fbdev->plane, struct am_osd_plane, base);

	DRM_DEBUG("%s - %d\n", __func__, osdplane->plane_index);
	ret = am_meson_fbdev_alloc_fb_gem(info);
	return ret;
}

static int am_meson_fbdev_release(struct fb_info *info, int arg)
{
	DRM_DEBUG("may no need to release memory\n");
	return 0;
}

static int am_meson_fbdev_mmap(struct fb_info *info,
			       struct vm_area_struct *vma)
{
	struct drm_fb_helper *helper = info->par;
	struct meson_drm_fbdev *fbdev = container_of(helper, struct meson_drm_fbdev, base);
	struct am_meson_gem_object *meson_gem;

	meson_gem = container_of(fbdev->fb_gem,
				 struct am_meson_gem_object, base);

	return am_meson_gem_object_mmap(meson_gem, vma);
}

static int am_meson_drm_fbdev_ioctl(struct fb_info *info,
				    unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	void __user *argp = (void __user *)arg;
	struct fb_dmabuf_export fbdma;
	struct drm_fb_helper *helper = info->par;
	struct meson_drm *private = helper->dev->dev_private;
	struct meson_drm_fbdev *fbdev = container_of(helper, struct meson_drm_fbdev, base);
	struct drm_plane *plane = fbdev->plane;
	struct ion_client *client = (struct ion_client *)private->gem_client;
	struct am_meson_fb *meson_fb;

	DRM_DEBUG("am_meson_drm_fbdev_ioctl CMD   [%x] - [%d] IN\n", cmd, plane->index);

	if (cmd == FBIOGET_OSD_DMABUF) {
		meson_fb = container_of(helper->fb, struct am_meson_fb, base);
		fbdma.fd = ion_share_dma_buf_fd(client, meson_fb->bufp[0]->handle);
		fbdma.flags = O_CLOEXEC;
		ret = copy_to_user(argp, &fbdma, sizeof(fbdma)) ? -EFAULT : 0;
	} else if (cmd == FBIO_WAITFORVSYNC) {
		if (plane->crtc)
			drm_wait_one_vblank(helper->dev, plane->crtc->index);
		else
			DRM_ERROR("crtc is not set for plane [%d]\n", plane->index);
	}

	DRM_DEBUG("am_meson_drm_fbdev_ioctl CMD   [%x] - [%d] OUT\n", cmd, plane->index);
	return ret;
}

/*sync from pan_display_atomic to adatp to the case of
 *input and output size is different
 */
static int am_meson_pan_display_atomic(struct fb_var_screeninfo *var,
				       struct fb_info *info)
{
	struct drm_fb_helper *fb_helper = info->par;
	struct meson_drm_fbdev *fbdev = container_of(fb_helper, struct meson_drm_fbdev, base);
	struct drm_device *dev = fb_helper->dev;
	struct drm_atomic_state *state;
	struct drm_plane_state *plane_state;
	struct drm_plane *plane = fbdev->plane;
	unsigned int plane_mask;
	struct drm_mode_set *mode_set;
	int hdisplay, vdisplay;
	int ret;

	drm_modeset_lock_all(dev);
	DRM_DEBUG("%s IN [%d]\n", __func__, plane->index);

	state = drm_atomic_state_alloc(dev);
	if (!state) {
		ret = -ENOMEM;
		goto unlock_exit;
	}

	state->acquire_ctx = dev->mode_config.acquire_ctx;
retry:
	plane_mask = 0;
	DRM_DEBUG("am_meson_pan_display_atomic for plane [%d-%p]\n", plane->type, fb_helper->fb);

	mode_set = &fb_helper->crtc_info[0].mode_set;

	/*update plane state, refer to drm_atomic_plane_set_property()*/
	plane_state = drm_atomic_get_plane_state(state, plane);
	if (IS_ERR(plane_state)) {
		ret = PTR_ERR(plane_state);
		goto fail;
	}

	ret = drm_atomic_set_crtc_for_plane(plane_state, mode_set->crtc);
	if (ret != 0) {
		DRM_ERROR("set crtc for plane failed.\n");
		goto fail;
	}

	drm_crtc_get_hv_timing(&mode_set->crtc->mode, &hdisplay, &vdisplay);
	plane_state->crtc_x = 0;
	plane_state->crtc_y = 0;
	plane_state->crtc_w = hdisplay;
	plane_state->crtc_h = vdisplay;

	drm_atomic_set_fb_for_plane(plane_state, fb_helper->fb);
	if (fb_helper->fb) {
		plane_state->src_x = var->xoffset << 16;
		plane_state->src_y = var->yoffset << 16;
		plane_state->src_w = var->xres << 16;
		plane_state->src_h = var->yres << 16;
		plane_state->zpos = fbdev->zorder;
	} else {
		plane_state->src_x = 0;
		plane_state->src_y = 0;
		plane_state->src_w = 0;
		plane_state->src_h = 0;
		plane_state->zpos = fbdev->zorder;
	}

	DRM_DEBUG("update fb [%x-%x, %x-%x]-%d->[%d-%d]",
		plane_state->src_x, plane_state->src_y,
		plane_state->src_w, plane_state->src_h,
		plane_state->zpos, plane_state->crtc_w,
		plane_state->crtc_h);

	plane_mask |= (1 << drm_plane_index(plane));
	plane->old_fb = plane->fb;

	ret = meson_async_atomic_commit(state);
	if (ret == 0) {
		DRM_DEBUG("Async atomic success.\n");
	} else {
		DRM_DEBUG("Async atomic failed, try drm_atomic_commit.\n");
		ret = drm_atomic_commit(state);
	}
	if (ret != 0)
		goto fail;

	info->var.xoffset = var->xoffset;
	info->var.yoffset = var->yoffset;

fail:
	drm_atomic_clean_old_fb(dev, plane_mask, ret);

	if (ret == -EDEADLK)
		goto backoff;

	if (ret != 0)
		drm_atomic_state_free(state);

unlock_exit:
	drm_modeset_unlock_all(dev);
	DRM_DEBUG("%s OUT [%d]\n", __func__, plane->index);

	return ret;

backoff:
	drm_atomic_state_clear(state);
	drm_atomic_legacy_backoff(state);

	goto retry;
}

/*sync from drm_fb_helper_pan_display to adatp to the case of
 *input and output size is different
 */
static int am_meson_drm_fb_pan_display(struct fb_var_screeninfo *var,
				       struct fb_info *info)
{
	struct drm_fb_helper *fb_helper = info->par;
	struct drm_device *dev = fb_helper->dev;
	struct meson_drm_fbdev *fbdev = container_of(fb_helper, struct meson_drm_fbdev, base);
	int ret = 0;

	if (oops_in_progress)
		return -EBUSY;

	if (fbdev->blank) {
		DRM_DEBUG("am_meson_drm_fb_pan_display skip blank.\n");
		return 0;
	}

	if (dev->mode_config.funcs->atomic_commit) {
		ret = am_meson_pan_display_atomic(var, info);
	} else {
		DRM_ERROR("no atomic supported.\n");
	}

	return ret;
}

/**
 * am_meson_drm_fb_helper_check_var - implementation for ->fb_check_var
 * @var: screeninfo to check
 * @info: fbdev registered by the helper
 */
static int am_meson_drm_fb_helper_check_var(struct fb_var_screeninfo *var,
					    struct fb_info *info)
{
	struct drm_fb_helper *fb_helper = info->par;
	struct drm_framebuffer *fb = fb_helper->fb;
	int depth;

	if (var->pixclock != 0 || in_dbg_master()) {
		DRM_ERROR("am_meson_drm_fb_helper_check_var FAILED.\n");
		return -EINVAL;
	}

	/*
	 * Changes struct fb_var_screeninfo are currently not pushed back
	 * to KMS, hence fail if different settings are requested.
	 */
	DRM_DEBUG("fb requested w/h/bpp  %dx%d-%d (virtual %dx%d)\n",
		  var->xres, var->yres, var->bits_per_pixel,
		  var->xres_virtual, var->yres_virtual);
	DRM_DEBUG("current fb w/h/bpp %dx%d-%d\n",
		  fb->width, fb->height, fb->bits_per_pixel);
	if (var->bits_per_pixel != fb->bits_per_pixel ||
	    var->xres_virtual != fb->width ||
	    var->yres_virtual != fb->height)
		DRM_DEBUG("%s need realloc buffer\n", __func__);

	switch (var->bits_per_pixel) {
	case 16:
		depth = (var->green.length == 6) ? 16 : 15;
		break;
	case 32:
		depth = (var->transp.length > 0) ? 32 : 24;
		break;
	default:
		depth = var->bits_per_pixel;
		break;
	}

	switch (depth) {
	case 8:
		var->red.offset = 0;
		var->green.offset = 0;
		var->blue.offset = 0;
		var->red.length = 8;
		var->green.length = 8;
		var->blue.length = 8;
		var->transp.length = 0;
		var->transp.offset = 0;
		break;
	case 15:
		var->red.offset = 10;
		var->green.offset = 5;
		var->blue.offset = 0;
		var->red.length = 5;
		var->green.length = 5;
		var->blue.length = 5;
		var->transp.length = 1;
		var->transp.offset = 15;
		break;
	case 16:
		var->red.offset = 11;
		var->green.offset = 5;
		var->blue.offset = 0;
		var->red.length = 5;
		var->green.length = 6;
		var->blue.length = 5;
		var->transp.length = 0;
		var->transp.offset = 0;
		break;
	case 24:
		var->red.offset = 16;
		var->green.offset = 8;
		var->blue.offset = 0;
		var->red.length = 8;
		var->green.length = 8;
		var->blue.length = 8;
		var->transp.length = 0;
		var->transp.offset = 0;
		break;
	case 32:
		var->red.offset = 16;
		var->green.offset = 8;
		var->blue.offset = 0;
		var->red.length = 8;
		var->green.length = 8;
		var->blue.length = 8;
		var->transp.length = 8;
		var->transp.offset = 24;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

/*
 * update fbinfo according to current fb.
 */
static int am_meson_drm_fb_helper_update_fbinfo(struct fb_info *info)
{
	struct drm_fb_helper *helper = info->par;
	struct drm_framebuffer *fb = helper->fb;
	size_t size = fb->pitches[0] * fb->height;

	info->screen_size = size;
	info->fix.smem_len = size;
	drm_fb_helper_fill_fix(info, fb->pitches[0], fb->depth);

	DRM_DEBUG("FB update [%dx%d]-%d size=%zu\n",
		fb->width, fb->height, fb->depth, size);
	return 0;
}

/**
 * drm_fb_helper_set_par - implementation for ->fb_set_par
 * @info: fbdev registered by the helper
 *
 * This will let fbcon do the mode init and is called at initialization time by
 * the fbdev core when registering the driver, and later on through the hotplug
 * callback.
 */
int am_meson_drm_fb_helper_set_par(struct fb_info *info)
{
	struct drm_fb_helper *fb_helper = info->par;
	struct fb_var_screeninfo *var = &info->var;
	struct drm_framebuffer *fb = fb_helper->fb;
	struct meson_drm_fbdev *fbdev = container_of(fb_helper, struct meson_drm_fbdev, base);
	struct drm_gem_object *fb_gem = fbdev->fb_gem;

	if (oops_in_progress)
		return -EBUSY;

	if (var->pixclock != 0) {
		DRM_ERROR("PIXEL CLOCK SET\n");
		return -EINVAL;
	}

	DRM_INFO("am_meson_drm_fb_helper_set_par  IN\n");
	if (var->bits_per_pixel != fb->bits_per_pixel ||
	    var->xres_virtual != fb->width ||
	    var->yres_virtual != fb->height) {
		/*realloc framebuffer, free old then alloc new gem*/
		am_meson_fbdev_update_fb(info, var);
		am_meson_drm_fb_helper_update_fbinfo(info);

		if (fb_gem && fb_gem->size < info->fix.smem_len) {
			DRM_DEBUG("GEM SIZE is not enough, no re-allocate.\n");
			am_meson_fbdev_free_fb_gem(info);
			fb_gem = NULL;
		}

		if (!fb_gem) {
			if (am_meson_fbdev_alloc_fb_gem(info)) {
				DRM_ERROR("%s realloc fb fail\n", __func__);
				return -ENOMEM;
			}
			DRM_DEBUG("am_meson_drm_fb_helper_set_par reallocate success.\n");
		}
	}

	DRM_INFO("am_meson_drm_fb_helper_set_par  OUT\n");

	return 0;
}

/**
 * the implment if different from drm_fb_helper.
 * for plane based fbdev, we only disable corresponding plane
 * but not the whole crtc.
 */
int am_meson_drm_fb_blank(int blank, struct fb_info *info)
{
	struct drm_fb_helper *helper = info->par;
	struct meson_drm_fbdev *fbdev = container_of(helper, struct meson_drm_fbdev, base);
	struct drm_device *dev = helper->dev;
	int ret = 0;

	if (blank == 0) {
		DRM_DEBUG("meson_fbdev[%s] goto UNBLANK.\n", fbdev->plane->name);
		fbdev->blank = false;
		ret = am_meson_drm_fb_pan_display(&info->var, info);
	} else {
		DRM_DEBUG("meson_fbdev[%s] goto blank.\n", fbdev->plane->name);
		drm_modeset_lock_all(dev);
		drm_plane_force_disable(fbdev->plane);
		drm_modeset_unlock_all(dev);
		fbdev->blank = true;
	}

	return ret;
}

static struct fb_ops meson_drm_fbdev_ops = {
	.owner		= THIS_MODULE,
	.fb_open        = am_meson_fbdev_open,
	.fb_release     = am_meson_fbdev_release,
	.fb_mmap	= am_meson_fbdev_mmap,
	.fb_fillrect	= drm_fb_helper_cfb_fillrect,
	.fb_copyarea	= drm_fb_helper_cfb_copyarea,
	.fb_imageblit	= drm_fb_helper_cfb_imageblit,
	.fb_check_var	= am_meson_drm_fb_helper_check_var,
	.fb_set_par	= am_meson_drm_fb_helper_set_par,
	.fb_blank	= am_meson_drm_fb_blank,
	.fb_pan_display	= am_meson_drm_fb_pan_display,
	.fb_setcmap	= drm_fb_helper_setcmap,
	.fb_ioctl       = am_meson_drm_fbdev_ioctl,
#ifdef CONFIG_COMPAT
	.fb_compat_ioctl = am_meson_drm_fbdev_ioctl,
#endif
};

static int am_meson_drm_fbdev_probe(struct drm_fb_helper *helper,
				     struct drm_fb_helper_surface_size *sizesxx)
{
	struct drm_device *dev = helper->dev;
	struct meson_drm *private = dev->dev_private;
	struct meson_drm_fbdev *fbdev = container_of(helper, struct meson_drm_fbdev, base);
	struct drm_mode_fb_cmd2 mode_cmd = { 0 };
	struct drm_framebuffer *fb;
	struct fb_info *fbi;
	uint32_t bpp;
	unsigned int bytes_per_pixel;
	u32 ui_w, ui_h;
	int ret;

	if (private->ui_config.fb_w  &&
		private->ui_config.fb_h &&
		private->ui_config.fb_bpp &&
		private->ui_config.ui_w &&
		private->ui_config.ui_h) {
		mode_cmd.width = private->ui_config.fb_w;
		mode_cmd.height = private->ui_config.fb_h;
		bpp = private->ui_config.fb_bpp;
		ui_w = private->ui_config.ui_w;
		ui_h = private->ui_config.ui_h;
	} else {
		DRM_ERROR("no fbdev size specified in dts.\n");
		mode_cmd.width = ui_w =  logo.width;
		mode_cmd.height = ui_h = logo.height;
		bpp = PREFERRED_BPP;
	}

	mode_cmd.pixel_format = drm_mode_legacy_fb_format(bpp, PREFERRED_DEPTH);
	bytes_per_pixel = DIV_ROUND_UP(bpp, 8);
	mode_cmd.pitches[0] = ALIGN(mode_cmd.width * bytes_per_pixel, 64);

	DRM_INFO("mode_cmd.width = %d\n", mode_cmd.width);
	DRM_INFO("mode_cmd.height = %d\n", mode_cmd.height);
	DRM_INFO("mode_cmd.pixel_format = %d-%d\n", mode_cmd.pixel_format, DRM_FORMAT_ARGB8888);

	fbi = drm_fb_helper_alloc_fbi(helper);
	if (IS_ERR(fbi)) {
		dev_err(dev->dev, "Failed to create framebuffer info.\n");
		ret = PTR_ERR(fbi);
		return ret;
	}

	helper->fb = am_meson_drm_framebuffer_init(dev, &mode_cmd,
						   fbdev->fb_gem);
	if (IS_ERR(helper->fb)) {
		dev_err(dev->dev, "Failed to allocate DRM framebuffer.\n");
		ret = PTR_ERR(helper->fb);
		goto err_release_fbi;
	}
	fb = helper->fb;
	fb->bits_per_pixel = bpp;
	fb->depth = PREFERRED_DEPTH;

	fbi->par = helper;
	fbi->flags = FBINFO_FLAG_DEFAULT;
	fbi->fbops = &meson_drm_fbdev_ops;
	fbi->skip_vt_switch = true;
	am_meson_drm_fb_helper_update_fbinfo(fbi);
	drm_fb_helper_fill_var(fbi, helper, ui_w, ui_h);
	return 0;

err_release_fbi:
	drm_fb_helper_release_fbi(helper);
	return ret;
}

static const struct drm_fb_helper_funcs meson_drm_fb_helper_funcs = {
	.fb_probe = am_meson_drm_fbdev_probe,
};

static void am_meson_fbdev_parse_config(struct drm_device *dev)
{
	int ret;
	struct meson_drm *private = dev->dev_private;

	ret = of_property_read_u32(dev->dev->of_node,
				   "fbdev_ui_w", &private->ui_config.ui_w);
	if (ret) {
		DRM_ERROR("don't find  match fbdev_ui_w\n");
		private->ui_config.ui_w = 0;
	}
	ret = of_property_read_u32(dev->dev->of_node,
				   "fbdev_ui_h", &private->ui_config.ui_h);
	if (ret) {
		DRM_ERROR("don't find  match fbdev_ui_h\n");
		private->ui_config.ui_h = 0;
	}
	ret = of_property_read_u32(dev->dev->of_node,
				   "fbdev_fb_w", &private->ui_config.fb_w);
	if (ret) {
		DRM_ERROR("don't find  match fbdev_fb_w\n");
		private->ui_config.fb_w = 0;
	}
	ret = of_property_read_u32(dev->dev->of_node,
				   "fbdev_fb_h", &private->ui_config.fb_h);
	if (ret) {
		DRM_ERROR("don't find  match fbdev_fb_h\n");
		private->ui_config.fb_h = 0;
	}
	ret = of_property_read_u32(dev->dev->of_node,
				   "fbdev_fb_bpp",
				   &private->ui_config.fb_bpp);
	if (ret) {
		DRM_ERROR("don't find  match fbdev_fb_bpp\n");
		private->ui_config.fb_bpp = 0;
	}
}

struct meson_drm_fbdev *am_meson_create_drm_fbdev(struct drm_device *dev)
{
	struct meson_drm *drmdev = dev->dev_private;
	struct meson_drm_fbdev *fbdev;
	struct drm_fb_helper *helper;
	int ret, bpp;

	if (drmdev->ui_config.fb_bpp)
		bpp = drmdev->ui_config.fb_bpp;
	else
		bpp = PREFERRED_BPP;

	fbdev = devm_kzalloc(dev->dev, sizeof(struct meson_drm_fbdev), GFP_KERNEL);
	if (!fbdev)
		return NULL;

	helper = &fbdev->base;

	drm_fb_helper_prepare(dev, helper, &meson_drm_fb_helper_funcs);

	ret = drm_fb_helper_init(dev, helper, 1,
				 MESON_DRM_MAX_CONNECTOR);
	if (ret < 0) {
		dev_err(dev->dev, "Failed to initialize drm fb helper - %d.\n",
			ret);
		goto err_free;
	}

	ret = drm_fb_helper_single_add_all_connectors(helper);
	if (ret < 0) {
		dev_err(dev->dev, "Failed to add connectors - %d.\n", ret);
		goto err_drm_fb_helper_fini;
	}

	ret = drm_fb_helper_initial_config(helper, bpp);
	if (ret < 0) {
		dev_err(dev->dev, "Failed to set initial hw config - %d.\n",
			ret);
		goto err_drm_fb_helper_fini;
	}

	fbdev->blank = false;

	DRM_INFO("create fbdev success.\n");
	return fbdev;

err_drm_fb_helper_fini:
	drm_fb_helper_fini(helper);
err_free:
	kfree(fbdev);
	fbdev = NULL;
	DRM_INFO("create drm fbdev failed[%d]\n", ret);
	return NULL;
}

int am_meson_drm_fbdev_init(struct drm_device *dev)
{
	struct meson_drm *drmdev = dev->dev_private;
	struct meson_drm_fbdev *fbdev;
	struct am_osd_plane *osd_plane;
	int i, fbdev_cnt = 0;

	DRM_INFO("%s in\n", __func__);
	am_meson_fbdev_parse_config(dev);

	if (!dev->mode_config.num_crtc || !dev->mode_config.num_connector)
		return -EINVAL;

	if (drmdev->primary_plane) {
		fbdev = am_meson_create_drm_fbdev(dev);
		fbdev->plane = drmdev->primary_plane;
		fbdev->zorder = OSD_PLANE_BEGIN_ZORDER;
		DRM_INFO("create fbdev for primary plane [%p]\n", fbdev);
	}

	/*only create fbdev for viu1*/
	for (i = 0; i < MESON_MAX_OSD; i++) {
		osd_plane = drmdev->osd_planes[i];
		if (!osd_plane)
			break;

		if (osd_plane->base.type == DRM_PLANE_TYPE_PRIMARY)
			continue;

		fbdev = am_meson_create_drm_fbdev(dev);
		fbdev->plane = &osd_plane->base;
		fbdev->zorder = OSD_PLANE_BEGIN_ZORDER + fbdev_cnt;
		fbdev_cnt++;
		DRM_INFO("create fbdev for plane [%d]-[%p]\n", osd_plane->plane_index, fbdev);
	}

	DRM_INFO("%s create %d out\n", __func__, fbdev_cnt);
	return 0;
}

void am_meson_drm_fbdev_fini(struct drm_device *dev)
{
	struct meson_drm *private = dev->dev_private;
	struct meson_drm_fbdev *fbdev;
	struct drm_fb_helper *helper;
	int i;

	for (i = 0; i < MESON_MAX_OSD; i++) {
		fbdev = private->osd_fbdevs[i];
		if (fbdev) {
			helper = &fbdev->base;
			drm_fb_helper_unregister_fbi(helper);
			drm_fb_helper_release_fbi(helper);
			if (helper->fb)
				drm_framebuffer_unreference(helper->fb);
			fbdev->fb_gem = NULL;
			drm_fb_helper_fini(helper);
			kfree(fbdev);
		}
	}
}

