/*
 * include/linux/amlogic/media/vcapture/vcapture.h
 *
 * Copyright (C) 2022 Amlogic, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 */

#ifndef VCAPTURE_H_
#define VCAPTURE_H_

#include <linux/mutex.h>
#include <linux/completion.h>
#include <media/v4l2-device.h>
#include <linux/amlogic/media/ge2d/ge2d.h>
#include <linux/amlogic/media/vfm/vframe_provider.h>
#include <linux/amlogic/media/vfm/vframe_receiver.h>
#include <linux/amlogic/media/v4l_util/videobuf-res.h>

#define RECEIVER_NAME_SIZE 32
#define PROVIDER_NAME_SIZE 32
#define MAX_VBUF 4
#define CMA_ALLOC_SIZE 16
#define CAPTURE_LINES 16
#define THUMNAIL_MAX_SIZE 65536
#define THUMNAIL_MIN_WIDTH 128
#define THUMNAIL_MIN_HEIGHT 96
#define THUMNAIL_DEFAULT_WIDTH 160
#define THUMNAIL_DEFAULT_HEIGHT 192

#define MAX_ZONE 256
#define HIST_BIN 8

#define USE_TEE_API

enum vcapture_vf_status {
	VCAPTURE_INIT = 0,
	VCAPTURE_PREPARE,
	VCAPTURE_DROP_VF,
};

enum vframe_source_type {
	DECODER_8BIT_NORMAL = 0,
	DECODER_8BIT_BOTTOM,
	DECODER_8BIT_TOP,
	DECODER_10BIT_NORMAL,
	DECODER_10BIT_BOTTOM,
	DECODER_10BIT_TOP,
	VDIN_8BIT_NORMAL,
	VDIN_10BIT_NORMAL,
};

struct vcapture_fmt {
	char *name;
	u32 fourcc; /* v4l2 format id */
	u8 depth;
};

struct vcapture_output {
	void *vbuf;
	int width;
	int height;
	u32 v4l2_format;

	void *thumbnail_vbuf;
	int thumbnail_width;
	int thumbnail_height;
	u32 thumbnail_v4l2_format;
};

struct vcapture_buffer {
	/* common v4l buffer stuff -- must be first */
	struct videobuf_buffer vb;
	struct vcapture_fmt *fmt;
};

struct vcapture_dmaqueue {
	struct list_head active;

	/* thread for generating video stream*/
	struct task_struct *kthread;
	wait_queue_head_t wq;
	unsigned char task_running;
#ifdef USE_SEMA_QBUF
	struct completion qbuf_comp;
#endif
};

struct vcapture_dev {
	struct v4l2_device v4l2_dev;
	struct videobuf_res_privdata res;
	struct video_device *vdev;
	bool inited;
	atomic_t capture_frame_per_n;
	atomic_t capture_lines_per_n;
	int frame_counter;
	int lines_counter;
	int fd_num;
	int src_canvas[3];
	int dst_canvas[2];
	struct timeval begin_time;
	atomic_t is_playing;
	atomic_t is_capturing;
	bool set_format_flag;
	struct vframe_s *new_vf;
	struct vframe_s *ge2d_vf;
	atomic_t vcap_status;
	unsigned int f_flags;
	struct semaphore thread_sem;
	resource_size_t buffer_start;
	unsigned int buffer_size;
	unsigned int each_buffer_size;
	struct vcapture_fmt *fmt;
	unsigned int width, height;
	unsigned int dst_width, dst_height;
	spinlock_t slock;
	struct mutex vcapture_mutex /*for v4l2 */;
	struct mutex vf_mutex /*for vframe */;
	enum v4l2_buf_type type;
	struct videobuf_queue vbuf_queue;
	struct vcapture_dmaqueue dma_q;
	struct ge2d_context_s *context;
	struct vframe_receiver_s recv;
	struct vframe_provider_s prov;
	char recv_name[RECEIVER_NAME_SIZE];
	char prov_name[PROVIDER_NAME_SIZE];
	int vbuf_count;
	int cur_vbuf_index;
	void *vbuf_addr[MAX_VBUF];
	unsigned int vbuf_fmt[MAX_VBUF];
	struct vm_area_struct *vma[MAX_VBUF];
	unsigned int vbuf_field[MAX_VBUF];

	atomic_t is_capturing_thumbnail;
	int thumbnail_canvas[2];
	struct vframe_s *vf_thumbnail[MAX_VBUF];
	unsigned int thumbnail_width, thumbnail_height;
	void *thumbnail_addr[MAX_VBUF];
	int thumbnail_index;
};

/* format */
static struct vcapture_fmt formats[] = {
	{
		.name = "NV12 (12)",
		.fourcc = V4L2_PIX_FMT_NV12, /* rrrrrggg gggbbbbb */
		.depth = 12,
	},
	{
		.name = "NV21 (12)",
		.fourcc = V4L2_PIX_FMT_NV21, /* rrrrrggg gggbbbbb */
		.depth = 12,
	},
	{
		.name = "RGB565 (BE16)",
		.fourcc = V4L2_PIX_FMT_RGB565X, /* rrrrrggg gggbbbbb */
		.depth = 16,
	},
	{
		.name = "RGB888 (24)",
		.fourcc = V4L2_PIX_FMT_RGB24, /* 24  RGB-8-8-8 */
		.depth = 24,
	},
	{
		.name = "BGR888 (24)",
		.fourcc = V4L2_PIX_FMT_BGR24, /* 24  BGR-8-8-8 */
		.depth = 24,
	},
	{
		.name = "YUV888 (24)",
		.fourcc = V4L2_PIX_FMT_YUV444, /* 24  YUV-8-8-8 */
		.depth = 24,
	},
	{
		.name = "ARGB888 (32)",
		.fourcc = V4L2_PIX_FMT_ARGB32, /* 32  ARGB-8-8-8 */
		.depth = 32,
	},
	{
		.name = "ABGR888 (32)",
		.fourcc = V4L2_PIX_FMT_ABGR32, /* 32  ABGR-8-8-8 */
		.depth = 32,
	},
	{
		.name = "Y8 (8)",
		.fourcc = V4L2_PIX_FMT_GREY, /* 8  Y only */
		.depth = 8,
	},
};

static struct vcapture_fmt *get_format(struct v4l2_format *f)
{
	struct vcapture_fmt *fmt;
	unsigned int k;

	for (k = 0; k < ARRAY_SIZE(formats); k++) {
		fmt = &formats[k];
		if (fmt->fourcc == f->fmt.pix.pixelformat)
			break;
	}
	if (k == ARRAY_SIZE(formats))
		fmt = NULL;
	return fmt;
}

static int get_source_type(struct vframe_s *vf)
{
	enum vframe_source_type ret;
	int interlace_mode;

	interlace_mode = vf->type & VIDTYPE_TYPEMASK;
	if ((vf->source_type == VFRAME_SOURCE_TYPE_HDMI) ||
	    (vf->source_type == VFRAME_SOURCE_TYPE_CVBS)) {
		if ((vf->bitdepth & BITDEPTH_Y10) &&
		    (!(vf->type & VIDTYPE_COMPRESS)) &&
		    (get_cpu_type() >= MESON_CPU_MAJOR_ID_TXL))
			ret = VDIN_10BIT_NORMAL;
		else
			ret = VDIN_8BIT_NORMAL;
	} else {
		if ((vf->bitdepth & BITDEPTH_Y10) &&
		    (!(vf->type & VIDTYPE_COMPRESS)) &&
		    (get_cpu_type() >= MESON_CPU_MAJOR_ID_TXL)) {
			if (interlace_mode == VIDTYPE_INTERLACE_TOP)
				ret = DECODER_10BIT_TOP;
			else if (interlace_mode == VIDTYPE_INTERLACE_BOTTOM)
				ret = DECODER_10BIT_BOTTOM;
			else
				ret = DECODER_10BIT_NORMAL;
		} else {
			if (interlace_mode == VIDTYPE_INTERLACE_TOP)
				ret = DECODER_8BIT_TOP;
			else if (interlace_mode == VIDTYPE_INTERLACE_BOTTOM)
				ret = DECODER_8BIT_BOTTOM;
			else
				ret = DECODER_8BIT_NORMAL;
		}
	}
	return ret;
}

static int get_input_format(struct vframe_s *vf)
{
	int format = GE2D_FORMAT_M24_YUV420;
	enum vframe_source_type soure_type;

	soure_type = get_source_type(vf);
	switch (soure_type) {
	case DECODER_8BIT_NORMAL:
		if (vf->type & VIDTYPE_VIU_422)
			format = GE2D_FORMAT_S16_YUV422;
		else if (vf->type & VIDTYPE_VIU_NV21)
			format = GE2D_FORMAT_M24_NV21
				| GE2D_LITTLE_ENDIAN;
		else if (vf->type & VIDTYPE_VIU_NV12)
			format = GE2D_FORMAT_M24_NV12
				| GE2D_LITTLE_ENDIAN;
		else if (vf->type & VIDTYPE_VIU_444)
			format = GE2D_FORMAT_S24_YUV444;
		else if (vf->type & VIDTYPE_COMPRESS)
			format = GE2D_FORMAT_M24_NV12
				| GE2D_LITTLE_ENDIAN;
		else
			format = GE2D_FORMAT_M24_YUV420;
		break;
	case DECODER_8BIT_BOTTOM:
		if (vf->type & VIDTYPE_VIU_422)
			format = GE2D_FORMAT_S16_YUV422
				| (GE2D_FORMAT_S16_YUV422B & (3 << 3));
		else if (vf->type & VIDTYPE_VIU_NV21)
			format = GE2D_FORMAT_M24_NV21
				| (GE2D_FORMAT_M24_NV21B & (3 << 3))
				| GE2D_LITTLE_ENDIAN;
		else if (vf->type & VIDTYPE_VIU_NV12)
			format = GE2D_FORMAT_M24_NV12
				| (GE2D_FORMAT_M24_NV12B & (3 << 3))
				| GE2D_LITTLE_ENDIAN;
		else if (vf->type & VIDTYPE_VIU_444)
			format = GE2D_FORMAT_S24_YUV444
				| (GE2D_FORMAT_S24_YUV444B & (3 << 3));
		else
			format = GE2D_FORMAT_M24_YUV420
				| (GE2D_FMT_M24_YUV420B & (3 << 3));
		break;
	case DECODER_8BIT_TOP:
		if (vf->type & VIDTYPE_VIU_422)
			format = GE2D_FORMAT_S16_YUV422
				| (GE2D_FORMAT_S16_YUV422T & (3 << 3));
		else if (vf->type & VIDTYPE_VIU_NV21)
			format = GE2D_FORMAT_M24_NV21
				| (GE2D_FORMAT_M24_NV21T & (3 << 3))
				| GE2D_LITTLE_ENDIAN;
		else if (vf->type & VIDTYPE_VIU_NV12)
			format = GE2D_FORMAT_M24_NV12
				| (GE2D_FORMAT_M24_NV12T & (3 << 3))
				| GE2D_LITTLE_ENDIAN;
		else if (vf->type & VIDTYPE_VIU_444)
			format = GE2D_FORMAT_S24_YUV444
				| (GE2D_FORMAT_S24_YUV444T & (3 << 3));
		else
			format = GE2D_FORMAT_M24_YUV420
				| (GE2D_FMT_M24_YUV420T & (3 << 3));
		break;
	case DECODER_10BIT_NORMAL:
		if (vf->type & VIDTYPE_VIU_422) {
			if (vf->bitdepth & FULL_PACK_422_MODE)
				format = GE2D_FORMAT_S16_10BIT_YUV422;
			else
				format = GE2D_FORMAT_S16_12BIT_YUV422;
		}
		break;
	case DECODER_10BIT_BOTTOM:
		if (vf->type & VIDTYPE_VIU_422) {
			if (vf->bitdepth & FULL_PACK_422_MODE)
				format = GE2D_FORMAT_S16_10BIT_YUV422
					| (GE2D_FORMAT_S16_10BIT_YUV422B
					& (3 << 3));
			else
				format = GE2D_FORMAT_S16_12BIT_YUV422
					| (GE2D_FORMAT_S16_12BIT_YUV422B
					& (3 << 3));
		}
		break;
	case DECODER_10BIT_TOP:
		if (vf->type & VIDTYPE_VIU_422) {
			if (vf->bitdepth & FULL_PACK_422_MODE)
				format = GE2D_FORMAT_S16_10BIT_YUV422
					| (GE2D_FORMAT_S16_10BIT_YUV422T
					& (3 << 3));
			else
				format = GE2D_FORMAT_S16_12BIT_YUV422
					| (GE2D_FORMAT_S16_12BIT_YUV422T
					& (3 << 3));
		}
		break;
	case VDIN_8BIT_NORMAL:
		if (vf->type & VIDTYPE_VIU_422)
			format = GE2D_FORMAT_S16_YUV422;
		else if (vf->type & VIDTYPE_VIU_NV21)
			format = GE2D_FORMAT_M24_NV21;
		else if (vf->type & VIDTYPE_VIU_444)
			format = GE2D_FORMAT_S24_YUV444;
		else
			format = GE2D_FORMAT_M24_YUV420;
		break;
	case VDIN_10BIT_NORMAL:
		if (vf->type & VIDTYPE_VIU_422) {
			if (vf->bitdepth & FULL_PACK_422_MODE)
				format = GE2D_FORMAT_S16_10BIT_YUV422;
			else
				format = GE2D_FORMAT_S16_12BIT_YUV422;
		}
		break;
	default:
		format = GE2D_FORMAT_M24_YUV420;
	}
	return format;
}

static int get_output_format(int v4l2_format)
{
	int format = GE2D_FORMAT_S24_YUV444;

	switch (v4l2_format) {
	case V4L2_PIX_FMT_NV12:
		format = GE2D_FORMAT_M24_NV12;
		break;
	case V4L2_PIX_FMT_NV21:
		format = GE2D_FORMAT_M24_NV21;
		break;
	case V4L2_PIX_FMT_RGB565X:
		format = GE2D_FORMAT_S16_RGB_565;
		break;
	case V4L2_PIX_FMT_YUV444:
		format = GE2D_FORMAT_S24_YUV444;
		break;
	case V4L2_PIX_FMT_BGR24:
		format = GE2D_FORMAT_S24_RGB;
		break;
	case V4L2_PIX_FMT_RGB24:
		format = GE2D_FORMAT_S24_BGR;
		break;
	case V4L2_PIX_FMT_ARGB32:
		format = GE2D_FORMAT_S32_ARGB;
		break;
	case V4L2_PIX_FMT_ABGR32:
		format = GE2D_FORMAT_S32_ABGR;
		break;
	case V4L2_PIX_FMT_GREY:
		format = GE2D_FORMAT_S8_Y;
		break;
	default:
		break;
	}
	return format;
}

/* canvas */
static int alloc_canvas(struct vcapture_dev *dev)
{
	const char *keep_owner = "vcapture";

	if (dev->src_canvas[0] < 0)
		dev->src_canvas[0] =
			canvas_pool_map_alloc_canvas(keep_owner);
	if (dev->src_canvas[1] < 0)
		dev->src_canvas[1] =
			canvas_pool_map_alloc_canvas(keep_owner);
	if (dev->src_canvas[2] < 0)
		dev->src_canvas[2] =
			canvas_pool_map_alloc_canvas(keep_owner);
	if (dev->dst_canvas[0] < 0)
		dev->dst_canvas[0] =
			canvas_pool_map_alloc_canvas(keep_owner);
	if (dev->dst_canvas[1] < 0)
		dev->dst_canvas[1] =
			canvas_pool_map_alloc_canvas(keep_owner);
	if (dev->thumbnail_canvas[0] < 0)
		dev->thumbnail_canvas[0] =
			canvas_pool_map_alloc_canvas(keep_owner);
	if (dev->thumbnail_canvas[1] < 0)
		dev->thumbnail_canvas[1] =
			canvas_pool_map_alloc_canvas(keep_owner);
	if (dev->src_canvas[0] < 0 ||
	    dev->src_canvas[1] < 0 ||
	    dev->src_canvas[2] < 0 ||
	    dev->dst_canvas[0] < 0 ||
	    dev->dst_canvas[1] < 0 ||
		dev->thumbnail_canvas[0] < 0 ||
		dev->thumbnail_canvas[1] < 0) {
		return -1;
	}
	return 0;
}

static int free_canvas(struct vcapture_dev *dev)
{
	if (dev->src_canvas[0] >= 0) {
		canvas_pool_map_free_canvas(dev->src_canvas[0]);
		dev->src_canvas[0] = -1;
	}
	if (dev->src_canvas[1] >= 0) {
		canvas_pool_map_free_canvas(dev->src_canvas[1]);
		dev->src_canvas[1] = -1;
	}
	if (dev->src_canvas[2] >= 0) {
		canvas_pool_map_free_canvas(dev->src_canvas[2]);
		dev->src_canvas[2] = -1;
	}
	if (dev->dst_canvas[0] >= 0) {
		canvas_pool_map_free_canvas(dev->dst_canvas[0]);
		dev->dst_canvas[0] = -1;
	}
	if (dev->dst_canvas[1] >= 0) {
		canvas_pool_map_free_canvas(dev->dst_canvas[1]);
		dev->dst_canvas[1] = -1;
	}
	if (dev->thumbnail_canvas[0] >= 0) {
		canvas_pool_map_free_canvas(dev->thumbnail_canvas[0]);
		dev->thumbnail_canvas[0] = -1;
	}
	if (dev->thumbnail_canvas[1] >= 0) {
		canvas_pool_map_free_canvas(dev->thumbnail_canvas[1]);
		dev->thumbnail_canvas[1] = -1;
	}
	if (dev->src_canvas[0] >= 0 ||
	    dev->src_canvas[1] >= 0 ||
	    dev->src_canvas[2] >= 0 ||
	    dev->dst_canvas[0] >= 0 ||
		dev->dst_canvas[1] >= 0 ||
		dev->thumbnail_canvas[0] >=0 ||
		dev->thumbnail_canvas[1] >=0) {
		return -1;
	}
	return 0;
}

int get_vcapture_canvas_index(struct vcapture_output *output,
			int *start_canvas)
{
	int canvas = start_canvas[0];
	int v4l2_format = output->v4l2_format;
	void *buf = (void *)output->vbuf;
	int width = output->width;
	int height = output->height;
	int canvas_height = height;

	switch (v4l2_format) {
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV21:
		canvas_config(start_canvas[0], (unsigned long)buf, width,
			      canvas_height, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);
		canvas_config(start_canvas[1], (unsigned long)buf + width * canvas_height, width,
			      canvas_height >> 1, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);
		canvas = (start_canvas[1] << 8) | start_canvas[0];
		break;
	case V4L2_PIX_FMT_RGB565X:
	case V4L2_PIX_FMT_VYUY:
		canvas = start_canvas[0];
		canvas_config(canvas, (unsigned long)buf, width * 2,
			      canvas_height, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);
		break;
	case V4L2_PIX_FMT_YUV444:
	case V4L2_PIX_FMT_BGR24:
	case V4L2_PIX_FMT_RGB24:
		canvas = start_canvas[0];
		canvas_config(canvas, (unsigned long)buf, width * 3,
			      canvas_height, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);
		break;
	case V4L2_PIX_FMT_ABGR32:
	case V4L2_PIX_FMT_ARGB32:
		canvas = start_canvas[0];
		canvas_config(canvas, (unsigned long)buf, width * 4,
			      canvas_height, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);
		break;
	case V4L2_PIX_FMT_GREY:
		canvas = start_canvas[0];
		canvas_config(canvas, (unsigned long)buf, width,
			      canvas_height, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);				  
		break;
	default:
		break;
	}
	return canvas;
}

int get_vcapture_thumbnail_canvas_index(struct vcapture_output *output,
			int *start_canvas)
{
	int canvas = start_canvas[0];
	int v4l2_format = output->thumbnail_v4l2_format;
	void *buf = (void *)output->thumbnail_vbuf;
	int width = output->thumbnail_width;
	int height = output->thumbnail_height;
	int canvas_height = height;

	switch (v4l2_format) {
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV21:
		canvas_config(start_canvas[0], (unsigned long)buf, width,
			      canvas_height, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);
		canvas_config(start_canvas[1], (unsigned long)buf + width * canvas_height, width,
			      canvas_height >> 1, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);
		canvas = (start_canvas[1] << 8) | start_canvas[0];
		break;
	case V4L2_PIX_FMT_GREY:
		canvas = start_canvas[0];
		canvas_config(canvas, (unsigned long)buf, width,
			      canvas_height, CANVAS_ADDR_NOWRAP,
			      CANVAS_BLKMODE_LINEAR);				  
		break;
	default:
		break;
	}
	return canvas;
}

#define VCAPTURE_IOC_MAGIC 'T'
#define VCAPTURE_IOCTL_SET_SKIP_FRAMES _IOW(VCAPTURE_IOC_MAGIC, 0x00, int)
#define VCAPTURE_IOCTL_SET_SKIP_2_LINE _IOW(VCAPTURE_IOC_MAGIC, 0x01, int)
#define VCAPTURE_IOCTL_SET_SECURE _IOW(VCAPTURE_IOC_MAGIC, 0x02, int)
#define VCAPTURE_IOCTL_SET_SCALE _IOW(VCAPTURE_IOC_MAGIC, 0x03, int)
#define VCAPTURE_IOCTL_SET_THUMBNAIL_WIDTH_HIGHT _IOW(VCAPTURE_IOC_MAGIC, 0x04, int)
#define VCAPTURE_IOCTL_SET_THUMBNAIL_SCALE _IOW(VCAPTURE_IOC_MAGIC, 0x05, int)

#endif /* VCAPTURE_H_ */

