#include <linux/version.h>
#include <linux/string.h>
#include <linux/io.h>
#include <linux/mm.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/vmalloc.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/kthread.h>
#include <linux/wait.h>
#include <linux/platform_device.h>
#include <linux/amlogic/media/vout/lcd/aml_ldim.h>
#include <linux/amlogic/media/vout/lcd/aml_bl.h>
#include <linux/amlogic/media/vout/lcd/ldim_alg.h>
#include <linux/amlogic/media/vout/lcd/ldim_support.h>
#include "ldim_drv.h"
#include "ldim_reg.h"
#include "ldim_wdog.h"
#include "ldim_proc.h"
#ifdef CONFIG_AMLOGIC_LOCAL_DIMMING_AS3824
#include "as3824.h"
#endif


int print_box_header(char *buf, int len, int buff_size, int col,
					int box_size_before, int box_size_after)
{
	int i;
	char line1[128];
	char line2[128];

	box_size_before %= 128;	// make sure line buf doesn't over run
	box_size_after %= 128;	// make sure line buf doesn't over run

	for (i = 0; i < box_size_before; i++)
		line1[i] = ' ';
	line1[i] = 0;

	for (i = 0; i < box_size_after; i++)
		line2[i] = ' ';
	line2[i] = 0;

	len += snprintf(buf+len, buff_size - (len + 1), "%sCol %d%s", line1, 0, line2);

	for (i = 1; i < col-1; i++)
		len += snprintf(buf+len, buff_size - (len + 1), "%sCol %d%s", line1, i, line2);

	len += snprintf(buf+len, buff_size - (len + 1), "%sCol %d\n", line1, i);

	return len;
}

int print_box_line(char *buf, int len, int buff_size, int index,
						int row, int col, int box_size)
{
	int i, j;
	char line1[128];
	char line2[128];
	char line3[128];

	if (box_size < 7) {
		pr_warn("Warning: %s :: length '%d' is too small, min is 7.\n", __func__, box_size);
		return 0;
	}

	box_size %= 128;	// make sure line buf doesn't over run

	for (i = 0; i < box_size-6; i++)
		line1[i] = '-';
	line1[i] = 0;

	for (i = 0; i < box_size-7; i++)
		line2[i] = '-';
	line2[i] = 0;

	for (i = 0; i < box_size; i++)
		line3[i] = '-';
	line3[i] = 0;

	if (index >= 0 && index < row) {
		if (index > 9)
			len += snprintf(buf + len, buff_size - (len + 1),
							"Row %d %s+", index, line2);
		else
			len += snprintf(buf + len, buff_size - (len + 1),
							"Row %d %s+", index, line1);
	} else
		len += snprintf(buf + len, buff_size - (len + 1), "%s+", line3);

	for (j = 1; j < col-1; j++)
		len += snprintf(buf + len, buff_size - (len + 1), "%s+", line3);
	len += snprintf(buf + len, buff_size - (len + 1), "%s|\n", line3);

	return len;
}


static void ldim_dump_matrix_proc(struct aml_ldim_driver_s *ldim_drv, struct seq_file *sfile,unsigned int *matrix_in,
								int box_size, int box_size_before, int box_size_after	)
{
	unsigned int i, j, n, len;
	unsigned int *ldim_matrix_t = NULL;
	char *buf;

	n = PAGE_SIZE;
	buf = kcalloc(n, sizeof(char), GFP_KERNEL);
	if (!buf)
		return;

	len = ldim_drv->conf->hist_row * ldim_drv->conf->hist_col;
	ldim_matrix_t = kcalloc(len, sizeof(unsigned int), GFP_KERNEL);
	if (!ldim_matrix_t) {
		kfree(buf);
		return;
	}
	memcpy(ldim_matrix_t, matrix_in,
	       len * sizeof(unsigned int));

	if (!sfile)
		pr_info("%s:\n", __func__);

	len = 0;

	if (ldim_drv->colour_debug) {
		len += snprintf(buf + len, PAGE_SIZE-(len+1), "        ");
		len += print_box_header(buf, len, PAGE_SIZE-(len+1),
							ldim_drv->conf->hist_col, box_size_before, box_size_after-1);
	} else{
		len += print_box_header(buf, len, PAGE_SIZE-(len+1),
							ldim_drv->conf->hist_col, box_size_before, box_size_after);
		len += print_box_line(buf, len, PAGE_SIZE-(len+1), 0,
						ldim_drv->conf->hist_row,
						ldim_drv->conf->hist_col, box_size);
	}
	if (sfile)
		seq_printf(sfile, "%s", buf);
	else
		pr_info("%s", buf);
	len = 0;

	for (i = 0; i < ldim_drv->conf->hist_row; i++) {

		if (ldim_drv->colour_debug)
			len += snprintf(buf + len, PAGE_SIZE-(len+1), " Row %2d ", i);

		for (j = 0; j < ldim_drv->conf->hist_col; j++) {
			n = ldim_drv->conf->hist_col * i + j;
			SET_COLOUR(255, 255-(ldim_matrix_t[n]/16));

			if (ldim_drv->colour_debug)
				len += snprintf(buf + len, PAGE_SIZE-(len+1),
								"  %4d  ", ldim_matrix_t[n]);
			else
				len += snprintf(buf + len, PAGE_SIZE-(len+1),
								"  %4d  |", ldim_matrix_t[n]);

			RESET_COLOUR();
		}
		len += snprintf(buf + len, PAGE_SIZE-(len+1), "\n");

		if (!ldim_drv->colour_debug)
			len += print_box_line(buf, len, PAGE_SIZE-(len+1), i+1,
							ldim_drv->conf->hist_row,
							ldim_drv->conf->hist_col, box_size);
		if (sfile)
			seq_printf(sfile, "%s", buf);
		else
			pr_info("%s", buf);
		len = 0;
	}
	kfree(buf);
	kfree(ldim_matrix_t);
}

int print_matrix(struct seq_file *m, int row, int column, int color_debug, unsigned int *value)
{
    ssize_t len = 0;
	char *buf;
	unsigned int i, j, n;

	n = PAGE_SIZE;
	buf = kcalloc(n, sizeof(char), GFP_KERNEL);
	if (!buf)
		return -1;

	for (i = 0; i < row; i++) {
		len = 0;
		for (j = 0; j < column; j++) {
			n = column * i + j;
			SET_COLOUR_NEW(color_debug, 255, 255-(value[n]/16));
			len += snprintf(buf + len, PAGE_SIZE-(len+1), "\t%4d",
							value[n]);
			RESET_COLOUR_NEW(color_debug);
		}
		seq_printf(m, "%s\n", buf);
	}

	kfree(buf);
	return 0;
}

int ldim_proc_get_hist_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nLDIM INFO MATRIX FORMAT\n");
	seq_puts(m, "----------Under Construction---------------\n");

	ldim_dump_histgram_matrix(ldim_drv, false, m);

	return 0;
}

int ldim_proc_get_hist_percent(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nLDIM INFO PERECENT FORMAT\n");
	seq_puts(m, "----------Under Construction---------------\n");

	ldim_dump_histgram_matrix(ldim_drv, true, m);

	return 0;
}

int ldim_proc_get_maxrgb(struct seq_file *m, void *v)
{
//	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nLDIM MAX RGB\n");
	seq_puts(m, "----------Under Construction---------------\n");

	//ldim_get_matrix_info_max_rgb(ldim_drv, m);

	return 0;
}

int ldim_proc_get_boost_info(struct seq_file *m, void *v)
{
	unsigned int i;
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();
	struct ldim_boost_s *boost = ldim_drv->boost;

	seq_puts(m, "\nBOOST INFO ( aml_ldim_info_s )\n");
	seq_puts(m, "-----------------------------\n");
	seq_printf(m, "%-20s %s\n", "boost_en:", boost->boost_en ? "TRUE" : "FALSE");
	seq_printf(m, "%-20s %s\n", "burst_en:", boost->burst_en ? "TRUE" : "FALSE");
	seq_printf(m, "%-20s %s\n", "contrast_en:",
					boost->contrast_en ? "TRUE" : "FALSE");
	seq_printf(m, "%-20s %s\n", "constraint_en:",
					boost->constraint_en ? "TRUE" : "FALSE");
	seq_printf(m, "%-20s %d\n", "code_value_on(0 to 1023):", boost->code_value_on);
	seq_printf(m, "%-20s %d\n", "dac_ref_on(800 or 500):", boost->dac_ref_on);
	seq_printf(m, "%-20s %d\n", "code_value_off(0 to 1023):", boost->code_value_off);
	seq_printf(m, "%-20s %d\n", "dac_ref_off(800 or 500):", boost->dac_ref_off);
	seq_printf(m, "%-20s %d\n", "tot_in_vol in mV:", boost->tot_in_vol);
	seq_printf(m, "%-20s %d\n", "max_tot_pow in watts:", boost->max_tot_pow);
	seq_printf(m, "%-20s %d\n", "max_boost_zone:", boost->max_boost_zone);
	seq_printf(m, "%-20s %d\n", "max_boost_duration(Seconds):", boost->max_boost_duration);
	seq_printf(m, "%-20s %d(%d pct)\n", "duty_high_level (0 to 4095):",
				boost->duty_high_level, (boost->duty_high_level * 100)/4095);
	seq_printf(m, "%-20s %d\n", "cool_prot_duration(Seconds):", boost->cool_prot_duration);
	seq_printf(m, "%-20s %d\n", "cool_prot_level:", boost->cool_prot_level);
	seq_printf(m, "%-20s %d\n", "boost_gain(0 to to 4095):", boost->boost_gain);
	seq_printf(m, "%-20s %d\n", "grad_step:", boost->grad_step);
	seq_printf(m, "%-20s %d\n", "comp_en:", boost->comp_en);

	seq_puts(m, "contrast_gain_lut[17]:\n");
	seq_puts(m, "\t");
	for (i = 0; i < 17; i++)	// this value if 17 comes from default_contrast_gain_lut[17]
		seq_printf(m, "%4d, ", boost->contrast_gain_lut[i]);
	seq_puts(m, "\n");
	seq_printf(m, "%-20s %d\n", "contrast_gain:", boost->contrast_gain);
	seq_printf(m, "%-20s %d\n", "contrast_gain_max_delta:", boost->contrast_gain_max_delta);
	seq_printf(m, "%-20s %d\n", "duty_high_th:", boost->duty_high_th);
	seq_printf(m, "%-20s %d\n", "tf_high_alpha:", boost->tf_high_alpha);
	seq_printf(m, "%-20s %d\n", "tf_low_alpha:", boost->tf_low_alpha);
	seq_printf(m, "%-20s %d\n", "tf_diff_th:", boost->tf_diff_th);
	seq_printf(m, "%-20s %d\n", "max_regulator_pow:", boost->max_regulator_pow);
	seq_printf(m, "%-20s %d\n", "reg_in_vol:", boost->reg_in_vol);
	seq_printf(m, "%-20s", "regulator_zones: ");
	for (i = 0; i < LDIM_MAX_REGULATORS; i++) {
		seq_printf(m, "%d ", boost->regulator_zone[i]);
	}
	seq_puts(m, "\n");
	seq_printf(m, "%-20s %d\n", "curr_boost_en:", boost->curr_boost_en);
	seq_printf(m, "%-20s %d\n", "pre_boost_en:", boost->pre_boost_en);
	seq_printf(m, "%-20s %d\n", "switch_en_flag:", boost->switch_en_flag);
	seq_printf(m, "%-20s %d\n", "ref_vol_on(mV):", boost->ref_vol_on);
	seq_printf(m, "%-20s %d\n", "ref_vol_off(mV):", boost->ref_vol_off);
	seq_printf(m, "%-20s %d\n", "min_duty_gain:", boost->min_duty_gain);
	seq_printf(m, "%-20s %d\n", "max_boost_duty(max_boost_zone x 4095):",
								boost->max_boost_duty);
	seq_printf(m, "%-20s %d\n", "max_tot_duty:", boost->max_tot_duty);
	seq_printf(m, "%-20s %d\n", "curr_sum_duty:", boost->curr_sum_duty);
	seq_printf(m, "%-20s %d\n", "pre_sum_duty:", boost->pre_sum_duty);
	seq_printf(m, "%-20s %d\n", "curr_sum_boost_duty:", boost->curr_sum_boost_duty);
	seq_printf(m, "%-20s %d\n", "pre_sum_boost_duty:", boost->pre_sum_boost_duty);
	seq_printf(m, "%-20s %d\n", "prot_state:", boost->prot_state);
	seq_printf(m, "%-20s %d\n", "curr_grad_level:", boost->curr_grad_level);
	seq_printf(m, "%-20s %d\n", "cool_prot_cnt:", boost->cool_prot_cnt);
	seq_printf(m, "%-20s %d\n", "max_boost_cnt_th:", boost->max_boost_cnt_th);
	seq_printf(m, "%-20s %d\n", "cool_prot_cnt_th:", boost->cool_prot_cnt_th);
	seq_printf(m, "%-20s %d\n", "curr_framerate:", boost->curr_framerate);
	seq_printf(m, "%-20s %d\n", "sum_cool_cnt:", boost->sum_cool_cnt);
	seq_printf(m, "%-20s %d\n", "max_regulator_duty:", boost->max_regulator_duty);
	seq_printf(m, "%-20s %d\n", "curr_regulator_duty:", boost->curr_regulator_duty);
#if BOOST_PROC_INFO_DEBUG	
	seq_printf(m, "%-20s %d\n", "1: sum_duty_post_reg_const:", boost->sum_duty_post_reg_const);
	seq_printf(m, "%-20s %d\n", "2: sum_duty_post_pow_const:", boost->sum_duty_post_pow_const);
	seq_printf(m, "%-20s %d\n", "3: sum_duty_post_max_boost_const:", boost->sum_duty_post_max_boost_const);
	seq_printf(m, "%-20s %d\n", "4: sum_duty_post_alpa_filt:", boost->sum_duty_post_alpa_filt);
	seq_printf(m, "%-20s %d\n", "5: sum_duty_post_duration_constr:", boost->sum_duty_post_duration_constr);
#endif	
	seq_puts(m, "boost duty_high_cnt:\n");
	print_matrix(m, ldim_drv->conf->hist_row, ldim_drv->conf->hist_col, ldim_drv->colour_debug, boost->duty_high_cnt);

	seq_puts(m, "\n");
	seq_printf(m, "%-20s %lu\n", "vsync isr count:", ldim_drv->boost->vsync_count);


	return 0;
}

static int ldim_proc_get_litgain_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nLITGAIN MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->litgain_matrix,8,2,2);
	return 0;
}
static int ldim_proc_get_boostgain_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nBOOST GAIN MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->boost_gain_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_reg_constraint_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nREGULATOR CONSTRAINT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->reg_constraint_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_pow_constraint_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nTOTAL POWER CONSTRAINT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->pow_constraint_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_maxbzone_constraint_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nMAX BOOST ZONE CONSTRAINT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->maxbzone_constraint_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_time_constraint_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nTIME CONSTRAINT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->time_constraint_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_temp_filter_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nTEMPORAL FILTER MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->temp_filter_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_pause_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nPAUSE OUTPUT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->pause_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_ldim_algo_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nLDIM FW OUTPUT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->fw_para->nprm->post_bl_matrix,8,2,2);
	return 0;
}

static int ldim_proc_get_bcontrast_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nBOOST CONTRAST OUTPUT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->bl_matrix_contrast,8,2,2);
	return 0;
}

static int ldim_proc_get_tally_matrix(struct seq_file *m, void *v)
{
	struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nTALLY OUTPUT MATRIX\n");
	seq_puts(m, "-------------------------\n");

	ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost->tally->output,8,2,2);
	return 0;
}

static int ldim_proc_get_smooth_matrix(struct seq_file *m, void *v)
{
	//struct aml_ldim_driver_s *ldim_drv = aml_ldim_get_driver();

	seq_puts(m, "\nSMOOTH STAGE MATRIX\n");
	seq_puts(m, "-------------------------\n");

	//ldim_dump_matrix_proc(ldim_drv,m,ldim_drv->boost ??????? ,8,2,2);
	return 0;
}

LDIM_PROC_ENTRY(litgain_matrix, NULL);
LDIM_PROC_ENTRY(boostgain_matrix, NULL);
LDIM_PROC_ENTRY(temp_filter_matrix, NULL);
LDIM_PROC_ENTRY(pause_matrix, NULL);
LDIM_PROC_ENTRY(reg_constraint_matrix, NULL);
LDIM_PROC_ENTRY(time_constraint_matrix, NULL);
LDIM_PROC_ENTRY(pow_constraint_matrix, NULL);
LDIM_PROC_ENTRY(maxbzone_constraint_matrix, NULL);
LDIM_PROC_ENTRY(ldim_algo_matrix, NULL);
LDIM_PROC_ENTRY(bcontrast_matrix, NULL);
LDIM_PROC_ENTRY(tally_matrix, NULL);
LDIM_PROC_ENTRY(smooth_matrix, NULL);

static struct proc_dir_entry *parent;

int ldim_proc_init(void)
{
	// Create proc file system for ldim_algo
	parent = proc_mkdir("driver/ldim/ldim_algo", NULL);
	proc_create(ldim_algo_matrix_proc_name, 0444, parent, &ldim_ldim_algo_matrix_proc_fops);

	// Create proc file system for boost contrast
	parent = proc_mkdir("driver/ldim/bcontrast", NULL);
	proc_create(bcontrast_matrix_proc_name, 0444, parent, &ldim_bcontrast_matrix_proc_fops);

	// Create proc file system for tally
	parent = proc_mkdir("driver/ldim/tally", NULL);
	proc_create(tally_matrix_proc_name, 0444, parent, &ldim_tally_matrix_proc_fops);

	// Create proc file system for smooth
	parent = proc_mkdir("driver/ldim/smooth", NULL);
	proc_create(smooth_matrix_proc_name, 0444, parent, &ldim_smooth_matrix_proc_fops);

	// Create proc file system for litgain
	parent = proc_mkdir("driver/ldim/litgain", NULL);
	proc_create(litgain_matrix_proc_name, 0444, parent, &ldim_litgain_matrix_proc_fops);

	// Create proc file system for boostgain
	parent = proc_mkdir("driver/ldim/boostgain", NULL);
	proc_create(boostgain_matrix_proc_name, 0444, parent, &ldim_boostgain_matrix_proc_fops);

	// Create proc file system for time constraint
	parent = proc_mkdir("driver/ldim/time_constraint", NULL);
	proc_create(time_constraint_matrix_proc_name, 0444, parent, &ldim_time_constraint_matrix_proc_fops);
	
	// Create proc file system for temporal filter
	parent = proc_mkdir("driver/ldim/temp_filter", NULL);
	proc_create(temp_filter_matrix_proc_name, 0444, parent, &ldim_temp_filter_matrix_proc_fops);

	// Create proc file system for pause
	parent = proc_mkdir("driver/ldim/pause", NULL);
	proc_create(pause_matrix_proc_name, 0444, parent, &ldim_pause_matrix_proc_fops);

	// Create proc file system for Regulator constraint
	parent = proc_mkdir("driver/ldim/reg_constraint", NULL);
	proc_create(reg_constraint_matrix_proc_name, 0444, parent, &ldim_reg_constraint_matrix_proc_fops);

	// Create proc file system for total power constraint
	parent = proc_mkdir("driver/ldim/pow_constraint", NULL);
	proc_create(pow_constraint_matrix_proc_name, 0444, parent, &ldim_pow_constraint_matrix_proc_fops);

	// Create proc file system for max booze zone constraint
	parent = proc_mkdir("driver/ldim/maxbzone_constraint", NULL);
	proc_create(maxbzone_constraint_matrix_proc_name, 0444, parent, &ldim_maxbzone_constraint_matrix_proc_fops);
	
	return 0;

}
