/*
 * drivers/amlogic/media/video_processor/vcapture/vcapture.c
 *
 * Copyright (C) 2022 Amlogic, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/semaphore.h>
#include <linux/atomic.h>
#include <asm/cacheflush.h>
#include <media/v4l2-ioctl.h>
#include <media/videobuf2-core.h>
#include <media/videobuf2-v4l2.h>
#include <linux/amlogic/major.h>
#include <linux/platform_device.h>
#include <linux/amlogic/cpu_version.h>
#include <linux/amlogic/media/vout/vout_notify.h>
#include <linux/amlogic/media/canvas/canvas_mgr.h>
#include <linux/amlogic/media/codec_mm/codec_mm.h>
#include <linux/amlogic/tee.h>
#include "vcapture.h"

#define RECEIVER_NAME "vcapture"
#define PROVIDER_NAME "vcapture"

#define VCAPTURE_MAJOR_VERSION 1
#define VCAPTURE_MINOR_VERSION 0
#define VCAPTURE_RELEASE 1
#define VCAPTURE_VERSION \
	KERNEL_VERSION(VCAPTURE_MAJOR_VERSION, VCAPTURE_MINOR_VERSION,\
		       VCAPTURE_RELEASE)

#define MAGIC_RE_MEM 0x123039dc
#define VCAPTURE_MODULE_NAME "vcapture"

#define V4L2_AMLOGIC_VCAPTURE_BASE  (V4L2_CID_USER_BASE | 0x1200)

#define PRINT_ERROR			0x0000
#define PRINT_CAPTUREINFO	0x0001
#define PRINT_BUFFERINFO	0x0002
#define PRINT_THREADINFO	0x0004
#define PRINT_PATHINFO		0x0008
#define PRINT_TIMESTAMP		0x0010
#define PRINT_GETINFO		0x0020
#define PRINT_CAP_OTHER		0x0040
#define PRINT_THUMBNAIL		0x0080
#define DUMP_SRC			0x0100
#define DUMP_DST			0x0200
#define DUMP_THUMBNAIL		0x0400
#define DUMP_THUMBNAIL_AVG	0x0800
#define PRINT_TEEAPI		0x1000

static int capture_debug;
static int capture_secure_mode = 1;
static unsigned int secure_mem_handle;  /* secure mem handle */
static bool capture_use_thread = true;
static int capture_thread;
static int capture_frame_per_n = 1;
static int capture_lines_per_n = 1;
static int capture_scale_down;
static int thumbnail_scale_down = (10 << 16) | 16; /* 10x16 */
static unsigned int vcapture_base = 60;
static int dump_counter;
static unsigned char thumbnail[MAX_VBUF][MAX_ZONE];
static unsigned char histogram[MAX_VBUF][MAX_ZONE][8];
#ifdef USE_TEE_API
static unsigned char databuf[MAX_ZONE * 9];
#endif

extern int ldim_get_vf(struct vframe_s *vf);

int vcapture_print(int debug_flag, const char *fmt, ...)
{
	if ((capture_debug & debug_flag) ||
	    (debug_flag == PRINT_ERROR)) {
		unsigned char buf[256];
		int len = 0;
		va_list args;

		va_start(args, fmt);
		len = sprintf(buf, "vcapture.0 ");
		vsnprintf(buf + len, 256 - len, fmt, args);
		pr_info("%s", buf);
		va_end(args);
	}
	return 0;
}

/* ------------------------------------------------------------------
 * cma
 * ------------------------------------------------------------------
 */
static int vcapture_cma_buf_init(struct vcapture_dev *dev)
{
	int flags;
	int ret;

	flags = CODEC_MM_FLAGS_DMA | CODEC_MM_FLAGS_CMA_CLEAR;
	ret = alloc_canvas(dev);
	if (ret < 0) {
		vcapture_print(PRINT_ERROR,
			"alloc canvas failed!\n");
		return -1;
	}
	dev->buffer_start =
		codec_mm_alloc_for_dma(dev->recv_name,
			(CMA_ALLOC_SIZE * SZ_1M) / PAGE_SIZE, 0, flags);
	if (!(dev->buffer_start)) {
		vcapture_print(PRINT_ERROR,
			"alloc cma buffer failed\n");
		return -1;
	}
	dev->buffer_size = CMA_ALLOC_SIZE * SZ_1M;
	if (capture_secure_mode) {
		tee_protect_mem_by_type(TEE_MEM_TYPE_HCODEC,
					dev->buffer_start, dev->buffer_size,
					&(secure_mem_handle));
		vcapture_print(PRINT_BUFFERINFO,"secure_mem_handle 0x%x secure size 0x%x\n",
			secure_mem_handle, dev->buffer_size);
	}
	vcapture_print(PRINT_BUFFERINFO,
		"vcapture video cma memory is %x , size is  %x\n",
		(unsigned int)dev->buffer_start,
		(unsigned int)dev->buffer_size);
	return 0;
}

static int vcapture_cma_buf_uninit(struct vcapture_dev *dev)
{
	int ret;

	ret = free_canvas(dev);
	if (ret < 0)
		vcapture_print(PRINT_ERROR,
			"%s free canvas fail! line: %d",
			__func__, __LINE__);
	if (dev->buffer_start != 0) {
		if (capture_secure_mode) {
			tee_unprotect_mem(secure_mem_handle);
			secure_mem_handle = 0;
			vcapture_print(PRINT_BUFFERINFO,
			      "tee_unprotect_mem succeed!\n");
		}
		codec_mm_free_for_dma("vcapture", dev->buffer_start);
		dev->buffer_start = 0;
		dev->buffer_size = 0;
		vcapture_print(PRINT_BUFFERINFO,
			      "vcapture cma memory release succeed!\n");
	}
	return 0;
}

/* ------------------------------------------------------------------
 * vfm operations
 * ------------------------------------------------------------------
 */
static struct vframe_s *vcapture_vf_peek(void *op_arg)
{
	struct vcapture_dev *dev = (struct vcapture_dev *)op_arg;

	return vf_peek(dev->recv_name);
}

static struct vframe_s *vcapture_vf_get(void *op_arg)
{
	struct vframe_s *vf;
	struct vcapture_dev *dev = (struct vcapture_dev *)op_arg;
	bool capture_frame = false;
	bool capture_lines = false;
	bool capture_thumbnail = false;
	int f_per_n;
	int l_per_n;

	vf = vf_get(dev->recv_name);
	if (!vf)
		return NULL;
	if (!dev)
		return vf;

	if (atomic_read(&dev->is_playing) == 0) {
		atomic_set(&dev->is_playing, 1);
		dev->frame_counter = 0;
		dev->lines_counter = 0;
	}
	if ((atomic_read(&dev->vcap_status) != VCAPTURE_PREPARE)
	&& (vf != dev->ge2d_vf)) {
		f_per_n = atomic_read(&dev->capture_frame_per_n);
		l_per_n = atomic_read(&dev->capture_lines_per_n);
		dev->frame_counter++;
		dev->lines_counter++;
		capture_frame = (f_per_n <= dev->frame_counter);
		capture_lines = (l_per_n <= dev->lines_counter);
		capture_thumbnail = atomic_read(&dev->is_capturing_thumbnail);
		if (capture_frame || capture_lines || capture_thumbnail) {
			vf->flag &= ~(VFRAME_FLAG_VIDEO_VCAPTURE_FRAME
				| VFRAME_FLAG_VIDEO_VCAPTURE_FRAME
				| VFRAME_FLAG_VIDEO_VCAPTURE_THUMBNAIL);
			if (capture_frame)
				vf->flag |= VFRAME_FLAG_VIDEO_VCAPTURE_FRAME;
			else if (capture_lines)
				vf->flag |= VFRAME_FLAG_VIDEO_VCAPTURE_LINE;
			if (capture_thumbnail)
				vf->flag |= VFRAME_FLAG_VIDEO_VCAPTURE_THUMBNAIL;
			dev->new_vf = vf;
			if (capture_thread)
				up(&dev->thread_sem);
			if (capture_frame)
				dev->frame_counter = 0;
			if (capture_lines)
				dev->lines_counter = 0;
		}
	}

	return vf;
}

static void vcapture_vf_put(struct vframe_s *vf, void *op_arg)
{
	struct vcapture_dev *dev = (struct vcapture_dev *)op_arg;

	if (atomic_read(&dev->vcap_status) == VCAPTURE_PREPARE ||
	(vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE))
		vf->flag |= VFRAME_FLAG_VIDEO_VCAPTURE_PUT;
	else
		vf_put(vf, dev->recv_name);
}

/* ------------------------------------------------------------------
 * ge2d thread and operations
 * ------------------------------------------------------------------
 */
static int copy_phybuf_to_file(ulong phys, u32 size,
			       struct file *fp, loff_t pos)
{
	u32 span = SZ_1M;
	u8 *p;
	int remain_size = 0;
	ssize_t ret;

	remain_size = size;
	while (remain_size > 0) {
		if (remain_size < span)
			span = remain_size;
		p = codec_mm_vmap(phys, PAGE_ALIGN(span));
		if (!p) {
			pr_info("vcapture: vmap failed\n");
			return -1;
		}
		codec_mm_dma_flush(p, span, DMA_FROM_DEVICE);
		ret = vfs_write(fp, (char *)p, span, &pos);
		if (ret <= 0)
			pr_info("vcapture: vfs write failed!\n");
		phys += span;
		codec_mm_unmap_phyaddr(p);
		remain_size -= span;

		pr_info("pos: %lld, phys: %lx, remain_size: %d\n",
			pos, phys, remain_size);
	}
	return 0;
}

static struct vframe_s *vcapture_src_vf(struct vframe_s *vf)
{
	struct vframe_s *src_vf = vf;

	if ((vf->type & (VIDTYPE_COMPRESS | VIDTYPE_DI_PW)) == (VIDTYPE_COMPRESS | VIDTYPE_DI_PW)
	|| (vf->type & VIDTYPE_PRE_INTERLACE)) {
#if 0
		if (vf->uvm_vf) {
			src_vf = vf->uvm_vf;
		} else
#endif
		if (vf->vf_ext) {
			src_vf = (struct vframe_s *)vf->vf_ext;
		}
	}
	return src_vf;
}

static int vcapture_ge2d_process(struct config_para_ex_s *ge2d_config,
				struct vcapture_output *output,
				struct vcapture_dev *dev)
{
	struct vframe_s *vf = dev->ge2d_vf;
	struct ge2d_context_s *context = dev->context;
	struct canvas_s cs0, cs1, cs2, cd, cd1;
	int interlace_mode;
	int output_canvas;
	int input_width, input_height;
	int output_width, output_height;
	struct vframe_s *src_vf;
	int index;
	struct timeval time_begin;
	struct timeval time_end;
	int cost_time;
	unsigned char *pdst;
	int w, h, i, j;
	unsigned char lvl[HIST_BIN] = {224, 192, 160, 128, 96, 64, 32, 0};
#ifdef USE_TEE_API
	phys_addr_t buf_pa;
	phys_addr_t data_pa;
	int ret;
	void __iomem *data_va;
	unsigned char *tbuf_va;
#else
	unsigned int cnt[HIST_BIN];
	int k, l;
	unsigned int sum;
	unsigned char *psrc;
	unsigned char *phist;
	unsigned char *buf_va;
	unsigned char pix;
#endif

	memset(ge2d_config, 0, sizeof(struct config_para_ex_s));
	mutex_lock(&dev->vf_mutex);
	if (atomic_read(&dev->vcap_status) == VCAPTURE_DROP_VF) {
		mutex_unlock(&dev->vf_mutex);
		atomic_set(&dev->vcap_status, VCAPTURE_INIT);
		vcapture_print(PRINT_ERROR,
			"%s exit\n", __func__);
		return -2;
	}

	src_vf = vcapture_src_vf(vf);
	input_width = src_vf->width;
	input_height = src_vf->height;

	if (src_vf->canvas0Addr == (u32)-1) {
		canvas_config_config(dev->src_canvas[0],
			&src_vf->canvas0_config[0]);
		if (src_vf->plane_num > 1) {
			canvas_config_config(dev->src_canvas[1],
				&src_vf->canvas0_config[1]);
		} else if (src_vf->plane_num > 2) {
			canvas_config_config(dev->src_canvas[2],
				&src_vf->canvas0_config[2]);
		}
		ge2d_config->src_para.canvas_index =
			dev->src_canvas[0]
			| (dev->src_canvas[1] << 8)
			| (dev->src_canvas[2] << 16);
		ge2d_config->src_planes[0].addr =
				src_vf->canvas0_config[0].phy_addr;
		ge2d_config->src_planes[0].w =
				src_vf->canvas0_config[0].width;
		ge2d_config->src_planes[0].h =
				src_vf->canvas0_config[0].height;
		ge2d_config->src_planes[1].addr =
				src_vf->canvas0_config[1].phy_addr;
		ge2d_config->src_planes[1].w =
				src_vf->canvas0_config[1].width;
		ge2d_config->src_planes[1].h =
				src_vf->canvas0_config[1].height >> 1;
		if (src_vf->plane_num == 3) {
			ge2d_config->src_planes[2].addr =
				src_vf->canvas0_config[2].phy_addr;
			ge2d_config->src_planes[2].w =
				src_vf->canvas0_config[2].width;
			ge2d_config->src_planes[2].h =
				src_vf->canvas0_config[2].height >> 1;
		}
	} else {
		canvas_read(src_vf->canvas0Addr & 0xff, &cs0);
		canvas_read((src_vf->canvas0Addr >> 8) & 0xff, &cs1);
		canvas_read((src_vf->canvas0Addr >> 16) & 0xff, &cs2);
		ge2d_config->src_planes[0].addr = cs0.addr;
		ge2d_config->src_planes[0].w = cs0.width;
		ge2d_config->src_planes[0].h = cs0.height;
		ge2d_config->src_planes[1].addr = cs1.addr;
		ge2d_config->src_planes[1].w = cs1.width;
		ge2d_config->src_planes[1].h = cs1.height;
		ge2d_config->src_planes[2].addr = cs2.addr;
		ge2d_config->src_planes[2].w = cs2.width;
		ge2d_config->src_planes[2].h = cs2.height;
		ge2d_config->src_para.canvas_index = src_vf->canvas0Addr;
	}

	ge2d_config->src_para.format = get_input_format(src_vf);
	mutex_unlock(&dev->vf_mutex);

	ge2d_config->src_key.key_enable = 0;
	ge2d_config->src_key.key_mask = 0;
	ge2d_config->src_key.key_mode = 0;
	ge2d_config->src_para.mem_type = CANVAS_TYPE_INVALID;
	ge2d_config->src_para.fill_color_en = 0;
	ge2d_config->src_para.fill_mode = 0;
	ge2d_config->src_para.x_rev = 0;
	ge2d_config->src_para.y_rev = 0;
	ge2d_config->src_para.color = 0xffffffff;
	ge2d_config->src_para.top = 0;
	ge2d_config->src_para.left = 0;
	ge2d_config->src_para.width = input_width;
	ge2d_config->src_para.height = input_height;
	ge2d_config->alu_const_color = 0;
	ge2d_config->bitmask_en = 0;
	ge2d_config->src1_gb_alpha = 0;/* 0xff; */
	ge2d_config->src2_para.mem_type = CANVAS_TYPE_INVALID;

	vcapture_print(PRINT_CAPTUREINFO,
			"src addr: %lx %lx %lx, canvas 0x%x, width: %d, height: %d, stride: %d, format: %x %x %x%s\n",
			ge2d_config->src_planes[0].addr,
			ge2d_config->src_planes[1].addr,
			ge2d_config->src_planes[2].addr,
			ge2d_config->src_para.canvas_index,
			ge2d_config->src_para.width,
			ge2d_config->src_para.height,
			ge2d_config->src_planes[0].w,
			src_vf->type,
			get_source_type(src_vf),
			get_input_format(src_vf),
			src_vf == vf ? " " : " use ext");

	if (vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_FRAME ||
		vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_LINE) {
		if (vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_LINE)
			input_height = src_vf->height < CAPTURE_LINES ? src_vf->height : CAPTURE_LINES;
		else
			input_height = src_vf->height;
		if (input_width > output->width)
			input_width = output->width;
		if (input_height > output->height)
			input_height = output->height;
		interlace_mode = src_vf->type & VIDTYPE_TYPEMASK;
		if (interlace_mode == VIDTYPE_INTERLACE_BOTTOM ||
		    interlace_mode == VIDTYPE_INTERLACE_TOP)
			input_height >>= 1;

		ge2d_config->src_para.width = input_width;
		ge2d_config->src_para.height = input_height;

		if (vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_LINE) {
			output_width = input_width;
			output_height = input_height;
		} else {
			if (capture_scale_down <= 8) {
				output_width = input_width >> capture_scale_down;
				output_height = input_height >> capture_scale_down;
			} else {
				output_width = (capture_scale_down >> 16) & 0xffff;
				if (output_width > input_width)
					output_width = input_width;
				output_height = capture_scale_down & 0xffff;
				if (output_height > input_height)
					output_height = input_height;
			}				
		}
		output->width = (output_width + 31) & (~31);
		output->height = (output_height + 1) & (~1);
		if (output->vbuf) { 
			output_canvas =
				get_vcapture_canvas_index(output, dev->dst_canvas);
			canvas_read(output_canvas & 0xff, &cd);
			ge2d_config->dst_planes[0].addr = cd.addr;
			ge2d_config->dst_planes[0].w = cd.width;
			ge2d_config->dst_planes[0].h = cd.height;
			if (output->v4l2_format == V4L2_PIX_FMT_NV12
			|| output->v4l2_format == V4L2_PIX_FMT_NV21) {
				canvas_read((output_canvas >> 8) & 0xff, &cd1);
				ge2d_config->dst_planes[1].addr = cd1.addr;
				ge2d_config->dst_planes[1].w = cd1.width;
				ge2d_config->dst_planes[1].h = cd1.height;
			}
			ge2d_config->dst_para.canvas_index = output_canvas;
			ge2d_config->dst_para.mem_type = CANVAS_TYPE_INVALID;
			ge2d_config->dst_para.format =
				get_output_format(output->v4l2_format) | GE2D_LITTLE_ENDIAN;
			ge2d_config->dst_para.fill_color_en = 0;
			ge2d_config->dst_para.fill_mode = 0;
			ge2d_config->dst_para.x_rev = 0;
			ge2d_config->dst_para.y_rev = 0;
			ge2d_config->dst_para.color = 0;
			ge2d_config->dst_para.top = 0;
			ge2d_config->dst_para.left = 0;
			ge2d_config->dst_para.width = output_width;
			ge2d_config->dst_para.height = output_height;
			ge2d_config->dst_xy_swap = 0;
			ge2d_config->mem_sec = capture_secure_mode;
			for (index = 0; index < dev->vbuf_count; index++) {
				if (output->vbuf == dev->vbuf_addr[index]) {
					dev->vbuf_fmt[index] =
						((ge2d_config->dst_planes[0].w >> 5) << 24) |
						(ge2d_config->dst_para.width << 12) |
						(ge2d_config->dst_para.height << 0);
					if (vf->type & VIDTYPE_COMPRESS)
						dev->vbuf_field[index] =
							(vf->compWidth << 12) | vf->compHeight;
					else
						dev->vbuf_field[index] =
							(vf->width << 12) | vf->height;

					if (vf->duration)
						dev->vbuf_field[index] |= ((96000 / vf->duration) & 0x3f) << 25;

					dev->vbuf_field[index] |=
						(src_vf->type & VIDTYPE_INTERLACE) ? (1 << 31) : 0;
					dev->vbuf_field[index] |=
						(vf->type & VIDTYPE_INTERLACE) ? (1 << 31) : 0;
					dev->vbuf_field[index] |=
						(vf->type_original & VIDTYPE_INTERLACE) ? (1 << 31) : 0;
					break;
				}
			}
			if (index == dev->vbuf_count) {
				vcapture_print(PRINT_ERROR,
					"++buf error vbuf=%p, buf[0]=%p, buf[1]=%p.\n",
					output->vbuf,
					dev->vbuf_addr[0],
					dev->vbuf_addr[1]);
				return -2;
			}

			vcapture_print(PRINT_CAPTUREINFO,
				"frame dst addr: %lx %lx, canvas 0x%x, width: %d, height: %d, stride: %d, format: %x, field: %x\n",
				ge2d_config->dst_planes[0].addr,
				ge2d_config->dst_planes[1].addr,
				ge2d_config->dst_para.canvas_index,
				(dev->vbuf_fmt[index] >> 12) & 0xfff,
				(dev->vbuf_fmt[index] >> 0) & 0xfff,
				((dev->vbuf_fmt[index] >> 24) & 0xff) << 5,
				get_output_format(output->v4l2_format),
				dev->vbuf_field[index]);
			if (ge2d_context_config_ex(context, ge2d_config) < 0) {
				vcapture_print(PRINT_ERROR,
					"++ge2d configing error.\n");
				return -1;
			}

			do_gettimeofday(&time_begin);
			if (!capture_scale_down)
				bitblt_noalpha(context, 0, 0, input_width, input_height, 0, 0);
			else
				stretchblt_noalpha(context, 0, 0, input_width, input_height,
					0, 0, output_width, output_height);
			do_gettimeofday(&time_end);
			cost_time = (time_end.tv_sec * 1000 * 1000 + time_end.tv_usec
					- time_begin.tv_sec * 1000 * 1000
					- time_begin.tv_usec) / 1000;
			vcapture_print(PRINT_TIMESTAMP,
					"%s line: %d frame ge2d done, cost time: %d ms\n", __func__, __LINE__, cost_time);
		}
	}

	if (vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_THUMBNAIL) {
		input_width = src_vf->width;
		input_height = src_vf->height;
		interlace_mode = src_vf->type & VIDTYPE_TYPEMASK;
		if (interlace_mode == VIDTYPE_INTERLACE_BOTTOM ||
	    	interlace_mode == VIDTYPE_INTERLACE_TOP)
			input_height >>= 1;
		ge2d_config->src_para.width = input_width;
		ge2d_config->src_para.height = input_height;

		output_width = output->thumbnail_width;
		output_height = output->thumbnail_height;
		output->thumbnail_width = (output_width + 31) & (~31);
		output->thumbnail_height = (output_height + 1) & (~1);
		//output->thumbnail_v4l2_format = V4L2_PIX_FMT_NV12;
		output->thumbnail_v4l2_format = V4L2_PIX_FMT_GREY;

#ifdef USE_TEE_API
		ret = tee_get_video_buf(&buf_pa);
		if (ret == 0) {
			tbuf_va = (unsigned char * )__phys_to_virt(buf_pa);
			output->thumbnail_vbuf =(void *)buf_pa;
			if (capture_debug & PRINT_TEEAPI)
				vcapture_print(PRINT_TEEAPI,
				"%s line:%d, buf_pa=0x%x, output->thumbnail_vbuf=%p\n",
				__func__, __LINE__,  buf_pa, output->thumbnail_vbuf);
		} else {
			vcapture_print(PRINT_ERROR,
					"get buf error.\n");
			return -1;
		}
#else
		output->thumbnail_vbuf = dev->thumbnail_addr[dev->thumbnail_index];
#endif
		if (output->thumbnail_vbuf) {
			output_canvas =
				get_vcapture_thumbnail_canvas_index(output, &dev->thumbnail_canvas[0]);
			canvas_read(output_canvas & 0xff, &cd);
			ge2d_config->dst_planes[0].addr = cd.addr;
			ge2d_config->dst_planes[0].w = cd.width;
			ge2d_config->dst_planes[0].h = cd.height;
			if (output->thumbnail_v4l2_format == V4L2_PIX_FMT_NV12
			|| output->thumbnail_v4l2_format == V4L2_PIX_FMT_NV21) {
				canvas_read((output_canvas >> 8) & 0xff, &cd1);
				ge2d_config->dst_planes[1].addr = cd1.addr;
				ge2d_config->dst_planes[1].w = cd1.width;
				ge2d_config->dst_planes[1].h = cd1.height;
			}
			ge2d_config->dst_para.canvas_index = output_canvas;
			ge2d_config->dst_para.mem_type = CANVAS_TYPE_INVALID;
			ge2d_config->dst_para.format =
				get_output_format(output->thumbnail_v4l2_format) | GE2D_LITTLE_ENDIAN;
			ge2d_config->dst_para.fill_color_en = 0;
			ge2d_config->dst_para.fill_mode = 0;
			ge2d_config->dst_para.x_rev = 0;
			ge2d_config->dst_para.y_rev = 0;
			ge2d_config->dst_para.color = 0;
			ge2d_config->dst_para.top = 0;
			ge2d_config->dst_para.left = 0;
			ge2d_config->dst_para.width = output_width;
			ge2d_config->dst_para.height = output_height;
			ge2d_config->dst_xy_swap = 0;
			ge2d_config->mem_sec = capture_secure_mode;

			vcapture_print(PRINT_CAPTUREINFO,
				"thumbnail dst addr: %lx, canvas 0x%x, width: %d, height: %d, stride: %d, format: %x\n",
				ge2d_config->dst_planes[0].addr,
				ge2d_config->dst_para.canvas_index,
				output_width, output_height,
				ge2d_config->dst_planes[0].w,
				get_output_format(output->v4l2_format));

			if (ge2d_context_config_ex(context, ge2d_config) < 0) {
				output->thumbnail_vbuf = NULL;
				vcapture_print(PRINT_ERROR,
					"++ge2d thumbnail configing error.\n");
			} else {
				do_gettimeofday(&time_begin);
				stretchblt_noalpha(context, 0, 0, input_width, input_height,
						0, 0, output_width, output_height);

				// call TEE API here
				vf->thumbnail_width = 0;
				vf->thumbnail_height = 0;
				vf->thumbnail = NULL;
				vf->histogram = NULL;
				if (vf->vf_ext) {
					src_vf = (struct vframe_s *)vf->vf_ext;
					src_vf->thumbnail_width = 0;
					src_vf->thumbnail_height = 0;
					src_vf->thumbnail = NULL;
					src_vf->histogram = NULL;
				}
#ifdef USE_TEE_API
				if (capture_debug & PRINT_TEEAPI)
					vcapture_print(PRINT_TEEAPI,
					"%s line:%d, thumbnail_scale_down=0x%x\n",
					__func__, __LINE__, thumbnail_scale_down);
				w = (thumbnail_scale_down >> 16) & 0xffff;
				h = thumbnail_scale_down & 0xffff; 
				ret = tee_get_video_data(output_width, output_height,
					output->thumbnail_width,
					w, h, &data_pa);
				if (ret == 0) {
					if (pfn_valid(__phys_to_pfn(data_pa))) {
						data_va = (void __iomem * )__phys_to_virt(data_pa);
					} else {
						data_va = ioremap_nocache(data_pa,
						w * h *(HIST_BIN +1));
					}
					if (!data_va) {
						pr_err("map video data failed\n");
					} else {
						memcpy(databuf, data_va, w*h*9);

						memcpy(thumbnail[dev->thumbnail_index],
						&databuf[0], w * h);

						memcpy(histogram[dev->thumbnail_index],
						&databuf[w*h], w * h * HIST_BIN);

						vf->thumbnail_width = w;
						vf->thumbnail_height = h;
						vf->thumbnail = thumbnail[dev->thumbnail_index];
						vf->histogram = histogram[dev->thumbnail_index];
						if (capture_debug & PRINT_TEEAPI)
							vcapture_print(PRINT_TEEAPI,
							"%s line:%d,w=%d, h=%d, nail_idx=%d\n",
							__func__, __LINE__,
							vf->thumbnail_width,
							vf->thumbnail_height,
							dev->thumbnail_index);
						if (vf->vf_ext) {
							src_vf = (struct vframe_s *)vf->vf_ext;
							src_vf->thumbnail_width = vf->thumbnail_width;
							src_vf->thumbnail_height = vf->thumbnail_height;
							src_vf->thumbnail = vf->thumbnail;
							src_vf->histogram = vf->histogram;
						}
					}
				} else {
					if (capture_debug & PRINT_TEEAPI)
						vcapture_print(PRINT_TEEAPI,
						"%s tee_get_video_data ret=%d\n", __func__, ret);
				}
#else
				canvas_read(output_canvas & 0xff, &cd);
				buf_va = codec_mm_vmap(cd.addr, PAGE_ALIGN(THUMNAIL_MAX_SIZE));
				if (!buf_va) {
					vcapture_print(PRINT_ERROR, "vmap failed for thumbnail\n");
					return -1;
				}
				codec_mm_dma_flush(buf_va, THUMNAIL_MAX_SIZE, DMA_FROM_DEVICE);
				vf->thumbnail_width = (thumbnail_scale_down >> 16) & 0xffff; // 10
				vf->thumbnail_height = thumbnail_scale_down & 0xffff; // 16
				vf->thumbnail = thumbnail[dev->thumbnail_index];
				vf->histogram = histogram[dev->thumbnail_index];
				if (vf->vf_ext) {
					src_vf = (struct vframe_s *)vf->vf_ext;
					src_vf->thumbnail_width = vf->thumbnail_width;
					src_vf->thumbnail_height = vf->thumbnail_height;
					src_vf->thumbnail = vf->thumbnail;
					src_vf->histogram = vf->histogram;
				}
				w = output_width / vf->thumbnail_width; // 160 / 10 = 16
				h = output_height / vf->thumbnail_height; // 96 / 16 = 6
				pdst = thumbnail[dev->thumbnail_index];
				phist = histogram[dev->thumbnail_index][0];
				for (i = 0; i < vf->thumbnail_height; i++) { // 0..15
					for (j = 0; j < vf->thumbnail_width; j++) { // 0..9
						psrc = buf_va + i * output->thumbnail_width * h + j * w; // i*160*6 + j*16
						sum = 0;
						for (k = 0; k < HIST_BIN; k++)
							cnt[k] = 0;
						for (k = 0; k < h; k++) { // 0..5
							for (l = 0; l < w; l++) { // 0..15
								//sum += psrc[l];
								//cnt[psrc[l] >> 5]++; // 8 bin
								pix = 255 * (psrc[l] - 16) / 219;
								sum += pix;
								cnt[pix >> 5]++; // 8 bin
							}
							psrc += output->thumbnail_width; // 160
						}
						for (k = 0; k < HIST_BIN; k++)
							phist[k] = cnt[k] * 256 / (w * h + 1) ;
						phist += HIST_BIN;
						*pdst++ = sum / (w * h);
					}
				}
				codec_mm_unmap_phyaddr(buf_va);
#endif
				do_gettimeofday(&time_end);
				cost_time = (time_end.tv_sec * 1000 * 1000 + time_end.tv_usec
						- time_begin.tv_sec * 1000 * 1000
						- time_begin.tv_usec) / 1000;
				vcapture_print(PRINT_TIMESTAMP,
						"%s line: %d thumbnail ge2d done, cost time: %d ms\n", __func__, __LINE__, cost_time);
				if ((vf->thumbnail) && (capture_debug & PRINT_THUMBNAIL)) {
					pdst = vf->thumbnail;
					vcapture_print(PRINT_THUMBNAIL, "thumbnail %p %dx%d\n",
						vf->thumbnail, vf->thumbnail_width, vf->thumbnail_height);
					for (i = 0; i < vf->thumbnail_height * vf->thumbnail_width; i += 10) {
						vcapture_print(PRINT_THUMBNAIL,
							"\t%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
							pdst[0], pdst[1], pdst[2], pdst[3], pdst[4],
							pdst[5], pdst[6], pdst[7], pdst[8], pdst[9]);
						pdst += 10;
					}
					if (vf->histogram) {
						vcapture_print(PRINT_THUMBNAIL, "histogram %p %dx%d*%d\n",
							vf->histogram, vf->thumbnail_width, vf->thumbnail_height, HIST_BIN);
						pdst = vf->histogram[0];
						for (i = 0; i < vf->thumbnail_height; i++) {
							for (j = 0; j < vf->thumbnail_width; j++) {
								vcapture_print(PRINT_THUMBNAIL,
									"\thist[%d][%d] = %02x %02x %02x %02x %02x %02x %02x %02x\n",
									i, j, pdst[0], pdst[1], pdst[2], pdst[3], pdst[4], pdst[5], pdst[6], pdst[7]);
								pdst += 8;
							}
						}
						pdst = vf->histogram[0];
						vcapture_print(PRINT_THUMBNAIL, "\t|--------|--------|--------|--------|--------|--------|--------|--------|--------|--------|\n");
						for (i = 0; i < vf->thumbnail_height; i++) {
							for (j = 0; j < HIST_BIN; j++) {
								vcapture_print(PRINT_THUMBNAIL,
									"\t|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|%s%s%s%s%s%s%s%s|\n",
									pdst[0] > lvl[j] ? "*" : " ", pdst[1] > lvl[j] ? "*" : " ", pdst[2] > lvl[j] ? "*" : " ", pdst[3] > lvl[j] ? "*" : " ",
									pdst[4] > lvl[j] ? "*" : " ", pdst[5] > lvl[j] ? "*" : " ", pdst[6] > lvl[j] ? "*" : " ", pdst[7] > lvl[j] ? "*" : " ",
									pdst[8] > lvl[j] ? "*" : " ", pdst[9] > lvl[j] ? "*" : " ", pdst[10] > lvl[j] ? "*" : " ", pdst[11] > lvl[j] ? "*" : " ",
									pdst[12] > lvl[j] ? "*" : " ", pdst[13] > lvl[j] ? "*" : " ", pdst[14] > lvl[j] ? "*" : " ", pdst[15] > lvl[j] ? "*" : " ",
									pdst[16] > lvl[j] ? "*" : " ", pdst[17] > lvl[j] ? "*" : " ", pdst[18] > lvl[j] ? "*" : " ", pdst[19] > lvl[j] ? "*" : " ",
									pdst[20] > lvl[j] ? "*" : " ", pdst[21] > lvl[j] ? "*" : " ", pdst[22] > lvl[j] ? "*" : " ", pdst[23] > lvl[j] ? "*" : " ",
									pdst[24] > lvl[j] ? "*" : " ", pdst[25] > lvl[j] ? "*" : " ", pdst[26] > lvl[j] ? "*" : " ", pdst[27] > lvl[j] ? "*" : " ",
									pdst[28] > lvl[j] ? "*" : " ", pdst[29] > lvl[j] ? "*" : " ", pdst[30] > lvl[j] ? "*" : " ", pdst[31] > lvl[j] ? "*" : " ",
									pdst[32] > lvl[j] ? "*" : " ", pdst[33] > lvl[j] ? "*" : " ", pdst[34] > lvl[j] ? "*" : " ", pdst[35] > lvl[j] ? "*" : " ",
									pdst[36] > lvl[j] ? "*" : " ", pdst[37] > lvl[j] ? "*" : " ", pdst[38] > lvl[j] ? "*" : " ", pdst[39] > lvl[j] ? "*" : " ",
									pdst[40] > lvl[j] ? "*" : " ", pdst[41] > lvl[j] ? "*" : " ", pdst[42] > lvl[j] ? "*" : " ", pdst[43] > lvl[j] ? "*" : " ",
									pdst[44] > lvl[j] ? "*" : " ", pdst[45] > lvl[j] ? "*" : " ", pdst[46] > lvl[j] ? "*" : " ", pdst[47] > lvl[j] ? "*" : " ",
									pdst[48] > lvl[j] ? "*" : " ", pdst[49] > lvl[j] ? "*" : " ", pdst[50] > lvl[j] ? "*" : " ", pdst[51] > lvl[j] ? "*" : " ",
									pdst[52] > lvl[j] ? "*" : " ", pdst[53] > lvl[j] ? "*" : " ", pdst[54] > lvl[j] ? "*" : " ", pdst[55] > lvl[j] ? "*" : " ",
									pdst[56] > lvl[j] ? "*" : " ", pdst[57] > lvl[j] ? "*" : " ", pdst[58] > lvl[j] ? "*" : " ", pdst[59] > lvl[j] ? "*" : " ",
									pdst[60] > lvl[j] ? "*" : " ", pdst[61] > lvl[j] ? "*" : " ", pdst[62] > lvl[j] ? "*" : " ", pdst[63] > lvl[j] ? "*" : " ",
									pdst[64] > lvl[j] ? "*" : " ", pdst[65] > lvl[j] ? "*" : " ", pdst[66] > lvl[j] ? "*" : " ", pdst[67] > lvl[j] ? "*" : " ",
									pdst[68] > lvl[j] ? "*" : " ", pdst[69] > lvl[j] ? "*" : " ", pdst[70] > lvl[j] ? "*" : " ", pdst[71] > lvl[j] ? "*" : " ",
									pdst[72] > lvl[j] ? "*" : " ", pdst[73] > lvl[j] ? "*" : " ", pdst[74] > lvl[j] ? "*" : " ", pdst[75] > lvl[j] ? "*" : " ",
									pdst[76] > lvl[j] ? "*" : " ", pdst[77] > lvl[j] ? "*" : " ", pdst[78] > lvl[j] ? "*" : " ", pdst[79] > lvl[j] ? "*" : " ");
							}
							vcapture_print(PRINT_THUMBNAIL, "\t|--------|--------|--------|--------|--------|--------|--------|--------|--------|--------|\n");
							pdst += HIST_BIN * vf->thumbnail_width;
						}
					}
					/* only dump once */
					capture_debug &= ~PRINT_THUMBNAIL;
				}
				if (vf->thumbnail) {
					ldim_get_vf(vf);
					dev->vf_thumbnail[dev->thumbnail_index] = vf;
					dev->thumbnail_index++;
					if (dev->thumbnail_index == MAX_VBUF)
						dev->thumbnail_index = 0;
				}
			}
		} else {
			if (capture_debug & PRINT_TEEAPI)
				vcapture_print(PRINT_TEEAPI,
							"%s output->thumbnail_vbuf = null\n",
							__func__);
		}
	}

	mutex_lock(&dev->vf_mutex);
	if (atomic_read(&dev->vcap_status) == VCAPTURE_DROP_VF) {
		mutex_unlock(&dev->vf_mutex);
		atomic_set(&dev->vcap_status, VCAPTURE_INIT);
		vcapture_print(PRINT_ERROR,
			"%s line: %d\n", __func__, __LINE__);
		return -2;
	}

	if (vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE)
		vf->flag &= ~VFRAME_FLAG_VIDEO_VCAPTURE;

	if (vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_PUT) {
		vf->flag &= ~VFRAME_FLAG_VIDEO_VCAPTURE_PUT;
		vf_put(vf, dev->recv_name);
	}
	mutex_unlock(&dev->vf_mutex);
	return 0;
}

static int vcapture_fill_buffer(struct vcapture_dev *dev)
{
	struct vcapture_buffer *buf = NULL;
	struct config_para_ex_s ge2d_config;
	struct vcapture_output output;
	struct vcapture_dmaqueue *dma_q = &dev->dma_q;
	int ret = 0;
	void *vbuf = NULL;
	unsigned long flags = 0;
	mm_segment_t old_fs;
	struct file *filp_scr = NULL;
	loff_t pos = 0;
	struct canvas_s cd;
	int result = 0;
	ulong phys;
	u32 width, height;
	struct vframe_s *vf;
	struct vframe_s *src_vf;
	char fname[32];
	int capture_type = 0;

	if (dev->ge2d_vf == dev->new_vf) {
		vcapture_print(PRINT_CAPTUREINFO,
			"the same frame!\n");
		return -1;
	}

	if (IS_ERR_OR_NULL(dev->new_vf)) {
		vcapture_print(PRINT_ERROR,
			"%s vf is NULL!\n", __func__);
		return -1;
	}

	vf = dev->new_vf;
	vf->thumbnail = NULL;
	vf->histogram = NULL;
	if (atomic_read(&dev->is_capturing) &&
	(vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_FRAME ||
	vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_LINE))
		capture_type = 1;
	if (atomic_read(&dev->is_capturing_thumbnail) &&
	(vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_THUMBNAIL))
		capture_type |= 2;
	if (!capture_type)
		return -1;

	memset(&output, 0, sizeof(struct vcapture_output));
	if (capture_type & 1) {
		spin_lock_irqsave(&dev->slock, flags);

		if (list_empty(&dma_q->active)) {
			spin_unlock_irqrestore(&dev->slock, flags);
			vcapture_print(PRINT_BUFFERINFO,
				"dma queue is empty!\n");
			return -1;
		}

		buf = list_entry(dma_q->active.next, struct vcapture_buffer, vb.queue);
		buf->vb.state = VIDEOBUF_ACTIVE;
		list_del(&buf->vb.queue);
		spin_unlock_irqrestore(&dev->slock, flags);

		vbuf = (void *)videobuf_to_res(&buf->vb);
		if (!vbuf) {
			vcapture_print(PRINT_ERROR, "vbuf is invalue!\n");
			goto fail;
		}
		output.v4l2_format = dev->fmt->fourcc;
		output.vbuf = vbuf;
		output.width = buf->vb.width;
		output.height = buf->vb.height;
	}
	if (capture_type & 2) {
		output.thumbnail_width = dev->thumbnail_width;
		output.thumbnail_height = dev->thumbnail_height;
	}

	mutex_lock(&dev->vf_mutex);
	if (atomic_read(&dev->vcap_status) == VCAPTURE_DROP_VF) {
		mutex_unlock(&dev->vf_mutex);
		atomic_set(&dev->vcap_status, VCAPTURE_INIT);
		vcapture_print(PRINT_CAP_OTHER,
			"%s line: %d drop\n", __func__, __LINE__);
		goto fail;
	}

	atomic_set(&dev->vcap_status, VCAPTURE_PREPARE);
	dev->new_vf->flag |= VFRAME_FLAG_VIDEO_VCAPTURE;
	dev->ge2d_vf = dev->new_vf;
	atomic_set(&dev->vcap_status, VCAPTURE_INIT);
	mutex_unlock(&dev->vf_mutex);

	ret = vcapture_ge2d_process(&ge2d_config, &output, dev);
	if (ret < 0) {
		if ((ret == -2) || atomic_read(&dev->is_playing) == 0) {
			dev->ge2d_vf = NULL;
			vcapture_print(PRINT_ERROR,
				"%s exit\n", __func__);
			goto fail;
		}
		mutex_lock(&dev->vf_mutex);
		if (atomic_read(&dev->vcap_status) == VCAPTURE_DROP_VF) {
			mutex_unlock(&dev->vf_mutex);
			dev->ge2d_vf = NULL;
			atomic_set(&dev->vcap_status, VCAPTURE_INIT);
			vcapture_print(PRINT_ERROR,
				"%s line: %d drop\n", __func__, __LINE__);
			goto fail;
		}

		if (dev->ge2d_vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE)
			dev->ge2d_vf->flag &= ~VFRAME_FLAG_VIDEO_VCAPTURE;
		if (dev->ge2d_vf->flag & VFRAME_FLAG_VIDEO_VCAPTURE_PUT) {
			dev->ge2d_vf->flag &= ~VFRAME_FLAG_VIDEO_VCAPTURE_PUT;
			vf_put(dev->ge2d_vf, dev->recv_name);
		}
		mutex_unlock(&dev->vf_mutex);
		dev->ge2d_vf = NULL;
		vcapture_print(PRINT_ERROR, "ge2d deal failed!\n");
		goto fail;
	}
	if (capture_type & 1)
		buf->vb.state = VIDEOBUF_DONE;

	if ((capture_debug & DUMP_SRC) || (capture_debug & DUMP_DST) || (capture_debug & DUMP_THUMBNAIL)) {
		old_fs = get_fs();
		set_fs(KERNEL_DS);
		if (capture_debug & DUMP_SRC) {
			src_vf = vcapture_src_vf(dev->ge2d_vf);
			sprintf(fname, "/nvram/src%d.bin", dump_counter);
			filp_scr = filp_open(fname, O_RDWR | O_CREAT, 0666);
			if (IS_ERR(filp_scr)) {
				vcapture_print(PRINT_ERROR, "dump open /nvram/src%d.bin failed\n", dump_counter);
			} else {
				if (src_vf->canvas0Addr == (u32)-1) {
					phys = src_vf->canvas0_config[0].phy_addr;
					width = src_vf->canvas0_config[0].width;
					height = src_vf->canvas0_config[0].height;
					vcapture_print(DUMP_SRC,
						"dump src use canvas config 0x%x\n", phys);
				} else {
					canvas_read(src_vf->canvas0Addr & 0xff, &cd);
					phys = cd.addr;
					width = cd.width;
					height = cd.height;
					vcapture_print(DUMP_SRC,
						"dump src use canvas 0x%x\n",
						vf->canvas0Addr);
				}
				vcapture_print(DUMP_SRC, "dump scr addr: %0lx, width: %d, height: %d, fmt: %x %x\n",
					phys, width, height, get_source_type(vf), get_input_format(vf));
				result = copy_phybuf_to_file(phys, width * height, filp_scr, 0);
				if (result < 0)
					vcapture_print(PRINT_ERROR, "dump write src%d.bin failed\n", dump_counter);
				vfs_fsync(filp_scr, 0);
				filp_close(filp_scr, NULL);
			}
		}
		if (capture_debug & DUMP_DST) {
			sprintf(fname, "/nvram/dst%d.bin", dump_counter);
			filp_scr = filp_open(fname, O_RDWR | O_CREAT, 0666);
			if (IS_ERR(filp_scr)) {
				vcapture_print(PRINT_ERROR, "dump open /nvram/dst%d.bin failed\n", dump_counter);
			} else {
				canvas_read(dev->dst_canvas[0] & 0xff, &cd);
				phys = cd.addr;
				width = cd.width;
				height = cd.height;
				vcapture_print(DUMP_DST, "dump dst addr: %0lx, width: %d, height: %d, fmt: %x\n",
					phys, width, height, get_output_format(output.v4l2_format));
				result = copy_phybuf_to_file(phys, width * height * 3 / 2, filp_scr, 0);
				if (result < 0)
					vcapture_print(PRINT_ERROR, "dump write dst%d.bin failed\n", dump_counter);
				vfs_fsync(filp_scr, 0);
				filp_close(filp_scr, NULL);
			}
		}
		if (capture_debug & DUMP_THUMBNAIL) {
			sprintf(fname, "/nvram/thumb%d.bin", dump_counter);
			filp_scr = filp_open(fname, O_RDWR | O_CREAT, 0666);
			if (IS_ERR(filp_scr)) {
				vcapture_print(PRINT_ERROR, "dump open /nvram/thumb%d.bin failed\n", dump_counter);
			} else {
				canvas_read(dev->thumbnail_canvas[0] & 0xff, &cd);
				phys = cd.addr;
				width = cd.width;
				height = cd.height;
				vcapture_print(DUMP_DST, "dump thumb addr: %0lx, width: %d, height: %d, fmt: %x\n",
					phys, width, height, get_output_format(output.thumbnail_v4l2_format));
				result = copy_phybuf_to_file(phys, width * height, filp_scr, 0);
				if (result < 0)
					vcapture_print(PRINT_ERROR, "dump write thumb%d.bin failed\n", dump_counter);
				vfs_fsync(filp_scr, 0);
				filp_close(filp_scr, NULL);
			}
		}
		if ((capture_debug & DUMP_THUMBNAIL_AVG) && vf->thumbnail) {
			sprintf(fname, "/nvram/avg_%d.bin", dump_counter);
			filp_scr = filp_open(fname, O_RDWR | O_CREAT, 0666);
			if (IS_ERR(filp_scr)) {
				vcapture_print(PRINT_ERROR, "dump open /nvram/avg%d.bin failed\n", dump_counter);
			} else {
				width = vf->thumbnail_width;
				height = vf->thumbnail_height;
				vcapture_print(DUMP_DST, "dump avg addr: %0lx, width: %d, height: %d, fmt: %x\n",
					vf->thumbnail, width, height, get_output_format(output.thumbnail_v4l2_format));
				result = vfs_write(filp_scr, vf->thumbnail, width * height, &pos);
				if (result <= 0)
					vcapture_print(PRINT_ERROR, "dump write avg%d.bin failed\n", dump_counter);
				vfs_fsync(filp_scr, 0);
				filp_close(filp_scr, NULL);
			}
		}
		set_fs(old_fs);
		dump_counter++;
	} else
		dump_counter = 0;

	/*wait up vb done buffer workqueue*/
	if (capture_type & 1) {
		if (waitqueue_active(&buf->vb.done))
			wake_up(&buf->vb.done);
	}
	return ret;
fail:
	if (capture_type & 1) {
		spin_lock_irqsave(&dev->slock, flags);
		buf->vb.state = VIDEOBUF_QUEUED;
		list_add_tail(&buf->vb.queue, &dma_q->active);
		spin_unlock_irqrestore(&dev->slock, flags);
	}
	return ret;
}

static int vcapture_thread(void *data)
{
	struct vcapture_dev *dev = data;
	struct sched_param param = {.sched_priority = MAX_RT_PRIO - 1};
	int ret = 0;

	sched_setscheduler(current, SCHED_FIFO, &param);
	allow_signal(SIGTERM);
	vcapture_print(PRINT_THREADINFO,
		      "start %s kthread.\n", dev->recv_name);

	while (down_interruptible(&dev->thread_sem) == 0) {
		if (atomic_read(&dev->is_playing) == 0) {
			vcapture_print(PRINT_CAP_OTHER,
				      "%s has no vf source\n", __func__);
			continue;
		}
		if (dev->begin_time.tv_sec == 0 &&
		    dev->begin_time.tv_usec == 0)
			do_gettimeofday(&dev->begin_time);
		if (kthread_should_stop()) {
			vcapture_print(PRINT_THREADINFO,
				      "stop %s kthread before capture.\n", dev->recv_name);
			break;
		}

#ifdef USE_SEMA_QBUF
		if (atomic_read(&dev->is_capturing))
			ret = wake_up_interruptible(&dev->dma_q.qbuf_comp);
#endif
		vcapture_print(PRINT_THREADINFO,
			"task %s running\n",
			dev->dma_q.task_running ? "is" : "not");
		if (!dev->dma_q.task_running)
			break;
		if (dev->new_vf)
			vcapture_fill_buffer(dev);
		if (kthread_should_stop()) {
			vcapture_print(PRINT_THREADINFO,
				"stop %s kthread after capture.\n", dev->recv_name);
			break;
		}
	}
	while (!kthread_should_stop()) {
		vcapture_print(PRINT_THREADINFO,
			"%s kthread wait stop.\n", dev->recv_name);
		usleep_range(9000, 10000);
	}
	vcapture_print(PRINT_THREADINFO,
		"exit %s kthread.\n", dev->recv_name);
	return ret;
}

/* ------------------------------------------------------------------
 * Videobuf operations
 * ------------------------------------------------------------------
 */
static int buffer_setup(struct videobuf_queue *vq, unsigned int *count,
			unsigned int *size)
{
	struct videobuf_res_privdata *res =
		(struct videobuf_res_privdata *)vq->priv_data;
	struct vcapture_dev *dev = (struct vcapture_dev *)res->priv;
	int height = dev->height;
	int i;

	if (height % 16 != 0)
		height = ((height + 15) >> 4) << 4;

	*size = (dev->width * height * dev->fmt->depth) >> 3;
	if (*count == 0)
		*count = 32;

	while (*size * *count + THUMNAIL_MAX_SIZE * MAX_VBUF > CMA_ALLOC_SIZE * 1024 * 1024 || *count > MAX_VBUF)
		(*count)--;
	dev->vbuf_count = *count;
	vcapture_print(PRINT_BUFFERINFO,
		"%s line: %d count=%d, size=%d %dx%d\n",
		__func__,  __LINE__, *count, *size, dev->width, height);
	for (i = 0; i < *count; i++) {
		dev->vbuf_addr[i] = (void *)(dev->res.start + *size * i);
		dev->vbuf_fmt[i] = ((dev->width >> 5) << 24) | (dev->width << 12) | height;
		dev->vbuf_field[i] = (dev->width << 12) | height;
		vcapture_print(PRINT_BUFFERINFO,
			"%s line: %d vbuf[%d] addr=0x%lx, fmt=0x%x, size=%d %dx%d\n",
			__func__,  __LINE__, i, dev->vbuf_addr[i], dev->vbuf_fmt[i], *size, dev->width, height);
	}
	return 0;
}

static void free_buffer(struct videobuf_queue *vq, struct vcapture_buffer *buf)
{
	vcapture_print(PRINT_BUFFERINFO,
		"%s line: %d state:%i\n", __func__, __LINE__, buf->vb.state);
	videobuf_waiton(vq, &buf->vb, 0, 0);
	if (in_interrupt())
		WARN_ON(1);
	videobuf_res_free(vq, &buf->vb);
	buf->vb.state = VIDEOBUF_NEEDS_INIT;
}

#define norm_maxw() 1920
#define norm_maxh() 1080

static int buffer_prepare(struct videobuf_queue *vq,
			  struct videobuf_buffer *videobuf,
			  enum v4l2_field field)
{
	struct videobuf_res_privdata *res =
		(struct videobuf_res_privdata *)vq->priv_data;
	struct vcapture_dev *dev = (struct vcapture_dev *)res->priv;
	struct vcapture_buffer *buf =
		container_of(videobuf, struct vcapture_buffer, vb);
	void *vbuf;
	int rc;

	vcapture_print(PRINT_BUFFERINFO,
		      "%s field: %d\n", __func__, field);
	if (dev->width < 16 || dev->width > norm_maxw() ||
	    dev->height < 16 || dev->height > norm_maxh()) {
		vcapture_print(PRINT_ERROR,
			      "%s line: %d\n", __func__, __LINE__);
		return -EINVAL;
	}
	buf->vb.size = (dev->width * dev->height * dev->fmt->depth) >> 3;
	if (buf->vb.baddr != 0 && buf->vb.bsize < buf->vb.size) {
		vcapture_print(PRINT_ERROR,
			"%s line: %d\n", __func__, __LINE__);
		return -EINVAL;
	}
	/* These properties only change when queue is idle, see s_fmt */
	buf->fmt = dev->fmt;
	buf->vb.width = dev->width;
	buf->vb.height = dev->height;
	buf->vb.field = field;
	vbuf = (void *)videobuf_to_res(&buf->vb);
	if (buf->vb.state == VIDEOBUF_NEEDS_INIT) {
		rc = videobuf_iolock(vq, &buf->vb, NULL);
		vcapture_print(PRINT_BUFFERINFO,
			"%s addr: 0x%lx, i: %d, magic:%d, size:%d", __func__,
			(unsigned long)vbuf, buf->vb.i, buf->vb.magic, buf->vb.size);
		if (rc < 0) {
			vcapture_print(PRINT_ERROR,
				"%s line: %d\n", __func__, __LINE__);
			goto fail;
		}
		dev->vbuf_addr[buf->vb.i] = vbuf;
		dev->vbuf_fmt[buf->vb.i] = ((buf->vb.width >> 5) << 24) | (buf->vb.width << 12) | buf->vb.height;
		dev->vbuf_field[buf->vb.i] = (buf->vb.width << 12) | buf->vb.height;
	}
	dev->each_buffer_size = buf->vb.size;
	buf->vb.state = VIDEOBUF_PREPARED;
	return 0;

fail: free_buffer(vq, buf);
	return rc;
}

static void buffer_queue(struct videobuf_queue *vq,
			 struct videobuf_buffer *videobuf)
{
	struct vcapture_buffer *buf =
		container_of(videobuf, struct vcapture_buffer, vb);
	struct videobuf_res_privdata *res =
		(struct videobuf_res_privdata *)vq->priv_data;
	struct vcapture_dev *dev = (struct vcapture_dev *)res->priv;
	struct vcapture_dmaqueue *dma_q = &dev->dma_q;

	vcapture_print(PRINT_BUFFERINFO,
		      "%s line: %d\n", __func__, __LINE__);

	buf->vb.state = VIDEOBUF_QUEUED;
	list_add_tail(&buf->vb.queue, &dma_q->active);
}

static void buffer_release(struct videobuf_queue *vq,
			   struct videobuf_buffer *videobuf)
{
	struct vcapture_buffer *buf =
		container_of(videobuf, struct vcapture_buffer, vb);

	vcapture_print(PRINT_BUFFERINFO,
		      "%s line: %d\n", __func__, __LINE__);

	free_buffer(vq, buf);
}

static struct videobuf_queue_ops vcapture_qops = {
	.buf_setup = buffer_setup,
	.buf_prepare = buffer_prepare,
	.buf_queue = buffer_queue,
	.buf_release = buffer_release,
};

static int vcapture_event_cb(int type, void *data, void *private_data)
{
	if (type & VFRAME_EVENT_RECEIVER_PUT)
		;
	else if (type & VFRAME_EVENT_RECEIVER_GET)
		;
	else if (type & VFRAME_EVENT_RECEIVER_FRAME_WAIT)
		vcapture_print(PRINT_PATHINFO,
			      "receiver is waiting\n");
	else if (type & VFRAME_EVENT_RECEIVER_FRAME_WAIT)
		vcapture_print(PRINT_PATHINFO,
			      "frame wait\n");
	return 0;
}

static int vcapture_vf_states(struct vframe_states *states, void *op_arg)
{
	states->vf_pool_size = 0;
	states->buf_recycle_num = 0;
	states->buf_free_num = 0;
	states->buf_avail_num = 0;
	/* spin_unlock_irqrestore(&lock, flags); */
	return 0;
}

static const struct vframe_operations_s vcapture_vf_prov = {
		.peek = vcapture_vf_peek,
		.get = vcapture_vf_get,
		.put = vcapture_vf_put,
		.event_cb = vcapture_event_cb,
		.vf_states = vcapture_vf_states,
};

static int video_receiver_event_fun(int type, void *data, void *private_data)
{
	struct vcapture_dev *dev = (struct vcapture_dev *)private_data;
	struct vframe_states states;

	switch (type) {
	case VFRAME_EVENT_PROVIDER_VFRAME_READY:
		vf_notify_receiver(dev->prov_name, type, NULL);
		vcapture_print(PRINT_PATHINFO, "ready\n");
		break;
	case VFRAME_EVENT_PROVIDER_QUREY_STATE:
		vcapture_print(PRINT_PATHINFO, "state\n");
		vcapture_vf_states(&states, dev);
		if (states.buf_avail_num >= 0)
			return RECEIVER_ACTIVE;
		if (vf_notify_receiver(dev->prov_name, type, NULL)
				       == RECEIVER_ACTIVE) {
			return RECEIVER_ACTIVE;
		}
		return RECEIVER_INACTIVE;
	case VFRAME_EVENT_PROVIDER_START:
		vf_notify_receiver(dev->prov_name, type, NULL);
		vcapture_print(PRINT_PATHINFO, "start\n");
		break;
	case VFRAME_EVENT_PROVIDER_REG:
		vf_reg_provider(&dev->prov);
		vcapture_print(PRINT_PATHINFO, "reg\n");
		break;
	case VFRAME_EVENT_PROVIDER_UNREG:
		mutex_lock(&dev->vf_mutex);
		atomic_set(&dev->vcap_status, VCAPTURE_DROP_VF);
		mutex_unlock(&dev->vf_mutex);
		atomic_set(&dev->is_playing, 0);
		vf_unreg_provider(&dev->prov);
		vcapture_print(PRINT_PATHINFO, "unreg\n");
		break;
	case VFRAME_EVENT_PROVIDER_LIGHT_UNREG:
		mutex_lock(&dev->vf_mutex);
		atomic_set(&dev->vcap_status, VCAPTURE_DROP_VF);
		mutex_unlock(&dev->vf_mutex);
		atomic_set(&dev->is_playing, 0);
		vf_light_unreg_provider(&dev->prov);
		vcapture_print(PRINT_PATHINFO, "light unreg\n");
		break;
	case VFRAME_EVENT_PROVIDER_RESET:
		mutex_lock(&dev->vf_mutex);
		atomic_set(&dev->vcap_status, VCAPTURE_DROP_VF);
		mutex_unlock(&dev->vf_mutex);
		atomic_set(&dev->is_playing, 0);
		vcapture_print(PRINT_PATHINFO, "reset\n");
		vf_notify_receiver(dev->prov_name, type, NULL);
		break;
	case VFRAME_EVENT_PROVIDER_FR_HINT:
	case VFRAME_EVENT_PROVIDER_FR_END_HINT:
		vf_notify_receiver(dev->prov_name, type, data);
		break;
	default:
		break;
	}
	return 0;
}

static const struct vframe_receiver_op_s vcapture_vf_recv = {
	.event_cb = video_receiver_event_fun
};

/* ------------------------------------------------------------------
 * v4l2 operations
 * ------------------------------------------------------------------
 */
static int vidioc_querycap(struct file *file, void *priv,
			   struct v4l2_capability *cap)
{
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);

	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);

	strcpy(cap->driver, "vcapture");
	strcpy(cap->card, "vcapture");
	strlcpy(cap->bus_info,
		dev->v4l2_dev.name,
		sizeof(cap->bus_info));
	cap->version = VCAPTURE_VERSION;
	cap->device_caps = V4L2_CAP_VIDEO_CAPTURE;
	cap->capabilities = cap->device_caps | V4L2_CAP_DEVICE_CAPS
			| V4L2_CAP_STREAMING | V4L2_CAP_READWRITE;
	return 0;
}

static int vidioc_reqbufs(struct file *file, void *priv,
			  struct v4l2_requestbuffers *p)
{
	struct vcapture_dev *dev = NULL;
#ifdef USE_SEMA_QBUF
	struct vcapture_dmaqueue *dma_q = NULL;
#endif
	dev = video_drvdata(file);
#ifdef USE_SEMA_QBUF
	dma_q = &dev->dma_q;
	init_completion(&dma_q->qbuf_comp);
#endif
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	return videobuf_reqbufs(&dev->vbuf_queue, p);
}

static int vidioc_querybuf(struct file *file, void *priv, struct v4l2_buffer *p)
{
	struct vcapture_dev *dev = NULL;
	int ret = 0;

	dev = video_drvdata(file);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);

	ret = videobuf_querybuf(&dev->vbuf_queue, p);

	dev->cur_vbuf_index = p->index;
	if (ret == 0) {
		p->reserved = dev->vbuf_fmt[p->index];
		p->reserved2 = (unsigned int)dev->vbuf_addr[p->index];
		p->field = dev->vbuf_field[p->index];
	} else {
		p->reserved = 0;
		p->reserved2 = 0;
	}

	return ret;
}

static int vidioc_qbuf(struct file *file, void *priv, struct v4l2_buffer *p)
{
	struct vcapture_dev *dev = NULL;
	#ifdef USE_SEMA_QBUF
	struct vcapture_dmaqueue *dma_q = NULL;
	#endif
	int ret = 0;

	dev = video_drvdata(file);

#ifdef USE_SEMA_QBUF
	dma_q = &dev->dma_q;
	complete(&dma_q->qbuf_comp);
#endif
	if (capture_thread) {
		dev->begin_time.tv_sec = 0;
		dev->begin_time.tv_usec = 0;
		if (atomic_read(&dev->is_capturing) &&
		(dev->ge2d_vf != dev->new_vf))
			up(&dev->thread_sem);
	}
	ret = videobuf_qbuf(&dev->vbuf_queue, p);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s ret: %d line: %d\n", __func__, ret, __LINE__);
	return ret;
}

static int vidioc_dqbuf(struct file *file, void *priv, struct v4l2_buffer *p)
{
	struct vcapture_dev *dev = video_drvdata(file);
	struct timeval time_begin;
	struct timeval time_end;
	int use_time;
	int ret;

	if (atomic_read(&dev->is_playing) == 0) {
		vcapture_print(PRINT_CAP_OTHER,
			      "%s has no vf source\n", __func__);
		return -ENOMEM;
	}

	if (!capture_thread) {
		do_gettimeofday(&time_begin);
		vcapture_fill_buffer(dev);
		do_gettimeofday(&time_end);
		use_time = (time_end.tv_sec * 1000 * 1000 + time_end.tv_usec
			    - time_begin.tv_sec * 1000 * 1000
			    - time_begin.tv_usec) / 1000;
		vcapture_print(PRINT_TIMESTAMP,
			      "%s block mode all cost time: %d ms line: %d\n",
			      __func__, use_time, __LINE__);
	} else {
		atomic_set(&dev->is_capturing, 1);
	}

	if (!capture_secure_mode) {
		flush_cache_range(dev->vma[p->index],
			dev->vma[p->index]->vm_start, dev->vma[p->index]->vm_end);
	}
	ret = videobuf_dqbuf(&dev->vbuf_queue, p, file->f_flags & O_NONBLOCK);
	if (ret >= 0) {
		p->reserved = dev->vbuf_fmt[p->index];
		p->reserved2 = (unsigned)dev->vbuf_addr[p->index];
		p->field = dev->vbuf_field[p->index];
		vcapture_print(PRINT_CAP_OTHER,
			"%s line: %d return: %d, index: %d %x %x %x %x\n",
			__func__, __LINE__, ret, p->index, p->reserved, p->reserved2,
				dev->vma[p->index]->vm_start, dev->vma[p->index]->vm_end);
	} else if (capture_thread && (dev->ge2d_vf != dev->new_vf))
			up(&dev->thread_sem);

	vcapture_print(PRINT_CAP_OTHER,
		"%s ret: %d line: %d\n", __func__, ret, __LINE__);

	return ret;
}

static int vidioc_g_fmt_vid_cap(struct file *file, void *priv,
				struct v4l2_format *f)
{
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	if (dev->set_format_flag) {
		f->fmt.pix.width = dev->width;
		f->fmt.pix.height = dev->height;
		f->fmt.pix.field = dev->vbuf_queue.field;
		f->fmt.pix.pixelformat = dev->fmt->fourcc;
		f->fmt.pix.bytesperline =
		(f->fmt.pix.width * dev->fmt->depth) >> 3;
		f->fmt.pix.sizeimage =
		f->fmt.pix.height * f->fmt.pix.bytesperline;
	} else {
		f->fmt.pix.width = 0;
		f->fmt.pix.height = 0;
	}

	return 0;
}

static int vidioc_try_fmt_vid_cap(struct file *file, void *priv,
				  struct v4l2_format *f)
{
	struct vcapture_fmt *fmt = NULL;
	struct vcapture_dev *dev = NULL;
	enum v4l2_field field;
	unsigned int maxw, maxh;

	dev = video_drvdata(file);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	fmt = get_format(f);
	if (!fmt) {
		vcapture_print(PRINT_ERROR,
			"Fourcc format (0x%08x) invalid.\n",
			f->fmt.pix.pixelformat);
		return -EINVAL;
	}

	field = f->fmt.pix.field;

	if (field == V4L2_FIELD_ANY) {
		field = V4L2_FIELD_INTERLACED;
	} else if (field != V4L2_FIELD_INTERLACED) {
		vcapture_print(PRINT_ERROR,
			"Field type invalid.\n");
		return -EINVAL;
	}

	maxw = norm_maxw();
	maxh = norm_maxh();

	f->fmt.pix.field = field;
	v4l_bound_align_image(&f->fmt.pix.width, 16, maxw, 2,
			      &f->fmt.pix.height, 16, maxh, 0, 0);
	f->fmt.pix.bytesperline = (f->fmt.pix.width * fmt->depth) >> 3;
	f->fmt.pix.sizeimage = f->fmt.pix.height * f->fmt.pix.bytesperline;
	return 0;
}

static int vidioc_s_fmt_vid_cap(struct file *file, void *priv,
				struct v4l2_format *f)
{
	int ret = 0;
	struct vcapture_dev *dev = NULL;
	struct videobuf_queue *q = NULL;

	dev = video_drvdata(file);
	q = &dev->vbuf_queue;
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	dev->dst_width = f->fmt.pix.width;
	dev->dst_height = f->fmt.pix.height;
	f->fmt.pix.width = (f->fmt.pix.width + 31) & (~31);
	// 420 need 64

	ret = vidioc_try_fmt_vid_cap(file, priv, f);
	if (ret < 0) {
		vcapture_print(PRINT_CAP_OTHER,
			      "%s line: %d\n", __func__, __LINE__);
		return ret;
	}
	mutex_lock(&q->vb_lock);

	if (videobuf_queue_is_busy(&dev->vbuf_queue)) {
		vcapture_print(PRINT_ERROR,
			"%s queue busy\n", __func__);
		ret = -EBUSY;
		goto out;
	}

	dev->fmt = get_format(f);
	dev->width = f->fmt.pix.width;
	dev->height = f->fmt.pix.height;
	dev->vbuf_queue.field = f->fmt.pix.field;
	dev->type = f->type;
	dev->set_format_flag = true;
	ret = 0;
out: mutex_unlock(&q->vb_lock);
	return ret;
}

static int vidioc_g_parm(struct file *file, void *priv,
			 struct v4l2_streamparm *parms)
{
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	return 0;
}

static int vidioc_s_parm(struct file *file, void *priv,
			 struct v4l2_streamparm *parms)
{
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);

	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	return 0;
}

static int vidioc_g_input(struct file *file, void *priv, unsigned int *i)
{
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);

	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	return 0;
}

static int vidioc_s_input(struct file *file, void *priv, unsigned int i)
{
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);

	vcapture_print(PRINT_CAP_OTHER,
		      "%s line: %d\n", __func__, __LINE__);
	return 0;
}

static int start_capture_thread(struct vcapture_dev *dev)
{
	struct vcapture_dmaqueue *dma_q = &dev->dma_q;

	if (capture_thread)
		return 0;
	if (capture_use_thread)
		capture_thread = 1;
	else
		capture_thread = 0;

	if (capture_thread) {
		mutex_lock(&dev->vcapture_mutex);
		if (dma_q->task_running) {
			mutex_unlock(&dev->vcapture_mutex);
			return 0;
		}

		dma_q->task_running = 1;
		dma_q->kthread = kthread_run(vcapture_thread,
					     dev, dev->recv_name);

		if (IS_ERR(dma_q->kthread)) {
			dma_q->task_running = 0;
			dma_q->kthread = NULL;
			mutex_unlock(&dev->vcapture_mutex);
			vcapture_print(PRINT_ERROR,
				"start thread error.....\n");
			return PTR_ERR(dma_q->kthread);
		}
		mutex_unlock(&dev->vcapture_mutex);
		dev->begin_time.tv_sec = 0;
		dev->begin_time.tv_usec = 0;
		vcapture_print(PRINT_THREADINFO,
			      "%s thread start successful!\n", __func__);
	}
	return 0;
}

static int vidioc_streamon(struct file *file, void *priv, enum v4l2_buf_type i)
{
	int ret;
	struct vcapture_dev *dev = NULL;
	struct vcapture_dmaqueue *dma_q = NULL;

	dev = video_drvdata(file);
	dma_q = &dev->dma_q;
	vcapture_print(PRINT_CAP_OTHER,
		      "%s begin\n", __func__);
	if ((dev->type != V4L2_BUF_TYPE_VIDEO_CAPTURE) || (i != dev->type))
		return -EINVAL;
	ret = videobuf_streamon(&dev->vbuf_queue);
	if (ret < 0) {
		vcapture_print(PRINT_ERROR,
			"%s ret:%d\n", __func__, ret);
		return ret;
	}
#ifdef USE_SEMA_QBUF
	init_completion(&dma_q->qbuf_comp);
#endif

	start_capture_thread(dev);

	return 0;
}

static void stop_capture_thread(struct vcapture_dev *dev)
{
	int ret;

	if (capture_thread && !atomic_read(&dev->is_capturing) &&
		!atomic_read(&dev->is_capturing_thumbnail)) {
		vcapture_print(PRINT_THREADINFO,
			      "begin to stop %s thread\n", dev->recv_name);
		mutex_lock(&dev->vcapture_mutex);
		/* shutdown control thread */
		if (!IS_ERR_OR_NULL(dev->dma_q.kthread)) {
			up(&dev->thread_sem);
			dev->dma_q.task_running = 0;
			send_sig(SIGTERM, dev->dma_q.kthread, 1);
#ifdef USE_SEMA_QBUF
			complete(&dev->dma_q.qbuf_comp);
#endif
			wake_up_interruptible(&dev->dma_q.wq);
			vcapture_print(PRINT_THREADINFO,
				      "ready to stop %s thread\n",
				      dev->recv_name);

			ret = kthread_stop(dev->dma_q.kthread);
			if (ret < 0)
				vcapture_print(PRINT_ERROR,
					      "%s, ret = %d .\n",
					      __func__, ret);

			dev->dma_q.kthread = NULL;
			dev->new_vf = NULL;
			dev->ge2d_vf = NULL;
		}
		capture_thread = 0;
		mutex_unlock(&dev->vcapture_mutex);
		vcapture_print(PRINT_THREADINFO,
			      "finish stop %s thread\n", dev->recv_name);
	}
}

static int vidioc_streamoff(struct file *file,
			    void *priv, enum v4l2_buf_type i)
{
	int ret;
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);

	if ((dev->type != V4L2_BUF_TYPE_VIDEO_CAPTURE) || (i != dev->type))
		return -EINVAL;
	ret = videobuf_streamoff(&dev->vbuf_queue);
	if (ret < 0) {
		vcapture_print(PRINT_ERROR, "%s fail\n", __func__);
		return ret;
	}

	if (IS_ERR_OR_NULL(&dev->dma_q)) {
		vcapture_print(PRINT_ERROR,
			      "%s dma_q is NULL\n", __func__);
		return -1;
	}

	atomic_set(&dev->is_capturing, 0);
	stop_capture_thread(dev);

	vcapture_print(PRINT_CAP_OTHER,
		      "%s end line %d\n", __func__, __LINE__);
	return 0;
}

static const struct v4l2_ioctl_ops vcapture_ioctl_ops = {
	.vidioc_querycap = vidioc_querycap,
	.vidioc_reqbufs = vidioc_reqbufs,
	.vidioc_querybuf = vidioc_querybuf,
	.vidioc_qbuf = vidioc_qbuf,
	.vidioc_dqbuf = vidioc_dqbuf,
	.vidioc_g_fmt_vid_cap = vidioc_g_fmt_vid_cap,
	.vidioc_try_fmt_vid_cap = vidioc_try_fmt_vid_cap,
	.vidioc_s_fmt_vid_cap = vidioc_s_fmt_vid_cap,
	.vidioc_g_parm = vidioc_g_parm,
	.vidioc_s_parm = vidioc_s_parm,
	.vidioc_s_input = vidioc_s_input,
	.vidioc_g_input = vidioc_g_input,
	.vidioc_streamon = vidioc_streamon,
	.vidioc_streamoff = vidioc_streamoff,
};


/* ------------------------------------------------------------------
 * v4l2 file operations
 * ------------------------------------------------------------------
 */
static int init_vcapture(struct vcapture_dev *dev)
{
	int ret, i; 

	if (dev->inited)
		return 0;
	ret = vcapture_cma_buf_init(dev);
	if (ret < 0) {
		vcapture_print(PRINT_ERROR,
			      "%s alloc cma buffer fail\n", __func__);
		return -ENOMEM;
	}
	dev->res.start = dev->buffer_start;
	dev->res.end = dev->buffer_start + dev->buffer_size;
	for (i = 0; i < MAX_VBUF; i++) {
		dev->thumbnail_addr[i] = (void *)(dev->res.end - THUMNAIL_MAX_SIZE * (MAX_VBUF - i));
		vcapture_print(PRINT_BUFFERINFO,
			"%s line: %d thumbnail[%d] addr=0x%lx, size=%d\n",
			__func__,  __LINE__, i, dev->thumbnail_addr[i], THUMNAIL_MAX_SIZE);
	}

	dev->context = create_ge2d_work_queue();
	if (!dev->context) {
		vcapture_cma_buf_uninit(dev);
		vcapture_print(PRINT_ERROR,
			      "%s create ge2d fail\n", __func__);
		return -ENOMEM;
	}

	dev->ge2d_vf = NULL;
	if (capture_use_thread) {
		sema_init(&dev->thread_sem, 1);
		dev->dma_q.kthread = NULL;
	}
	atomic_set(&dev->vcap_status, VCAPTURE_INIT);
	dev->dma_q.kthread = NULL;
	dev->res.priv = (void *)dev;
	dev->inited = true;
	return 0;
}

static int vidioc_open(struct file *file)
{
	struct vcapture_dev *dev = NULL;
	struct videobuf_res_privdata *res = NULL;

	dev = video_drvdata(file);
	mutex_lock(&dev->vcapture_mutex);
	if (dev->fd_num > 0) {
		vcapture_print(PRINT_ERROR,
			      "%s line %d\n", __func__, __LINE__);
		mutex_unlock(&dev->vcapture_mutex);
		return -EBUSY;
	}

	dev->fd_num++;

	if (init_vcapture(dev)) {
		dev->fd_num--;
		mutex_unlock(&dev->vcapture_mutex);
		return -ENOMEM;
	}

	/* init video dma queues */
	INIT_LIST_HEAD(&dev->dma_q.active);
	init_waitqueue_head(&dev->dma_q.wq);
	mutex_unlock(&dev->vcapture_mutex);
	file->private_data = dev;
	dev->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	dev->fmt = &formats[0];
	dev->width = 1920;
	dev->height = 1080;
	atomic_set(&dev->capture_frame_per_n, capture_frame_per_n);
	atomic_set(&dev->capture_lines_per_n, capture_lines_per_n);
	dev->set_format_flag = false;
	dev->f_flags = file->f_flags;
	res = &dev->res;

	videobuf_queue_res_init(&dev->vbuf_queue, &vcapture_qops, NULL,
				&dev->slock, dev->type, V4L2_FIELD_INTERLACED,
				sizeof(struct vcapture_buffer), (void *)res,
				NULL);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s vcapture.0\n", __func__);
	return 0;
}

static ssize_t vidioc_read(struct file *file, char __user *data,
			   size_t count, loff_t *ppos)
{
	struct vcapture_dev *dev = NULL;

	dev = video_drvdata(file);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line %d\n", __func__, __LINE__);
	if (dev->type == V4L2_BUF_TYPE_VIDEO_CAPTURE) {
		return videobuf_read_stream(&dev->vbuf_queue,
					    data, count, ppos, 0,
					    file->f_flags & O_NONBLOCK);
	}
	return 0;
}

static unsigned int vidioc_poll(struct file *file,
				struct poll_table_struct *wait)
{
	struct vcapture_dev *dev = NULL;
	struct videobuf_queue *q = &dev->vbuf_queue;

	dev = video_drvdata(file);
	vcapture_print(PRINT_CAP_OTHER,
		      "%s line %d\n", __func__, __LINE__);

	if (dev->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return POLLERR;

	return videobuf_poll_stream(file, q, wait);
}

static int uninit_vcapture(struct vcapture_dev *dev)
{
	if (!dev->inited)
		return 0;
	if (atomic_read(&dev->is_capturing))
		return 0;
	if (atomic_read(&dev->is_capturing_thumbnail))
		return 0;
	if (dev->context)
		destroy_ge2d_work_queue(dev->context);
	vcapture_cma_buf_uninit(dev);
	dev->inited = false;
	return 0;
}
static int vidioc_close(struct file *file)
{
	struct vcapture_dev *dev = NULL;
	struct vcapture_dmaqueue *dma_q = NULL;

	dev = video_drvdata(file);
	dma_q = &dev->dma_q;
	vcapture_print(PRINT_CAP_OTHER,
		      "%s vcapture.0\n", __func__);
	atomic_set(&dev->vcap_status, VCAPTURE_INIT);
	if (dma_q->kthread && atomic_read(&dev->is_capturing))
		vidioc_streamoff(file, (void *)dev, dev->type);
	videobuf_stop(&dev->vbuf_queue);
	videobuf_mmap_free(&dev->vbuf_queue);
	mutex_lock(&dev->vcapture_mutex);
	uninit_vcapture(dev);
	dev->fd_num--;
	mutex_unlock(&dev->vcapture_mutex);
	return 0;
}

static int vcapture_mmap(struct file *file, struct vm_area_struct *vma)
{
	int ret = 0;
	struct vcapture_dev *dev = video_drvdata(file);

	if (capture_secure_mode)
		vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	ret = videobuf_mmap_mapper(&dev->vbuf_queue, vma);
	dev->vma[dev->cur_vbuf_index] = vma;
	vcapture_print(PRINT_BUFFERINFO,
			"%s line %d vma=0x%08lx start=0x%08lx size=%ld ret=%d\n",
			__func__, __LINE__,
			(unsigned long)vma,
			(unsigned long)vma->vm_start,
			(unsigned long)vma->vm_end - (unsigned long)vma->vm_start,
			ret);

	return ret;
}

static const struct v4l2_file_operations vcapture_v4l2_fops = {
	.owner = THIS_MODULE,
	.open = vidioc_open,
	.release = vidioc_close,
	.read = vidioc_read,
	.poll = vidioc_poll,
	.unlocked_ioctl = video_ioctl2, /* V4L2 ioctl handler */
	.mmap = vcapture_mmap,
};

static struct video_device vcapture_template = {
	.name = "vcapture",
	.fops =  &vcapture_v4l2_fops,
	.ioctl_ops = &vcapture_ioctl_ops,
	.release = video_device_release,
	.tvnorms = V4L2_STD_525_60,
};


/* ------------------------------------------------------------------
 * capture device file ops
 * ------------------------------------------------------------------
 */
struct vcapture_dev *vcapture_device;

static int vcapture_open(struct inode *inode, struct file *file)
{
	int *vcapture_id = kzalloc(sizeof(int), GFP_KERNEL);

	if (vcapture_id) {
		*vcapture_id = -1;
		file->private_data = vcapture_id;
	}
	return 0;
}

static int vcapture_release(struct inode *inode, struct file *file)
{
	if (file->private_data != NULL)
		kfree(file->private_data);
	file->private_data = NULL;

	return 0;
}

static long vcapture_ioctl(struct file *file, unsigned int cmd, ulong arg)
{
	long ret = 0;
	void __user *argp = (void __user *)arg;
	unsigned int capture_thumbnail_width_height = 0;

	if (!vcapture_device)
		return 0;
	switch (cmd) {
	case VCAPTURE_IOCTL_SET_SKIP_FRAMES:
		get_user(capture_frame_per_n, (u32 __user *)argp);
		atomic_set(&vcapture_device->capture_frame_per_n, capture_frame_per_n);
		break;
	case VCAPTURE_IOCTL_SET_SKIP_2_LINE:
		get_user(capture_lines_per_n, (u32 __user *)argp);
		atomic_set(&vcapture_device->capture_lines_per_n, capture_lines_per_n);
		break;
	case VCAPTURE_IOCTL_SET_SECURE:
		get_user(capture_secure_mode, (u32 __user *)argp);
		break;
	case VCAPTURE_IOCTL_SET_SCALE:
		get_user(capture_scale_down, (u32 __user *)argp);
		break;
	case VCAPTURE_IOCTL_SET_THUMBNAIL_WIDTH_HIGHT:
		get_user(capture_thumbnail_width_height, (u32 __user *)argp);
		if (capture_thumbnail_width_height == 0) {
			atomic_set(&vcapture_device->is_capturing_thumbnail, 0);
			stop_capture_thread(vcapture_device);
			uninit_vcapture(vcapture_device);
			pr_info("stop capturing frame thumbnail\n");
		} else {
			vcapture_device->thumbnail_width = (capture_thumbnail_width_height >> 16) & 0xffff;
			vcapture_device->thumbnail_height = capture_thumbnail_width_height & 0xffff;
			if ((vcapture_device->thumbnail_width * vcapture_device->thumbnail_height > THUMNAIL_MAX_SIZE)
			|| (vcapture_device->thumbnail_width < THUMNAIL_MIN_WIDTH)
			|| (vcapture_device->thumbnail_height < THUMNAIL_MIN_HEIGHT)) {
				vcapture_device->thumbnail_width = THUMNAIL_DEFAULT_WIDTH;
				vcapture_device->thumbnail_height = THUMNAIL_DEFAULT_HEIGHT;
			}
			if (vcapture_device->thumbnail_width && vcapture_device->thumbnail_height) {
				atomic_set(&vcapture_device->is_capturing_thumbnail, 1);
				init_vcapture(vcapture_device);
				start_capture_thread(vcapture_device);
				pr_info("start capturing frame thumbnail %d x %d\n",
					vcapture_device->thumbnail_width,
					vcapture_device->thumbnail_height);
			}
		}
		break;
	case VCAPTURE_IOCTL_SET_THUMBNAIL_SCALE:
		get_user(thumbnail_scale_down, (u32 __user *)argp);
		break;
	default:
		return -EINVAL;
	}
	return ret;
}

#ifdef CONFIG_COMPAT
static long vcapture_compat_ioctl(struct file *file,
				 unsigned int cmd, ulong arg)
{
	long ret = 0;

	ret = vcapture_ioctl(file, cmd, (ulong)compat_ptr(arg));
	return ret;
}
#endif

static const struct file_operations vcapture_fops = {
	.owner = THIS_MODULE,
	.open = vcapture_open,
	.release = vcapture_release,
	.unlocked_ioctl = vcapture_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl = vcapture_compat_ioctl,
#endif
	.poll = NULL,
};

/* ------------------------------------------------------------------
 * capture device attr
 * ------------------------------------------------------------------
 */
static ssize_t debug_show(struct class *class,
	struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "vcapture debug: %d\n", capture_debug);
}

static ssize_t debug_store(struct class *class,
	struct class_attribute *attr,
	const char *buf, size_t count)
{
	ssize_t r;
	int val;

	r = kstrtoint(buf, 0, &val);
	if (r < 0)
		return -EINVAL;

	if (val > 0)
		capture_debug = val;
	else
		capture_debug = 0;
	pr_info("set capture_debug val:%d\n", capture_debug);
	return count;
}

static ssize_t use_thread_show(struct class *class,
	struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "vcapture use thread: %d\n", capture_use_thread ? 1 : 0);
}

static ssize_t use_thread_store(struct class *class,
	struct class_attribute *attr,
	const char *buf, size_t count)
{
	ssize_t r;
	int val;

	r = kstrtoint(buf, 0, &val);
	if (r < 0)
		return -EINVAL;

	if (val > 0)
		capture_use_thread = true;
	else
		capture_use_thread = false;
	pr_info("vcapture %s using thread\n", capture_use_thread ? " " : "not");
	return count;
}

static ssize_t frame_rate_show(struct class *class,
	struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "capture frame per %d frame(s), and capture lines per %d frame(s), 0: no capture\n",
		capture_frame_per_n, capture_lines_per_n);
}

static ssize_t frame_rate_store(struct class *class,
	struct class_attribute *attr,
	const char *buf, size_t count)
{
	ssize_t r;
	int val;

	r = kstrtoint(buf, 0, &val);
	if (r < 0)
		return -EINVAL;

	if (val > 0) {
		capture_frame_per_n = val & 0xff;
		capture_lines_per_n = (val >> 8) & 0xff;
	} else {
		capture_frame_per_n = 0;
		capture_lines_per_n = 0;
	}
	atomic_set(&vcapture_device->capture_frame_per_n, capture_frame_per_n);
	atomic_set(&vcapture_device->capture_lines_per_n, capture_lines_per_n);
	pr_info("capture frame per %d frame(s), and capture lines per %d frame(s), 0: no capture\n",
		capture_frame_per_n, capture_lines_per_n);
	return count;
}

static ssize_t scale_show(struct class *class,
	struct class_attribute *attr, char *buf)
{
	if (capture_scale_down <= 8)
		return sprintf(buf, "scale to 1/2^n: n=%d\n", capture_scale_down);
	else
		return sprintf(buf, "scale to %d x %d\n", 
			(capture_scale_down >> 16) & 0xffff, capture_scale_down & 0xffff);
}

static ssize_t scale_store(struct class *class,
	struct class_attribute *attr,
	const char *buf, size_t count)
{
	ssize_t r;
	int val;

	r = kstrtoint(buf, 0, &val);
	if (r < 0)
		return -EINVAL;

	capture_scale_down = val;
	if (capture_scale_down <= 8)
		pr_info("set scale to 1/2^n: n=%d\n", capture_scale_down);
	else
		pr_info("set scale to %d x %d\n",
			(capture_scale_down >> 16) & 0xffff, capture_scale_down & 0xffff);
	return count;
}

static ssize_t thumbnail_scale_show(struct class *class,
	struct class_attribute *attr, char *buf)
{
	return sprintf(buf, "scale to %d x %d\n", 
		(thumbnail_scale_down >> 16) & 0xffff, thumbnail_scale_down & 0xffff);
}

static ssize_t thumbnail_scale_store(struct class *class,
	struct class_attribute *attr,
	const char *buf, size_t count)
{
	ssize_t r;
	int val;

	r = kstrtoint(buf, 0, &val);
	if (r < 0)
		return -EINVAL;

	thumbnail_scale_down = val;
	pr_info("set scale to %d x %d\n",
		(thumbnail_scale_down >> 16) & 0xffff, thumbnail_scale_down & 0xffff);
	return count;
}

static ssize_t thumbnail_show(struct class *class,
	struct class_attribute *attr, char *buf)
{
	if (atomic_read(&vcapture_device->is_capturing_thumbnail))
		return sprintf(buf, "capture thumbnail %d * %d, max res = %d, min width = %d, min height = %d\n",
			vcapture_device->thumbnail_width,
			vcapture_device->thumbnail_height,
			THUMNAIL_MAX_SIZE, THUMNAIL_MIN_WIDTH, THUMNAIL_MIN_HEIGHT);
	else
		return sprintf(buf, "no capture thumbnail\n");
}

static ssize_t thumbnail_store(struct class *class,
	struct class_attribute *attr,
	const char *buf, size_t count)
{
	ssize_t r;
	int val;

	r = kstrtoint(buf, 0, &val);
	if (r < 0)
		return -EINVAL;

	if (val != 0) {
		vcapture_device->thumbnail_height = val & 0xffff;
		vcapture_device->thumbnail_width = (val >> 16) & 0xffff;
		if ((vcapture_device->thumbnail_width * vcapture_device->thumbnail_height > THUMNAIL_MAX_SIZE)
		|| (vcapture_device->thumbnail_width < THUMNAIL_MIN_WIDTH)
		|| (vcapture_device->thumbnail_height < THUMNAIL_MIN_HEIGHT)) {
			vcapture_device->thumbnail_width = THUMNAIL_DEFAULT_WIDTH;
			vcapture_device->thumbnail_height = THUMNAIL_DEFAULT_HEIGHT;
		}
		if (capture_debug & 0x4000)
			capture_secure_mode = 1;
		else if (capture_debug & 0x8000)
			capture_secure_mode = 0;
		pr_info("capture_secure_mode = %d\n",
			capture_secure_mode);
		atomic_set(&vcapture_device->is_capturing_thumbnail, 1);
		init_vcapture(vcapture_device);
		start_capture_thread(vcapture_device);
		pr_info("start capturing frame thumbnail %d x %d\n",
			vcapture_device->thumbnail_width,
			vcapture_device->thumbnail_height);
	} else {
		vcapture_device->thumbnail_height = 0;
		vcapture_device->thumbnail_width = 0;
		atomic_set(&vcapture_device->is_capturing_thumbnail, 0);
		stop_capture_thread(vcapture_device);
		uninit_vcapture(vcapture_device);
		ldim_get_vf(vcapture_device->ge2d_vf);//for debug
		pr_info("stop capturing frame thumbnail\n");
	}
	return count;
}

static struct class_attribute vcapture_class_attrs[] = {
	__ATTR(debug, 0664,
	       debug_show,
	       debug_store),
	__ATTR(use_thread, 0664,
	       use_thread_show,
	       use_thread_store),
	__ATTR(frame_rate, 0664,
	       frame_rate_show,
	       frame_rate_store),
	__ATTR(scale, 0664,
	       scale_show,
	       scale_store),
	__ATTR(thumbnail, 0664,
	       thumbnail_show,
	       thumbnail_store),
	__ATTR(thumbnail_scale, 0664,
	       thumbnail_scale_show,
	       thumbnail_scale_store),
	__ATTR_NULL
};

static struct class vcapture_class = {
	.name = "vcapture",
	.class_attrs = vcapture_class_attrs,
};

/* ------------------------------------------------------------------
 * capture device driver module
 * ------------------------------------------------------------------
 */
static int vcapture_create_instance(void)
{
	struct vcapture_dev *dev = NULL;
	struct video_device *vfd;
	int ret;

	if (vcapture_device) {
		vcapture_print(PRINT_ERROR,
			"%s vcapture device already created!\n",
			__func__);
		return -ENOMEM;
	}

	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev) {
		vcapture_print(PRINT_ERROR,
			      "%s vcapture.0 device alloc failed!\n",
			      __func__);
		return -ENOMEM;
	}
	dev->res.magic = MAGIC_RE_MEM;
	dev->res.priv = NULL;
	snprintf(dev->v4l2_dev.name, sizeof(dev->v4l2_dev.name),
		 "%s-0", VCAPTURE_MODULE_NAME);
	ret = v4l2_device_register(NULL, &dev->v4l2_dev);
	if (ret) {
		vcapture_print(PRINT_ERROR,
			      "v4l2 device register fail!\n");
		goto free_dev;
	}
	pr_info("vcapture.0 v4l2_dev.name=:%s\n", dev->v4l2_dev.name);
	dev->fmt = &formats[0];
	dev->width = 1920;
	dev->height = 1080;
	dev->thumbnail_width = THUMNAIL_DEFAULT_WIDTH;
	dev->thumbnail_height = THUMNAIL_DEFAULT_HEIGHT;
	dev->thumbnail_index = 0;
	atomic_set(&dev->capture_frame_per_n, capture_frame_per_n);
	atomic_set(&dev->capture_lines_per_n, capture_lines_per_n);
	dev->fd_num = 0;
	atomic_set(&dev->is_playing, 0);
	atomic_set(&dev->is_capturing, 0);
	atomic_set(&dev->is_capturing_thumbnail, 0);
	atomic_set(&dev->vcap_status, VCAPTURE_INIT);
	/* init video dma queues */
	INIT_LIST_HEAD(&dev->dma_q.active);
	init_waitqueue_head(&dev->dma_q.wq);
	/* initialize locks */
	spin_lock_init(&dev->slock);
	mutex_init(&dev->vcapture_mutex);
	mutex_init(&dev->vf_mutex);
	vfd = video_device_alloc();
	if (!vfd)
		goto unreg_dev;
	*vfd = vcapture_template;
	vfd->dev_debug = capture_debug;
	vfd->v4l2_dev = &dev->v4l2_dev;
	vfd->device_caps = V4L2_CAP_VIDEO_CAPTURE | V4L2_CAP_STREAMING
			| V4L2_CAP_READWRITE;
	/*
	 * Provide a mutex to v4l2 core. It will be used to protect
	 * all fops and v4l2 ioctls.
	 */
	ret = video_register_device(vfd, VFL_TYPE_GRABBER,
		vcapture_base);
	if (ret < 0)
		goto unreg_dev;

	video_set_drvdata(vfd, dev);
	dev->vdev = vfd;
	dev->src_canvas[0] = -1;
	dev->src_canvas[1] = -1;
	dev->src_canvas[2] = -1;
	dev->dst_canvas[0] = -1;
	dev->dst_canvas[1] = -1;
	dev->thumbnail_canvas[0] = -1;
	dev->thumbnail_canvas[1] = -1;
	dev->context = NULL;
	dev->new_vf = NULL;
	dev->ge2d_vf = NULL;
	snprintf(dev->recv_name, RECEIVER_NAME_SIZE,
		 RECEIVER_NAME ".0");
	vf_receiver_init(&dev->recv, dev->recv_name, &vcapture_vf_recv, dev);
	vf_reg_receiver(&dev->recv);
	snprintf(dev->prov_name, RECEIVER_NAME_SIZE,
		 PROVIDER_NAME ".0");
	vf_provider_init(&dev->prov, dev->prov_name, &vcapture_vf_prov, dev);
	pr_info("vcapture.0 v4l2 device registered as %s\n",
		video_device_node_name(vfd));
	vcapture_device = dev;
	return 0;
unreg_dev:
	v4l2_device_unregister(&dev->v4l2_dev);
free_dev:
	kfree(dev);
	return ret;
}

static int vcapture_v4l2_release(void)
{
	int ret;

	if (vcapture_device) {
		if (vcapture_device->context)
			destroy_ge2d_work_queue(vcapture_device->context);
		vf_unreg_receiver(&vcapture_device->recv);
		ret = free_canvas(vcapture_device);
		if (ret < 0)
			pr_info("vcapture free canvas failed!\n");
		pr_info("vcapture unregistering %s\n",
			video_device_node_name(vcapture_device->vdev));
		video_unregister_device(vcapture_device->vdev);
		video_device_release(vcapture_device->vdev);
		v4l2_device_unregister(&vcapture_device->v4l2_dev);
		kfree(vcapture_device);
		vcapture_device = NULL;
	}
	return 0;
}

static int vcapture_drv_probe(struct platform_device *pdev)
{
	int ret;
	struct device *devp;

	pr_info("%s\n", __func__);
	ret = class_register(&vcapture_class);
	if (ret < 0)
		return ret;

	ret = register_chrdev(VCAPTURE_MAJOR, "vcapture", &vcapture_fops);
	if (ret < 0) {
		pr_err("Can't allocate major for vcapture device\n");
		goto error1;
	}

	devp = device_create(&vcapture_class, NULL,
			     MKDEV(VCAPTURE_MAJOR, 0), NULL,
			     VCAPTURE_MODULE_NAME);
	if (IS_ERR(devp)) {
		pr_err("failed to create vcapture device node\n");
		ret = PTR_ERR(devp);
		return ret;
	}

	ret = vcapture_create_instance();
	if (ret < 0) {
		pr_err("vcapture: error %d while loading driver\n", ret);
		goto error1;
	}
	return 0;

error1:
	unregister_chrdev(VCAPTURE_MAJOR, "vcapture");
	class_unregister(&vcapture_class);
	return ret;
}

static int vcapture_drv_remove(struct platform_device *pdev)
{
	pr_info("%s\n", __func__);
	vcapture_v4l2_release();
	device_destroy(&vcapture_class, MKDEV(VCAPTURE_MAJOR, 0));
	unregister_chrdev(VCAPTURE_MAJOR, VCAPTURE_MODULE_NAME);
	class_unregister(&vcapture_class);
	return 0;
}

static const struct of_device_id vcapture_dt_match[] = {
	{
		.compatible = "amlogic, vcapture",
	},
	{}
};

static struct platform_driver vcapture_drv = {
	.probe = vcapture_drv_probe,
	.remove = vcapture_drv_remove,
	.driver = {
			.name = "vcapture",
			.owner = THIS_MODULE,
			.of_match_table = vcapture_dt_match,
		}
};

static int __init vcapture_init(void)
{
	pr_info("%s\n", __func__);
	if (platform_driver_register(&vcapture_drv)) {
		pr_err("Failed to register vcapture driver\n");
			return -ENODEV;
	}
	return 0;
}

static void __exit vcapture_exit(void)
{
	pr_info("%s\n", __func__);
	platform_driver_unregister(&vcapture_drv);
}

MODULE_DESCRIPTION("video frame capture");
MODULE_AUTHOR("Amlogic, Robin.zhu@amlogic.com>");
MODULE_LICENSE("GPL");
MODULE_VERSION("1.0");

module_init(vcapture_init);
module_exit(vcapture_exit);
