/*
 * Copyright (C) 2010-2016 ARM Limited. All rights reserved.
 *
 * This program is free software and is provided to you under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation, and any use by you of this program is subject to the terms of such GNU licence.
 *
 * A copy of the licence is included with the program, and can also be obtained from Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <linux/fs.h>       /* file system operations */
#include <linux/slab.h>     /* memort allocation functions */
#include <asm/uaccess.h>    /* user space access */

#include "mali_ukk.h"
#include "mali_osk.h"
#include "mali_kernel_common.h"
#include "mali_session.h"
#include "mali_ukk_wrappers.h"

int get_api_version_wrapper(struct mali_session_data *session_data, _mali_uk_get_api_version_s __user *uargs)
{
	_mali_uk_get_api_version_s kargs;
	_mali_osk_errcode_t err;

	MALI_CHECK_NON_NULL(uargs, -EINVAL);

	if (get_user(kargs.version, &uargs->version) != 0) return -EFAULT;

	kargs.ctx = (uintptr_t)session_data;
	err = _mali_ukk_get_api_version(&kargs);
	if (err != _MALI_OSK_ERR_OK) return map_errcode(err);

	if (put_user(kargs.version, &uargs->version) != 0) return -EFAULT;
	if (put_user(kargs.compatible, &uargs->compatible) != 0) return -EFAULT;

	return 0;
}

int get_api_version_v2_wrapper(struct mali_session_data *session_data, _mali_uk_get_api_version_v2_s __user *uargs)
{
	_mali_uk_get_api_version_v2_s kargs;
	_mali_osk_errcode_t err;

	MALI_CHECK_NON_NULL(uargs, -EINVAL);

	if (get_user(kargs.version, &uargs->version) != 0) return -EFAULT;

	kargs.ctx = (uintptr_t)session_data;
	err = _mali_ukk_get_api_version_v2(&kargs);
	if (err != _MALI_OSK_ERR_OK) return map_errcode(err);

	if (put_user(kargs.version, &uargs->version) != 0) return -EFAULT;
	if (put_user(kargs.compatible, &uargs->compatible) != 0) return -EFAULT;

	return 0;
}

int wait_for_notification_wrapper(struct mali_session_data *session_data, _mali_uk_wait_for_notification_s __user *uargs)
{
	_mali_uk_wait_for_notification_s kargs;
	_mali_osk_errcode_t err;

	MALI_CHECK_NON_NULL(uargs, -EINVAL);

	kargs.ctx = (uintptr_t)session_data;
	err = _mali_ukk_wait_for_notification(&kargs);
	if (err != _MALI_OSK_ERR_OK) return map_errcode(err);

	if (kargs.type != _MALI_NOTIFICATION_CORE_SHUTDOWN_IN_PROGRESS) {
		kargs.ctx = (uintptr_t)NULL; /* prevent kernel address to be returned to user space */
		if (copy_to_user(uargs, &kargs, sizeof(_mali_uk_wait_for_notification_s)) != 0) return -EFAULT;
	} else {
		if (put_user(kargs.type, &uargs->type) != 0) return -EFAULT;
	}

	return 0;
}

int post_notification_wrapper(struct mali_session_data *session_data, _mali_uk_post_notification_s __user *uargs)
{
	_mali_uk_post_notification_s kargs;
	_mali_osk_errcode_t err;

	MALI_CHECK_NON_NULL(uargs, -EINVAL);

	kargs.ctx = (uintptr_t)session_data;

	if (get_user(kargs.type, &uargs->type) != 0) {
		return -EFAULT;
	}

	err = _mali_ukk_post_notification(&kargs);
	if (err != _MALI_OSK_ERR_OK) {
		return map_errcode(err);
	}

	return 0;
}

int get_user_settings_wrapper(struct mali_session_data *session_data, _mali_uk_get_user_settings_s __user *uargs)
{
	_mali_uk_get_user_settings_s kargs;
	_mali_osk_errcode_t err;

	MALI_CHECK_NON_NULL(uargs, -EINVAL);

	kargs.ctx = (uintptr_t)session_data;
	err = _mali_ukk_get_user_settings(&kargs);
	if (err != _MALI_OSK_ERR_OK) {
		return map_errcode(err);
	}

	kargs.ctx = 0; /* prevent kernel address to be returned to user space */
	if (copy_to_user(uargs, &kargs, sizeof(_mali_uk_get_user_settings_s)) != 0) return -EFAULT;

	return 0;
}

int request_high_priority_wrapper(struct mali_session_data *session_data, _mali_uk_request_high_priority_s __user *uargs)
{
	_mali_uk_request_high_priority_s kargs;
	_mali_osk_errcode_t err;

	MALI_CHECK_NON_NULL(uargs, -EINVAL);

	kargs.ctx = (uintptr_t)session_data;
	err = _mali_ukk_request_high_priority(&kargs);

	kargs.ctx = 0;

	return map_errcode(err);
}

int pending_submit_wrapper(struct mali_session_data *session_data, _mali_uk_pending_submit_s __user *uargs)
{
	_mali_uk_pending_submit_s kargs;
	_mali_osk_errcode_t err;

	MALI_CHECK_NON_NULL(uargs, -EINVAL);

	kargs.ctx = (uintptr_t)session_data;
	err = _mali_ukk_pending_submit(&kargs);
	if (err != _MALI_OSK_ERR_OK) return map_errcode(err);

	return 0;
}
