#!/bin/bash

# Used by the App, logs eth and wifi connectivity data in the following format:
#SKY ETH0 0c:08:b4:13:65:7e
#SKY CHANNEL WL0 1597848552 1 0
#SKY RSSI f4:f2:6d:94:1f:b1 WL0 1597848552 2500 [-49 0 0 0] [-47 0 0 0] [-44 0 0 0] [-49 0 0 0] [-46 0 0 0] [-50 0 0 0] [-44 0 0 0] [-46 0 0 0] [-44 0 0 0] [-47 0 0 0]


# Configurable variables:
ETH_ADAPTER='eth0'
WIFI_ADAPTER='wlan0'
WIFI_ON_FILEPATH='/tmp/wifi-on'
WIFI_ADAPTER_OPERSTATE_FILEPATH="/sys/class/net/${WIFI_ADAPTER}/operstate"
LOG_FILE_PATH='/tmp/wifi_rssi_log'
RSSI_HISTORY_SIZE=10
FREQ_USECONDS=2500000

# /proc/net/wireless output is spaces separated - reduce all spaces to one:
REDUCE_SPACES_TO_ONE="tr -s ' '"

# /proc/net/wireless "Quality Level" field is RSSI value:
QUALITY_LEVEL_FIELD=5
CHANNEL_EPOCH=0
RSSI_EPOCH=0
RSSI_LOG_ARRAY=()


# This function first checks that there is a Wi-Fi connection. If it is connected 
# it gets the RSSI value from /proc/net/wireless. This is then returned by reference 
# using the first paramater. 
# Return codes: 0 - Wifi is connected
#               1 - Bad param passed in (not set)
#               2 - No Wifi connected
getCurrentRssi ()
{
    # Return the RSSI via the first parameter passed in
    local __resultvar=$1
    local WIFI_STATE=`wpa_cli status | grep wpa_state | cut -d'=' -f2`
    local RSSI_RESULT=0

    if [ -z "$1" ]; then
        echo "getCurrentRssi() No parameter specified" 
        return 1
    fi

    if [ $WIFI_STATE == "COMPLETED" ]; then
        RSSI_RESULT=`cat /proc/net/wireless | grep ${WIFI_ADAPTER} | eval "$REDUCE_SPACES_TO_ONE" | cut -d' ' -f5`
        # Return the RSSI value via the first param ($1), also use % param expansion to trim any decimal points (needs to return an int)
        eval $__resultvar="'${RSSI_RESULT%.}'"
        return 0
    fi
    
    # No connection return error code:
    return 2
}

# Adds RSSI value passed by first parameter ($1) to an array
# If the array size is equal RSSI_HISTORY_SIZE+1, the oldest is removed.
# Populates the RSSI string in format "[ -23 0 0 0 ] [ -24 0 0 0 ]..."
addRssiToLog ()
{
    # Append the RSSI value into the array:
    RSSI_LOG_ARRAY+=( "$1" )

    # Get the Array size (@ symbol)
    if [ ${#RSSI_LOG_ARRAY[@]} == $((RSSI_HISTORY_SIZE+1)) ]; then
        # Remove the first RSSI array element
        RSSI_LOG_ARRAY=(${RSSI_LOG_ARRAY[@]:1:$RSSI_HISTORY_SIZE})
    fi

    # Create the RSSI log output string (which needs to be in the format: "[ -23 0 0 0  ] [ -24 0 0 0 ]..."
    RSSI_OUT_STRING=""
    for rssi in ${RSSI_LOG_ARRAY[@]};
    do
        RSSI_OUT_STRING+="[ $rssi 0 0 0 ] "
    done
}

clearRssiLog ()
{
    RSSI_LOG_ARRAY=()
}

# Unfortunately there is no utility that returns the connected channel.
# The workaround is to cross-reference the available channel-frequency table (from "iwlist channel") with the connected frequency (in "wpa_cli status")
getChannel ()
{
    local FREQ=`wpa_cli status | grep freq | cut -d'=' -f2`
    local FREQ_IN_DECIMAL="${FREQ:0:1}.${FREQ:1}"

    # Remove any zeros (up to 2) at the end of the freq:
    FREQ_IN_DECIMAL=${FREQ_IN_DECIMAL/0}
    FREQ_IN_DECIMAL=${FREQ_IN_DECIMAL/0}

    if [ -z "$FREQ" ]; then
        WIFI_CHANNEL=0
    else
        # The channel list to be compared against is of format:
        #    "          Channel 01 : 2.412 GHz"
        # ..."          Channel 149 : 5.745 GHz"
        local CHANNEL_STRING=`iwlist p2p0 channel 2> /dev/null | grep -w $FREQ_IN_DECIMAL | eval "$REDUCE_SPACES_TO_ONE" | cut -d' ' -f3`

        # Guard against empty string:
        if [ -z "$CHANNEL_STRING" ]; then
            WIFI_CHANNEL=0
        else
            # Remove any leading zeros:
            WIFI_CHANNEL=${CHANNEL_STRING#0}
            echo "CHANNEL FOUND: $WIFI_CHANNEL"
        fi
    fi
}



ETH_MAC=`ifconfig $ETH_ADAPTER | grep -w $ETH_ADAPTER | eval "$REDUCE_SPACES_TO_ONE" | cut -d' ' -f5`
if [ -z "$ETH_MAC" ]; then
    ETH_MAC="XX:XX:XX:XX:XX:XX"
fi

ETH_ADAPTER_UPPER=`echo $ETH_ADAPTER | tr '[:lower:]' '[:upper:]'`

# First line (Ethernet) of log output:
ETH_LINE="SKY $ETH_ADAPTER_UPPER $ETH_MAC"

systemd-notify --ready --status="Starting Main loop"

while :
do
    getChannel
    if [ ! $WIFI_CHANNEL == 0 ]; then
        CHANNEL_EPOCH=`date '+%s'`
    fi
   
    WIFI_OPERSTATE_STATUS=`ifconfig | grep ${WIFI_ADAPTER}`

    if [ -f $WIFI_ON_FILEPATH ] && [ -n "$WIFI_OPERSTATE_STATUS" ]; then
        # Channel Output line:
        CHANNEL_LINE="SKY CHANNEL $WIFI_ADAPTER $CHANNEL_EPOCH $WIFI_CHANNEL 0"

        CONNECTED_AP_MAC=`wpa_cli status | grep bssid | cut -d= -f2 | tr '[:lower:]' '[:upper:]'`
        if [ -z "$CONNECTED_AP_MAC" ]; then 
            CONNECTED_AP_MAC="XX:XX:XX:XX:XX:XX"
        fi 

        # Create and pass "RSSI" variable as a parameter - RSSI value is returned via this.
        getCurrentRssi RSSI
        RSSI_STATUS=$?

        # Only update the RSSI and epoch if RSSI value is valid:
        if [ $RSSI_STATUS == 0 ]; then
            addRssiToLog $RSSI
            RSSI_EPOCH=`date '+%s'`
        fi

        SAMPLE_FREQ_IN_MSECS=$((FREQ_USECONDS/1000))

        # RSSI Output line:
        systemd-notify --status="Running: wlan0 UP"
        RSSI_LINE="SKY RSSI $CONNECTED_AP_MAC $WIFI_ADAPTER $RSSI_EPOCH $SAMPLE_FREQ_IN_MSECS $RSSI_OUT_STRING"
    
        echo -e "$ETH_LINE\n$CHANNEL_LINE\n$RSSI_LINE" > $LOG_FILE_PATH 
    else
        clearRssiLog
        systemd-notify --status="Running: wlan0 DOWN"
        echo -e "$ETH_LINE" > $LOG_FILE_PATH 
    fi

    usleep ${FREQ_USECONDS}
done 

systemd-notify --status="Not Running"
