 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/inetdevice.h>
#include "bcmflow.h"
#include "flowmgr_fap_ops.h"
#include "dqnet_priv.h"
#include "BcmGfapApi.h"
#include "ethsw.h"
#include "gFapConfig.h"

DEFINE_SPINLOCK(gfap_ops_lock);
/* GFAP only supports single tunnel for now */
#define MAX_GRE_TUNNELS 1
static int gre_tunnel_mask;

/* Utility function to get interface ip address in kernel space. */
static inline int get_ip_and_netmask(struct net_device *dev,
				     uint32 *ip,
				     uint32 *mask)
{
	struct net_device *pnet_device;
	int ret = -1;
	read_lock(&dev_base_lock);
	/* read all devices */
	for_each_netdev(&init_net, pnet_device) {
		if ((netif_running(pnet_device)) &&
				(pnet_device->ip_ptr != NULL) &&
				(!strcmp(pnet_device->name, dev->name))) {
			struct in_device *pin_dev;
			pin_dev = (struct in_device *)(pnet_device->ip_ptr);
			if (pin_dev && pin_dev->ifa_list) {
				*ip = htonl(pin_dev->ifa_list->ifa_address);
				*mask = htonl(pin_dev->ifa_list->ifa_mask);
				ret = 0;
			}
			break;
		}
	}
	read_unlock(&dev_base_lock);
	return ret;
}

static int fap_enable(bool enable)
{
	if (enable)
		BcmGfapApiConfig_Init(enable);
	return BcmGfapApiConfig_RoutingEnable(enable);
}

static int fap_config(char *pri_lan_dev_name, char *wan_dev_name,
		      char *sec_lan_dev_name)
{
	struct net_device *lan_dev;
	struct net_device *wan_dev;
	GfapNetworkParams pri;
	GfapNetworkParams sec;

	if (pri_lan_dev_name) {
		memset(&pri, 0, sizeof(GfapNetworkParams));
		lan_dev = __dev_get_by_name(&init_net, pri_lan_dev_name);
		if (!lan_dev) {
			pr_warn("%s: can't find  %s interface.\n",
				__func__, pri_lan_dev_name);
			return -1;
		}
		if (lan_dev)
			memcpy(pri.lan_mac_address, lan_dev->dev_addr, 6);
		wan_dev = __dev_get_by_name(&init_net, wan_dev_name);
		if (wan_dev)
			memcpy(pri.wan_mac_address, wan_dev->dev_addr, 6);
		pri.local_network = 0xc0a80001; /* Default value */
		pri.local_network_mask =  0xffffff00; /* Default value */
		if (get_ip_and_netmask(lan_dev, &pri.local_network,
				       &pri.local_network_mask)) {
			/* this will fail in primary VLAN bridged mode.
			   retry with wanbridge where the local address
			   is in this mode */
			if (get_ip_and_netmask(wan_dev, &pri.local_network,
					       &pri.local_network_mask))
				pr_warn(
				   "%s: IP address not initialized for device %s. Using default values.\n",
				   __func__, pri_lan_dev_name);
		 }
	}
	if (sec_lan_dev_name) {
		memset(&sec, 0, sizeof(GfapNetworkParams));
		lan_dev = __dev_get_by_name(&init_net, sec_lan_dev_name);
		if (!lan_dev) {
			pr_warn("%s: can't find  %s interface.\n",
				__func__, sec_lan_dev_name);
			return -1;
		}
		if (lan_dev)
			memcpy(sec.lan_mac_address, lan_dev->dev_addr, 6);
		wan_dev = __dev_get_by_name(&init_net, wan_dev_name);
		if (wan_dev)
			memcpy(sec.wan_mac_address, wan_dev->dev_addr, 6);
		sec.local_network = 0xc0a86401; /* Default value */
		sec.local_network_mask =  0xffffff00; /* Default value */
		if (get_ip_and_netmask(lan_dev, &sec.local_network,
				       &sec.local_network_mask))
			pr_warn(
			   "%s: IP address not initialized for device %s. Using default values.\n",
			   __func__, sec_lan_dev_name);
	}
	if (pri_lan_dev_name && sec_lan_dev_name)
		BcmGfapApiConfig_Network(&pri, &sec);
	else if (pri_lan_dev_name)
		BcmGfapApiConfig_Network(&pri, NULL);
	else if (sec_lan_dev_name)
		BcmGfapApiConfig_Network(NULL, &sec);
	else
		return -1;
	return BcmGfapApiConfig_Router();
}

static int fap_config_dslite(void *params, int oindex)
{
	GfapDsliteTunnelParams *dslite = NULL;
	if (params)
		dslite = (GfapDsliteTunnelParams *) params;
	return BcmGfapApiConfig_Dslite(dslite, false);
}

static int fap_config_gre(void *params)
{
	GfapGreTunnelParams *gre = NULL;
	int ret = 0, id;
	spin_lock(&gfap_ops_lock);

	for (id = 0; id < MAX_GRE_TUNNELS; id++) {
		if (!(gre_tunnel_mask & (1<<id))) {
			gre_tunnel_mask |= (1<<id);
			break;
		}
	}

	if (id == MAX_GRE_TUNNELS) {
		pr_err("All HW GRE tunnel resources are used\n");
		ret = -1;
		goto done;
	}
	if (params)
		gre = (GfapGreTunnelParams *) params;
	ret = BcmGfapApiConfig_Gre(gre);
	if (ret == 0) {
		gre_tunnel_mask |= (1<<id);
		gre->tunnel_id++; /* Return 1 based index */
	}
done:
	spin_unlock(&gfap_ops_lock);
	return ret;
}

static int fap_unconfig_gre(int id)
{
	GfapGreTunnelParams gre;
	int ret = 0;

	id = id - 1; /* Convert it to 0 based index */

	if ((id < 0) || (id >= MAX_GRE_TUNNELS)) {
		pr_err("GRE tunnel %d invalid\n", id);
		return -1;
	}

	spin_lock(&gfap_ops_lock);
	if (!(gre_tunnel_mask & (1<<id))) {
		pr_err("GRE tunnel %d not configured\n", id);
		ret = -1;
		goto done;
	}

	memset(&gre, 0, sizeof(GfapGreTunnelParams));
	gre.tunnel_id = id;
	ret = BcmGfapApiConfig_Gre(&gre);
	if (ret == 0)
		gre_tunnel_mask &= ~(1<<id);
done:
	spin_unlock(&gfap_ops_lock);
	return ret;
}

static int fap_flow_add(void *ptr, int *flow_id)
{
	struct flow_params *params;
	struct net_device *dev;
	struct dqnet_netdev *ndev;
	GfapTxParams gfap_tx;
	int i, ret = -1;

	params = (struct flow_params *)ptr;
	dev = __dev_get_by_index(&init_net,
				 params->tx.interface);
	if (!dev)
		goto _ret_fap_flow_add;
	ndev = netdev_priv(dev);
	if (!ndev)
		goto _ret_fap_flow_add;
	memset(&gfap_tx, 0, sizeof(GfapTxParams));
	gfap_tx.tx_interface.interface_bits.mac_id = ndev->if_id;
	gfap_tx.tx_interface.interface_bits.sub_id = ndev->if_sub_id;
	gfap_tx.tx_control_bits.mark_dscp = params->tx.control.dscp_mark;
	gfap_tx.tx_control_bits.dscp_val = params->tx.control.dscp_val;
	if (params->type == ft_mac_bridge) {
		GfapMacBridgeFlowParams gfap;
		dev = __dev_get_by_index(&init_net,
					 params->mac_bridge.rx_interface);
		if (!dev)
			goto _ret_fap_flow_add;
		ndev = netdev_priv(dev);
		if (!ndev)
			goto _ret_fap_flow_add;
		memcpy(gfap.src_mac_address,
		       params->mac_bridge.mac_src,
		       6);

		memcpy(gfap.dst_mac_address,
		       params->mac_bridge.mac_dst,
		       6);
		gfap.flow_type = params->mac_bridge.type;
		gfap.ether_type = params->mac_bridge.ether_type;
		gfap.rx_interface.interface = 0;
		gfap.rx_interface.interface_bits.mac_id = ndev->if_id;
		gfap.rx_interface.interface_bits.sub_id = ndev->if_sub_id;
		spin_lock(&gfap_ops_lock);
		ret = BcmGfapApiFlow_Add(mac_bridge,
					  &gfap,
					  &gfap_tx,
					  flow_id);
		spin_unlock(&gfap_ops_lock);
	} else if (params->type == ft_ipv4) {
		GfapIpv4FlowParams gfap;
		memcpy(&gfap,
		       &params->ipv4,
		       sizeof(GfapIpv4FlowParams));
		if (params->ipv4.type == ft_ipv4_gre_decap ||
		    params->ipv4.type == ft_ipv4_gre_encap) {
			/* copy per-flow tunnel parameters. These are inside
			   the GFAP IPv4 flow parameters but are part of the
			   generic flow for the flow manager */
			gfap.gre_tunnel_params.tunnel_id =
				ntohs(params->tunnel.gre.tunnel_id);
			gfap.gre_tunnel_params.tunnel_id--;
			gfap.gre_tunnel_params.vlan_tag =
				ntohs(params->tunnel.gre.vlan_tag);
			gfap.gre_tunnel_params.mpls_header =
				ntohl(params->tunnel.gre.mpls_hdr);
		}
		/* Byte Swapping */
		gfap.src_ip = ntohl(gfap.src_ip);
		gfap.dst_ip = ntohl(gfap.dst_ip);
		gfap.src_port = ntohs(gfap.src_port);
		gfap.dst_port = ntohs(gfap.dst_port);
		gfap.replacement_ip = ntohl(gfap.replacement_ip);
		gfap.replacement_port = ntohs(gfap.replacement_port);
		spin_lock(&gfap_ops_lock);
		ret = BcmGfapApiFlow_Add(ipv4,
					  &gfap,
					  &gfap_tx,
					  flow_id);
		spin_unlock(&gfap_ops_lock);
	} else if (params->type == ft_ipv6) {
		GfapIpv6FlowParams gfap;

		memcpy(&gfap,
		       &params->ipv6,
		       sizeof(GfapIpv6FlowParams));
		/* Byte Swapping */
		gfap.src_port = ntohs(gfap.src_port);
		gfap.dst_port = ntohs(gfap.dst_port);
		spin_lock(&gfap_ops_lock);
		ret = BcmGfapApiFlow_Add(ipv6,
					  &gfap,
					  &gfap_tx,
					  flow_id);
		spin_unlock(&gfap_ops_lock);
	} else if (params->type == ft_multicast) {
		GfapMulticastFlowParams	gfap;
		char mcast_mac[6];
		int ethsw_mask = 0;
		int iface_mask = 0;
		dev = __dev_get_by_index(&init_net,
					 params->multicast.rx_interface);
		if (!dev)
			goto _ret_fap_flow_add;
		ndev = netdev_priv(dev);
		if (!ndev)
			goto _ret_fap_flow_add;
		memset(&gfap, 0, sizeof(GfapMulticastFlowParams));
		gfap.flow_type = params->multicast.type;
		memcpy(gfap.group_id, params->multicast.group_id,
		       sizeof(gfap.group_id));
		if (gfap.group_id[0] == 0) {
			/* IPv4 case */
			gfap.group_id[3] = ntohl(gfap.group_id[3]);
			mcast_mac[0] = 0x01;
			mcast_mac[1] = 0x00;
			mcast_mac[2] = 0x5E;
			mcast_mac[3] = (gfap.group_id[3]>>16) & 0x7F;
			mcast_mac[4] = (gfap.group_id[3]>>8) & 0xFF;
			mcast_mac[5] = gfap.group_id[3] & 0xFF;
		}
		gfap.rx_interface.interface = 0;
		gfap.rx_interface.interface_bits.mac_id = ndev->if_id;
		gfap.rx_interface.interface_bits.sub_id = ndev->if_sub_id;
		for (i = 0;  i < NUM_MCTX; i++) {
			if (!params->multicast.tx_interface_mask[i])
				continue;

			dev = __dev_get_by_index(&init_net, i);
			if (!dev)
				goto _ret_fap_flow_add;
			ndev = netdev_priv(dev);
			if (!ndev)
				goto _ret_fap_flow_add;

			switch (ndev->if_id) {
			case 0: /* ETH0 */
				iface_mask |= (1 << (ndev->if_sub_id+1));
				ethsw_mask |= 0x100;
				ethsw_mask |= (1 << ndev->if_sub_id);
				break;
			case 1: /* ETH1 */
				iface_mask |= (1 << 9);
				break;
			case 2: /* WIFI0 */
				iface_mask |=
				(1 << kGfapMcastIf_Wifi0_0);
				break;
			case 3: /* WIFI1 */
				iface_mask |=
				(1 << kGfapMcastIf_Wifi1_0);
				break;
			case 4: /* CM */
				break;
			case 7: /* WIFI0 GST */
				iface_mask |=
				(1 << (ndev->if_sub_id+kGfapMcastIf_Wifi0_0));
				break;
			case 8: /* WIFI1 GST */
				iface_mask |=
				(1 << (ndev->if_sub_id+kGfapMcastIf_Wifi1_0));
				break;
			case 14: /* APP: STB */
				iface_mask |= (1 << (ndev->if_sub_id+10));
				break;
			case 15: /* EROUTER */
				iface_mask |= (1 << 31);
				break;
			}
		}
		gfap.replication_iface_mask = iface_mask;
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
		if (ethsw_mask)
			ethsw_arl_hw_table_add(mcast_mac, 0, ethsw_mask);
#endif
		for (i = 0;  i < NUM_MC2UC; i++) {
			if (!params->multicast.mc2uc[i].interface)
				break;

			dev = __dev_get_by_index(&init_net,
				params->multicast.mc2uc[i].interface);
			if (!dev)
				break;
			ndev = netdev_priv(dev);
			if (!ndev)
				break;

			switch (ndev->if_id) {
			case 0: /* ETH0 */
				gfap.mcast2ucast[i].iface_num =
					(ndev->if_sub_id + 1);
				break;
			case 1: /* ETH1 */
				gfap.mcast2ucast[i].iface_num = (9);
				break;
			case 2: /* WIFI0 */
				gfap.mcast2ucast[i].iface_num =
					(kGfapMcastIf_Wifi0_0);
				break;
			case 3: /* WIFI1 */
				gfap.mcast2ucast[i].iface_num =
					(kGfapMcastIf_Wifi1_0);
				break;
			case 4: /* CM */
				break;
			case 7: /* WIFI0 GST */
				gfap.mcast2ucast[i].iface_num =
				(ndev->if_sub_id + kGfapMcastIf_Wifi0_0);
				break;
			case 8: /* WIFI1 GST */
				gfap.mcast2ucast[i].iface_num =
				(ndev->if_sub_id + kGfapMcastIf_Wifi1_0);
				break;
			case 14: /* APP: STB */
				gfap.mcast2ucast[i].iface_num =
					(ndev->if_sub_id + 10);
				break;
			case 15: /* EROUTER */
				gfap.mcast2ucast[i].iface_num = (31);
				break;
			}

			memcpy(gfap.mcast2ucast[i].dst_mac_address,
			       params->multicast.mc2uc[i].mac_dst, 6);
		}
		spin_lock(&gfap_ops_lock);
		ret = BcmGfapApiFlow_Add(multicast,
					  &gfap,
					  &gfap_tx,
					  flow_id);
		spin_unlock(&gfap_ops_lock);
	}

_ret_fap_flow_add:
	return ret;
}

static int fap_flow_remove(int flow_id)
{
	int ret = 0;

	if (flow_id) {
		spin_lock(&gfap_ops_lock);
		ret = BcmGfapApiFlow_Remove(flow_id);
		spin_unlock(&gfap_ops_lock);
	}

	return ret;
}

static int fap_flow_remove_mcast(int flow_id, __be32 *group_id, int oindex)
{
	int ret = 0;

	if (flow_id) {
		if (group_id[0] == 0) {
			char mcast_mac[6];
			/* IPv4 case */
			group_id[3] = ntohl(group_id[3]);

			mcast_mac[0] = 0x01;
			mcast_mac[1] = 0x00;
			mcast_mac[2] = 0x5E;
			mcast_mac[3] = (group_id[3]>>16) & 0x7F;
			mcast_mac[4] = (group_id[3]>>8) & 0xFF;
			mcast_mac[5] = (group_id[3]) & 0xFF;
#if defined(CONFIG_BCM_ETHSW) || defined(CONFIG_BCM_ETHSW_MODULE)
			ethsw_arl_hw_table_add(mcast_mac, 0, 0);
#endif
		}

		spin_lock(&gfap_ops_lock);
		ret = BcmGfapApiFlow_Remove(flow_id);
		spin_unlock(&gfap_ops_lock);
	}

	return ret;
}

static int fap_flow_get_counter(int flow_id, u32 *packets, u32 *bytes,
				bool reset)
{
	*bytes = 0;
	return BcmGfapApiFlow_GetCounter(flow_id, (uint32 *)packets, reset);
}

static int fap_flow_get_mcast_counter(int flow_id, u32 *packets, u32 *bytes,
				bool reset)
{
	return 0;
}

static int fap_dump(void)
{
	pr_info("--- GFAP debug dump ----\n");
	pr_info("gre_tunnel_mask        : 0x%x\n", gre_tunnel_mask);
	return 0;
}

static int fap_flow_max(void)
{
	return 1024;
}

static struct fap_ops fapops = {
	.type = GFAP,
	.name = "HW FAP - GFAP",
	.features = (FLOW_F_L4_ALL | FLOW_F_GRETAP4WAN |
		     FLOW_F_GRETAP6WAN | FLOW_F_DSLITE |
		     FLOW_F_MACBR | FLOW_F_MCAST),
	.dump = fap_dump,
	.enable = fap_enable,
	.config = fap_config,
	.config_dslite = fap_config_dslite,
	.config_gre = fap_config_gre,
	.unconfig_gre = fap_unconfig_gre,
	.flow_add = fap_flow_add,
	.flow_remove = fap_flow_remove,
	.flow_remove_mcast = fap_flow_remove_mcast,
	.flow_get_counter = fap_flow_get_counter,
	.flow_get_mcast_counter = fap_flow_get_mcast_counter,
	.flow_max = fap_flow_max,
};

struct fap_ops *gfap(void)
{
	return &fapops;
}

static int __init flowmgr_gfap_init(void)
{
	flowmgr_register_fap(gfap());
	return 0;
}

static void __exit flowmgr_gfap_exit(void)
{
	flowmgr_unregister_fap(gfap());
}

module_init(flowmgr_gfap_init);
module_exit(flowmgr_gfap_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("flowmgr_gfap");
