 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2015-2016 Broadcom. All rights reserved.
 * The term Broadcom refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/inet.h>
#include <linux/proc_fs.h>
#include "sfap.h"
#include <proc_cmd.h>

#define SFAP_PROC_DIR_NAME	"driver/sfap"

struct private_data {
	int  type;
	int  count;
};

static void *sfap_inf_seq_start(struct seq_file *seq, loff_t *pos)
{
	if (*pos == 0)
		return SEQ_START_TOKEN;
	if ((*pos-1) < sfap_inf_num())
		return sfap_inf_get_by_index(*pos-1);

	return NULL;
}

static void *sfap_inf_seq_next(struct seq_file *seq, void *v,
				      loff_t *pos)
{
	++*pos;
	if ((*pos-1) < sfap_inf_num())
		return sfap_inf_get_by_index(*pos-1);

	return NULL;
}

static void sfap_inf_seq_stop(struct seq_file *seq, void *v)
{
}

static int sfap_inf_seq_show(struct seq_file *seq, void *v)
{
	struct sfap_inf *inf = v;
	if (v == SEQ_START_TOKEN) {
		seq_printf(seq,
			   "index ifindex flag     name     rx pkts  tx pkts  rx bytes tx bytes\n");
		return 0;
	}

	seq_printf(seq,
		   "%-2d    %-2d      %-8x %-8s %08x %08x %08x %08x\n",
		   inf->index,
		   inf->dev->ifindex,
		   inf->flags,
		   inf->dev->name,
		   inf->rx_stats.packets,
		   inf->tx_stats.packets,
		   inf->rx_stats.bytes,
		   inf->tx_stats.bytes);
	return 0;
}

static const struct seq_operations sfap_inf_seq_ops = {
	.start	= sfap_inf_seq_start,
	.next	= sfap_inf_seq_next,
	.stop	= sfap_inf_seq_stop,
	.show	= sfap_inf_seq_show,
};

static int sfap_inf_seq_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &sfap_inf_seq_ops);

	if (!ret) {
		struct seq_file *sf = file->private_data;
		sf->private = PDE_DATA(inode);
	}
	return ret;
};

static void *sfap_flow_seq_start(struct seq_file *seq, loff_t *pos)
{
	struct private_data *data = (struct private_data *) seq->private;
	struct sfap_flow *flow = NULL;
	if (*pos == 0)
		return SEQ_START_TOKEN;
	if ((*pos-1) >= sfap_flow_get_total())
		return NULL;
	if (data->type != 0xFF) {
		flow = sfap_flow_get_by_index(*pos-1);
		while (data->type && flow &&
		       (flow->params.type != data->type)) {
			++*pos;
			if ((*pos-1) >= sfap_flow_get_total())
				return NULL;
			flow = sfap_flow_get_by_index(*pos-1);
		}
	} else {
		flow = sfap_flow_free_get_by_index(*pos-1);
	}
	return flow;
}

static void *sfap_flow_seq_next(struct seq_file *seq, void *v, loff_t *pos)
{
	struct private_data *data = (struct private_data *) seq->private;
	struct sfap_flow *flow = NULL;
	++*pos;
	if ((*pos-1) >= sfap_flow_get_total())
		return NULL;
	if (data->type != 0xFF) {
		flow = sfap_flow_get_by_index(*pos-1);
		while (data->type && flow &&
		       (flow->params.type != data->type)) {
			++*pos;
			if ((*pos-1) >= sfap_flow_get_total())
				return NULL;
			flow = sfap_flow_get_by_index(*pos-1);
		}
	} else {
		flow = sfap_flow_free_get_by_index(*pos-1);
	}
	return flow;
}

static void sfap_flow_seq_stop(struct seq_file *seq, void *v)
{
	/* Strange... this is called in between show */
}

static long timespec_sub(struct timespec64 a, struct timespec64 b)
{
        long ret = b.tv_sec + b.tv_nsec / NSEC_PER_SEC;

        ret -= a.tv_sec + a.tv_nsec / NSEC_PER_SEC;
        return ret;
}

static int sfap_flow_seq_show(struct seq_file *seq, void *v)
{
	struct private_data *data = (struct private_data *) seq->private;
	struct sfap_flow *flow = v;
	if (v == SEQ_START_TOKEN) {
		data->count = 0;
		if ((data->type == 0) || (data->type == 0xFF))
			seq_printf(seq,
				   "index flowid hash type ctime   utime\n");
		else if (data->type == ft_ipv4)
			seq_printf(seq,
				   "index flowid hash t d p  "
				   "src ip-port"
				   "           "
				   "dst ip-port"
				   "           "
				   "replacement ip-port"
				   "   "
				   "rep_src_mac"
				   "       "
				   "rep_dst_mac"
				   "       "
				   "tx_idx "
				   "pkts"
				   "     "
				   "bytes\n");
		else if (data->type == ft_ipv6)
			seq_printf(seq,
				   "index flowid hash t d p  "
				   "src ip-port"
				   "                                   "
				   "dst ip-port"
				   "                                   "
				   "rep_src_mac"
				   "       "
				   "rep_dst_mac"
				   "       "
				   "tx_idx "
				   "pkts"
				   "     "
				   "bytes\n");
		else if (data->type == ft_mac_bridge)
			seq_printf(seq,
				    "index flowid hash t "
				   "src_mac"
				   "           "
				   "dst_mac"
				   "           "
				   "rx_idx "
				   "tx_idx "
				   "pkts"
				   "     "
				   "bytes\n");
		return 0;
	} else if (v == NULL) {
		return 0;
	}
	if ((data->type == 0) || (data->type == 0xFF)) {
		long cdelta, udelta;
		struct	timespec64 now;
		ktime_get_real_ts64(&now);
		cdelta = timespec_sub(now, flow->ctime);
		udelta = timespec_sub(now, flow->utime);
		seq_printf(seq,
			   "%-5d %-5d  %-4x %-1d     %-8ld %-8ld\n",
			   data->count, flow->index+1,
			   flow->hash, flow->params.type,
			   cdelta, udelta);
	} else if (data->type == ft_ipv4) {
		struct flow_ipv4_params *params = &flow->params.ipv4;
		seq_printf(seq,
			   "%-5X %-5d  %-4X %-1d %-1d %-2d %-15pI4-%-5hu %-15pI4-%-5hu %-15pI4-%-5hu %pM %pM %-5d  %08x %08x\n",
			   data->count, flow->index+1,
			   flow->hash, flow->params.type,
			   params->type, params->ip_prot,
			   &params->src_ip, ntohs(params->src_port),
			   &params->dst_ip, ntohs(params->dst_port),
			   &params->replacement_ip,
			   ntohs(params->replacement_port),
			   params->replacement_mac_src,
			   params->replacement_mac_dst,
			   flow->params.tx.interface,
			   flow->stats.packets, flow->stats.bytes);
	} else if (data->type == ft_ipv6) {
		struct flow_ipv6_params *params = &flow->params.ipv6;
		seq_printf(seq,
			   "%-5X %-5d  %-4X %-1d %-1d %-2d %-39pI6-%-5hu %-39pI6-%-5hu %pM %pM %-5d  %08x %08x\n",
			   data->count, flow->index+1,
			   flow->hash, flow->params.type,
			   params->type, params->ip_prot,
			   params->src_ip, ntohs(params->src_port),
			   params->dst_ip, ntohs(params->dst_port),
			   params->replacement_mac_src,
			   params->replacement_mac_dst,
			   flow->params.tx.interface,
			   flow->stats.packets, flow->stats.bytes);
	} else if (data->type == ft_mac_bridge) {
		struct flow_mac_bridge_params *params;
		struct sfap_inf *inf;
		params = &flow->params.mac_bridge;
		inf = sfap_inf_get_by_dev_ifindex(params->rx_interface);
		if (inf == NULL)
			return 0;
		seq_printf(seq,
			   "%-5d %-5d  %-4X %-1d %pM %pM %-5d  %-5d  %08x %08x\n",
			   data->count, flow->index+1,
			   flow->hash, flow->params.type,
			   params->mac_src,
			   params->mac_dst,
			   inf->index,
			   flow->params.tx.interface,
			   flow->stats.packets, flow->stats.bytes);
	} else {
	}
	data->count++;
	return 0;
}

static const struct seq_operations sfap_flow_seq_ops = {
	.start = sfap_flow_seq_start,
	.next  = sfap_flow_seq_next,
	.stop  = sfap_flow_seq_stop,
	.show  = sfap_flow_seq_show,
};

static int sfap_flow_seq_open(struct inode *inode, struct file *file)
{
	int ret = seq_open(file, &sfap_flow_seq_ops);

	if (!ret) {
		struct seq_file *sf = file->private_data;
		sf->private = PDE_DATA(inode);
	}
	return ret;
};

static int sfap_cmd_add_flow_ipv4_nat(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_ipv4_params *fparams = &params.ipv4;
	int ret = -1;
	params.type = ft_ipv4;
	if (kstrtoint(argv[2], 10, (int *)&fparams->type) != 0) {
		pr_info("Adding Flow: wrong value - IPv4 flow type\n");
	} else if (kstrtou8(argv[3], 10, &fparams->ip_prot) != 0) {
		pr_info("Adding Flow: wrong value - ip_prot\n");
	} else if (in4_pton(argv[4], strlen(argv[4]),
			    (u8 *)&fparams->src_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - src_ip\n");
	} else if (in4_pton(argv[5], strlen(argv[5]),
			    (u8 *)&fparams->dst_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - dst_ip\n");
	} else if (kstrtou16(argv[6], 10, &fparams->src_port) != 0) {
		pr_info("Adding Flow: wrong value - src_port\n");
	} else if (kstrtou16(argv[7], 10, &fparams->dst_port) != 0) {
		pr_info("Adding Flow: wrong value - dst_port\n");
	} else if (in4_pton(argv[8], strlen(argv[8]),
			    (u8 *)&fparams->replacement_ip,
			     -1, NULL) == 0) {
		pr_info("Adding Flow: wrong value - replacement_ip\n");
	} else if (kstrtou16(argv[9], 10, &fparams->replacement_port) != 0) {
		pr_info("Adding Flow: wrong value - replacement_port\n");
	} else if (mac_pton(argv[10], fparams->replacement_mac_src) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_src\n");
	} else if (mac_pton(argv[11], fparams->replacement_mac_dst) == 0) {
		pr_info("Adding Flow: wrong value - replacement_mac_dst\n");
	} else if (kstrtou32(argv[12], 10, &params.tx.interface) != 0) {
		pr_info("Adding Flow: wrong value - tx_interface\n");
	} else {
		if (sfap_flow_add(&params, &flow_id) == 0) {
			pr_info("Flow (ft_ipv4_nat ) added: %d\n", flow_id);
			ret = 0;
		}
	}
	return 0;
}

static int sfap_cmd_add_flow_mac_bridge(int argc, char *argv[])
{
	int flow_id;
	struct flow_params params;
	struct flow_mac_bridge_params *fparams = &params.mac_bridge;
	int ret = -1;
	params.type = ft_mac_bridge;
	if (mac_pton(argv[2], fparams->mac_src) == 0) {
		pr_info("Adding Flow: wrong value - src_mac_address\n");
	} else if (mac_pton(argv[3], fparams->mac_dst) == 0) {
		pr_info("Adding Flow: wrong value - dst_mac_address\n");
	} else if (kstrtou32(argv[4], 10, &fparams->rx_interface) != 0) {
		pr_info("Adding Flow: wrong value - rx_interface\n");
	} else if (kstrtou32(argv[5], 10, &params.tx.interface) != 0) {
		pr_info("Adding Flow: wrong value - tx_interface\n");
	} else {
		if (sfap_flow_add(&params, &flow_id) == 0) {
			pr_info("Flow (mac_bridge) added: %d\n", flow_id);
			ret = 0;
		}
	}
	return 0;
}

static const struct proc_ops sfap_proc_inf_fops = {
	.proc_open     = sfap_inf_seq_open,
	.proc_read     = seq_read,
	.proc_release  = seq_release,
};

static const struct proc_ops sfap_proc_flow_fops = {
	.proc_open     = sfap_flow_seq_open,
	.proc_read     = seq_read,
	.proc_lseek    = seq_lseek,
	.proc_release  = seq_release,
};

struct sfap_proc_table {
	const char                    *procname;
	const struct proc_ops         *fops;
	umode_t                       mode;
	struct private_data           data;
};

struct sfap_proc_table sfap_proc_entries_name[] = {
	{	.procname = "inf",
		.fops = &sfap_proc_inf_fops,
		.mode = S_IRUGO,
		.data = {0, 0}
	},
	{	.procname = "flow_all",
		.fops = &sfap_proc_flow_fops,
		.mode = S_IRUGO,
		.data = {0, 0}
	},
	{.	procname = "flow_free",
		.fops = &sfap_proc_flow_fops,
		.mode = S_IRUGO,
		.data = {0xFF, 0}
	},
	{	.procname = "flow_ipv4",
		.fops = &sfap_proc_flow_fops,
		.mode = S_IRUGO,
		.data = {ft_ipv4, 0}
	},
	{	.procname = "flow_ipv6",
		.fops = &sfap_proc_flow_fops,
		.mode = S_IRUGO,
		.data = {ft_ipv6, 0}
	},
	{	.procname = "flow_bridge",
		.fops = &sfap_proc_flow_fops,
		.mode = S_IRUGO,
		.data = {ft_mac_bridge, 0}
	},
	/*
	{	.procname = "flow_mcast",
		.fops = &sfap_proc_flow_fops,
		.mode = S_IRUGO,
		.data = {ft_multicast, 0}
	},
	*/
};
static int cmd_status(int argc, char *argv[])
{
	pr_info("Total flows\t: %d\n",
		sfap_flow_get_total());
	pr_info("Active flows\t: %d\n",
		sfap_flow_get_active());
	pr_info("Max Active flows\t: %d\n",
		sfap_flow_get_active_max());
	pr_info("Cumulative mac flows match: %ld\n",
		sfap_flow_match_get_mac_cnt());
	pr_info("Cumulative iv4 flows match: %ld\n",
		sfap_flow_match_get_ipv4_cnt());
	pr_info("Cumulative iv6 flows match: %ld\n",
		sfap_flow_match_get_ipv6_cnt());
	sfap_inf_show();
	return 0;
}
static int cmd_clear(int argc, char *argv[])
{
	int i;
	if (argc == 1) {
		sfap_flow_stats_clear();
		for (i = 0; i < sfap_inf_num(); i++) {
			struct sfap_inf *inf;
			inf = sfap_inf_get_by_index(i);
			if (inf) {
				sfap_inf_stats_clear(inf);
				pr_info("Clear counters - %s\n",
					inf->dev->name);
				goto done;
			}
		}
	} else {
		struct sfap_inf *inf;
		inf = sfap_inf_get_by_name(argv[1]);
		if (inf) {
			sfap_inf_stats_clear(inf);
			pr_info("Clear counters - %s\n",
				inf->dev->name);
			goto done;
		}
	}
/* help */
	pr_info("Clear counters for specific device:\n");
	pr_info(" %s <dev_name>\n", argv[0]);
	pr_info("If no arguments: Clear counters for all devices\n");
done:
	return 0;
}
static int cmd_flow_add(int argc, char *argv[])
{
	if (argc > 2) {
		int type, ret = -1;
		if (kstrtoint(argv[1], 10, &type) == 0) {
			pr_info("Adding Flow Type %d\n", type);
			if ((type == ft_ipv4) && argc == 13) {
				ret = sfap_cmd_add_flow_ipv4_nat(argc,
								 argv);
			/*} else if (type == ft_ipv6) {*/
			} else if ((type == ft_mac_bridge) && argc == 6) {
				ret = sfap_cmd_add_flow_mac_bridge(argc,
								   argv);
			}
			if (ret == 0)
				goto done;
		}
	}
/* help */
	pr_info("For adding ipv4_nat flow:\n");
	pr_info("  %s 1 <dir> <ip_prot> <src_ip> <dst_ip> <src_port> <dst_port> <rep_ip> <rep_port> <rep_mac_src> <rep_mac_dst> <tx_inf>\n",
		argv[0]);
	pr_info("For adding bridge flow:\n");
	pr_info("  %s 3 <mac_src> <mac_dst> <rx_inf> <tx_inf>\n", argv[0]);
done:
	return 0;
}
static int cmd_flow_delall(int argc, char *argv[])
{
	if (argc == 1) {
		sfap_flow_delete_all();
		goto done;
	}
/* help */
	pr_info("Delete all flows\n");
done:
	return 0;
}
static int cmd_flow_del(int argc, char *argv[])
{
	if (argc > 1) {
		int index;
		if (kstrtoint(argv[1], 10, &index) == 0) {
			sfap_flow_remove(index);
			pr_info("Flow %d Deleted\n", index);
			goto done;
		}
	}
/* help */
	pr_info("Delete flow with flow id\n");
	pr_info("%s <flow_id>\n", argv[0]);
done:
	return 0;
}
static int cmd_enable(int argc, char *argv[])
{
	if (argc > 1) {
		int index;
		if (kstrtoint(argv[1], 10, &index) == 0) {
			sfap_enable(index);
			pr_info("Software FAP %s\n",
				index?"Enable":"Disable");
			goto done;
		}
	}
/* help */
	pr_info("Enable/Disable Software FAP\n");
	pr_info("%s <0|1>\n", argv[0]);
done:
	return 0;
}

static int cmd_inf(int argc, char *argv[])
{
	if (argc > 2) {
		if (strcasecmp("add", argv[1]) == 0) {
			if (sfap_inf_register_netdevice(argv[2]) == 0)
				goto done;
		} else if (strcasecmp("del", argv[1]) == 0) {
			if (sfap_inf_unregister_netdevice(argv[2]) == 0)
				goto done;
		}
	}
/* help */
	pr_info("%s <add|del> <dev_name>\n", argv[0]);
done:
	return 0;
}

static struct proc_cmd_ops command_entries[] = {
	{ .name = "flow_add", .do_command = cmd_flow_add},
	{ .name = "flow_del", .do_command = cmd_flow_del},
	{ .name = "flow_delall", .do_command = cmd_flow_delall},
	{ .name = "status", .do_command = cmd_status},
	{ .name = "show", .do_command = cmd_status},
	{ .name = "clear", .do_command = cmd_clear},
	{ .name = "enable", .do_command = cmd_enable},
	{ .name = "inf", .do_command = cmd_inf},
};

static struct proc_cmd_table command_table = {
	.module_name = "SFAP",
	.size = ARRAY_SIZE(command_entries),
	.ops = command_entries
};

static struct
proc_dir_entry *sfap_proc_dir;
static struct
proc_dir_entry *sfap_proc_entries[ARRAY_SIZE(sfap_proc_entries_name)];

void sfap_procfs_init(void)
{
	int i;
	sfap_proc_dir = proc_mkdir(SFAP_PROC_DIR_NAME, NULL);
	if (sfap_proc_dir == NULL) {
		pr_warn("SFAP Warning: cannot create /proc/%s\n",
			SFAP_PROC_DIR_NAME);
		return;
	}

	for (i = 0; i < ARRAY_SIZE(sfap_proc_entries_name); i++) {
		sfap_proc_entries[i] =
			proc_create_data(sfap_proc_entries_name[i].procname,
					 sfap_proc_entries_name[i].mode,
					 sfap_proc_dir,
					 sfap_proc_entries_name[i].fops,
					 &sfap_proc_entries_name[i].data);
	}
	proc_create_cmd("cmd", sfap_proc_dir, &command_table);
}

void sfap_procfs_exit(void)
{
	int i;
	if (sfap_proc_dir) {
		for (i = 0; i < ARRAY_SIZE(sfap_proc_entries_name); i++)
			remove_proc_entry(sfap_proc_entries_name[i].procname,
					  sfap_proc_dir);
		remove_proc_entry("cmd", sfap_proc_dir);
		remove_proc_entry(SFAP_PROC_DIR_NAME, NULL);
		sfap_proc_dir = NULL;
	}
}
