/*
 * Driver for BCM fixed regulator
 *
 * Copyright (C) 2020 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef CONFIG_BCM_KF_CM

#ifndef __REGULATOR_GPIOS_PINCTRL_H
#define __REGULATOR_GPIOS_PINCTRL_H

struct regulator_init_data;
struct platform_device;

#define NR_PIN_STATES 2
#define BCM_FIXED_SYSCON_MAGIC ((unsigned int *)0x42465343) /*'B' 'F' 'S' 'C' */

/**
 * struct bcm_fixed_gpios -
 * bcm_fixed_gpios structure
 * @gpios:	GPIOs to use for enable power/reset control
 * @ngpios:	Number of power/reset GPIOs
 * set to 0 if not used
 */
struct bcm_fixed_gpios {
	int *gpios;
	int ngpios;
};

/**
 * struct bcm_fixed_pinctrl -
 * bcm_fixed_pinctrl structure
 * @pinctrl:	pinctrl to use changing pin states
 * @states:	pin states
 * @names:	Number of pin states
 * set to 0 if not used
 */
struct bcm_fixed_pinctrl {
	struct pinctrl *pinctrl;
	struct pinctrl_state *states[NR_PIN_STATES];
	const char *names[NR_PIN_STATES];
};

/**
 * bcm_fixed_notify syscon notify function handler -
 * bcm_fixed_notify arguments
 * @handle:	syscon handle
 * @symbol:	syscon symbol
 * @status:	status to be notified
 *
 * This is callback notification function from BCM fixed regulator
 * to inform/update the syscon about regulator status change.
 */
typedef int (*bcm_fixed_notify)(struct platform_device *handle, const char *symbol, const char *status);

/**
 * struct bcm_fixed_syscon -
 * bcm_fixed_syscon structure
 * @kmagic:	syscon kmagic
 * @notify:	syscon notify function handler
 * @handle:	syscon handle
 * @symbol:	syscon symbol
 */
struct bcm_fixed_syscon {
	unsigned int *kmagic;
	bcm_fixed_notify notify;
	struct platform_device *handle;
	const char *symbol;
	const char *states[NR_PIN_STATES];
};

/**
 * struct bcm_fixed_config -
 * bcm_fixed_config structure
 * @init_data:		regulator_init_data
 * @power:		GPIO structure for enable power control
 * @reset:		GPIO structure for enable reset control
 * @pinctrl:		pinctrl structure gpio and pinmux state
 * @syscon:		syscon callback structure for notifying status change
 * @startup_delay:	delay time after power on in
 * microseconds
 * @shutdown_delay:	delay time after power off in
 * microseconds
 * @assert_delay:	delay time after reset active in
 * microseconds
 * @deassert_delay:	delay time after reset inactive
 * in microseconds
 *
 * This structure contains BCM fixed regulator
 * configuration information that must be passed by platform
 * code to the BCM fixed regulator driver.
 */
struct bcm_fixed_config {
	struct regulator_init_data *init_data;
	struct bcm_fixed_gpios power;
	struct bcm_fixed_gpios reset;
	struct bcm_fixed_pinctrl pinctrl;
	struct bcm_fixed_syscon syscon;
	unsigned int startup_delay;
	unsigned int shutdown_delay;
	unsigned int assert_delay;
	unsigned int deassert_delay;
};

#endif

#endif /* CONFIG_BCM_KF_CM */
